/*
 * last modified---
 * 	03-27-25 fall back to passed AES keys for input eNFTs if not returned by AUD
 * 	02-24-24 limit memo lines to 1024 characters
 * 	03-08-24 try additional AUDs as key servers if we get a failure
 * 	12-04-23 do not auto-upload receipts; remove M_ReceiptUpload state
 * 	07-04-23 after getNFTsById() call, check for greylisting and dwell time
 * 	06-29-23 getABI() moved to SmartContractConfig
 * 	06-14-23 remove AssetConfig usage
 * 	06-12-23 remove standalone mode requirement
 * 	04-11-23 check inputs and payee counts vs M_ARRAY_MAX
 * 	04-05-23 build receipts with EIP-55 addresses and token contract assets
 * 	03-02-23 check and generate valid eNFT sigs, generate real receipt sigs
 * 	03-01-23 sign entire OB, entire AB
 * 	02-28-23 also sign the ArgumentsHash block in the AB
 * 	02-24-23 add signing of ArgumentsHash block in OB, verification of OB sig
 * 	12-02-22 handle eNFTmetadata.m_Rand field, payee m_Rand
 * 	10-13-22 use EncodingUtils.buildDetailsHash()
 * 	08-19-22 finish debugging
 * 	08-04-22 complete draft
 * 	07-26-22 begin actual implementation
 * 	04-06-22 new (stub)
 *
 * purpose---
 * 	state machine transition object for handling dApp spend requests
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.AuditorBlock;
import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.eNFTmetadata;
import cc.enshroud.jetty.ReceiptBlock;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.WalletUtils;
import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;

import java.util.Map;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Base64;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.net.URI;
import java.text.NumberFormat;
import java.nio.charset.StandardCharsets;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.AsyncContext;


/**
 * This class provides the implementation of control logic to handle the various
 * stages required to perform a spend of new eNFTs for one or more input eNFTs,
 * along with error handling and recovery for any problems encountered at
 * each stage.  These objects are created from within {@link ClientHandler}
 * in response to user requests (implying m_IsLead is true).  They are also
 * created from {@link MVOHandler} when a lead MVO picks us as a committee
 * member (implying m_IsLead is false).
 */
public final class SpendState extends ReqSvcState {
	// BEGIN data members
	// aliases for steps in the spend process
	/**
	 * request received from dApp (parsed okay, sig verified)
	 */
	public static final int	M_ReqReceived = 1;
	/**
	 * request verified as acceptable (sanity, checked vs SC config, input
	 * eNFTs exist and are valid, etc.)
	 */
	public static final int M_ReqVerified = 2;
	/**
	 * keys obtained from an Auditor for listed input eNFTs, and all provided
	 * inputs compared against actual eNFTs retrieved from blockchain
	 */
	public static final int M_InputKeys = 3;
	/**
	 * keys obtained from an Auditor for required output eNFTs (note a
	 * committee member MVO will lookup the keys rather than creating them)
	 */
	public static final int M_NewKeys = 4;
	/**
	 * OperationsBlock generated (and all output eNFTs generated)
	 */
	public static final int M_OB_Generated = 5;
	/**
	 * OperationsBlock signed (note a committee member compares its own OB
	 * against the one passed by the lead, and signs the passed one) -- the
	 * completion of step implies that ALL committee MVOs have signed, not just
	 * the lead
	 */
	public static final int M_OB_Signed = 6;
	/**
	 * receipt keys fetched and receipts generated (lead MVO only)
	 */
	public static final int M_ReceiptGen = 7;
	/**
	 * reply sent (to dApp for lead, to lead MVO for committee member)
	 */
	public static final int M_Replied = 8;
	/**
	 * transaction mined, mints/burns of eNFTs appear in chain event log (lead)
	 */
	public static final int	M_Spent = 9;
	/**
	 * processing is done, clean up
	 */
	public static final int M_Completed = 10;

	/**
	 * flag indicating whether we're the lead MVO, or just a committee member
	 * (true if lead)
	 */
	private boolean			m_IsLead;

	/**
	 * local shorthand copy of MVO's logging object
	 */
	private Log				m_Log;

	/**
	 * list of change outputs to catch any overage, assets mapped to amounts
	 */
	private HashMap<String, BigInteger>	m_ChangeOutputs;

	/**
	 * list of assets and totals to be emitted as output eNFTs (asset to amount)
	 */
	private Hashtable<String, BigInteger>	m_OutputAssets;

	/**
	 * list of assets and totals found in the input eNFTs (asset to amount)
	 */
	private Hashtable<String, BigInteger>	m_InputAssets;

	/**
	 * list of input eNFTs retrieved from on-chain
	 */
	private ArrayList<NFTmetadata>	m_FetchedInputs;

	/**
	 * the reply we got from an Auditor fetching the keys to decrypt input eNFTs
	 */
	private AuditorKeyBlock	m_InputKeyResponse;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param brokerEnt the parent object to which we are attached
	 */
	public SpendState(MVOBrokerEntry brokerEnt) {
		super(brokerEnt);
		m_FinalStep = M_Completed;
		m_Log = m_Parent.getLog();
		m_ChangeOutputs = new HashMap<String, BigInteger>();
		m_OutputAssets = new Hashtable<String, BigInteger>();
		m_InputAssets = new Hashtable<String, BigInteger>();
		m_FetchedInputs = new ArrayList<NFTmetadata>();
	}

	/**
	 * method to advance the state (required to extend ReqSvcState)
	 * @param last the last step we attempted to perform (the from state)
	 * @param next the step number to go to
	 * @param reason the success/failure code causing us to go to this event
	 * (if an error code, the error will have occurred doing the previous step)
	 * @param errMsg a text error message describing the failure, or empty
	 * @return whether the state advance worked successfully (can be true even
	 * if handling an error reason code)
	 */
	public boolean advanceState(int last, int next, int reason, String errMsg) {
		final String lbl = this.getClass().getSimpleName() + ".advanceState: ";
	/*
		m_Log.debug(lbl + "transition from " + last + " to " + next
					+ ", reason " + reason + ", emsg \"" + errMsg + "\"");
	 */
		
		// examine inputs
		if (next > m_FinalStep) {
			m_Log.error(lbl + "invalid next step number: " + next);
			return false;
		}
		if (next < last) {
			m_Log.error(lbl + "reverse state transition from " + next
						+ " to " + last);
			return false;
		}
		if (reason < 0) {
			m_Log.error(lbl + "invalid step advance reason: " + reason);
			return false;
		}
		m_PrevStep = last;
		m_CurrStep = next;
		m_FailureMode = reason;
		if (errMsg != null) {
			m_FailureMsg = errMsg;
		}

		// local convenience definitions
		MVO mvo = m_Parent.getMVO();
		MVOState stObj = mvo.getStateObj();
		AsyncContext rep = m_Parent.getDappReply();
		if (m_IsLead && rep == null) {
			m_Log.error(lbl + "lead without an AsyncContext, abort");
			return false;
		}
		String mvoRep = m_Parent.getMVOReplyId();
		String leadMVOId = "";
		ClientHandler cHandler = mvo.getClientHandler();

		// handle errors, based on the 'last' step completed
		boolean ret = true;
		switch (reason) {
			// request failed to parse, failed signature check etc.
			case M_InvalidReq:
				// context: M_ReqReceived --> M_ReqVerified
				if (m_PrevStep == M_ReqReceived) {
					// nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_BAD_REQUEST,
								  "parse error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not reach a committee MVO at some point
			case M_UnavailMVO:
				// context: M_OB_Generated --> M_OB_Signed
				if (m_PrevStep == M_OB_Generated) {
					// nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "validator access error: " + m_FailureMsg);
				}
				// context: M_OB_Signed --> M_Completed (after a reply error)
				else if (m_PrevStep >= M_OB_Signed && m_CurrStep == M_Replied) {
					// no point in trying to reply again; do nothing
					ret = false;
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not reach an Auditor
			case M_UnavailAud:
				// context: M_ReqVerified --> M_InputKeys
				// or context: M_InputKeys --> M_NewKeys
				if (m_PrevStep == M_ReqVerified || m_PrevStep == M_InputKeys) {
					// failed to get eNFT keys; nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "key server error: " + m_FailureMsg);
				}
				// context: M_OB_Generated --> M_OB_Signed (lead or not)
				else if (m_PrevStep == M_OB_Generated) {
					// we could not send {OperationsBlock} to an Auditor
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "Auditor access error: " + m_FailureMsg);
				}
				// context: M_OB_Signed --> M_ReceiptGen
				else if (m_IsLead && m_PrevStep == M_OB_Signed) {
					// we could not get keys to generate receipts, must nack
					ret = nackDapp(rep, HttpServletResponse.SC_GATEWAY_TIMEOUT,
								   "receipt key server error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not access the appropriate blockchain
			case M_UnavailABI:
				// context: M_ReqReceived --> M_ReqVerified
				if (m_PrevStep == M_ReqReceived) {
					// couldn't check vs. SC config; nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_BAD_GATEWAY,
								  "blockchain access error: " + m_FailureMsg);
				}
				// context: M_Replied --> M_Spent
				else if (m_IsLead && m_PrevStep == M_Replied) {
					// we can't nack the request because we already replied
					return advanceState(M_Replied, m_FinalStep, reason,
										"could not verify spend on chain: "
										+ m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			case M_ProcError:
				/* Internal processing error, theoretically valid for any step.
				 * Unless we have some kind of coding error, these should not
				 * occur. To avoid the possibility of a recursive endless loop,
				 * we always bail with a nack when one of these crops up.
				 */
				String fail = "";
				int rCode = HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
				switch (m_PrevStep) {
					case M_ReqReceived:
						fail = "request processing error: " + m_FailureMsg;
						rCode = HttpServletResponse.SC_BAD_REQUEST;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_ReqVerified:
					//FALLTHROUGH
					case M_InputKeys:
						fail = "key server access problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_NewKeys:
						fail = "spend processing problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_OB_Generated:
						fail = "MVO committee access problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_OB_Signed:
						fail = "receipt generation problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_ReceiptGen:
						fail = "reply send problem: " + m_FailureMsg;
						// this will probably fail too of course:
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_Replied:
						fail = "blockchain spend confirmation problem: "
								+ m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;
					
					case M_Spent:
						fail = "receipt upload problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				m_Log.error(lbl + "processing error: " + fail);
				break;

			case M_ParseError:
				/* Partners can send us bogosity that didn't parse.  If they do,
				 * we have to bail because resending will probably generate the
				 * same bogosity again.  (This condition probably indicates a
				 * coding error in a partner server.)
				 */
				switch (m_PrevStep) {
					case M_ReqReceived:
					case M_ReqVerified:
					case M_InputKeys:
					case M_NewKeys:
					case M_OB_Generated:
					case M_OB_Signed:
					case M_ReceiptGen:
					case M_Replied:
					case M_Spent:
						m_Log.error(lbl + "parse error seen from partner, "
									+ m_FailureMsg);
						ret = nackReq(rep, mvoRep,
								HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
								"error processing partner reply, aborting: "
								+ m_FailureMsg);
						break;

					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				break;

			// partners can of course also send us error replies
			case M_GotNack:
				String nack = "";
				int nCode = HttpServletResponse.SC_BAD_GATEWAY;
				switch (m_PrevStep) {
					case M_ReqReceived:
						nack = "nack accessing blockchain config, "
								+ m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_ReqVerified:
					// FALLTHROUGH
					case M_InputKeys:
						nack = "nack obtaining eNFT keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_OB_Generated:
						nack = "nack from committee MVO, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_OB_Signed:
						nack = "nack obtaining receipt keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_Replied:
						nack = "nack accessing blockchain for spend conf, "
								+ m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;
					
					case M_Spent:
						nack = "nack uploading receipts, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					// no peer replies expected in these previous states
					case M_NewKeys:
					case M_ReceiptGen:
					case M_Completed:
					default:
						// should be impossible
						m_Log.error(lbl + "unexpected reply parse error in "
									+ "state " + m_CurrStep + ", "
									+ m_FailureMsg);
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				break;

			// an external store (such as for receipts) was unavailable
			case M_ExtUnavail:
				// context: M_OB_Signed --> M_ReceiptGen
				if (m_IsLead && m_PrevStep == M_OB_Signed) {
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_GATEWAY,
									"receipt storage error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			case M_NoFailure:
				// fall through below
				break;

			default:
				// should be impossible
				ret = inconsistentState(rep, mvoRep);
				break;
		}
		if (reason != M_NoFailure) return ret;

		// handle normal (non-error) cases
		ret = true;
		m_FailureMode = M_NoFailure;
		ClientRequest orgReq = m_Parent.getDappRequest();
		ClientMVOBlock clientReq = null;
		ArrayList<ClientMVOBlock.ClientPayee> payees = null;
		ArrayList<ClientMVOBlock.ClientInput> inputs = null;
		OperationsBlock opBlock = m_Parent.getOperationsBlock();
		SmartContractConfig scc = null;
		MVOConfig mvoConf = mvo.getConfig();
		final BigInteger one100 = new BigInteger("100");
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		final String chgMemo = "auto-generated change amount";
		switch (m_CurrStep) {
			case M_ReqVerified:
				/* Here we have successfully parsed and signature-verified the
				 * request from the client (iff lead) or lead MVO (if not).
				 * Our next step is to verify the parameters in the request vs.
				 * the SC config for the indicated blockchain.  If that's all
				 * okay, we need to do basic sanity-checking.  If any problems
				 * are found, we nack the request.
				 */
				if (m_Achieved < M_ReqReceived) {
					m_Log.error(lbl + "verify requested without valid "
								+ "request received");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				// get the request as passed by originator
				if (!(orgReq instanceof ClientMVOBlock)) {
					m_FailureMsg = "client request is not a ClientMVOBlock";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}
				clientReq = (ClientMVOBlock) orgReq;
				if (!clientReq.getOpcode().equals(clientReq.M_OpSpend)) {
					// we don't belong here
					m_FailureMsg = "client request is not a spend";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// check versus SC config
				scc = mvo.getSCConfig(clientReq.getChainId());
				if (scc == null) {
					// unknown chain
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " is not supported";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// confirm availability of JSON-RPC client
				if (!scc.isEnabled()) {
					// Web3j URL must be offline or not reachable
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " support is not currently available";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_UnavailABI,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// if we're not the lead, we must verify the lead's sig on OB
				if (!m_IsLead) {
					leadMVOId = m_Parent.getTagField(mvoRep, 1);
					MVOSignature leadSig
						= MVOSignature.findMVOSig(leadMVOId,
												  opBlock.getSignatures());
					if (leadSig == null) {
						m_FailureMsg = "lead MVO signature on OB not found";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// find the signingAddress for the lead MVO on this chain
					Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
					MVOConfig leadConf = (MVOConfig) mvoMap.get(leadMVOId);
					String signingAddress = "";
					BlockchainConfig bcConf
						= leadConf.getChainConfig(clientReq.getChainId());
					if (bcConf != null) {
						signingAddress = bcConf.getSigningAddress();
					}

					// validate lead MVO's signature using signingAddress
					String opData = opBlock.buildSignedData();
					String sigAddress
						= EncodingUtils.getDataSignerAddress(opData,
														leadSig.m_Signature);
					if (sigAddress.isEmpty()
						|| !sigAddress.equalsIgnoreCase(signingAddress))
					{
						m_FailureMsg = "lead MVO signature on OB invalid";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// confirm sender address is valid
				if (!WalletUtils.isValidAddress(clientReq.getSender())) {
					m_FailureMsg = "sender address (" + clientReq.getSender()
									+ ") appears invalid";
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				inputs = clientReq.getInputs();
				// ensure that we don't have more inputs than the SC allows
				if (inputs.size() > ClientMVOBlock.M_ARRAY_MAX) {
					m_FailureMsg = "too many input eNFTs (" + inputs.size()
									+ "), max is " + ClientMVOBlock.M_ARRAY_MAX;
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				payees = clientReq.getPayees();
				// ensure that we don't have more payees than the SC allows
				if (payees.size() > ClientMVOBlock.M_ARRAY_MAX) {
					m_FailureMsg = "too many output eNFTs (" + payees.size()
									+ "), max is " + ClientMVOBlock.M_ARRAY_MAX;
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// check asset type(s) is/are supported
				// NB: empty inputs or payees is disallowed by ClientMVOBlock

				/* Build a tally of the amounts of each asset in the inputs.
				 * Also make a list of the IDs of all input eNFTs.
				 */
				ArrayList<String> idList = new ArrayList<String>(inputs.size());
				m_InputAssets.clear();
				for (ClientMVOBlock.ClientInput reqInput : inputs) {
					String id = reqInput.m_eNFT.getID();
					// check for dup inputs
					if (idList.contains(id)) {
						m_FailureMsg = "input ID " + id + " included twice";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 	 m_FailureMsg);
						ret = false;
						return ret;
					}
					idList.add(id);
					String asst = reqInput.m_eNFT.getAsset();
					BigInteger inpAmt = reqInput.m_eNFT.getAmount();
					if (!m_InputAssets.containsKey(asst)) {
						// initialize with amount
						m_InputAssets.put(asst, inpAmt);
					}
					else {
						// add amount
						BigInteger amt = m_InputAssets.get(asst);
						BigInteger sum = amt.add(inpAmt);
						m_InputAssets.replace(asst, sum);
					}
				}

				/* Before proceeding any further, validate that all inputs are
				 * still circulating.  We could trust the signature of the
				 * issuing MVO on the passed eNFT input, but we cannot trust
				 * that it is indeed still circulating and hasn't been burned.
				 * (Particularly since that status could change during our
				 * processing of a transaction.)
				 */
				BlockchainAPI web3j = mvo.getWeb3(clientReq.getChainId());
				m_FetchedInputs.clear();
				Future<ArrayList<NFTmetadata>> nftFut
					= web3j.getNFTsById(scc.getChainId(),
										scc.getABI(),
										clientReq.getSender(),
										idList, false);
				try {
					m_FetchedInputs = nftFut.get();
				}
				catch (InterruptedException | ExecutionException
						| CancellationException e)
				{
					m_FailureMsg = "could not confirm input eNFTs on chain";
					m_Log.error(lbl + m_FailureMsg, e);
					advanceState(m_PrevStep, m_CurrStep, M_UnavailABI,
								 m_FailureMsg);
					ret = false;
					//scc.setEnabled(false);
					break;
				}
				if (m_FetchedInputs == null
					|| m_FetchedInputs.size() != inputs.size())
				{
					m_FailureMsg = "one or more input eNFTs not found on chain";
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// check for possible greylisting and insufficient dwell time
				EnftCache eCache = scc.getCache();
				if (eCache != null) {
					boolean greyListedInput = false;
					boolean unconfInput = false;
					BigInteger curBlock = eCache.getLatestBlock();
					EnshroudProtocol protoWrapper = eCache.getProtocolWrapper();
					for (String Id : idList) {
						// see if ID is in the greylisted list in cache
						BigInteger eId = Numeric.parsePaddedNumberHex(Id);
						if (eCache.isIdGreyListed(eId)) {
							m_FailureMsg = "greylisted input eNFT, ID " + Id;
							m_Log.error(lbl + m_FailureMsg);
							greyListedInput = true;
							break;
						}

						// see if ID is vested/confirmed at this time
						BigInteger unlockBlock = null;
						try {
							unlockBlock
								= protoWrapper.enftUnlockTime(eId).send();
						}
						catch (Exception eee) {
							m_Log.error(lbl + "unable to obtain enftUnlockTime "
										+ "for ID " + Id, eee);
						}
						if (unlockBlock == null
							|| unlockBlock.compareTo(curBlock) > 0)
						{
							m_FailureMsg = "unconfirmed input eNFT, ID " + Id;
							m_Log.error(lbl + m_FailureMsg);
							unconfInput = true;
							break;
						}
					}
					if (greyListedInput || unconfInput) {
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}
				// else: we have no cache to check against for this chain

				// NB: we save inputs for later use in the M_InputKeys step

				// build a tally of the amounts of each asset in the outputs
				boolean assetsOk = true;
				m_OutputAssets.clear();
				for (ClientMVOBlock.ClientPayee reqOutput : payees) {
					String asst
						= Keys.toChecksumAddress(reqOutput.m_OutputAsset);
					BigInteger inpTotal = m_InputAssets.get(asst);
					if (inpTotal == null) {
						// asset is not found in inputs
						assetsOk = false;
						m_FailureMsg = "output asset " + asst
										+ " not found in inputs";
						break;
					}

					// confirm that payee address is valid
					if (!reqOutput.m_Address.isEmpty()
						&& !WalletUtils.isValidAddress(reqOutput.m_Address))
					{
						assetsOk = false;
						m_FailureMsg = "payee address " + reqOutput.m_Address
										+ " appears invalid";
						break;
					}

					BigInteger outAmt = reqOutput.m_OutputAmount;
					// this could be an absolute or percentage amount
					if (reqOutput.m_Units.equals("%")) {
						// calculate percentage of input total
						BigInteger incr = inpTotal.multiply(outAmt);
						outAmt = incr.divide(one100);
					}
					// else: absolute amount
					if (!m_OutputAssets.containsKey(asst)) {
						// initialize with amount
						m_OutputAssets.put(asst, outAmt);
					}
					else {
						// add amount to running total
						BigInteger amt = m_OutputAssets.get(asst);
						BigInteger sum = amt.add(outAmt);
						m_OutputAssets.replace(asst, sum);
					}
				}

				// now check that all outputs are sufficiently funded by inputs
				m_ChangeOutputs.clear();
				if (assetsOk) {
					for (String outAsset : m_OutputAssets.keySet()) {
						BigInteger outAmt = m_OutputAssets.get(outAsset);
						BigInteger inpAmt = m_InputAssets.get(outAsset);
						int amtComp = inpAmt.compareTo(outAmt);
						if (amtComp > 0) {
							// supply catch-all eNFT back to sender for overage
							BigInteger diff = inpAmt.subtract(outAmt);
							m_ChangeOutputs.put(outAsset, diff);
						}
						else if (amtComp < 0) {
							// inputs are short
							assetsOk = false;
							m_FailureMsg = "output asset " + outAsset
										+ ": insufficient inputs to pay "
										+ outAmt;
							break;
						}
					}
				}
				if (!assetsOk) {
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// worked!
				recordSuccess(M_ReqVerified);

				// advance state to next step, which is fetching input keys
				ret = advanceState(m_CurrStep, M_InputKeys, M_NoFailure, "");
				break;

			case M_InputKeys:
				if (m_Achieved < M_ReqVerified) {
					m_Log.error(lbl + "input keys requested without "
								+ "request validation");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				/* Here we have successfully validated the request versus the
				 * relevant SC config, plus internal consistency, etc.
				 * First, we'll pick a random Auditor from our list.  If we
				 * don't have a connection to it, we'll open one (and do the
				 * AUTHREQ:AUTHREP dance, in which Auditor does REQ).
				 * We must then fetch the decryption keys for each eNFT found
				 * in the m_FetchedInputs list (which we retrieved earlier).
				 * These must then be decrypted and compared against the inputs
				 * provided in the client request.
				 */
				if (!m_ExpectingKeyServerReply) {
					String randAud2 = mvo.getRandomAuditor(null);
					if (randAud2.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get keys failure, no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// build the key block to send to the Auditor
					MVOKeyBlock inputKeyBlock = new MVOKeyBlock(m_Log);
					inputKeyBlock.setMapping(inputKeyBlock.M_MapENFT);
					inputKeyBlock.setOpcode(inputKeyBlock.M_OpGet);
					clientReq = (ClientMVOBlock) orgReq;
					// must supply the user's orig req
					inputKeyBlock.setClientBlockJson(
											clientReq.getDecryptedPayload());
					inputs = clientReq.getInputs();
					for (ClientMVOBlock.ClientInput reqInput : inputs) {
						// compute chainId+ID+address and add to new key request
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(clientReq.getChainId()));
						hashComps.add(reqInput.m_eNFT.getID());
						hashComps.add(clientReq.getSender().toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						inputKeyBlock.addHashToList(keyHash);
					}

					// sign the MVOKeyBlock
					MVOSignature ourSig2 = new MVOSignature();
					ourSig2.m_Signer = mvo.getMVOId();
					PrivateKey sgKey = mvoConf.getCommPrivKey();
					String signedData2 = inputKeyBlock.buildSignedData();
					boolean sgOk = true;
					if (inputKeyBlock.getErrCode() != 0) {
						sgOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sgKey, signedData2);
						if (sig == null) {
							sgOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							ourSig2.m_Signature = sig;
						}
					}
					if (!sgOk) {
						m_FailureMsg = "could not build key block for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					inputKeyBlock.setSignature(ourSig2);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud2,
													   inputKeyBlock))
						{
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud2 + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud2);
							randAud2 = mvo.getRandomAuditor(excludeAIds);
							if (randAud2.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg
							= "error looking up input eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// handling key server response
					m_ExpectingKeyServerReply = false;
					if (!m_KeyResponse.getStatus().equals(
													m_KeyResponse.M_Success))
					{
						m_FailureMsg
							= "error looking up input eNFT decryption keys";
						m_Log.error(lbl + "key lookup failure, Auditor "
									+ "says: \"" + m_KeyResponse.getStatus()
									+ "\"");
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify Auditor's signature
					String signedAudData2 = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature audSig2
						= m_KeyResponse.getSignature();
					boolean sgVerf = true;
					PublicKey audKey2 = mvoConf.getPeerPubkey(audSig2.m_Signer);
					if (audKey2 == null) {
						sgVerf = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ audSig2.m_Signer + ", cannot verify sig "
									+ "on returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(audKey2,
														   signedAudData2,
														   audSig2.m_Signature))
						{
							sgVerf = false;
							m_Log.error(lbl + "sig verify failed for Aud "
										+ audSig2.m_Signer + ", cannot verify "
										+ "sig on returned AuditorKeyBlock");
						}
					}
					if (!sgVerf) {
						m_FailureMsg
							= "error looking up input eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + audSig2.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// record response for later use
					m_InputKeyResponse = m_KeyResponse;
				}

				/* Now for each input we have, fetch the corresponding NFT
				 * and decrypt it with the fetched key.  By construction, these
				 * should all be in the same order (inputs, NFTs, AES keys).
				 */
				clientReq = (ClientMVOBlock) orgReq;
				inputs = clientReq.getInputs();
				boolean inputNFTsOk = true;
				int nftIdx = 0;
				ArrayList<String> hashParts = new ArrayList<String>(3);
				for (ClientMVOBlock.ClientInput reqInput : inputs) {
					// build the hash anew and find the key in the key response
					String id = reqInput.m_eNFT.getID();
					hashParts.add(Long.toString(clientReq.getChainId()));
					hashParts.add(id);
					hashParts.add(clientReq.getSender().toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashParts);
					String keyHash = EncodingUtils.sha3(keyIdx);
					hashParts.clear();

					// find the key in the list by looking up the hash
					String b64KeyData = "";
					AuditorKeyBlock.KeySpec keySpec
						= m_InputKeyResponse.getKeyForHash(keyHash);
					if (keySpec == null) {
						// use passed reqInput.m_AESkey if available
						if (reqInput.m_AESkey.isEmpty()) {
							inputNFTsOk = false;
							m_FailureMsg = "no key avail for input eNFT " + id;
							m_Log.error(lbl + m_FailureMsg);
							break;
						}
						else {
							b64KeyData = reqInput.m_AESkey;
							m_Log.debug(lbl + "fallback to dApp's key value "
										+ "for eNFT id " + id);
						}
					}
					else {
						// use what key server returned
						b64KeyData = keySpec.m_KeyData;
						// check it matches any value passed with dApp's request
						if (!reqInput.m_AESkey.isEmpty()) {
							if ( !b64KeyData.equals(reqInput.m_AESkey)) {
								m_Log.error(lbl + "returned key for eNFT " + id
											+ " does not match passed by dApp");
								// NB: attempt to use it to decrypt anyhow
							}
						}
					}

					// make key from this key data
					Base64.Decoder b64d = Base64.getUrlDecoder();
					byte[] keyData = b64d.decode(b64KeyData);
					SecretKey secretKey = null;
					// do not assume key is okay, as it could be user input
					try {
						secretKey = new SecretKeySpec(keyData, "AES");
					}
					catch (IllegalArgumentException iae) {
						m_Log.error(lbl + "bad AES key for eNFT at index "
									+ nftIdx, iae);
						m_FailureMsg = "bad decryption key for input " + id;
						inputNFTsOk = false;
						break;
					}

					// decrypt the eNFT using this key
					NFTmetadata nft = null;
					try {
						nft = m_FetchedInputs.get(nftIdx);
					}
					catch (IndexOutOfBoundsException ioobe) {
						m_Log.error(lbl + "no NFT at index " + nftIdx, ioobe);
						m_FailureMsg = "no eNFT avail for input " + id;
						inputNFTsOk = false;
						break;
					}
					if (nft == null) {
						m_Log.error(lbl + "no NFT at index " + nftIdx);
						m_FailureMsg = "no eNFT avail for input " + id;
						inputNFTsOk = false;
						break;
					}
					nftIdx++;
					eNFTmetadata eNft = nft.getProperties();
					String enshrouded = eNft.getEnshrouded();
					if (!eNft.isEncrypted() || enshrouded.isEmpty()) {
						m_Log.error(lbl + "weird, eNFT Id " + id
									+ " does not appear encrypted");
					}
					else {
						String decNFT = EncodingUtils.decWithAES(secretKey,
																 enshrouded);
						if (decNFT == null || decNFT.isEmpty()) {
							m_FailureMsg = "processing error decrypting eNFTs";
							inputNFTsOk = false;
							break;
						}
						String decMap = "{" + decNFT + "}";
						// parse this as JSON
						Object parsed = null;
						try {
							parsed = JSON.parse(decMap);
						}
						catch (IllegalStateException ise) {
							m_FailureMsg = "processing error parsing eNFTs";
							m_Log.error(lbl + "could not parse decrypted "
										+ "eNFT ID " + id, ise);
							inputNFTsOk = false;
							break;
						}
						if (parsed instanceof Map) {
							Map eNFTMap = (Map) parsed;
							if (!eNft.buildFromMap(eNFTMap)) {
								m_FailureMsg = "processing error parsing eNFTs";
								m_Log.error(lbl + "could not parse decrypted "
											+ "eNFT ID " + id);
								inputNFTsOk = false;
								break;
							}
						}
						else {
							m_FailureMsg = "processing error parsing eNFTs";
							m_Log.error(lbl + "could not parse decrypted "
										+ "eNFT ID " + id);
							inputNFTsOk = false;
							break;
						}
					}

					// get the signer and check their signature on the eNFT
					String signer = eNft.getSigner();
					// get their signing key for this blockchain
					scc = mvo.getSCConfig(clientReq.getChainId());
					MVOConfig leadConf
						= (MVOConfig) scc.getMVOMap().get(signer);
					if (leadConf == null) {
						m_Log.error(lbl + "cannot find config for "
									+ "eNFT MVO signer, " + signer);
						m_FailureMsg = "eNFT signature validation failure";
						inputNFTsOk = false;
						break;
					}
					BlockchainConfig leadChConf
						= leadConf.getChainConfig(clientReq.getChainId());
					if (leadChConf == null) {
						m_Log.error(lbl + "cannot find chain config"
									+ " for eNFT signer " + signer
									+ " on chainId " + clientReq.getChainId());
						m_FailureMsg = "eNFT signature validation failure";
						inputNFTsOk = true;
						break;
					}
					String leadSigAddr = leadChConf.getSigningAddress();

					// verify sig of leadSigAddr on eNft (allow for grandfather)
					if (!validateENFTsig(eNft, leadSigAddr)) {
						m_Log.error(lbl + "signature does not match on eNFT ID "
									+ eNft.getID());
						m_FailureMsg = "eNFT signature validation failure";
						inputNFTsOk = false;
						break;
					}

					/* Now that we have the decrypted input eNFT, we need to
					 * compare two fields to establish equality: the ID and
					 * the signature value.  If the signature is the same, then
					 * the hash is the same, hence no need to compare all parts.
					 */
					if (!id.equals(eNft.getID())
						|| !reqInput.m_eNFT.getSigner().equals(eNft.getSigner())
						|| !reqInput.m_eNFT.getSignature().equals(
														eNft.getSignature()))
					{
						// this input does not match!
						inputNFTsOk = false;
						m_FailureMsg = "input eNFT ID " + id
										+ " does not match blockchain copy!";
						m_Log.error(lbl + m_FailureMsg);
						break;
					}
				}
				if (!inputNFTsOk) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				/* NB: at the end of this loop, all NFTs in m_FetchedInputs
				 * 	   are now in decrypted form.
				 */

				if (m_IsLead) {
					/* Lastly, we must replicate all inputs into the
					 * {OperationsBlock}'s own list of inputs.
					 */
					for (ClientMVOBlock.ClientInput reqInput : inputs) {
						OperationsBlock.ClientInput obInput
							= opBlock.new ClientInput();
						obInput.m_Input = reqInput.m_Input.substring(5);
						obInput.m_ID = reqInput.m_eNFT.getID();
						// compute the details hash from values in the input
						obInput.m_DetailsHash = EncodingUtils.buildDetailsHash(
												reqInput.m_eNFT.getOwner(),
												obInput.m_ID,
												reqInput.m_eNFT.getAsset(),
									reqInput.m_eNFT.getAmount().toString(16),
												reqInput.m_eNFT.getRand());
						opBlock.addInput(obInput);
					}
				}
				recordSuccess(M_InputKeys);

				// advance state to next step, which is fetching output keys
				ret = advanceState(m_CurrStep, M_NewKeys, M_NoFailure, "");
				break;

			case M_NewKeys:
				if (m_Achieved < M_InputKeys) {
					m_Log.error(lbl + "new keys requested without input keys");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				/* Here we have successfully validated the request and inputs.
				 * First, we'll pick a random Auditor from our list.  If we
				 * don't have a connection to it, we'll open one (and do the
				 * AUTHREQ:AUTHREP dance, in which Auditor does REQ).
				 * If we're the lead MVO, we'll need to do a create request.
				 * Otherwise we do a fetch for the keys, as the lead will have
				 * created them already.  In both cases we do this by sending a
				 * MVOKeyBlock to the Auditor and waiting for the reply.
				 * Normally this is all asynchronous, but temporarily we'll be
				 * doing it synchronously using AuditorHandler.handleKeyBlock().
				 * If any problems occur, we must nack the request.
				 */
				if (!m_ExpectingKeyServerReply) {	// making the key request
					String randAud = mvo.getRandomAuditor(null);
					if (randAud.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get keys failure, no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// build the key block to send to the Auditor
					MVOKeyBlock keyBlock = new MVOKeyBlock(m_Log);
					keyBlock.setMapping(keyBlock.M_MapENFT);
					if (m_IsLead) {
						keyBlock.setOpcode(keyBlock.M_OpNew);
					}
					else {
						keyBlock.setOpcode(keyBlock.M_OpGet);
					}

					/* If we're lead, we must assign a new unique eNFT ID for 
					 * each output (checking to be sure no such ID currently
					 * exists), and compute: sha3(chainId+ID+address).  Add
					 * these to the map and request them from the Auditor.
					 * If we're not lead, we'll find the list of keys in the
					 * passed OperationsBlock from our committee lead, in
					 * (ArrayList):
					 * m_Parent.getOperationsBlock().getPayees().m_ID.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					// must supply the user's orig req
					keyBlock.setClientBlockJson(
											clientReq.getDecryptedPayload());
					payees = clientReq.getPayees();
					if (m_IsLead) {
						/* Loop through payees and copy data to the
						 * {OperationsBlock}, obtaining a random ID from the
						 * state object for each eNFT that will be required.
						 * Confirm that the key doesn't already exist (despite
						 * the low odds of a random collision generating 256-bit
						 * values!), and add it to the {MVOKeyBlock} we'll send
						 * to the Auditor.
						 */
						opBlock.setOpcode(clientReq.M_OpSpend);
						scc = mvo.getSCConfig(clientReq.getChainId());
						Base64.Encoder b64enp
							= Base64.getEncoder().withoutPadding();
						for (ClientMVOBlock.ClientPayee payee : payees) {
							String asset
								= Keys.toChecksumAddress(payee.m_OutputAsset);
							// ignore any improper negative amounts
							if (payee.m_OutputAmount.compareTo(BigInteger.ZERO)
								< 0)
							{
								m_Log.error(lbl + "ignoring illegal amount of "
											+ payee.m_OutputAmount
											+ " on eNFT for " + payee.m_Payee);
								// NB: this should cause a peer OB mismatch
								continue;
							}

							// make a payee for the OB
							OperationsBlock.ClientPayee obPayee
								= opBlock.new ClientPayee();
							// strip off "^payee" to leave just the sequence #
							obPayee.m_Payee = payee.m_Payee.substring(5);
							obPayee.m_Address = new String(payee.m_Address);
							/* NB: obPayee.m_OutputAmount is not emitted for a
							 * spend
							 */
							if (!payee.m_Units.equals("%")) {
								// absolute amount
								obPayee.m_OutputAmount = new BigInteger(
											payee.m_OutputAmount.toString());
							}
							else {
								// interpret as percentage of total asset amount
								BigInteger totOut = m_OutputAssets.get(asset);
								BigInteger incr
									= totOut.multiply(payee.m_OutputAmount);
								obPayee.m_OutputAmount = incr.divide(one100);
							}

							// pick an ID for the eNFT we'll create (unique)
							obPayee.m_ID
								= getUniqueNFTId(clientReq.getChainId());

							// copy any randomizer for the eNFT we'll create
							obPayee.m_Rand = payee.m_Rand;

							// compute the details hash
							obPayee.m_DetailsHash
								= EncodingUtils.buildDetailsHash(
															obPayee.m_Address,
															obPayee.m_ID,
															asset,
											obPayee.m_OutputAmount.toString(16),
															obPayee.m_Rand);
							opBlock.addPayee(obPayee);
						/*
							m_Log.debug(lbl + "added regular payee "
										+ obPayee.m_Payee
										+ " for " + obPayee.m_OutputAmount
										+ " of " + asset + ", ID "
										+ obPayee.m_ID + ", detHash = "
										+ obPayee.m_DetailsHash);
						 */
						}

						// add change outputs, if any, as appended payees
						if (!m_ChangeOutputs.isEmpty()) {
							int extPayee = payees.size() + 1;
							for (String changeAsset : m_ChangeOutputs.keySet())
							{
								// create additional payee
								OperationsBlock.ClientPayee chgPayee
									= opBlock.new ClientPayee();
								chgPayee.m_Payee = nf.format(extPayee++);
								chgPayee.m_OutputAmount = new BigInteger(
											m_ChangeOutputs.get(changeAsset)
											.toString());
								chgPayee.m_Address = clientReq.getSender();
								chgPayee.m_ID
									= getUniqueNFTId(clientReq.getChainId());
								// save the asset address for later use below
								chgPayee.m_Asset = changeAsset;

								// add randomizer value (we must gen this one)
								byte[] chgRand = new byte[M_RandomizerSize];
								stObj.getRNG().nextBytes(chgRand);
								byte[] chgRand64 = b64enp.encode(chgRand);
								chgPayee.m_Rand = new String(chgRand64,
														StandardCharsets.UTF_8);

								// set details hash
								chgPayee.m_DetailsHash
									= EncodingUtils.buildDetailsHash(
															chgPayee.m_Address,
															chgPayee.m_ID,
															chgPayee.m_Asset,
										chgPayee.m_OutputAmount.toString(16),
															chgPayee.m_Rand);

								opBlock.addPayee(chgPayee);
							/*
								m_Log.debug("Added change payee "
											+ chgPayee.m_Payee + " for "
											+ chgPayee.m_OutputAmount + " of "
											+ changeAsset + ", rand "
											+ chgPayee.m_Rand + ", ID "
											+ chgPayee.m_ID);
							 */
							}
						}

						// loop through all output payees, compute hash, get key
						ArrayList<OperationsBlock.ClientPayee> obPayees
							= opBlock.getPayees();
						for (OperationsBlock.ClientPayee payee : obPayees) {
							// compute chainId+ID+address and add to new key req
							ArrayList<String> hashComps
								= new ArrayList<String>(3);
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(payee.m_ID);
							hashComps.add(payee.m_Address.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							keyBlock.addHashToList(keyHash);
						}
					}
					else {	// not lead
						/* NB: in this case we'll verify the passed OB data, but
						 * we'll do it in the M_OB_Generated step, not here.
						 */
						boolean noId = false;
						// our passed OB should contain a complete payee list
						if (opBlock.getPayees().isEmpty()) {
							noId = true;
						}
						for (OperationsBlock.ClientPayee payee 
							 : opBlock.getPayees())
						{
							if (payee.m_ID.isEmpty()) {
								noId = true;
								continue;
							}

							// compute chainId+ID+address and fetch key
							ArrayList<String> hashComps
								= new ArrayList<String>(3);
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(payee.m_ID);
							hashComps.add(payee.m_Address.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							keyBlock.addHashToList(keyHash);
						}
						if (noId) {
							m_FailureMsg = "missing passed payee list in OB";
							m_Log.error(lbl + "missing ID in OB payees passed");
							advanceState(m_PrevStep, m_CurrStep, M_ParseError,
										 m_FailureMsg);
							ret = false;
							break;
						}
					}

					// sign the MVOKeyBlock
					MVOSignature ourSig = new MVOSignature();
					ourSig.m_Signer = mvo.getMVOId();
					PrivateKey sigKey = mvoConf.getCommPrivKey();
					String signedData = keyBlock.buildSignedData();
					boolean sigOk = true;
					if (keyBlock.getErrCode() != 0) {
						sigOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKey, signedData);
						if (sig == null) {
							sigOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							ourSig.m_Signature = sig;
						}
					}
					if (!sigOk) {
						m_FailureMsg = "could not build key block for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					keyBlock.setSignature(ourSig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud, keyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud);
							randAud = mvo.getRandomAuditor(excludeAIds);
							if (randAud.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg
							= "error looking up output eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// handling key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String signedAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature audSig
						= m_KeyResponse.getSignature();
					boolean sigVerf = true;
					PublicKey audKey = mvoConf.getPeerPubkey(audSig.m_Signer);
					if (audKey == null) {
						sigVerf = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ audSig.m_Signer
									+ ", cannot verify sig on returned "
									+ "AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(audKey,
														   signedAudData,
														   audSig.m_Signature))
						{
							sigVerf = false;
							m_Log.error(lbl + "sig verify failed for Aud "
										+ audSig.m_Signer + ", cannot verify "
										+ "sig on returned AuditorKeyBlock");
						}
					}
					if (!sigVerf) {
						m_FailureMsg = "error looking up eNFT encryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + audSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up eNFT encryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all keys, move to next state
				recordSuccess(M_NewKeys);
				ret = advanceState(m_CurrStep, M_OB_Generated, M_NoFailure, "");
				break;

			case M_OB_Generated:
				if (m_Achieved < M_NewKeys) {
					m_Log.error(lbl + "OB generation without eNFT keys");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				/* Here we have successfully obtained from an Auditor the keys
				 * we need for the involved eNFTs. Those for inputs are stored
				 * in m_InputKeyResponse while those for pending outputs are
				 * stored in m_KeyResponse.  The actual input eNFTs have been
				 * retrieved from the blockchain and stored in m_FetchedInputs.
				 * These have been decrypted and compared with the inputs sent
				 * to us by the client.  We now do one of two things:
				 * 1) If we're the lead MVO, we must generate and sign new eNFTs
				 * 	  using each key, and store them in m_Parent.m_LeadMVOBlock.
				 * 	  We then sign our generated OB, complete this step, and
				 * 	  transition to M_OB_SIGNED to deal with the committee.
				 * 2) If we're not the lead, we must decrypt the eNFTs found
				 * 	  in the m_LeadMVOBlock, and confirm that their details do
				 * 	  match the data in the {OperationsBlock}.  If they
				 * 	  don't, we nack.  If they do, we move to M_OB_SIGNED to
				 * 	  sign the original passed OB and return it to the lead MVO.
				 */
				// confirm we do indeed have the keys needed for OB processing
				ArrayList<AuditorKeyBlock.KeySpec> keyList
					= m_KeyResponse.getKeys();
				if (m_KeyResponse == null || keyList.isEmpty()) {
					m_Log.error(lbl + "no output eNFT keys found, abort");
					m_FailureMsg = "no eNFT keys found, cannot process OB";
					advanceState(m_PrevStep, m_CurrStep, M_ParseError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				clientReq = (ClientMVOBlock) orgReq;
				ArrayList<OperationsBlock.ClientPayee> outputs
					= opBlock.getPayees();
				scc = mvo.getSCConfig(clientReq.getChainId());
				payees = clientReq.getPayees();

				// check for lead
				if (m_IsLead) {
					// first get our signing key for this chain
					BlockchainConfig bConfig
						= mvoConf.getChainConfig(scc.getChainId());
					if (bConfig == null) {
						// shouldn't occur!
						m_Log.error(lbl + "no MVO config for chain "
									+ scc.getChainName() + ", cannot "
									+ "sign eNFTs for it");
						m_FailureMsg = "missing MVO signature key";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					BigInteger bcSigKey = bConfig.getSigningKey();

					// loop through all the outputs and create eNFT objects
					boolean gotNFTerr = false;
					for (OperationsBlock.ClientPayee outp : outputs) {
						// make outer NFT using defaults
						NFTmetadata nft = new NFTmetadata(m_Log);
						eNFTmetadata eNft = new eNFTmetadata(m_Log);

						// get the asset type for this payee output
						String asset = "";
						for (ClientMVOBlock.ClientPayee payee : payees) {
							// match on the payee sequence number
							if (payee.m_Payee.endsWith(outp.m_Payee)) {
								asset = Keys.toChecksumAddress(
														payee.m_OutputAsset);
								// also copy the memo line, if any
								if (!payee.m_Memo.isEmpty()) {
									// check for memo line too long
									if (payee.m_Memo.length()
										> eNFTmetadata.M_MEMO_MAX)
									{
										m_FailureMsg = "memo line too long, "
											+ "max " + eNFTmetadata.M_MEMO_MAX
											+ " characters";
										gotNFTerr = true;
									}
									else {
										eNft.setMemo(payee.m_Memo);
									}
								}
								break;
							}
						}
						if (asset.isEmpty()) {
							/* This must be a change payee.  Find the asset
							 * address in the OB's ClientPayee record itself.
							 */
							asset = Keys.toChecksumAddress(outp.m_Asset);
							eNft.setMemo(chgMemo);
						}

						// generate the other enshrouded properties
						eNft.setID(outp.m_ID);
						eNft.setOwner(outp.m_Address);
						eNft.setAsset(asset);
						eNft.setAmount(outp.m_OutputAmount.toString());
						eNft.setRand(outp.m_Rand);
						eNft.setSigner(mvo.getMVOId());

						/* cycle through inputs of this asset type to determine
						 * the generation, which is the minimum plus 1
						 */
						int minGen = 0;
						for (NFTmetadata iNft : m_FetchedInputs) {
							eNFTmetadata eNFT = iNft.getProperties();
							if (eNFT.getAsset().equalsIgnoreCase(asset)) {
								if (minGen == 0) {
									// first asset match
									minGen = eNFT.getGeneration();
								}
								else if (eNFT.getGeneration() < minGen) {
									minGen = eNFT.getGeneration();
								}
							}
						}
						minGen += 1;
						eNft.setGeneration(minGen);

						// TBD: if required, deal with expiration/growth/cost
					
						// attach encrypted properties to NFT
						nft.setProperties(eNft);

						// generate our signature on the eNFT
						String eNftSig = eNft.calcSignature(bcSigKey);
						if (eNftSig == null) {
							m_Log.error(lbl + "unable to sign new eNFT");
							m_FailureMsg = "error generating eNFT signature";
							gotNFTerr = true;
							break;
						}
						eNft.setSignature(eNftSig);

						/* encrypt the entire thing including the signature,
						 * using the AES key corresponding to the hash:
						 * chainId+ID+address
						 */
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(scc.getChainId()));
						hashComps.add(eNft.getID());
						hashComps.add(eNft.getOwner().toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						String useKey = null;
						AuditorKeyBlock.KeySpec keySpec
							= m_KeyResponse.getKeyForHash(keyHash);
						if (keySpec != null) {
							useKey = keySpec.m_KeyData;
						}
						if (useKey == null) {
							m_Log.error(lbl + "no AES key found for hash "
										+ keyHash + ", cannot "
										+ "encrypt eNFTs with it");
							m_FailureMsg = "missing eNFT encryption key";
							gotNFTerr = true;
							break;
						}

						// make key from this key data
						Base64.Decoder b64d = Base64.getUrlDecoder();
						byte[] keyData = b64d.decode(useKey);
						SecretKey secretKey = new SecretKeySpec(keyData, "AES");

						// encrypt the entire {enshrouded} value, including sig
						StringBuilder enshroudedData = new StringBuilder(512);
						eNft.addJSON(enshroudedData);
						String encData
							= EncodingUtils.encWithAES(secretKey,
													  enshroudedData.toString(),
													   stObj.getRNG());
						if (encData == null) {
							m_FailureMsg = "eNFT encryption failure";
							m_Log.error(lbl + "could not encrypt eNFT ID "
										+ eNft.getID()
										+ " with AES key, abort");
							gotNFTerr = true;
							break;
						}
						eNft.setEnshrouded(encData);
						eNft.setEncrypted(true);

						// record full NFT metadata (including enc {enshrouded})
						outp.m_Metadata = nft;
						StringBuilder encMetadata = new StringBuilder(1024);
						nft.addJSON(encMetadata);
						outp.m_EncMetadata = encMetadata.toString();

						// if we're doing local blockchain API, save the file
						//TEMPCODE (local dev testing only)
						BlockchainAPI web3Api = mvo.getWeb3(scc.getChainId());
						if (web3Api instanceof LocalBlockchainAPI) {
							// save the file
							LocalBlockchainAPI lbApi
								= (LocalBlockchainAPI) mvo.getWeb3(
															scc.getChainId());
							if (!lbApi.createENFTfile(scc.getChainId(),
													  eNft.getID(),
													  eNft.getOwner(),
													  outp.m_EncMetadata))
							{
								m_FailureMsg = "could not create eNFT file for "
												+ "ID " + eNft.getID();
								m_Log.error(lbl + m_FailureMsg);
								gotNFTerr = true;
								break;
							}
						}
						//TEMPCODE
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// this completes this step; record success and move on
					recordSuccess(M_OB_Generated);
					ret = advanceState(m_CurrStep, M_OB_Signed, M_NoFailure,
										"");
				}
				else {	// not lead
					/* Here we do not generate the OB, because it was passed to
					 * us completed and signed.  Instead, we only need to walk
					 * through it and verify that what the lead MVO constructed
					 * is copacetic with the dApp's original request (also
					 * forwarded to us, and its signature verified already).
					 * To do so we must decrypt the existing eNFTs shown in the
				 	 * m_LeadMVOBlock, and confirm that their details do match
				 	 * the output data in the {ClientMVOBlock}.  If they don't,
				 	 * we must nack.  If they do, we move to M_OB_SIGNED to
				 	 * sign the original passed OB and return it to lead MVO.
					 * We must also allow for any m_ChangeOutputs, which won't
					 * have an analog in the {ClientMVOBlock}.
					 */
					boolean gotNFTerr = false;
					ArrayList<ClientMVOBlock.ClientPayee> orgOutputs
						= clientReq.getPayees();

					// make certain every original output is found in the OB
					final String genErr = "eNFT data consistency error";
					for (ClientMVOBlock.ClientPayee orgOut : orgOutputs) {
						boolean present = false;
						for (OperationsBlock.ClientPayee output : outputs) {
							// strip off "payee"
							String ptag = orgOut.m_Payee.substring(5);
							if (ptag.equals(output.m_Payee)) {
								present = true;
								break;
							}
						}
						if (!present) {
							m_Log.error(lbl + "CRIT - client payee "
										+ orgOut.m_Payee + " not found in OB");
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// loop through all the outputs and validate eNFT objects
					ArrayList<OperationsBlock.ClientPayee> obPayees
						= opBlock.getPayees();
					for (OperationsBlock.ClientPayee outp : obPayees) {
						// get the original client request payee that matches
						ClientMVOBlock.ClientPayee orgOutput = null;
						for (ClientMVOBlock.ClientPayee orgPayee : orgOutputs) {
							if (orgPayee.m_Payee.equals("payee"+outp.m_Payee)) {
								orgOutput = orgPayee;
								break;
							}
						}
						if (orgOutput == null) {
							/* This could be a change payee.  In that case, the
							 * address MUST be the sender, and we must have at
							 * least one change amount in our list.
							 */
							if (!outp.m_Address.equalsIgnoreCase(
														clientReq.getSender())
								|| m_ChangeOutputs.isEmpty())
							{
								// we don't have an original payee like this
								m_Log.error(lbl + "CRIT - OB payee "
											+ outp.m_Payee + " not found in "
											+ "original client request!");
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}
						}

						// parse outer NFT with {enshrouded} data encrypted 
						NFTmetadata nft = new NFTmetadata(m_Log);
						if (!nft.buildFromJSON(outp.m_EncMetadata)) {
							m_FailureMsg = "output eNFT did not parse";
							m_Log.error(lbl + "could not parse eNFT in "
										+ "OB, bad metadata");
							gotNFTerr = true;
							break;
						}

						// obtain the {enshrouded} data
						eNFTmetadata metadata = nft.getProperties();
						if (metadata == null || !metadata.isEncrypted()) {
							m_FailureMsg = "output eNFT metadata not found";
							m_Log.error(lbl + "eNFT metadata not found "
										+ "or not encrypted");
							gotNFTerr = true;
							break;
						}

						/* decrypt the entire thing including the signature,
						 * using the AES key corresponding to the hash:
						 * chainId+ID+address
						 */
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(scc.getChainId()));
						hashComps.add(outp.m_ID);
						hashComps.add(outp.m_Address.toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						String useKey = null;
						AuditorKeyBlock.KeySpec aesSpec
							= m_KeyResponse.getKeyForHash(keyHash);
						if (aesSpec != null) {
							useKey = aesSpec.m_KeyData;
						}
						if (useKey == null) {
							m_Log.error(lbl + "no AES key found for hash "
										+ keyHash + ", cannot "
										+ "decrypt eNFTs with it");
							m_FailureMsg = "missing output eNFT decryption key";
							gotNFTerr = true;
							break;
						}

						// make key from this key data
						Base64.Decoder b64d = Base64.getUrlDecoder();
						byte[] keyData = b64d.decode(useKey);
						SecretKey secretKey = new SecretKeySpec(keyData, "AES");
						String encMeta = metadata.getEnshrouded();
						String decMeta = EncodingUtils.decWithAES(secretKey,
																  encMeta);
						if (decMeta == null) {
							m_Log.error(lbl + "error decrypting eNFT "
										+ "metadata for " + outp.m_ID);
							m_FailureMsg = "eNFT metdata decryption error";
							gotNFTerr = true;
							break;
						}
						eNFTmetadata eNFTdecMetadata = new eNFTmetadata(m_Log);
						String parseTxt = "{" + decMeta + "}";
						Object parseObj = null;
						try {
							parseObj = JSON.parse(parseTxt);
						}
						catch (IllegalStateException ise) { /* log below */ }
						if (!(parseObj instanceof Map)) {
							m_Log.error(lbl + "error parsing JSON: \""
										+ parseTxt + "\"");
							m_FailureMsg = "eNFT metdata parse error";
							gotNFTerr = true;
							break;
						}
						Map parseMap = (Map) parseObj;
						if (!eNFTdecMetadata.buildFromMap(parseMap)) {
							m_Log.error(lbl + "error parsing eNFT map");
							m_FailureMsg = "eNFT metdata parse error";
							gotNFTerr = true;
							break;
						}

						// get the signer and check their signature on the eNFT
						String signer = eNFTdecMetadata.getSigner();
						// this should equal lead MVO
						String lead = m_Parent.getTagField(mvoRep, 1);
						if (!signer.equals(lead)) {
							m_Log.error(lbl + "sig on OB is that of "
										+ signer + ", not lead MVO " + lead);
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						// get their signing key for this blockchain
						MVOConfig leadConf
							= (MVOConfig) scc.getMVOMap().get(signer);
						if (leadConf == null) {
							m_Log.error(lbl + "cannot find config for "
										+ "lead MVO signer, " + signer);
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						BlockchainConfig leadChConf
							= leadConf.getChainConfig(clientReq.getChainId());
						if (leadChConf == null) {
							m_Log.error(lbl + "cannot find chain config"
										+ " for eNFT signer " + signer
									+ " on chainId " + clientReq.getChainId());
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						String leadSigAddr = leadChConf.getSigningAddress();

						// verify sig of leadSigAddr on eNFT
						String sigAddr = eNFTdecMetadata.getSigAddr();
						if (!sigAddr.equalsIgnoreCase(leadSigAddr)) {
							m_Log.error(lbl + "signature does not match on ID "
										+ eNFTdecMetadata.getID());
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}

						/* We now have a decrypted version of the eNFT data.
						 * Compare each relevant bit against the output data,
						 * to ensure the lead MVO pulled no shenaningans.  We
						 * need to check against BOTH the values in the OB, and
						 * the original outputs specified in the signed client
						 * request which was forwarded.  If no client payee is
						 * available, we can check for a change output.
						 */
						String asset = eNFTdecMetadata.getAsset();

						// check ID value
						if (!eNFTdecMetadata.getID().equals(outp.m_ID)) {
							// falsifying the ID of the eNFT is very serious...
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "ID = " + eNFTdecMetadata.getID()
										+ " while OB has " + outp.m_ID);
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}

						// check payee address
						if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(
																outp.m_Address))
						{
							// falsifying the payee is also very serious...
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "payee " + eNFTdecMetadata.getOwner()
										+ " while OB has " + outp.m_Address);
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}

						// NB: asset is not emitted on a spend

						// NB: randomiser value is not emitted on a spend
						String eRand = eNFTdecMetadata.getRand();

						// check fields against client payee if there is one
						BigInteger orgAmt = null;
						if (orgOutput != null) {	// eNFT from client output
							// check address against the client's payee output
							if (!outp.m_Address.equalsIgnoreCase(
														orgOutput.m_Address))
							{
								m_Log.error(lbl + "CRIT - output eNFT with "
										+ "payee " + eNFTdecMetadata.getOwner()
											+ " while org client request has "
											+ orgOutput.m_Address);
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							// check asset
							if (!asset.equalsIgnoreCase(
													orgOutput.m_OutputAsset))
							{
								// changing asset is also very serious...
								m_Log.error(lbl + "CRIT - output eNFT with "
											+ "asset " + asset
											+ ", while org client request has "
											+ orgOutput.m_OutputAsset);
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							/* NB: we can't check eNFTdecMetadata.getAmount()
							 * versus the outp.m_OutputAmount, because the
							 * latter isn't included in the JSON output when
							 * the OB is passed to us, in a spend context.
							 * (This field is used only for withdrawals.)
							 */
							// determine amount from org request based on units
							orgAmt = orgOutput.m_OutputAmount;
							BigInteger orgInputTotal
								= clientReq.getAssetInputTotal(
													orgOutput.m_OutputAsset);
							if (orgOutput.m_Units.equals("%")) {
								// interpret as a percentage of total
								BigInteger
									incr = orgInputTotal.multiply(orgAmt);
								orgAmt = incr.divide(one100);
							}
							if (!orgAmt.equals(eNFTdecMetadata.getAmount())) {
								m_Log.error(lbl + "CRIT - output eNFT with "
											+ "amount "
										+ eNFTdecMetadata.getAmount().toString()
											+ " while org client request has "
											+ orgAmt.toString());
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							// check the randomizer value
							if (!orgOutput.m_Rand.equals(eRand)) {
								m_Log.error(lbl + "CRIT - output eNFT with "
											+ "randomizer " + eRand
											+ " while org client request has "
											+ orgOutput.m_Rand);
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							// verify the memo line, if present, has not changed
							String memo = orgOutput.m_Memo;
							if (!memo.isEmpty()) {
								if (!memo.equals(eNFTdecMetadata.getMemo())) {
									m_Log.error(lbl + "memo on eNFT, ID "
												+ eNFTdecMetadata.getID()
											+ " doesn't match org client req");
									m_FailureMsg = genErr;
									gotNFTerr = true;
									break;
								}
							}
						}
						else {	// eNFT for change / overage
							// record the asset type so we can use for receipts
							outp.m_Asset = asset;

							// validate that this eNFT is to be issued to sender
							if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(
														clientReq.getSender()))
							{
								m_Log.error(lbl + "CRIT - output change eNFT "
											+ "with payee "
											+ eNFTdecMetadata.getOwner()
											+ " does not match sender");
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							/* If this is a change eNFT (for the overage), it
							 * must correlate to an amount found in the list of
							 * change outputs.
							 */
							orgAmt = eNFTdecMetadata.getAmount();
							BigInteger changeAmt = m_ChangeOutputs.get(asset);
							if (changeAmt == null || !orgAmt.equals(changeAmt))
							{
								m_Log.error(lbl + "CRIT - output change eNFT "
											+ "seen for asset " + asset
											+ ", but no change value for that "
											+ "amount exists");
								m_FailureMsg = genErr;
								gotNFTerr = true;
								break;
							}

							// memo line should say change amount
							if (!chgMemo.equals(eNFTdecMetadata.getMemo())) {
								m_Log.error(lbl + "memo on eNFT, ID "
											+ eNFTdecMetadata.getID()
											+ " not as expected for change");
								// don't fail because of this, though
							}
						}

						// make sure the details hash computes correctly
						String hashStr = EncodingUtils.buildDetailsHash(
																outp.m_Address,
																outp.m_ID,
																asset,
															orgAmt.toString(16),
																eRand);
						// check it matches
						if (!hashStr.equals(outp.m_DetailsHash)) {
							m_Log.error(lbl + "CRIT - output " + outp.m_Payee
										+ " with ID " + outp.m_ID
										+ ", details hash doesn't match OB");
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}

						// compute details hash for output eNFT also, and check
						String eNFTHash = EncodingUtils.buildDetailsHash(
													eNFTdecMetadata.getOwner(),
													eNFTdecMetadata.getID(),
													eNFTdecMetadata.getAsset(),
									eNFTdecMetadata.getAmount().toString(16),
													eNFTdecMetadata.getRand());
						if (!eNFTHash.equals(hashStr)) {
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "ID " + eNFTdecMetadata.getID()
										+ ", details hash doesn't match OB");
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}

						/* cycle through inputs of this asset type to determine
						 * the generation, which is the minimum plus 1
						 */
						int minGen = 0;
						for (NFTmetadata iNft : m_FetchedInputs) {
							eNFTmetadata eNFT = iNft.getProperties();
							if (eNFT.getAsset().equalsIgnoreCase(asset)) {
								if (minGen == 0) {
									// first asset match
									minGen = eNFT.getGeneration();
								}
								else if (eNFT.getGeneration() < minGen) {
									minGen = eNFT.getGeneration();
								}
							}
						}
						minGen += 1;
						if (eNFTdecMetadata.getGeneration() != minGen) {
							m_Log.error(lbl + "illegal generation, ID "
										+ eNFTdecMetadata.getID() + ": "
										+ eNFTdecMetadata.getGeneration());
							m_FailureMsg = genErr;
							gotNFTerr = true;
							break;
						}

						// TBD: if required, deal with expiration/growth/cost
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// this completes this step; record success and move on
					recordSuccess(M_OB_Generated);
					ret = advanceState(m_CurrStep, M_OB_Signed, M_NoFailure,
										"");
				}
				break;

			case M_OB_Signed:
				if (m_Achieved < M_OB_Generated) {
					m_Log.error(lbl + "OB signature without OB generated");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				/* Here we have generated or verified the {OperationsBlock}.
				 * If we're lead, we must now pick a MVO committee, and set up
				 * to forward it to each member.  We'll complete this step only
				 * once all signed replies are seen.  We also send a CC to
				 * all known and connected Auditors as a MVOAuditorBlock.
				 *
				 * If we're not lead, we sign the original OB with our key and
				 * send it back, which completes this step for us.  We will
				 * likewise broadcast our signed OB to all Auditors, included in
				 * a MVOAuditorBlock message.
				 *
				 * As a pre-step, because the entire OB cannot be passed to the
				 * SC for verification (too large to parse), we must generate
				 * the OperationsBlock.ArgumentsHash for the argument data
				 * which will actually get passed to the SC by the dApp.  The
				 * lead MVO will set this in the OB; committee members merely
				 * verify it.  All MVOs will sign this hash data in addition to
				 * the complete OB (which itself includes the args hash value).
				 */
				clientReq = (ClientMVOBlock) orgReq;
				scc = mvo.getSCConfig(clientReq.getChainId());
				payees = clientReq.getPayees();

				// first get our signing key for this chain
				BlockchainConfig bConfig
					= mvoConf.getChainConfig(scc.getChainId());
				if (bConfig == null) {
					// shouldn't occur!
					m_Log.error(lbl + "no MVO config for chain "
								+ scc.getChainName() + ", cannot "
								+ "sign eNFTs for it");
					m_FailureMsg = "missing MVO signature keys";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				BigInteger bcSigKey = bConfig.getSigningKey();

				// sign the ArgumentsBlock passed by the dApp to the SC's API
				OperationsBlock.ArgumentsHash argsHash = opBlock.getArgsBlock();
				// NB: this arguments data is opcode-dependent
				String argData = argsHash.buildArgsData();
				if (argData == null) {
					m_Log.error(lbl + "could not build OB args hash data");
					m_FailureMsg = "could not build operations arguments hash";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				argsHash.m_ArgumentsData = argData;
				if (m_IsLead) {
					String hashValue = argsHash.argsHash();
					argsHash.m_ArgsHash = hashValue;
				}
				else {
					// double-check this is the data lead MVO actually signed
					if (!argsHash.m_ArgsHash.equals(argsHash.argsHash())) {
						m_Log.error(lbl + "could not verify OB args hash");
						m_FailureMsg
							= "could not verify operations arguments hash";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// validate the hash sig of lead MVO
					String leadArgSig = "";
					leadMVOId = m_Parent.getTagField(mvoRep, 1);
					MVOSignature leadSig = MVOSignature.findMVOSig(leadMVOId,
													opBlock.getSignatures());
					if (leadSig != null) {
						leadArgSig = leadSig.m_ArgsSig;
					}
					if (leadArgSig.isEmpty()) {
						m_FailureMsg = "no OB args signature for lead MVO";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// find the signingAddress for the lead MVO on this chain
					Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
					MVOConfig leadConf = (MVOConfig) mvoMap.get(leadMVOId);
					String signingAddress = "";
					BlockchainConfig leadBcConf
						= leadConf.getChainConfig(clientReq.getChainId());
					if (leadBcConf != null) {
						signingAddress = leadBcConf.getSigningAddress();
					}

					// verify this was the address which signed
					String sigAddress = EncodingUtils.getHashSignerAddress(
															argsHash.m_ArgsHash,
															leadArgSig);
					if (sigAddress.isEmpty()
						|| !sigAddress.equalsIgnoreCase(signingAddress))
					{
						m_FailureMsg = "lead MVO OB args signature failure";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// produce our signature on the arguments hash
				MVOSignature obSig = new MVOSignature();
				obSig.m_Signer = mvo.getMVOId();
				String digestSig = EncodingUtils.signHash(argsHash.m_ArgsHash,
														  bcSigKey);
				if (digestSig == null || digestSig.length() != 130) {
					m_FailureMsg = "error signing OB args hash";
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				obSig.m_ArgsSig = digestSig;

				// everything was okay, so sign the whole OB (lead or not)
				String hashableData = opBlock.buildSignedData();
				if (hashableData == null || hashableData.isEmpty()) {
					m_Log.error(lbl + "could not build OB signed data");
					m_FailureMsg = "could not sign operations block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				String ourSig = EncodingUtils.signData(hashableData, bcSigKey);
				if (ourSig == null) {
					m_Log.error(lbl + "could not sign OB data");
					m_FailureMsg = "could not sign operations block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				obSig.m_Signature = ourSig;
				// NB: if we're lead, this will be the first sig
				opBlock.addSignature(obSig);

				// build the MVOAuditorBlock message corresponding to our OB
				AuditorBlock audBlk = new AuditorBlock(m_Log);
				audBlk.setOpcode(clientReq.M_OpSpend);
				audBlk.setSender(clientReq.getSender());
				ArrayList<OperationsBlock.ClientPayee> obOutputs
						= opBlock.getPayees();

				// add all the outputs
				boolean gotNFTerr = false;
				for (OperationsBlock.ClientPayee outp : obOutputs) {
					AuditorBlock.ClientPayee audPayee
						= audBlk.new ClientPayee();
					audPayee.m_Payee = outp.m_Payee;
					audPayee.m_Address = outp.m_Address;
					audPayee.m_ID = outp.m_ID;
					audPayee.m_DetailsHash = outp.m_DetailsHash;
					audPayee.m_EncMetadata = outp.m_EncMetadata;

					// the NFT in an AuditorBlock must be constructed in clear
					audPayee.m_Metadata = new NFTmetadata(m_Log);
					// parse outer NFT with {enshrouded} data encrypted
					if (!audPayee.m_Metadata.buildFromJSON(
														audPayee.m_EncMetadata))
					{
						m_Log.error(lbl + "could not parse eNFT in OB for "
									+ "payee " + audPayee.m_Payee
									+ ", bad metadata");
						gotNFTerr = true;
						break;
					}

					// to get asset and amount, decrypt the {enshrouded} data
					eNFTmetadata realNFT = audPayee.m_Metadata.getProperties();
					if (realNFT == null || !realNFT.isEncrypted()) {
						m_Log.error(lbl + "eNFT metadata not found or "
									+ "not encrypted, payee " + outp.m_Payee);
						gotNFTerr = true;
						break;
					}
					eNFTmetadata eNft = new eNFTmetadata(m_Log);

					/* decrypt the entire thing including the signature,
					 * using the AES key corresponding to the hash:
					 * chainId+ID+address
					 */
					ArrayList<String> hashComps = new ArrayList<String>(3);
					hashComps.add(Long.toString(scc.getChainId()));
					hashComps.add(outp.m_ID);
					hashComps.add(outp.m_Address.toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashComps);
					String keyHash = EncodingUtils.sha3(keyIdx);
					String useKey = null;
					AuditorKeyBlock.KeySpec aesSpec
						= m_KeyResponse.getKeyForHash(keyHash);
					if (aesSpec != null) {
						useKey = aesSpec.m_KeyData;
					}
					if (useKey == null) {
						m_Log.error(lbl + "no AES key found for hash "
									+ keyHash + ", cannot "
									+ "decrypt eNFT with it");
						m_FailureMsg = "missing output eNFT decryption key";
						gotNFTerr = true;
						break;
					}

					// make key from this key data
					Base64.Decoder b64d = Base64.getUrlDecoder();
					byte[] keyData = b64d.decode(useKey);
					SecretKey secretKey = new SecretKeySpec(keyData, "AES");
					String encMeta = realNFT.getEnshrouded();
					String decMeta = EncodingUtils.decWithAES(secretKey,
															  encMeta);
					if (decMeta == null) {
						m_Log.error(lbl + "error decrypting eNFT "
									+ "metadata for " + outp.m_ID);
						m_FailureMsg = "eNFT metdata decryption error";
						gotNFTerr = true;
						break;
					}
					String parseTxt = "{" + decMeta + "}";
					Object parseObj = null;
					try {
						parseObj = JSON.parse(parseTxt);
					}
					catch (IllegalStateException ise) { /* log below */ }
					if (!(parseObj instanceof Map)) {
						m_Log.error(lbl + "error parsing JSON: \""
									+ parseTxt + "\"");
						m_FailureMsg = "eNFT metdata parse error";
						gotNFTerr = true;
						break;
					}
					Map parseMap = (Map) parseObj;
					if (!eNft.buildFromMap(parseMap)) {
						m_Log.error(lbl + "error parsing eNFT map");
						m_FailureMsg = "eNFT metdata parse error";
						gotNFTerr = true;
						break;
					}
					// NB: all fields of eNft should now be set properly
					audPayee.m_OutputAmount = eNft.getAmount();
					audPayee.m_OutputAsset = eNft.getAsset();
					audPayee.m_Rand = eNft.getRand();

					// recompute the details hash and compare as a check
					String hashStr = EncodingUtils.buildDetailsHash(
														audPayee.m_Address,
														audPayee.m_ID,
														audPayee.m_OutputAsset,
										audPayee.m_OutputAmount.toString(16),
														audPayee.m_Rand);
					if (!hashStr.equals(audPayee.m_DetailsHash)) {
						m_Log.error(lbl + "Auditor block payee "
									+ audPayee.m_Payee + " hash did not match");
						m_FailureMsg = "hash error building AuditorBlock";
						gotNFTerr = true;
						break;
					}
					audBlk.addPayee(audPayee);
				}
				if (gotNFTerr) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// add all the inputs
				ArrayList<OperationsBlock.ClientInput> obInputs
					= opBlock.getInputs();
				for (OperationsBlock.ClientInput clInp : obInputs) {
					AuditorBlock.ClientInput audInput
						= audBlk.new ClientInput();
					audInput.m_Input = clInp.m_Input;
					audInput.m_ID = clInp.m_ID;
					audInput.m_DetailsHash = clInp.m_DetailsHash;
					audBlk.addInput(audInput);
				}

				// construct the AB's ArgumentsHash and sign it
				AuditorBlock.ArgumentsHash abArgsHash = audBlk.getArgsBlock();
				// NB: this arguments data is opcode-dependent
				String abArgData = abArgsHash.buildArgsData();
				if (argData == null) {
					m_Log.error(lbl + "could not build AB args hash data");
					m_FailureMsg = "could not build Auditor arguments hash";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				abArgsHash.m_ArgumentsData = abArgData;
				abArgsHash.m_ArgsHash = abArgsHash.argsHash();

				// produce our signature on the arguments hash
				MVOSignature audBlSig = new MVOSignature();
				audBlSig.m_Signer = mvo.getMVOId();
				String abDigestSig
					= EncodingUtils.signHash(abArgsHash.m_ArgsHash, bcSigKey);
				if (digestSig == null || digestSig.length() != 130) {
					m_FailureMsg = "error signing AB args hash";
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				audBlSig.m_ArgsSig = abDigestSig;

				// sign the whole AuditorBlock with bcSigKey
				String audHashData = audBlk.buildSignedData();
				if (audHashData == null || audHashData.isEmpty()) {
					m_Log.error(lbl
								+ "could not build AuditorBlock signed data");
					m_FailureMsg = "could not sign Auditor block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				String aBlkSig = EncodingUtils.signData(audHashData, bcSigKey);
				if (aBlkSig == null) {
					m_Log.error(lbl + "could not sign AB data");
					m_FailureMsg = "could not sign Auditor block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				audBlSig.m_Signature = aBlkSig;
				audBlk.addSignature(audBlSig);

				// build the MVOAuditorBlock wrapper
				MVOAuditorBlock mvoAud = new MVOAuditorBlock(m_Log);
				mvoAud.setMVOBlock(audBlk);
				mvoAud.setClientBlock(clientReq);
				StringBuilder audJson = new StringBuilder(4096);
				audBlk.addJSON(audJson);
				mvoAud.setMVOBlockJson(audJson.toString());
				// append original client spend request text
				String clientJSON = clientReq.getDecryptedPayload();
				mvoAud.setClientBlockJson(clientJSON);

				ArrayList<MVOConfig> mvoCommittee = null;
				if (m_IsLead) {
					// select the MVO committee based on stakings etc.
					mvoCommittee
						= m_Parent.selectMVOCommittee(clientReq.getChainId());
					if (mvoCommittee == null
						|| mvoCommittee.size() < scc.getNumSigs()-1)
					{
						m_Log.error(lbl + "could not form committee of "
									+ (scc.getNumSigs()-1) + " MVOs for chain "
									+ scc.getChainName());
						m_FailureMsg = "could not form MVO committee";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					/* Blast the auditor report to all known auditors.  At least
					 * one must succeed, or we cannot proceed with the request.
					 */
					if (!m_Parent.broadcastToAuditors(mvoAud)) {
						m_Log.error(lbl + "CRIT - could not broadcast to any "
									+ "Auditors");
						m_FailureMsg = "no Auditors available, cannot proceed";
						ret = false;
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						break;
					}

					// send our data to the committee (asynchronously)
					if (!m_Parent.sendToMVOCommittee(mvoCommittee, opBlock)) {
						m_Log.error(lbl + "could not send to MVO "
									+ "committee, abort");
						m_FailureMsg = "could not send to MVO committee";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailMVO,
									 m_FailureMsg);
						ret = false;
						break;
					}
					/* NB:	the callback will advance state to M_ReceiptGen
					 * 		when the last committee response has been received.
					 */
				}
				else {	// not lead
					/* Blast the auditor report to all known auditors.  At least
					 * one Auditor must succeed, or we must nack the lead MVO.
					 */
					if (!m_Parent.broadcastToAuditors(mvoAud)) {
						m_Log.error(lbl + "CRIT - could not broadcast to any "
									+ "Auditors");
						m_FailureMsg = "no Auditors available, cannot proceed";
						ret = false;
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						break;
					}

					// go on to return the reply to the lead MVO
					recordSuccess(M_OB_Signed);
					ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				}
				break;

			case M_ReceiptGen:
				if (m_Achieved < M_OB_Signed) {
					m_Log.error(lbl + "receipt generation without OB signed");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				/* Here we have finished all OB verification and signing tasks,
				 * whether or not we were the lead in the transaction.  If we
				 * are the lead, then it also falls to us to build and sign all
				 * of the receipts required.  In the case of a spend, this
				 * means one receipt for each unique payee address in the list
				 * of outputs.  Always the spender gets one payer receipt.
				 *
				 * If we're not the lead, it's a state error to come here.
				 * However, we do not fail with an error, instead we'll just
				 * go to the next step.
				 */
				if (!m_IsLead) {
					// this is a state error, but do not return failure for it
					m_Log.error(lbl + "hit receipt generation step on "
								+ "!lead MVO, skipping to reply step");
					ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
					break;
				}
				if (!m_ExpectingKeyServerReply) {
					/* Generate receipts corresponding to outputs.  Also
					 * make a key generation request to an Auditor, and
					 * record the response in m_KeyResponse.  We can
					 * advance to the next step (for real) only once the
					 * Auditor replies to us successfully.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					ArrayList<OperationsBlock.ClientPayee> obPayees
						= opBlock.getPayees();

					// init a receipt for the spender (payer)
					ReceiptBlock spendReceipt = new ReceiptBlock(m_Log);
					spendReceipt.setReceiptId(stObj.getNextID());
					spendReceipt.setReceiptType(spendReceipt.M_Sender);
					spendReceipt.setSource(clientReq.getSender());
					spendReceipt.setChainId(clientReq.getChainId());
					m_Receipts.add(spendReceipt);

					// build a payee receipt for each payee in the OB
					int payeeIdx = 1;
					// map of payee addresses + asset combos we've seen:
					HashMap<String, ReceiptBlock.ReceiptPayee> seenPayees
						= new HashMap<String, ReceiptBlock.ReceiptPayee>();
					boolean rctsOk = true;

					for (OperationsBlock.ClientPayee payee : obPayees) {
						// allocate a receipt for the recipient
						ReceiptBlock receipt = new ReceiptBlock(m_Log);
						receipt.setReceiptType(receipt.M_Recipient);
						// pick a random ID for the receipt
						receipt.setReceiptId(stObj.getNextID());
						receipt.setSource(clientReq.getSender());
						receipt.setChainId(clientReq.getChainId());

						// make a single ReceiptPayee for the receiver
						ReceiptBlock.ReceiptPayee rPayee
							= receipt.new ReceiptPayee();

						// memo/asset found only in the original client req
						ArrayList<ClientMVOBlock.ClientPayee> orgPayees
							= clientReq.getPayees();
						String memo = "";
						String ast = "";
						for (ClientMVOBlock.ClientPayee orgPayee : orgPayees) {
							// find the matching payee output
							if (orgPayee.m_Payee.endsWith(payee.m_Payee)) {
								// copy the memo line
								memo = new String(orgPayee.m_Memo);
								// assets can vary for a spend
								ast = Keys.toChecksumAddress(
														orgPayee.m_OutputAsset);
								break;
							}
						}
						if (ast.isEmpty()) {
							// must be an output for change/overage
							if (payee.m_Asset.isEmpty()) {
								// this should be impossible
								m_FailureMsg
									= "cannot determine receipt asset type";
								m_Log.error(lbl + "receipt construction "
											+ "failure, " + m_FailureMsg);
								rctsOk = false;
								break;
							}
							else {
								ast = Keys.toChecksumAddress(payee.m_Asset);
								memo = new String(chgMemo);
							}
						}
						rPayee.m_Asset = new String(ast);
						rPayee.m_Memo = memo;
						// show payee only one payee (themselves)
						rPayee.m_Payee = new String("payee001");
						rPayee.m_Address
							= Keys.toChecksumAddress(payee.m_Address);
						rPayee.m_Amount = payee.m_OutputAmount;
						rPayee.m_ID = payee.m_ID;
						// set tag for sender receipt to first output ID
						if (spendReceipt.getTagID().isEmpty()) {
							spendReceipt.setTagID(payee.m_ID);
						}
						// set tag for receiver receipt to their issued ID
						receipt.setTagID(payee.m_ID);
						receipt.addPayee(rPayee);		// the only one
						m_Receipts.add(receipt);

						/* Now build a list element ReceiptPayee for the sender.
						 * We consolidate totals and memo lines into one, if
						 * there are multiple occurrences of a given address
						 * and asset type.
						 */
						ReceiptBlock.ReceiptPayee sPayee = null;
						String rctKey = payee.m_Address + ":" + ast;
						// use existing one if found
						ReceiptBlock.ReceiptPayee existPayee
							= seenPayees.get(rctKey);
						if (existPayee == null) {
							// make new one and add to map
							sPayee = spendReceipt.new ReceiptPayee();
							sPayee.m_Address
								= Keys.toChecksumAddress(payee.m_Address);
							sPayee.m_Asset = new String(ast);
							seenPayees.put(rctKey, sPayee);
						}
						else {
							sPayee = existPayee;
						}

						// re-use payee sequence number if required
						String payeeSeq = "payee" + nf.format(payeeIdx);
						if (sPayee.m_Payee.isEmpty()) {
							sPayee.m_Payee = new String(payeeSeq);
							payeeIdx++;
						}
						// else: leave it alone (i.e. use the existing value)

						// propagate the memo line
						if (sPayee.m_Memo.isEmpty()) {
							sPayee.m_Memo = memo;
						}
						else {
							// append
							sPayee.m_Memo += ("; " + memo);
						}
						// (NB: no ID is set for a sender receipt)

						// aggregate amount if required
						if (existPayee != null) {
							BigInteger newTot
								= sPayee.m_Amount.add(payee.m_OutputAmount);
							sPayee.m_Amount = newTot;
						}
						else {
							sPayee.m_Amount = payee.m_OutputAmount;
							spendReceipt.addPayee(sPayee);	// add to list
						}
					}
					if (!rctsOk) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					/* Now, for every unique receipt ID found in m_Receipts,
					 * request an AES key for it from the Auditor,
					 * using chain+ID+address as the hash key.  Send it
					 * asynchronously and advance when the callback sees
					 * the response, recorded in m_KeyResponse.
					 */
					// start by picking a random auditor
					String randAud3 = mvo.getRandomAuditor(null);
					if (randAud3.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get receipt keys failure, "
									+ "no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// now build the key block to send to the Auditor
					MVOKeyBlock rctKeyBlock = new MVOKeyBlock(m_Log);
					rctKeyBlock.setMapping(rctKeyBlock.M_MapReceipt);
					rctKeyBlock.setOpcode(rctKeyBlock.M_OpNew);

					// must supply the user's orig req
					String cliJSON = clientReq.getDecryptedPayload();
					rctKeyBlock.setClientBlockJson(cliJSON);

					// for each receipt, compute chainId+ID+address and get key
					for (ReceiptBlock rct : m_Receipts) {
						String rId = rct.getReceiptId();
						String addy = "";
						if (rct.getReceiptType().equals(rct.M_Sender)) {
							addy = rct.getSource();
						}
						else {
							ReceiptBlock.ReceiptPayee rPay
								= rct.getPayees().get(0);
							if (rPay == null) {
								m_Log.error(lbl + "receipt Id " + rId
										+ " has no ReceiptPayee record, skip");
								continue;
							}
							addy = rPay.m_Address;
						}
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(clientReq.getChainId()));
						hashComps.add(rId);
						hashComps.add(addy.toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						rctKeyBlock.addHashToList(keyHash);
					}

					// sign the MVOKeyBlock
					MVOSignature rctSig = new MVOSignature();
					rctSig.m_Signer = mvo.getMVOId();
					PrivateKey sigKy = mvoConf.getCommPrivKey();
					String sData = rctKeyBlock.buildSignedData();
					boolean sOk = true;
					if (rctKeyBlock.getErrCode() != 0) {
						sOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKy, sData);
						if (sig == null) {
							sOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							rctSig.m_Signature = sig;
						}
					}
					if (!sOk) {
						m_FailureMsg = "could not build key block for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					rctKeyBlock.setSignature(rctSig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud3, rctKeyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud3 + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud3);
							randAud3 = mvo.getRandomAuditor(excludeAIds);
							if (randAud3.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg
							= "error looking up receipt eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// processing key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String sigAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature adSig
						= m_KeyResponse.getSignature();
					boolean sigVer = true;
					PublicKey adKey = mvoConf.getPeerPubkey(adSig.m_Signer);
					if (adKey == null) {
						sigVer = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ adSig.m_Signer + ", cannot verify "
									+ "sig on returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(adKey,
														   sigAudData,
													   	   adSig.m_Signature))
						{
							sigVer = false;
							m_Log.error(lbl + "sig verify failed for "
										+ "Aud " + adSig.m_Signer
										+ ", cannot verify sig on returned "
										+ "AuditorKeyBlock");
						}
					}
					if (!sigVer) {
						m_FailureMsg = "error looking up receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + adSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all keys, move to next state
				/* (NB: we do not sign or encrypt receipts until we hit the
				 * M_Spent state)
				 */
				recordSuccess(M_ReceiptGen);
				ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				break;

			case M_Replied:
				/* Here we're ready to reply to the party who sent us this
				 * whole request.  As a lead, that's a dApp client.  Otherwise
				 * it's the lead MVO who picked us for their committee.
				 */
				if (!m_IsLead) {
					// affix our signature to the passed OB block JSON text
					PrivateKey ourKey = mvoConf.getCommPrivKey();
					StringBuilder opbJSON = new StringBuilder(2048);
					opBlock.addJSON(opbJSON);
					String fullTxt = opbJSON.toString();
					String ourL2Sig = EncodingUtils.signStr(ourKey, fullTxt);

					// the reply ID in the broker entry specifies lead MVO
					if (mvoRep == null || mvoRep.isEmpty()) {
						m_Log.error(lbl + "could not send reply to "
									+ "lead MVO (no tag), abort");
						m_FailureMsg = "no reply tag to send committee reply "
										+ "to lead";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					leadMVOId = m_Parent.getTagField(mvoRep, 1);

					// encrypt to this pubkey
					PublicKey leadKey = mvoConf.getPeerPubkey(leadMVOId);
					if (leadKey == null) {
						m_Log.error(lbl + "could not encrypt OB to "
									+ "lead MVO, no pubkey");
						m_FailureMsg = "no pubkey found for lead MVO "
										+ leadMVOId;
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					String encTxt
						= EncodingUtils.base64PubkeyEncStr(fullTxt, leadKey);

					// send it back to the lead MVO as one message
					String leadMsg = mvoRep + "::" + encTxt + "::" + ourL2Sig;
					PeerConnector peerConn = stObj.getPeerConnection(leadMVOId);
					if (peerConn == null) {
						// this is an error, because we're doing a reply
						m_Log.error(lbl + "no connection to lead MVO Id "
									+ leadMVOId);
						m_FailureMsg = "no lead MVO connection";
						ret = false;
						// cannot nack w/out connection, so just clean up
						advanceState(m_PrevStep, m_FinalStep,
									 M_UnavailMVO, m_FailureMsg);
						break;
					}

					// send as asynchronous write
					if (!peerConn.sendString(leadMsg, null)) {
						m_Log.error(lbl + "could not write reply to "
									+ "lead MVO Id " + leadMVOId);
						m_FailureMsg = "lead MVO reply failure";
						ret = false;
						// cannot nack w/out connection, so just clean up
						advanceState(m_PrevStep, m_FinalStep,
									 M_UnavailMVO, m_FailureMsg);
						break;
					}

					// as a non-lead MVO, we are now done, go to cleanup
					recordSuccess(M_Replied);
					ret = advanceState(m_CurrStep, m_FinalStep,
									   M_NoFailure, "");
				}
				else {
					/* Send our reply to the original client request.  This
					 * includes the OB we created, with all signatures of MVO
					 * committee members attached to it.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					HttpServletResponse httpRep
						= (HttpServletResponse) rep.getResponse();
					if (!cHandler.sendNormalRep(httpRep, opBlock,
												clientReq.getReplyKey()))
					{
						m_Log.error(lbl + "unable to send client "
									+ "reply after successful completion, "
									+ "transId = " + m_Parent.getTransId());
						rep.complete();
						ret = false;
						// skip over final 2 states to cleanup
						advanceState(m_CurrStep, m_FinalStep, M_ProcError,
									"client reply failure");
					}
					else {
						// normal progression to next step
						rep.complete();
						recordSuccess(M_Replied);
						ret = advanceState(m_CurrStep, M_Spent,
										   M_NoFailure, "");
					}
				}
				break;

			case M_Spent:
				/* Here we've sent our reply already.  In the lead MVO case
				 * (only), we need to know when the transaction has been mined
				 * on-chain, so that we can upload the transaction receipts we
				 * built in a previous step. To do this, we do have a list of
				 * eNFT IDs which the SC will only mint or burn once the trans
				 * has been performed and mined.  In particular, our m_Receipts
				 * list features tags which will prove that the transaction
				 * was finalized and minted.  We'll use the Web3j interface to
				 * watch for the mint or burn event(s), based on the list of
				 * tags found in ReceiptBlock.m_TagIDs for our receipts.
				 *
				 * To avoid stopping our thread, we'll make an async ABI request
				 * and set a timer which will come back here to register an
				 * error in case of timeout (i.e. transaction never mined).
				 *
				 * We also note the block number of the trans, and record it
				 * in the ReceiptBlock.m_BlockNumber field for each receipt.
				 * We then sign and encrypt each receipt, and pass to the queue.
				 *
				 * It's an error to arrive here if not the lead MVO.
				 */
				if (!m_IsLead) {
					m_Log.error(lbl + "non-lead MVO hit spend check");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				// must have gotten key response too
				if (m_KeyResponse == null || m_Achieved < M_ReceiptGen) {
					m_Log.error(lbl + "no Auditor key response but hit "
								+ "spend check state");
					ret = false;
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 "receipts not generated before check for "
								 + "transaction mined");
					break;
				}

				/* Save all receipts with the queueing manager, marking each
				 * as not yet being ready to upload.  Save the keys also.
				 * This is done so that the manager can recover and upload the
				 * receipts for us in the event that we exit in the interim.
				 */
				clientReq = (ClientMVOBlock) orgReq;
				scc = mvo.getSCConfig(clientReq.getChainId());
				EnftCache enCache = scc.getCache();
				BigInteger currBlock = enCache.getLatestBlock();
				ReceiptQueue rctUploader = m_Parent.getMVO().getReceiptQueue();
				boolean queueOk = true;
				for (ReceiptBlock rct : m_Receipts) {
					ReceiptQueue.ReceiptEntry qEntry
						= rctUploader.new ReceiptEntry();
					qEntry.m_ChainId = rct.getChainId();
					qEntry.m_Receipt = rct;
					qEntry.m_BlockWhenQueued = currBlock;

					// find the corresponding key
					String rId = rct.getReceiptId();
					String addy = "";
					if (rct.getReceiptType().equals(rct.M_Sender)) {
						addy = rct.getSource();
					}
					else {
						ReceiptBlock.ReceiptPayee rPay
							= rct.getPayees().get(0);
						if (rPay == null) {
							m_Log.error(lbl + "receipt Id " + rId
										+ " has no ReceiptPayee record, skip");
							queueOk = false;
							continue;
						}
						addy = rPay.m_Address;
					}
					ArrayList<String> hashComps = new ArrayList<String>(3);
					hashComps.add(Long.toString(rct.getChainId()));
					hashComps.add(rId);
					hashComps.add(addy.toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashComps);
					String keyHash = EncodingUtils.sha3(keyIdx);
					AuditorKeyBlock.KeySpec aesKey
						= m_KeyResponse.getKeyForHash(keyHash);
					if (aesKey == null) {
						m_Log.error(lbl + "no AES key found for rId "
									+ rId + ", cannot queue");
						queueOk = false;
						continue;
					}
					qEntry.m_AESkey = aesKey.m_KeyData;

					/* What we actually store in the queue is the raw receipt,
					 * including any tagID values.  It is not yet signed or
					 * encrypted using the indicated key.
					 */
					qEntry.updateJson();
					if (!rctUploader.queueReceipt(qEntry)) {
						m_Log.error(lbl + "cannot queue receipt, "
									+ qEntry.asText());
						queueOk = false;
						continue;
					}
				}
				if (!queueOk) {
					m_Log.error(lbl + "error queueing prepared but "
								+ "unsigned receipts");
					ret = false;
					advanceState(m_CurrStep, m_FinalStep, M_ProcError,
								"receipt upload failure");
					break;
				}

				recordSuccess(M_Spent);
				// go to cleanup
				ret = advanceState(m_CurrStep, M_Completed, M_NoFailure, "");
				break;

			case M_Completed:
				/* Here all of our tasks have been completed, and the last step
				 * is to clean up data structures, removing temporary data.
				 * Garbage collection and finalize() calls will auto-trigger
				 * the next time System.gc() runs.
				 */
				// if we're lead, remove ourselves from the broker map
				if (m_IsLead) {
					Map<String, MVOBrokerEntry> brokerMap
						= stObj.getBrokerMap();
					brokerMap.remove(m_Parent.getTransId(), m_Parent);
					m_Parent = null;
				}
				m_Receipts.clear();
				m_KeyResponse = null;
				m_InputKeyResponse = null;
				m_ChangeOutputs.clear();
				m_InputAssets.clear();
				m_OutputAssets.clear();
				m_FetchedInputs.clear();
				// NB: we do not need to close any connections or listeners
				break;

			default:
				// all other states don't make sense
				m_Log.error(lbl + "illegal state transition to " + m_CurrStep);
				ret = inconsistentState(rep, mvoRep);
				break;
		}
		return ret;
	}

	/**
	 * obtain whether we're the lead MVO on this request
	 * @return true if lead, false if committee member
	 */
	public boolean isLead() { return m_IsLead; }

	/**
	 * set whether we're the lead MVO on this request
	 * @param lead true if lead
	 */
	public void setLead(boolean lead) { m_IsLead = lead; }

	/**
	 * method to send pending error reply to dApp via ClientHandler
	 * @param resp the dApp response object
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackDapp(AsyncContext resp, int sc, String errTxt) {
		if (resp != null) {
			HttpServletResponse httpRep
				= (HttpServletResponse) resp.getResponse();
			m_Parent.getMVO().getClientHandler().sendErrRep(httpRep,
															sc, errTxt);
			resp.complete();
			recordSuccess(M_Replied);
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * method to send pending error reply to lead MVO via MVOHandler
	 * @param mvoRep the MVOId:transId:msgId needed to send a response to the
	 * lead MVO
	 * @param the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackMVO(String mvoRep, int sc, String errTxt) {
		if (mvoRep != null && !mvoRep.isEmpty()) {
			// do actual response write on WebSocket
			String leadMVOId = m_Parent.getTagField(mvoRep, 1);
			MVOState mvoState = m_Parent.getMVO().getStateObj();
			PeerConnector peerConn = mvoState.getPeerConnection(leadMVOId);
			if (peerConn == null) {
				// very odd since we got the committee request from the lead
				m_Log.error("nackMVO(): no return connection to lead MVO to "
							+ "send nack: " + errTxt);
				return false;
			}

			// send with asynchronous write, opening connection if necessary
			String errRep = mvoRep + "::ERR" + sc + "::" + errTxt;
			if (!peerConn.sendString(errRep, null)) {
				m_Log.error("nackMVO(): error writing nack to " + leadMVOId);
			}
			else {
				m_Log.debug("nackMVO(): sent err reply to " + leadMVOId + ": "
							+ errRep);
				recordSuccess(M_Replied);
			}
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * method to send a generic reply to the originator
	 * @param resp the dApp response object
	 * @param mvoRep the MVO reply Id
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackReq(AsyncContext resp, String mvoRep,
							int sc, String errTxt)
	{
		if (m_IsLead && resp != null) {
			// don't try to send a reply if we're past that
			if (m_Achieved >= M_Replied) {
				return false;
			}
			return nackDapp(resp, sc, errTxt);
		}
		else if (mvoRep != null) {
			return nackMVO(mvoRep, sc, errTxt);
		}
		m_Log.error("Unable to send nack!");
		return false;
	}

	/**
	 * Method to log when reason versus state is inconsistent.  Note this
	 * implies an internal programming error. The dApp client is nack'd.
	 * @param rep the dApp response object
	 * @param mvoRep the MVO reply Id
	 * @return true on success
	 */
	private boolean inconsistentState(AsyncContext rep, String mvoRep) {
		m_Log.error("SpendState internal inconsistency, step " + m_CurrStep
					+ ", reason " + m_FailureMode + ", emsg \"" + m_FailureMsg
					+ "\"");
		return nackReq(rep, mvoRep,
					   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
					   "Internal state error, abort");
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_ChangeOutputs != null) {
				m_ChangeOutputs.clear();
			}
			if (m_OutputAssets != null) {
				m_OutputAssets.clear();
			}
			if (m_InputAssets != null) {
				m_InputAssets.clear();
			}
			if (m_FetchedInputs != null) {
				m_FetchedInputs.clear();
			}
			m_InputKeyResponse = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
