/*
 * last modified---
 * 	10-27-25 record m_SubStartBlock and throw away events from earlier blocks
 * 	10-11-25 implement adding a new MVO via MVOStatusChanged event handling;
 * 			 use Log object from SmartContractConfig not MVO, since per-chain
 * 	03-18-24 start from EnftCache.m_LastProcessedBlock; pull Web3j from m_SCC
 * 	03-15-24 add shutdown()
 * 	03-12-24 new
 *
 * purpose---
 * 	implement a listener for MVOStakeUpdated and MVOStatusChanged events from
 * 	the MVOStaking contract
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.wrappers.MVOStaking;

import org.web3j.protocol.Web3j;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.Keys;
import org.web3j.crypto.WalletUtils;
import org.web3j.tx.gas.DefaultGasProvider;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.protocol.core.DefaultBlockParameterNumber;
import io.reactivex.Flowable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Action;

import java.util.Hashtable;
import java.math.BigInteger;
import java.io.IOException;
import java.net.URI;


/**
 * Provide a listener (one per chain) which can update our existing MVOStaking
 * record data when an event changing that data is seen on the blockchain.
 * This class should be initialized immediately after all the current records
 * are downloaded via BlockchainConnectListener.getOtherNodeConfigs().
 */
public final class MVOStakingListener {
	// BEGIN data members
	
	/**
	 * the owning MVO
	 */
	private MVO					m_MVO;

	/**
	 * the smart contract config which we serve
	 */
	private SmartContractConfig	m_SCC;

	/**
	 * credentials to use for our read-only Web3j access
	 */
	private Credentials 		m_Credentials;

	/**
	 * the initialized (loaded) MVOStaking wrapper
	 */
	private MVOStaking			m_Wrapper;

	/**
	 * convenience logging object (copy of m_SCC's)
	 */
	private Log					m_Log;

	/**
	 * the starting block for our event subscriptions
	 */
	private BigInteger			m_SubStartBlock;

	/**
	 * inner class to supply callbacks for events and errors on MVOStakeUpdated
	 * events
	 */
	private final class MVOStakeUpdatedHandler implements
		Consumer<MVOStaking.MVOStakeUpdatedEventResponse>, Action
	{
		/**
		 * method to consume events retrieved from m_MVOStakeUpdatedSub
		 * @param event the data from the event log
		 */
		public void accept(MVOStaking.MVOStakeUpdatedEventResponse event)
			throws Exception
		{
			final String lbl = this.getClass().getSimpleName() + ".accept: ";
			final String mvoId = event.mvoID;
			BigInteger evBlock = event.log.getBlockNumber();
			final BigInteger staked = event.stakingQuantum;

			/* Check that our block source is not suddenly handing us old stale
			 * events from blocks prior to our subscription start.
			 */
			if (evBlock.compareTo(m_SubStartBlock) < 0) {
				m_Log.debug(lbl + "discarding event from block " + evBlock
							+ " because it's before sub start block of "
							+ m_SubStartBlock);
				return;
			}
			m_Log.debug(lbl + "got MVOStakeUpdated event in block " + evBlock
						+ ": mvoId = " + mvoId + ", new quanta = " + staked);

			// find the BlockchainConfig record for this MVO on this chain
			Hashtable<String, MVOGenConfig> mvoMap = m_SCC.getMVOMap();
			MVOGenConfig mvoConf = mvoMap.get(event.mvoID);
			if (!(mvoConf instanceof MVOConfig)) {
				m_Log.error(lbl + "no MVO config for mvoId " + mvoId
							+ ", cannot process staking update to "
							+ staked + " on chain " + m_SCC.getChainName());
				return;
			}
			MVOConfig mConf = (MVOConfig) mvoConf;
			BlockchainConfig mbc = mConf.getChainConfig(m_SCC.getChainId());
			if (mbc == null) {
				m_Log.error(lbl + "no BC config for mvoId " + mvoId
							+ " on chain " + m_SCC.getChainName()
							+ ", cannot process MVOStakeUpdated event");
				return;
			}

			// get the old value, compute difference, and update value
			BigInteger oldStake = mbc.getStaking();
			BigInteger stakeDiff = staked.subtract(oldStake);
			m_Log.debug(lbl + "mvoId " + mvoId + " staking diff = "
						+ stakeDiff);
			mbc.setStaking(staked.toString());

			// if active, adjust total staked on this chain, based on difference
			if (mConf.getStatus()) {
				BigInteger oldStakingTot = m_SCC.getTotalStaking();
				BigInteger newStakingTot = oldStakingTot.add(stakeDiff);
				m_SCC.setTotalStaking(newStakingTot.toString());
				m_Log.debug(lbl + "total MVO staking on " + m_SCC.getChainName()
							+ " now = " + newStakingTot);
			}
		}

		/**
		 * method to consume errors occurring on m_MVOStakeUpdatedSub
		 * -- reconnects to publisher to pick up from where the error occurred
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on MVOStakeUpdated subscription: "
						+ err.toString());

			// cancel any previous subscription now voided by exception
			if (m_MVOStakeUpdatedSub != null
				&& !m_MVOStakeUpdatedSub.isDisposed())
			{
				try {
					m_MVOStakeUpdatedSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}

			// re-init wrapper, using current Web3j obj
			String mvoStakingAddr = MVOStaking.getPreviouslyDeployedAddress(
											Long.toString(m_SCC.getChainId()));
			m_Wrapper = MVOStaking.load(mvoStakingAddr, m_SCC.getABI(),
										m_Credentials,
										new DefaultGasProvider());

			// the only other thing we have to do is restart subscription
			m_SubStartBlock = m_SCC.getCache().getLastProcessedBlock();
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(m_SubStartBlock);

			// request a filter for all subsequent MVOStakeUpdated events
			Flowable<MVOStaking.MVOStakeUpdatedEventResponse> mvoUpdPub
				= m_Wrapper.mVOStakeUpdatedEventFlowable(startBlock,
											DefaultBlockParameterName.LATEST);
			// use form with callback and error handler
			m_MVOStakeUpdatedSub = mvoUpdPub.subscribe(m_MVOStakeUpdatedHandler,
					throwable -> m_MVOStakeUpdatedHandler.onError(throwable));
			m_Log.debug(lbl + "restarted sub for MVOStakeUpdated events from "
						+ "block " + m_SubStartBlock);
		}

		/**
		 * method required to handle completion action after last event seen
		 * (will never be called with a 2-argument subscription stipulating
		 * LATEST block, non-terminating)
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			final String lbl = this.getClass().getSimpleName() + ".run: ";
			m_Log.error(lbl + "unexpected invocation on non-terminating sub");
		}

		/**
		 * nullary constructor
		 */
		public MVOStakeUpdatedHandler() { }
	}

	/**
	 * handler for MVOStakeUpdated events
	 */
	private MVOStakeUpdatedHandler	m_MVOStakeUpdatedHandler;

	/**
	 * our subscription to MVOStakeUpdated events (one open at a time)
	 */
	private Disposable				m_MVOStakeUpdatedSub;

	/**
	 * inner class to supply callbacks for events and errors on MVOStatusChanged
	 * events
	 */
	private final class MVOStatusChangedHandler implements
		Consumer<MVOStaking.MVOStatusChangedEventResponse>, Action
	{
		/**
		 * method to consume events retrieved from m_MVOStatusChangedSub
		 * @param event the data from the event log
		 */
		public void accept(MVOStaking.MVOStatusChangedEventResponse event)
			throws Exception
		{
			final String lbl = this.getClass().getSimpleName() + ".accept: ";

			final String mvoId = event.mvoID;
			final MVOStaking.MVO mvoRec = event.updatedMVO;
			final BigInteger chainId = event.chainId;
			BigInteger evBlock = event.log.getBlockNumber();

			/* Check that our block source is not suddenly handing us old stale
			 * events from blocks prior to our subscription start.
			 */
			if (evBlock.compareTo(m_SubStartBlock) <= 0) {
				m_Log.debug(lbl + "discarding event from block " + evBlock
							+ " because it's before sub start block of "
							+ m_SubStartBlock);
				return;
			}
			m_Log.debug(lbl + "got MVOStatusChanged event in block " + evBlock
						+ " on chainId " + chainId + ": mvoId = " + mvoId);

			if (chainId.longValue() != m_SCC.getChainId()) {
				// this should be impossible
				m_Log.error(lbl + "MVOStatusChanged event for chain Id "
							+ chainId + " seen on callback for chain "
							+ m_SCC.getChainId());
				return;
			}

			BigInteger oldStakingTot = m_SCC.getTotalStaking();
			Hashtable<String, MVOGenConfig> mvoMap = m_SCC.getMVOMap();
			MVOGenConfig mvoConf = mvoMap.get(mvoId);
			MVOConfig mConf = null;
			BlockchainConfig mbc = null;
			if (!(mvoConf instanceof MVOConfig)) {
				// create a record to store data about this MVO
				mConf = new MVOConfig(mvoId, m_Log);
				mConf.setStatus(mvoRec.active.booleanValue());
				mConf.addBlockchainConfig(chainId.longValue(),
										  m_SCC.getChainName());
				mbc = mConf.getChainConfig(m_SCC.getChainId());
				mbc.configSigningAddress(Keys.toChecksumAddress(
														mvoRec.signingAddress));
				mbc.configStakingAddress(Keys.toChecksumAddress(
														mvoRec.stakingAddress));
				mbc.setStaking(mvoRec.stakingQuantum.toString());
				BigInteger mvoStake = mbc.getStaking();
				URI calcURI = m_MVO.computePeerURI(mvoId, false);
				mConf.setMVOURI(calcURI);
				// insert this record into map for this chain
				mvoMap.put(mvoId, mConf);

				// if active and non-zero, add stake into total chain stakings
				if (mConf.getStatus()
					&& mvoStake.compareTo(BigInteger.ZERO) > 0)
				{
					BigInteger newStakingTot = oldStakingTot.add(mvoStake);
					m_SCC.setTotalStaking(newStakingTot.toString());
					m_Log.debug(lbl + "total MVO staking increased to: "
								+ m_SCC.getTotalStaking().toString()
								+ " on chain " + m_SCC.getChainName());
				}
				m_Log.debug(lbl + "recorded new MVO Id = " + mvoId + " with "
							+ "stakingAddr = " + mbc.getStakingAddress()
							+ ", signingAddr = " + mbc.getSigningAddress()
							+ ", user URL = " + mvoRec.listenURI
							+ ", staking = " + mbc.getStaking().toString()
							+ ", active = " + mConf.getStatus());
				// no further processing required
				return;
			}

			// change to an existing MVO configured at startup
			mConf = (MVOConfig) mvoConf;
			// find the BlockchainConfig record for this MVO on this chain
			mbc = mConf.getChainConfig(m_SCC.getChainId());
			if (mbc == null) {
				m_Log.error(lbl + "no BC config for mvoId " + mvoId
							+ " on chain " + m_SCC.getChainName()
							+ ", cannot process MVOStatusChanged event");
				return;
			}

			/* These are the only fields of the MVOStaking.MVO class which we
			 * care about: stakingAddress, signingAddress, stakingQuantum,
			 * and active.  In each case we need to determine whether this is
			 * a change since we downloaded all the current MVO records during
			 * initialization (triggered by a call to
			 * RemoteBlockchainAPI.getConfig() made in
			 * BlockchainConnectListener.getOtherNodeConfigs())
			 * and deal with any change appropriately.
			 */
			// check for update to signing address
			String newSigning = Keys.toChecksumAddress(mvoRec.signingAddress);
			if (WalletUtils.isValidAddress(newSigning)
				&& !newSigning.equals(mbc.getSigningAddress()))
			{
				// record update
				m_Log.debug(lbl + "changed mvoId " + mvoId + " signing address "
							+ "from: " + mbc.getSigningAddress()
							+ "to: " + newSigning);
				mbc.configSigningAddress(newSigning);
			}

			// check for update to staking address
			String newStaking = Keys.toChecksumAddress(mvoRec.stakingAddress);
			if (WalletUtils.isValidAddress(newStaking)
				&& !newStaking.equals(mbc.getStakingAddress()))
			{
				// record update
				m_Log.debug(lbl + "changed mvoId " + mvoId + " staking address "
							+ "from: " + mbc.getStakingAddress()
							+ "to: " + newStaking);
				mbc.configStakingAddress(newStaking);
			}

			// see whether active setting changed
			boolean isActive = mvoRec.active.booleanValue();
			boolean wasActive = mConf.getStatus();
			if (isActive != wasActive) {
				m_Log.debug(lbl + "mvoId " + mvoId + " changed status from "
							+ wasActive + " to " + isActive);
				mConf.setStatus(isActive);
			}

			// obtain old (current) staking quantum
			BigInteger oldStake = mbc.getStaking();
			if (oldStake.compareTo(mvoRec.stakingQuantum) != 0) {
				// compute difference, and update value
				BigInteger stakeDiff = mvoRec.stakingQuantum.subtract(oldStake);
				m_Log.debug(lbl + "mvoId " + mvoId + " staking diff = "
							+ stakeDiff);
				mbc.setStaking(mvoRec.stakingQuantum.toString());

				if (wasActive) {
					if (isActive) {
						// adjust total staked on this chain, based on diff
						BigInteger newStakingTot = oldStakingTot.add(stakeDiff);
						m_SCC.setTotalStaking(newStakingTot.toString());
						m_Log.debug(lbl + "total MVO staking on "
									+ m_SCC.getChainName() + " now = "
									+ newStakingTot + " after staking quantum "
									+ "for mvoId " + mvoId + " changed by "
									+ stakeDiff);
					}
					else {
						// subtract entire original staking amount from total
						BigInteger newStakingTot
							= oldStakingTot.subtract(oldStake);
						m_SCC.setTotalStaking(newStakingTot.toString());
						m_Log.debug(lbl + "total MVO staking on "
									+ m_SCC.getChainName() + " now reduced to "
									+ newStakingTot + " from " + oldStakingTot
									+ " due to deactivation of mvoId " + mvoId);
					}
				}
				else {
					if (isActive) {
						// add entire new staking amount to chain total
						BigInteger newStakingTot
							= oldStakingTot.add(mvoRec.stakingQuantum);
						m_SCC.setTotalStaking(newStakingTot.toString());
						m_Log.debug(lbl + "total MVO staking on "
									+ m_SCC.getChainName() + " increased to "
									+ newStakingTot + " from " + oldStakingTot
									+ " due to activation of mvoId " + mvoId);
					}
					// else: nothing to do
				}
			}
			else {	// staking amount did not change
				// see whether active setting changed
				if (isActive && !wasActive) {
					// add entire new staking amount to chain total
					BigInteger newStakingTot
						= oldStakingTot.add(mvoRec.stakingQuantum);
					m_SCC.setTotalStaking(newStakingTot.toString());
					m_Log.debug(lbl + "total MVO staking on "
								+ m_SCC.getChainName() + " increased from "
								+ oldStakingTot + " to " + newStakingTot
								+ " due to activation of mvoId " + mvoId);
				}
				else if (!isActive && wasActive) {
					// subtract entire original staking amount from total
					BigInteger newStakingTot
						= oldStakingTot.subtract(oldStake);
					m_SCC.setTotalStaking(newStakingTot.toString());
					m_Log.debug(lbl + "total MVO staking on "
								+ m_SCC.getChainName() + " now reduced from "
								+ oldStakingTot + " to " + newStakingTot
								+ " due to deactivation of mvoId " + mvoId);
				}
				// else: nothing to do
			}
		}

		/**
		 * method to consume errors occurring on m_MVOStatusChangedSub
		 * -- reconnects to publisher to pick up from where the error occurred
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on MVOStatusChanged subscription: "
						+ err.toString());

			// cancel any previous subscription now voided by exception
			if (m_MVOStatusChangedSub != null
				&& !m_MVOStatusChangedSub.isDisposed())
			{
				try {
					m_MVOStatusChangedSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}

			// re-init wrapper, using current Web3j obj
			String mvoStakingAddr = MVOStaking.getPreviouslyDeployedAddress(
											Long.toString(m_SCC.getChainId()));
			m_Wrapper = MVOStaking.load(mvoStakingAddr, m_SCC.getABI(),
										m_Credentials,
										new DefaultGasProvider());

			// the only other thing we have to do is restart subscription
			m_SubStartBlock = m_SCC.getCache().getLastProcessedBlock();
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(m_SubStartBlock);

			// request a filter for all subsequent MVOStatusChanged events
			Flowable<MVOStaking.MVOStatusChangedEventResponse> mvoStatUpd
				= m_Wrapper.mVOStatusChangedEventFlowable(startBlock,
											DefaultBlockParameterName.LATEST);
			// use form with callback and error handler
			m_MVOStatusChangedSub = mvoStatUpd.subscribe(
					m_MVOStatusChangedHandler,
					throwable -> m_MVOStatusChangedHandler.onError(throwable));
			m_Log.debug(lbl + "restarted sub for MVOStatusChanged events "
						+ "from block " + m_SubStartBlock);
		}

		/**
		 * method required to handle completion action after last event seen
		 * (will never be called with a 2-argument subscription stipulating
		 * LATEST block, non-terminating)
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			final String lbl = this.getClass().getSimpleName() + ".run: ";
			m_Log.error(lbl + "unexpected invocation on non-terminating sub");
		}

		/**
		 * nullary constructor
		 */
		public MVOStatusChangedHandler() { }
	}

	/**
	 * handler for MVOStakeUpdated events
	 */
	private MVOStatusChangedHandler	m_MVOStatusChangedHandler;

	/**
	 * our subscription to MVOStatusChanged events (one open at a time)
	 */
	private Disposable				m_MVOStatusChangedSub;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the owning MVO
	 * @param scConfig the relevant smart contract config record
	 */
	public MVOStakingListener(MVO mvo, SmartContractConfig scConfig) {
		m_MVO = mvo;
		m_SCC = scConfig;
		m_Log = scConfig.log();
		m_MVOStakeUpdatedHandler = new MVOStakeUpdatedHandler();
		m_MVOStatusChangedHandler = new MVOStatusChangedHandler();
		m_SubStartBlock = BigInteger.ZERO;
	}

	/**
	 * initialize operation
	 * @return true on success
	 */
	public boolean initialize() {
		final String lbl = this.getClass().getSimpleName() + ".initialize: ";
		if (!m_SCC.isEnabled()) {
			m_Log.warning(lbl + "called while SCC not yet initialized for "
						+ "chainId " + m_SCC.getChainId());
		}

		// get the deployed MVOStaking contract for this chain
		String mvoStakingAddr = MVOStaking.getPreviouslyDeployedAddress(
											Long.toString(m_SCC.getChainId()));
		if (mvoStakingAddr == null) {
			m_Log.error(lbl + "MVOStaking contract not deployed on chain Id "
						+ m_SCC.getChainId());
			return false;
		}

		// since we're strictly view-only, can dummy up Credentials (and no gas)
		try {
			m_Credentials = Credentials.create(Keys.createEcKeyPair());
		}
		catch (Exception eee) {
			m_Log.error(lbl + "exception generating dummy Credentials", eee);
			return false;
		}

		// init wrapper
		m_Wrapper = MVOStaking.load(mvoStakingAddr, m_SCC.getABI(),
									m_Credentials, new DefaultGasProvider());

		// obtain last block known to the cache
		m_SubStartBlock = m_SCC.getCache().getLastProcessedBlock();
		DefaultBlockParameterNumber startBlock
			= new DefaultBlockParameterNumber(m_SubStartBlock);

		// request a filter for all subsequent MVOStakeUpdated events
		Flowable<MVOStaking.MVOStakeUpdatedEventResponse> mvoUpdPub
			= m_Wrapper.mVOStakeUpdatedEventFlowable(startBlock,
											DefaultBlockParameterName.LATEST);
		// use form with callback and error handler
		m_MVOStakeUpdatedSub = mvoUpdPub.subscribe(m_MVOStakeUpdatedHandler,
					throwable -> m_MVOStakeUpdatedHandler.onError(throwable));

		// request a filter for all subsequent MVOStatusChanged events
		Flowable<MVOStaking.MVOStatusChangedEventResponse> mvoStatUpd
			= m_Wrapper.mVOStatusChangedEventFlowable(startBlock,
											DefaultBlockParameterName.LATEST);
		// use form with callback and error handler
		m_MVOStatusChangedSub = mvoStatUpd.subscribe(m_MVOStatusChangedHandler,
					throwable -> m_MVOStatusChangedHandler.onError(throwable));
		m_Log.debug(lbl + "started MVOStakeUpdated and MVOStatusChanged event "
					+ "subscriptions from block " + m_SubStartBlock);
		return true;
	}

	/**
	 * shut down the listener and clean up subscriptions
	 */
	public void shutdown() {
		final String lbl = this.getClass().getSimpleName() + ".shutdown: ";
		if (m_SCC.isEnabled()) {
			if (m_MVOStakeUpdatedSub != null
				&& !m_MVOStakeUpdatedSub.isDisposed())	
			{
				try {
					m_MVOStakeUpdatedSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}
			if (m_MVOStatusChangedSub != null
				&& !m_MVOStatusChangedSub.isDisposed())
			{
				try {
					m_MVOStatusChangedSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}
		}
		// else: calling dispose() on subs is likely to throw exceptions

		m_MVOStakeUpdatedSub = null;
		m_MVOStatusChangedSub = null;
		m_SubStartBlock = BigInteger.ZERO;
	}

	// END methods
}
