/*
 * last modified---
 * 	07-16-25 allow failover to other VPN in openPeerSocket()
 * 	07-18-24 set max idle time on peer sockets to 5 min (5 pings)
 * 	07-09-24 shut down PeerConnector if an exception occurs initializing it
 * 	02-12-24 suppress SslContextFactory configuration, as we won't be using it
 * 	07-12-22 improve error message labeling
 * 	06-10-22 use PeerConnector
 * 	06-06-22 use separate SslContextFactory.Client() with trustAll=true
 * 	04-18-22 new
 *
 * purpose---
 * 	provide a class to make outgoing WebSocket connections to MVO/Auditor peers
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.client.HttpClient;
import org.eclipse.jetty.websocket.client.WebSocketClient;
import org.eclipse.jetty.websocket.client.WebSocketUpgradeRequest;
import org.eclipse.jetty.websocket.client.ClientUpgradeRequest;
import org.eclipse.jetty.websocket.server.WebSocketServerFactory;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.WebSocketBehavior;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.CloseStatus;
import org.eclipse.jetty.websocket.api.StatusCode;
import org.eclipse.jetty.websocket.common.WebSocketSession;
import org.eclipse.jetty.websocket.common.WebSocketSessionFactory;
import org.eclipse.jetty.websocket.common.SessionTracker;
import org.eclipse.jetty.websocket.common.scopes.DelegatedContainerScope;
import org.eclipse.jetty.websocket.common.events.EventDriverFactory;
import org.eclipse.jetty.util.ssl.SslContextFactory;

import java.net.URI;
import java.io.IOException;
import java.util.Properties;
import java.util.ArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeoutException;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.SSLSession;


/**
 * This class is used for creating outbound WebSocket connections to our peers,
 * by managing a WebSocketClient.  It leverages structures created in the
 * {@link MVOHandler}, which auto-creates server mode connections from a
 * listener.  The WebSocket callbacks are managed by {@link PeerConnector}.
 */
public final class MVOClient implements HostnameVerifier {
	// BEGIN data members
	/**
	 * the owning MVO
	 */
	private MVO							m_MVOServer;

	/**
	 * the secure client conection factory used by connectors, built using our
	 * configured keystore/
	 */
	private SslContextFactory.Client	m_SslFactory;

	/**
	 * the HttpClient utilized by the m_WebSocketClient to establish connections
	 */
	private HttpClient					m_HttpClient;

	/**
	 * the factory used to create sessions from client side
	 */
	private WebSocketSessionFactory		m_SessionFactory;

	/**
	 * the WebSocket client object we manage
	 */
	private WebSocketClient				m_WebSocketClient;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which owns us
	 */
	public MVOClient(MVO mvo) {
		m_MVOServer = mvo;

		// eventually, should remove the "trust all" option
		m_SslFactory = new SslContextFactory.Client(true);
		//m_HttpClient = new HttpClient(m_SslFactory);
		m_HttpClient = new HttpClient();
		m_WebSocketClient = new WebSocketClient(m_HttpClient);

		/* NB: we do not init remaining objects here because we want to borrow
		 * 	   some of them from MVOHandler (server side), which doesn't define
		 * 	   all its own objects completely until MVOHandler.initialize()
		 * 	   has been called.
		 */
	}

	/**
	 * initialize the object
	 * @return true on success
	 */
	public boolean initialize() {
	/* NB: wss TLS negotiation is broken in Jetty!  Also, we don't want to
	 * 	   require site certificates for MVOs, since they have no domains.
	 * 	   Therefore, we comment all of this out.
		// set up the client SSL connector based on keystore and properties
		Properties props = m_MVOServer.getProperties();
		String keystorePath = props.getProperty("PeerKeyStore");
		m_SslFactory.setKeyStorePath(keystorePath);
		String keystorePass = props.getProperty("KeyStorePass");
		m_SslFactory.setKeyStorePassword(keystorePass);
		String keystoreType = props.getProperty("KeyStoreType");
		m_SslFactory.setKeyStoreType(keystoreType);
		String truststorePath = props.getProperty("TrustStore");
		m_SslFactory.setTrustStorePath(truststorePath);
		String truststoreType = props.getProperty("TrustStoreType");
		m_SslFactory.setTrustStoreType(truststoreType);
		String trustStorePass = props.getProperty("TrustStorePass");
		if (trustStorePass != null) {
			m_SslFactory.setTrustStorePassword(trustStorePass);
		}
		String keyAlias = props.getProperty("MVOKeyAlias", "mvopeers");
		m_SslFactory.setCertAlias(keyAlias);
		m_SslFactory.addExcludeProtocols("SSLv2Hello", "SSLv3");
		// turn on hostname verification
		m_SslFactory.setEndpointIdentificationAlgorithm("HTTPS");
		m_SslFactory.setHostnameVerifier(this);
		try {
			m_SslFactory.start();
		}
		catch (Exception eee) {
			m_MVOServer.log().error("MVOClient.initialize: unable to start "
									+ "client SSL factory", eee);
			return false;
		}
	 */

		// borrow what we can from the MVOHandler (server side)
		MVOHandler serverHandler = m_MVOServer.getMVOHandler();
		WebSocketServerFactory wssFactory = serverHandler.getFactory();
		WebSocketPolicy serverPolicy = serverHandler.getSocketPolicy();
		WebSocketPolicy ourPolicy
			= serverPolicy.delegateAs(WebSocketBehavior.CLIENT);
		// to propagate what we need from the server side factory, we do this:
		DelegatedContainerScope ourContainerScope
			= new DelegatedContainerScope(ourPolicy, wssFactory);
		EventDriverFactory edf = new EventDriverFactory(ourContainerScope);
		m_SessionFactory = new WebSocketSessionFactory(ourContainerScope);
		m_WebSocketClient.setEventDriverFactory(edf);
		m_WebSocketClient.setSessionFactory(m_SessionFactory);
		// max idle timeout = 5 ping intervals (close after 5 pings missed)
		m_WebSocketClient.setMaxIdleTimeout(300000L);

		// start the HttpClient and the websocket client
		try {
			m_HttpClient.start();
			m_WebSocketClient.start();
		}
		catch (Exception eee) {
			m_MVOServer.log().error("MVOClient.initialize: could not start "
									+ "MVOClient", eee);
			return false;
		}
		return true;
	}

	/**
	 * shutdown the client object
	 */
	public void shutdown() {
		// kill all peer sessions
		CloseStatus cStat = new CloseStatus(StatusCode.SHUTDOWN,
											"Normal MVO shutdown");
		MVOState mvoSt = m_MVOServer.getStateObj();
		// loop through snapshot of concurrent map
		for (String mvoId : mvoSt.getConnectedPeers()) {
			PeerConnector peerConn = mvoSt.getPeerConnection(mvoId);

			// allow for possibility that this connector disappeared
			if (peerConn == null) continue;

			// close sockets, if any
			peerConn.shutdown();
			mvoSt.purgePeerConnection(mvoId);
		}

		try {
			if (m_WebSocketClient.isRunning()) {
				m_WebSocketClient.stop();
			}
			if (m_HttpClient.isRunning()) {
				m_HttpClient.stop();
			}
			if (m_SslFactory != null && m_SslFactory.isRunning()) {
				m_SslFactory.stop();
			}
		}
		catch (Exception eee) {
			m_MVOServer.log().error("MVOClient.shutdown: could not stop "
									+ "MVOClient", eee);
		}
	}

	/**
	 * open a connection to a peer if we don't already have one
	 * @param peerId the ID of the MVO or Auditor
	 * @param uri the URI to connect to
	 * @param allowFailover true if we haven't yet tried other VPN's URI
	 * @return the session connected to that peer, or null on errors
	 */
	public synchronized PeerConnector.MVOClientWebSocket openPeerSocket(
														String peerId,
														URI uri,
														boolean allowFailover)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".openPeerSocket: ";
		Log log = m_MVOServer.log();
		if (peerId == null || peerId.isEmpty() || uri == null) {
			log.error(lbl + "missing connect input");
			return null;
		}

		// see if we already have a connector for this object
		MVOState stateObj = m_MVOServer.getStateObj();
		PeerConnector peerConn = stateObj.getPeerConnection(peerId);
		if (peerConn != null) {
			PeerConnector.MVOClientWebSocket existing
				= peerConn.getClientSocket();
			if (existing != null && existing.isAuthenticated()) {
				// use the one we have
				return existing;
			}
		}
		else {
			// create a new connector and record it
			peerConn = new PeerConnector(m_MVOServer, peerId);
			stateObj.addPeerConnection(peerId, peerConn);
		}

		// create a new one using our client
		if (m_WebSocketClient == null || !m_WebSocketClient.isRunning()) {
			log.error(lbl + "WebSocketClient not ready");
			peerConn.shutdown();
			return null;
		}
		PeerConnector.MVOClientWebSocket newSocket
			= peerConn.new MVOClientWebSocket();

		// build the upgrade request we need to identify who we are to peer
		WebSocketUpgradeRequest wsuReq
			= new WebSocketUpgradeRequest(m_WebSocketClient, m_HttpClient,
										  uri, newSocket);
		wsuReq.header("PeerId", m_MVOServer.getMVOId());
		ClientUpgradeRequest clientUpgrade = new ClientUpgradeRequest(wsuReq);
		// allow text protocol only, no binary
		ArrayList<String> protocols = new ArrayList<String>(1);
		protocols.add("text");
		clientUpgrade.setSubProtocols(protocols);
		clientUpgrade.setTimeout(10L, TimeUnit.SECONDS);
		Future<Session> doConn = null;
		Session newSess = null;
		try {
			doConn = m_WebSocketClient.connect(newSocket, uri, clientUpgrade);
			newSess = doConn.get(15L, TimeUnit.SECONDS);
		}
		catch (IOException ioe) {
			log.error(lbl + "I/O exception connecting to "
					+ peerId + " at " + uri, ioe);
		}
		catch (TimeoutException te) {
			log.error(lbl + "timeout exception connecting to "
					+ peerId + " at " + uri, te);
		}
		catch (ExecutionException ee) {
			log.error(lbl + "execution exception connecting to "
					+ peerId + " at " + uri);
		}
		catch (InterruptedException ie) {
			log.error(lbl + "interruption exception connecting to "
					+ peerId + " at " + uri, ie);
		}
		finally {
			if (newSess instanceof WebSocketSession) {
				WebSocketSession webSess = (WebSocketSession) newSess;
				// record websocket with the connector in the client slot
				peerConn.registerClientSocket(newSocket);

				// add to session tracker (holds both server and client sockets)
				MVOHandler serverHandler = m_MVOServer.getMVOHandler();
				serverHandler.getSessionTracker().onSessionCreated(webSess);
			}
			else {
				// see if we can flip the VPN between odd and even
				if (allowFailover) {
					// permute the URI
					URI altUri = m_MVOServer.computePeerURI(peerId, false);
					if (altUri.compareTo(uri) == 0) {
						altUri = m_MVOServer.computePeerURI(peerId, true);
						if (altUri.compareTo(uri) == 0) {
							/* This can happen if URI hardcoded in .properties.
							 * In this case we have no alternate URI available.
							 */
							peerConn.shutdown();
							return null;
						}
					}

					// recurse and retry with alternate URI
					return openPeerSocket(peerId, altUri, false);
				}
				else {
					// shut down the PeerConnector
					peerConn.shutdown();
					return null;
				}
			}
		}
		return newSocket;
	}

	/**
	 * obtain the web socket client
	 * @return the client object
	 */
	public WebSocketClient getClient() { return m_WebSocketClient; }

	// implement HostnameVerifier
	/**
	 * implement the single HostnameVerifier method
	 * @param host the hostname
	 * @param session the SSL session
	 * @return true if valid (always true)
	 */
	public boolean verify(String host, SSLSession session) {
		m_MVOServer.log().debug("MVOClient.verify(): verified host " + host
								+ " okay to connect");
		return true;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			m_SslFactory = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
