/*
 * last modified---
 * 	01-27-25 purge existing PeerConnectionS when they have null client sockets
 * 	03-08-24 prevent pings from starting on new connections when open has failed
 * 	05-26-23 adjust MVO random selection algorithm
 * 	10-01-22 in writeFailed() and run() methods, transition from prev step
 * 	08-19-22 broadcastToAuditors() fails only if 0 Auditors are reached
 * 	08-10-22 add AuditorQueue usage
 * 	08-03-22 add KeyServerTarget class, sendKeyServerReq()
 * 	08-02-22 add broadcastToAuditors() (stub)
 * 	07-12-22 improve error message labeling
 * 	06-22-22 flesh out FutureWriteCallback.writeFailed() and writeSuccess()
 * 	06-21-22 use AsyncContext
 * 	06-13-22 use PeerConnector
 * 	06-01-22 debug selectMVOCommittee()
 * 	05-11-22 add m_TransId, sendToMVOCommittee()
 * 	05-10-22 add selectMVOCommittee(), getMVOMsgTag(), getAUDMsgTag()
 * 	04-29-22 allow MVO and Aud req/rep to be ConcurrentHashMapS
 * 	03-30-22 new
 *
 * purpose---
 * 	provides a class that acts as a broker for requests made to the MVO by a
 * 	dApp client, encapsulating other necessary asynchronous requests
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.ClientReceiptBlock;
import cc.enshroud.jetty.ClientWalletBlock;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.websocket.common.WebSocketSession;
import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.RemoteEndpoint;
import org.eclipse.jetty.websocket.common.io.FutureWriteCallback;

import java.util.ArrayList;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.text.NumberFormat;
import java.net.URI;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.AsyncContext;


/**
 * This class is used as a HashMap entry.  It is used to associate a reply
 * received from other entities with the original transaction from a dApp
 * which precipitated it.
 */
public final class MVOBrokerEntry {
	// BEGIN data members
	/**
	 * the owning MVO
	 */
	private MVO							m_MVOServer;

	/**
	 * the transaction ID we were created for (index to this record in
	 * MVOState.m_BrokeredRequests)
	 */
	private String						m_TransId;

	/**
	 * the original message received from a dApp, parsed and validated
	 * (either received by us -- when lead -- or forwarded to us by the lead)
	 */
	private ClientRequest				m_DappRequest;

	/**
	 * the {OperationsBlock} received from the lead MVO, if we're not lead
	 * (for a lead MVO this will be the block we ourselves created)
	 */
	private OperationsBlock				m_LeadMVOBlock;

	/**
	 * the async response context built for replies to the requesting dApp user
	 * (NB: if we're a committee MVO, this will be null)
	 */
	private AsyncContext				m_DappReply;

	/**
	 * the tag of the request we received from the lead MVO (if we're a
	 * committee member; if we're the lead this will be null).  Format:
	 * "$MVOId:$transId:$msgId", generated by getMVOMsgTag().  (Note the mvoId
	 * field will be the lead MVO, not ourselves, because this is for a reply.)
	 */
	private String						m_MVOReplyId;

	/**
	 * outgoing brokered async raw requests to other MVOs, which we sent on
	 * behalf of the original request (such requests may be outstanding),
	 * indexed by partner MVOId:transId:msgId (which should always be unique)
	 */
	private ConcurrentHashMap<String, String>	m_BrokerMVOReqs;

	/**
	 * the raw replies to the current outgoing MVO requests, indexed by
	 * MVOId:transId:msgId (note we should never have a hash index collision)
	 */
	private ConcurrentHashMap<String, String>	m_BrokerMVOReps;

	/**
	 * outgoing brokered async raw requests to Auditors, which we sent in the
	 * context of processing the dApp request (either a key create or fetch,
	 * or an {OperationsBlock} broadcast to all known Auditors), indexed by
	 * partner AUDId:msgId (as generated by getAUDMsgTag())
	 */
	private ConcurrentHashMap<String, String>	m_BrokerAudReqs;

	/**
	 * the raw replies to the current outstanding Auditor requests, indexed by
	 * AUDId:msgId
	 */
	private ConcurrentHashMap<String, String>	m_BrokerAudReps;

	/**
	 * The state-tracking object.  This is used to specify steps in processing
	 * the original dApp request, to hold temporary data, and to facilitate
	 * error handling.  The type here is a base class, which will actually be
	 * a sub-classed instance depending on the nature (opcode) in m_DappRequest.
	 */
	private ReqSvcState					m_StateTracker;

	/**
	 * local copy of error logging object for MVO
	 */
	private Log							m_Log;

	/**
	 * inner class to represent the members and state of a MVO committee
	 */
	public class MVOCommitteeMember extends FutureWriteCallback {
		/**
		 * MVO Id
		 */
		public String					m_Id;

		/**
		 * waiting on establishment of WebSocket connection before sending
		 */
		public boolean					m_ConnectWait;

		/**
		 * request sent
		 */
		public boolean					m_SigRequested;

		/**
		 * signature reply received
		 */
		public boolean					m_SigReceived;

		/**
		 * the context data for the committee
		 */
		public MVOVerifyBlock			m_Context;

		/**
		 * the message tag assigned
		 */
		public String					m_MsgTag;

		/**
		 * an error occurred (this MVO could be replaced in committee)
		 */
		public boolean					m_Error;

		/**
		 * timer used to wait for asynchronous reply on PeerConnection
		 */
		public Timer					m_Timer;

		/**
		 * inner class to provide a TimerTask
		 */
		protected class MVOTimeout extends TimerTask {
			/**
			 * nullary constructor
			 */
			public MVOTimeout() {
				super();
			}

			/**
			 * method triggered by m_Timer, which must handle timeouts on
			 * completed write requests to the committee member
			 */
			@Override
			public void run() {
				m_Error = true;
				final String lbl = "MVOCommitteeMember.run: ";
				m_Log.error(lbl + "reply timeout from MVO " + m_Id);

				// propagate abort to state engine
				int endState = 0;
				int step = m_StateTracker.getPreviousStep();
				if (m_StateTracker instanceof DepositState) {
					endState = DepositState.M_Replied;
				}
				else if (m_StateTracker instanceof WithdrawState) {
					endState = WithdrawState.M_Replied;
				}
				else if (m_StateTracker instanceof SpendState) {
					endState = SpendState.M_Replied;
				}
				else {
					m_Log.error(lbl + "CRIT - unexpected state engine type, "
								+ m_StateTracker.getClass().getSimpleName());
					return;
				}

				// move from current state to appropriate error
				if (m_StateTracker.advanceState(step, endState,
												ReqSvcState.M_UnavailMVO,
												"Reply timeout from " + m_Id))
				{
					m_Log.debug(lbl + "successful "
								+ "processing of timeout from MVOId " + m_Id
								+ ", opcode " + m_LeadMVOBlock.getOpcode());
				}
				else {
					m_Log.error(lbl + "unsuccessful "
								+ "processing of timeout from MVOId " + m_Id
								+ ", opcode " + m_LeadMVOBlock.getOpcode());
				}
			}
		}

		/**
		 * timer task to go with timer
		 */
		protected MVOTimeout			m_TimeoutTask;

		/**
		 * constructor
		 * @param id the MVO Id
		 */
		public MVOCommitteeMember(String id) {
			super();
			m_Id = id;
			m_MsgTag = "";
			m_TimeoutTask = new MVOTimeout();
		}

		// implement FutureWriteCallback
		/**
		 * handle async write failure
		 * @param cause what made the write fail
		 */
		@Override
		public void writeFailed(Throwable cause) {
			m_Error = true;
			final String lbl = "MVOCommitteeMember.writeFailed: ";
			m_Log.error(lbl + "could not write to MVO " + m_Id, cause);

			// propagate abort to state engine
			int endState = 0;
			int step = m_StateTracker.getPreviousStep();
			if (m_StateTracker instanceof DepositState) {
				endState = DepositState.M_Replied;
			}
			else if (m_StateTracker instanceof WithdrawState) {
				endState = WithdrawState.M_Replied;
			}
			else if (m_StateTracker instanceof SpendState) {
				endState = SpendState.M_Replied;
			}
			else {
				m_Log.error(lbl + "CRIT - unexpected state engine type, "
							+ m_StateTracker.getClass().getSimpleName());
				return;
			}

			// move from current state to appropriate error
			if (m_StateTracker.advanceState(step, endState,
											ReqSvcState.M_UnavailMVO,
											"Write failure to " + m_Id))
			{
				m_Log.debug(lbl + "successful "
							+ "processing of write fail to MVOId " + m_Id
							+ ", opcode " + m_LeadMVOBlock.getOpcode());
			}
			else {
				m_Log.error(lbl + "unsuccessful "
							+ "processing of write fail to MVOId " + m_Id
							+ ", opcode " + m_LeadMVOBlock.getOpcode());
			}
		}

		/**
		 * handle async write success
		 */
		@Override
		public void writeSuccess() {
			m_SigRequested = true;
		/*
			m_Log.debug("MVOCommitteeMember.writeSuccess: successful write to "
						+ m_Id);
		 */

			// set a timer on the response
			if (m_Timer != null) {
				m_Timer.cancel();
			}
			m_Timer = new Timer("Committee:" + m_Id, true);
			// allow 20 secs for MVO's reply
			m_Timer.schedule(m_TimeoutTask, 20000L);
		}
	}

	/**
	 * the MVO committee state, if any (created only by lead MVOs)
	 */
	public ArrayList<MVOCommitteeMember>	m_MVOCommittee;

	/**
	 * inner class to represent the Auditor target of a key server request
	 */
	public class KeyServerTarget extends FutureWriteCallback {
		/**
		 * Auditor peer Id
		 */
		public String				m_Id;

		/**
		 * waiting on establishment of WebSocket connection before sending
		 */
		public boolean				m_ConnectWait;

		/**
		 * request sent
		 */
		public boolean				m_ReqSent;

		/**
		 * reply received
		 */
		public boolean				m_RepReceived;

		/**
		 * the context data to send
		 */
		public MVOKeyBlock			m_Context;

		/**
		 * the message tag assigned
		 */
		public String				m_MsgTag;

		/**
		 * an error occurred (another Auditor could be tried)
		 */
		public boolean				m_Error;

		/**
		 * timer used to wait for asynchronous reply on PeerConnection
		 */
		public Timer				m_Timer;

		/**
		 * inner class to provide a TimerTask
		 */
		protected class AudTimeout extends TimerTask {
			/**
			 * nullary constructor
			 */
			public AudTimeout() {
				super();
			}

			/**
			 * method triggered by m_Timer, which must handle timeouts on
			 * completed write requests to the committee member
			 */
			@Override
			public void run() {
				m_Error = true;
				m_Log.error("KeyServerTarget.run(): reply timeout from Auditor "
							+ m_Id);

				// propagate abort to state engine, from prev step to end state
				int endState = 0;
				int step = m_StateTracker.getPreviousStep();
				if (m_StateTracker instanceof WalletState) {
					endState = WalletState.M_Replied;
				}
				else if (m_StateTracker instanceof ReceiptState) {
					endState = ReceiptState.M_Replied;
				}
				else if (m_StateTracker instanceof DepositState) {
					endState = DepositState.M_Replied;
				}
				else if (m_StateTracker instanceof WithdrawState) {
					endState = WithdrawState.M_Replied;
				}
				else if (m_StateTracker instanceof SpendState) {
					endState = SpendState.M_Replied;
				}
				else {
					m_Log.error("KeyServerTarget.run(): CRIT - unexpected "
								+ "state engine type, "
								+ m_StateTracker.getClass().getSimpleName());
					return;
				}

				// move from current state to appropriate error
				if (m_StateTracker.advanceState(step, endState,
												ReqSvcState.M_UnavailAud,
												"Reply timeout from " + m_Id))
				{
					m_Log.debug("KeyServerTarget.run(): successful "
								+ "processing of timeout from AudId " + m_Id
								+ ", key opcode " + m_Context.getOpcode());
				}
				else {
					m_Log.error("KeyServerTarget.run(): unsuccessful "
								+ "processing of timeout from AudId " + m_Id
								+ ", key opcode " + m_Context.getOpcode());
				}
			}
		}

		/**
		 * timer task to go with timer
		 */
		protected AudTimeout		m_TimeoutTask;

		/**
		 * constructor
		 * @param id the Auditor peer Id
		 */
		public KeyServerTarget(String id) {
			super();
			m_Id = id;
			m_MsgTag = "";
			m_TimeoutTask = new AudTimeout();
		}

		/**
		 * timer cancel method
		 */
		public void cancel() {
			if (m_Timer != null) {
				m_Timer.cancel();
			}
		}

		// implement FutureWriteCallback
		/**
		 * handle async write failure
		 * @param cause what made the write fail
		 */
		@Override
		public void writeFailed(Throwable cause) {
			m_Error = true;
			final String lbl = "KeyServerTarget.writeFailed: ";
			m_Log.error(lbl + "could not write to Auditor " + m_Id, cause);

			// propagate abort to state engine
			int endState = 0;
			int step = m_StateTracker.getPreviousStep();
			if (m_StateTracker instanceof ReceiptState) {
				endState = ReceiptState.M_Replied;
			}
			else if (m_StateTracker instanceof WalletState) {
				endState = WalletState.M_Replied;
			}
			else if (m_StateTracker instanceof DepositState) {
				endState = DepositState.M_Replied;
			}
			else if (m_StateTracker instanceof WithdrawState) {
				endState = WithdrawState.M_Replied;
			}
			else if (m_StateTracker instanceof SpendState) {
				endState = SpendState.M_Replied;
			}
			else {
				m_Log.error(lbl + "CRIT - unexpected state engine type, "
							+ m_StateTracker.getClass().getSimpleName());
				return;
			}

			// move from current state to appropriate error
			if (m_StateTracker.advanceState(step, endState,
											ReqSvcState.M_UnavailAud,
											"Write failure to " + m_Id))
			{
				m_Log.debug(lbl + "successful "
							+ "processing of write fail to AudId " + m_Id
							+ ", key opcode " + m_Context.getOpcode());
			}
			else {
				m_Log.error(lbl + "unsuccessful "
							+ "processing of write fail to AudId " + m_Id
							+ ", key opcode " + m_Context.getOpcode());
			}
		}

		/**
		 * handle async write success
		 */
		@Override
		public void writeSuccess() {
			m_ReqSent = true;
		/*
			m_Log.debug("KeyServerTarget.writeSuccess: successful write to "
						+ m_Id);
		 */
			// set a timer on the response
			if (m_Timer != null) {
				m_Timer.cancel();
			}
			m_Timer = new Timer("Keyserver:" + m_Id, true);
			// allow 10 secs for Auditor's reply
			m_Timer.schedule(m_TimeoutTask, 10000L);
		}
	}

	/**
	 * the Auditor key server request state
	 */
	private KeyServerTarget				m_KeyServerTarget;

	// END data members

	// BEGIN methods
	/**
	 * constructor for creation in client dApp request context
	 * @param dAppReq the original request we are brokering
	 * @param mvo the MVO which created us for its brokered request map
	 */
	public MVOBrokerEntry(ClientRequest dAppReq, MVO mvo) {
		m_MVOServer = mvo;
		m_TransId = "";
		m_Log = m_MVOServer.log();
		if (dAppReq == null) {
			m_Log.error("MVOBrokerEntry constructor invoked with null orig "
						+ "dApp message");
		}
		m_DappRequest = dAppReq;
		m_LeadMVOBlock = new OperationsBlock(m_Log);
		m_BrokerMVOReqs = new ConcurrentHashMap<String, String>();
		m_BrokerMVOReps = new ConcurrentHashMap<String, String>();
		m_BrokerAudReqs = new ConcurrentHashMap<String, String>();
		m_BrokerAudReps = new ConcurrentHashMap<String, String>();
		m_MVOCommittee = new ArrayList<MVOCommitteeMember>();
	}

	/**
	 * constructor for creation in MVO committee context
	 * @param dAppReq the original request from the client, forwarded to us
	 * @param opBlock the forwarded {OperationsBlock} sent by the lead MVO
	 * @param replyId the ID we must use to reply to the sending (lead) MVO
	 * @param mvo the MVO which created us for its brokered request map
	 */
	public MVOBrokerEntry(ClientRequest dAppReq,
						  OperationsBlock opBlock,
						  String replyId,
						  MVO mvo)
	{
		m_MVOServer = mvo;
		m_Log = m_MVOServer.log();
		if (dAppReq == null) {
			m_Log.error("MVOBrokerEntry constructor invoked with null orig "
						+ "dApp message");
		}
		if (opBlock == null) {
			m_Log.error("MVOBrokerEntry constructor invoked with null OB");
		}
		if (replyId == null || replyId.isEmpty()) {
			m_Log.error("MVOBrokerEntry constructor invoked with no replyId");
		}
		m_DappRequest = dAppReq;
		m_LeadMVOBlock = opBlock;
		m_MVOReplyId = replyId;
		m_BrokerMVOReqs = new ConcurrentHashMap<String, String>();
		m_BrokerMVOReps = new ConcurrentHashMap<String, String>();
		m_BrokerAudReqs = new ConcurrentHashMap<String, String>();
		m_BrokerAudReps = new ConcurrentHashMap<String, String>();
	}

	// GET accessor methods
	/**
	 * obtain the original dApp message
	 * @return the message (from the dApp)
	 */
	public ClientRequest getDappRequest() { return m_DappRequest; }

	/**
	 * obtain the transId we're a part of
	 * @return the Id, index to this record in brokered request map
	 */
	public String getTransId() { return m_TransId; }

	/**
	 * obtain the forwarded {OperationsBlock} message
	 * @return the message, created by us (if lead), or forwarded from the lead
	 */
	public OperationsBlock getOperationsBlock() { return m_LeadMVOBlock; }

	/**
	 * obtain the pending reply being constructed
	 * @return the message (to the dApp)
	 */
	public AsyncContext getDappReply() { return m_DappReply; }

	/**
	 * obtain the message Id required to send a reply to the originating MVO
	 * @return the Id, equal to "leadMVOId:trans:seqNumber"
	 */
	public String getMVOReplyId() { return m_MVOReplyId; }

	/**
	 * obtain the brokered requests to another MVO
	 * @return the outgoing WebSocket requests
	 */
	public ConcurrentHashMap<String, String> getBrokerMVOReqs() {
		return m_BrokerMVOReqs;
	}

	/**
	 * obtain the incoming replies to the MVO brokered requests
	 * @return the WebSocket replies (from the other MVO)
	 */
	public ConcurrentHashMap<String, String> getBrokerMVOReps() {
		return m_BrokerMVOReps;
	}

	/**
	 * obtain the latest brokered requests sent to an Auditor
	 * @return the outgoing WebSocket requests
	 */
	public ConcurrentHashMap<String, String> getBrokerAudReqs() {
		return m_BrokerAudReqs;
	}

	/**
	 * obtain the incoming replies to the Auditor brokered requests
	 * @return the WebSocket replies (from the Auditors)
	 */
	public ConcurrentHashMap<String, String> getBrokerAudReps() {
		return m_BrokerAudReps;
	}

	/**
	 * obtain the owning MVO
	 * @return the MVO operating us
	 */
	public MVO getMVO() { return m_MVOServer; }

	/**
	 * obtain the state tracker
	 * @return the tracker, a subclass of ReqSvcState
	 */
	public ReqSvcState getStateTracker() { return m_StateTracker; }

	/**
	 * obtain the committee tracking records
	 * @return the list and status of MVO committee members
	 */
	public ArrayList<MVOCommitteeMember> getCommittee() {
		return m_MVOCommittee;
	}

	/**
	 * obtain the key server targeted
	 * @return the key server we last sent an async request to
	 */
	public KeyServerTarget getKeyServer() { return m_KeyServerTarget; }

	/**
	 * obtain the logging object
	 * @return the logger copy
	 */
	public Log getLog() { return m_Log; }


	// SET methods
	/**
	 * set the asynch reply context
	 * @param aContext the object we'll use to send replies after async started
	 */
	public void setAsyncContext(AsyncContext aContext) {
		if (aContext == null) {
			m_Log.error("MVOBrokerEntry.setAsyncContext: missing context");
			return;
		}
		if (m_DappReply != null) {
			m_Log.warning("MVOBrokerEntry.setAsyncContext: replacing "
						+ "AsyncContext for request");
		}
		m_DappReply = aContext;
	}

	/**
	 * set the transaction ID
	 * @param tId the ID, index to this record in MVOState.m_BrokeredRequests
	 */
	public void setTransId(String tId) { 
		if (tId != null && !tId.isEmpty()) {
			m_TransId = new String(tId);
		}
	}

	/**
	 * set the outgoing brokered MVO request (invoked when we send it)
	 * @param target the target we sent it to (MVOId:transId:msgId)
	 * @param reqSent the request (TEXT WebSocket message)
	 */
	public void setBrokerMVOReq(String target, String reqSent) {
		if (reqSent != null && !reqSent.isEmpty()
			&& target != null && !target.isEmpty())
		{
			m_BrokerMVOReqs.put(target, reqSent);
		}
	}

	/**
	 * set the incoming brokered MVO reply (invoked when we receive one)
	 * @param target the target we got it from (MVOId:transId:msgId)
	 * @param repRecv the reply (TEXT WebSocket message)
	 */
	public void setBrokerMVORep(String target, String repRecv) {
		if (repRecv != null && !repRecv.isEmpty()
			&& target != null && !target.isEmpty())
		{
			m_BrokerMVOReps.put(target, repRecv);
		}
	}

	/**
	 * set the outgoing brokered Auditor request (invoked when we send it)
	 * @param target the target we sent it to (AUDId:transId:msgId)
	 * @param reqSent the request (TEXT WebSocket message)
	 */
	public void setBrokerAudReq(String target, String reqSent) {
		if (reqSent != null && !reqSent.isEmpty()
			&& target != null && !target.isEmpty())
		{
			m_BrokerAudReqs.put(target, reqSent);
		}
	}

	/**
	 * set the incoming brokered Auditor reply (invoked when we receive one)
	 * @param audId the target we got it from (AUDId:transId:msgId)
	 * @param repRecv the reply (TEXT WebSocket message)
	 */
	public void setBrokerAudRep(String target, String repRecv) {
		if (repRecv != null && !repRecv.isEmpty()
			&& target != null && !target.isEmpty())
		{
			m_BrokerAudReps.put(target, repRecv);
		}
	}

	/**
	 * set the state tracker
	 * @param tracker a subclass of ReqSvcState
	 */
	public void setStateTracker(ReqSvcState tracker) {
		if (tracker != null) {
			m_StateTracker = tracker;
		}
	}

	/**
	 * randomly determine a committee of MVOs for this request (called only
	 * when we're the lead MVO)
	 * @param chainId the chain we need a committee for
	 * @return a list of MVO configs, null on errors
	 */
	public ArrayList<MVOConfig> selectMVOCommittee(long chainId) {
		final String lbl = "MVOBrokerEntry.selectMVOCommittee: ";
		if (chainId <= 0L) {
			m_Log.error(lbl + "invalid chain Id, " + chainId);
			return null;
		}

		// obtain the chain's config
		SmartContractConfig scc = m_MVOServer.getSCConfig(chainId);
		if (scc == null) {
			m_Log.error(lbl + "no config found for chain Id, " + chainId);
			return null;
		}

		/* Examine the total staked by all MVOs on this chain.  If this is zero,
		 * then we need to check the total on Ethereum (chainId = 1) instead.
		 * Because we can't pick ourselves for the committee, we'll subtract
		 * our own staking amount from the total of all MVOs.
		 */
		BigInteger stakingTot = scc.getTotalStaking();
		SmartContractConfig ethConfig = m_MVOServer.getSCConfig(1L);
		boolean useEth = false;
		if (stakingTot.equals(BigInteger.ZERO)) {
			// fall back to Eth, for both total and individual MVO stakings
			if (ethConfig == null) {
				m_Log.error(lbl + "no config for Mainnet!");
				return null;
			}
			useEth = true;
			stakingTot = ethConfig.getTotalStaking();
			m_Log.debug(lbl + "using ETH staking "
						+ "as the determinant, total = " + stakingTot);
		}
		final BigInteger oneE18 = new BigInteger("1000000000000000000");
		// scale down staking total by 1e18 so we can do math in doubles
		stakingTot = stakingTot.divide(oneE18);

		// now subtract our own staking value from the total
		MVOConfig ourConf = (MVOConfig) m_MVOServer.getConfig();
		BlockchainConfig ourBcc = null;
		if (useEth) {
			// use the MVO's BlockchainConfig from Ethereum
			ourBcc = ourConf.getChainConfig(1L);
		}
		else {
			ourBcc = ourConf.getChainConfig(chainId);
			if (ourBcc == null) {
				// fallback to Eth mainnet
				ourBcc = ourConf.getChainConfig(1L);
			}
		}
		if (ourBcc == null) {
			m_Log.error(lbl + "could not find BlockchainConfig "
						+ "for MVOId " + m_MVOServer.getMVOId());
			return null;
		}
		BigInteger ourStaking = ourBcc.getStaking().divide(oneE18);
		double allStakings = stakingTot.subtract(ourStaking).doubleValue();

		// determine number of MVOs required
		Hashtable<String, MVOGenConfig> availMVOs = scc.getMVOMap();
		int sigs = scc.getNumSigs() - 1;
		if (sigs < 1) {
			m_Log.error(lbl + "bad number of sigs "
						+ "(" + scc.getNumSigs() + ") for chain Id " + chainId);
			return null;
		}
		NumberFormat nf = NumberFormat.getNumberInstance();
		ArrayList<MVOConfig> pickedMVOs = new ArrayList<MVOConfig>(sigs);
		ArrayList<String> pickedMVOIds = new ArrayList<String>(sigs);

		// NB - assumption here: sum of stakings for all MVOs == stakingTot
		for (int iii = 0; iii < sigs; iii++) {
			boolean selectedMVO = false;
			Set<String> candidateMVOs = availMVOs.keySet();
			int itCnt = 0;
			// loop until we get a pick, or run out of possible new picks
			while (!selectedMVO
				   && candidateMVOs.size() - 1 - pickedMVOs.size() > 0)
			{
				// failsafe to prevent endless loop if something misconfigured
				if (++itCnt > sigs * 10) {
					break;
				}
				double runningTotal = 0.0;
				double pickPct
					= m_MVOServer.getStateObj().getRNG().nextDouble();

				// for each MVO in the list, compare against a weighted percent
				for (String mvoId : candidateMVOs) {
					// don't pick ourselves
					if (mvoId.equals(m_MVOServer.getMVOId())) {
						continue;
					}

					// get this MVO's config
					MVOConfig mvoConf = (MVOConfig) availMVOs.get(mvoId);
					if (!mvoConf.getStatus()) {
						// MVO marked disabled; skip
						//m_Log.debug("MVOId " + mvoId + " disabled; skipped");
						continue;
					}

					// also get the MVO's config for this blockchain
					BlockchainConfig bcc = null;
					if (useEth) {
						// use the MVO's BlockchainConfig from Ethereum
						bcc = mvoConf.getChainConfig(1L);
					}
					else {
						bcc = mvoConf.getChainConfig(chainId);
						if (bcc == null) {
							// fallback to Eth mainnet
							bcc = mvoConf.getChainConfig(1L);
						}
					}
					if (bcc == null) {
						m_Log.error(lbl + "could not find BlockchainConfig "
									+ "for MVOId " + mvoId);
						itCnt += 1;
						break;
					}
					BigInteger staking = bcc.getStaking();
					if (staking.equals(BigInteger.ZERO)) {
						m_Log.error(lbl + "MVO " + mvoId
									+ " has 0 staking on chain "
									+ bcc.getChainId() + ", error");
						itCnt += 1;
						break;
					}
					// scale staking by 1e-18
					staking = staking.divide(oneE18);
					double mvoStake = staking.doubleValue();

					// compute ratio of MVO staking and add to running total
					double ratio = mvoStake / allStakings;
					runningTotal += ratio;

					// disallow putting the same MVO in twice
					if (pickedMVOIds.contains(mvoId)) {
						continue;
					}
					if (pickPct <= runningTotal) {
						// add this one and continue to next pick
						pickedMVOs.add(mvoConf);
						pickedMVOIds.add(mvoId);
						selectedMVO = true;
					/*
						m_Log.debug(lbl + "added "
									+ "MVO " + mvoId + " to picked list at "
									+ nf.format(pickPct) + " <= "
									+ nf.format(runningTotal) + ", iteration "
									+ itCnt);
					 */
						break;
					}
				}
			}
		}
		if (pickedMVOs.size() != sigs) {
			m_Log.error(lbl + "picked only "
						+ pickedMVOs.size() + " of " + sigs + " MVOs");
		}
		return pickedMVOs;
	}

	/**
	 * helper method to pick the next MVO message tag
	 * @param mvo the Id of the MVO we're sending to
	 * @return the unique identifier to use as a prefix in a WebSocketMessage
	 */
	public String getMVOMsgTag(String mvo) {
		if (mvo == null || mvo.isEmpty()) {
			m_Log.error("MVOBrokerEntry.getMVOMsgTag: no MVOId");
			return "";
		}
		String tag = mvo + ":" + m_TransId + ":"
					+ Long.valueOf(m_MVOServer.getStateObj().getNextMVOid());
		return tag;
	}

	/**
	 * helper method to pick the next AUD message tag
	 * @param aud the Id of the Auditor we're sending to
	 * @return the unique identifier to use as a prefix in a WebSocketMessage
	 */
	public String getAUDMsgTag(String aud) {
		if (aud == null || aud.isEmpty()) {
			m_Log.error("MVOBrokerEntry.getAUDMsgTag: no AudId");
			return "";
		}
		String tag = aud + ":" + m_TransId + ":"
					+ Long.valueOf(m_MVOServer.getStateObj().getNextAUDid());
		return tag;
	}

	/**
	 * method to parse a tag generated by getMVOMsgTag() or getAUDMsgTag()
	 * @param tag the full tag value
	 * @param field which field is wanted (1, 2, 3)
	 * @return the field selected, or empty string on errors
	 */
	public String getTagField(String tag, int field) {
		String ret = "";
		if (tag == null || tag.isEmpty() || field <= 0 || field > 3) {
			m_Log.error("MVOBrokerEntry.getTagField: illegal input");
			return ret;
		}
		if (tag.indexOf(":") == -1) {
			m_Log.error("MVOBrokerEntry.getTagField: not a message tag");
			return ret;
		}
		String[] tagFields = tag.split(":");
		if (tagFields.length != 3) {
			m_Log.error("MVOBrokerEntry.getTagField: not a message tag");
			return ret;
		}
		return tagFields[field-1];
	}

	/**
	 * method to send an {OperationsBlock} to a MVO committee (asynchronously)
	 * @param mvoList the configs of the MVOs in the committee
	 * @param opBlock the block to send (must be signed by us)
	 * @return true on success
	 */
	public boolean sendToMVOCommittee(ArrayList<MVOConfig> mvoList,
									  OperationsBlock opBlock)
	{
		final String label = "MVOBrokerEntry.sendToMVOCommittee: ";
		if (mvoList == null || mvoList.isEmpty()) {
			m_Log.error(label + "missing MVO list");
			return false;
		}
		if (opBlock == null) {
			m_Log.error(label + "missing OperationsBlock");
			return false;
		}

		// make sure OB is signed by us
		ArrayList<MVOSignature> obSigs = opBlock.getSignatures();
		String us = m_MVOServer.getMVOId();
		boolean signed = false;
		for (MVOSignature sig : obSigs) {
			if (sig.m_Signer.equals(us)) {
				signed = true;
				break;
			}
		}
		if (!signed) {
			m_Log.error(label + "OperationsBlock unsigned");
			return false;
		}

		// we must have a ClientMVOBlock available
		if (!(m_DappRequest instanceof ClientMVOBlock)) {
			m_Log.error(label + "missing ClientMVOBlock");
			return false;
		}
		ClientMVOBlock clReq = (ClientMVOBlock) m_DappRequest;
		String opcode = clReq.getOpcode();

		// build MVOVerifyBlock to send
		MVOVerifyBlock committeeMsg = new MVOVerifyBlock(m_Log);
		committeeMsg.setClientBlock(clReq);
		committeeMsg.setMVOBlock(opBlock);
		//HttpServletRequest orgReq = m_DappRequest.getOrigRequest();
		// fetch original client JSON
		String clientJSON = "";
		if (opcode.equals(clReq.M_OpDeposit)
			|| opcode.equals(clReq.M_OpSpend)
			|| opcode.equals(clReq.M_OpWithdraw))
		{
			// ClientRequest recorded this here after decryption
			clientJSON = clReq.getDecryptedPayload();
		}
		else {
			m_Log.error(label + "unexpected opcode, " + opcode);
			return false;
		}
		committeeMsg.setClientBlockJson(clientJSON);
		StringBuilder opbJSON = new StringBuilder(2048);
		opBlock.addJSON(opbJSON);
		committeeMsg.setMVOBlockJson(opbJSON.toString());
		StringBuilder completeJSON = new StringBuilder(4096);
		committeeMsg.addJSON(completeJSON);

		// build our signature on the actual data to be sent
		MVOConfig ourConf = m_MVOServer.getConfig();
		PrivateKey ourKey = ourConf.getCommPrivKey();
		String fullTxt = completeJSON.toString();
		String ourL2Sig = EncodingUtils.signStr(ourKey, fullTxt);

		// process the committee list
		MVOHandler mvoHandler = m_MVOServer.getMVOHandler();
		MVOClient mvoClient = m_MVOServer.getMVOClient();
		MVOState stObj = m_MVOServer.getStateObj();
		SmartContractConfig scc = m_MVOServer.getSCConfig(clReq.getChainId());
		m_MVOCommittee.clear();
		for (MVOConfig mvoConf : mvoList) {
			String mvoId = mvoConf.getId();

			// create record for the member
			MVOCommitteeMember member = new MVOCommitteeMember(mvoId);
			member.m_Context = committeeMsg;

			// ask for an open WebSocket to this MVO
			PeerConnector peerConn = stObj.getPeerConnection(mvoId);
			if (peerConn == null) {
				// ask to open one
				PeerConnector.MVOClientWebSocket webConn
					= mvoClient.openPeerSocket(mvoId,
											   mvoConf.getMVOURI(),
											   true);
				if (webConn == null) {
					m_Log.error(label + "cannot open connection to committee "
								+ "MVO " + mvoId + ", abort " + opcode
								+ " request");
					// remove PeerConnection that failed from MVOState
					stObj.purgePeerConnection(mvoId);
					return false;
				}
				// flag that we're waiting for an asynchronous connection
				member.m_ConnectWait = true;
			}
			else {
				PeerConnector.MVOClientWebSocket webConn
					= peerConn.getClientSocket();
				/* an existing PeerConnection should not have a null client
				 * socket, but might if there was an earlier timeout on the
				 * connection
				 */
				if (webConn == null) {
					// remove failed PeerConnection from MVOState
					peerConn.shutdown();
					stObj.purgePeerConnection(mvoId);

					// ask to open new one
					webConn = mvoClient.openPeerSocket(mvoId,
													   mvoConf.getMVOURI(),
													   true);
					if (webConn == null) {
						m_Log.error(label + "cannot open connection to "
									+ "committee MVO " + mvoId + ", abort "
									+ opcode + " request");
						// remove PeerConnection that failed from MVOState
						stObj.purgePeerConnection(mvoId);
						return false;
					}
					// flag that we're waiting for an asynchronous connection
					member.m_ConnectWait = true;
				}
				else if (!webConn.isAuthenticated()) {
					// flag that we have a connection still waiting on Auth
					member.m_ConnectWait = true;
				}
			}
			m_MVOCommittee.add(member);

			// if we're waiting for a connect or auth, we're done here for now
			if (member.m_ConnectWait) {
				// we'll generate the tag and do the send when auth completes
				continue;
			}

			// encrypt text to target's key
			PublicKey theirKey = ourConf.getPeerPubkey(mvoId);
			if (theirKey == null) {
				m_Log.error(label + "no pubkey found for MVO " + mvoId);
				return false;
			}
			String encTxt = EncodingUtils.base64PubkeyEncStr(fullTxt, theirKey);

			// get a message tag
			String tag = getMVOMsgTag(mvoId);

			// send the message over the socket (async)
			String mvoMsg = tag + "::" + encTxt + "::" + ourL2Sig;
			member.m_MsgTag = new String(tag);
			if (peerConn.sendString(mvoMsg, member)) {
				// record in broker map
				setBrokerMVOReq(member.m_MsgTag, mvoMsg);
			}
			else {
				m_Log.error(label + "client write fail to " + tag
							+ ", abort " + opcode + " request");
				return false;
			}
		}
		return true;
	}

	/**
	 * method to send an {MVOKeyBlock} to an Auditor (asynchronously)
	 * @param audId the configs of the MVOs in the committee
	 * @param opBlock the block to send (must be signed by us)
	 * @return true on success
	 */
	public boolean sendKeyServerReq(String audId, MVOKeyBlock keyBlock) {
		final String label = "MVOBrokerEntry.sendKeyServerReq: ";
		if (audId == null || audId.isEmpty()) {
			m_Log.error(label + "missing Auditor Id");
			return false;
		}
		if (keyBlock == null) {
			m_Log.error(label + "missing MVOKeyBlock");
			return false;
		}

		// make sure key request is signed by us
		MVOSignature reqSig = keyBlock.getSignature();
		String us = m_MVOServer.getMVOId();
		if (!reqSig.m_Signer.equals(us)) {
			m_Log.error(label + "MVOKeyBlock unsigned");
			return false;
		}

		// our ClientRequest could be one of any derived type
		ClientMVOBlock clReq = null;
		ClientReceiptBlock rctReq = null;
		ClientWalletBlock walReq = null;
		String opcode = "";
		if (m_DappRequest instanceof ClientMVOBlock) {
			clReq = (ClientMVOBlock) m_DappRequest;
			opcode = clReq.getOpcode();
		}
		else if (m_DappRequest instanceof ClientReceiptBlock) {
			rctReq = (ClientReceiptBlock) m_DappRequest;
			opcode = rctReq.getOpcode();
		}
		else if (m_DappRequest instanceof ClientWalletBlock) {
			walReq = (ClientWalletBlock) m_DappRequest;
			opcode = "wallet";
		}
		else {
			m_Log.error(label + "unsupported ClientRequest type, "
						+ m_DappRequest.getClass().getSimpleName());
			return false;
		}

		// process the key server request
		MVOClient mvoClient = m_MVOServer.getMVOClient();
		MVOState stObj = m_MVOServer.getStateObj();

		// create record for the selected Auditor
		KeyServerTarget selAud = new KeyServerTarget(audId);
		selAud.m_Context = keyBlock;

		// ask for an open WebSocket to this Auditor
		PeerConnector peerConn = stObj.getPeerConnection(audId);
		if (peerConn == null) {
			// ask to open one
			URI audURI = m_MVOServer.getURIforMVO(audId);
			if (audURI == null) {
				m_Log.error(label + "URI for Auditor " + audId + " unknown");
				return false;
			}
			PeerConnector.MVOClientWebSocket webConn
				= mvoClient.openPeerSocket(audId, audURI, true);
			if (webConn == null) {
				m_Log.error(label + "cannot open connection to Auditor "
							+ audId + ", abort " + opcode + " request");
				// remove new failed connection from MVOState so we don't ping
				stObj.purgePeerConnection(audId);
				return false;
			}
			// flag that we're waiting for an asynchronous connection
			selAud.m_ConnectWait = true;
		}
		else {
			PeerConnector.MVOClientWebSocket webConn
				= peerConn.getClientSocket();
			/* an existing PeerConnection should not have a null client socket,
			 * but might if there was an earlier timeout on the connection
			 */
			if (webConn == null) {
				m_Log.debug(label + "cannot utilize connection to Auditor "
							+ audId + ", retry connect for " + opcode
							+ " request");
				// remove failed connection from MVOState to force recreation
				peerConn.shutdown();
				stObj.purgePeerConnection(audId);

				// ask to open new one
				URI audURI = m_MVOServer.getURIforMVO(audId);
				webConn = mvoClient.openPeerSocket(audId, audURI, true);
				if (webConn == null) {
					m_Log.error(label + "cannot open connection to Auditor "
								+ audId + ", abort " + opcode + " request");
					// remove PeerConnection that failed from MVOState
					stObj.purgePeerConnection(audId);
					return false;
				}
				// flag that we're waiting for an asynchronous connection
				selAud.m_ConnectWait = true;
			}
			else if (!webConn.isAuthenticated()) {
				// flag that we have a connection still waiting on Auth
				selAud.m_ConnectWait = true;
			}
		}
		if (m_KeyServerTarget != null) {
			// cancel any existing timer
			m_KeyServerTarget.cancel();
		}
		m_KeyServerTarget = selAud;

		// if we're waiting for a connect or auth, we're done here for now
		if (selAud.m_ConnectWait) {
			// we'll generate the tag and do the send when auth completes
			return true;
		}

		// build complete message to send
		StringBuilder completeJSON = new StringBuilder(2048);
		keyBlock.addJSON(completeJSON);
		String fullTxt = completeJSON.toString();

		// build our signature on the actual data to be sent
		MVOConfig ourConf = m_MVOServer.getConfig();
		PrivateKey ourKey = ourConf.getCommPrivKey();
		String ourL2Sig = EncodingUtils.signStr(ourKey, fullTxt);

		// encrypt text to target's key
		PublicKey theirKey = ourConf.getPeerPubkey(audId);
		if (theirKey == null) {
			m_Log.error(label + "no pubkey found for Auditor " + audId);
			return false;
		}
		String encTxt = EncodingUtils.base64PubkeyEncStr(fullTxt, theirKey);

		// get a message tag
		String tag = getAUDMsgTag(audId);

		// send the message over the socket (async)
		String audMsg = tag + "::" + encTxt + "::" + ourL2Sig;
		selAud.m_MsgTag = new String(tag);
		if (peerConn.sendString(audMsg, selAud)) {
			// record in broker map
			setBrokerAudReq(selAud.m_MsgTag, audMsg);
		}
		else {
			m_Log.error(label + "client write fail to " + tag
						+ ", abort " + opcode + " request");
			return false;
		}
		return true;
	}

	/**
	 * method to send an {MVOAuditorBlock} to all Auditors (asynchronously)
	 * @param mvoBlock the block to send (OB part must be signed by us)
	 * @return true on success (meaning at least one Auditor was reached)
	 */
	public boolean broadcastToAuditors(MVOAuditorBlock mvoBlock) {
		final String lbl = "MVOBrokerEntry.broadcastToAuditors: ";
		if (mvoBlock == null) {
			m_Log.error(lbl + "missing MVOAuditorBlock");
			return false;
		}

		// make sure AuditorBlock is signed by us
		AuditorBlock audBlock = mvoBlock.getMVOBlock();
		if (audBlock == null) {
			m_Log.error(lbl + "missing AuditorBlock");
			return false;
		}
		boolean signed = false;
		ArrayList<MVOSignature> audSigs = audBlock.getSignatures();
		String us = m_MVOServer.getMVOId();
		for (MVOSignature sig : audSigs) {
			if (sig.m_Signer.equals(us)) {
				signed = true;
				break;
			}
		}
		if (!signed) {
			m_Log.error(lbl + "OperationsBlock unsigned");
			return false;
		}
		MVOConfig mvoConf = m_MVOServer.getConfig();
		HashMap<String, Integer> audList = mvoConf.getAuditorList();
		if (audList.isEmpty()) {
			m_Log.error(lbl + "no Auditors defined");
			return false;
		}

		AuditorQueue audQueue = m_MVOServer.getAuditorManager();
		int successCnt = 0;
		for (String auditor : audList.keySet()) {
			// get the queue for this Auditor
			AuditorQueue.PerAuditorQueue audQ
				= audQueue.getAuditorQueue(auditor);
			AuditorQueue.BroadcastEntry item = audQueue.new BroadcastEntry();
			item.m_Block = mvoBlock;
			item.updateJson();
			if (!audQ.queueBlock(item)) {
				m_Log.error(lbl + "error queueing block to Auditor " + auditor);
			}
			else {
				successCnt++;
			}
		}
		return successCnt > 0;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			if (m_BrokerMVOReqs != null) {
				m_BrokerMVOReqs.clear();
			}
			if (m_BrokerMVOReps != null) {
				m_BrokerMVOReps.clear();
			}
			if (m_BrokerAudReqs != null) {
				m_BrokerAudReqs.clear();
			}
			if (m_BrokerAudReps != null) {
				m_BrokerAudReps.clear();
			}
			if (m_MVOCommittee != null) {
				m_MVOCommittee.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
