/*
 * last modified---
 * 	06-20-23 add acctId arg to getNFTsById(); add checkIfIdUsed()
 * 	06-14-23 remove AssetConfig usage
 * 	06-12-23 pass Web3j objects instead of URLs (ignored in any case)
 * 	04-05-23 use EIP-55 format for account owner addresses
 * 	03-07-23 init test data with contract addresses, not token symbols
 * 	12-01-22 actually read Credentials for chain
 * 	10-26-22 add chainId arg to getAccountNFTs(), getNFTsById(), and
 * 			 createENFTfile()
 * 	07-29-22 add demintedOk arg to getNFTsById()
 * 	07-06-22 no need for HexBinaryAdapter
 * 	06-27-22 make getAccountNFTs() return a HashMap<ID, NFT> so we have IDs
 * 	05-09-22 add additional methods createENFTfile() and backupProps()
 * 	04-26-22 flesh out
 * 	04-22-22 new (skeleton)
 *
 * purpose---
 * 	provide a local filesystem / properties implementation of BlockchainAPI
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.log.Log;

import org.web3j.crypto.WalletUtils;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.CipherException;
import org.web3j.crypto.Keys;
import org.web3j.protocol.Web3j;

import java.math.BigInteger;
import java.net.URL;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Properties;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Date;
import java.util.concurrent.Future;
import java.util.concurrent.CompletableFuture;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.FilenameFilter;
import java.io.FileFilter;


/**
 * Implementation of BlockchainAPI (for Web3j functions) using the local file
 * system and properties to access the test data.  This class is used when
 * we want to run in using limited test data (prefabricated eNFTs and accounts,
 * etc. stored locally) without touching any ABI nodes.  In this class
 * the URLs are ignored.  Iff the Web3 ABI URL in properties is a file:// type,
 * methods of this class should be used.  Otherwise, use RemoteBlockchainAPI.
 * Effectively, this class implements a crude flat file database.
 */
public final class LocalBlockchainAPI
	implements BlockchainAPI, FilenameFilter, FileFilter
{
	// BEGIN data members
	/**
	 * owning MVO which creates us
	 */
	private MVO			m_MVO;

	/**
	 * the chain we support
	 */
	private long		m_ChainId;

	/**
	 * logging object (from owning MVO)
	 */
	private Log			m_Log;

	/**
	 * list of locally stored test eNFTs, mapped to owning account IDs and
	 * stored as a file in $RunDir/eData/eNFT.properties
	 */
	private Properties	m_eNFTList;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which creates us
	 * @param chId the chainId per chainlist.org
	 */
	public LocalBlockchainAPI(MVO mvo, long chId) {
		m_MVO = mvo;
		m_Log = mvo.log();
		m_ChainId = chId;
		m_eNFTList = new Properties();
	}

	/**
	 * obtain chain we support
	 * @return supported chain Id
	 */
	public long getChainId() { return m_ChainId ; }

	// methods to implement BlockchainAPI
	/**
	 * initialize the object
	 * @return true on success
	 */
	public synchronized boolean init() {
		// locate and read in the list of eNFTs in our test data set
		Properties props = m_MVO.getProperties();
		String runDir = props.getProperty("RunDir", "");
		String propPath = runDir + File.separator + "eData" + File.separator
						+ "eNFT.properties";
		boolean err = false;
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(propPath);
			// accommodate any deletions we may not have seen
			m_eNFTList.clear();
			m_eNFTList.load(fis);
			fis.close();
		}
		catch (FileNotFoundException fnfe) {
			m_Log.error("LocalBlockchainAPI.init: could not find eNFT list, "
						+ propPath, fnfe);
			err = true;
		}
		catch (IOException ioe) {
			m_Log.error("LocalBlockchainAPI.init: error reading properties "
						+ "file, " + propPath, ioe);
			err = true;
		}
		return !err;
	}

	/**
	 * write the properties file
	 * @return true on success
	 */
	private synchronized boolean backupProps() {
		// locate and write out the properties file after changes
		Properties props = m_MVO.getProperties();
		String runDir = props.getProperty("RunDir", "");
		String propPath = runDir + File.separator + "eData" + File.separator
						+ "eNFT.properties";
		boolean err = false;
		FileWriter fw = null;
		try {
			fw = new FileWriter(propPath, false);
			m_eNFTList.store(fw, "Last update: " + new Date());
			fw.flush();
			fw.close();
		}
		catch (IOException ioe) {
			m_Log.error("LocalBlockchainAPI.backupProps: error writing "
						+ "properties file, " + propPath, ioe);
			err = true;
		}
		return !err;
	}

	/**
	 * shutdown handler
	 * @return true on success
	 */
	public boolean shutdown() {
		/* Since in testing mode we share the eNFT.properties file between
		 * MVOs, there is a possibility that an MVO which shuts down later
		 * could overwrite the file with a version which does not contain
		 * one or more entries which were added by a peer but which it has
		 * never seen.  Therefore, we re-init from the file immediately before
		 * shutdown.
		 */
		if (init()) {
			return backupProps();
		}
		return false;
	}

	/**
	 * obtain the smart contract configuration for a given blockchain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @return the configuration, with data supplied appropriately
	 */
	public Future<SmartContractConfig> getConfig(long chainId,
												 String chain,
												 Web3j abiSession)
	{
		final String lbl = this.getClass().getSimpleName() + ".getConfig: ";
		if (chainId <= 0L || chain == null || chain.isEmpty()) {
			m_Log.error(lbl + "missing input");
			return null;
		}
		// NB: we ignore the abiSession value

		CompletableFuture<SmartContractConfig> retFut
			= new CompletableFuture<SmartContractConfig>();

		// we'll simply build a generic template response and set a few fields
		SmartContractConfig scc
			= new SmartContractConfig(chain, chainId, m_Log, m_MVO);
		Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();

		/* add test MVO configs
		 * 	We do this by looking for security/MVO-NNN.properties for each peer
		 * 	MVO for which we have a MVO-NNN.pubkey.  This will be a
		 * 	properties file containing:
		 * 		MVOURI=URI
		 * 		status=true
		 * 		staked=BigInteger (stored in m_ChainConfigs for main chain)
		 * 	We don't really care about anything else, and we'll use the
		 * 	same values for all supported chains.
		 */
		String secDir = m_MVO.getProperties().getProperty("RunDir", "");
		StringBuilder keyPath = new StringBuilder(64);
		if (secDir.isEmpty()) {
			keyPath.append("security");
		}
		else {
			keyPath.append(secDir + File.separator + "security");
		}
		// find all files MVO-NNN.properties
		File securityDir = new File(keyPath.toString());
		File[] mvoConfigs = null;
		try {
			mvoConfigs = securityDir.listFiles((FileFilter)this);
		}
		catch (SecurityException se) {
			m_Log.error(lbl + "no permission to access peer properties", se);
			retFut.completeExceptionally(se);
			return retFut;
		}
		boolean propErr = false;
		if (mvoConfigs == null || mvoConfigs.length == 0) {
			m_Log.error(lbl + "no peer properties found in config");
			propErr = true;
		}
		else {
			for (int mmm = 0; mmm < mvoConfigs.length; mmm++) {
				File propFile = mvoConfigs[mmm];
				if (!propFile.exists() || !propFile.canRead()) {
					m_Log.error(lbl + "cannot open properties file "
								+ propFile + " for read");
					propErr = true;
					continue;
				}
				Properties mvoProps = new Properties();
				FileInputStream fis = null;
				try {
					fis = new FileInputStream(propFile);
					mvoProps.load(fis);
					fis.close();
				}
				catch (FileNotFoundException fnfe) {
					// we already checked for this
					propErr = true;
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "error reading properties file, "
								+ propFile, ioe);
					propErr = true;
				}
				if (propErr) continue;

				// get the MVOID from the filename
				String fName = propFile.getName();
				int dot = fName.indexOf(".properties");
				String mvoId = fName.substring(0, dot);
				MVOConfig mConf = new MVOConfig(mvoId, m_Log);

				// record the data from the file in the config, and add to list
				String mvoURI = mvoProps.getProperty("MVOURI");
				URI uri = null;
				try {
					uri = new URI(mvoURI);
					mConf.setMVOURI(uri);
				}
				catch (URISyntaxException ufe) {
					m_Log.error(lbl + "bad format, " + mvoURI, ufe);
					propErr = true;
				}
				Boolean stat = Boolean.valueOf(mvoProps.getProperty("status"));
				mConf.setStatus(stat);

				// the "staked" value in the file represents a cross-chain total
				String staked = mvoProps.getProperty("staked");
				if (staked == null || staked.isEmpty()) {
					m_Log.error(lbl + "missing staked value for " + mvoId);
					propErr = true;
				}
				else {
					// add to Ethereum mainnet, even if this isn't supported
					BlockchainConfig bc
						= mConf.addBlockchainConfig(1L, "mainnet");
					if (bc != null) {
						bc.setStaking(staked);
					}
				}

				// find the SigningAddress of the MVO for this chain
				mConf.addBlockchainConfig(chainId, chain);
				BlockchainConfig cbc = mConf.getChainConfig(chainId);
				String sigAddrTag = "SigningAddress-" + chainId;
				String signingAddress = mvoProps.getProperty(sigAddrTag);
				if (signingAddress != null && !signingAddress.isEmpty()) {
					cbc.configSigningAddress(signingAddress);
					cbc.setStaking(staked);
				}
				else {
					propErr = true;
					m_Log.error(lbl + "missing " + sigAddrTag + " for MVOId "
								+ mvoId);
				}
				// record identical staked value for this MVO for this chain too
				cbc.setStaking(staked);
				mvoMap.put(mvoId, mConf);
			}
			if (propErr) {
				m_Log.error(lbl + "error processing MVO properties files");
				Exception ee = new Exception("Error processing MVO props");
				retFut.completeExceptionally(ee);
				return retFut;
			}
		}

		scc.setNumSigs(2);
		scc.setDwellTime(2);
		// 300k in 18-digit precision
		scc.setTotalStaking("300000000000000000000000");
		retFut.complete(scc);
		return retFut;
	}

	/**
	 * obtain the configuration particular to a given MVO on the chain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @param mvoId the ID of the MVO (in local usage, only called for self)
	 * @return the configuration (public parts only)
	 */
	public Future<BlockchainConfig> getMVOConfig(long chainId,
												 String chain,
												 Web3j abiSession,
												 String mvoId)
	{
		final String lbl = this.getClass().getSimpleName() + ".getMVOConfig: ";
		if (mvoId == null || mvoId.isEmpty()) {
			m_Log.error(lbl + "missing MVO Id");
			return null;
		}
		// NB: we ignore the abiSession value

		CompletableFuture<BlockchainConfig> retFut
			= new CompletableFuture<BlockchainConfig>();

		// must be called for self only, since we don't have peer Credentials
		if (!mvoId.equals(m_MVO.getMVOId())) {
			Exception self = new Exception(lbl + "was not called for self");
			retFut.completeExceptionally(self);
			return retFut;
		}

		// find our signing wallet credentials for this chain
		int numIdx = mvoId.indexOf("-") + 1;
		String mvoNum = mvoId.substring(numIdx);
		BlockchainConfig bConfig = new BlockchainConfig(chainId, chain, m_Log);
		Properties props = m_MVO.getProperties();
		String credTag = "SigningWallet-" + chainId;
		String passTag = "SigningWalletPass-" + chainId;
		String credPass = props.getProperty(passTag);
		if (credPass == null || credPass.isEmpty()) {
			m_Log.error(lbl + "no " + passTag + " property defined");
			Exception err = new Exception(lbl + "no " + passTag + " property");
			retFut.completeExceptionally(err);
			return retFut;
		}
		String credWallet = props.getProperty(credTag,
											  chainId + "-signwallet.json");
		if (credWallet == null || credWallet.isEmpty()) {
			m_Log.error(lbl + "no " + credWallet + " property defined");
			Exception err = new Exception(lbl + "no " + credWallet
										+ " property");
			retFut.completeExceptionally(err);
			return retFut;
		}
		String secDir = props.getProperty("RunDir", "");
		StringBuilder keyPath = new StringBuilder(128);
		if (secDir.isEmpty()) {
			keyPath.append("security");
		}
		else {
			keyPath.append(secDir + File.separator + "security");
		}
		keyPath.append(File.separator + mvoId);
		String walletDir = keyPath.toString();
		String walletFile = walletDir + File.separator + credWallet;
		Credentials credentials = null;
		try {
			credentials = WalletUtils.loadCredentials(credPass, credWallet);
		}
		catch (CipherException ce) {
			m_Log.error(lbl + "exception loading signing Credentials", ce);
			retFut.completeExceptionally(ce);
			return retFut;
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "exception loading signing Credentials", ioe);
			retFut.completeExceptionally(ioe);
			return retFut;
		}
		bConfig.configSigningAddress(credentials.getAddress());
		ECKeyPair ecKeys = credentials.getEcKeyPair();
		bConfig.configSigningKey(ecKeys.getPrivateKey());

		// set dummy value
		bConfig.setStaking("150000000000000000000000");
		// in reality the staking address would not be the same as signing addr
		bConfig.configStakingAddress(credentials.getAddress());

		// normal completion
		retFut.complete(bConfig);
		return retFut;
	}

	/**
	 * obtain all the eNFTs on-chain for a given account address
	 * @param chainId the chain Id per standard, used to select files
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @param acct the account address (as EIP-55)
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted)
	 */
	public Future<HashMap<String, NFTmetadata>> getAccountNFTs(long chainId,
															   Web3j abiSession,
														 	   String acct,
														 	boolean demintedOk)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".getAccountNFTs: ";
		if (acct == null || acct.isEmpty()) {
			m_Log.error(lbl + "missing acct Id");
			return null;
		}
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain Id");
			return null;
		}
		// NB: we don't use the passed abiSession, and ignore demintedOk

		CompletableFuture<HashMap<String, NFTmetadata>> retFut
			= new CompletableFuture<HashMap<String, NFTmetadata>>();

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null) {
			// don't have this chain defined
			Exception err2 = new Exception("chainId " + chainId
											+ " is not supported");
			retFut.completeExceptionally(err2);
			return retFut;
		}

		// build base path to account directory
		Properties props = m_MVO.getProperties();
		String acctPath
			= "eData" + File.separator + Keys.toChecksumAddress(acct);
		File acctDir = new File(acctPath);
		File nftDir = null;
		File[] eNFTfiles = null;
		boolean err = false;
		if (!acctDir.isDirectory()) {
			m_Log.error(lbl + acctDir + " is not a directory");
			err = true;
		}
		else {
			// move to ./chainId subdirectory
			nftDir = new File(acctDir, Long.toString(chainId));
			if (!nftDir.isDirectory()) {
				m_Log.error(lbl + nftDir + " is not a directory");
				err = true;
			}
			else {
				try {
					eNFTfiles = nftDir.listFiles((FilenameFilter)this);
				}
				catch (SecurityException se) {
					m_Log.error(lbl + "no permission to access " + nftDir);
					err = true;
				}
			}
		}
		if (eNFTfiles == null) {
			m_Log.error(lbl + "acctId " + acct
						+ ": could not access eNFTs in " + nftDir);
			err = true;
		}
		else {
			HashMap<String, NFTmetadata> eNFTlist
				= new HashMap<String, NFTmetadata>(eNFTfiles.length);
			for (int nnn = 0; nnn < eNFTfiles.length; nnn++) {
				File eNFTfile = eNFTfiles[nnn];
				if (!eNFTfile.exists() || !eNFTfile.canRead()) {
					m_Log.error(lbl + "cannot open eNFT file "
								+ eNFTfile + " for read");
					err = true;
					continue;
				}
				int eLen = (int) eNFTfile.length();
				char[] eBuff = new char[eLen+1];
				try {
					FileReader efr = new FileReader(eNFTfile);
					efr.read(eBuff, 0, eLen);
					efr.close();
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "unable to read eNFT file, "
								+ eNFTfile, ioe);
					err = true;
					continue;
				}
				String eNFTtxt = new String(eBuff);
				NFTmetadata eNFT = new NFTmetadata(m_Log);
				eNFT.buildFromJSON(eNFTtxt);
				// the filename is the ID
				String fName = eNFTfile.getName();
				int idx = fName.indexOf(".json");
				if (idx == -1) {
					m_Log.error(lbl + "unable to find file basename, " + fName);
					err = true;
					continue;
				}
				eNFTlist.put(fName.substring(0, idx), eNFT);
			}
			if (!err) {
				retFut.complete(eNFTlist);
			}
		}
		if (err) {
			Exception errRet
				= new Exception(lbl + "got errors processing files");
			retFut.completeExceptionally(errRet);
		}
		return retFut;
	}

	/**
	 * obtain a list of eNFTs, given their IDs
	 * @param chainId the chain Id per standard, used to select files
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @param acct the account address (as EIP-55)
	 * @param idList the list of eNFT IDs we want
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted)
	 */
	public Future<ArrayList<NFTmetadata>> getNFTsById(long chainId,
													  Web3j abiSession,
													  String acct,
													  ArrayList<String> idList,
													  boolean demintedOk)
	{
		final String lbl = this.getClass().getSimpleName() + ".getNFTsById: ";
		if (acct == null || acct.isEmpty()) {
			m_Log.error(lbl + "missing acct Id");
			return null;
		}
		if (idList == null || idList.isEmpty()) {
			m_Log.error(lbl + "missing idList");
			return null;
		}
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain Id");
			return null;
		}
		// NB: we don't use the abiSession, and demintedOk is always effectively
		// 	   false, because a deminted eNFT's file will not be found at all.
		// 	   The acctId is also irrelevant for this implementation.
		String acctId = Keys.toChecksumAddress(acct);

		CompletableFuture<ArrayList<NFTmetadata>> retFut
			= new CompletableFuture<ArrayList<NFTmetadata>>();

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null) {
			// don't have this chain defined
			Exception err2 = new Exception(lbl + "chainId " + chainId
											+ " is not supported");
			retFut.completeExceptionally(err2);
			return retFut;
		}

		// build base path
		Properties props = m_MVO.getProperties();
		String baseDir = props.getProperty("RunDir", "") + File.separator
						+ "eData" + File.separator;

		// look up each ID in the list in our local properties files
		ArrayList<NFTmetadata> eNFTlist
			= new ArrayList<NFTmetadata>(idList.size());
		boolean gotErr = false;
		// re-init the properties list in case another MVO changed it
		if (!init()) {
			gotErr = true;
		}
		for (String id : idList) {
			// find the owning account
			String ownerAcct = m_eNFTList.getProperty(id);
			if (ownerAcct == null) {
				m_Log.error(lbl + "eNFT ID " + id + " not found, skipping");
				gotErr = true;
				continue;
			}

			// build path to file, $RunDir/eData/$acct/$chainId/$id.json
			String ePath = baseDir + Keys.toChecksumAddress(ownerAcct)
							+ File.separator + Long.toString(chainId)
							+ File.separator + id + ".json";
			File eFile = new File(ePath);
			if (!eFile.exists() || !eFile.canRead()) {
				m_Log.error(lbl + "cannot open eNFT file, " + ePath);
				gotErr = true;
				continue;
			}
			int eLen = (int) eFile.length();
			char[] eBuff = new char[eLen+1];
			try {
				FileReader efr = new FileReader(eFile);
				efr.read(eBuff, 0, eLen);
				efr.close();
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "unable to read eNFT file, " + ePath, ioe);
				gotErr = true;
				continue;
			}
			String eNFTtxt = new String(eBuff);
			NFTmetadata eNFT = new NFTmetadata(m_Log);
			if (eNFT.buildFromJSON(eNFTtxt)) {
				eNFTlist.add(eNFT);
			}
			else {
				m_Log.error(lbl + "unable to build NFT for Id " + id);
				gotErr = true;
			}
		}
		if (gotErr) {
			Exception errRet = new Exception(lbl
											+ "got errors processing files");
			retFut.completeExceptionally(errRet);
		}
		else {
			retFut.complete(eNFTlist);
		}
		return retFut;
	}

	/**
	 * method to determine whether a given ID is used (including deminted eNFTs)
	 * @param chainId the chain Id per chainlist.org
	 * @param abiService the object at which we can access Web3j service
	 * @param id the eNFT Id
	 * @return true if an eNFT exists with that Id, else false (null on error)
	 */
	public Future<Boolean> checkIfIdUsed(long chainId,
										 Web3j abiService,
										 String id)
	{
		final String lbl = this.getClass().getSimpleName() + ".checkIfIdUsed: ";
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}
		if (id == null || id.isEmpty()) {
			m_Log.error(lbl + "missing eNFT Id");
			return null;
		}
		// NB: we ignore abiService
		CompletableFuture<Boolean> retFut = new CompletableFuture<Boolean>();

		// re-init the properties list in case another MVO changed it
		if (!init()) {
			Exception err = new Exception(lbl + "ID properties list not found");
			retFut.completeExceptionally(err);
			return retFut;
		}

		// look for the owning account
		Boolean used = Boolean.valueOf(false);
		String ownerAcct = m_eNFTList.getProperty(id);
		if (ownerAcct != null) {
			// was used
			used = Boolean.valueOf(true);
		}
		retFut.complete(used);
		return retFut;
	}

	// implement FilenameFilter
	/**
	 * method to select files in RunDir/eData/acctId that are ID.json
	 * @param dir the directory the file was found in
	 * @param name the name of the file
	 * @return true if we should accept this file
	 */
	public boolean accept(File dir, String name) {
		if (dir == null || name == null) {
			return false;
		}
		if (dir.getPath().contains("eData")) {
			if (name.endsWith(".json")) {
				return true;
			}
		}
		return false;
	}

	// implement FileFilter
	/**
	 * method to select files matching security/MVO-*.properties
	 * @param path the file to examine
	 * @return true if we should accept this file
	 */
	public boolean accept(File path) {
		if (path == null) {
			return false;
		}
		String name = path.getName();
		if (name.startsWith("MVO-") && name.endsWith(".properties")) {
			return true;
		}
		return false;
	}

	/**
	 * supplemental method to create an eNFT file (not in BlockchainAPI)
	 * @param ID the Id of the eNFT to create
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param owner the owning account
	 * @param contents the data to store (NFTmetadata.toString())
	 * @return true on successful creation
	 */
	public boolean createENFTfile(long chainId,
								  String ID,
								  String owner,
								  String contents)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".createENFTfile: ";
		if (ID == null || owner == null || ID.isEmpty() || owner.isEmpty()
			|| contents == null || contents.isEmpty() || chainId <= 0L)
		{
			m_Log.error(lbl + "missing input");
			return false;
		}

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null) {
			// don't have this chain defined
			m_Log.error(lbl + "chainId " + chainId + " is not supported");
			return false;
		}

		// build base path
		Properties props = m_MVO.getProperties();
		String acctPath = props.getProperty("RunDir", "") + File.separator
						+ "eData" + File.separator
						+ Keys.toChecksumAddress(owner) + File.separator
						+ Long.toString(chainId);
		File acctDir = new File(acctPath);
		if (!acctDir.exists()) {
			if (!acctDir.mkdirs()) {
				m_Log.error(lbl + "could not create dir " + acctDir);
				return false;
			}
		}
		String filePath = acctPath + File.separator + ID + ".json";
		File enftFile = new File(filePath);
		if (enftFile.exists()) {
			m_Log.error(lbl + filePath + " already exists");
			return false;
		}

		// create the file with the contents
		boolean err = false;
		try {
			FileWriter fw = new FileWriter(enftFile, false);
			fw.write(contents, 0, contents.length());
			fw.flush();
			fw.close();
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "unable to write to " + enftFile, ioe);
			err = true;
		}
		if (err) {
			return !err;
		}

		// add to properties file
		m_eNFTList.setProperty(ID, Keys.toChecksumAddress(owner));
		// flush props to disk
		if (!backupProps()) {
			err = true;
		}
		return !err;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_eNFTList != null) {
				m_eNFTList.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
