/*
 * last modified---
 * 	02-12-24 suppress SslContextFactory.Server setup, as we won't be using it
 * 	06-08-23 handle EIP-712 signed requests
 * 	05-01-23 handle AES-decrypting requests per ECIES algorithm
 * 	04-21-23 add doOptions(), handle decryption prior to determining payload
 * 	04-14-23 handle encrypting replies with the client's specified replyKey
 * 	03-20-23 encrypt normal replies with replyKey if client supplied one
 * 	03-17-23 handle decrypting the payload if indicated by a encrchain parameter
 * 	01-04-23 support CORS by adding CrossOriginFilter
 * 	07-12-22 improve error message labeling
 * 	06-21-22 supply AsyncContextState to MVOBrokerEntry; implement
 * 				  AsyncListener
 * 	04-07-22 add invocation of state handlers
 * 	03-15-22 move SslContextFactory config here from MVO
 * 	03-11-22 new
 *
 * purpose---
 * 	handle MVO Layer2 requests from dApp clients
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.servlet.ServletHandler;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.NetworkTrafficServerConnector;
import org.eclipse.jetty.server.AsyncContextState;
import org.eclipse.jetty.server.AsyncContextEvent;
import org.eclipse.jetty.util.ajax.JSON;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;

import java.util.Base64;
import java.util.Properties;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.SecureRandom;
import java.nio.charset.StandardCharsets;
import java.nio.ByteBuffer;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.AsyncContext;
import javax.servlet.AsyncListener;
import javax.servlet.AsyncEvent;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * This class is instantiated once per MVO and provides a listener and request
 * handler for client requests to make deposit, spend, and withdrawal actions.
 * All of these arrive on the Jetty servlet handler as POSTs.
 */
public final class ClientHandler extends HttpServlet implements AsyncListener {
	// BEGIN data members
	/**
	 * the owning MVO
	 */
	private MVO					m_MVOServer;

	/**
	 * the logging object (shared with owning MVO)
	 */
	private Log					m_Log;

	/**
	 * the port number at which we are to listen for incoming requests
	 */
	private final int			m_ListenPort;

	/**
	 * the secure connection factory used by connectors, built using our
	 * configured keystore
	 */
	private SslContextFactory.Server	m_SslFactory;

	/**
	 * the Jetty server used to listen for incoming messages
	 */
	private Server				m_JettyServer;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which owns us
	 * @param log the error/debug logger (normally mvo.log())
	 * @param listenPort the port at which we're to listen for requests
	 */
	public ClientHandler(MVO mvo, Log log, int listenPort) {
		super();
		m_MVOServer = mvo;
		m_Log = log;
		m_ListenPort = listenPort;
		m_SslFactory = new SslContextFactory.Server();
		// customize thread pool
		QueuedThreadPool qtp = new QueuedThreadPool(255, 8, 600000);
		m_JettyServer = new Server(qtp);
	}

	/**
	 * initialize the Jetty instance
	 * @return true on success
	 */
	public boolean initialize() {
		boolean secureOk = true;
	/* NB: wss TLS negotiation is broken in Jetty!  Also, we don't want to
	 * 	   require site certificates for MVOs, since they have no domains.
	 * 	   Therefore, we comment all of this out.
		// set up the SSL connector factory based on our keystore and properties
		Properties props = m_MVOServer.getProperties();
		String keystorePath = props.getProperty("ClientKeyStore");
		m_SslFactory.setKeyStorePath(keystorePath);
		String keystorePass = props.getProperty("KeyStorePass");
		m_SslFactory.setKeyStorePassword(keystorePass);
		String keystoreType = props.getProperty("KeyStoreType");
		m_SslFactory.setKeyStoreType(keystoreType);
		String truststorePath = props.getProperty("TrustStore");
		m_SslFactory.setTrustStorePath(truststorePath);
		String truststoreType = props.getProperty("TrustStoreType");
		m_SslFactory.setTrustStoreType(truststoreType);
		String trustStorePass = props.getProperty("TrustStorePass");
		if (trustStorePass != null) {
			m_SslFactory.setTrustStorePassword(trustStorePass);
		}
		String keyAlias = props.getProperty("ClientKeyAlias", "mvoclient");
		m_SslFactory.setCertAlias(keyAlias);
		m_SslFactory.setNeedClientAuth(false);
		m_SslFactory.setWantClientAuth(false);
		m_SslFactory.addExcludeProtocols("SSLv2Hello", "SSLv3");
		try {
			m_SslFactory.start();
		}
		catch (Exception eee) {
			m_Log.error("ClientHandler.initialize: unable to start SSL factory",
						eee);
			secureOk = false;
		}
	 */
		secureOk = false;

		if (secureOk) {
			final NetworkTrafficServerConnector httpsConnector
				= new NetworkTrafficServerConnector(m_JettyServer,
													m_SslFactory);
			httpsConnector.setHost(null);
			httpsConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpsConnector);
		}
		else {
			NetworkTrafficServerConnector httpConnector
				= new NetworkTrafficServerConnector(m_JettyServer);
			httpConnector.setHost(null);
			httpConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpConnector);
		}

		// define servlet handler
		ServletHandler handler = new ServletHandler();
		m_JettyServer.setHandler(handler);
		handler.addServletWithMapping(new ServletHolder(this), "/mvo");

		// so we can support CORS requests, add filter with defaults
		handler.addFilterWithMapping(
						"org.eclipse.jetty.servlets.CrossOriginFilter", "/*",
							org.eclipse.jetty.servlet.FilterMapping.DEFAULT);

		// attempt to start the server and handler
		try {
			m_JettyServer.start();
			handler.initialize();
			m_Log.debug("ClientHandler.initialize: started ClientHandler Jetty "
						+ "listener (secure=" + secureOk + ") on port "
						+ m_ListenPort);
		}
		catch (Exception e) {
			m_Log.error("ClientHandler.initialize: could not start Jetty "
						+ "listener for ClientHandler", e);
			return false;
		}
		return true;
	}

	/**
	 * shutdown Jetty listener
	 */
	public void shutdown() {
		// shut down the listener
		if (m_JettyServer.isRunning()) {
			try {
				m_JettyServer.stop();
				m_Log.debug("ClientHandler.shutdown: Jetty listener stopped");
			}
			catch (Exception e) {
				m_Log.error("ClientHandler.shutdown: error stopping "
							+ "Jetty listener", e);
			}
		}

		// shut down the SSL factory
		if (m_SslFactory != null && m_SslFactory.isRunning()) {
			try {
				m_SslFactory.stop();
			}
			catch (Exception ee) {
				m_Log.error("ClientHandler.shutdown: exception stopping SSL "
							+ "factory", ee);
			}
		}
	}

	/**
	 * obtain the Jetty server
	 * @return the Jetty server object we use to handle requests
	 */
	public Server getJettyServer() { return m_JettyServer; }

	// methods to extend the HttpServlet class
	/**
	 * handler method for GETs from the dApp clients (not used)
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doGet(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ClientHandler.doGet: null dApp request");
		}
		if (response == null) {
			throw new IOException("ClientHandler.doGet: null dApp response "
								+ "object");
		}

		// GET is not used
		//response.addHeader("Access-Control-Allow-Origin", "*");
		response.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED,
							"GET not supported for this function");
	}

	/**
	 * handler for OPTIONS preflights from dApp clients
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doOptions(HttpServletRequest request,
						  HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ClientHandler.doOptions: null dApp receipt "
								+ "request");
		}
		if (response == null) {
			throw new IOException("ClientHandler.doOptions: null dApp receipt "
								+ "response object");
		}

		// echo back same origin we were sent
		response.addHeader("Access-Control-Allow-Origin",
							request.getHeader("ORIGIN"));
		response.addHeader("Access-Control-Allow-Headers", "*");
		response.addHeader("Access-Control-Allow-Methods",
							"POST, HEAD, TRACE, OPTIONS");
		response.setStatus(HttpServletResponse.SC_NO_CONTENT);
		response.flushBuffer();
	}

	/**
	 * handler method for POSTs from the dApp clients
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doPost(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ClientHandler.doPost: null dApp reqeust");
		}
		if (response == null) {
			throw new IOException("ClientHandler.doPost: null dApp response "
								+ "object");
		}
		final String lbl = this.getClass().getSimpleName() + ".doPost: ";

		response.setContentType("text/html");
		//response.addHeader("Access-Control-Allow-Origin", "*");
		response.setStatus(HttpServletResponse.SC_OK);

		/* See whether the payload has been encrypted.  This is indicated by
		 * sending the encrchain as a separate POST parameter.  In this case we
		 * must decrypt with the private ECIES key for the indicated chain
		 * (matching the pubkey shown in the MVO config).
		 */
		String payload = "";
		String parsePayload = payload;
		String opcode = "";
		String chainParm = request.getHeader("encrchain");
		ClientMVOBlock cBlock = new ClientMVOBlock(request, m_Log);
		String origDappReq = "";
		Long chId = null;
		boolean encrypted = chainParm != null;
		if (encrypted) {
			try {
				chId = Long.parseLong(chainParm.trim());
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "bad encryption encrchain passed, "
							+ chainParm);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Invalid encrchain parameter value, " + chainParm);
				return;
			}

			// get our key for this chain
			BlockchainConfig bcConf
				= m_MVOServer.getConfig().getChainConfig(chId);
			if (bcConf == null) {
				m_Log.error(lbl + "no BlockchainConfig found for encrchain "
							+ chId);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Unsupported encrchain, " + chId);
				return;
			}
			PrivateKey ecPriv = bcConf.getECPrivKey();
			if (ecPriv == null) {
				m_Log.error(lbl + "no EC privkey found for encrchain " + chId);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Decrypt key not found for  encrchain " + chId);
				return;
			}

			// obtain the data, which will be a single line of Base64Url
			BufferedReader reqStream = null;
			try {
				reqStream = request.getReader();

				// allow for possible chunking
				String line = "";
				do {
					line = reqStream.readLine();
					if (line != null && !line.isEmpty()) {
						payload += line;
					}
				} while (line != null);
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "error reading encrypted request data", ioe);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Error reading encrypted request data");
			}

			// decrypt payload using this key
			String decPayload = EncodingUtils.decWithECIES(ecPriv, payload);
			if (decPayload == null) {
				m_Log.error(lbl + "error decrypting payload for chain " + chId);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Error decrypting ECIES payload");
				return;
			}

			// JSON parse this data
			Object parsedDec = null;
			try {
				parsedDec = JSON.parse(decPayload);
			}
			catch (IllegalStateException ise) {
				m_Log.error(lbl + "decrypted JSON did not parse: \""
							+ decPayload + "\"", ise);
			}
			if (!(parsedDec instanceof Map)) {
				m_Log.error(lbl + "decrypted JSON was not a Map");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Parse error on decrypted JSON");
				throw new ServletException("Parse error on decrypted JSON");
			}
			Map decMap = (Map) parsedDec;

			/* Record entire decrypted payload as original, including EIP-712
			 * data plus the client's signature.  This must be passed to the
			 * AUD key server so it can validate the original client request,
			 * and to every AUD node.
			 */
			origDappReq = decPayload;

			// find message.requestJson, which should be a Map
			Object messageObj = decMap.get("message");
			if (!(messageObj instanceof Map)) {
				m_Log.error(lbl + "could not find message in signed data");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Could not find message in signed data");
				throw new ServletException("Missing message signed data");
			}
			Map messageMap = (Map) messageObj;
			Object requestJson = messageMap.get("requestJson");
			if (!(requestJson instanceof Map)) {
				m_Log.error(lbl + "could not find message.requestJson");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Could not find message.requestJson data");
				throw new ServletException("Missing message.requestJson data");
			}
			Map jsonMap = (Map) requestJson;

			// look for one of the three request payloads we support as mapping
			Map decPayloadMap = null;
			Object depReq = jsonMap.get("depositspec");
			if (depReq instanceof Map) {
				decPayloadMap = (Map) depReq;
				opcode = cBlock.M_OpDeposit;
			}
			else {
				Object spendReq = jsonMap.get("spendspec");
				if (spendReq instanceof Map) {
					decPayloadMap = (Map) spendReq;
					opcode = cBlock.M_OpSpend;
				}
				else {
					Object withReq = jsonMap.get("withdrawspec");
					if (withReq instanceof Map) {
						decPayloadMap = (Map) withReq;
						opcode = cBlock.M_OpWithdraw;
					}
				}
			}
			if (decPayloadMap == null) {
				// nothing else is valid
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"No recognized payload type");
				m_Log.error(lbl + "no recognized payload type");
				throw new ServletException("No recognized payload type");
			}

			// convert back to a string -- this will reorder the map elements
			parsePayload = JSON.toString(decPayloadMap);
		}
		else {
			// unencrypted, so look for request parameters (note: unsigned)
			payload = request.getParameter("depositspec");
			if (payload != null && !payload.isEmpty()) {
				opcode = cBlock.M_OpDeposit;
			}
			else {
				payload = request.getParameter("spendspec");
				if (payload != null && !payload.isEmpty()) {
					opcode = cBlock.M_OpSpend;
				}
				else {
					payload = request.getParameter("withdrawspec");
					if (payload != null && !payload.isEmpty()) {
						opcode = cBlock.M_OpWithdraw;
					}
					else {
						// nothing else is valid
						sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
									"No recognized payload type");
						m_Log.error(lbl + "no recognized payload type");
						throw new ServletException(
												"No recognized payload type");
					}
				}
			}
			parsePayload = payload;
			origDappReq = payload;
		}

		/* Three kinds of request payloads can arrive here:
		 * depositspec - a deposit request, JSON forming a ClientMVOBlock
		 *				 with {opcode}=deposit
		 * spendspec - a spend request, with JSON forming a ClientMVOBlock
		 * 			   with {opcode}=spend
		 * withdrawspec - a withdrawal request, with JSON forming a
		 * 				  ClientMVOBlock with {opcode}=withdraw
		 * We will return an OperationsBlock after processing.
		 */
		cBlock.setDecryptedPayload(origDappReq);

		// parse the dApp client's payload, which should be valid JSON
		if (!cBlock.buildFromString(parsePayload)) {
			m_Log.error(lbl + "could not parse opcode "
						+ opcode + ", payload = \"" + parsePayload + "\"");
			sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
						"Payload for " + opcode + " did not parse as JSON");
			return;
		}

		// verify the opcode matches the one stipulated by parameter name
		if (!cBlock.getOpcode().equals(opcode)) {
			// do not attempt to continue in this case
			m_Log.error(lbl + "dApp opcode " + cBlock.getOpcode()
						+ " received, but " + opcode
						+ " expected from payload name");
			opcode = cBlock.getOpcode();
			sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
						"opcode " + opcode + " inconsistent with payload name");
			return;
		}

		// verify the chainId inside matches the parameter, if we got one
		if (chId != null && cBlock.getChainId() != chId.longValue()) {
			// do not attempt to continue in this case
			m_Log.error(lbl + "dApp chainId " + cBlock.getChainId()
						+ " received, but " + chId
						+ " expected from parameter value");
			sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
						"chainId " + cBlock.getChainId()
						+ " inconsistent with parameter value");
			return;
		}

		// create a broker map entry for this request
		MVOBrokerEntry mvoBroEnt = new MVOBrokerEntry(cBlock, m_MVOServer);
		DepositState depSt = null;
		SpendState spSt = null;
		WithdrawState withSt = null;
		if (opcode.equals(cBlock.M_OpDeposit)) {
			// deposit
			depSt = new DepositState(mvoBroEnt);
			depSt.setLead(true);
			mvoBroEnt.setStateTracker(depSt);
		}
		else if (opcode.equals(cBlock.M_OpSpend)) {
			// spend
			spSt = new SpendState(mvoBroEnt);
			spSt.setLead(true);
			mvoBroEnt.setStateTracker(spSt);
		}
		else if (opcode.equals(cBlock.M_OpWithdraw)) {
			// withdraw
			withSt = new WithdrawState(mvoBroEnt);
			withSt.setLead(true);
			mvoBroEnt.setStateTracker(withSt);
		}
		else {
			// impossible opcode, shouldn't have parsed
			m_Log.error(lbl + "impossible opcode on request, "
						+ opcode + ", json = \"" + parsePayload + "\"");
			sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
						"Unsupported opcode: " + opcode);
			return;
		}

		// confirm signature on dApp request
		if (encrypted && !verifyClientSig(origDappReq, cBlock)) {
			m_Log.error(lbl + "signature did not verify, opcode "
						+ opcode + ", json = \"" + origDappReq + "\"");
			sendErrRep(response, HttpServletResponse.SC_UNAUTHORIZED,
						"Signature verification failure on " + opcode
						+ " request");
			return;
		}

		// to permit async request processing, create a context from the req/rep
		AsyncContext async = null;
		try {
			async = request.startAsync();
		}
		catch (IllegalStateException ise) {
			m_Log.error(lbl + "could not start async processing", ise);
			sendErrRep(response, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						"Could not switch to async request processing");
			return;
		}
		mvoBroEnt.setAsyncContext(async);
		// set a 1 min timeout
		async.setTimeout(60000L);
		// add listener (for timeouts)
		async.addListener(this);

		// since request appears legit, create transId and add to broker map
		MVOState stObj = m_MVOServer.getStateObj();
		BigInteger val = new BigInteger(256, stObj.getRNG());
		String transId = val.toString();
		mvoBroEnt.setTransId(transId);
		Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();
		brMap.put(transId, mvoBroEnt);
		
		// record progress (we have parsed and verified sig) and advance step
		boolean advOk = true;
		if (depSt != null) {
			depSt.recordSuccess(DepositState.M_ReqReceived);
			// go on to verify request vs SC config
			advOk = depSt.advanceState(DepositState.M_ReqReceived,
									   DepositState.M_ReqVerified,
					 				   ReqSvcState.M_NoFailure, "");
		}
		else if (spSt != null) {
			spSt.recordSuccess(SpendState.M_ReqReceived);
			// go on to verify request vs SC config
			advOk = spSt.advanceState(SpendState.M_ReqReceived,
									  SpendState.M_ReqVerified,
								 	  ReqSvcState.M_NoFailure, "");
		}
		else if (withSt != null) {
			withSt.recordSuccess(WithdrawState.M_ReqReceived);
			// go on to verify request vs SC config
			advOk = withSt.advanceState(WithdrawState.M_ReqReceived,
										WithdrawState.M_ReqVerified,
										ReqSvcState.M_NoFailure, "");
		}
		else {
			m_Log.error(lbl + "impossible, no state object");
			throw new ServletException("Missing state tracker object");
		}
	}

	/**
	 * method to supply servlet information
	 * @return the name of the servlet (arbitrary)
	 */
	@Override
	public String getServletInfo() {
		return "MVO-Jetty:ClientHandler servlet";
	}

	// utility methods
	/**
	 * obtain the SSL factory
	 * @return the factory
	 */
	public SslContextFactory.Server getSslFactory() { return m_SslFactory; }

	/**
	 * send a correctly formatted HTTP error response to a POST
	 * @param resp the response for the reply
	 * @param sc the status code (SC_* constant in HttpServletResponse)
	 * @param errTxt the error message to return to the client
	 */
	public void sendErrRep(HttpServletResponse resp, int sc, String errTxt) {
		if (resp == null || errTxt == null) {
			m_Log.error("ClientHandler.sendErrRep: missing input");
			return;
		}
		try {
			resp.setContentType("text/html");
			//resp.addHeader("Access-Control-Allow-Origin", "*");
			resp.sendError(sc, errTxt);
		}
		catch (IOException ioe) {
			m_Log.error("ClientHandler.sendErrRep: exception sending error "
						+ "reply " + errTxt, ioe);
		}
		catch (IllegalStateException ise) {
			m_Log.error("ClientHandler.sendErrRep: exception sending error "
						+ "reply " + errTxt, ise);
		}
	}

	/**
	 * validate client signature on an EIP712-signed request
	 * @param requestJson the entire JSON string received from the user
	 * @param cliReq the parsed client block containing the signature
	 * @return true if client's signature was valid
	 */
	private boolean verifyClientSig(String requestJson, ClientMVOBlock cliReq) {
		if (requestJson == null || requestJson.isEmpty() || cliReq == null) {
			m_Log.error("ClientHandler.verifyClientSig: missing input");
			return false;
		}

		/* to extract the signed portion, take everything up to the last:
		 * ',\"signature\":'
		 */
		int sigIdx = requestJson.lastIndexOf(",\"signature\":");
		if (sigIdx == -1) {
			// json was not signed
			m_Log.error("ClientHandler.verifyClientSig: missing signature");
			return false;
		}
		String eip712Data = requestJson.substring(0, sigIdx) + "}";

		// obtain the signature from full input data
		String rawSig = requestJson.substring(sigIdx);
		int sigStart = rawSig.lastIndexOf(":\"");
		// strip off leading :" and trailing "}
		String signature = rawSig.substring(sigStart+2, rawSig.length()-2);
		String signingAddress = cliReq.getSender();

		// determine which address actually signed the data
		String actualSigAddr
			= EncodingUtils.getEIP712SignerAddress(eip712Data, signature);
		return actualSigAddr.equalsIgnoreCase(signingAddress);
	}

	/**
	 * send a correctly formatted JSON reply to a client POST
	 * @param resp the response for the reply
	 * @param opBlock the complete object (signed) to send
	 * @param repKey client's AES key to encrypt the text back to client (opt)
	 */
	public boolean sendNormalRep(HttpServletResponse resp,
								 OperationsBlock opBlock,
								 String repKey)
	{
		final String lbl = this.getClass().getSimpleName() + ".sendNormalRep: ";
		if (resp == null || opBlock == null || repKey == null) {
			m_Log.error(lbl + "missing input");
			return false;
		}
		resp.setContentType("application/json");
		//resp.addHeader("Access-Control-Allow-Origin", "*");
		resp.setStatus(HttpServletResponse.SC_OK);

		// obtain the writer and append the data to it
		try {
			PrintWriter writer = resp.getWriter();
			StringBuilder reply = new StringBuilder(10240);
			opBlock.addJSON(reply);

			// see if we need to encrypt our reply to the client's requested key
			if (repKey.isEmpty()) {
				writer.append(reply);
			}
			else {
				// convert back to bytes from Base64
				Base64.Decoder b64d = Base64.getUrlDecoder();
				byte[] keyBytes = b64d.decode(repKey);

				// verify we have an AES-256 size key
				if (keyBytes.length != 32) {
					m_Log.error(lbl + "bad length on replyKey, "
								+ keyBytes.length);
					return false;
				}
				SecureRandom rng = m_MVOServer.getStateObj().getRNG();
				SecretKey aesKey = new SecretKeySpec(keyBytes, "AES");
				String encReply = EncodingUtils.encWithAES(aesKey,
														   reply.toString(),
														   rng);
				if (encReply == null) {
					m_Log.error(lbl + "error encrypting client reply to key "
								+ repKey);
					return false;
				}
				else {
					writer.append(encReply);
				}
			}
			resp.flushBuffer();
		}
		catch (NumberFormatException nfe) {
			m_Log.error(lbl + "replyKey was not hex: " + repKey, nfe);
			return false;
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "error writing response", ioe);
			return false;
		}
		return true;
	}

	// methods to implement AsyncListener
	/**
	 * notification that an async operation has completed
	 * @param event the event describing the async operation
	 */
	public void onComplete(AsyncEvent event) {
		// normal; nothing to do here unless there's an exception
		Throwable thrown = event.getThrowable();
		if (thrown != null) {
			m_Log.error("ClientHandler.onComplete() with exception present",
						thrown);
		}
	}

	/**
	 * Notification that an async operation has timed out.  Normally this
	 * indicates that a peer never gave us a reply, so we have to report that
	 * and trigger a response to the client.
	 * @param event the event describing the async operation
	 */
	public void onTimeout(AsyncEvent event) {
		// find the broker entry which contains our context
		AsyncContext repContext = event.getAsyncContext();
		MVOState stObj = m_MVOServer.getStateObj();
		Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();
		boolean found = false;
		MVOBrokerEntry brEnt = null;
		for (String tId : brMap.keySet()) {
			brEnt = brMap.get(tId);
			AsyncContext cont = brEnt.getDappReply();
			if (cont == repContext) {
				found = true;
				break;
			}
		}
		if (found) {
			/* Look for an outstanding request without a reply from an MVO.  If
			 * one is found, we register a M_UnavailMVO error.
			 *
			 * If none is found, look for an outstanding request to an Auditor.
			 * If found, we register a M_UnavailAud error.
			 *
			 * Otherwise, we log an error and register a M_ProcError by default.
			 * There can be circumstances where we had trouble getting data
			 * from a blockchain, in which case it's M_UnavailABI instead.
			 */
			ConcurrentHashMap<String, String> mvoReqs
				= brEnt.getBrokerMVOReqs();
			ConcurrentHashMap<String, String> mvoReps
				= brEnt.getBrokerMVOReps();
			// assume MVO timed out
			int errType = ReqSvcState.M_UnavailMVO;
			String guiltyParty = "MVO node";
			if (mvoReqs.size() <= mvoReps.size()) {
				// wasn't MVO, try Auditor
				ConcurrentHashMap<String, String> audReqs
					= brEnt.getBrokerAudReqs();
				ConcurrentHashMap<String, String> audReps
					= brEnt.getBrokerAudReps();
				if (audReqs.size() <= audReps.size()) {
					// was neither
					errType = ReqSvcState.M_ProcError;
					guiltyParty = "Partner node (or NPE)";
				}
				else {
					// was Auditor
					errType = ReqSvcState.M_UnavailAud;
					guiltyParty = "Auditor node";
				}
			}

			ReqSvcState stateEngine = brEnt.getStateTracker();
			int endState = 0;
			// examine type of engine (note every possible type can occur here)
			DepositState depEng = null;
			WithdrawState withEng = null;
			SpendState spendEng = null;
			WalletState wallEng = null;
			ReceiptState rctEng = null;
			int step = stateEngine.getCurrentStep();
			if (stateEngine instanceof DepositState) {
				depEng = (DepositState) stateEngine;
				endState = DepositState.M_Replied;
				// make errors more specific if possible
				if (step == DepositState.M_ReceiptGen) {
					errType = ReqSvcState.M_ExtUnavail;
					guiltyParty = "Receipt store";
				}
				// NB: M_Deposited occurs after dApp reply was already sent
				else if (step == DepositState.M_Deposited
						 || step == DepositState.M_ReqReceived)
				{
					errType = ReqSvcState.M_UnavailABI;
					guiltyParty = "Blockchain API";
				}
			}
			else if (stateEngine instanceof WithdrawState) {
				withEng = (WithdrawState) stateEngine;
				endState = WithdrawState.M_Replied;
			}
			else if (stateEngine instanceof SpendState) {
				spendEng = (SpendState) stateEngine;
				endState = SpendState.M_Replied;
			}
			else if (stateEngine instanceof WalletState) {
				wallEng = (WalletState) stateEngine;
				endState = WalletState.M_Replied;
			}
			else if (stateEngine instanceof ReceiptState) {
				rctEng = (ReceiptState) stateEngine;
				endState = ReceiptState.M_Replied;
			}
			else {
				m_Log.error("ClientHandler.onTimeout: unexpected state engine "
							+ "type, " + stateEngine.getClass().getName());
				// we always have to complete nonetheless
				repContext.complete();
				return;
			}

			// move from current state to appropriate error
			if (stateEngine.advanceState(step, endState, errType,
										 "Timeout from " + guiltyParty))
			{
				m_Log.debug("ClientHandler.onTimeout: successful processing of "
							+ "errcode " + errType + " (" + guiltyParty
							+ ") by state engine "
							+ stateEngine.getClass().getName());
			}
			else {
				m_Log.error("ClientHandler.onTimeout: unsuccessful processing "
							+ "of errcode " + errType + " (" + guiltyParty
							+ ") by state engine "
							+ stateEngine.getClass().getName());
			}
			/* NB: we expect the endState to trigger the call to complete(),
			 * 	   after sending a reply to the dApp user.
			 */
		}
		else {
			m_Log.error("ClientHandler.onTimeout: found no match for async "
						+ "context");
			// we always have to complete nonetheless
			repContext.complete();
		}
	}

	/**
	 * notification that an async operation has terminated with an error
	 * @param event the event describing the async operation
	 */
	public void onError(AsyncEvent event) {
		// fetch and log any exception reported
		Throwable thrown = event.getThrowable();
		if (thrown == null) {
			m_Log.error("ClientHandler.onError: Async terminated with error "
						+ "(probably a timeout did not call complete())");
		}
		else {
			m_Log.error("ClientHandler.onError: Async terminated with error "
						+ "(probably a timeout did not call complete())",
						thrown);
		}
	}

	/**
	 * notification that an async operation has been started
	 * @param event the event describing the async operation
	 */
	public void onStartAsync(AsyncEvent event) {
		// nothing to do here
		m_Log.warning("ClientHandler.onStartAsync: started with pre-existing "
					+ "listener");
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			m_SslFactory = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
