/*
 * last modified---
 * 	02-25-22 new
 *
 * purpose---
 * 	map Jetty logging interface onto Log interface
 */

package cc.enshroud.jetty.log;

import org.eclipse.jetty.util.log.Logger;


/**
 * A Logger implementation to override the default Jetty logging behavior.
 * All log statements are written to the MVO/Auditor server logs. Info level
 * logging is sent to debug.  Anything with an exception is sent to error.
 */
public class JettyLog implements Logger {
	/**
	 * the actual logging object we use
	 */
	private static Log		m_Log;
	
	/**
	 * whether debug is enabled (this produces lots of output from Jetty)
	 */
	private boolean			m_Debug;

	/**
	 * nullary constructor (caller must set System.PEER_ID property first)
	 */
	public JettyLog() {
		String peerId = System.getProperty("PEER_ID");
		m_Log = new Log("/var/log/enshroud/" + peerId + "-jetty.log");
		m_Log.Init();
	}

	// methods to implement Jetty's Logger interface
	/**
	 * determine if debug is enabled
	 * @return current setting of debug
	 */
	@Override
	public boolean isDebugEnabled() {
		return m_Debug;
   }

	/**
	 * obtain name of this logging class
	 * @return the name of this logger
	 */
	@Override
	public String getName() {
		return "JettyLog";
	}

	/**
	 * set debug enabled
	 * @param enabled as specified
	 */
	@Override
	public void setDebugEnabled(boolean enabled) {
		m_Debug = enabled;
	}

	/**
	 * log an info message, mapped to debug
	 * @param message the message to log
	 */
	public void info(String message) {
      // send info log messages to debug
		if (message != null) {
			m_Log.debug(message);
		}
	}

	/**
	 * log an info message, with associated objects
	 * @param msg the message
	 * @param args the objects
	 */
	@Override
	public void info(String msg, Object... args) {
		// send info log messages to debug
		m_Log.debug(format(msg, args));
	}

	/**
	 * log an info message, with an exception
	 * @param msg the message
	 * @param ee the exception
	 */
	@Override
	public void info(String msg, Throwable ee) {
		m_Log.warning(msg, ee);
	}

	/**
	 * log an info exception (mapped to error)
	 * @param ee the exception
	 */
	@Override
	public void info(Throwable ee) {
		m_Log.error("exception occurred", ee);
	}
	
	/**
	 * log a debug message
	 * @param msg the message
	 */
	public void debug(String msg) {
		if (msg != null) {
			m_Log.debug(msg);
		}
	}

	/**
	 * log a debug message, with an exception (mapped to error)
	 * @param msg the message
	 * @param ee the exception
	 */
	@Override
	public void debug(String msg, Throwable ee) {
		if (m_Debug) {
			m_Log.error(msg, ee);
		}
	}

	/**
	 * log a debug exception (mapped to error)
	 * @param ee the exception
	 */
	public void debug(Throwable ee) {
		if (m_Debug) {
			m_Log.error("exception occurred", ee);
		}
	}
	
	/**
	 * log a debug message, with associated objects
	 * @param msg the message
	 * @param args the objects
	 */
	@Override
	public void debug(String msg, Object... args) {
		if (m_Debug) {
			m_Log.debug(format(msg, args));
		}
	}

	/**
	 * log a debug message, with associated long value
	 * @param msg the message
	 * @param lVal long value
	 */
	@Override
	public void debug(String msg, long lVal) {
		if (m_Debug) {
			m_Log.debug(msg + ": " + lVal);
		}
	}

	/**
	 * log a warning message
	 * @param msg the message
	 */
	public void warn(String msg) {
		m_Log.warning(msg);
	}

	/**
	 * log a warning message with associated objects
	 * @param msg the message
	 * @param args the objects
	 */
	@Override
	public void warn(String msg, Object... args) {
		m_Log.warning(format(msg, args));
	}

	/**
	 * log a warning message, with an exception
	 * @param msg the message
	 * @param ee the exception
	 */
	@Override
	public void warn(String msg, Throwable ee) {
		m_Log.warning(msg, ee);
	}

	/**
	 * log a warning exception (mapped to error)
	 * @param ee the exception
	 */
	@Override
	public void warn(Throwable ee) {
		m_Log.error("exception occurred", ee);
	}
	
	/**
	 * obtain a default logger object
	 * @param name the label for the logger
	 */
	@Override
	public Logger getLogger(String name) {
		return new JettyLog();
	}

	/**
	 * ignore an exception (mapped to debug)
	 * @param ee the exception
	 */
	@Override
	public void ignore(Throwable ee) {
		if (ee == null) {
			return;
		}
		String msg = ee.getMessage();
		/* Nix the following message, which appears to occur when site certs
		 * are used which have been signed using SHA256withRSA.  It is not
		 * actually a problem, but an artifact of browser clients closing a
		 * connection ungracefully and failing a keep-alive ping from server.
		 */
		if (msg != null && !msg.contains(": possible truncation attack")) {
			m_Log.debug(msg);
		}
	}

	/**
	 * utility method to format a message alongside its objects
	 * @param msg the text message
	 * @param args the objects to format
	 * @return the formatted string representation
	 */
	private String format(String msg, Object... args) {
		if (args.length == 0) {
			return msg;
		}
		StringBuilder out = new StringBuilder(256);
		int sub = msg.indexOf("{}");
		out.append(msg.substring(0, sub));
		for (int iii = 0; iii < args.length; iii++) {
			out.append(args[iii]);
			int oldsub = sub;
			sub = msg.indexOf("{}", oldsub+2);
			if (sub > 0) {
				out.append(msg.substring(oldsub+2, sub));
			}
			else {
				out.append(msg.substring(oldsub+2));
			}
		}
		return out.toString();
	}
}
