/*
 * last modified---
 * 	10-11-22 add m_DetailsHash and methods, M_Deleted
 * 	10-10-22 add m_Narrative and methods
 * 	09-26-22 new
 *
 * purpose--
 * 	provide a Java bean for a record in the ENSH_ENFT_status table
 */

package cc.enshroud.jetty.aud.db;

import java.util.Date;


/**
 * This class supplies an instantiation of a row in the ENSH_ENFT_STATUS
 * table.  It has bean methods to get and set all fields.
 */
public final class EnftStatus {
	// BEGIN data members
	// constants
	/**
	 * status meaning that the eNFT should be treated as valid (default)
	 */
	public static final String	M_Valid;
	/**
	 * status meaning that the eNFT is believed to be possibly invalid, but
	 * has not yet been marked as greylisted in the chain's smart contract
	 */
	public static final String	M_Suspect;
	/**
	 * status meaning that the eNFT is believed to be invalid, and has been
	 * added to the greylist in the smart contract on m_Chain
	 */
	public static final String	M_Blocked;
	/**
	 * status meaning that the eNFT has been deminted and is no longer listed
	 * in the mapping on-chain
	 */
	public static final String	M_Deleted;

	// initialize
	static {
		M_Valid = "valid";
		M_Suspect = "suspect";
		M_Blocked = "blocked";
		M_Deleted = "deleted";
	}

	/**
	 * the unique ID, a zero-padded 64-digit hex number
	 */
	private String		m_ID;

	/**
	 * the chain ID on which the eNFT was issued
	 */
	private long		m_Chain;

	/**
	 * the details hash of the eNFT, defined as:
	 * keccak256("[{address},{id},{asset},{amount}]")
	 */
	private String		m_DetailsHash;

	/**
	 * the current status value, one of valid|suspect|blocked, set originally
	 * at m_InsertDate and possibly reset at m_LmodDate
	 */
	private String		m_Status;

	/**
	 * the initial insertion date (in the database, not on the blockchain --
	 * it is possible that the eNFT will _never_ be emitted on the chain)
	 */
	private Date		m_InsertDate;

	/**
	 * the ID of the Auditor node which originally (first) inserted the record
	 * (of the form AUD-NNN)
	 */
	private String		m_Submitter;

	/**
	 * the justification for setting suspect|blocked status, typically an
	 * extract from the Auditor's error log
	 */
	private String		m_Narrative;

	/**
	 * the latest modification date, if any
	 */
	private Date		m_LmodDate;

	/**
	 * the ID of the Auditor node which made the most recent modification if any
	 */
	private String		m_Updater;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param Id the ID value assigned to the eNFT (unique on its chain)
	 * @param chain the ID of the relevant blockchain
	 */
	public EnftStatus(String Id, long chain) {
		this();
		setID(Id);
		setChain(chain);
	}

	/**
	 * nullary constructor
	 */
	public EnftStatus() {
		m_ID = m_Submitter = m_Updater = m_Narrative = m_DetailsHash = "";
		m_Status = M_Valid;
	}

	// GET methods
	/**
	 * fetch the Id
	 * @return the Id of the eNFT
	 */
	public String getID() { return m_ID; }

	/**
	 * fetch the chain ID
	 * @return the ID of the chain (per standards) where the eNFT exists
	 */
	public long getChain() { return m_Chain; }

	/**
	 * fetch the details hash
	 * @return the hash value
	 */
	public String getDetailsHash() { return m_DetailsHash; }

	/**
	 * fetch the status
	 * @return the status value
	 */
	public String getStatus() { return m_Status; }

	/**
	 * fetch the original insertion time
	 * @return the timestamp when the record was added to the database (could
	 * be null)
	 */
	public Date getInsertDate() { return m_InsertDate; }

	/**
	 * fetch the original submitter
	 * @return the Auditor ID who created the record
	 */
	public String getSubmitter() { return m_Submitter; }

	/**
	 * fetch the justification for marking the record suspect or blocked
	 * @return the Auditor log extract explaining the rationale
	 */
	public String getNarrative() { return m_Narrative; }

	/**
	 * fetch the modification time
	 * @return the timestamp when the record was last updated (could be null)
	 */
	public Date getLmodDate() { return m_LmodDate; }

	/**
	 * fetch the last updater
	 * @return the Auditor ID which last modified the record (may differ from
	 * m_Submitter)
	 */
	public String getUpdater() { return m_Updater; }


	// SET methods
	/**
	 * set the hash
	 * @param Id the eNFT ID value
	 */
	public void setID(String Id) {
		if (Id != null) {
			m_ID = Id;
		}
	}

	/**
	 * set the chain Id
	 * @param chain the ID of the blockchain
	 */
	public void setChain(long chain) {
		if (chain > 0L) {
			m_Chain = chain;
		}
	}

	/**
	 * set the details hash value (separately)
	 * @param hash the value to set
	 */
	public void setDetailsHash(String hash) {
		if (hash != null) {
			m_DetailsHash = hash;
		}
	}

	/**
	 * set the status value (separately)
	 * @param stat the value to set (database should guarantee within ENUM)
	 */
	public void setStatus(String stat) {
		if (stat != null) {
			m_Status = stat;
		}
	}

	/**
	 * set the submitting Auditor (separately)
	 * @param aud the Auditor who first submitted the record
	 */
	public void setSubmitter(String aud) {
		if (aud != null && !aud.isEmpty()) {
			m_Submitter = aud;
		}
	}

	/**
	 * set the Auditor's reasons for marking the ID with bad status
	 * @param reasons typically an extract from the Auditor's error log
	 */
	public void setNarrative(String reasons) {
		if (reasons != null && !reasons.isEmpty()) {
			m_Narrative = reasons;
		}
	}

	/**
	 * set the record creation date (separately)
	 * @param ins the insert date/time (can be null)
	 */
	public void setInsertDate(Date ins) {
		m_InsertDate = ins;
	}

	/**
	 * set the record updater (separately)
	 * @param aud the Auditor who last updated the record
	 */
	public void setUpdater(String aud) {
		if (aud != null && !aud.isEmpty()) {
			m_Updater = aud;
		}
	}

	/**
	 * set the record update time (separately)
	 * @param upd the update timestamp (can be null)
	 */
	public void setLmodDate(Date upd) {
		m_LmodDate = upd;
	}

	/**
	 * configure the status (for the first time)
	 * @param stat the status value
	 * @param submitter the Auditor ID
	 * @param ts the timestamp to set, or null to use current time
	 */
	public boolean configStatus(String stat, String submitter, Date ts) {
		// validate inputs
		if (stat == null
			|| (!stat.equals(M_Valid)
				&& !stat.equals(M_Suspect)
				&& !stat.equals(M_Blocked)))
		{
			return false;
		}
		if (submitter == null || !submitter.startsWith("AUD-")) {
			return false;
		}

		// verify this is the first time
		if (!m_Submitter.isEmpty() || m_InsertDate != null) {
			// should use updStatus()
			return false;
		}

		// set fields
		m_Status = stat;
		m_Submitter = submitter;
		if (ts == null) {
			m_InsertDate = new Date();
		}
		else {
			m_InsertDate = ts;
		}
		return true;
	}

	/**
	 * update the status (not for the first time)
	 * @param stat the status value
	 * @param updater the Auditor ID
	 * @param ts the timestamp to set, or null to use current time
	 * @param reason the textual explanation for the status, empty to let stand
	 */
	public boolean updStatus(String stat,
							 String updater,
							 Date ts,
							 String reason)
	{
		// validate inputs
		if (stat == null
			|| (!stat.equals(M_Valid)
				&& !stat.equals(M_Suspect)
				&& !stat.equals(M_Blocked)))
		{
			return false;
		}
		if (updater == null || !updater.startsWith("AUD-")) {
			return false;
		}

		// verify this isn't the first time
		if (m_Submitter.isEmpty() || m_InsertDate == null) {
			// should use setStatus()
			return false;
		}

		// set fields
		m_Status = stat;
		m_Updater = updater;
		if (ts == null) {
			m_LmodDate = new Date();
		}
		else {
			m_LmodDate = ts;
		}
		if (reason != null) {
			m_Narrative = reason;
		}
		return true;
	}

	// END methods
}
