/*
 * last modified---
 * 	09-06-22 new, stripped down from ../mvo's version
 *
 * purpose---
 * 	encapsulate certain configuration data for an MVO
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.log.Log;

import java.math.BigInteger;


/**
 * This class holds the configuration parameters specific to a particular
 * MVO server.  The values should be loaded from the corresponding records in
 * the smart contract for the blockchain.  Configs will exist for the same
 * MVO in every relevant blockchain's configuration.  As opposed to the full
 * version in cc.enshroud.jetty.mvo.MVOConfig, this version used by Auditors
 * does not contain certain data, e.g. the MVO's private key.
 */
public final class MVOConfig extends MVOGenConfig {
	// BEGIN data members

	// NB: these 3 items are in lieu of separate BlockchainConfig records
	/**
	 * quantity of $ENSHROUD staked by this MVO (normally on chainId = 1, but
	 * it can also be specified for individual chains)
	 */
	private BigInteger				m_Staked;

	/**
	 * address associated with on-chain signing credentials
	 */
	private String					m_SigningAddress;

	/**
	 * address associated with on-chain staking of m_Staked $ENSHROUD
	 */
	private String					m_StakingAddress;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param mvoId Id of server
	 * @param log logging object
	 */
	public MVOConfig(String mvoId, Log log) {
		super(mvoId, log);
		m_Staked = new BigInteger("0");
		m_SigningAddress = m_StakingAddress = "";
	}

	// GET methods
	/**
	 * obtain the staked quantity on this particular blockchain
	 * @return the quantity this MVO has staked here (can be zero)
	 */
	public BigInteger getStaking() { return m_Staked; }

	/**
	 * obtain the signing address for this chain
	 * @return the address whose private key signs for this MVO
	 */
	public String getSigningAddress() { return m_SigningAddress; }

	/**
	 * obtain the staking address for this chain
	 * @return the address where this MVO has their $ENSHROUD staked (can be "")
	 */
	public String getStakingAddress() { return m_StakingAddress; }


	// SET methods
	/**
	 * configure the staked quantity for this MVO on this chain
	 * @param staking the amount staked
	 */
	public void setStaked(BigInteger staking) {
		if (staking != null) {
			m_Staked = staking;
		}
		else {
			m_Log.error("MVOConfig.setStaked: missing staking amount");
		}
	}

	/**
	 * configure the signing address
	 * @param addr the address whose private key the MVO uses to sign here
	 */
	public void setSigningAddress(String addr) {
		if (addr == null || addr.isEmpty()) {
			m_Log.error("MVOConfig.setSigningAddress: missing address");
		}
		else {
			m_SigningAddress = new String(addr);
		}
	}

	/**
	 * configure the staking address
	 * @param addr the address where the MVO stakes on this chain (can be "")
	 */
	public void setStakingAddress(String addr) {
		if (addr == null) {
			m_Log.error("MVOConfig.setStakingAddress: missing address");
		}
		else {
			m_StakingAddress = new String(addr);
		}
	}

	// END methods
}
