/*
 * last modified---
 * 	12-14-23 add m_GreyListers
 * 	12-06-22 add m_BlockchainConfigs
 * 	09-06-22 new, based on MVOConfig
 *
 * purpose---
 * 	encapsulate certain configuration data for an Auditor
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.log.Log;

import java.net.URL;
import java.net.URI;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.Hashtable;


/**
 * This class holds the configuration parameters specific to a particular
 * AUD server.  The values should be loaded from the corresponding records in
 * the smart contract for the blockchain.  Configs will exist for the same
 * AUD in every relevant blockchain's configuration.
 */
public final class AUDConfig {
	// BEGIN data members
	/**
	 * ID of the Auditor, used as an index and as address owner
	 * (should match the one in AUD object)
	 */
	private String							m_AUDId;

	/**
	 * endpoint where this Auditor listens for peer MVO requests (wss:)
	 */
	private URI								m_MVOURI;

	/**
	 * current status (up | down)
	 */
	private boolean							m_Status;

	/**
	 * public key used for Layer2 communications with MVOs
	 */
	private PublicKey						m_CommPubKey;

	/**
	 * private key used to decrypt and sign Layer2 communications
	 */
	private PrivateKey						m_CommPrivKey;

	/**
	 * the configs of MVOs we know about, ID mapped to port number
	 */
	private HashMap<String, Integer>		m_MVOList;

	/**
	 * the map of public keys for other peer MVOs and Auditors, indexed by ID
	 * (all MVOs in the m_MVOList must have an entry here)
	 */
	private Hashtable<String, PublicKey>	m_PeerPubkeys;

	/**
	 * map of this Auditor's configuration on each supported blockchain,
	 * chainId to config
	 */
	private HashMap<Long, BlockchainConfig>	m_BlockchainConfigs;

	/**
	 * map of this Auditor's greylisting utility on each supported blockchain,
	 * chainId to object
	 */
	private HashMap<Long, GreyLister>		m_GreyListers;

	/**
	 * logging object
	 */
	private Log								m_Log;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param mvoId Id of server
	 * @param log logging object
	 */
	public AUDConfig(String mvoId, Log log) {
		m_AUDId = new String(mvoId);
		m_Log = log;
		m_MVOList = new HashMap<String, Integer>();
		m_PeerPubkeys = new Hashtable<String, PublicKey>(100);
		m_BlockchainConfigs = new HashMap<Long, BlockchainConfig>();
		m_GreyListers = new HashMap<Long, GreyLister>();
	}

	// GET methods
	/**
	 * obtain AUD Id
	 * @return the Id for the server
	 */
	public String getId() { return m_AUDId; }

	/**
	 * obtain MVO request URI
	 * @return the endpoint where this MVO listens for peer MVO requests
	 */
	public URI getMVOURI() { return m_MVOURI; }

	/**
	 * obtain the last reported status of the MVO
	 * @return up|down status
	 */
	public boolean getStatus() { return m_Status; }

	/**
	 * obtain the pubkey used to encrypt messages sent to this Auditor
	 * @return the key used for securing messages sent to this AUD
	 */
	public PublicKey getCommPubKey() { return m_CommPubKey; }

	/**
	 * obtain the privkey used to decrypt messages sent to this AUD, and sign
	 * messages originating from it
	 * @return the key used for decrypt/sign of peer L2 messages
	 */
	public PrivateKey getCommPrivKey() { return m_CommPrivKey; }

	/**
	 * look up a pubkey for a given peer
	 * @param peerId the ID of the other MVO or Auditor
	 * @return the public key for that ID, or null if not found
	 */
	public PublicKey getPeerPubkey(String peerId) {
		if (peerId == null || peerId.isEmpty()) {
			return null;
		}
		return m_PeerPubkeys.get(peerId);
	}

	/**
	 * obtain the list of mandatory MVO IDs and listen port numbers
	 * @return the list, as configured in properties
	 */
	public HashMap<String, Integer> getMVOList() { return m_MVOList; }

	/**
	 * obtain the list of blockchain configs for this Auditor
	 * @return the map, indexed by chain IDs
	 */
	public HashMap<Long, BlockchainConfig> getChainConfigs() {
		return m_BlockchainConfigs;
	}

	/**
	 * obtain the greylisting utility for a given chain
	 * @param chainId the chain whose GreyLister we want
	 * @return the GreyLister specific to the passed chain, null if not found
	 */
	public GreyLister getGreyLister(long chainId) {
		return m_GreyListers.get(chainId);
	}


	// SET methods
	/**
	 * configure the peer MVO request endpoint
	 * @param uri the relevant network endpoint serving requests
	 */
	public void setMVOURI(URI uri) {
		if (uri != null) {
			m_MVOURI = uri;
		}
		else {
			m_Log.error("AUDConfig.setMVOURI: missing URI");
		}
	}

	/**
	 * update status of MVO
	 * @param stat up (true) or down (false)
	 */
	public void setStatus(boolean stat) { m_Status = stat; }

	/**
	 * configure the public key used to secure messages sent to this MVO
	 * @param pubkey the appropriate public key
	 */
	public void configCommPubkey(PublicKey pubkey) {
		if (pubkey != null) {
			m_CommPubKey = pubkey;
		}
		else {
			m_Log.error("AUDConfig.configCommPubkey: missing key");
		}
	}

	/**
	 * configure the private key used to secure messages sent to this MVO
	 * @param privkey the appropriate private key
	 */
	public void configCommPrivkey(PrivateKey privkey) {
		if (privkey != null) {
			m_CommPrivKey = privkey;
		}
		else {
			m_Log.error("AUDConfig.configCommPrivkey: missing key");
		}
	}

	/**
	 * add an MVO Id to the list
	 * @param id the ID to add (from properties)
	 * @param port the port number to construct VPN URI (from properties)
	 */
	public void addMVO(String id, Integer port) {
		if (id != null && !id.isEmpty() && port != null && port.intValue() > 0)
		{
			m_MVOList.put(id, port);
		}
	}

	/**
	 * add a public key to the map of peer keys
	 * @param peerId the ID of the MVO or Auditor
	 * @param pubKey their public key
	 * @return previous value, or null if there wasn't one
	 */
	public PublicKey addPeerPubkey(String peerId, PublicKey pubKey) {
		if (peerId == null || peerId.isEmpty() || pubKey == null) {
			return null;
		}
		return m_PeerPubkeys.put(peerId, pubKey);
	}

	/**
	 * add a blockchain config for a chain
	 * @param chainId the ID of the blockchain
	 * @param config the blockchain config
	 * @return previous value, or null if there was none
	 */
	public BlockchainConfig addChainConfig(long chainId,
										   BlockchainConfig config)
	{
		if (chainId <= 0 || config == null) {
			m_Log.error("addChainConfig(): missing input");
			return null;
		}
		return m_BlockchainConfigs.put(chainId, config);
	}

	/**
	 * add a greylister object for a chain
	 * @param chainId the ID of the blockchain
	 * @param greyLister the greylister utility object
	 * @return previous value, or null of there was none
	 */
	public GreyLister addGreyLister(long chainId, GreyLister greyLister) {
		if (chainId <= 0 || greyLister == null) {
			m_Log.error("addGreyLister: missing input");
			return null;
		}
		return m_GreyListers.put(chainId, greyLister);
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			if (m_CommPrivKey != null) {
				m_CommPrivKey.destroy();
			}
			if (m_PeerPubkeys != null) {
				m_PeerPubkeys.clear();
			}
			if (m_MVOList != null) {
				m_MVOList.clear();
			}
			if (m_BlockchainConfigs != null) {
				m_BlockchainConfigs.clear();
			}
			if (m_GreyListers != null) {
				m_GreyListers.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
