/*
 * last modified---
 * 	09-25-23 parse only ClientMVOBlock.message.requestJson, due to EIP-712
 * 	07-12-22 improve error output labeling
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-04-22 new
 *
 * purpose---
 * 	encapsulate client requests forwarded from a lead/committee MVO to Auditors
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.io.IOException;


/**
 * This class holds the data for a forward request (over a WebSocket connection)
 * from one MVO to Auditors.  The first (lead) MVO attaches the client's
 * original {@link ClientMVOBlock} request verbatim, together with the
 * {@link AuditorBlock} which it generated in response.  Each committee MVO
 * will re-verify everything the first MVO did, and then send the
 * {AuditorBlock} as a broadcast, bearing its own signature.
 */
public final class MVOAuditorBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * the dApp client's original request
	 */
	private ClientMVOBlock		m_ClientBlock;

	/**
	 * original received JSON string representing the ClientMVOBlock
	 */
	private String				m_ClientBlockJson;

	/**
	 * the MVO's signed {AuditorBlock}, which it created based on the request
	 */
	private AuditorBlock		m_MVOBlock;

	/**
	 * original received JSON string representing the AuditorBlock
	 */
	private String				m_MVOBlockJson;

	/**
	 * logging object
	 */
	private Log					m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param log logging object
	 */
	public MVOAuditorBlock(Log log) {
		m_Log = log;
		m_ClientBlockJson = m_MVOBlockJson = "";
	}

	// GET methods
	/**
	 * obtain the client request block
	 * @return the client's original request, forwarded
	 */
	public ClientMVOBlock getClientBlock() { return m_ClientBlock; }

	/**
	 * obtain the MVO block
	 * @return the signed operations block created by the broadcasting MVO
	 */
	public AuditorBlock getMVOBlock() { return m_MVOBlock; }

	/**
	 * obtain the client request block JSON
	 * @return the original string received before parsing
	 */
	public String getClientBlockJson() { return m_ClientBlockJson; }

	/**
	 * obtain the operations block JSON
	 * @return the original string received before parsing
	 */
	public String getMVOBlockJson() { return m_MVOBlockJson; }

	// SET methods
	/**
	 * configure the client block
	 * @param the client's original forwarded block
	 */
	public void setClientBlock(ClientMVOBlock cBlock) {
		if (cBlock != null) {
			m_ClientBlock = cBlock;
		}
	}

	/**
	 * configure the client block json
	 * @param json the original JSON string received
	 */
	public void setClientBlockJson(String json) {
		if (json != null && !json.isEmpty()) {
			m_ClientBlockJson = new String(json);
		}
	}

	/**
	 * configure the MVO operations block json
	 * @param json the original JSON string received
	 */
	public void setMVOBlockJson(String json) {
		if (json != null && !json.isEmpty()) {
			m_MVOBlockJson = json;
		}
	}

	/**
	 * configure the MVO block
	 * @param the MVO's forwarded block
	 */
	public void setMVOBlock(AuditorBlock mvoBlock) {
		if (mvoBlock != null) {
			m_MVOBlock = mvoBlock;
		}
	}

	/**
	 * parse a JSON string containing both of these blocks
	 * @param forwardMsg the forwarded message from the MVO
	 * @return true on success
	 */
	public boolean buildFromString(String forwardMsg) {
		if (forwardMsg == null || forwardMsg.isEmpty()) {
			m_Log.error("MVOAuditorBlock.buildFromString: missing data");
			return false;
		}
		boolean ret = true;
		Object data = null;
		try {
			data = JSON.parse(forwardMsg);
		}
		catch (IllegalStateException ise) {
			m_Log.error("MVOAuditorBlock.buildFromString: forwarded data was "
						+ "not JSON: \"" + forwardMsg + "\"");
			ret = false;
			return ret;
		}
		if (data instanceof Map) {
			Map reqData = (Map) data;
			ret = buildFromMap(reqData, forwardMsg);
		}
		else {
			m_Log.error("MVOAuditorBlock.buildFromString: forwarded data was "
						+ "not a Map");
			ret = false;
		}
		return ret;
	}

	/**
	 * build the objects from a passed Map, created from this JSON structure.
	 * {
	 * 	"ClientMVOBlock":{the ClientMVOBlock Map},
	 * 	"AuditorBlock":{the AuditorBlock Map}
	 * }
	 * @param map the mapping, created by parsing JSON
	 * @param orgJson the original string passed (known to be valid JSON)
	 * @return true on success
	 */
	public boolean buildFromMap(Map map, String orgJson) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (map == null || map.isEmpty()
			|| orgJson == null || orgJson.isEmpty())
		{
			m_Log.error(lbl + "missing or empty top Map");
			return false;
		}
		boolean ret = true;

		// get the first object, which includes all the EIP-712 sig data + types
		Object client = map.get("ClientMVOBlock");
		if (client instanceof Map) {
			// find message.requestJson, which should also be a Map
			Map payloadMap = null;
			Map clientBlock = (Map) client;
			Object messageObj = clientBlock.get("message");
			if (!(messageObj instanceof Map)) {
				m_Log.error(lbl + "could not find message in signed data");
				ret = false;
			}
			else {
				Map messageMap = (Map) messageObj;
				Object requestJson = messageMap.get("requestJson");
				if (!(requestJson instanceof Map)) {
					m_Log.error(lbl + "could not find message.requestJson");
					ret = false;
				}
				else {
					Map jsonMap = (Map) requestJson;
					/* one of 3 request payloads come here: depositspec,
					 * spendspec, or withdrawspec
					 */
					Object depReq = jsonMap.get("depositspec");
					if (depReq instanceof Map) {
						payloadMap = (Map) depReq;
					}
					else {
						Object spendReq = jsonMap.get("spendspec");
						if (spendReq instanceof Map) {
							payloadMap = (Map) spendReq;
						}
						else {
							Object withReq = jsonMap.get("withdrawspec");
							if (withReq instanceof Map) {
								payloadMap = (Map) withReq;
							}
							else {
								ret = false;
								m_Log.error(lbl + "no recognized payload type");
							}
						}
					}
				}
			}

			m_ClientBlock = new ClientMVOBlock(null, m_Log);
			if (ret && payloadMap != null) {
				ret = m_ClientBlock.buildFromMap(payloadMap);
				if (ret) {
					// capture the original JSON for this element for addJSON()
					String startTag = "{\"ClientMVOBlock\":";
					int startIdx = startTag.length();
					String endTag = ",\"AuditorBlock\":";
					int endIdx = orgJson.indexOf(endTag);
					if (endIdx == -1) {
						ret = false;
						m_Log.error(lbl
									+ "no end tag found for ClientMVOBlock");
					}
					else {
						String orgCli = orgJson.substring(startIdx, endIdx);
						setClientBlockJson(orgCli);
					}
				}
			}
		}
		else {
			m_Log.error(lbl + "missing ClientMVOBlock Map");
			ret = false;
		}
		if (ret) {
			// get the second object
			Object ops = map.get("AuditorBlock");
			if (ops instanceof Map) {
				Map opsBlock = (Map) ops;
				m_MVOBlock = new AuditorBlock(m_Log);
				ret = m_MVOBlock.buildFromMap(opsBlock);
			}
			else {
				m_Log.error(lbl + "missing AuditorBlock Map");
				ret = false;
			}
		}
		return ret;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		if (m_ClientBlockJson.isEmpty() || m_MVOBlockJson.isEmpty()) {
			m_Log.error("MVOAuditorBlock.addJSON(): Cannot render JSON without "
						+ "components set");
			return;
		}
		StringBuilder out = new StringBuilder(20480);
		out.append("{\"ClientMVOBlock\":");
		out.append(m_ClientBlockJson + ",");
		out.append("\"AuditorBlock\":");
		out.append(m_MVOBlockJson + "}");
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("MVOAuditorBlock.addJSON(): exception appending", ioe);
		}
	}

	// END methods
}
