/*
 * last modified---
 * 	09-29-23 pass filespecs[] as Filespec[] to fix issue with EIP712 signatures
 * 	06-05-23 remove m_Signature, as this is now at a higher level
 * 	03-20-23 add handling of m_ReplyKey if present
 * 	09-14-22 supply addJSON()
 * 	07-12-22 improve error output labeling
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-30-22 extend ClientRequest
 * 	03-23-22 add {capability} field
 * 	03-08-22 new
 *
 * purpose---
 * 	encapsulate receipt-related request messages to MVOs from dApp clients
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.util.ArrayList;
import java.io.IOException;
import javax.servlet.http.HttpServletRequest;


/**
 * This class holds the data for dApp receipt requests made to MVOs.  Such
 * requests are received on the https endpoint, parsed, and stored in these
 * objects for processing by MVOs.  The output of MVO processing will be a
 * request/reply to the receipt storage engine for the relevant blockchain,
 * which reply data is passed back to the dApp in the MVO's reply.
 * This class knows how to build itself from input JSON.
 */
public final class ClientReceiptBlock extends ClientRequest
	implements JSON.Generator
{
	// BEGIN data members
	/**
	 * opcode constant: get
	 */
	public final static String	M_ReceiptGet = "get";
	/**
	 * opcode constant: list
	 */
	public final static String	M_ReceiptList = "list";
	/**
	 * opcode constant: delete
	 */
	public final static String	M_ReceiptDel = "delete";

	/**
	 * capability bits (the signature of m_Sender on m_Sender address)
	 */
	private String			m_Capability;

	/**
	 * opcode for request, one of list|get|delete (see constants above)
	 */
	private String			m_Opcode;

	/**
	 * list of filenames being manipulated (for get and delete opcodes)
	 */
	private ArrayList<String>	m_FileSpecs;

	/**
	 * logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param orgReq the original HTTP level request
	 * @param logger the logging object
	 */
	public ClientReceiptBlock(HttpServletRequest orgReq, Log logger) {
		super(orgReq);
		m_Log = logger;
		m_Opcode = m_Capability = "";
		m_FileSpecs = new ArrayList<String>();
	}

	// GET methods
	/**
	 * obtain the client's capability bits
	 * @return the access capability (normally sig(getSender()))
	 */
	public String getCapability() { return m_Capability; }

	/**
	 * obtain the opcode of this request
	 * @return the code, one of deposit|spend|withdraw
	 */
	public String getOpcode() { return m_Opcode; }

	/**
	 * obtain the list of file specifications
	 * @return the list, empty if none
	 */
	public ArrayList<String> getFileSpecs() { return m_FileSpecs; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * config the chain Id
	 * @param id the ID of the blockchain this request is for, per chainlist.org
	 */
	public void setChainId(long id) {
		if (id > 0L) {
			m_ChainId = id;
		}
		else {
			m_Log.error("ClientReceiptBlock.setChainId(): illegal chainId, "
						+ id);
		}
	}

	/**
	 * set the sender of the request
	 * @param sender the wallet address which sent (and signed) the request
	 */
	public void setSender(String sender) {
		if (sender != null && !sender.isEmpty()) {
			m_Sender = new String(sender);
		}
		else {
			m_Log.error("ClientReceiptBlock.setSender(): missing sender");
		}
	}

	/**
	 * set the sender's capability bits
	 * @param cap the capability (normally sig(getSender()))
	 */
	public void setCapability(String cap) {
		if (cap != null && !cap.isEmpty()) {
			m_Capability = new String(cap);
		}
		else {
			m_Log.error("ClientReceiptBlock.setCapability(): missing "
						+ "capabilities bits");
		}
	}

	/**
	 * set the opcode for the request
	 * @param op the opcode, one of deposit|spend|withdraw
	 */
	public void setOpcode(String op) {
		if (op != null && !op.isEmpty()) {
			m_Opcode = new String(op);
		}
		else {
			m_Log.error("ClientReceiptBlock.setOpcode(): missing opcode");
		}
	}

	/**
	 * add a filename to the list of files
	 * @param file the filespec to add
	 */
	public void addFileSpec(String file) {
		if (file != null) {
			m_FileSpecs.add(file);
		}
	}


	/**
	 * method to build object from a Map
	 * @param request the mapping, the result of a JSON parse
	 * @return true on success
	 */
	public boolean buildFromMap(Map request) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (request == null || request.isEmpty()) {
			m_Log.error(lbl + "missing top Map");
			return false;
		}
		boolean ret = true;

		// required: the chainId
		Object chain = request.get("chainId");
		if (chain instanceof String) {
			String chainId = (String) chain;
			try {
				Long cId = Long.parseLong(chainId.trim());
				setChainId(cId.longValue());
			}
			catch (NumberFormatException nfe) {
				ret = false;
				m_Log.error(lbl + "illegal chain Id, " + chainId, nfe);
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing chain Id");
		}

		// required: the sender address
		Object send = request.get("sender");
		if (send instanceof String) {
			String sender = (String) send;
			setSender(sender);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing sender");
		}

		// required: the capabilities bits
		Object cap = request.get("capability");
		if (cap instanceof String) {
			String capabilities = (String) cap;
			setCapability(capabilities);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing capabilities bits");
		}

		// optional: the AES encryption reply key
		Object rKey = request.get("replyKey");
		if (rKey instanceof String) {
			String repKey = (String) rKey;
			setReplyKey(repKey);
		}

		// required: the opcode
		Object op = request.get("opcode");
		if (op instanceof String) {
			String opcode = (String) op;
			setOpcode(opcode);
			if (!(m_Opcode.equals(M_ReceiptGet)
				  || m_Opcode.equals(M_ReceiptList)
				  || m_Opcode.equals(M_ReceiptDel)))
			{
				ret = false;
				m_Log.error(lbl + "illegal client opcode, " + m_Opcode);
				return ret;
			}
		}
		else {
			ret = false;
			m_Log.error("buildFromMap(): missing opcode");
			return ret;
		}

		// get filespec inputs (at least one required except for list)
		Object files = request.get("filespecs");
		if (files instanceof Object[]) {
			Object[] filespecs = (Object[]) files;
			if (filespecs.length < 1 && !m_Opcode.equals(M_ReceiptList)) {
				ret = false;
				m_Log.error(lbl + "missing filespec inputs");
			}
			for (int iii = 0; iii < filespecs.length; iii++) {
				Object fileSpec = filespecs[iii];
				if (fileSpec instanceof Map) {
					Map fSpec = (Map) fileSpec;
					Object filename = fSpec.get("receiptID");
					if (filename instanceof String) {
						String fileName = (String) filename;
						if (fileName.endsWith(".json")) {
							addFileSpec(fileName);
						}
						else {
							ret = false;
							m_Log.error(lbl + "illegal filename, " + fileName);
						}
					}
					else {
						ret = false;
						m_Log.error(lbl + "non-String Filespec input "
									+ (iii+1));
					}
				}
				else {
					ret = false;
					m_Log.error(lbl + "missing Filespec input " + (iii+1));
				}
			}
		}
		else if (!m_Opcode.equals(M_ReceiptList)) {
			ret = false;
			m_Log.error(lbl + "missing filespec inputs");
		}

		return ret;
	}

	/**
	 * method to build entire request object from a JSON string
	 * @param reqData the request data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String reqData) {
		if (reqData == null | reqData.isEmpty()) {
			m_Log.error("ClientReceiptBlock.buildFromString: missing client "
						+ "request data");
			return false;
		}
		Object req = null;
		boolean ret = false;
		try {
			req = JSON.parse(reqData);
		}
		catch (IllegalStateException ise) {
			m_Log.error("ClientReceiptBlock.buildFromString: client JSON did "
						+ "not parse: \"" + reqData + "\"");
			return ret;
		}

		// parsed object should consist of a Map
		if (!(req instanceof Map)) {
			m_Log.error("ClientReceiptBlock.buildFromString: client JSON was "
						+ "not a Map");
			return ret;
		}
		else {
			Map map = (Map) req;
			ret = buildFromMap(map);
		}
		return ret;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		out.append("{\"chainId\":\"" + m_ChainId + "\",");
		out.append("\"sender\":\"" + m_Sender + "\",");
		out.append("\"capability\":\"" + m_Capability + "\",");
		out.append("\"opcode\":\"" + m_Opcode + "\"");
		if (!m_ReplyKey.isEmpty()) {
			out.append(",\"replyKey\":\"" + m_ReplyKey + "\"");
		}
		int fIdx = 0;
		if (!m_FileSpecs.isEmpty()) {
			out.append(",\"filespecs\":[");
			for (String fSpec : m_FileSpecs) {
				out.append("{\"receiptId\":\"" + fSpec + "\"}");
				// add comma except last time
				if (++fIdx < m_FileSpecs.size()) {
					out.append(",");
				}
			}
			out.append("]");
		}
		out.append("}");

		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("ClientReceiptBlock.addJSON(): exception appending",
						ioe);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_FileSpecs != null) {
				m_FileSpecs.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
