/*
 * last modified---
 * 	04-05-23 add chainId as parameter to getReceiptIds()
 * 	03-15-23 new
 *
 * purpose---
 * 	provide an API to manipulate records in the ENSH_RECEIPT_STORAGE table
 */

package cc.enshroud.jetty.mvo.db;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.GenericDb;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.db.DbConnectionManager;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Date;
import java.util.ArrayList;


/**
 * This class supplies atomic operations which act on the ENSH_RECEIPT_STORAGE
 * table in the MVO's database.  These methods correspond to logical operations,
 * e.g. create, get.  They closely interoperate with the class that instantiates
 * a row in the table, {@link ReceiptStorage ReceiptStorage}.  Read-write
 * methods allow for passed connections, to permit use within a transaction.
 */
public final class ReceiptStorageDb extends GenericDb {
	// BEGIN data members

    // END data members

    // BEGIN methods
    /**
     * constructor
	 * @param manager the DB connection manager
	 * @param logger the error/debug logger
	 */
	public ReceiptStorageDb(DbConnectionManager manager, Log logger) {
		super(manager, logger);
		m_Table = "ENSH_RECEIPT_STORAGE";
	}

	/**
	 * obtain the list of receipt IDs stored for a particular owner hash
	 * @param owner the SHA256 hash of the owning address
	 * @param chainId the ID of the relevant blockchain
	 * @return the list, empty if not found, or null on errors
	 */
	public ArrayList<String> getReceiptIds(String owner, long chainId) {
		final String lbl = this.getClass().getSimpleName() + ".getReceiptIds: ";
		if (owner == null || owner.length() != 64) {
			m_Log.error(lbl + "missing owner address hash");
			return null;
		}
		if (chainId <= 0L) {
			m_Log.error(lbl + "illegal chainId, " + chainId);
			return null;
		}

		// get a DB connection
		Connection dbConn = getDbConnection();
		if (dbConn == null) {
			m_Log.error(lbl + "cannot get DB connection");
			return null;
		}
		ArrayList<String> ownedIds = new ArrayList<String>(100);

		// build and execute query
		String getSql = "select RECEIPT_ID from " + m_Table + " where OWNER=? "
						+ "and CHAIN_ID=?;";
		PreparedStatement query = null;
		ResultSet rctRS = null;
		try {
			query = dbConn.prepareStatement(getSql);
			query.setString(1, owner);
			query.setLong(2, chainId);
			rctRS = query.executeQuery();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running receipt ID query for " + m_Table,
						sqle);
			ownedIds = null;
		}

		// examine results and load a return object
		try {
			while (ownedIds != null && rctRS != null && rctRS.next()) {
				ownedIds.add(rctRS.getString("RECEIPT_ID"));
			}
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error parsing result set for " + m_Table, sqle);
			ownedIds = null;
		}
		finally {
			m_DbManager.closeConnection(rctRS, query, dbConn);
		}
		return ownedIds;
	}

	/**
	 * obtain the receipt text for a list of receipts
	 * @param Ids the IDs of the receipts whose data is sought
	 * @param chainId the ID of the relevant blockchain
	 * @param conn DB connection to use (create one if passed as null)
	 * @return the list of status records for these IDs, empty if not found, or
	 * null on errors
	 */
	public ArrayList<ReceiptStorage> getReceipts(ArrayList<String> Ids,
												 long chainId,
												 Connection conn)
	{
		final String lbl = this.getClass().getSimpleName() + ".getReceipts: ";
		if (Ids == null || Ids.isEmpty() || chainId <= 0L) {
			m_Log.error(lbl + "missing input");
			return null;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				return null;
			}
		}
		ArrayList<ReceiptStorage> rctRecs
			= new ArrayList<ReceiptStorage>(Ids.size());

		// build and execute query
		StringBuilder getSql = new StringBuilder(1024);
		getSql.append("select * from " + m_Table + " where RECEIPT_ID in (");
		int idCnt = 1;
		for (String id : Ids) {
			getSql.append("?");
			if (idCnt++ < Ids.size()) {
				getSql.append(",");
			}
		}
		getSql.append(") and CHAIN_ID=?;");
		PreparedStatement query = null;
		ResultSet rctRS = null;
		try {
			query = dbConn.prepareStatement(getSql.toString());
			idCnt = 1;
			for (String id : Ids) {
				query.setString(idCnt++, id);
			}
			query.setLong(idCnt, chainId);
			rctRS = query.executeQuery();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running receipt query for " + m_Table,
						sqle);
			rctRecs = null;
		}

		// examine results and load a return object
		try {
			while (rctRecs != null && rctRS != null && rctRS.next()) {
				String rId = rctRS.getString("RECEIPT_ID");
				long chain = rctRS.getLong("CHAIN_ID");
				ReceiptStorage rctStore = new ReceiptStorage(rId, chain);
				rctStore.setOwner(rctRS.getString("OWNER"));
				rctStore.setReceipt(rctRS.getString("RECEIPT"));
				rctRecs.add(rctStore);
			}
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error parsing result set for " + m_Table, sqle);
			rctRecs = null;
		}
		finally {
			// don't close a passed connection
			if (conn != null) {
				m_DbManager.closeResultSet(rctRS);
				m_DbManager.closeStatement(query);
			}
			else {
				m_DbManager.closeConnection(rctRS, query, dbConn);
			}
		}
		return rctRecs;
	}

	/**
	 * create a record for a receipt
	 * @param rId the ID of the receipt
	 * @param chainId the ID of the blockchain to which the receipt applies
	 * @param owner the SHA256 hash of the address of the owning wallet
	 * @param encText the AES-encrypted, base64-encoded text of the receipt
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on success, false on bad input
	 * @throws EnshDbException on all database errors, dup record, etc.
	 */
	public boolean storeReceipt(String rId,
								long chainId,
								String owner,
								String encText,
								Connection conn)
		throws EnshDbException
	{
		final String lbl = this.getClass().getSimpleName() + ".storeReceipt: ";
		// check inputs
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain ID");
			return false;
		}
		if (rId == null || rId.isEmpty()) {
			m_Log.error(lbl + "missing receipt ID");
			return false;
		}
		if (owner == null || owner.length() != 64) {
			m_Log.error(lbl + "illegal receipt owner address hash, " + owner);
			return false;
		}
		if (encText == null || encText.isEmpty()) {
			m_Log.error(lbl + "missing receipt data");
			return false;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				throw new EnshDbException("unable to get database connection");
			}
		}

		// build and execute SQL insertion string
		String insSql = "insert into " + m_Table + " values(?, ?, ?, ?);";
		PreparedStatement insert = null;
		int insCnt = 0;
		try {
			insert = dbConn.prepareStatement(insSql);
			insert.setString(1, rId);
			insert.setLong(2, chainId);
			insert.setString(3, owner);
			insert.setString(4, encText);
			insCnt = insert.executeUpdate();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running insert for " + m_Table, sqle);
			if (conn != null) {
				m_DbManager.closeStatement(insert);
			}
			else {
				m_DbManager.closeConnection(insert, dbConn);
			}
			throw new EnshDbException(sqle.getMessage(), insSql,
									  sqle.getErrorCode());
		}
		finally {
			// don't close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(insert);
			}
			else {
				m_DbManager.closeConnection(insert, dbConn);
			}
		}
		if (insCnt != 1) {
			m_Log.error(lbl + "unexpected insert count on " + m_Table
						+ ": " + insCnt);
			throw new EnshDbException("Bad insert count, " + insCnt, insSql);
		}
		return true;
	}

	/**
	 * delete a receipt record
	 * @param rId the ID of the receipt
	 * @param chainId the ID of the blockchain
	 * @param owner the SHA256 hash of the owner of the receipt (must match)
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on success, false on errors or if not found
	 */
	public boolean purgeReceipt(String rId,
								long chainId,
								String owner,
								Connection conn)
	{
		final String lbl = this.getClass().getSimpleName() + ".purgeReceipt: ";

		// check inputs
		if (rId == null || rId.isEmpty()) {
			m_Log.error(lbl + "missing receipt ID");
			return false;
		}
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain ID");
			return false;
		}
		if (owner == null || owner.length() != 64) {
			m_Log.error(lbl + "illegal receipt owner address hash, " + owner);
			return false;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				return false;
			}
		}

		// build and execute update string
		String updSql = "delete from " + m_Table + " where RECEIPT_ID=? and "
						+ "CHAIN_ID=? and OWNER=?;";
		PreparedStatement delRct = null;
		int updCnt = 0;
		try {
			delRct = dbConn.prepareStatement(updSql);
			delRct.setString(1, rId);
			delRct.setLong(2, chainId);
			delRct.setString(3, owner);
			updCnt = delRct.executeUpdate();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "exception running " + updSql, sqle);
		}
		finally {
			// do not close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(delRct);
			}
			else {
				m_DbManager.closeConnection(delRct, dbConn);
			}
		}
		// unique index
		if (updCnt != 1) {
			m_Log.error(lbl + "bad update count deleting receipt " + rId);
			return false;
		}
		return true;
	}

	// END methods
}
