/*
 * last modified---
 * 	06-29-23 getABI() moved to SmartContractConfig
 * 	06-20-23 use checkIfIdUsed() rather than getNFTsById()
 * 	06-14-23 remove URL param from getUniqueNFTId()
 * 	12-02-22 add M_RandomizerSize
 * 	08-04-22 add m_ExpectingKeyServerReply, recordKeyServerReply(),
 * 				  getPreviousStep()
 * 	07-27-22 add getUniqueNFTId()
 * 	07-20-22 add M_JoinChar
 * 	05-05-22 add m_KeyResponse and methods
 * 	04-06-22 add m_Receipts and methods
 * 	03-30-22 new
 *
 * purpose---
 * 	base class for request state machine objects
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ReceiptBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.eNFTmetadata;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.SmartContractConfig;

import java.util.ArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.net.URL;


/**
 * This class is a base class for all state machine classes which correspond to
 * requests which an MVO can receive from a dApp client.
 */
abstract class ReqSvcState {
	// BEGIN data members

	// possible common failure modes (sub-classes may add to this list)
	/**
	 * no error occurred
	 */
	public static final int	M_NoFailure = 0;
	/**
	 * client request did not parse or was otherwise defective
	 */
	public static final int	M_InvalidReq = 1;
	/**
	 * a peer MVO could not be accessed
	 */
	public static final int	M_UnavailMVO = 2;
	/**
	 * an Auditor could not be accessed
	 */
	public static final int M_UnavailAud = 3;
	/**
	 * a blockchain ABI node could not be accessed
	 */
	public static final int	M_UnavailABI = 4;
	/**
	 * some generic processing error occurred
	 */
	public static final int	M_ProcError = 5;
	/**
	 * a partner entity sent us a bad reply we could not understand
	 */
	public static final int	M_ParseError = 6;
	/**
	 * a partner entity sent us an error response
	 */
	public static final int	M_GotNack = 7;
	/**
	 * external store (such as for receipts) unavailable
	 */
	public static final int M_ExtUnavail = 8;

	/**
	 * the number of (random) bytes used for randomizer elements
	 */
	public static final int	M_RandomizerSize = 16;

	/**
	 * the character used to join components of key hashes
	 */
	protected final String	M_JoinChar = "+";

	/**
	 * the owning broker entry object
	 */
	protected MVOBrokerEntry	m_Parent;

	/**
	 * the current step number, which begins at 1 when the object is created
	 * and continues increasing until the end of processing is reached
	 */
	protected int				m_CurrStep;

	/**
	 * the step number from which we reached the current step (previous)
	 */
	protected int				m_PrevStep;

	/**
	 * the last step number we achieved successfully without an error
	 */
	protected int				m_Achieved;

	/**
	 * the maximum number of steps (the step constituting completion, where we
	 * would send the reply to the original request)
	 */
	protected int				m_FinalStep;

	/**
	 * the failure mode code, if we got to this step because of a timeout or
	 * some other error (see M_* modes described above)
	 */
	protected int				m_FailureMode;

	/**
	 * an optional textual error message which accompanies the failure mode
	 */
	protected String			m_FailureMsg;

	/**
	 * a list of receipts, either generated or fetched for the dApp user
	 */
	protected ArrayList<ReceiptBlock>	m_Receipts;

	/**
	 * the reply we got from an Auditor generating or fetching a list of keys
	 * (either for eNFTs or for receipts)
	 */
	protected AuditorKeyBlock	m_KeyResponse;

	/**
	 * whether we're currently expecting a reply from an Auditor acting as a
	 * keyserver
	 */
	protected boolean			m_ExpectingKeyServerReply;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param brokerEnt the parent object to which we are attached
	 */
	public ReqSvcState(MVOBrokerEntry brokerEnt) {
		m_Parent = brokerEnt;
		m_CurrStep = 1;
		m_FinalStep = 2;
		m_FailureMsg = "";
		m_Receipts = new ArrayList<ReceiptBlock>();
		m_KeyResponse = new AuditorKeyBlock(m_Parent.getLog());
	}

	/**
	 * method to advance the state
	 * @param last the last step we attempted to perform (the from state)
	 * @param next the step number to go to
	 * @param reason the success/failure code causing us to go to this event
	 * @param errMsg a text error message describing the failure, or empty
	 * @return whether the state advance worked successfully
	 */
	abstract boolean advanceState(int last, int next, int reason,
								  String errMsg);

	/**
	 * method to record that we successfully fulfilled a given step
	 * @param step the step number we just completed okay.
	 * If this step is out of range or prior to a step already completed, this
	 * call will do nothing.
	 */
	public void recordSuccess(int step) {
		if (step > 0 && step <= m_FinalStep && step > m_Achieved) {
			m_Achieved = step;
		}
	}

	/**
	 * method to return the current step being worked on (for calls outside of
	 * a subclass)
	 * @return the current state
	 */
	public int getCurrentStep() { return m_CurrStep; }

	/**
	 * method to return the previous step worked on (for calls outside of
	 * a subclass)
	 * @return the previous state
	 */
	public int getPreviousStep() { return m_PrevStep; }

	/**
	 * obtain the receipt list (for calls outside of a subclass)
	 * @return the list of receipts created or obtained (side-effects intended)
	 */
	public ArrayList<ReceiptBlock> getReceipts() { return m_Receipts; }

	/**
	 * obtain the key block from an Auditor
	 * @return the reply received from the Auditor to our key create/fetch
	 */
	public AuditorKeyBlock getKeyReply() { return m_KeyResponse; }

	/**
	 * utility method to obtain a guaranteed unique eNFT ID value
	 * @param chainId the chain ID per standard
	 * @return a suitable ID, guaranteed to be unique on the chain
	 */
	public String getUniqueNFTId(long chainId) {
		if (chainId <= 0L) {
			m_Parent.getLog().error("getUniqueNFTId: missing chain Id");
			return "";
		}
		MVO mvo = m_Parent.getMVO();
		MVOState stObj = mvo.getStateObj();
		BlockchainAPI web3j = mvo.getWeb3(chainId);
		SmartContractConfig sConfig = mvo.getSCConfig(chainId);
		boolean unique = false;
		String newID = null;
		while (!unique) {
			newID = stObj.getNextID();
			// find whether an eNFT exists with this ID (circulating or not)
			Future<Boolean> nftFut
				= web3j.checkIfIdUsed(chainId, sConfig.getABI(), newID);
			Boolean exists = null;
			try {
				exists = nftFut.get();
				if (exists != null) {
					unique = !exists.booleanValue();
				}
				else {
					m_Parent.getLog().error(
							"getUniqueNFTId: checkIfIdUsed() returned null");
					unique = true;	// assume
				}
			}
			catch (InterruptedException | ExecutionException
					| CancellationException e)
			{
				m_Parent.getLog().error("getUniqueNFTId: exception checking "
									+ "for existing eNFT with ID " + newID, e);
				unique = true;	// very safe assumption anyway
			}
		}
		return newID;
	}

	/**
	 * record a key server response
	 * @param keyServerReply the reply received
	 */
	public void recordKeyServerReply(AuditorKeyBlock keyServerReply) {
		if (keyServerReply != null) {
			m_KeyResponse = keyServerReply;
			// complete the current step, in a new thread
			Runnable completeStep = new Runnable() {
				public void run() {
					advanceState(m_PrevStep, m_CurrStep, M_NoFailure, "");
				}
			};
			// run task in WebSocketClient's Executor environment
			Executor exec
				= m_Parent.getMVO().getMVOClient().getClient().getExecutor();
			if (exec != null) {
				exec.execute(completeStep);
			}
			else {
				m_Parent.getLog().warning("recordKeyServerReply: no Executor");
				new Thread(completeStep).start();
			}
		}
	}

	/**
	 * validate the signature on an eNFT
	 * @param eNft the eNFT object
	 * @param sigAddr the address which is supposed to have signed it
	 * @return true if validated, false otherwise
	 */
	protected boolean validateENFTsig(eNFTmetadata eNft, String sigAddr) {
		if (eNft == null || sigAddr == null || sigAddr.isEmpty()) {
			return false;
		}
		String signingAddress = eNft.getSigAddr();
		if (signingAddress.equalsIgnoreCase(sigAddr)) {
			return true;
		}
		return false;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Receipts != null) {
				m_Receipts.clear();
			}
			if (m_KeyResponse != null) {
				ArrayList<AuditorKeyBlock.KeySpec>
					keys = m_KeyResponse.getKeys();
				if (keys != null) {
					keys.clear();
				}
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
