/*
 * last modified---
 * 	03-07-24 try additional AUDs as key servers if we get a failure
 * 	04-11-23 destroy ephemeral AES keys after use
 * 	04-05-23 use EIP-55 address format to find receipts
 * 	03-20-23 pass decrypted payload of original clientReq
 * 	03-16-23 retool to use ReceiptStorageDb where specified by chain's URI
 * 	03-02-23 add validateReceiptSig(), create and check real sigs
 * 	08-05-22 code for async key server accesses
 * 	07-12-22 complete debugging
 * 	07-01-22 implementation draft
 * 	04-07-22 new (placeholder)
 *
 * purpose---
 * 	state machine transition object for handling dApp receipt requests
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.MVOReceiptBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.ClientReceiptBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.ReceiptBlock;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.mvo.db.ReceiptStorageDb;
import cc.enshroud.jetty.mvo.db.ReceiptStorage;

import org.web3j.crypto.Keys;

import java.text.NumberFormat;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Map;
import java.util.Base64;
import java.net.URI;
import java.io.File;
import java.io.FileReader;
import java.io.FileFilter;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.math.BigInteger;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Savepoint;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.AsyncContext;


/**
 * This class provides the implementation of control logic to handle the various
 * stages required to perform a lookup or purge of receipts owned by a wallet.
 * It also does error handling and recovery for any problems encountered at
 * each stage.
 */
public final class ReceiptState extends ReqSvcState implements FileFilter {
	// BEGIN data members
	// aliases for steps in the deposit process
	/**
	 * request received from dApp (parsed okay, sig verified)
	 */
	public static final int M_ReqReceived = 1;
	/**
	 * receipts retrieved via receipt URI for chain, confirmed to belong to
	 * the requestor
	 */
	public static final int	M_GotReceipts = 2;
	/**
	 * keys obtained from an Auditor for receipt decryption
	 */
	public static final int M_GotKeys = 3;
	/**
	 * receipt operation completed (list | get | delete)
	 */
	public static final int	M_ReceiptOpDone = 4;
	/**
	 * reply sent to dApp
	 */
	public static final int M_Replied = 5;
	/**
	 * processing is done, clean up
	 */
	public static final int M_Completed = 6;

	/**
	 * the reply we're building for the client, containing all data
	 */
	private MVOReceiptBlock	m_ReceiptReply;

	/**
	 * local shorthand copy of MVO's logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param brokerEnt the parent object to which we are attached
	 */
	public ReceiptState(MVOBrokerEntry brokerEnt) {
		super(brokerEnt);
		m_FinalStep = M_Completed;
		m_Log = m_Parent.getMVO().log();
		m_ReceiptReply = new MVOReceiptBlock(m_Log);
	}

	/**
	 * method to advance the state (required to extend ReqSvcState)
	 * @param last the last step we attempted to perform (the from state)
	 * @param next the step number to go to
	 * @param reason the success/failure code causing us to go to this event
	 * (if an error code, the error will have occurred doing the previous step)
	 * @param errMsg a text error message describing the failure, or empty
	 * @return whether the state advance worked successfully (can be true even
	 * if handling an error reason code)
	 */
	public boolean advanceState(int last, int next, int reason, String errMsg) {
		final String lbl = this.getClass().getSimpleName() + ".advanceState: ";
	/*
		m_Log.debug(lbl + "transition from " + last + " to " + next
					+ ", reason " + reason + ", emsg \"" + errMsg + "\"");
	 */

		// examine inputs
		if (next > m_FinalStep) {
			m_Log.error(lbl + "invalid next step number: " + next);
			return false;
		}
		if (next < last) {
			m_Log.error(lbl + "reverse state transition from " + next
						+ " to " + last);
			return false;
		}
		if (reason < 0) {
			m_Log.error(lbl + "invalid step advance reason: " + reason);
			return false;
		}
		m_PrevStep = last;
		m_CurrStep = next;
		m_FailureMode = reason;
		if (errMsg != null) {
			m_FailureMsg = errMsg;
		}

		// local convenience definitions
		MVO mvo = m_Parent.getMVO();
		MVOState stObj = mvo.getStateObj();
		AsyncContext rep = m_Parent.getDappReply();
		if (rep == null) {
			m_Log.error(lbl + "missing an AsyncContext, abort");
			return false;
		}
		ReceiptHandler rHandler = mvo.getReceiptHandler();

		// handle errors, based on the 'last' step completed
		boolean ret = true;
		switch (reason) {
			case M_InvalidReq:
				// context: M_ReqReceived, bad request seen
				if (m_PrevStep != M_ReqReceived) {
					// nack with state error
					ret = inconsistentState(rep);
				}
				else {
					// nack original request
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_REQUEST,
								   "parse error: " + m_FailureMsg);
				}
				break;

			// we could not access the receipt storage
			case M_ExtUnavail:
				// context: M_ReqReceived --> M_GotReceipts
				if (m_PrevStep == M_ReqReceived) {
					// couldn't obtain receipts from blockchain's storage
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_GATEWAY,
								 	"receipt storage access error: "
									+ m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep);
				}
				break;

			// we could not reach an Auditor (for decryption keys)
			case M_UnavailAud:
				// context: M_GotReceipts --> M_GotKeys
				if (m_PrevStep == M_GotReceipts) {
					// failed to get receipt keys; nack original request
					ret = nackDapp(rep, HttpServletResponse.SC_GATEWAY_TIMEOUT,
								   "key server error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep);
				}
				break;

			case M_ProcError:
				/* Internal processing error, theoretically valid for any step.
				 * Unless we have some kind of coding error, these should not
				 * occur. To avoid the possibility of a recursive endless loop,
				 * we always bail with a nack when one of these crops up.
				 */
				String fail = "";
				int rCode = HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
				switch (m_PrevStep) {
					case M_ReqReceived:
						fail = "problem retrieving receipts: " + m_FailureMsg;
						rCode = HttpServletResponse.SC_BAD_REQUEST;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_GotReceipts:
						fail = "key server access problem: " + m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_GotKeys:
						fail = "receipt decryption or construction problem: "
								+ m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_ReceiptOpDone:
					//FALLTHROUGH
					case M_Replied:
						fail = "reply send problem: " + m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;
					
					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep);
						break;
				}
				m_Log.error(lbl + "processing error: " + fail);
				break;

			case M_ParseError:
				/* Partners can send us bogosity that didn't parse.  If they do,
				 * we have to bail because resending will probably generate the
				 * same bogosity again.  (This condition probably indicates a
				 * coding error in a partner server.)
				 */
				switch (m_PrevStep) {
					case M_ReqReceived:
					case M_GotReceipts:
					case M_GotKeys:
						m_Log.error(lbl + "parse error seen from "
									+ "partner, " + m_FailureMsg);
						ret = nackDapp(rep,
								HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
								"error processing partner reply, aborting: "
								+ m_FailureMsg);
						break;

					case M_ReceiptOpDone:
					case M_Replied:
					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep);
						break;
				}
				break;

			// partners can of course also send us error replies
			case M_GotNack:
				String nack = "";
				int nCode = HttpServletResponse.SC_BAD_GATEWAY;
				switch (m_PrevStep) {
					case M_GotReceipts:
						nack = "nack obtaining receipts, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackDapp(rep, nCode, nack);
						break;

					case M_GotKeys:
						nack = "nack obtaining receipt keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackDapp(rep, nCode, nack);
						break;

					// no peer replies expected in these states
					case M_ReqReceived:
					case M_ReceiptOpDone:
					case M_Replied:
					case M_Completed:
					default:
						// should be impossible
						m_Log.error(lbl + "unexpected reply parse error in "
									+ "state " + m_CurrStep + ", "
									+ m_FailureMsg);
						ret = inconsistentState(rep);
						break;
				}
				break;

			case M_NoFailure:
				// fall through below
				break;

			default:
				// should be impossible
				break;
		}
		if (reason != M_NoFailure) return ret;

		// handle normal (non-error) cases
		ret = true;
		m_FailureMode = M_NoFailure;
		// get the request as passed by dApp client
		ClientRequest orgReq = m_Parent.getDappRequest();
		if (!(orgReq instanceof ClientReceiptBlock)) {
			m_FailureMsg = "client request is not a ClientReceiptBlock";
			advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
						 m_FailureMsg);
			ret = false;
			return ret;
		}
		ClientReceiptBlock clientReq = (ClientReceiptBlock) orgReq;
		SmartContractConfig scc = null;
		MVOConfig mvoConf = mvo.getConfig();
		String opcode = clientReq.getOpcode();
		NumberFormat nf = NumberFormat.getNumberInstance();
		nf.setMinimumIntegerDigits(3);
		ReceiptStorageDb rctDb
			= new ReceiptStorageDb(mvo.getDbManager(), m_Log);
		switch (m_CurrStep) {
			case M_GotReceipts:
				/* Here we have successfully parsed and signature-verified the
				 * request from the client.  Our next step is to do basic
				 * sanity-checking.  If any problems are found, we nack request.
				 */
				if (m_Achieved < M_ReqReceived) {
					m_Log.error(lbl + "verify requested without valid "
								+ "request received");
					ret = inconsistentState(rep);
					break;
				}

				// obtain SC config
				scc = mvo.getSCConfig(clientReq.getChainId());
				if (scc == null) {
					// unknown chain
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " is not supported";
					m_Log.error(lbl + "req verify failure: "
								+ m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}
				URI receiptStore = scc.getReceiptURI();

				/* validate the sig of {sender} on {sender}
				 * TBD: can use real EDCSA sig validation to resolve to {sender}
				 * (currently the dApp simply passes SHA256 as the sig, because
				 * it's redundant with the signature on the entire block)
				 */
				String capSig
					= EncodingUtils.sha3(clientReq.getSender().toLowerCase());
				String capBits = clientReq.getCapability();
				if (!capBits.equals(capSig)) {
					m_FailureMsg = "capabilities signature check failure on "
									+ "receipt " + opcode + " request";
					m_Log.error(lbl + "capabilities sig check failure "
								+ "for " + clientReq.getSender() + ", S/B: "
								+ capSig);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}
				boolean gotRctErr = false;
				m_ReceiptReply.setOpcode(opcode);

				/* Use the receipt URI for this chain to
				 * obtain the receipts for this account according to protocol.
				 */
				String storePath = receiptStore.getPath();
				if (storePath.startsWith("eData")) {
					/* Temp algorithm is to support only a relative file:// URI
					 * to fetch them from disk files.  Here we assume [as does
					 * ReceiptQueue.uploadReceipt()] that all receipts are
					 * stored in eData/receipts/{chainId}/{acctId}
					 * and named {receiptId}.json.
					 */
					//TEMPCODE
					// build relative path to storage location for this chain
					String rctDir = receiptStore.getPath() + File.separator
									+ clientReq.getChainId() + File.separator
									+ Keys.toChecksumAddress(
														clientReq.getSender());
					File receiptDir = new File(rctDir);
					// make sure this exists
					if (!(receiptDir.exists() && receiptDir.isDirectory())) {
						// this isn't necessarily an error (no receipts legal)
						recordSuccess(M_GotReceipts);
						// remainder of process is moot, regardless of opcode
						ret = advanceState(m_CurrStep, M_ReceiptOpDone,
										   M_NoFailure, "");
						break;
					}

					// list all files in this directory which are .json
					File[] rctFiles = receiptDir.listFiles(this);
					if (rctFiles.length == 0) {
						// this isn't necessarily an error (no receipts is okay)
						recordSuccess(M_GotReceipts);
						// remainder of process is moot, regardless of opcode
						ret = advanceState(m_CurrStep, M_ReceiptOpDone,
										   M_NoFailure, "");
						break;
					}

					// examine opcode
					if (opcode.equals(ClientReceiptBlock.M_ReceiptList)) {
						// we have what we need: the list of filenames
						for (int fff = 0; fff < rctFiles.length; fff++) {
							File rctFile = rctFiles[fff];
							m_ReceiptReply.addFileSpec(rctFile.getName());
						}

						// go on to building response
						recordSuccess(M_GotReceipts);
						ret = advanceState(m_CurrStep, M_ReceiptOpDone,
										   M_NoFailure, "");
						break;
					}
					else if (opcode.equals(ClientReceiptBlock.M_ReceiptGet)) {
						// for each file that matches, read the file
						ArrayList<String> reqFiles = clientReq.getFileSpecs();
						int outIdx = 1;
						for (int fff = 0; fff < rctFiles.length; fff++) {
							File rctFile = rctFiles[fff];
							if (reqFiles.contains(rctFile.getName())) {
								// user wants this one
								String rctContents = "";
								try {
									FileReader fread = new FileReader(rctFile);
									Long fileLen = rctFile.length();
									char[] rctData
										= new char[fileLen.intValue()];
									int rdLen = 0;
									while (rdLen != -1 && rdLen < fileLen) {
										rdLen = fread.read(rctData);
									}
									fread.close();
									rctContents = new String(rctData);
								}
								catch (FileNotFoundException fnfe) {
									m_Log.error(lbl + "file not found", fnfe);
									gotRctErr = true;
								}
								catch (IOException ioe) {
									m_Log.error(lbl + "error reading file "
												+ rctFile, ioe);
									gotRctErr = true;
								}
								if (!gotRctErr) {
									// this should be a receipt (encrypted)
									MVOReceiptBlock.ReceiptSpec rctSpec
										= m_ReceiptReply.new ReceiptSpec();
									rctSpec.m_Filename = rctFile.getName();
									rctSpec.m_Sequence
										= "receipt" + nf.format(outIdx++);
									if (!rctSpec.m_ReceiptData.buildFromString(
																rctContents))
									{
										m_FailureMsg = "receipt file " + rctFile
														+ "did not parse";
										m_Log.error(lbl + "receipt file "
													+ rctFile
													+ " did not parse");
										gotRctErr = true;
									}
									else {
										// add receipt to output and echo name
										m_ReceiptReply.addReceipt(rctSpec);
										m_ReceiptReply.addFileSpec(
															rctFile.getName());
									}
								}
							}
						}

						// check that we got some matches
						if (m_ReceiptReply.m_Receipts.isEmpty()) {
							recordSuccess(M_GotReceipts);
							// remainder of process is moot regardless of opcode
							ret = advanceState(m_CurrStep, M_ReceiptOpDone,
												M_NoFailure, "");
							break;
						}

						// if all is well we need to go get keys now
						if (!gotRctErr) {
							recordSuccess(M_GotReceipts);
							ret = advanceState(m_CurrStep, M_GotKeys,
											   M_NoFailure, "");
							break;
						}
					}
					else if (opcode.equals(ClientReceiptBlock.M_ReceiptDel)) {
						// for each file that matches, remove the file
						ArrayList<String> reqFiles = clientReq.getFileSpecs();
						int delCnt = 0;
						for (int fff = 0; fff < rctFiles.length; fff++) {
							File rctFile = rctFiles[fff];
							if (reqFiles.contains(rctFile.getName())) {
								// user wants this one purged
								if (rctFile.delete()) {
									m_ReceiptReply.addFileSpec(
															rctFile.getName());
									delCnt++;
								}
								else {
									m_Log.error(lbl + "could not delete "
												+ "receipt file " + rctFile);
									// we must continue nonetheless
								}
							}
						}
						recordSuccess(M_GotReceipts);

						// if we deleted anything we need to go purge keys too
						if (delCnt == 0) {
							// done; go to reply instead
							ret = advanceState(m_CurrStep, M_ReceiptOpDone,
											   M_NoFailure, "");
							break;
						}
					}
					else {
						// impossible
						m_FailureMsg = "illegal receipt opcode, " + opcode;
						ret = false;
						advanceState(m_CurrStep, M_Replied, M_InvalidReq,
									 m_FailureMsg);
						break;
					}
					//TEMPCODE
				}
				else if (storePath.startsWith("dbData")) {
					String acctId = clientReq.getSender();
					String ownerHash = EncodingUtils.sha3(acctId.toLowerCase());

					// examine opcode and process
					if (opcode.equals(ClientReceiptBlock.M_ReceiptList)) {
						// get the list of all receipt IDs stored for this user
						ArrayList<String> allRctIds
							= rctDb.getReceiptIds(ownerHash,
												  clientReq.getChainId());
						if (allRctIds != null) {
							// add all found to response, as .json filenames
							for (String rId : allRctIds) {
								m_ReceiptReply.addFileSpec(rId + ".json");
							}

							// we're done, go to completion
							recordSuccess(M_GotReceipts);
							ret = advanceState(m_CurrStep, M_ReceiptOpDone,
											   M_NoFailure, "");
							break;
						}
						else {
							m_FailureMsg = "error listing receipts";
							m_Log.error(lbl + m_FailureMsg + " for address "
										+ acctId);
							gotRctErr = true;
						}
					}
					else if (opcode.equals(ClientReceiptBlock.M_ReceiptGet)) {
						// for each file requested, fetch receipt from DB
						ArrayList<String> reqFiles = clientReq.getFileSpecs();
						ArrayList<String> reqIds
							= new ArrayList<String>(reqFiles.size());
						for (String rFile : reqFiles) {
							int suffixIdx = rFile.lastIndexOf(".json");
							if (suffixIdx == -1) {
								m_Log.error("Bad input filename, " + rFile);
							}
							else {
								reqIds.add(rFile.substring(0, suffixIdx));
							}
						}
						ArrayList<ReceiptStorage> retReceipts
							= rctDb.getReceipts(reqIds, clientReq.getChainId(),
												null);
						if (retReceipts == null) {
							m_FailureMsg = "error fetching requested receipts";
							m_Log.error(lbl + m_FailureMsg + " for address "
										+ acctId);
							gotRctErr = true;
						}
						else {
							// loop through retrieved receipts
							int outIdx = 1;
							for (ReceiptStorage rct : retReceipts) {
								// this should be a receipt (still encrypted)
								MVOReceiptBlock.ReceiptSpec rctSpec
									= m_ReceiptReply.new ReceiptSpec();
								rctSpec.m_Filename = rct.getID() + ".json";
								rctSpec.m_Sequence
									= "receipt" + nf.format(outIdx++);
								if (!rctSpec.m_ReceiptData.buildFromString(
															rct.getReceipt()))
								{
									m_FailureMsg = "receipt file "
													+ rctSpec.m_Filename
													+ " did not parse";
									m_Log.error(lbl + m_FailureMsg);
									gotRctErr = true;
								}
								else {
									// must be owned by requestor
									if (rct.getOwner().equals(ownerHash)) {
										// add receipt to output and echo name
										m_ReceiptReply.addReceipt(rctSpec);
										m_ReceiptReply.addFileSpec(
															rctSpec.m_Filename);
									}
									else {
										m_Log.error(lbl + "user address "
													+ acctId + " requested a "
													+ "receipt rId "
													+ rct.getID()
													+ " not owned by them");
										m_FailureMsg = "one or more receipts "
												+ "requested not owned by you";
										gotRctErr = true;
									}
								}
							}

							// check that we got some matches
							if (!gotRctErr &&
								m_ReceiptReply.m_Receipts.isEmpty())
							{
								// empty is perfectly legal
								recordSuccess(M_GotReceipts);
								ret = advanceState(m_CurrStep, M_ReceiptOpDone,
													M_NoFailure, "");
								break;
							}
						}
					}
					else if (opcode.equals(ClientReceiptBlock.M_ReceiptDel)) {
						ArrayList<String> reqFiles = clientReq.getFileSpecs();
						if (reqFiles.isEmpty()) {
							// nothing to do
							recordSuccess(M_GotReceipts);
							// remainder of process is moot
							ret = advanceState(m_CurrStep, M_ReceiptOpDone,
											   M_NoFailure, "");
							break;
						}

						// for each file that matches, remove the receipt
						int delCnt = 0;
						// do in a single transaction
						Connection dbConn = null;
						Savepoint savePt = null;
						boolean needRollback = false;
						try {
							dbConn = mvo.getDbManager().getConnection();
							dbConn.setAutoCommit(false);
							savePt = dbConn.setSavepoint("rctDel" + acctId);
						}
						catch (SQLException se) {
							m_Log.error(lbl + "can't set up transaction for "
										+ "receipt deletions", se);
							gotRctErr = true;
						}
						if (!gotRctErr) {
							for (String delFile : reqFiles) {
								int suffixIdx = delFile.lastIndexOf(".json");
								if (suffixIdx == -1) {
									m_Log.error(lbl + "bad input filename, "
												+ delFile);
									m_FailureMsg = "illegal receipt file for "
													+ "deletion";
									needRollback = true;
									break;
								}
								String delId = delFile.substring(0, suffixIdx);
								// NB: if ownerHash doesn't match, silently fail
								if (rctDb.purgeReceipt(delId,
													   clientReq.getChainId(),
													   ownerHash,
													   dbConn))
								{
									m_ReceiptReply.addFileSpec(delFile);
									delCnt++;
								}
								else {
									m_Log.error(lbl + "error deleting receipt "
												+ "Id " + delId + " for addr "
												+ acctId);
									// we must continue nonetheless
								}
							}
						}

						// rollback or commit changes
						if (needRollback) {
							try {
								dbConn.rollback(savePt);
								// reset auto-commit
								dbConn.setAutoCommit(true);
							}
							catch (SQLException se) {
								m_Log.error(lbl
									+ "error rolling back to save point!", se);
								gotRctErr = true;
								m_FailureMsg = "receipt storage access error";
							}
							finally {
								mvo.getDbManager().closeConnection(dbConn);
							}
						}
						else {
							try {
								dbConn.commit();
								// reset auto-commit
								dbConn.setAutoCommit(true);
							}
							catch (SQLException se) {
								m_Log.error(lbl + "error committing receipt "
											+ " deletion trans");
								gotRctErr = true;
								m_FailureMsg = "receipt storage access error";
							}
							finally {
								mvo.getDbManager().closeConnection(dbConn);
							}
						}
						if (!gotRctErr) {
							// if we deleted anything we need to purge keys too
							if (delCnt == 0) {
								// done; go to reply instead
								ret = advanceState(m_CurrStep, M_ReceiptOpDone,
												   M_NoFailure, "");
								break;
							}
						}
					}
					else {
						// impossible
						m_FailureMsg = "illegal receipt opcode, " + opcode;
						ret = false;
						advanceState(m_CurrStep, M_Replied, M_InvalidReq,
									 m_FailureMsg);
					}
				}
				// TBD: add else if clauses for additional receipt storage types
				else {
					m_FailureMsg = "unsupported receipt store URI";
					m_Log.error(lbl + m_FailureMsg + ", " + receiptStore
								+ " for chainId " + clientReq.getChainId());
					gotRctErr = true;
				}

				// abort on errors
				if (gotRctErr) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// this completes this step; record success and move on
				recordSuccess(M_GotReceipts);
				ret = advanceState(m_CurrStep, M_GotKeys, M_NoFailure, "");
				break;

			case M_GotKeys:
				// it's a state error to come here on a list operation
				if (opcode.equals(ClientReceiptBlock.M_ReceiptList)) {
					m_Log.error(lbl + "erroneous invocation of GotKeys "
								+ "on a " + opcode + " opcode");

					// go on to building response, since we can
					recordSuccess(M_GotKeys);
					ret = advanceState(m_CurrStep, M_ReceiptOpDone, M_NoFailure,
										"");
					break;
				}

				/* At this point we have all receipts included in the wallet.
				 * We now need to obtain the AES-256 encryption key from the key
				 * server (any Auditor node) for each ID found in the list.
				 * To do this, we pick a random Auditor and compute:
				 * chainId+ID+owner_address and add it to the fetch request.
				 */
				if (!m_ExpectingKeyServerReply) {
					String randAud = mvo.getRandomAuditor(null);
					if (randAud.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get receipt keys failure, no "
									+ "Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// now build the key block to send to the Auditor
					MVOKeyBlock rctKeyBlock = new MVOKeyBlock(m_Log);
					rctKeyBlock.setMapping(rctKeyBlock.M_MapReceipt);
					if (opcode.equals(ClientReceiptBlock.M_ReceiptDel)) {
						// we need to purge the indicated keys
						rctKeyBlock.setOpcode(rctKeyBlock.M_OpDelete);
					}
					else {
						// we're looking up some existing keys
						rctKeyBlock.setOpcode(rctKeyBlock.M_OpGet);
					}

					// we must supply the user's orig req
					String clientJSON = clientReq.getDecryptedPayload();
					rctKeyBlock.setClientBlockJson(clientJSON);

					// for each receipt, compute chainId+ID+owner and get key
					String ownerAddr = clientReq.getSender();
					ArrayList<String> hashComps = new ArrayList<String>(3);
					if (opcode.equals(ClientReceiptBlock.M_ReceiptGet)) {
						ArrayList<MVOReceiptBlock.ReceiptSpec> receipts
							= m_ReceiptReply.getReceipts();
						for (MVOReceiptBlock.ReceiptSpec rct : receipts) {
							ReceiptBlock receipt = rct.m_ReceiptData;
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(receipt.getReceiptId());
							hashComps.add(ownerAddr.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							rctKeyBlock.addHashToList(keyHash);
							hashComps.clear();
						}
					}
					else {
						ArrayList<String> reqFiles = clientReq.getFileSpecs();
						for (String fname : reqFiles) {
							int suffixIdx = fname.indexOf(".json");
							if (suffixIdx == -1) {
								m_Log.error(lbl + "delete filename "
											+ fname + " did not contain .json");
								continue;
							}
							String rctId = fname.substring(0, suffixIdx);
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(rctId);
							hashComps.add(ownerAddr.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							rctKeyBlock.addHashToList(keyHash);
							hashComps.clear();
						}
					}

					// sign the MVOKeyBlock
					MVOSignature keySig = new MVOSignature();
					keySig.m_Signer = mvo.getMVOId();
					PrivateKey sigKey = mvoConf.getCommPrivKey();
					String sData = rctKeyBlock.buildSignedData();
					boolean sOk = true;
					if (rctKeyBlock.getErrCode() != 0) {
						sOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, therefore no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKey, sData);
						if (sig == null) {
							sOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							keySig.m_Signature = sig;
						}
					}
					if (!sOk) {
						m_FailureMsg
							= "could not build MVOKeyBlock for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					rctKeyBlock.setSignature(keySig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud, rctKeyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud);
							randAud = mvo.getRandomAuditor(excludeAIds);
							if (randAud.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg = "error getting receipt decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// processing a key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String sigAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature adSig
						= m_KeyResponse.getSignature();
					boolean sigVer = true;
					PublicKey audKey = mvoConf.getPeerPubkey(adSig.m_Signer);
					if (audKey == null) {
						sigVer = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ adSig.m_Signer + ", cannot verify sig on "
									+ "returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(audKey,
														   sigAudData,
														   adSig.m_Signature))
						{
							sigVer = false;
							m_Log.error(lbl + "sig verify failed for Aud "
										+ adSig.m_Signer + ", cannot verify sig"
										+ " on returned AuditorKeyBlock");
						}
					}
					if (!sigVer) {
						m_FailureMsg = "error looking up receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + adSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error obtaining receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all the keys, move on to next state
				recordSuccess(M_GotKeys);
				ret = advanceState(m_CurrStep, M_ReceiptOpDone, M_NoFailure,
									"");
				break;

			case M_ReceiptOpDone:
				/* Here we have all required data and need to build the
				 * appropriate response as reply to the user's request.
				 */
				if (clientReq == null || m_KeyResponse == null) {
					m_FailureMsg = "processing error, no keys found";
					m_Log.error(lbl + "missing client request and/or receipt "
								+ "AES keys, cannot build receipt reply");
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// obtain SC config
				scc = mvo.getSCConfig(clientReq.getChainId());

				// examine opcode
				if (opcode.equals(ClientReceiptBlock.M_ReceiptGet)) {
					/* Here we need to use the AES keys we retrieved from the
					 * Auditor to decrypt the receipts we already retrieved.
					 * These are then sent back to the dApp client verbatim.
					 */
					// for each receipt, decrypt it and add it to the output
					int rctIdx = 1;
					boolean allOk = true;
					ArrayList<MVOReceiptBlock.ReceiptSpec> retReceipts
						= m_ReceiptReply.getReceipts();
					ArrayList<String> hashParts = new ArrayList<String>(3);
					for (MVOReceiptBlock.ReceiptSpec rct : retReceipts) {
						ReceiptBlock receipt = rct.m_ReceiptData;
						String rctID = receipt.getReceiptId();
						hashParts.add(Long.toString(clientReq.getChainId()));
						hashParts.add(rctID);
						hashParts.add(clientReq.getSender().toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashParts);
						String keyHash = EncodingUtils.sha3(keyIdx);
						hashParts.clear();

						// find the key in the list by looking up the hash
						AuditorKeyBlock.KeySpec keySpec
							= m_KeyResponse.getKeyForHash(keyHash);
						if (keySpec == null) {
							m_FailureMsg = "processing error, missing receipt "
											+ "key";
							m_Log.error(lbl + "missing receipt AES key "
										+ "for receipt hash index " + keyIdx
										+ ", cannot build receipt reply");
							allOk = false;
							break;
						}

						// make key from this key data
						Base64.Decoder b64d = Base64.getUrlDecoder();
						byte[] keyData = b64d.decode(keySpec.m_KeyData);
						SecretKey secretKey = new SecretKeySpec(keyData, "AES");

						// decrypt the receipt using this key
						String encRctData = receipt.getEncData();
						if (encRctData.isEmpty()) {
							m_Log.error(lbl + "weird, receipt Id "
										+ rctID + " does not appear encrypted");
						}
						else {
							String decRct = EncodingUtils.decWithAES(secretKey,
																 encRctData);
							if (decRct == null || decRct.isEmpty()) {
								m_Log.error(lbl + "could not decrypt "
											+ "receipt ID " + rctID);
								m_FailureMsg
									= "processing error decrypting receipts";
								allOk = false;
								break;
							}

							// build the receipt object fields from decrypt
							if (!receipt.buildFromDecrypt(decRct)) {
								m_FailureMsg
									= "processing error parsing receipts";
								m_Log.error(lbl + "could not parse "
											+ "decrypted receipt ID " + rctID);
								allOk = false;
								break;
							}
							receipt.setEncData("");

							// get signing key for signing MVO
							MVOSignature rctSig = receipt.getSignature();
							MVOConfig signerConf
								= (MVOConfig) scc.getMVOMap().get(
															rctSig.m_Signer);
							if (signerConf == null) {
								m_FailureMsg = "error validating receipt sig";
								m_Log.error(lbl + "cannot find config for "
											+ "receipt signer, "
											+ rctSig.m_Signer);
								allOk = false;
								break;
							}
							BlockchainConfig signerChConf
								= signerConf.getChainConfig(
														clientReq.getChainId());
							if (signerChConf == null) {
								m_FailureMsg = "error validating receipt sig";
								m_Log.error(lbl + "cannot find chain config for"
											+ " receipt signer "
											+ rctSig.m_Signer + " on chainId "
											+ clientReq.getChainId());
								allOk = false;
								break;
							}
							String signerAddr
								= signerChConf.getSigningAddress();

							// verify the signature on the receipt
							if (!validateReceiptSig(receipt, signerAddr)) {
								m_FailureMsg = "signature did not validate on "
												+ "decrypted receipt";
								m_Log.error(lbl + "sig verification failed on "
											+ "decrypted receipt ID " + rctID);
								allOk = false;
								break;
							}
						}
					}
					if (!allOk) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}
				else {
					/* ClientReceiptBlock.M_ReceiptList
					 * Here we simply return the filenames we retrieved.
					 * (The client will presumably select any or all and make a
					 * subsequent Get request to obtain the actual receipts.)
					 * We already have the filenames loaded in the reply, so
					 * all we need to do in this step is sign and send.
					 */
					/* ClientReceiptBlock.M_ReceiptDel
					 * Here we have already deleted the receipts, together with
					 * their corresponding AES keys.  We need only return the
					 * filenames of the purged receipts as confirmation.  So
					 * again, all we do is sign and send.
					 */
					//m_Log.debug(lbl + "M_ReceiptOpDone for opcode " + opcode);
				}

				// affix our signature to the block
				String sigData = m_ReceiptReply.buildSignedData();
				// get our signing key for this chain
				BlockchainConfig bConfig
					= mvoConf.getChainConfig(scc.getChainId());
				if (bConfig == null) {
					// shouldn't occur!
					m_Log.error(lbl + "no MVO config for chain "
								+ scc.getChainName() + ", cannot "
								+ "sign MVOReceiptBlock");
					m_FailureMsg = "could not sign receipt reply block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				BigInteger bcSigKey = bConfig.getSigningKey();
				MVOSignature receiptSig = new MVOSignature();
				receiptSig.m_Signer = mvo.getMVOId();
				String ourSig = EncodingUtils.signData(sigData, bcSigKey);
				if (ourSig == null) {
					m_Log.error(lbl + "error signing receipt reply block");
					m_FailureMsg = "could not sign receipt reply block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				receiptSig.m_Signature = ourSig;
				m_ReceiptReply.setSignature(receiptSig);

				// send back the reply to the dApp client
				recordSuccess(M_ReceiptOpDone);
				ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				break;

			case M_Replied:
				/* Send our reply to the original client request.  This
				 * includes the MVOWalletBlock we created, with our signature.
				 */
				HttpServletResponse httpRep
					= (HttpServletResponse) rep.getResponse();
				if (!rHandler.sendNormalRep(httpRep, m_ReceiptReply,
											clientReq.getReplyKey()))
				{
					m_Log.error(lbl + "unable to send client reply "
								+ "after successful completion");
					rep.complete();
					ret = false;
					advanceState(m_CurrStep, m_FinalStep, M_ProcError,
								"client reply failure");
				}
				else {
					// normal progression to next step
					rep.complete();
					recordSuccess(M_Replied);
					ret = advanceState(m_CurrStep, m_FinalStep, M_NoFailure,
										"");
				}
				break;

			case M_Completed:
				/* Here all of our tasks have been completed, and the last step
				 * is to clean up data structures, removing temporary data.
				 * Garbage collection and finalize() calls will auto-trigger
				 * the next time System.gc() runs.
				 */
				// remove ourselves from the broker map if we are in it
				Map<String, MVOBrokerEntry> brokerMap
					= stObj.getBrokerMap();
				String tId = m_Parent.getTransId();
				if (tId != null && !tId.isEmpty()) {
					brokerMap.remove(tId, m_Parent);
					m_Parent = null;
				}

				// if we have a receipt reply built, forget the receipts
				ArrayList<MVOReceiptBlock.ReceiptSpec> receiptEntries
					= m_ReceiptReply.getReceipts();
				receiptEntries.clear();
				ArrayList<String> receiptFilenames
					= m_ReceiptReply.getFileSpecs();
				receiptFilenames.clear();
				m_KeyResponse = null;
				break;

			default:
				// all other states don't make sense
				m_Log.error(lbl + "illegal state transition to "
							+ m_CurrStep);
				ret = inconsistentState(rep);
				break;
		}
		return ret;
	}

	/**
	 * obtain receipt reply
	 * @return the object being built for the reply
	 */
	public MVOReceiptBlock getReceiptReply() { return m_ReceiptReply; }

	// implement FileFilter (TEMPCODE, used for file:// receipt storage chains)
	/**
	 * select files in a directory
	 * @param file an input file
	 * @return true if file should be accepted
	 */
	public boolean accept(File file) {
		if (file.getName().endsWith(".json")) {
			return true;
		}
		return false;
	}

	/**
	 * method to send pending error reply to dApp via ReceiptHandler
	 * @param resp the dApp response object
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackDapp(AsyncContext resp, int sc, String errTxt) {
		if (resp != null) {
			HttpServletResponse httpRep
				= (HttpServletResponse) resp.getResponse();
			m_Parent.getMVO().getReceiptHandler().sendErrRep(httpRep,
															sc, errTxt);
			resp.complete();
			recordSuccess(M_Replied);
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * Method to log when reason versus state is inconsistent.  Note this
	 * implies an internal programming error. The dApp client is nack'd.
	 * @param rep the dApp response object
	 * @return true on success
	 */
	private boolean inconsistentState(AsyncContext rep) {
		m_Log.error("ReceiptState internal inconsistency, step " + m_CurrStep
					+ ", reason " + m_FailureMode + ", emsg \"" + m_FailureMsg
					+ "\"");
		return nackDapp(rep, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						"Internal state error, abort");
	}

	/**
	 * method to validate the signature on a receipt
	 * @param receipt the receipt object
	 * @param sigAddr the address which is supposed to have signed it
	 * @return true if validate, else false
	 */
	public boolean validateReceiptSig(ReceiptBlock receipt, String sigAddr) {
		if (receipt == null || sigAddr == null || sigAddr.isEmpty()) {
			return false;
		}
		String signingAddress = receipt.getSigAddr();
		if (signingAddress.equalsIgnoreCase(sigAddr)) {
			return true;
		}
		return false;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Receipts != null) {
				m_Receipts.clear();
			}
			m_ReceiptReply = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
