/*
 * last modified---
 * 	01-16-24 debug recordEnftBurn()
 * 	12-20-23 add recordUnGreylist() stub to EnftListener
 * 	12-18-23 add recordBlock() stub to EnftListener
 * 	11-30-23 implement checks for uploadability of queued receipts
 * 	11-28-23 supply an EnftListener
 * 	04-05-23 use EIP-55 format for account owner addresses for eData method,
 * 			 and lowercase for hash for dbData method
 * 	03-16-23 implement dbData/receipts storage URIs using DB
 * 	07-20-22 fix parsing an empty receipt
 * 	07-12-22 improve error message labeling
 * 	06-29-22 new
 *
 * purpose---
 * 	provide an object which can buffer up receipts to be uploaded, and guarantee
 * 	delivery of them in order when a storage connection becomes available
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ReceiptBlock;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.EnftListener;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.mvo.db.ReceiptStorageDb;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;

import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;
import org.web3j.protocol.Web3j;
import org.web3j.protocol.core.methods.request.EthFilter;
import org.web3j.protocol.core.methods.response.EthLog;
import org.web3j.protocol.core.DefaultBlockParameterNumber;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.crypto.Credentials;
import org.web3j.tx.gas.DefaultGasProvider;
import org.web3j.abi.EventEncoder;
import org.web3j.abi.TypeEncoder;
import org.web3j.abi.datatypes.Address;
import org.web3j.abi.EventValues;
import org.web3j.tx.Contract;

import java.util.Iterator;
import java.util.ArrayList;
import java.util.List;
import java.util.Base64;
import java.util.concurrent.LinkedBlockingQueue;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.PrintWriter;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.FileWriter;
import java.net.URI;
import java.math.BigInteger;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;


/**
 * This class implements a mechanism for delayed (but guaranteed) delivery of
 * receipts which must be sent to the storage.  Its behavior is such that a 
 * separate work thread is provided which does nothing until an entry appears
 * in the queue, at which point staggered delivery attempts commence.  It is
 * assumed that the original request which caused the entry to be placed in
 * the queue has already been replied to.
 */
public final class ReceiptQueue implements Runnable, EnftListener {
	// BEGIN data members
	/**
	 * constant defining length of time to wait between queue and connection
	 * checks (msec) for queue drain attempts
	 */
	public final long		M_CheckInterval = 10000L;

	/**
	 * constant defining spacing between messages sent when draining the queue
	 * (to prevent message flooding, msec)
	 */
	private final long		M_MsgSpacing = 500L;

	/**
	 * constant defining how long to wait for the upload before timing out
	 * (msec)
	 */
	private final long		M_Timeout = 60000L;

	/**
	 * constant defining the multiple of the "dwell time" on a given chain,
	 * which we use to determine when to give up on uploading a queued receipt
	 */
	private final int		M_DwellMultiple = 100;

	/**
	 * the MVO object which owns us
	 */
	private MVO				m_MVO;

	/**
	 * inner class to describe a queued entry (also represented as a single line
	 * in the backing file)
	 */
	public final class ReceiptEntry {
		/**
		 * flag whether this receipt entry is eligible for upload
		 */
		public boolean		m_Uploadable;

		/**
		 * ID of blockchain for which this receipt was generated
		 */
		public long			m_ChainId;

		/**
		 * block number when this entry was originally queued (used for age)
		 */
		public BigInteger	m_BlockWhenQueued;

		/**
		 * json text of ReceiptBlock
		 */
		public String		m_ReceiptJson;

		/**
		 * the ReceiptBlock itself
		 */
		public ReceiptBlock	m_Receipt;

		/**
		 * the AES key which has been assigned to this receipt ID
		 */
		public String		m_AESkey;

		/**
		 * nullary constructor
		 */
		public ReceiptEntry() {
			m_ReceiptJson = m_AESkey = "";
		}

		/**
		 * emit entry as text suitable for inclusion in the queue
		 */
		public String asText() {
			String output = Boolean.toString(m_Uploadable) + "::"
							+ m_ChainId + "::" + m_BlockWhenQueued + "::"
							+ m_ReceiptJson + "::" + m_AESkey;
			return output;
		}

		/**
		 * build receipt block from JSON text
		 * @return true on success
		 */
		public boolean buildReceipt() {
			if (m_Receipt == null) {
				// allocate one
				m_Receipt = new ReceiptBlock(m_Log);
			}
			if (!m_Receipt.buildFromString(m_ReceiptJson)) {
				m_Log.error("ReceiptEntry.buildReceipt: parse error "
							+ m_Receipt.getErrCode());
				return false;
			}
			return true;
		}

		/**
		 * update JSON text based on receipt block
		 */
		public void updateJson() {
			StringBuilder json = new StringBuilder(1024);
			if (m_Receipt == null) {
				m_Log.error("ReceiptEntry.updateJson: no receipt built yet");
				return;
			}
			m_Receipt.addJSON(json);
			m_ReceiptJson = json.toString();
		}
	}

	/**
	 * the message queue
	 */
	private LinkedBlockingQueue<ReceiptEntry>	m_Queue;

	/**
	 * the thread that does all the work
	 */
	private Thread			m_WorkThread;

	/**
	 * the stop trigger, to cause the work thread to exit when it becomes null
	 */
	private volatile Thread	m_StopTrigger;

	/**
	 * the path to the disc file backing our queue
	 */
	private String			m_QueuePath;

	/**
	 * dummy credentials used to search for events (randomly generated)
	 */
	private Credentials		m_EventCreds;

	/**
	 * local copy of MVO's logging object
	 */
	private Log				m_Log;

	// END data member

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the plugin object of which we are a part
	 * @param qPath the full pathname to the disc file used to back the queue
	 */
	public ReceiptQueue(MVO mvo, String qPath) {
		m_MVO = mvo;
		m_Log = m_MVO.log();
		if (qPath == null || qPath.isEmpty()) {
			m_Log.error("ReceiptQueue constructor: null queue pathname");
		}
		m_QueuePath = new String(qPath);
		m_Queue = new LinkedBlockingQueue<ReceiptEntry>();
	}

	/**
	 * Start method: parses all entries in queue backing file, and starts up
	 * the thread.  Also checks queued entries to see whether they should now
	 * be uploaded, in the event this MVO has been offline for an interval.
	 * @return true if all went okay, false if queue could not be started
	 */
	public boolean start() {
		final String lbl = this.getClass().getSimpleName() + ".start: ";
		// pre-load the queue with any records found saved in the disc file
		if (m_QueuePath == null || m_QueuePath.isEmpty()) {
			m_Log.error(lbl + "queue path file not set");
			return false;
		}

		// open reader on file
		FileReader fr = null;
		try {
			fr = new FileReader(m_QueuePath);
		}
		catch (FileNotFoundException fnfe) {
			m_Log.error(lbl + "could not open queue backing file", fnfe);
			return false;
		}
		BufferedReader br = new BufferedReader(fr);

		String line = null;
		boolean ok = true;
		int cnt = 0;
		do {
			// each line should be one saved request message
			try {
				line = br.readLine();
				// allow for empty line
				if (line == null || line.isEmpty()) {
					continue;
				}

				// try to interpret this as a queued ReceiptEntry
				String[] lineParts = line.split("::");
				if (lineParts.length != 5) {
					m_Log.error(lbl + "illegal format, receipt queue entry: "
								+ line);
					continue;
				}
				cnt++;

				String uploadable = lineParts[0];
				String chain = lineParts[1];
				String block = lineParts[2];
				String receipt = lineParts[3];
				// if there is no receipt text, throw it out
				if (receipt.isEmpty()) {
					m_Log.warning(lbl + "empty receipt in queue, line " + cnt);
					continue;
				}
				String key = lineParts[4];
				ReceiptEntry entry = new ReceiptEntry();
				boolean lineOk = true;

				// parse each of the 5 parts for correctness
				entry.m_Uploadable = Boolean.valueOf(uploadable);
				try {
					Long chId = Long.parseLong(chain.trim());
					entry.m_ChainId = chId;
				}
				catch (NumberFormatException nfe) {
					m_Log.error(lbl + "illegal chainId, " + chain
								+ " in receipt queue line " + cnt);
					lineOk = false;
				}
				try {
					BigInteger blk = new BigInteger(block);
					entry.m_BlockWhenQueued = blk;
				}
				catch (NumberFormatException nfe) {
					m_Log.error(lbl + "illegal blockId, " + block
								+ " in receipt queue line " + cnt);
					lineOk = false;
				}
				entry.m_ReceiptJson = receipt;
				if (!entry.buildReceipt()) {
					m_Log.error(lbl + "receipt data in receipt queue line "
								+ cnt + " does not parse, " + receipt);
					lineOk = false;
				}
				if (!key.isEmpty()) {
					// assume this is a correct field (should be base64)
					entry.m_AESkey = key;
				}

				if (lineOk) {
					// it's okay, add it to the queue
					try {
						m_Queue.put(entry);
					}
					catch (InterruptedException ie) {
						m_Log.error(lbl + "interrupted adding to "
									+ "receipt queue from disc file", ie);
						ok = false;
					}
				}
				else {
					m_Log.error(lbl + "non-receipt entry found "
								+ "in queue file, " + line);
					ok = false;
				}
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "receipt backing queue file read error", ioe);
				ok = false;
			}
		} while (line != null);
		try {
			fr.close();
		}
		catch (IOException ioe) { /* ignore */ }

		// dummy up Credentials for read-only blockchain queries for events
		try {
			m_EventCreds = Credentials.create(Keys.createEcKeyPair());
		}
		catch (Exception eee) {
			m_Log.error(lbl + "exception generating dummy Credentials", eee);
			ok = false;
		}

		// bail if we saw a problem (MVO will exit)
		if (!ok) return ok;

		// now deal with thread startup
		m_WorkThread = new Thread(this);
		m_WorkThread.setDaemon(true);
		m_StopTrigger = m_WorkThread;
		m_WorkThread.start();
		m_Log.debug(lbl + "receipt queue initialized with "
					+ m_Queue.size() + " elements from file");

		/* Examine all entries for which m_Uploadable is false.  These should
		 * be checked for tagID values which have indeed been minted on the
		 * relevant chain.
		 *
		 * If found, these should be finalized, signed, encrypted, and
		 * marked uploadable.  (Whereupon the run() loop will upload them.)
		 */
		Iterator<ReceiptEntry> queueIt = m_Queue.iterator();
		boolean didRms = false;
		while (queueIt.hasNext()) {
			ReceiptEntry qEntry = (ReceiptEntry) queueIt.next();
			if (!qEntry.m_Uploadable) {
				String rctId = qEntry.m_Receipt.getReceiptId();

				// grab the EnftCache object for this chain
				SmartContractConfig scc = m_MVO.getSCConfig(qEntry.m_ChainId);
				if (scc == null) {
					m_Log.error(lbl + "no SmartContractConfig for chainId "
								+ qEntry.m_ChainId + ", cannot check queued "
								+ "receipt Id " + rctId
								+ " for upload readiness");
					continue;
				}
				EnftCache eCache = scc.getCache();
				if (eCache == null) {
					m_Log.error(lbl + "no EnftCache for chainId "
								+ qEntry.m_ChainId + ", cannot check queued "
								+ "receipt Id " + rctId
								+ " for upload readiness");
					continue;
				}

				/* Ask cache whether the tag entry has been minted or burned.
				 * This means we have seen URI metadata for it, regardless of
				 * whether it's been burned also.  We do not check greylisting,
				 * because even if the output got greylisted it still arguably
				 * "happened", it just isn't usable as a new transaction input.
				 * Also, if the eNFT ended up greylisted for longer than our
				 * expiration interval, we'd purge it before it ever got stored.
				 */
				BigInteger eId = null;
				try {
					eId = new BigInteger(qEntry.m_Receipt.getReceiptId(), 16);
				}
				catch (NumberFormatException nfe) {
					m_Log.error(lbl + "illegal receiptId value, " + rctId);
					// NB: do not dequeue because of this
					continue;
				}
				String metadata = eCache.getMetadataForId(eId);
				if (metadata != null) {
					// this receipt may be eligible for completion and upload
					if (!prepReceiptForUpload(qEntry, null)) {
						m_Log.error(lbl + "error preparing receiptId " + rctId
									+ " for upload; removing from queue");
						// remove from queue to prevent further errors
						queueIt.remove();
						didRms = true;
					}
					else {
						m_Log.debug(lbl + "receiptId " + rctId
									+ " prepped and readied for upload");
					}
				}
				else {
					/* Check whether elapsed number of blocks exceeds our
					 * receipt expiration interval.  If it does then we must
					 * conclude that this item was never submitted or failed.
					 * The interval is arbitrarily set to 100x the dwell time
					 * for this blockchain.  Note that even an extended period
					 * of downtime for this MVO will not hit this case, because
					 * it should get picked up above with non-null metadata.
					 */
					Integer blockCnt = scc.getDwellTime() * M_DwellMultiple;
					BigInteger lifetime = new BigInteger(blockCnt.toString());
					BigInteger blockAge
						= qEntry.m_BlockWhenQueued.add(lifetime);
					if (eCache.getLatestBlock().compareTo(blockAge) > 0) {
						m_Log.debug(lbl + "receiptId " + rctId + " on chain "
									+ qEntry.m_ChainId + " ("
									+ scc.getChainName()
									+ ") has reached EOL after > " + lifetime
									+ " blocks; dequeuing");
						queueIt.remove();
						didRms = true;
					}
				}
			}
		}

		// resync queue file if we changed it
		if (didRms) {
			syncQueueFile();
		}
		return ok;
	}

	/**
	 * stop method
	 * @return true if all queue contents were written to backing file,
	 * false otherwise
	 */
	public boolean stop() {
		// set the trigger that will make the run loop exit
		m_StopTrigger = null;

		// NB: this will overwrite with a blank file if queue is now empty
		return syncQueueFile();
	}

	/**
	 * private helper method to write all queue elements to the backing file
	 * @return true if all queue contents were written to backing file,
	 * false otherwise
	 */
	private synchronized boolean syncQueueFile() {
		// take any remaining queue entries and save them to backing disc file
		PrintWriter pw = null;
		try {
			pw = new PrintWriter(m_QueuePath);
		}
		catch (FileNotFoundException fnfe) {
			m_Log.error("ReceiptQueue.syncQueueFile: could not open receipt "
						+ "queue backing file for write");
			return false;
		}

		/* output the text for each entry in our list
		 * NB: if there are no elements, we will write an empty file.  This may
		 * be the intended effect, if we just removed the only element present.
		 */
		for (ReceiptEntry qEntry : m_Queue) {
			pw.println(qEntry.asText());
		}
		pw.flush();
		pw.close();
		return true;
	}

	// method to implement Runnable
	/**
	 * main method to poll queue and perform uploads
	 */
	public void run() {
		for (;;) {
			// break out of loop if we've been stopped
			Thread thisThread = Thread.currentThread();
			if (thisThread != m_StopTrigger) {
				break;
			}

			/* See if anything is in the queue.  We use peek() and not take()
			 * because we cannot be guaranteed that this thread will never be
			 * terminated ungracefully, and we don't want to remove a message
			 * that wasn't actually processed.
			 */
			ReceiptEntry qEnt = m_Queue.peek();
			ReceiptEntry uploadRct = null;
			if (qEnt == null) {
				// wait the recheck interval
				try {
					Thread.sleep(M_CheckInterval);
				}
				catch (InterruptedException ie) { /* ignore */ }
				continue;
			}

			// see whether this item is uploadable
			if (qEnt.m_Uploadable) {
				uploadRct = qEnt;
			}

			// if there is more than one item, iterate looking for uploadable
			if (uploadRct == null && m_Queue.size() > 1) {
				Iterator<ReceiptEntry> queueIt = m_Queue.iterator();
				while (queueIt.hasNext()) {
					ReceiptEntry qEntry = (ReceiptEntry) queueIt.next();
					if (qEntry.m_Uploadable) {
						uploadRct = qEntry;
						break;
					}
				}
			}

			// if we didn't find anything uploadable, wait recheck interval
			if (uploadRct == null) {
				try {
					Thread.sleep(M_CheckInterval);
				}
				catch (InterruptedException ie) { /* ignore */ }
				continue;
			}

			// see if we currently have a valid connection to the storage
			boolean uploadOk = true;
			SmartContractConfig scc = m_MVO.getSCConfig(uploadRct.m_ChainId);
			if (scc == null) {
				m_Log.error("ReceiptQueue.run: no chain Id "
							+ uploadRct.m_ChainId
							+ " support for queued receipt, skipping");
				// remove so we don't stick on this item
				m_Queue.remove(uploadRct);
				continue;
			}
			URI storage = scc.getReceiptURI();

			// upload the item
			if (!uploadReceipt(uploadRct, storage)) {
				// failed; wait interval before trying again
				try {
					Thread.sleep(M_CheckInterval);
				}
				catch (InterruptedException ie) { /* ignore */ }
				continue;
			}

			// remove from list
			if (!dequeueReceipt(uploadRct)) {
				m_Log.error("ReceiptQueue.run: error dequeuing uploaded "
							+ "receipt: " + uploadRct.asText());
			}

			// wait drain interval and continue
			try {
				Thread.sleep(M_MsgSpacing);
			}
			catch (InterruptedException ie) { /* ignore */ }
		}

		m_Log.debug("ReceiptQueue.run: work thread terminated");
	}

	/**
	 * method to queue a receipt (which must be correct and complete)
	 * in both the memory queue and the backing flat file
	 * @param receipt the wrapped receipt to be sent
	 * @return true on success, false on failure
	 */
	public boolean queueReceipt(ReceiptEntry receipt) {
		if (receipt == null) {
			m_Log.error("ReceiptQueue.queueReceipt: attempt to queue missing "
						+ "receipt");
			return false;
		}

		// add to in-memory queue
		try {
			m_Queue.put(receipt);
		}
		catch (InterruptedException ie) {
			m_Log.error("ReceiptQueue.queueReceipt: interrupted queuing "
						+ "receipt", ie);
			return false;
		}

		// now add to backing flat file by re-writing queue into it
		return syncQueueFile();
	}

	/**
	 * method to remove a receipt from the queue (both in-memory and flat file)
	 * @param receipt the receipt to be removed
	 * @return true on success, false on failure
	 */
	private boolean dequeueReceipt(ReceiptEntry receipt) {
		if (receipt == null) {
			m_Log.error("ReceiptQueue.dequeueReceipt: attempt to dequeue "
						+ "improper receipt");
			return false;
		}

		// take out of in-memory queue
		if (!m_Queue.remove(receipt)) {
			return false;
		}

		/* now remove from backing flat file by re-writing queue into it
		 * (may result in overwriting our backing file with one of zero length)
		 */
		return syncQueueFile();
	}

	/**
	 * method to find a given receipt entry in the queue
	 * @param receiptId the unique Id to search for
	 * @return the entry containing this receipt, or null if not found
	 */
	public ReceiptEntry findReceipt(String receiptId) {
		ReceiptEntry rctFound = null;
		if (receiptId == null || receiptId.isEmpty()) {
			return rctFound;
		}
		Iterator<ReceiptEntry> queueIt = m_Queue.iterator();
		while (queueIt.hasNext()) {
			ReceiptEntry qEntry = (ReceiptEntry) queueIt.next();
			if (qEntry.m_Receipt.getReceiptId().equals(receiptId)) {
				rctFound = qEntry;
				break;
			}
		}
		return rctFound;
	}

	/**
	 * Method to prepare a receipt for upload, by completing, signing, and
	 * encrypting it.  This is done only for receipts which have seen their
	 * tag eNFT Id appear on the blockchain.
	 * @param qEntry the queue entry we are to prepare
	 * @param block the block in which the tag appeared, or null if not known
	 * @return true on success, false on failure
	 */
	public boolean prepReceiptForUpload(ReceiptEntry qEntry, BigInteger block) {
		final String lbl
			= this.getClass().getSimpleName() + ".prepReceiptForUpload: ";
		if (qEntry == null || qEntry.m_Receipt == null || qEntry.m_Uploadable) {
			m_Log.error(lbl + "missing receipt, or already uploadable");
			return false;
		}
		ReceiptBlock receipt = qEntry.m_Receipt;
		String rctId = receipt.getReceiptId();
		// grab the config and EnftCache object for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(qEntry.m_ChainId);
		if (scc == null) {
			m_Log.error(lbl + "no SmartContractConfig for chainId "
						+ qEntry.m_ChainId + ", cannot prepare queued "
						+ "receipt Id " + rctId + " for upload");
			return false;
		}
		EnftCache eCache = scc.getCache();
		if (eCache == null) {
			m_Log.error(lbl + "no EnftCache for chainId "
						+ qEntry.m_ChainId + ", cannot prepare queued "
						+ "receipt Id " + rctId + " for upload");
			return false;
		}

		// get our signing key for this chain
		MVOConfig mvoConf = m_MVO.getConfig();
		BlockchainConfig bConf = mvoConf.getChainConfig(qEntry.m_ChainId);
		if (bConf == null) {
			m_Log.error(lbl + "no MVO config for chain " + scc.getChainName()
						+ ", cannot sign receipts for it");
			return false;
		}
		BigInteger sigKey = bConf.getSigningKey();

		// entry must have an AES key already generated
		if (qEntry.m_AESkey.isEmpty()) {
			m_Log.error(lbl + "receiptId " + rctId
					+ " had no AES key in queue, cannot prepare for upload");
			return false;
		}

		// obtain the block where the receipt.m_TagID appeared, if not passed
		if (block == null) {
			/* First we must determine whether this receipt is a candidate for
			 * checking a burn value as its tag.  This is only done in the case
			 * where a Burn/Withdraw was done without any change output eNFT.
			 * In this case, the tag value will have "BURN:" prepended to it.
			 *
			 * In this case we'll look for a matching TransferSingle where:
			 * 	m_Source = TransferSingle.operator (i.e. user does own burn)
			 * 	m_Source = TransferSingle.from (user that burned it)
			 *	TransferSingle.to = 0x0 (i.e. it's a burn)
			 *	m_TagID = "BURN:" + TransferSingle.id (not indexed)
			 *
			 * In all normal cases we'll look for a mint event, one in which:
			 * 	m_Source = TransferSingle.operator (i.e. sent from source)
			 * 	TransferSingle.from = 0x0 (i.e. it's a mint)
			 * 	m_TagID = TransferSingle.id (not indexed)
			 */
			boolean burnCandidate = false;
			String realTagId = receipt.getTagID();
			if (realTagId.startsWith("BURN:")) {
				burnCandidate = true;
				realTagId = realTagId.substring(5);
			}

			// get the deployed EnshroudProtocol contract for this chain
			String protocolAddr = EnshroudProtocol.getPreviouslyDeployedAddress(
											Long.toString(scc.getChainId()));
			// search from deployment block on chain through Latest
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(scc.getDeploymentBlock());
			DefaultBlockParameterName endBlock
				= DefaultBlockParameterName.LATEST;
			// get loaded EnshroudProtocol wrapper object
			Web3j web3j = scc.getABI();
			EnshroudProtocol wrapper
				= EnshroudProtocol.load(protocolAddr,
										web3j,
										m_EventCreds,
										new DefaultGasProvider());
			EthLog ethLog = null;

			// look for a Burn special case
			if (burnCandidate) {
				// now try to find a matching burn using a different filter
				EthFilter burnFilter
					= new EthFilter(startBlock, endBlock, protocolAddr);
				burnFilter.addSingleTopic(EventEncoder.encode(
										EnshroudProtocol.TRANSFERSINGLE_EVENT));
				// first indexed topic: TransferSingle.operator
				burnFilter.addOptionalTopics("0x" + TypeEncoder.encode(
											new Address(receipt.getSource())));
				// second indexed topic: TransferSingle.from
				burnFilter.addOptionalTopics("0x" + TypeEncoder.encode(
											new Address(receipt.getSource())));
				// third indexed topic: TransferSingle.to
				burnFilter.addOptionalTopics(
									"0x" + TypeEncoder.encode(Address.DEFAULT));
				boolean burnMatch = false;
				try {
					ethLog = web3j.ethGetLogs(burnFilter).send();
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "unable to get EthLogs for burn filter",
								ioe);
					return false;
				}
				List<EthLog.LogResult> burnEvents = ethLog.getLogs();
				for (EthLog.LogResult eLog : burnEvents) {
					EthLog.LogObject logObj = (EthLog.LogObject) eLog;
					org.web3j.protocol.core.methods.response.Log log
																= logObj.get();
					EventValues eventValues
						= Contract.staticExtractEventParameters(
									EnshroudProtocol.TRANSFERSINGLE_EVENT, log);
					BigInteger enftId = (BigInteger) eventValues
									.getNonIndexedValues().get(0).getValue();
					String eId
						= Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
					if (eId.equals(realTagId)) {
						burnMatch = true;
						block = log.getBlockNumber();
						break;
					}
				}
				if (!burnMatch) {
					return false;
				}
			}
			else {
				// regular case: look for a mint
				EthFilter mintFilter
					= new EthFilter(startBlock, endBlock, protocolAddr);
				mintFilter.addSingleTopic(EventEncoder.encode(
										EnshroudProtocol.TRANSFERSINGLE_EVENT));
				// first indexed topic: TransferSingle.operator
				mintFilter.addOptionalTopics("0x"
						+ TypeEncoder.encode(new Address(receipt.getSource())));
				// second indexed topic: TransferSingle.from
				mintFilter.addOptionalTopics("0x"
										+ TypeEncoder.encode(Address.DEFAULT));
				boolean gotMatch = false;
				try {
					ethLog = web3j.ethGetLogs(mintFilter).send();
				}
				catch (IOException ioe) {
					m_Log.error(lbl
								+ "unable to get EthLogs for mint filter", ioe);
					return false;
				}
				List<EthLog.LogResult> mintEvents = ethLog.getLogs();
				for (EthLog.LogResult eLog : mintEvents) {
					EthLog.LogObject logObj = (EthLog.LogObject) eLog;
					org.web3j.protocol.core.methods.response.Log log
						= logObj.get();
					EventValues eventValues
						= Contract.staticExtractEventParameters(
									EnshroudProtocol.TRANSFERSINGLE_EVENT, log);
					BigInteger enftId = (BigInteger) eventValues
									.getNonIndexedValues().get(0).getValue();
					String eId
						= Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
					if (eId.equals(realTagId)) {
						gotMatch = true;
						block = log.getBlockNumber();
						break;
					}
				}
				if (!gotMatch) {
					return false;
				}
			}
			if (block == null) {
				return false;
			}
		}
		// else: we already have the block number from a current event

		// Prepare the receipt for upload.  Start by zeroing out the tags field.
		receipt.setTagID("");

		// set block number
		receipt.setBlockId(block.toString());

		// add our signature
		MVOSignature receiptSig = receipt.getSignature();
		receiptSig.m_Signer = m_MVO.getMVOId();
		String sig = receipt.calcSignature(sigKey);
		if (sig == null) {
			m_Log.error(lbl + "could not sign receiptId " + rctId);
			return false;
		}
		receiptSig.m_Signature = sig;

		// encrypt signed data (including sig) with indicated AES key
		Base64.Decoder b64d = Base64.getUrlDecoder();
		byte[] keyData = b64d.decode(qEntry.m_AESkey);
		SecretKey secretKey = new SecretKeySpec(keyData, "AES");
		String sigData = receipt.buildSignedData(false, true);
		String encReceipt = EncodingUtils.encWithAES(secretKey,
													 sigData,
												m_MVO.getStateObj().getRNG());
		if (encReceipt == null) {
			m_Log.error(lbl + "receiptId " + rctId + " could not be encrypted");
			return false;
		}
		receipt.setEncData(encReceipt);
		qEntry.updateJson();

		// mark as uploadable -- queue will now auto-upload
		qEntry.m_Uploadable = true;
		m_Log.debug(lbl + "receiptId " + rctId
					+ " successfully prepped for upload");
		return true;
	}

	/**
	 * method to upload a receipt to the indicated storage
	 * @param receipt the receipt to be uploaded
	 * @param storage the storage location
	 */
	public boolean uploadReceipt(ReceiptEntry receipt, URI storage) {
		final String lbl = this.getClass().getSimpleName() + ".uploadReceipt: ";
		if (receipt == null || storage == null) {
			m_Log.error(lbl + "missing input");
			return false;
		}
		ReceiptBlock rct = receipt.m_Receipt;
		String acctId = "";
		if (rct.getReceiptType().equals(rct.M_Sender)) {
			acctId = rct.getSource();
		}
		else {
			// find account address of first (only) payee
			ArrayList<ReceiptBlock.ReceiptPayee> payees = rct.getPayees();
			if (payees.isEmpty()) {
				m_Log.error(lbl + "no payees on sender "
							+ "receipt, cannot find acctId");
				return false;
			}
			ReceiptBlock.ReceiptPayee payee = payees.get(0);
			acctId = payee.m_Address;
		}

		// branch on the URI specification
		String storageUri = storage.getPath();

		//TEMPCODE -- used for testing only, with local file DB not blockchain
		if (storageUri.startsWith("eData")) {
			/*	For now we'll store receipts in:
			 *		eData/receipts/{chainId}/{acctId}
			 *	and name the files {receiptId}.json. (Normally they'd be named
			 *	{block}-NNN.json, based on the block the transaction was mined
			 *	in.)
			 */
			// build relative path to storage location for this chain
			String rctDir = storage.getPath() + File.separator
							+ receipt.m_ChainId + File.separator
							+ Keys.toChecksumAddress(acctId);
			File receiptDir = new File(rctDir);

			// make sure this directory exists
			if (!(receiptDir.exists() && receiptDir.isDirectory())) {
				try {
					receiptDir.mkdirs();
				}
				catch (SecurityException se) {
					m_Log.error(lbl + "could not create "
								+ "receipt storage dir, " + rctDir, se);
					return false;
				}
			}

			// create the receipt file
			String rctFile
				= rctDir + File.separator + rct.getReceiptId() + ".json";
			File receiptFile = new File(rctFile);
			if (receiptFile.exists()) {
				// this is an error, must not overwrite
				m_Log.error(lbl + "receipt file " + rctFile
							+ " already exists!");
				return false;
			}

			// write contents into new file
			try {
				receiptFile.createNewFile();
				FileWriter fWriter = new FileWriter(receiptFile);
				// NB: assumption is that m_ReceiptJson was configured first
				fWriter.write(receipt.m_ReceiptJson);
				fWriter.flush();
				fWriter.close();
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "error saving file " + rctFile, ioe);
				return false;
			}
		}
		//END TEMPCODE
		else if (storageUri.startsWith("dbData")) {
			// here we use the receipts database, which can be shared among MVOs
			ReceiptStorageDb rctDb = new ReceiptStorageDb(m_MVO.getDbManager(),
														  m_Log);
			boolean store = true;
			try {
				store = rctDb.storeReceipt(rct.getReceiptId(),
										   receipt.m_ChainId,
									EncodingUtils.sha3(acctId.toLowerCase()),
										   receipt.m_ReceiptJson,
										   null);
			}
			catch (EnshDbException edbe) {
				m_Log.error("Unable to store receipt Id " + rct.getReceiptId(),
							edbe);
				store = false;
			}
			if (!store) {
				m_Log.error("Unable to store receipt Id " + rct.getReceiptId());
				return false;
			}
		}
		else {
			/* 	In future, we'll need to accommodate ipfs:// or wss:// or
			 *  https:// or whatever blockchain-based storage protocol we
			 *  utilize for a given chain.  TBD.
			 */
			m_Log.error(lbl + "unsupported receipt storage URI, " + storageUri);
			return false;
		}

		return true;
	}

	// methods required to implement EnftListener (called from EnftCacheS)
	/**
	 * register a eNFT mint event (occurring after start() has completed)
	 * @param chainId the chain on which the mint occurred
	 * @param enftId the ID of the eNFT minted
	 * @param addr the wallet address to which the eNFT was minted
	 * @param block the block number in which the mint occurred
	 */
	public void recordEnftMint(long chainId,
							   BigInteger enftId,
							   String addr,
							   BigInteger block)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".recordEnftMint: ";
		// convert eId to zero-padded base64
		String eId = Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
	/*
		m_Log.debug(lbl + "cId " + chainId + ", eId " + eId + ", to "
					+ addr + ", block " + block);
	 */

		// look for queued receipts with this eId for a m_Tag value
		ArrayList<ReceiptEntry> receipts = new ArrayList<ReceiptEntry>();
		Iterator<ReceiptEntry> queueIt = m_Queue.iterator();
		while (queueIt.hasNext()) {
			ReceiptEntry qEntry = (ReceiptEntry) queueIt.next();
			ReceiptBlock rct = qEntry.m_Receipt;
			if (qEntry.m_ChainId == chainId
				&& !qEntry.m_Uploadable
				&& rct.getTagID().equals(eId))
			{
				receipts.add(qEntry);
			}
		}
		// NB: an empty list is no surprise, since another MVO might have it

		// loop through receipts keyed to this eNFT being minted
		for (ReceiptEntry rctFound : receipts) {
			// prepare this receipt entry for upload as of indicated block
			if (!prepReceiptForUpload(rctFound, block)) {
				m_Log.error(lbl + "error prepping receiptId "
							+ rctFound.m_Receipt.getReceiptId()
							+ " for upload to chainId " + chainId);
			}
		}
	}

	/**
	 * register a eNFT burn event (occurring after start() has completed)
	 * @param chainId the chain on which the burn occurred
	 * @param enftId the ID of the eNFT burned
	 * @param addr the wallet address by which the eNFT was burned
	 * @param block the block number in which the burn occurred
	 */
	public void recordEnftBurn(long chainId,
							   BigInteger enftId,
							   String addr,
							   BigInteger block)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".recordEnftBurn: ";
		// convert eId to zero-padded base64
		String eId = Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
	/*
		m_Log.debug(lbl + "cId " + chainId + ", eId " + eId
					+ ", by " + addr + ", block " + block);
	 */

		/* Loop through receipts tagged to this eNFT being burned.  An eId
		 * burned is used as a tag only in one circumstance: where a Burn
		 * (Withdraw) is performed and there was no change eNFT issued.
		 * This case is indicated by prepending "BURN:" to the receipt tag.
		 */
		ArrayList<ReceiptEntry> receipts = new ArrayList<ReceiptEntry>();
		Iterator<ReceiptEntry> queueIt = m_Queue.iterator();
		while (queueIt.hasNext()) {
			final String searchTag = "BURN:" + eId;
			ReceiptEntry qEntry = (ReceiptEntry) queueIt.next();
			ReceiptBlock rct = qEntry.m_Receipt;
			if (qEntry.m_ChainId == chainId
				&& !qEntry.m_Uploadable
				&& rct.getTagID().equals(searchTag)
				&& rct.getSource().equals("0x" + addr)
				// NB: no Recipient receipts use burned eIDs as tags
				&& rct.M_Sender.equals(rct.getReceiptType()))
			{
				receipts.add(qEntry);
			}
		}
		if (receipts.isEmpty()) {
			return;
		}

		/* Because existing IDs can be used in multiple attempted transactions
		 * but can only be burned once, if we have multiple matches we will do
		 * only the last-submitted one, as shown by their m_BlockWhenQueued
		 * value.  Naturally this precaution doesn't help when multiple
		 * unsubmitted burn requests are distributed across several MVOs.
		 *
		 * However, the worst that can happen is that more than one MVO uploads
		 * a Burn/Withdraw receipt for the user, as a consequence of their own
		 * multiple burn attempts.  Therefore we are never creating a situation
		 * where a third party receives what appears to be evidence of a new
		 * eNFT being minted to them which they never actually got.
		 */
		BigInteger highestBlk = BigInteger.ZERO;
		ReceiptEntry rctToDo = null;
		for (ReceiptEntry rctFound : receipts) {
			if (highestBlk.compareTo(rctFound.m_BlockWhenQueued) < 0) {
				highestBlk = rctFound.m_BlockWhenQueued;
				rctToDo = rctFound;
			}
		}

		// loop through again and process the one selected to do, purge the rest
		for (ReceiptEntry rctFound : receipts) {
			if (rctFound == rctToDo) {
				// prepare this receipt entry for upload as of indicated block
				if (!prepReceiptForUpload(rctFound, block)) {
					m_Log.error(lbl + "error prepping receiptId "
								+ rctFound.m_Receipt.getReceiptId()
								+ " for upload to chainId " + chainId);
				}
			}
			else {
				// dequeue this, as no earlier entry can be mined now
				dequeueReceipt(rctFound);
				m_Log.warning(lbl + "dequeued burned Sender receipt for Id "
							+ eId + " on chain " + rctFound.m_ChainId);
			}
		}
	}

	/**
	 * process a notification of a new block
	 * @param chainId the chain on which the block was mined
	 * @param block the new block number
	 */
	public void recordBlock(long chainId, BigInteger block) {
		// we have no special handling for this event in an MVO
	/*
		m_Log.debug("recordBlock: new block on chId " + chainId + " = "
					+ block);
	 */
	}

	/**
	 * process a notification of an ungreylist event
	 * @param chainId the chain on which the block was mined
	 * @param enftId the ID of the eNFT which has been ungreylisted
	 */
	public void recordUnGreylist(long chainId, BigInteger enftId) {
		/* we have no special handling for this event in an MVO, beyond the
		 * normal processing in the EnftCache (which calls us)
		 */
	/*
		m_Log.debug("recordUnGreylist: on chainId " + chainId
					+ ", eId "
					+ Numeric.toHexStringNoPrefixZeroPadded(enftId, 64)
					+ " was un-greylisted");
	 */
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Queue != null) {
				m_Queue.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
