/*
 * last modified---
 * 	02-12-24 suppress SslContextFactory.Server setup, as we won't be using it
 * 	06-05-23 handle EIP-712 signed requests
 * 	04-27-23 handle AES-decrypting requests per ECIES algorithm
 * 	04-21-23 add doOptions(), handle decryption prior to determining payload
 * 	04-14-23 handle encrypting replies with the client's specified replyKey
 * 	03-20-23 handle decrypting the payload if indicated by a encrchain param;
 * 			 encrypt normal replies with replyKey if client supplied one
 * 	01-04-23 support CORS by adding CrossOriginFilter
 * 	09-13-22 properly init broker entry in mapping
 * 	07-12-22 improve error message labeling
 * 	06-28-22 implement sendNormalRep() methods
 * 	06-21-22 enable async processing
 * 	04-07-22 add invocation of state handlers
 * 	04-05-22 handle wallet query requests as well as receipt queries
 * 	03-15-22 share SSL factory with ClientHandler
 * 	03-11-22 new
 *
 * purpose---
 * 	handle MVO Layer2 requests related to receipts and wallets from dApp clients
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ClientReceiptBlock;
import cc.enshroud.jetty.ClientWalletBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.MVOReceiptBlock;
import cc.enshroud.jetty.MVOWalletBlock;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.servlet.ServletHandler;
import org.eclipse.jetty.server.Connector;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.NetworkTrafficServerConnector;
import org.eclipse.jetty.util.ajax.JSON;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;

import java.util.Base64;
import java.util.Properties;
import java.util.Map;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.BufferedReader;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.SecureRandom;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.ServletException;
import javax.servlet.ServletContext;
import javax.servlet.AsyncContext;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * This class is instantiated once per MVO and provides a listener and request
 * handler for client requests to manipulate receipt records in the storage,
 * and to perform wallet downloads.  These arrive on the Jetty servlet handler
 * as POSTs.
 */
public final class ReceiptHandler extends HttpServlet {
	// BEGIN data members
	/**
	 * the owning MVO
	 */
	private MVO			m_MVOServer;

	/**
	 * the logging object (shared with owning MVO)
	 */
	private Log			m_Log;

	/**
	 * the port number at which we are to listen for incoming requests
	 */
	private final int	m_ListenPort;

	/**
	 * the secure connection factory used by connectors, built using our
	 * configured keystore
	 */
	private SslContextFactory.Server	m_SslFactory;

	/**
	 * the Jetty server used to listen for incoming messages
	 */
	private Server		m_JettyServer;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which owns us
	 * @param log the error/debug logger (normally mvo.log())
	 * @param listenPort the port at which we're to listen for requests
	 */
	public ReceiptHandler(MVO mvo, Log log, int listenPort) {
		super();
		m_MVOServer = mvo;
		m_Log = log;
		m_ListenPort = listenPort;
		m_SslFactory = new SslContextFactory.Server();
		// customize thread pool
		QueuedThreadPool qtp = new QueuedThreadPool(255, 8, 600000);
		m_JettyServer = new Server(qtp);
	}

	/**
	 * initialize the Jetty instance
	 * @return true on success
	 */
	public boolean initialize() {
		boolean secureOk = true;
	/* NB: wss TLS negotiation is broken in Jetty!  Also, we don't want to
	 * 	   require site certificates for MVOs, since they have no domains.
	 * 	   Therefore, we comment all of this out.
		// set up the SSL connector factory based on our keystore and properties
		Properties props = m_MVOServer.getProperties();
		String keystorePath = props.getProperty("ClientKeyStore");
		m_SslFactory.setKeyStorePath(keystorePath);
		String keystorePass = props.getProperty("KeyStorePass");
		m_SslFactory.setKeyStorePassword(keystorePass);
		String keystoreType = props.getProperty("KeyStoreType");
		m_SslFactory.setKeyStoreType(keystoreType);
		String truststorePath = props.getProperty("TrustStore");
		m_SslFactory.setTrustStorePath(truststorePath);
		String truststoreType = props.getProperty("TrustStoreType");
		m_SslFactory.setTrustStoreType(truststoreType);
		String trustStorePass = props.getProperty("TrustStorePass");
		if (trustStorePass != null) {
			m_SslFactory.setTrustStorePassword(trustStorePass);
		}
		String keyAlias = props.getProperty("ClientKeyAlias", "mvoclient");
		m_SslFactory.setCertAlias(keyAlias);
		m_SslFactory.setNeedClientAuth(false);
		m_SslFactory.setWantClientAuth(false);
		m_SslFactory.addExcludeProtocols("SSLv2Hello", "SSLv3");
		try {
			m_SslFactory.start();
		}
		catch (Exception eee) {
			m_Log.error("ReceiptHandler.initialize: unable to start SSL "
						+ "factory", eee);
			secureOk = false;
		}
	 */
		secureOk = false;

		if (secureOk) {
			final NetworkTrafficServerConnector httpsConnector
				= new NetworkTrafficServerConnector(m_JettyServer,
													m_SslFactory);
			httpsConnector.setHost(null);
			httpsConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpsConnector);
		}
		else {
			NetworkTrafficServerConnector httpConnector
				= new NetworkTrafficServerConnector(m_JettyServer);
			httpConnector.setHost(null);
			httpConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpConnector);
		}

		// define servlet handler
		ServletHandler handler = new ServletHandler();
		m_JettyServer.setHandler(handler);
		handler.addServletWithMapping(new ServletHolder(this), "/download");

		// so we can support CORS requests, add filter with defaults
		handler.addFilterWithMapping(
						"org.eclipse.jetty.servlets.CrossOriginFilter", "/*",
							org.eclipse.jetty.servlet.FilterMapping.DEFAULT);

		// attempt to start the server and handler
		try {
			m_JettyServer.start();
			handler.initialize();
			m_Log.debug("ReceiptHandler.initialize: ReceiptHandler Jetty "
						+ "listener (secure=" + secureOk + ") on port "
						+ m_ListenPort);
		}
		catch (Exception e) {
			m_Log.error("ReceiptHandler.initialize: could not start Jetty "
						+ "listener", e);
			return false;
		}
		return true;
	}

	/**
	 * shutdown Jetty listener
	 */
	public void shutdown() {
		if (m_JettyServer.isRunning()) {
			try {
				m_JettyServer.stop();
				m_Log.debug("ReceiptHandler.shutdown: Jetty listener stopped");
			}
			catch (Exception e) {
				m_Log.error("ReceiptHandler.shutdown: error stopping Jetty "
							+ "listener", e);
			}
		}

		// shut down the SSL factory
		if (m_SslFactory != null && m_SslFactory.isRunning()) {
			try {
				m_SslFactory.stop();
			}
			catch (Exception ee) {
				m_Log.error("ReceiptHandler.shutdown: exception stopping SSL "
							+ "factory", ee);
			}
		}
	}

	/**
	 * obtain the Jetty server
	 * @return the Jetty server object we use to handle requests
	 */
	public Server getJettyServer() { return m_JettyServer; }

	// methods to extend the HttpServlet class
	/**
	 * handler method for GETs from the dApp clients (not used)
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doGet(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ReceiptHandler.doGet: null dApp receipt "
								+ "request");
		}
		if (response == null) {
			throw new IOException("ReceiptHandler.doGet: null dApp receipt "
								+ "response object");
		}

		// GET is not used
		//response.addHeader("Access-Control-Allow-Origin", "*");
		response.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED,
							"GET not supported for this function");
	}

	/**
	 * handler for OPTIONS preflights from dApp clients
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doOptions(HttpServletRequest request,
						  HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ReceiptHandler.doOptions: null dApp receipt "
								+ "request");
		}
		if (response == null) {
			throw new IOException("ReceiptHandler.doOptions: null dApp receipt "
								+ "response object");
		}

		// echo back same origin we were sent
		response.addHeader("Access-Control-Allow-Origin",
							request.getHeader("ORIGIN"));
		response.addHeader("Access-Control-Allow-Headers", "*");
		response.addHeader("Access-Control-Allow-Methods",
							"POST, HEAD, TRACE, OPTIONS");
		response.setStatus(HttpServletResponse.SC_NO_CONTENT);
		response.flushBuffer();
	}

	/**
	 * handler method for POSTs from the dApp clients
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doPost(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("ReceiptHandler.doPost: null dApp receipt "
								+ "request");
		}
		if (response == null) {
			throw new IOException("ReceiptHandler.doPost: null dApp receipt "
								+ "response object");
		}
		final String lbl = this.getClass().getSimpleName() + ".doPost: ";

		response.setContentType("text/html");
		//response.addHeader("Access-Control-Allow-Origin", "*");
		response.setStatus(HttpServletResponse.SC_OK);

		/* See whether the payload has been encrypted.  This is indicated by
		 * sending the encrchain as a separate POST parameter.  In this case we
		 * must decrypt with the private ECIES key for the indicated chain
		 * (matching the pubkey shown in the MVO config).
		 */
		String payload = "";
		String parsePayload = payload;
		String requested = "";
		String chainParm = request.getHeader("encrchain");
		Long chId = null;
		String origDappReq = "";
		boolean encrypted = chainParm != null;
		if (encrypted) {
			try {
				chId = Long.parseLong(chainParm.trim());
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "bad encryption encrchain passed, "
							+ chainParm);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Invalid encrchain parameter value, " + chainParm);
				return;
			}

			// get our key for this chain
			BlockchainConfig bcConf
				= m_MVOServer.getConfig().getChainConfig(chId);
			if (bcConf == null) {
				m_Log.error(lbl + "no BlockchainConfig found for encrchain "
							+ chId);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Unsupported encrchain, " + chId);
				return;
			}
			PrivateKey ecPriv = bcConf.getECPrivKey();
			if (ecPriv == null) {
				m_Log.error(lbl + "no EC privkey found for encrchain " + chId);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Decrypt key not found for encrchain " + chId);
				return;
			}

			// obtain the data, which should be a single line of Base64Url
			BufferedReader reqStream = null;
			try {
				reqStream = request.getReader();

				// allow for possible chunking
				String line = "";
				do {
					line = reqStream.readLine();
					if (line != null && !line.isEmpty()) {
						payload += line;
					}
				} while (line != null);
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "error reading encrypted request data", ioe);
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Error reading encrypted request data");
				return;
			}

			// decrypt payload using this key
			String decPayload = EncodingUtils.decWithECIES(ecPriv, payload);
			if (decPayload == null) {
				m_Log.error(lbl + "error decrypting payload for chain " + chId);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Error decrypting ECIES payload");
				return;
			}

			// JSON parse this data
			Object parsedDec = null;
			try {
				parsedDec = JSON.parse(decPayload);
			}
			catch (IllegalStateException ise) {
				m_Log.error(lbl + "decrypted JSON did not parse: \""
							+ decPayload + "\"", ise);
			}
			if (!(parsedDec instanceof Map)) {
				m_Log.error(lbl + "decrypted JSON was not a Map");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Parse error on decrypted JSON");
				throw new ServletException("Parse error on decrypted JSON");
			}
			Map decMap = (Map) parsedDec;

			/* Record entire decrypted payload as original, including EIP-712
			 * data plus the client's signature.  This must be passed to the
			 * AUD key server so it can validate the original client request
			 * as required.
			 */
			origDappReq = decPayload;

			// find message.payloadJson, which should be a Map
			Object messageObj = decMap.get("message");
			if (!(messageObj instanceof Map)) {
				m_Log.error(lbl + "could not find message in signed data");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Could not find message in signed data");
				throw new ServletException("Missing message signed data");
			}
			Map messageMap = (Map) messageObj;
			Object requestJson = messageMap.get("requestJson");
			if (!(requestJson instanceof Map)) {
				m_Log.error(lbl + "could not find message.requestJson");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Could not find message.requestJson data");
				throw new ServletException("Missing message.requestJson data");
			}
			Map jsonMap = (Map) requestJson;

			// look for receiptRequest or walletDownload map key
			Map decPayloadMap = null;
			Object rctReq = jsonMap.get("receiptRequest");
			if (rctReq instanceof Map) {
				decPayloadMap = (Map) rctReq;
				requested = "receipts";
			}
			else {
				Object wallReq = jsonMap.get("walletDownload");
				if (wallReq instanceof Map) {
					decPayloadMap = (Map) wallReq;
					requested = "eNFTs";
				}
			}
			if (decPayloadMap == null) {
				// nothing else is valid
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"No recognized payload type");
				m_Log.error(lbl + "no recognized payload type");
				throw new ServletException("No recognized payload type");
			}

			// convert back to a string -- this will reorder the map elements
			parsePayload = JSON.toString(decPayloadMap);
		}
		else {
			// unencrypted, so look for request parameters (note: unsigned)
			payload = request.getParameter("receiptRequest");
			if (payload != null && !payload.isEmpty()) {
				requested = "receipts";
			}
			else {
				payload = request.getParameter("walletDownload");
				if (payload == null || payload.isEmpty()) {
					// nothing else is valid
					sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
								"No recognized payload type");
					m_Log.error(lbl + "no recognized payload type");
					throw new ServletException("No recognized payload type");
				}
				requested = "eNFTs";
			}
			parsePayload = payload;
			origDappReq = payload;
		}

		/* Two kinds of request payloads can arrive here:
		 * receiptRequest - a receipt manipulation request, JSON forming a
		 * 					ClientReceiptBlock.
		 * 					We will return a ReceiptBlock after processing.
		 * walletDownload -	a wallet download request, JSON forming a
		 * 					ClientWalletBlock.
		 * We will return a MVOWalletBlock after processing.
		 */
		ClientReceiptBlock rBlock = null;
		ClientWalletBlock wBlock = null;
		MVOBrokerEntry mvoBroEnt = null;

		// handle what was requested
		if (requested.equals("receipts")) {
			rBlock = new ClientReceiptBlock(request, m_Log);
			rBlock.setDecryptedPayload(origDappReq);

			// parse the dApp client's receipt payload, which should be JSON
			if (!rBlock.buildFromString(parsePayload)) {
				m_Log.error(lbl + "could not parse receipt payload = \""
							+ parsePayload + "\"");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Payload for receiptSpec did not parse as JSON");
				return;
			}

			// verify the chainId inside matches the parameter, if we got one
			if (chId != null && rBlock.getChainId() != chId.longValue()) {
				// do not attempt to continue in this case
				m_Log.error(lbl + "dApp chainId " + rBlock.getChainId()
							+ " received, but " + chId
							+ " expected from parameter value");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"chainId " + rBlock.getChainId()
							+ " inconsistent with parameter value");
				return;
			}

			// create a broker map entry for this request
			mvoBroEnt = new MVOBrokerEntry(rBlock, m_MVOServer);
			ReceiptState rcptSt = new ReceiptState(mvoBroEnt);
			mvoBroEnt.setStateTracker(rcptSt);

			// confirm signature on dApp receipt request
			if (encrypted && !verifyClientSig(origDappReq, rBlock)) {
				m_Log.error(lbl + "signature did not verify, opcode "
							+ rBlock.getOpcode() + " json = \""
							+ origDappReq + "\"");
				sendErrRep(response, HttpServletResponse.SC_UNAUTHORIZED,
							"Signature verification failure on "
							+ rBlock.getOpcode() + " receipt request");
				return;
			}

			// to permit async request processing, create an async context
			AsyncContext async = null;
			try {
				async = request.startAsync();
			}
			catch (IllegalStateException ise) {
				m_Log.error(lbl + "could not start async processing", ise);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Could not switch to async request processing");
				return;
			}
			mvoBroEnt.setAsyncContext(async);
			// set a 1 min timeout
			async.setTimeout(60000L);
			// ClientHandler is the event listener
			async.addListener(m_MVOServer.getClientHandler());

			// since request appears legit, create transId and add to broker map
			MVOState stObj = m_MVOServer.getStateObj();
			BigInteger val = new BigInteger(256, stObj.getRNG());
			String transId = val.toString();
			mvoBroEnt.setTransId(transId);
			Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();
			brMap.put(transId, mvoBroEnt);

			// record successful parse and sig verify
			rcptSt.recordSuccess(ReceiptState.M_ReqReceived);
			// go on to obtain necessary receipts
			rcptSt.advanceState(ReceiptState.M_ReqReceived,
								ReceiptState.M_GotReceipts,
								ReqSvcState.M_NoFailure, "");
		}
		else if (requested.equals("eNFTs")) {
			wBlock = new ClientWalletBlock(request, m_Log);
			wBlock.setDecryptedPayload(origDappReq);

			// parse the dApp's wallet request payload, which should be JSON
			if (!wBlock.buildFromString(parsePayload)) {
				m_Log.error(lbl + "could not parse wallet "
							+ "request payload = \"" + parsePayload + "\"");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"Payload for walletDownload did not parse as JSON");
				return;
			}

			// verify the chainId inside matches the parameter, if we got one
			if (chId != null && wBlock.getChainId() != chId.longValue()) {
				// do not attempt to continue in this case
				m_Log.error(lbl + "dApp chainId " + wBlock.getChainId()
							+ " received, but " + chId
							+ " expected from parameter value");
				sendErrRep(response, HttpServletResponse.SC_BAD_REQUEST,
							"chainId " + wBlock.getChainId()
							+ " inconsistent with parameter value");
				return;
			}

			// create a broker map entry for this request
			mvoBroEnt = new MVOBrokerEntry(wBlock, m_MVOServer);
			WalletState wallSt = new WalletState(mvoBroEnt);
			mvoBroEnt.setStateTracker(wallSt);

			// confirm signature on dApp wallet request
			if (encrypted && !verifyClientSig(origDappReq, wBlock)) {
				m_Log.error(lbl + "signature did not verify, json = \""
							+ origDappReq + "\"");
				sendErrRep(response, HttpServletResponse.SC_UNAUTHORIZED,
							"Signature verification failure on "
							+ " wallet download request");
				return;
			}

			// to permit async request processing, create an async context
			AsyncContext async = null;
			try {
				async = request.startAsync();
			}
			catch (IllegalStateException ise) {
				m_Log.error(lbl + "could not start async processing", ise);
				sendErrRep(response,
						   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
						   "Could not switch to async request processing");
				return;
			}
			mvoBroEnt.setAsyncContext(async);
			// set a 1 min timeout
			async.setTimeout(60000L);
			// ClientHandler is the event listener
			async.addListener(m_MVOServer.getClientHandler());

			// since request appears legit, create transId and add to broker map
			MVOState stObj = m_MVOServer.getStateObj();
			BigInteger val = new BigInteger(256, stObj.getRNG());
			String transId = val.toString();
			mvoBroEnt.setTransId(transId);
			Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();
			brMap.put(transId, mvoBroEnt);

			// record successful parse and sig verify
			wallSt.recordSuccess(WalletState.M_ReqReceived);
			// go on to fetch eNFTs for wallet (from event log)
			wallSt.advanceState(WalletState.M_ReqReceived,
								WalletState.M_GoteNFTs,
								ReqSvcState.M_NoFailure, "");
		}
		else {
			// should be unreachable
			throw new ServletException("No recognized payload type");
		}
	}

	/**
	 * method to supply servlet information
	 * @return the name of the servlet (arbitrary)
	 */
	@Override
	public String getServletInfo() {
		return "MVO-Jetty:ReceiptHandler servlet";
	}

	// utility methods
	/**
	 * obtain the SSL factory
	 * @return the factory
	 */
	public SslContextFactory.Server getSslFactory() { return m_SslFactory; }

	/**
	 * send a correctly formatted HTTP error response to a POST or GET
	 * @param resp the response for the reply
	 * @param sc the status code (SC_* constant in HttpServletResponse)
	 * @param errTxt the error message to return to the client
	 */
	public void sendErrRep(HttpServletResponse resp, int sc, String errTxt) {
		if (resp == null || errTxt == null) {
			m_Log.error("ReceiptHandler.sendErrRep: missing input");
			return;
		}
		try {
			resp.setContentType("text/html");
			//resp.addHeader("Access-Control-Allow-Origin", "*");
			resp.sendError(sc, errTxt);
		}
		catch (IOException ioe) {
			m_Log.error("ReceiptHandler.sendErrRep: exception sending error "
						+ "reply " + errTxt, ioe);
		}
		catch (IllegalStateException ise) {
			m_Log.error("ReceiptHandler.sendErrRep: exception sending error "
						+ "reply " + errTxt, ise);
		}
	}

	/**
	 * validate client signature on an EIP712-signed request
	 * @param requestJson the entire JSON string received from the user
	 * @param cliRequest the parsed client block containing the signature
	 * @return true if client's signature was valid
	 */
	private boolean verifyClientSig(String requestJson,
									ClientRequest cliRequest) 
	{
		if (requestJson == null || requestJson.isEmpty() || cliRequest == null)
		{
			m_Log.error("ReceiptHandler.verifyClientSig: missing input");
			return false;
		}

		/* to extract the signed portion, take everything up to:
		 * ',\"signature\":'
		 */
		int sigIdx = requestJson.lastIndexOf(",\"signature\":");
		if (sigIdx == -1) {
			m_Log.error("ReceiptHandler.verifyClientSig: missing signature");
			return false;
		}
		String eip712Data = requestJson.substring(0, sigIdx) + "}";

		// obtain the signature from full input data
		String rawSig = requestJson.substring(sigIdx);
		int sigStart = rawSig.lastIndexOf(":\"");
		// strip off leading :" and trailing "}
		String signature = rawSig.substring(sigStart+2, rawSig.length()-2);
		String signingAddress = cliRequest.getSender();

		// determine which address actually signed the data
		String actualSigAddr
			= EncodingUtils.getEIP712SignerAddress(eip712Data, signature);
		return actualSigAddr.equalsIgnoreCase(signingAddress);
	}

	/**
	 * send a correctly formatted JSON receipt reply to a client POST
	 * @param resp the response for the reply
	 * @param recBlock the complete text (signed) to send
	 * @param repKey client's AES key to encrypt the text back to client (opt)
	 */
	public boolean sendNormalRep(HttpServletResponse resp,
								 MVOReceiptBlock recBlock,
								 String repKey)
	{
		final String lbl = this.getClass().getSimpleName() + ".sendNormalRep: ";
		if (resp == null || recBlock == null || repKey == null) {
			m_Log.error(lbl + "missing input");
			return false;
		}
		resp.setContentType("application/json");
		//resp.addHeader("Access-Control-Allow-Origin", "*");
		resp.setStatus(HttpServletResponse.SC_OK);

		// obtain the writer and append the data to it
		try {
			PrintWriter writer = resp.getWriter();
			StringBuilder reply = new StringBuilder(10240);
			recBlock.addJSON(reply);

			// see if we need to encrypt our reply to the client's requested key
			if (repKey.isEmpty()) {
				writer.append(reply);
			}
			else {
				// convert back to bytes from Base64
				Base64.Decoder b64d = Base64.getUrlDecoder();
				byte[] keyBytes = b64d.decode(repKey);

				// verify we have an AES-256 size key
				if (keyBytes.length != 32) {
					m_Log.error(lbl + "bad length on replyKey, "
								+ keyBytes.length);
					return false;
				}
				SecretKey aesKey = new SecretKeySpec(keyBytes, "AES");
				SecureRandom rng = m_MVOServer.getStateObj().getRNG();
				String encReply = EncodingUtils.encWithAES(aesKey,
														   reply.toString(),
														   rng);
				if (encReply == null) {
					m_Log.error(lbl + "error encrypting client reply to key "
								+ repKey);
					return false;
				}
				else {
					writer.append(encReply);
				}
			}
			resp.flushBuffer();
		}
		catch (NumberFormatException nfe) {
			m_Log.error(lbl + "replyKey was not hex: " + repKey, nfe);
			return false;
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "error writing response", ioe);
			return false;
		}
		return true;
	}

	/**
	 * send a correctly formatted JSON wallet reply to a client POST
	 * @param resp the response for the reply
	 * @param walletBlock the complete text (signed) to send
	 * @param repKey client's AES key to encrypt the text back to client (opt)
	 */
	public boolean sendNormalRep(HttpServletResponse resp,
								 MVOWalletBlock walletBlock,
								 String repKey)
	{
		final String lbl = this.getClass().getSimpleName() + ".sendNormalRep: ";
		if (resp == null || walletBlock == null || repKey == null) {
			m_Log.error(lbl + "missing input");
			return false;
		}
		resp.setContentType("application/json");
		//resp.addHeader("Access-Control-Allow-Origin", "*");
		resp.setStatus(HttpServletResponse.SC_OK);

		// obtain the writer and append the data to it
		try {
			PrintWriter writer = resp.getWriter();
			StringBuilder reply = new StringBuilder(10240);
			walletBlock.addJSON(reply);

			// see if we need to encrypt our reply to the client's requested key
			if (repKey.isEmpty()) {
				writer.append(reply);
			}
			else {
				// convert back to bytes from Base64
				Base64.Decoder b64d = Base64.getUrlDecoder();
				byte[] keyBytes = b64d.decode(repKey);

				// verify we have an AES-256 size key
				if (keyBytes.length != 32) {
					m_Log.error(lbl + "bad length on replyKey, "
								+ keyBytes.length);
					return false;
				}
				SecureRandom rng = m_MVOServer.getStateObj().getRNG();
				SecretKey aesKey = new SecretKeySpec(keyBytes, "AES");
				String encReply = EncodingUtils.encWithAES(aesKey,
														   reply.toString(),
														   rng);
				if (encReply == null) {
					m_Log.error(lbl + "error encrypting client reply to key "
								+ repKey);
					return false;
				}
				else {
					writer.append(encReply);
				}
			}
			resp.flushBuffer();
		}
		catch (IllegalArgumentException iae) {
			m_Log.error(lbl + "replyKey was not Base64", iae);
			return false;
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "error writing response", ioe);
			return false;
		}
		return true;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			m_SslFactory = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
