/*
 * last modified---
 * 	01-12-26 version 0.2.7; recent minor fixes
 * 	12-05-25 also call MVOStakingListener.reinit() from HKP
 * 	12-04-25 add EnftCache.reinitWhileRunning() call to run() HKP
 * 	10-06-25 do logging for requestEnftCacheRestart() in SCC's logfile
 * 	08-07-25 version 0.2.6; switch to MariaDB driver with wsrep_sync_wait set
 * 	07-16-25 version 0.2.5; implement 2nd VPN failover, live release candidate
 * 	05-06-25 version 0.2.4 (final test versions of MVOStaking/EnshroudProtocol)
 * 	04-04-25 prevent duplicate WSS connections created by
 * 			 requestEnftCacheRestart() invocations
 * 	03-28-25 version 0.2.3; support fallback to dApp-supplied eNFT AES keys
 * 	11-15-24 bump timeouts to 60 secs; prevent dup EnftCacheResetter calls
 * 	10-30-24 implement EnftCacheResetter
 * 	09-24-24 specify 30 sec timeouts for http connect/write, 60 secs for read;
 * 			 ensure ShutdownHook is created prior to initSmartContracts()
 * 	05-21-24 add override to RxJavaPlugins.setErrorHandler() to debug
 * 			 UndeliverableExceptionS that occur during .dispose() calls
 * 	04-15-23 add m_HKPTimer
 * 	03-12-24 provide BlockchainConnectListener.setupMVOStakingListener()
 * 	03-07-24 add usedAuds param to getRandomAuditor()
 * 	02-14-24 remove unnecessary usage of UserURL and DownloadURL config params
 * 	09-25-23 changes to order of initialization of objects
 * 	07-07-23 implement BlockchainConnectListener
 * 	06-20-23 set genesis deployment block in initSmartContractConfigs()
 * 	06-09-23 rework to support Web3j URLs in each BlockchainConfig and distinct
 * 			 RemoteBlockchainAPIs for every chain
 * 	03-15-23 add m_DbManager
 * 	12-01-22 initialize EC keypair for each BlockchainConfig
 * 	09-28-22 fork Jetty logging into JettyLog
 * 	09-16-22 remove old AuditorHandler now that real AUD is available;
 * 			 also remove unnecessary separate Auditor listen port
 * 	08-10-22 add AuditorQueue support
 * 	08-04-22 support Auditor specs of ID:port
 * 	06-30-22 add ReceiptQueue
 * 	06-23-22 add ReceiptStore for m_ReceiptStore support
 * 	06-13-22 add getURIforMVO()
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	05-05-22 add getWeb3(), getAuditorHandler()
 * 	04-22-22 add Standalone mode, BlockchainAPI element
 * 	04-20-22 add m_SmartContracts and init
 * 	04-19-22 add m_MVOClient
 * 	04-15-22 add initialization of own keypair and other MVO pubkeys
 * 	04-05-22 add m_State plus methods
 * 	03-23-22 add m_Config plus methods
 * 	03-10-22 new
 *
 * purpose---
 * 	main class for MVO servers
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.*;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.DbConnectionManager;

import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.protocol.Web3j;
import org.web3j.protocol.http.HttpService;
import org.web3j.protocol.websocket.WebSocketService;
import org.web3j.protocol.exceptions.ClientConnectionException;
import org.web3j.protocol.core.filters.FilterException;
import okhttp3.OkHttpClient;
import io.reactivex.plugins.RxJavaPlugins;
import io.reactivex.exceptions.UndeliverableException;
import org.java_websocket.exceptions.WebsocketNotConnectedException;

import java.util.TimerTask;
import java.util.Timer;
import java.util.Properties;
import java.util.Hashtable;
import java.util.Map;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.FilenameFilter;
import java.text.NumberFormat;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.net.URI;
import java.net.URL;
import java.net.URISyntaxException;
import java.net.ConnectException;
import java.net.SocketException;
import java.sql.Connection;
import java.sql.SQLException;
import java.math.BigInteger;


/**
 * This class serves as the main for the MVO Jetty server.  It manages all the
 * handlers, endpoints, listeners, and contexts for the MVO, both as a server
 * and as a client of other MVOs, Auditors, and receipt storage archives.
 */
public final class MVO extends TimerTask
	implements FilenameFilter, BlockchainConnectListener, EnftCacheResetter
{
	// BEGIN data members
	/**
	 * constant for version/date string
	 */
	private final static String	M_Version = "0.2.7, 12 January, 2026";

	/**
	 * our local configuration properties, as read from the file
	 * $jetty_home/MVO-N.properties (where N is the MVO number of the instance)
	 */
	private Properties			m_PropConfig;

	/**
	 * our instance number (which MVO we are)
	 */
	public static int			m_MVONumber = 1;

	/**
	 * our MVO ID
	 */
	private String				m_MVOId;

	/**
	 * our MVO configuration, as read from the smart contract
	 */
	private MVOConfig			m_Config;

	/**
	 * processor for handling dApp client requests for Enshrouded operations
	 */
	private ClientHandler		m_ClientHandler;

	/**
	 * processor for handling dApp client requests for receipt operations
	 */
	private ReceiptHandler		m_ReceiptHandler;

	/**
	 * processor for handling peer MVO requests for committee actions
	 */
	private MVOHandler			m_MVOHandler;

	/**
	 * processor for outgoing connections to peer MVOs and Auditors
	 */
	private MVOClient			m_MVOClient;

	/**
	 * actual blockchain API objects which talk to real JSON-RPC nodes,
	 * indexed by chainId values
	 */
	private Hashtable<Long, BlockchainAPI>	m_BlockchainAPIs;

	/**
	 * queue manager for sending broadcasts to all Auditors
	 */
	private AuditorQueue		m_AuditorManager;

	/**
	 * state-tracking object
	 */
	private MVOState			m_State;

	/**
	 * list of blockchain smart contract configurations this MVO supports,
	 * indexed by chain Id
	 */
	private Hashtable<Long, SmartContractConfig>	m_SmartContracts;

	/**
	 * list of MVOStaking event monitors for this MVO, indexed by chain Id
	 */
	private Hashtable<Long, MVOStakingListener>		m_StakingListeners;

	/**
	 * the object which controls queueing uploads to receipt storage
	 */
	private ReceiptQueue		m_ReceiptUploader;

	/**
	 * database manager for accessing receipts storage
	 */
	private DbConnectionManager	m_DbManager;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * inner class to provide a shutdown hook for the server
	 */
	private class ShutdownHook extends Thread {
		/**
		 * the server we're to shut down
		 */
		private MVO			m_Server;

		/**
		 * constructor
		 * @param server the server we're a hook for
		 */
		public ShutdownHook(MVO server) {
			m_Server = server;
		}

		/**
		 * invokes shutdown methods and logs event
		 */
		@Override
		public void run() {
			if (m_HKPTimer != null) {
				m_HKPTimer.cancel();
			}

			// shut down the client request handlers
			// NB: ReceiptHandler depends on ClientHandler
			if (m_Server.m_ReceiptHandler != null) {
				m_Server.m_ReceiptHandler.shutdown();
			}
			if (m_Server.m_ClientHandler != null) {
				m_Server.m_ClientHandler.shutdown();
			}
			if (m_Server.m_MVOClient != null) {
				m_Server.m_MVOClient.shutdown();
			}
			if (m_Server.m_MVOHandler != null) {
				m_Server.m_MVOHandler.shutdown();
			}
			if (m_Server.m_AuditorManager != null) {
				m_Server.log().debug("ShutdownHook: stopping Auditor queues");
				if (!m_Server.m_AuditorManager.stopQueues()) {
					m_Server.log().error("MVO " + m_Server.m_MVONumber
										+ " got an error shutting down one or "
										+ "more Auditor queues");
				}
				// wait for completions of run threads
				try {
					Thread.sleep(m_Server.m_AuditorManager.M_CheckInterval);
				}
				catch (InterruptedException ie) { /* ignore */ }
			}
			if (m_Server.m_Config != null) {
				m_Server.m_Config.setStatus(false);
			}

			// clear outstanding requests
			if (m_Server.m_State != null) {
				m_Server.m_State.getBrokerMap().clear();
			}

			// terminate the receipt uploader / queue
			if (m_Server.m_ReceiptUploader != null) {
				m_Server.log().debug("ShutdownHook: stopping receipt uploader");
				m_Server.m_ReceiptUploader.stop();
				// wait for completion of the run thread
				try {
					Thread.sleep(m_Server.m_ReceiptUploader.M_CheckInterval);
				}
				catch (InterruptedException ie) { /* ignore */ }
			}

			// shut down all BlockchainAPI interfaces
			for (Long chain : m_BlockchainAPIs.keySet()) {
				// flushes the properties file to disk iff LocalBlockchainAPI
				BlockchainAPI chainAPI = m_BlockchainAPIs.get(chain);
				if (!chainAPI.shutdown()) {
					m_Server.log().error("MVO " + m_Server.m_MVONumber
										+ " got error shutting down chain API "
										+ "for chain " + chain);
				}
			}

			// stop the listeners for MVOStaking events
			for (Long chId : m_StakingListeners.keySet()) {
				MVOStakingListener mvoListener = m_StakingListeners.get(chId);
				// disposes of any active subscriptions
				mvoListener.shutdown();
			}

			// stop all the Web3j objects
			for (SmartContractConfig scc : m_SmartContracts.values()) {
				// shut down the cache
				EnftCache eCache = scc.getCache();
				if (eCache != null) {
					eCache.shutdown();
				}
				Web3j abiInterface = scc.getABI();
				if (abiInterface != null) {
					abiInterface.shutdown();
				}
				scc.log().deInit();
			}

			m_Server.log().debug("MVO " + m_Server.m_MVONumber
								+ " server shutdown");
			m_Server.log().deInit();
		}
	}

	/**
	 * timer to trigger running hourly housekeeping
	 */
	private Timer				m_HKPTimer;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param props the properties we read from our config file
	 */
	public MVO(Properties props) {
		super();
		m_PropConfig = props;

		// init logging
		String logDir = m_PropConfig.getProperty("LogDir");
		if (logDir == null) {
			logDir = "/var/log/enshroud";
		}
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		// NB: assumption is that static m_MVONumber was set by main()
		String logFile = m_PropConfig.getProperty("LogFile");
		if (logFile != null && !logFile.isEmpty()) {
			m_Log = new Log(logDir + File.separator + logFile);
		}
		else {
			m_Log = new Log(logDir + File.separator + "MVO-"
							+ nf.format(m_MVONumber) + ".log");
		}
		m_Log.Init();

		// assign System.out and System.err to our logging object
		PrintStream logOut = m_Log.getLogStream();
		System.setOut(logOut);
		System.setErr(logOut);

		// set MVO Id, with number in minimum 3 digits
		m_MVOId = m_PropConfig.getProperty("MVOId",
											"MVO-" + nf.format(m_MVONumber));

		// also set in system properties
		System.setProperty("PEER_ID", m_MVOId);

		// configure Jetty logging to use our Logger class
		System.setProperty("org.eclipse.jetty.util.log.class",
							"cc.enshroud.jetty.log.JettyLog");

		// create config
		m_Config = new MVOConfig(m_MVOId, m_Log);

		// create state tracker
		m_State = new MVOState();

		// create smart contract config map
		m_SmartContracts = new Hashtable<Long, SmartContractConfig>();

		// create blockchain API object map
		m_BlockchainAPIs = new Hashtable<Long, BlockchainAPI>();

		// create MVOStaking listener map
		m_StakingListeners = new Hashtable<Long, MVOStakingListener>();

		// create the client request handlers
		String clientPort = m_PropConfig.getProperty("ClientPort");
		if (clientPort == null || clientPort.isEmpty()) {
			clientPort = "10443";
			m_PropConfig.setProperty("ClientPort", clientPort);
		}
		int cPort = 10443;
		try {
			cPort = Integer.parseInt(clientPort.trim());
		}
		catch (NumberFormatException nfe) {
			m_Log.error("Bad ClientPort properties value, " + clientPort);
		}
		m_ClientHandler = new ClientHandler(this, m_Log, cPort);

		// create the MVO request handler
		String mvoPort = m_PropConfig.getProperty("MVOPort");
		if (mvoPort == null || mvoPort.isEmpty()) {
			mvoPort = "10444";
			m_PropConfig.setProperty("MVOPort", mvoPort);
		}
		int mPort = 10444;
		try {
			mPort = Integer.parseInt(mvoPort.trim());
		}
		catch (NumberFormatException nfe) {
			m_Log.error("Bad MVOPort properties value, " + mvoPort);
		}
		m_MVOHandler = new MVOHandler(this, m_Log, mPort);
		m_MVOClient = new MVOClient(this);

		// configure Apache logging to use our LogFactory class (for Proxool)
		System.setProperty("org.apache.commons.logging.LogFactory",
							"cc.enshroud.jetty.log.CommonsLogFactory");

		// init receipt DB interface
		String dbUrl = m_PropConfig.getProperty("DBUrl");
		if (dbUrl == null) {
			dbUrl = "jdbc:mysql://localhost:3306/enshroud";
		}
		String dbUser = m_PropConfig.getProperty("DBUser");
		if (dbUser == null) {
			m_Log.error("Missing DB user");
		}
		String dbPw = m_PropConfig.getProperty("DBPassword");
		if (dbPw == null) {
			m_Log.error("Missing DB pass");
		}
		m_DbManager = new DbConnectionManager(dbUrl, dbUser, dbPw, m_Log);

		// create the client receipt request handler
		String receiptPort = m_PropConfig.getProperty("ReceiptPort");
		if (receiptPort == null || receiptPort.isEmpty()) {
			receiptPort = "10447";
			m_PropConfig.setProperty("ReceiptPort", receiptPort);
		}
		int rPort = 10447;
		try {
			rPort = Integer.parseInt(receiptPort.trim());
		}
		catch (NumberFormatException nfe) {
			m_Log.error("Bad ReceiptPort properties value, " + receiptPort);
		}
		m_ReceiptHandler = new ReceiptHandler(this, m_Log, rPort);
	}

	/**
	 * main run method
	 * @param args array of arguments passed at invocation
	 */
	public static void main(String[] args) {
		// look for a -n config option, which tells us our instance number
		int instanceNum = 1;
		for (int iii = 0; iii < args.length; iii++) {
			String arg = args[iii];
			if (arg.equals("-n")) {
				// next arg will be the instance number
				String num = args[iii+1];
				try {
					int instance = Integer.parseInt(num.trim());
					instanceNum = instance;
				}
				catch (NumberFormatException nfe) {
					System.err.println("Illegal instance number, " + num);
					System.exit(2);
				}
			}
			else if (arg.startsWith("-")) {
				// something unsupported
				System.err.println("Unrecognized flag, " + arg);
				System.exit(1);
			}
		}

		// read the properties file for our instance
		m_MVONumber = instanceNum;
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		String confFile = "MVO-" + nf.format(m_MVONumber) + ".properties";
		Properties props = new Properties();
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(confFile);
			props.load(fis);
			fis.close();
		}
		catch (FileNotFoundException fnfe) {
			System.err.println("Could not find properties file, " + confFile);
			System.exit(3);
		}
		catch (IOException ioe) {
			System.err.println("Error reading properties file, " + confFile);
			System.exit(4);
		}

		// establish the server object
		final MVO mvoServer = new MVO(props);
		if (!mvoServer.initialize()) {
			mvoServer.log().error("Unable to initialize MVO server, exiting");
			System.exit(5);
		}
		mvoServer.log().debug("Enshroud MVO instance " + nf.format(m_MVONumber)
							+ " initialized, version " + M_Version);
	}

	/**
	 * initialize MVO server
	 * @return true on success
	 */
	public boolean initialize() {
		final String lbl = this.getClass().getSimpleName() + ".initialize: ";
		// provide additional properties defaults
		if (m_PropConfig.getProperty("TrustStore") == null) {
			m_PropConfig.setProperty("TrustStore", "security/truststore");
		}
		if (m_PropConfig.getProperty("TrustStoreType") == null) {
			m_PropConfig.setProperty("TrustStoreType", "JKS");
		}
		if (m_PropConfig.getProperty("ClientKeyStore") == null) {
			m_PropConfig.setProperty("ClientKeyStore",
									 "security/keystore_client");
		}
		if (m_PropConfig.getProperty("PeerKeyStore") == null) {
			m_PropConfig.setProperty("PeerKeyStore",
									 "security/keystore_peer");
		}
		if (m_PropConfig.getProperty("KeyStoreType") == null) {
			m_PropConfig.setProperty("KeyStoreType", "PKCS12");
		}
		if (m_PropConfig.getProperty("KeyStorePass") == null) {
			m_PropConfig.setProperty("KeyStorePass", "changeit");
		}

		// populate m_Config with keypair used on L2
		StringBuilder keyPath = new StringBuilder(128);
		String secDir = m_PropConfig.getProperty("RunDir", "");

		// private key first
		String privKeyPath = m_PropConfig.getProperty("PrivKeyFile");
		if (privKeyPath == null || privKeyPath.isEmpty()) {
			// build a default
			if (secDir.isEmpty()) {
				// rely on relative path
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + m_MVOId + File.separator);
			keyPath.append("privkey.asc");
		}
		else {
			if (!secDir.isEmpty()) {
				keyPath.append(secDir + File.separator + privKeyPath);
			}
			else {
				keyPath.append(privKeyPath);
			}
		}
		File privKeyFile = new File(keyPath.toString());
		if (!privKeyFile.exists() || !privKeyFile.canRead()) {
			m_Log.error(lbl + "cannot open private key file " + privKeyFile
						+ " for read");
			return false;
		}
		// skip newline at end
		int keyLen = (int) privKeyFile.length() - 1;
		char[] keyBuff = new char[keyLen];
		try {
			FileReader kfr = new FileReader(privKeyFile);
			kfr.read(keyBuff, 0, keyLen);
			kfr.close();
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "unable to read private key from file", ioe);
			return false;
		}
		String keyData = new String(keyBuff);
		PrivateKey privKey = EncodingUtils.getPrivkeyFromBase64Str(keyData);
		Arrays.fill(keyBuff, ' ');
		if (privKey == null) {
			m_Log.error(lbl + "bad MVO private key data, " + keyPath);
			return false;
		}
		m_Config.configCommPrivkey(privKey);

		// now the public key
		keyPath = new StringBuilder(128);
		String pubKeyPath = m_PropConfig.getProperty("PubKeyFile");
		if (pubKeyPath == null || pubKeyPath.isEmpty()) {
			// build a default
			if (secDir.isEmpty()) {
				// rely on relative path
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + m_MVOId + File.separator);
			keyPath.append("pubkey.asc");
		}
		else {
			if (!secDir.isEmpty()) {
				keyPath.append(secDir + File.separator + pubKeyPath);
			}
			else {
				keyPath.append(pubKeyPath);
			}
		}
		File pubKeyFile = new File(keyPath.toString());
		if (!pubKeyFile.exists() || !pubKeyFile.canRead()) {
			m_Log.error(lbl + "cannot open public key file " + pubKeyFile
						+ " for read");
			return false;
		}
		// skip newline at end
		keyLen = (int) pubKeyFile.length() - 1;
		keyBuff = new char[keyLen];
		try {
			FileReader kfr = new FileReader(pubKeyFile);
			kfr.read(keyBuff, 0, keyLen);
			kfr.close();
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "unable to read public key from file", ioe);
			return false;
		}
		keyData = new String(keyBuff);
		PublicKey pubKey = EncodingUtils.getPubkeyFromBase64Str(keyData);
		if (pubKey == null) {
			m_Log.error(lbl + "bad MVO public key data, " + keyPath);
			return false;
		}
		m_Config.configCommPubkey(pubKey);

		// init the default URI at which we listen for peer MVO connections
		URI mvoURI = computePeerURI(m_MVOId, false);
		/* NB: this is overridden in MVOHandler by listening on $MVOPort on
		 * 	   all supported interfaces, using ws://
		 */
		m_Config.setMVOURI(mvoURI);

		// init list of configured Auditors
		String audIdList = m_PropConfig.getProperty("AuditorList");
		if (audIdList == null || audIdList.isEmpty()) {
			m_Log.error(lbl + "missing AuditorList in properties");
			return false;
		}
		m_AuditorManager = new AuditorQueue(this);
		Object aListObj = null;
		try {
			aListObj = JSON.parse(audIdList);
		}
		catch (IllegalStateException ise) { /* log below */ }
		if (!(aListObj instanceof Object[])) {
			m_Log.error(lbl + "improper AuditorList in properties, "
						+ audIdList);
			return false;
		}
		Object[] audListObj = (Object[]) aListObj;
		for (int aaa = 0; aaa < audListObj.length; aaa++) {
			Object audIdObj = audListObj[aaa];
			if (!(audIdObj instanceof String)) {
				m_Log.error(lbl + "illegal Aud spec in AuditorList, "
							+ audIdList);
				return false;
			}
			// this must be a string AUD-NNN:port
			String audTxt = (String) audIdObj;
			String[] audSpec = audTxt.split(":");
			if (audSpec.length != 2) {
				m_Log.error(lbl + "illegal Aud spec in AuditorList, " + audTxt);
				return false;
			}
			String audId = audSpec[0];
			String portId = audSpec[1];
			Integer aPort = null;
			try {
				aPort = Integer.parseInt(portId.trim());
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "illegal Aud port in AuditorList, " + portId);
				return false;
			}
			m_Config.addAuditor(audId, aPort);

			/* NB: we cannot start any auditor queues until we have initialized
			 * our peer pubkeys and started the MVOHandler (else connects fail)
			 */
		}
		// check we have at least one
		HashMap<String, Integer> allAuds = m_Config.getAuditorList();
		if (allAuds.isEmpty()) {
			m_Log.error(lbl + "no valid Auditor configurations in properties");
			return false;
		}

		// init peer pubkeys
		if (!initPeerKeys(secDir)) {
			m_Log.error(lbl + "error initializing peer public keys");
			return false;
		}

		// now confirm that we have a key for every configured Auditor
		boolean audKeyErr = false;
		for (String aId : allAuds.keySet()) {
			if (m_Config.getPeerPubkey(aId) == null) {
				m_Log.error(lbl + "no peer pubkey found for Auditor Id " + aId);
				audKeyErr = true;
			}
		}
		if (audKeyErr) return false;

		/* set the override handler for RxJavaPlugins.onError() to help debug
		 * UndeliverableException handling when doing Subscription.dispose()
		 */
		RxJavaPlugins.setErrorHandler(e -> {
			if (e instanceof UndeliverableException) {
				e = e.getCause();
			}
			if ((e instanceof IOException)
				|| (e instanceof SocketException))
			{
				// fine, network problem or API that throws on cancelation
				m_Log.warning(lbl + "network error disposing of subscription: "
							  + e.toString());
				return;
			}
			if (e instanceof InterruptedException) {
				// fine, some blocking code was interrupted by a dispose()
				m_Log.warning(lbl + "blocking code interrupted by dispose", e);
				return;
			}
			if (e instanceof ClientConnectionException) {
				// fine, a dispose() call could not talk to the other side
				m_Log.warning(lbl + "connection failure during a dispose: "
							+ e.getMessage());
				return;
			}
			if ((e instanceof NullPointerException)
				|| (e instanceof IllegalArgumentException))
			{
				// that's likely a bug in the application
				m_Log.error(lbl + "possible bug due to NPE/IAE", e);
				Thread.currentThread().getUncaughtExceptionHandler()
					.uncaughtException(Thread.currentThread(), e);
				return;
			}
			if (e instanceof IllegalStateException) {
				// that's a bug in RxJava or in a custom operator
				m_Log.error(lbl + "possible bug in RxJava", e);
				Thread.currentThread().getUncaughtExceptionHandler()
					.uncaughtException(Thread.currentThread(), e);
				return;
			}
			if (e instanceof FilterException) {
				/* This occurs when the calling thread in the Executor has
				 * exited.  Because we call Executor.shutdownNow() first before
				 * calling .dispose(), this is expected.
				 */
				return;
			}
			if (e instanceof WebsocketNotConnectedException) {
				/* This occurs when a wss-based JSON-RPC connection is lost
				 * and reconnects, and EnftCache.shutdown() is called, which
				 * has no place to deliver the exception.  Expected.
				 */
				return;
			}
			m_Log.warning(lbl + "Undeliverable exception received, cause: "
						  + e.toString());
		});

		// establish a shutdown hook
		ShutdownHook sdh = new ShutdownHook(this);
		Runtime.getRuntime().addShutdownHook(sdh);

		// init blockchain configs from smart contracts, plus properties
		if (!initSmartContractConfigs()) {
			m_Log.error(lbl
						+ "unable to initialize smart contract configurations");
			return false;
		}
		// NB: at this point all the SmartContractConfigs should be enabled

		// setup the listeners for events which handle MVOStaking updates
		for (Long chId : m_SmartContracts.keySet()) {
			SmartContractConfig scConfig = m_SmartContracts.get(chId);
			if (!setupMVOStakingListener(scConfig)) {
				scConfig.log().error(lbl
								+ "error establishing listener for changes to "
								+ "MVOStakings for chainId " + chId);
			}
		}

		// test the DB connection for receipts (must do before receipt queue)
		Connection conn = null;
		try {
			conn = m_DbManager.getConnection();
			m_DbManager.closeConnection(conn);
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "could not obtain test DB connection", sqle);
			return false;
		}

		// initialize the receipt upload queue object
		String queuePath = m_PropConfig.getProperty("ReceiptQueueFile");
		m_ReceiptUploader = new ReceiptQueue(this, queuePath);
		if (!m_ReceiptUploader.start()) {
			m_Log.error(lbl + "could not start ReceiptQueue");
			return false;
		}

		/* Tell every SmartContractConfig we have that ReceiptQueue is the
		 * EnftListener for mint and burn events (whether or not chain is init).
		 */
		for (Long chaId : m_SmartContracts.keySet()) {
			SmartContractConfig cScc = m_SmartContracts.get(chaId);
			cScc.configEnftListener(m_ReceiptUploader);
			// also set listener in the EnftCache object if it exists yet
			EnftCache eCache = cScc.getCache();
			if (eCache != null) {
				eCache.registerListener(m_ReceiptUploader);
			}
			// else: the SCC itself will set listener when connect happens
		}

		/* now that we have pubkeys set up, establish the server and client
		 * objects for handling WebSocket connections to peers
		 */
		// start up the websocket incoming connection handler
		if (!m_MVOHandler.initialize()) {
			m_Log.error(lbl + "unable to start MVOHandler, exit");
			return false;
		}

		// start up the websocket outgoing handler (depends on m_MVOHandler)
		if (!m_MVOClient.initialize()) {
			m_Log.error(lbl + "unable to start MVOClient, exit");
			return false;
		}

		// now start a queue in the manager for each defined Auditor
		for (String audId : allAuds.keySet()) {
			// add a distinct queue in the manager for this Auditor
			String aPath = "security" + File.separator + audId + File.separator;
			if (!m_AuditorManager.configAuditorQueue(audId, aPath)) {
				m_Log.error(lbl + "could not init queue for Auditor " + audId);
				return false;
			}
		}

		// start up the dApp client handler (also does SSL factory)
		if (!m_ClientHandler.initialize()) {
			m_Log.error(lbl + "unable to start ClientHandler, exit");
			return false;
		}

		// start up the dApp receipt handler (depends on ClientHandler)
		if (!m_ReceiptHandler.initialize()) {
			m_Log.error(lbl + "unable to start ReceiptHandler, exit");
			return false;
		}

		// schedule hourly housekeeping
		m_HKPTimer = new Timer("housekeeping", true);
		m_HKPTimer.scheduleAtFixedRate(this,
									   javax.management.timer.Timer.ONE_HOUR,
									   javax.management.timer.Timer.ONE_HOUR);
		return true;
	}

	// GET methods
	/**
	 * obtain the MVO Id
	 * @return the ID of our signing key
	 */
	public String getMVOId() { return m_MVOId; }

	/**
	 * obtain the MVO config (also loaded from supported smart contracts)
	 * @return the configuration, which spans all the blockchains we support
	 */
	public MVOConfig getConfig() { return m_Config; }

	/**
	 * obtain the properties config
	 * @return the local config object
	 */
	public Properties getProperties() { return m_PropConfig; }

	/**
	 * obtain the state-tracker
	 * @return the object used to track state
	 */
	public MVOState getStateObj() { return m_State; }

	/**
	 * obtain the logging object
	 * @return the logger used for errors and debug messages
	 */
	public Log log() { return m_Log; }

	/**
	 * obtain the client request processor
	 * @return the object which handles dApp user requests
	 */
	public ClientHandler getClientHandler() { return m_ClientHandler; }

	/**
	 * obtain the receipt request processor
	 * @return the object which handles dApp receipt requests
	 */
	public ReceiptHandler getReceiptHandler() { return m_ReceiptHandler; }

	/**
	 * obtain the MVO WebSocket processor
	 * @return the object which handles peer MVO WebSocket requests
	 */
	public MVOHandler getMVOHandler() { return m_MVOHandler; }

	/**
	 * obtain the MVO WebSocket client
	 * @return the object which handles creating MVO WebSocket connections
	 */
	public MVOClient getMVOClient() { return m_MVOClient; }

	/**
	 * obtain the Auditor queue manager
	 * @return the object which manages the queues for all defined Auditors
	 */
	public AuditorQueue getAuditorManager() { return m_AuditorManager; }

	/**
	 * obtain the DB connection manager
	 * @return the manager object
	 */
	public DbConnectionManager  getDbManager() { return m_DbManager; }

	/**
	 * obtain a random Auditor
	 * @param usedAuds list of AUD Ids caller has already tried, or null
	 * @return the ID of the Auditor selected, or "" if usedAuds supplied and
	 * no ID is available which is not already in the list
	 */
	public String getRandomAuditor(ArrayList<String> usedAuds) {
		HashMap<String, Integer> allAuds = m_Config.getAuditorList();
		ArrayList<String> audIDs = new ArrayList<String>(allAuds.keySet());
		ArrayList<String> doNotUseIds = new ArrayList<String>(audIDs.size());
		if (usedAuds != null) {
			doNotUseIds.addAll(usedAuds);
		}
		String aId = "";
		// fail-fast check for condition that will cause endless loop
		if (doNotUseIds.size() >= audIDs.size()) {
			return aId;
		}
		SecureRandom rng = m_State.getRNG();
		int idx = 0;
		boolean foundId = false;
		// extra caution to prevent hang by limiting max iterations
		int itCnt = 0;
		final int maxIt = audIDs.size() * 50;

		// repeat random selection until we hit something not under restriction
		while (!foundId && itCnt < maxIt) {
			idx = rng.nextInt(audIDs.size());
			try {
				aId = audIDs.get(idx);
				if (!doNotUseIds.contains(aId)) {
					// this one is allowed
					foundId = true;
				}
			}
			catch (IndexOutOfBoundsException ioobe) {
				// ArrayList somehow changed during this call! (impossible)
				m_Log.error("getRandomAuditor: no such Auditor index, " + idx,
							ioobe);
				// return first if allowed
				aId = audIDs.get(0);
				if (doNotUseIds.contains(aId)) {
					aId = "";
				}
				// bail regardless
				foundId = true;
			}
			itCnt++;
		}
		return aId;
	}

	/**
	 * obtain the smart contract config object for a given blockchain Id
	 * @param chainId the Id of the chain we want the config for
	 * @return the config, or null if not found
	 */
	public SmartContractConfig getSCConfig(long chainId) {
		return m_SmartContracts.get(chainId);
	}

	/**
	 * calculate the URI for an MVO or AUD node, on the red/black VPNs, based
	 * on its ID
	 * @param nodeId the MVOId or AUDId
	 * @param reverse true if we want to use the IP from the opposite VPN
	 * @return the websocket URI to communicate with the node
	 */
	public URI computePeerURI(String nodeId, boolean reverse) {
		final String lbl
			= this.getClass().getSimpleName() + ".computePeerURI: ";
		URI nodeURI = null;
		String uriStr = "ws://";
		String vpnEven = m_PropConfig.getProperty("VPNEVEN", "10.2.0.");
		String vpnOdd = m_PropConfig.getProperty("VPNODD", "10.1.0.");

		// determine whether MVO or AUD
		if (nodeId.startsWith("MVO-")) {
			/* Build the URI based on properties values.  If the MVO's Id is
			 * even, use the formula VPNODD with IP = Id+20.  If it's even, use
			 * the formula VPNEVEN with IP = Id+20.  Examples: MVO-001 would be
			 * calculated as 10.1.0.21, MVO-002 as 10.2.0.22.  Specifying
			 * reverse=true results in the opposite.
			 */
			int idIdx = nodeId.indexOf("-") + 1;
			String mId = nodeId.substring(idIdx);
			try {
				int mvoId = Integer.parseUnsignedInt(mId);
				Integer mIP = mvoId + MVOGenConfig.MVO_VPN_OFFSET;
				if (mvoId % 2 == 0) {
					uriStr += (reverse ? vpnOdd : vpnEven);
				}
				else {
					uriStr += (reverse ? vpnEven : vpnOdd);
				}
				/* NB: a MVO with :port != 10444 must have its URI specified in
				 * a security/MVOId.properties file containing MVOURI=
				 */
				uriStr += (mIP + ":10444/mvo");
				nodeURI = new URI(uriStr);
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "illegal MVOId, " + nodeId, nfe);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "error building MVO URI from " + uriStr, use);
			}
		/*
			m_Log.debug(lbl + "computed URI for MVO " + nodeId + " = "
						+ nodeURI);
		 */
		}
		else if (nodeId.startsWith("AUD-")) {
			/* Build the URI based on properties values.  If the AUD's Id is
			 * even, use the formula VPNODD with IP = Id+1.  If it's even, use
			 * the formula VPNEVEN with IP = Id+1.  Examples: AUD-001 would be
			 * calculated as 10.1.0.2, AUD-002 as 10.2.0.3.  Specifying
			 * reverse=true results in the opposite.
			 */
			int idIdx = nodeId.indexOf("-") + 1;
			String aId = nodeId.substring(idIdx);
			HashMap<String, Integer> availAuds = m_Config.getAuditorList();
			Integer audPort = availAuds.get(nodeId);
			try {
				int audId = Integer.parseUnsignedInt(aId);
				Integer aIP = audId + MVOGenConfig.AUD_VPN_OFFSET;
				if (audId % 2 == 0) {
					uriStr += (reverse ? vpnOdd : vpnEven);
				}
				else {
					uriStr += (reverse ? vpnEven : vpnOdd);
				}
				uriStr += (aIP + ":" + audPort + "/keyserver");
				nodeURI = new URI(uriStr);
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "illegal AUDId, " + nodeId, nfe);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "error building AUD URI from " + uriStr, use);
			}
		/*
			m_Log.debug(lbl + "computed URI for AUD " + nodeId + " = "
						+ nodeURI);
		 */
		}
		else {
			m_Log.error(lbl + "illegal nodeId, " + nodeId);
		}
		return nodeURI;
	}

	/**
	 * find a URI for a given MVO or Auditor
	 * @param mvoId the ID of the MVO or Auditor node (e.g. MVO-001, AUD-001)
	 * @return the URI at which this MVO can be contacted, or null if not found
	 */
	public URI getURIforMVO(String mvoId) {
		if (mvoId == null || mvoId.isEmpty()) {
			return null;
		}
		final String lbl = this.getClass().getSimpleName() + ".getURIforMVO: ";

		URI mvoURI = null;
		if (mvoId.startsWith("MVO-")) {
			/* We make the assumption that the URI will be the same in the
			 * config for any blockchain in which the MVO appears.  Therefore,
			 * we simply take the first occurrence found.  These URIs are
			 * built by RemoteBlockchainAPI.getConfig(), but sourced via the
			 * algo used in computePeerURI(), with the MVOURI property
			 * found in any security/mvoId.properties file as an override.
			 */
			for (SmartContractConfig scc : m_SmartContracts.values()) {
				Hashtable<String, MVOGenConfig> availMVOs = scc.getMVOMap();
				MVOGenConfig mvoConf = availMVOs.get(mvoId);
				if (mvoConf != null) {
					mvoURI = mvoConf.getMVOURI();
					break;
				}
			}
		}
		else if (mvoId.startsWith("AUD-")) {
			// check that this is a valid Auditor
			HashMap<String, Integer> availAuds = m_Config.getAuditorList();
			if (!availAuds.containsKey(mvoId)) {
				m_Log.error(lbl + "unknown Auditor, " + mvoId);
				return null;
			}
			mvoURI = computePeerURI(mvoId, false);

			// check for an override, stored in security/$AudId.properties
			String secDir = m_PropConfig.getProperty("RunDir", "");
			StringBuilder propsPath = new StringBuilder(64);
			if (secDir.isEmpty()) {
				propsPath.append("security");
			}
			else {
				propsPath.append(secDir + File.separator + "security");
			}
			propsPath.append(File.separator + mvoId + ".properties");
			File propFile = new File(propsPath.toString());
			if (propFile.exists() && propFile.canRead()) {
				// read the file
				Properties audProps = new Properties();
				FileInputStream fis = null;
				try {
					fis = new FileInputStream(propFile);
					audProps.load(fis);
					fis.close();
				}
				catch (FileNotFoundException fnfe) {
					// (we already checked for this)
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "error reading properties file, "
								+ propFile, ioe);
				}
				String audURI = audProps.getProperty("MVOURI");
				URI uri = null;
				try {
					uri = new URI(audURI);
					// return override
					mvoURI = uri;
				}
				catch (URISyntaxException ufe) {
					m_Log.error(lbl + "bad format, " + audURI, ufe);
				}
			}
		}
		else {
			m_Log.error(lbl + "illegal MVO/AUD Id passed, " + mvoId);
		}
		return mvoURI;
	}

	/**
	 * obtain the blockchain interface handler
	 * @param chainId the chain for which we want the interface
	 * @return the object we use to talk to blockchains
	 */
	public BlockchainAPI getWeb3(long chainId) {
		return m_BlockchainAPIs.get(chainId);
	}

	/**
	 * obtain the receipt queueing object
	 * @return the receipt queue, used to upload receipts
	 */
	public ReceiptQueue getReceiptQueue() { return m_ReceiptUploader; }


	// SET methods
	/**
	 * initialize peer pubkeys for MVOs and Auditors from static files
	 * @param secDir security directory where these files are stored
	 * @return true on success
	 */
	public boolean initPeerKeys(String secDir) {
		if (secDir == null) {
			return false;
		}

		StringBuilder keyPath = new StringBuilder(64);
		if (secDir.isEmpty()) {
			// rely on relative path
			keyPath.append("security");
		}
		else {
			keyPath.append(secDir + File.separator + "security");
		}

		/* list all files in this directory which are of the form MVO-NNN.pubkey
		 * or AUD-NNN.pubkey, for peer MVOs and Auditors
		 */
		File securityDir = new File(keyPath.toString());
		File[] peerPubkeys = null;
		try {
			peerPubkeys = securityDir.listFiles(this);
		}
		catch (SecurityException se) {
			m_Log.error("No permission to access peer pubkeys", se);
			return false;
		}
		boolean keyErr = false;
		if (peerPubkeys == null || peerPubkeys.length == 0) {
			m_Log.error("No peer pubkeys found in config");
			keyErr = true;
		}
		else {
			for (int ppp = 0; ppp < peerPubkeys.length; ppp++) {
				File pubKeyFile = peerPubkeys[ppp];
				if (!pubKeyFile.exists() || !pubKeyFile.canRead()) {
					m_Log.error("Cannot open public key file " + pubKeyFile
								+ " for read");
					keyErr = true;
					continue;
				}
				// skip newline at end
				int keyLen = (int) pubKeyFile.length() - 1;
				char[] keyBuff = new char[keyLen];
				try {
					FileReader kfr = new FileReader(pubKeyFile);
					kfr.read(keyBuff, 0, keyLen);
					kfr.close();
				}
				catch (IOException ioe) {
					m_Log.error("Unable to read public key from file", ioe);
					keyErr = true;
					continue;
				}
				PublicKey pubKey
					= EncodingUtils.getPubkeyFromBase64Str(new String(keyBuff));
				if (pubKey == null) {
					m_Log.error("Bad peer public key data, "
								+ pubKeyFile.getPath());
					keyErr = true;
				}
				else {
					int dotPos = pubKeyFile.getName().indexOf(".");
					String peerId = pubKeyFile.getName().substring(0, dotPos);
					m_Config.addPeerPubkey(peerId, pubKey);
				}
			}
		}
		return !keyErr;
	}

	/**
	 * Initialize the smart contract configurations, from the blockchains
	 * themselves.  In our properties we have only a list
	 * of chains to deal with.  Our ABI partner(s) must provide access via a
	 * URL.  If we configure a blockchain, it's required that we be listed as
	 * a MVO for that blockchain (otherwise we'll get no business).
	 * @return true on success
	 */
	private boolean initSmartContractConfigs() {
		final String lbl = this.getClass().getSimpleName()
						+ ".initSmartContractConfigs: ";
		String scArray = m_PropConfig.getProperty("SmartContractList");
		if (scArray == null || scArray.isEmpty()) {
			m_Log.error(lbl + "missing smart contract config");
			return false;
		}
		Object scObj = null;
		try {
			scObj = JSON.parse(scArray);
		}
		catch (IllegalStateException ise) { /* log below */ }
		if (!(scObj instanceof Object[])) {
			m_Log.error(lbl + "parse error on SmartContractList value, "
						+ scArray);
			return false;
		}
		Object[] scAry = (Object[]) scObj;
		boolean scConfErr = false;
		for (int sss = 0; sss < scAry.length; sss++) {
			Object scItem = scAry[sss];
			// should have a list of longs
			if (!(scItem instanceof Long)) {
				m_Log.error(lbl + "item " + (sss+1)
							+ " in SmartContractList is not a Long");
				return false;
			}

			// look for detail entry in properties
			Long chainId = (Long) scItem;
			String scDetails
				= m_PropConfig.getProperty("SmartContract-" + chainId);
			if (scDetails == null || scDetails.isEmpty()) {
				m_Log.error(lbl + "missing SmartContract-" + chainId
							+ " properties");
				scConfErr = true;
				continue;
			}

			/**
			 * should yield a Map with these entries -
			 * name = common name for blockchain
			 * ABI = local URI at which we can contact the full blockchain node
			 * Genesis = block number in which EnshroudProtocol was deployed
			 * ReceiptStore = URI at which we can upload/download receipts
			 */
			Object scMap = null;
			try {
				scMap = JSON.parse(scDetails);
			}
			catch (IllegalStateException ise) { /* log below */ }
			if (!(scMap instanceof Map)) {
				m_Log.error(lbl + "SmartContract-" + chainId
							+ " details didn't parse");
				scConfErr = true;
				continue;
			}
			Map scDetMap = (Map) scMap;
			Object nmObj = scDetMap.get("name");
			String chainName = "";
			if (nmObj instanceof String) {
				chainName = (String) nmObj;
			}
			if (chainName == null || chainName.isEmpty()) {
				m_Log.error(lbl + "missing name= for " + chainId + " config");
				scConfErr = true;
			}
			Object uriObj = scDetMap.get("ABI");
			String abiUri = "";
			if (uriObj instanceof String) {
				abiUri = (String) uriObj;
			}
			if (abiUri == null || abiUri.isEmpty()) {
				m_Log.error(lbl + "missing ABI= for " + chainId + " config");
				scConfErr = true;
			}
			URI web3jURI = null;
			try {
				web3jURI = new URI(abiUri);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "Web3j URI for chainId " + chainId
							+ ", bad format", use);
				scConfErr = true;
			}
			Object genesisObj = scDetMap.get("Genesis");
			BigInteger deployBlock = null;
			if (genesisObj instanceof String) {
				String depBlock = (String) genesisObj;
				try {
					deployBlock = new BigInteger(depBlock);
				}
				catch (NumberFormatException nfe) {
					m_Log.error(lbl + "Genesis block for chainId " + chainId
								+ ", invalid block = " + depBlock);
				}
			}
			if (deployBlock == null) {
				m_Log.error(lbl + "missing Genesis= for " + chainId
							+ " config");
				// default to 1
				deployBlock = BigInteger.ONE;
			}
			Object receiptObj = scDetMap.get("ReceiptStore");
			String receiptLoc = "";
			if (receiptObj instanceof String) {
				receiptLoc = (String) receiptObj;
			}
			if (receiptLoc == null || receiptLoc.isEmpty()) {
				m_Log.error(lbl + "missing ReceiptStore= for " + chainId
							+ " config");
				scConfErr = true;
			}
			URI receiptURI = null;
			try {
				receiptURI = new URI(receiptLoc);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "ReceiptStore URI for chainId " + chainId
							+ ", bad format", use);
				scConfErr = true;
			}
			if (scConfErr) {
				m_Log.error(lbl + "fatal errors in SC properties config for "
							+ "chainId " + chainId + " or a predecessor");
				continue;
			}

			// create a smart contract config object and init it
			SmartContractConfig scConfig
				= new SmartContractConfig(chainName, chainId, m_Log, this);
			scConfig.setWeb3jURI(web3jURI);
			scConfig.setReceiptStore(receiptURI);
			scConfig.setDeploymentBlock(deployBlock);
			scConfig.configCacheRebuilder(this);

			// create the BlockchainConfig entry in our MVOConfig
			BlockchainConfig bConfig
				= m_Config.addBlockchainConfig(chainId, chainName);

			// form a connection to the Web3j URL
			Web3j abiSession = null;
			BlockchainAPI web3Api = null;
			String proto = web3jURI.getScheme();
			boolean deferChainInit = false;
			Log scLog = scConfig.log();
			scLog.debug(lbl + chainName
						+ ": connecting to JSON-RPC @" + web3jURI.toString());
			if (proto.equals("file")) {
				// use test flat-file implementation
				web3Api = new LocalBlockchainAPI(this, chainId);
				if (!web3Api.init()) {
					m_Log.error(lbl + "error initializing LocalBlockchainAPI");
					scConfErr = true;
					continue;
				}
			}
			else if (proto.startsWith("http")) {
				// NB: regular HttpClient will be slow and shouldn't be used
				OkHttpClient okClient = new OkHttpClient();
				/* configure 60 sec timeout for fetching events, to match value
				 * org.web3j.protocol.websocket.WebSocketService.REQUEST_TIMEOUT
				 */
				OkHttpClient configClient = okClient.newBuilder()
										.readTimeout(60L, TimeUnit.SECONDS)
										.writeTimeout(60L, TimeUnit.SECONDS)
										.connectTimeout(60L, TimeUnit.SECONDS)
					.build();
				abiSession = Web3j.build(new HttpService(web3jURI.toString(),
										 configClient));
			}
			else if (proto.startsWith("ws")) {
				// websockets are persistent, therefore we must connect first
				WebSocketService wss
					= new WebSocketService(web3jURI.toString(), false);
				scConfig.configWSS(wss);
				try {
					/* We use connect(Consumer<String> onMessage,
					 * 				Consumer<Throwable> onError,
					 * 				Runnable onClose())
					 * with the methods located in scConfig object,
					 * so we can tell when the connection drops and restore it.
					 */
					wss.connect(scConfig,
								throwable -> scConfig.accept(throwable),
								scConfig);
					abiSession = Web3j.build(wss);
				}
				catch (ConnectException ce) {
					m_Log.error(lbl + "error connecting to websocket for "
								+ web3jURI.toString());
					// defer to reconnect logic by invoking disconnect handler
					scConfig.run();		// (forks thread)
					// pre-config a BlockchainAPI so we'll have one
					web3Api = new RemoteBlockchainAPI(this, chainId);
					web3Api.init();		// (doesn't currently do anything)
					// certain inits below which need JSON-RPC we can't do yet
					deferChainInit = true;
				}
			}
			if (abiSession != null && !deferChainInit) {
				web3Api = new RemoteBlockchainAPI(this, chainId);
				scConfig.configABISession(abiSession);
				web3Api.init();		// (doesn't currently do anything)

				// create and init the eNFT data cache
				EnftCache eCache
					= new EnftCache(scConfig, abiSession, BigInteger.ZERO);
				scConfig.configCache(eCache);
				if (!eCache.initialize(false)) {
					// implies configuration error; we should exit app
					m_Log.error(lbl + "error initializing EnftCache for "
								+ chainName);
					scConfErr = true;
					continue;
				}
			}
			if (web3Api == null) {
				m_Log.error(lbl + "unsupported Web3j ABI protocol, " + proto);
				scConfErr = true;
				continue;
			}
			m_BlockchainAPIs.put(chainId, web3Api);

			// record the config
			m_SmartContracts.put(chainId, scConfig);

			/* fetch our own config for this chain and configure:
			 * 	bConfig.m_Staked
			 * 	bConfig.m_SigningAddress
			 * 	bConfig.m_StakingAddress
			 * 	this one is loaded from local Credentials, not from blockchain:
			 * 	bConfig.m_SigningKey (find in wallet file specified in
			 *				m_PropConfig.getProperty("chainId-signwallet.json"))
			 */
			if (!deferChainInit) {
				if (!getOwnNodeConfig(scConfig)) {
					scLog.error(lbl + "error fetching self BlockchainConfig "
								+ "for chainId " + chainId);
					scConfErr = true;
					continue;
				}
			}

			/* next, read local files to find these:
			 *	bConfig.m_ECPrivKey
			 *	bConfig.m_ECPubKey (although not actually used for anything)
			 * we look in RunDir/security/MVO-xxx/ directory for:
			 * 	chainId + "-ecprivkey.asc" and chainId + "-ecpubkey.asc"
			 */
			StringBuilder keyPath = new StringBuilder(128);
			String secDir = m_PropConfig.getProperty("RunDir", "");
			if (secDir.isEmpty()) {
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + m_MVOId);
			String keyDir = keyPath.toString();

			// read EC public key
			String pubKeyFilename
				= keyDir + File.separator + chainId + "-ecpubkey.asc";
			File pubKeyFile = new File(pubKeyFilename);
			if (!pubKeyFile.exists() || !pubKeyFile.canRead()) {
				m_Log.error(lbl + "cannot open EC public key file " + pubKeyFile
							+ " for read");
				scConfErr = true;
				continue;
			}
			// skip newline at end
			int keyLen = (int) pubKeyFile.length() - 1;
			char[] keyBuff = new char[keyLen];
			try {
				FileReader kfr = new FileReader(pubKeyFile);
				kfr.read(keyBuff, 0, keyLen);
				kfr.close();
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "unable to read EC public key from file",
							ioe);
				scConfErr = true;
				continue;
			}
			String keyData = new String(keyBuff);
			PublicKey pubKey = EncodingUtils.getECPubkeyFromBase64Str(keyData);
			if (pubKey == null) {
				m_Log.error(lbl + "error reading EC public key from file "
							+ pubKeyFilename);
				scConfErr = true;
			}
			else {
				bConfig.setECPublicKey(pubKey);
			}

			// read EC private key
			String privKeyFilename
				= keyDir + File.separator + chainId + "-ecprivkey.asc";
			File privKeyFile = new File(privKeyFilename);
			if (!privKeyFile.exists() || !privKeyFile.canRead()) {
				m_Log.error(lbl + "cannot open EC private key file "
							+ privKeyFile + " for read");
				scConfErr = true;
				continue;
			}
			// skip newline at end
			keyLen = (int) privKeyFile.length() - 1;
			keyBuff = new char[keyLen];
			try {
				FileReader kfr = new FileReader(privKeyFile);
				kfr.read(keyBuff, 0, keyLen);
				kfr.close();
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "unable to read EC private key from file",
							ioe);
				scConfErr = true;
				continue;
			}
			keyData = new String(keyBuff);
			PrivateKey privKey
				= EncodingUtils.getECPrivkeyFromBase64Str(keyData);
			if (privKey == null) {
				m_Log.error(lbl + "error reading EC private key from file "
							+ privKeyFilename);
				scConfErr = true;
			}
			else {
				bConfig.setECPrivateKey(privKey);
			}

			// add ourselves to the SCC's MVO map (selection algo assumes this)
			scConfig.getMVOMap().put(m_MVOId, m_Config);

			if (!deferChainInit) {
				// grab MVOStaking records for all peer nodes
				if (!getOtherNodeConfigs(scConfig)) {
					scLog.error(lbl + "error getting peer on-chain configs "
								+ "for chainId " + chainId);
					scConfErr = true;
					continue;
				}

			}
		}
		if (scConfErr) {
			m_Log.error(lbl + "fatal errors in SC properties configs");
			return false;
		}
		return true;
	}

	// override run() to implement TimerTask
	@Override
	public void run() {
		// reinitialize blockchain subscriptions since last HKP run
		for (SmartContractConfig scc : m_SmartContracts.values()) {
			// reinit the cache
			EnftCache eCache = scc.getCache();
			if (eCache != null) {
				if (!eCache.reinitWhileRunning()) {
					scc.log().error("HKP unable to reinit cache for chain "
									+ scc.getChainName());
				}
			}
		}

		// reinitialize MVO listener subscriptions since last HKP run
		for (MVOStakingListener msl : m_StakingListeners.values()) {
			// reinit the listener
			if (msl != null) {
				msl.reinit();
			}
		}

		// now reclaim any freed heap memory
		System.gc();
	}

	// implement FilenameFilter
	/**
	 * method to select files in RunDir/security that are peer MVO pubkeys
	 * @param dir the directory the file was found in
	 * @param name the name of the file
	 * @return true if we should accept this file
	 */
	public boolean accept(File dir, String name) {
		if (dir == null || name == null) {
			return false;
		}
		if (dir.getPath().contains("security")) {
			if (name.endsWith(".pubkey")
				&& (name.startsWith("MVO-") || name.startsWith("AUD-")))
			{
				return true;
			}
		}
		return false;
	}

	// methods to implement BlockchainConnectListener
	/**
	 * obtain node Id
	 * @return our Id
	 */
	public String getSelfId() { return m_MVOId; }

	/**
	 * perform download tasks related to obtaining our own on-chain config
	 * @param scc the smart contract we just achieved a connection to
	 * @return true on success
	 */
	public boolean getOwnNodeConfig(SmartContractConfig scc) {
		final String lbl
			= this.getClass().getSimpleName() + ".getOwnNodeConfig: ";
		if (scc == null) {
			m_Log.error(lbl + "no SmartContractConfig input");
			return false;
		}
		long chainId = scc.getChainId();

		// use the Web3j connection to obtain our on-chain config
		BlockchainAPI web3Api = m_BlockchainAPIs.get(chainId);
		Future<BlockchainConfig> mvoConfFuture
			= web3Api.getMVOConfig(chainId, scc.getChainName(),
								   scc.getABI(), getSelfId());
		BlockchainConfig mvoBCconf = null;
		try {
			mvoBCconf = mvoConfFuture.get();
		}
		catch (Exception ee) {
			m_Log.error(lbl + "exception getting BlockchainConfig from "
						+ "Web3j: " + ee.toString());
		}
		if (mvoBCconf == null) {
			m_Log.error(lbl + "could not fetch BlockchainConfig from "
						+ "Web3j");
			return false;
		}

		// record data in allocated blockchain config record
		BlockchainConfig bConfig = m_Config.getChainConfig(chainId);
		bConfig.setStaking(mvoBCconf.getStaking().toString());
		bConfig.configStakingAddress(mvoBCconf.getStakingAddress());
		// NB: this item is always set directly from local Credentials:
		bConfig.configSigningAddress(mvoBCconf.getSigningAddress());
		bConfig.configSigningKey(mvoBCconf.getSigningKey());
		return true;
	}

	/**
	 * perform download tasks related to obtaining other nodes' configs
	 * @param scc the smart contract we just achieved a connection to
	 * @return true on success
	 */
	public boolean getOtherNodeConfigs(SmartContractConfig scc) {
		final String lbl
			= this.getClass().getSimpleName() + ".getOtherNodeConfigs: ";
		if (scc == null) {
			m_Log.error(lbl + "no SmartContractConfig input");
			return false;
		}
		long chainId = scc.getChainId();

		/* use the Web3j connection to obtain and record this data:
		 *	scc.m_MVOs (i.e. populate rest of mvoMap)
		 *	scc.m_ReqSigs
		 *	scc.m_DwellTime
		 * 	scc.m_TotalStaked
		 * 	scc.m_BaseURI
		 */
		BlockchainAPI web3Api = m_BlockchainAPIs.get(chainId);
		Future<SmartContractConfig> scFuture
			= web3Api.getConfig(chainId, scc.getChainName(), scc.getABI());
		SmartContractConfig scConf = null;
		try {
			scConf = scFuture.get();
		}
		catch (Exception ee) {
			m_Log.error(lbl + "exception getting SmartContractConfig "
						+ "from Web3j: " + ee.toString());
		}
		if (scConf == null) {
			m_Log.error(lbl + "could not fetch SmartContractConfig "
						+ "from Web3j");
			return false;
		}
		scc.getMVOMap().putAll(scConf.getMVOMap());
		scc.setNumSigs(scConf.getNumSigs());
		scc.setDwellTime(scConf.getDwellTime());
		scc.setTotalStaking(scConf.getTotalStaking().toString());
		scc.setBaseURI(scConf.getBaseURI());
		return true;
	}

	/**
	 * begin watching for MVOStaking record change events, after our calls to
	 * getOwnNodeConfig() and getOtherNodeConfigs()
	 * @param scc the smart contract we just achieved a connection to
	 * @return true on success
	 */
	public boolean setupMVOStakingListener(SmartContractConfig scc) {
		if (scc == null) {
			m_Log.error("setupMVOStakingListener: no SCC passed");
			return false;
		}

		// see whether we're on a real chain or not
		BlockchainAPI web3Api = m_BlockchainAPIs.get(scc.getChainId());
		if (web3Api instanceof LocalBlockchainAPI) {
			// nothing to be done here
			return true;
		}
		MVOStakingListener mvoSL = new MVOStakingListener(this, scc);
		if (mvoSL.initialize()) {
			// record in table
			m_StakingListeners.put(scc.getChainId(), mvoSL);
			return true;
		}
		return false;
	}

	// method to implement EnftCacheResetter
	/**
	 * rebuild the ABI session and EnftCache for a specified blockchain
	 * @param chainId the chain ID (used to look up SmartContractConfig)
	 * @param reason the reason we were invoked (provided by old EnftCache)
	 * @return false if cache init is being deferred on a WS connection, true
	 * if cache init is proceeding normally
	 */
	public boolean requestEnftCacheRestart(long chainId, String reason) {
		final String lbl
			= this.getClass().getSimpleName() + ".requestEnftCacheRestart: ";

		// find the SCC object
		SmartContractConfig scConfig = m_SmartContracts.get(chainId);
		if (scConfig == null) {
			m_Log.error(lbl + "CRITICAL: no SCC found for existing chain Id "
						+ chainId);
			return false;
		}
		Log scLog = scConfig.log();
		scLog.error(lbl + "invoked due to: \"" + reason + "\"");

		// to prevent overlapping calls, check flag and set if required
		if (scConfig.isCacheReinitInProgress()) {
			scLog.error(lbl + "overlapping call for " + scConfig.getChainName()
						+ ", ignoring");
			return false;
		}
		scConfig.setCacheReinitInProgress(true);

		// shut down the existing cache
		scConfig.setEnabled(false);
		EnftCache oldCache = scConfig.getCache();
		if (oldCache != null) {
			oldCache.shutdown();
		}

		// pull latest block from old cache
		BigInteger previousLatestBlock = oldCache.getLatestBlock();

		// form a new connection to the Web3j URL
		Web3j abiSession = null;
		BlockchainAPI web3Api = null;
		URI web3jURI = scConfig.getWeb3jURI();
		String proto = web3jURI.getScheme();
		boolean deferCacheInit = false;
		if (proto.startsWith("http")) {
			// stop existing ABI interface
			Web3j abiInterface = scConfig.getABI();
			if (abiInterface != null) {
				abiInterface.shutdown();
			}

			scLog.debug(lbl + "reconnecting to JSON-RPC @"
						+ web3jURI.toString());
			// NB: regular HttpClient will be slow and shouldn't be used
			OkHttpClient okClient = new OkHttpClient();
			/* configure 60 sec timeout for fetching events, to match value
			 * org.web3j.protocol.websocket.WebSocketService.REQUEST_TIMEOUT
			 */
			OkHttpClient configClient = okClient.newBuilder()
									.readTimeout(60L, TimeUnit.SECONDS)
									.writeTimeout(60L, TimeUnit.SECONDS)
									.connectTimeout(60L, TimeUnit.SECONDS)
				.build();
			abiSession = Web3j.build(new HttpService(web3jURI.toString(),
									 configClient));
		}
		else if (proto.startsWith("ws")) {
			// defer to reconnect logic by invoking disconnect handler
			scConfig.run();		// (forks thread via WSS_Connector)

			/* Certain inits which need JSON-RPC we can't do yet.  Do not
			 * init EnftCache yet because it needs this to operate.
			 */
			deferCacheInit = true;
		}

		// iff we created new ABI session (http case), init cache now
		if (abiSession != null) {
			// record API object
			scConfig.configABISession(abiSession);

			// create and init the eNFT data cache
			EnftCache eCache
				= new EnftCache(scConfig, abiSession, previousLatestBlock);
			scConfig.configCache(eCache);
			if (!eCache.initialize(false)) {
				// NB: this can only happen due to a configuration error
				m_Log.error(lbl + "CRITICAL: error initializing EnftCache for "
							+ scConfig.getChainName() + ", exiting");
				System.exit(6);
			}
		}

		// reset nested call protection flag unless we deferred
		if (!deferCacheInit) {
			scConfig.setCacheReinitInProgress(false);
		}

		return !deferCacheInit;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any potentially sensitive data
		try {
			if (m_PropConfig != null) {
				m_PropConfig.clear();
			}
			if (m_BlockchainAPIs != null) {
				m_BlockchainAPIs.clear();
			}
			if (m_StakingListeners != null) {
				m_StakingListeners.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
