/*
 * last modified---
 * 	02-24-24 limit memo lines to 1024 characters
 * 	03-08-24 try additional AUDs as key servers if we get a failure
 * 	01-30-23 remove m_DepositFee (vestigial)
 * 	12-04-23 do not auto-upload receipts; remove M_ReceiptUpload state
 * 	06-14-23 remove AssetConfig usage
 * 	06-12-23 remove standalone mode requirement
 * 	04-11-23 check number of payees vs M_ARRAY_MAX
 * 	04-05-23 build receipts with EIP-55 addresses and token contract assets
 * 	03-02-23 check and generate valid eNFT sigs, generate real receipt sigs
 * 	02-28-23 also sign the ArgumentsHash block in the AB; sign entire OB
 * 	02-21-23 add signing of ArgumentsHash block in OB, verification of OB sig
 * 	12-02-22 handle eNFTmetadata.m_Rand field, payee m_Rand
 * 	10-13-22 use EncodingUtils.buildDetailsHash()
 * 	08-19-22 aggregate sender receipt payee totals for same asset
 * 	08-12-22 debug AuditorBlock generation; re-allow 0 eNFT outputs
 * 	08-05-22 code for async requests for keys to Auditors
 * 	08-03-22 add code to broadcast signed AuditorBlockS to Auditors
 * 	07-22-22 prevent emission of amount=0 eNFTs
 * 	07-06-22 debug receipt processing / uploading
 * 	07-01-22 actually trigger upload of receipts in M_ReceiptUpload
 * 	06-30-22 use ReceiptQueue to store generated receipts
 * 	06-21-22 use AsyncContext
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	05-23-22 complete initial implementation draft
 * 	04-28-22 begin actual implementation
 * 	04-06-22 new (stub)
 *
 * purpose---
 * 	state machine transition object for handling dApp deposit requests
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.AuditorBlock;
import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.eNFTmetadata;
import cc.enshroud.jetty.ReceiptBlock;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.WalletUtils;
import org.web3j.crypto.Keys;

import java.util.Map;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.Base64;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.net.URI;
import java.text.NumberFormat;
import java.nio.charset.StandardCharsets;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.AsyncContext;


/**
 * This class provides the implementation of control logic to handle the various
 * stages required to perform an issuance one or more eNFTs as receipt tokens
 * (against previously deposited balance credit), along with error handling
 * and recovery for any problems encountered at each stage.  These objects are
 * created from {@link ClientHandler} in response to user requests
 * (implying m_IsLead is true).  They are also created from {@link MVOHandler}
 * when a lead MVO picks us as a committee member (implying m_IsLead is false).
 */
public final class DepositState extends ReqSvcState {
	// BEGIN data members
	// aliases for steps in the deposit process
	/**
	 * request received from dApp (parsed okay, sig verified)
	 */
	public static final int	M_ReqReceived = 1;
	/**
	 * request verified as acceptable (sanity, checked vs SC config, deposited
	 * balance credit is available, etc.)
	 */
	public static final int M_ReqVerified = 2;
	/**
	 * keys obtained from an Auditor for required output eNFTs (note a
	 * committee member MVO will lookup the keys rather than creating them)
	 */
	public static final int M_NewKeys = 3;
	/**
	 * OperationsBlock generated (and all output eNFTs generated)
	 */
	public static final int M_OB_Generated = 4;
	/**
	 * OperationsBlock signed (note a committee member compares its own OB data
	 * against the one passed by the lead, and signs the passed one) -- the
	 * completion of step implies that ALL committee MVOs have signed, not just
	 * the lead
	 */
	public static final int M_OB_Signed = 5;
	/**
	 * receipt keys fetched and receipts generated (lead MVO only)
	 */
	public static final int M_ReceiptGen = 6;
	/**
	 * reply sent (to dApp for lead, to lead MVO for committee member)
	 */
	public static final int M_Replied = 7;
	/**
	 * transaction mined, eNFTs appear in chain event log, user's balance
	 * credit reduced by total issuance (lead MVO only)
	 */
	public static final int M_Deposited = 8;
	/**
	 * processing is done, clean up
	 */
	public static final int M_Completed = 9;

	/**
	 * flag indicating whether we're the lead MVO, or just a committee member
	 * (true if lead)
	 */
	private boolean			m_IsLead;

	/**
	 * local shorthand copy of MVO's logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param brokerEnt the parent object to which we are attached
	 */
	public DepositState(MVOBrokerEntry brokerEnt) {
		super(brokerEnt);
		m_FinalStep = M_Completed;
		m_Log = m_Parent.getLog();
	}

	/**
	 * method to advance the state (required to extend ReqSvcState)
	 * @param last the last step we attempted to perform (the from state)
	 * @param next the step number to go to
	 * @param reason the success/failure code causing us to go to this event
	 * (if an error code, the error will have occurred doing the previous step)
	 * @param errMsg a text error message describing the failure, or empty
	 * @return whether the state advance worked successfully (can be true even
	 * if handling an error reason code)
	 */
	public boolean advanceState(int last, int next, int reason, String errMsg) {
		final String lbl = this.getClass().getSimpleName() + ".advanceState: ";
	/*
		m_Log.debug(lbl + "transition from " + last + " to " + next
					+ ", reason " + reason + ", emsg \"" + errMsg + "\"");
	 */

		// examine inputs
		if (next > m_FinalStep) {
			m_Log.error(lbl + "invalid next step number: " + next);
			return false;
		}
		if (next < last) {
			m_Log.error(lbl + "reverse state transition from " + next
						+ " to " + last);
			return false;
		}
		if (reason < 0) {
			m_Log.error(lbl + "invalid step advance reason: " + reason);
			return false;
		}
		m_PrevStep = last;
		m_CurrStep = next;
		m_FailureMode = reason;
		if (errMsg != null) {
			m_FailureMsg = errMsg;
		}

		// local convenience definitions
		MVO mvo = m_Parent.getMVO();
		MVOState stObj = mvo.getStateObj();
		AsyncContext rep = m_Parent.getDappReply();
		if (m_IsLead && rep == null) {
			m_Log.error(lbl + "lead without an AsyncContext, abort");
			return false;
		}
		String mvoRep = m_Parent.getMVOReplyId();
		String leadMVOId = "";
		ClientHandler cHandler = mvo.getClientHandler();

		// handle errors, based on the 'last' step completed
		boolean ret = true;
		switch (reason) {
			// request failed to parse, failed signature check etc.
			case M_InvalidReq:
				// context: M_ReqReceived --> M_ReqVerified
				if (m_PrevStep == M_ReqReceived) {
					// nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_BAD_REQUEST,
								  "parse error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not reach a committee MVO at some point
			case M_UnavailMVO:
				// context: M_OB_Generated --> M_OB_Signed
				if (m_PrevStep == M_OB_Generated) {
					// nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "validator access error: " + m_FailureMsg);
				}
				// context: M_OB_Signed --> M_Completed (after a reply error)
				else if (m_PrevStep >= M_OB_Signed && m_CurrStep == M_Completed)
				{
					// no point in trying to reply again; do nothing
					ret = false;
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not reach an Auditor
			case M_UnavailAud:
				// context: M_ReqVerified --> M_NewKeys
				if (m_PrevStep == M_ReqVerified) {
					// failed to get eNFT keys; nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "key server error: " + m_FailureMsg);
				}
				// context: M_OB_Generated --> M_OB_Signed (lead or not)
				else if (m_PrevStep == M_OB_Generated) {
					// we could not send {OperationsBlock} to an Auditor
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_GATEWAY_TIMEOUT,
								  "Auditor access error: " + m_FailureMsg);
				}
				// context: M_OB_Signed --> M_ReceiptGen
				else if (m_IsLead && m_PrevStep == M_OB_Signed) {
					// we could not get keys to generate receipts, must nack
					ret = nackDapp(rep, HttpServletResponse.SC_GATEWAY_TIMEOUT,
								"receipt key server error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			// we could not access the appropriate blockchain
			case M_UnavailABI:
				// context: M_ReqReceived --> M_ReqVerified
				if (m_PrevStep == M_ReqReceived) {
					// couldn't check vs. SC config; nack original request
					ret = nackReq(rep, mvoRep,
								  HttpServletResponse.SC_BAD_GATEWAY,
								  "blockchain access error: " + m_FailureMsg);
				}
				// context: M_Replied --> M_Deposited
				else if (m_IsLead && m_PrevStep == M_Replied) {
					// we can't nack the request because we already replied
					return advanceState(M_Replied, m_FinalStep, reason,
										"could not verify deposit on chain: "
										+ m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			case M_ProcError:
				/* Internal processing error, theoretically valid for any step.
				 * Unless we have some kind of coding error, these should not
				 * occur. To avoid the possibility of a recursive endless loop,
				 * we always bail with a nack when one of these crops up.
				 */
				String fail = "";
				int rCode = HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
				switch (m_PrevStep) {
					case M_ReqReceived:
						fail = "blockchain access problem: " + m_FailureMsg;
						rCode = HttpServletResponse.SC_BAD_REQUEST;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_ReqVerified:
						fail = "key server access problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_NewKeys:
						fail = "deposit processing problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_OB_Generated:
						fail = "MVO committee access problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_OB_Signed:
						fail = "receipt generation problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_ReceiptGen:
						fail = "reply send problem: " + m_FailureMsg;
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_Replied:
						fail = "blockchain deposit confirmation problem: "
								+ m_FailureMsg;
						// this will probably fail too of course:
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;

					case M_Deposited:
						fail = "receipt upload problem: " + m_FailureMsg;
						// this will probably fail too of course:
						ret = nackReq(rep, mvoRep, rCode, fail);
						break;
					
					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				m_Log.error(lbl + "processing error: " + fail);
				break;

			case M_ParseError:
				/* Partners can send us bogosity that didn't parse.  If they do,
				 * we have to bail because resending will probably generate the
				 * same bogosity again.  (This condition probably indicates a
				 * coding error in a partner server.)
				 */
				switch (m_PrevStep) {
					case M_ReqReceived:
					case M_ReqVerified:
					case M_NewKeys:
					case M_OB_Generated:
					case M_OB_Signed:
					case M_ReceiptGen:
					case M_Replied:
					case M_Deposited:
						m_Log.error(lbl + "parse error seen from partner, "
									+ m_FailureMsg);
						ret = nackReq(rep, mvoRep,
								HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
								"error processing partner reply, aborting: "
								+ m_FailureMsg);
						break;

					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				break;

			// partners can of course also send us error replies
			case M_GotNack:
				String nack = "";
				int nCode = HttpServletResponse.SC_BAD_GATEWAY;
				switch (m_PrevStep) {
					case M_ReqReceived:
						nack = "nack accessing blockchain config, "
								+ m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_ReqVerified:
						nack = "nack obtaining eNFT keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_OB_Generated:
						nack = "nack from committee MVO, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_OB_Signed:
						nack = "nack obtaining receipt keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					case M_Replied:
						nack = "nack accessing blockchain for deposit conf, "
								+ m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;
					
					case M_Deposited:
						nack = "nack uploading receipts, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackReq(rep, mvoRep, nCode, nack);
						break;

					// no peer replies expected in these previous states
					case M_NewKeys:
					case M_ReceiptGen:
					case M_Completed:
					default:
						// should be impossible
						m_Log.error(lbl + "unexpected reply parse error in "
									+ "state " + m_CurrStep + ", "
									+ m_FailureMsg);
						ret = inconsistentState(rep, mvoRep);
						break;
				}
				break;

			// an external store (such as for receipts) was unavailable
			case M_ExtUnavail:
				// context: M_OB_Signed --> M_ReceiptGen
				if (m_IsLead && m_PrevStep == M_OB_Signed) {
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_GATEWAY,
									"receipt storage error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep, mvoRep);
				}
				break;

			case M_NoFailure:
				// fall through below
				break;

			default:
				// should be impossible
				ret = inconsistentState(rep, mvoRep);
				break;
		}
		if (reason != M_NoFailure) return ret;

		// handle normal (non-error) cases
		ret = true;
		m_FailureMode = M_NoFailure;
		ClientRequest orgReq = m_Parent.getDappRequest();
		ClientMVOBlock clientReq = null;
		ArrayList<ClientMVOBlock.ClientPayee> payees = null;
		OperationsBlock opBlock = m_Parent.getOperationsBlock();
		SmartContractConfig scc = null;
		MVOConfig mvoConf = mvo.getConfig();
		final BigInteger one100 = new BigInteger("100");
		switch (m_CurrStep) {
			case M_ReqVerified:
				/* Here we have successfully parsed and signature-verified the
				 * request from the client (iff lead) or lead MVO (if not).
				 * Our next step is to verify the parameters in the request vs.
				 * the SC config for the indicated blockchain.  If that's all
				 * okay, we need to do basic sanity-checking.  If any problems
				 * are found, we nack the request.
				 */
				if (m_Achieved < M_ReqReceived) {
					m_Log.error(lbl + "verify requested without valid "
								+ "request received");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				// get the request as passed by originator
				if (!(orgReq instanceof ClientMVOBlock)) {
					m_FailureMsg = "client request is not a ClientMVOBlock";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}
				clientReq = (ClientMVOBlock) orgReq;
				if (!clientReq.getOpcode().equals(clientReq.M_OpDeposit)) {
					// we don't belong here
					m_FailureMsg = "client request is not a deposit";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// check versus SC config 
				scc = mvo.getSCConfig(clientReq.getChainId());
				if (scc == null) {
					// unknown chain
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " is not supported";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// confirm availability of JSON-RPC client
				if (!scc.isEnabled()) {
					// Web3j URL must be offline or not reachable
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " support is not currently available";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_UnavailABI,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// if we're not the lead, we must verify the lead's sig on OB
				if (!m_IsLead) {
					leadMVOId = m_Parent.getTagField(mvoRep, 1);
					MVOSignature leadSig
						= MVOSignature.findMVOSig(leadMVOId,
												  opBlock.getSignatures());
					if (leadSig == null) {
						m_FailureMsg = "lead MVO signature on OB not found";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// find the signingAddress for the lead MVO on this chain
					Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
					MVOConfig leadConf = (MVOConfig) mvoMap.get(leadMVOId);
					String signingAddress = "";
					BlockchainConfig bcConf
						= leadConf.getChainConfig(clientReq.getChainId());
					if (bcConf != null) {	
						signingAddress = bcConf.getSigningAddress();
					}

					// validate lead MVO's signature using signingAddress
					String opData = opBlock.buildSignedData();
					String sigAddress
						= EncodingUtils.getDataSignerAddress(opData,
														leadSig.m_Signature);
					if (sigAddress.isEmpty()
						|| !sigAddress.equalsIgnoreCase(signingAddress))
					{
						m_FailureMsg = "lead MVO signature on OB invalid";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// confirm sender address is valid
				if (!WalletUtils.isValidAddress(clientReq.getSender())) {
					m_FailureMsg = "sender address (" + clientReq.getSender()
									+ ") appears invalid";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// ensure that we don't have more payees than the SC allows
				payees = clientReq.getPayees();
				if (payees.size() > ClientMVOBlock.M_ARRAY_MAX) {
					m_FailureMsg = "too many output eNFTs (" + payees.size()
									+ "), max is " + ClientMVOBlock.M_ARRAY_MAX;
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// ensure that all the client payee addresses appear valid
				BigInteger amt = clientReq.getAmount();
				boolean badAddress = false;
				BigInteger outputSum = new BigInteger("0");
				for (ClientMVOBlock.ClientPayee payee : payees) {
					if (!payee.m_Address.isEmpty()
						&& !WalletUtils.isValidAddress(payee.m_Address))
					{
						badAddress = true;
						m_Log.error(lbl + "payee address "
									+ payee.m_Address + " appears invalid");
					}

					/* Tally output amounts.  Note that the value of the m_Units
					 * field dictates whether amount is absolute or a percentage
					 * of the total.
					 */
					if (payee.m_Units.isEmpty()) {
						outputSum = outputSum.add(payee.m_OutputAmount);
					}
					else if (payee.m_Units.equals("%")) {
						// compute the percentage of the total
						BigInteger incr = amt.multiply(payee.m_OutputAmount);
						incr = incr.divide(one100);
						outputSum = outputSum.add(incr);
					}
					else {
						m_Log.error(lbl + "bad units on output eNFT "
									+ payee.m_Payee + ", ignoring amount");
					}
				}
				if (badAddress) {
					m_FailureMsg = "one or more payee addresses are invalid";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// ensure that deposit amount offered matches outputs
				if (outputSum.compareTo(amt) != 0) {
					m_FailureMsg = "output eNFT amounts do not match total";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg
								+ " (" + outputSum + " versus " + amt + ")");
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// worked!
				recordSuccess(M_ReqVerified);

				// advance state to next step, which is fetching/creating keys
				ret = advanceState(m_CurrStep, M_NewKeys, M_NoFailure, "");
				break;

			case M_NewKeys:
				/* Here we have successfully validated the request versus the
				 * relevant SC config, plus internal consistency, etc.
				 * First, we'll pick a random Auditor from our list.  If we
				 * don't have a connection to it, we'll open one (and do the
				 * AUTHREQ:AUTHREP dance, in which Auditor does REQ).
				 * If we're the lead MVO, we'll need to do a create request.
				 * Otherwise we do a fetch for the keys, as the lead will have
				 * created them already.  In both cases we do this by sending a
				 * MVOKeyBlock to the Auditor and waiting for the reply.
				 * Normally this is all asynchronous, but temporarily we'll be
				 * doing it synchronously using AuditorHandler.handleKeyBlock().
				 * If any problems occur, we must nack the request.
				 */
				if (!m_ExpectingKeyServerReply) {
					String randAud = mvo.getRandomAuditor(null);
					if (randAud.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get keys failure, no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// build the key block to send to the Auditor
					MVOKeyBlock keyBlock = new MVOKeyBlock(m_Log);
					keyBlock.setMapping(keyBlock.M_MapENFT);
					if (m_IsLead) {
						keyBlock.setOpcode(keyBlock.M_OpNew);
					}
					else {
						keyBlock.setOpcode(keyBlock.M_OpGet);
					}

					/* If we're lead, we must assign a new unique eNFT ID for 
					 * each output (checking to be sure no such ID currently
					 * exists), and compute: sha3(chainId+ID+address).  Add
					 * these to the map and request them from the Auditor.
					 *
					 * If we're not lead, we'll find the list of keys in the
					 * passed OperationsBlock from our committee lead, in
					 * (ArrayList):
					 * m_Parent.getOperationsBlock().getPayees().m_ID.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					// must supply user's original request
					keyBlock.setClientBlockJson(
											clientReq.getDecryptedPayload());
					payees = clientReq.getPayees();
					if (m_IsLead) {
						/* Loop through payees and copy data to
						 * {OperationsBlock}, obtaining a random ID from the
						 * state object for each eNFT that will be required.
						 * Confirm that the key doesn't already exist (despite
						 * the low odds of a random collision generating
						 * 256-bit values!), and add it to the {MVOKeyBlock}
						 * we'll send to the auditor.
						 */
						opBlock.setOpcode(clientReq.M_OpDeposit);
						// this must be a hex value:
						opBlock.setAmount(clientReq.getAmount().toString(16));
						opBlock.setAsset(Keys.toChecksumAddress(
														clientReq.getAsset()));
						scc = mvo.getSCConfig(clientReq.getChainId());
						for (ClientMVOBlock.ClientPayee payee : payees) {
							// ignore any improper negative amounts
							if (payee.m_OutputAmount.compareTo(BigInteger.ZERO)
								< 0)
							{
								m_Log.error(lbl + "ignoring illegal amount of "
											+ payee.m_OutputAmount
											+ " on eNFT for " + payee.m_Payee);
								// NB: this should cause a peer OB mismatch
								continue;
							}

							// make a payee for the OB
							OperationsBlock.ClientPayee obPayee
								= opBlock.new ClientPayee();
							// strip off "^payee" to leave just the sequence #
							obPayee.m_Payee = payee.m_Payee.substring(5);
							obPayee.m_Address = new String(payee.m_Address);
							/* NB: obPayee.m_OutputAmount is not emitted for a
							 * deposit
							 */
							if (!payee.m_Units.equals("%")) {
								// absolute amount
								obPayee.m_OutputAmount = new BigInteger(
											payee.m_OutputAmount.toString());
							}
							else {
								// interpret as percentage of total amount
								BigInteger incr
									= clientReq.getAmount().multiply(
														payee.m_OutputAmount);
								obPayee.m_OutputAmount = incr.divide(one100);
							}

							// pick an ID for the eNFT we'll create (unique)
							String newID = getUniqueNFTId(scc.getChainId());
							obPayee.m_ID = newID;

							// capture the randomizer value, if any
							if (!payee.m_Rand.isEmpty()) {
								obPayee.m_Rand = new String(payee.m_Rand);
							}

							// compute and set the details hash
							obPayee.m_DetailsHash
								= EncodingUtils.buildDetailsHash(
															obPayee.m_Address,
															newID,
															opBlock.getAsset(),
											obPayee.m_OutputAmount.toString(16),
															obPayee.m_Rand);
							opBlock.addPayee(obPayee);

							// compute chainId+ID+address and get key
							ArrayList<String>
								hashComps = new ArrayList<String>(3);
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(newID);
							hashComps.add(obPayee.m_Address.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							keyBlock.addHashToList(keyHash);
						}
					}
					else {	// not lead
						/* NB: in this case we'll verify the passed OB data, but
						 * we'll do it in the M_OB_Generated step, not here.
						 */
						boolean noId = false;
						// our passed OB should contain a complete payee list
						if (opBlock.getPayees().isEmpty()) {
							noId = true;
						}
						for (OperationsBlock.ClientPayee payee
							 : opBlock.getPayees())
						{
							if (payee.m_ID.isEmpty()) {
								noId = true;
								continue;
							}

							// compute chainId+ID+address and get key
							ArrayList<String> hashComps
								= new ArrayList<String>(3);
							hashComps.add(
										Long.toString(clientReq.getChainId()));
							hashComps.add(payee.m_ID);
							hashComps.add(payee.m_Address.toLowerCase());
							String keyIdx = String.join(M_JoinChar, hashComps);
							String keyHash = EncodingUtils.sha3(keyIdx);
							keyBlock.addHashToList(keyHash);
						}
						if (noId) {
							m_FailureMsg = "missing passed payee ID list in OB";
							m_Log.error(lbl + "no IDs in OB payees passed");
							advanceState(m_PrevStep, m_CurrStep, M_ParseError,
										 m_FailureMsg);
							ret = false;
							break;
						}
					}

					// sign the MVOKeyBlock
					MVOSignature ourSig = new MVOSignature();
					ourSig.m_Signer = mvo.getMVOId();
					PrivateKey sigKey = mvoConf.getCommPrivKey();
					String signedData = keyBlock.buildSignedData();
					boolean sigOk = true;
					if (keyBlock.getErrCode() != 0) {
						sigOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKey, signedData);
						if (sig == null) {
							sigOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							ourSig.m_Signature = sig;
						}
					}
					if (!sigOk) {
						m_FailureMsg = "could not build key block for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					keyBlock.setSignature(ourSig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud, keyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud);
							randAud = mvo.getRandomAuditor(excludeAIds);
							if (randAud.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg
							= "error fetching output eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// handling key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String signedAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature audSig
						= m_KeyResponse.getSignature();
					boolean sigVerf = true;
					PublicKey audKey = mvoConf.getPeerPubkey(audSig.m_Signer);
					if (audKey == null) {
						sigVerf = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ audSig.m_Signer
									+ ", cannot verify sig on "
									+ "returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(audKey,
														   signedAudData,
														   audSig.m_Signature))
						{
							sigVerf = false;
							m_Log.error(lbl + "sig verify failed for Aud "
										+ audSig.m_Signer
										+ ", cannot verify sig "
										+ "on returned AuditorKeyBlock");
						}
					}
					if (!sigVerf) {
						m_FailureMsg = "error looking up eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + audSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up eNFT AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all keys, move to next state
				recordSuccess(M_NewKeys);
				ret = advanceState(m_CurrStep, M_OB_Generated, M_NoFailure, "");
				break;

			case M_OB_Generated:
				/* Here we have successfully obtained from an Auditor the keys
				 * we need for the involved eNFTs, and stored them in our
				 * m_KeyResponse record.  These keys must now be used in one of
				 * two ways:
				 * 1) If we're the lead MVO, we must generate and sign new eNFTs
				 * 	  using each key, and store them in m_Parent.m_LeadMVOBlock.
				 * 	  We then sign our generated OB, complete this step,
				 * 	  and transition to M_OB_SIGNED to deal with the committee.
				 * 2) If we're not the lead, we must decrypt the existing eNFTs
				 * 	  in the m_LeadMVOBlock, and confirm that their details do
				 * 	  match the output data in the {OperationsBlock}.  If they
				 * 	  don't, we nack.  If they do, we move to M_OB_SIGNED to
				 * 	  sign the original passed OB and return it to the lead MVO.
				 */
				// confirm we do indeed have the keys needed for OB processing
				ArrayList<AuditorKeyBlock.KeySpec> keyList
					= m_KeyResponse.getKeys();
				if (m_KeyResponse == null || keyList.isEmpty()) {
					m_Log.error(lbl + "no eNFT keys found, abort");
					m_FailureMsg = "no eNFT keys found, cannot process OB";
					advanceState(m_PrevStep, m_CurrStep, M_ParseError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				clientReq = (ClientMVOBlock) orgReq;
				ArrayList<OperationsBlock.ClientPayee> outputs
					= opBlock.getPayees();
				scc = mvo.getSCConfig(clientReq.getChainId());
				payees = clientReq.getPayees();

				// check for lead
				if (m_IsLead) {
					// first get our signing key for this chain
					BlockchainConfig bConfig
						= mvoConf.getChainConfig(scc.getChainId());
					if (bConfig == null) {
						// shouldn't occur!
						m_Log.error(lbl + "no MVO config for chain "
									+ scc.getChainName() + ", cannot "
									+ "sign eNFTs for it");
						m_FailureMsg = "missing MVO signature key";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					BigInteger bcSigKey = bConfig.getSigningKey();

					// loop through all the outputs and create eNFT objects
					boolean gotNFTerr = false;
					for (OperationsBlock.ClientPayee outp : outputs) {
						// make outer NFT using defaults
						NFTmetadata nft = new NFTmetadata(m_Log);

						// generate the enshrouded properties
						eNFTmetadata eNft = new eNFTmetadata(m_Log);
						eNft.setID(outp.m_ID);
						eNft.setOwner(outp.m_Address);
						eNft.setAsset(opBlock.getAsset());
						eNft.setAmount(outp.m_OutputAmount.toString());
						eNft.setSigner(mvo.getMVOId());
						eNft.setRand(outp.m_Rand);

						// copy the memo line from the client request, if any
						for (ClientMVOBlock.ClientPayee payee : payees) {
							// match on the payee sequence number
							if (payee.m_Payee.endsWith(outp.m_Payee)) {
								if (!payee.m_Memo.isEmpty()) {
									// check for memo line too long
									if (payee.m_Memo.length()
										> eNFTmetadata.M_MEMO_MAX)
									{
										m_FailureMsg = "memo line too long, "
											+ "max " + eNFTmetadata.M_MEMO_MAX
											+ " characters";
										gotNFTerr = true;
									}
									else {
										eNft.setMemo(payee.m_Memo);
									}
								}
								break;
							}
						}

						// TBD: if required, deal with expiration/growth/cost
					
						// attach encrypted properties to NFT
						nft.setProperties(eNft);

						// generate our signature on the eNFT
						String eNftSig = eNft.calcSignature(bcSigKey);
						if (eNftSig == null) {
							m_Log.error(lbl + "unable to sign new eNFT");
							m_FailureMsg = "error generating eNFT signature";
							gotNFTerr = true;
							break;
						}
						eNft.setSignature(eNftSig);

						/* encrypt the entire thing including the signature,
						 * using the AES key corresponding to the hash:
						 * chainId+ID+address
						 */
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(scc.getChainId()));
						hashComps.add(eNft.getID());
						hashComps.add(eNft.getOwner().toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						String useKey = null;
						AuditorKeyBlock.KeySpec keySpec
							= m_KeyResponse.getKeyForHash(keyHash);
						if (keySpec != null) {
							useKey = keySpec.m_KeyData;
						}
						if (useKey == null) {
							m_Log.error(lbl + "no AES key found for hash "
										+ keyHash + ", cannot "
										+ "encrypt eNFTs with it");
							m_FailureMsg = "missing eNFT encryption key";
							gotNFTerr = true;
							break;
						}

						// make key from this key data
						Base64.Decoder b64d = Base64.getUrlDecoder();
						byte[] keyData = b64d.decode(useKey);
						SecretKey secretKey = new SecretKeySpec(keyData, "AES");

						// encrypt the entire {enshrouded} value, including sig
						StringBuilder enshroudedData = new StringBuilder(1024);
						eNft.addJSON(enshroudedData);
						String encData
							= EncodingUtils.encWithAES(secretKey,
													  enshroudedData.toString(),
													   stObj.getRNG());
						if (encData == null) {
							m_FailureMsg = "eNFT encryption failure";
							m_Log.error(lbl + "could not encrypt eNFT ID "
										+ eNft.getID()
										+ " with AES key, abort");
							gotNFTerr = true;
							break;
						}
						eNft.setEnshrouded(encData);
						eNft.setEncrypted(true);

						// record full NFT metadata (including enc {enshrouded})
						outp.m_Metadata = nft;
						StringBuilder encMetadata = new StringBuilder(2048);
						nft.addJSON(encMetadata);
						outp.m_EncMetadata = encMetadata.toString();

						// if we're doing local blockchain API, save the file
						//TEMPCODE (local dev test only)
						BlockchainAPI web3Api = mvo.getWeb3(scc.getChainId());
						if (web3Api instanceof LocalBlockchainAPI) {
							// save the file
							LocalBlockchainAPI lbApi
								= (LocalBlockchainAPI) web3Api;
							if (!lbApi.createENFTfile(scc.getChainId(),
													  eNft.getID(),
													  eNft.getOwner(),
													  outp.m_EncMetadata))
							{
								m_FailureMsg = "could not create eNFT file for "
												+ "ID " + eNft.getID();
								m_Log.error(lbl + m_FailureMsg);
								gotNFTerr = true;
								break;
							}
						}
						//TEMPCODE
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// this completes this step; record success and move on
					recordSuccess(M_OB_Generated);
					ret = advanceState(m_CurrStep, M_OB_Signed, M_NoFailure,
										"");
				}
				else {
					/* Here we do not generate the OB, because it was passed to
					 * us completed and signed.  Instead, we only need to walk
					 * through it and verify that what the lead MVO constructed
					 * is copacetic with the dApp's original request (also
					 * forwarded to us, and its signature verified already).
					 * To do so we must decrypt the existing eNFTs shown in the
				 	 * m_LeadMVOBlock, and confirm that their details do match
				 	 * the output data in the {ClientMVOBlock}.  If they don't,
				 	 * we must nack.  If they do, we move to M_OB_SIGNED to
				 	 * sign the original passed OB and return it to lead MVO.
					 */
					boolean gotNFTerr = false;
					ArrayList<ClientMVOBlock.ClientPayee> orgOutputs
						= clientReq.getPayees();

					// make certain every original output is found in the OB
					for (ClientMVOBlock.ClientPayee orgOut : orgOutputs) {
						boolean present = false;
						for (OperationsBlock.ClientPayee output : outputs) {
							// strip off "payee"
							String ptag = orgOut.m_Payee.substring(5);
							if (ptag.equals(output.m_Payee)) {
								present = true;
								break;
							}
						}
						if (!present) {
							m_Log.error(lbl + "CRIT - client payee "
										+ orgOut.m_Payee + " not found in OB");
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// loop through all the outputs and validate eNFT objects
					for (OperationsBlock.ClientPayee outp : outputs) {
						// get the original client request payee that matches
						ClientMVOBlock.ClientPayee orgOutput = null;
						for (ClientMVOBlock.ClientPayee orgPayee : orgOutputs) {
							if (orgPayee.m_Payee.equals("payee"+outp.m_Payee)) {
								orgOutput = orgPayee;
								break;
							}
						}
						if (orgOutput == null) {
							// we don't have an original payee like this
							m_Log.error(lbl + "CRIT - OB payee "
										+ outp.m_Payee + " not found in "
										+ "original client request!");
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// parse outer NFT with {enshrouded} data encrypted 
						NFTmetadata nft = new NFTmetadata(m_Log);
						if (!nft.buildFromJSON(outp.m_EncMetadata)) {
							m_Log.error(lbl + "could not parse eNFT in "
										+ "OB, bad metadata");
							gotNFTerr = true;
							break;
						}

						// obtain and decrypt the {enshrouded} data
						eNFTmetadata metadata = nft.getProperties();
						if (metadata == null || !metadata.isEncrypted()) {
							m_Log.error(lbl + "eNFT metadata not found "
										+ "or not encrypted");
							gotNFTerr = true;
							break;
						}

						/* decrypt the entire thing including the signature,
						 * using the AES key corresponding to the hash:
						 * chainId+ID+address
						 */
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(scc.getChainId()));
						hashComps.add(outp.m_ID);
						hashComps.add(outp.m_Address.toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						String useKey = null;
						AuditorKeyBlock.KeySpec aesSpec
							= m_KeyResponse.getKeyForHash(keyHash);
						if (aesSpec != null) {
							useKey = aesSpec.m_KeyData;
						}
						if (useKey == null) {
							m_Log.error(lbl + "no AES key found for hash "
										+ keyHash + ", cannot "
										+ "decrypt eNFTs with it");
							m_FailureMsg = "missing output eNFT decryption key";
							gotNFTerr = true;
							break;
						}

						// make key from this key data
						Base64.Decoder b64d = Base64.getUrlDecoder();
						byte[] keyData = b64d.decode(useKey);
						SecretKey secretKey = new SecretKeySpec(keyData, "AES");
						String encMeta = metadata.getEnshrouded();
						String decMeta = EncodingUtils.decWithAES(secretKey,
																  encMeta);
						if (decMeta == null) {
							m_Log.error(lbl + "error decrypting eNFT "
										+ "metadata for " + outp.m_ID);
							m_FailureMsg = "eNFT metdata decryption error";
							gotNFTerr = true;
							break;
						}
						eNFTmetadata eNFTdecMetadata = new eNFTmetadata(m_Log);
						String parseTxt = "{" + decMeta + "}";
						Object parseObj = null;
						try {
							parseObj = JSON.parse(parseTxt);
						}
						catch (IllegalStateException ise) { /* log below */ }
						if (!(parseObj instanceof Map)) {
							m_Log.error(lbl + "error parsing JSON: \""
										+ parseTxt + "\"");
							m_FailureMsg = "eNFT metdata parse error";
							gotNFTerr = true;
							break;
						}
						Map parseMap = (Map) parseObj;
						if (!eNFTdecMetadata.buildFromMap(parseMap)) {
							m_Log.error(lbl + "error parsing eNFT map");
							m_FailureMsg = "eNFT metdata parse error";
							gotNFTerr = true;
							break;
						}

						// get the signer and check their signature on the eNFT
						String signer = eNFTdecMetadata.getSigner();
						// this should equal lead MVO
						String lead = m_Parent.getTagField(mvoRep, 1);
						if (!signer.equals(lead)) {
							m_Log.error(lbl + "sig on OB is that of "
										+ signer + ", not lead MVO " + lead);
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						// get their signing key for this blockchain
						MVOConfig leadConf
							= (MVOConfig) scc.getMVOMap().get(signer);
						if (leadConf == null) {
							m_Log.error(lbl + "cannot find config for "
										+ "lead MVO signer, " + signer);
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						BlockchainConfig leadChConf
							= leadConf.getChainConfig(clientReq.getChainId());
						if (leadChConf == null) {
							m_Log.error(lbl + "cannot find chain config"
										+ " for eNFT signer " + signer
									+ " on chainId " + clientReq.getChainId());
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}
						String leadSigAddr = leadChConf.getSigningAddress();

						// verify sig of leadSigAddr on eNFT
						String sigAddr = eNFTdecMetadata.getSigAddr();
						if (!sigAddr.equalsIgnoreCase(leadSigAddr)) {
							m_Log.error(lbl + "signature does not match on ID "
										+ eNFTdecMetadata.getID());
							m_FailureMsg = "eNFT signature validation failure";
							gotNFTerr = true;
							break;
						}

						/* We now have a decrypted version of the eNFT data.
						 * Compare each relevant bit against the output data,
						 * to ensure the lead MVO pulled no shenaningans.  We
						 * need to check against BOTH the values in the OB, and
						 * the original outputs specified in the signed client
						 * request which was forwarded.
						 */
						// check ID value
						if (!eNFTdecMetadata.getID().equals(outp.m_ID)) {
							// falsifying the ID of the eNFT is very serious...
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "ID = " + eNFTdecMetadata.getID()
										+ " while OB has " + outp.m_ID);
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// check owner
						if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(
																outp.m_Address))
						{
							// falsifying the payee is also very serious...
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "payee " + eNFTdecMetadata.getOwner()
										+ " while OB has " + outp.m_Address);
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}
						if (!outp.m_Address.equalsIgnoreCase(
														orgOutput.m_Address))
						{
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "payee " + eNFTdecMetadata.getOwner()
										+ " while org client request has "
										+ orgOutput.m_Address);
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// check asset
						if (!eNFTdecMetadata.getAsset().equalsIgnoreCase(
															opBlock.getAsset()))
						{
							// changing asset is also very serious...
							m_Log.error(lbl + "CRIT - output eNFT with asset "
										+ eNFTdecMetadata.getAsset()
									+ " while OB has " + opBlock.getAsset());
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}
						if (!opBlock.getAsset().equalsIgnoreCase(
													orgOutput.m_OutputAsset))
						{
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "asset " + eNFTdecMetadata.getAsset()
										+ " while org client request has "
										+ orgOutput.m_OutputAsset);
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// check randomizer value against original dApp request
						// (not passed in OB on a deposit)
						String eRand = eNFTdecMetadata.getRand();
						if (!eRand.equals(orgOutput.m_Rand)) {
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "randomizer " + eRand
										+ " while org client request has "
										+ orgOutput.m_Rand);
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						/* NB: we can't check eNFTdecMetadata.getAmount() versus
						 * 	   the outp.m_OutputAmount, because the latter isn't
						 * 	   included in the JSON output when the OB is passed
						 * 	   to us, in a deposit context.  (This field is
						 * 	   used only for withdrawals.)
						 */
						// determine amount from org request based on units
						BigInteger orgAmt = orgOutput.m_OutputAmount;
						if (orgOutput.m_Units.equals("%")) {
							// interpret as a percentage of total
							BigInteger incr
								= opBlock.getAmount().multiply(orgAmt);
							orgAmt = incr.divide(one100);
						}
						if (!orgAmt.equals(eNFTdecMetadata.getAmount())) {
							m_Log.error(lbl + "CRIT - output eNFT with amount "
										+ eNFTdecMetadata.getAmount().toString()
										+ " while org client request has "
										+ orgAmt.toString());
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// make sure the details hash computes correctly
						String hashStr = EncodingUtils.buildDetailsHash(
															outp.m_Address,
															outp.m_ID,
															opBlock.getAsset(),
															orgAmt.toString(16),
															orgOutput.m_Rand);
						// check it matches
						if (!outp.m_DetailsHash.equals(hashStr)) {
							m_Log.error(lbl + "CRIT - output eNFT with "
										+ "ID " + eNFTdecMetadata.getID()
										+ ", details hash doesn't match OB");
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// on a deposit, generation should be 1
						if (eNFTdecMetadata.getGeneration() != 1) {
							m_Log.error(lbl + "illegal generation, ID "
										+ eNFTdecMetadata.getID() + ": "
										+ eNFTdecMetadata.getGeneration());
							m_FailureMsg = "eNFT data consistency error";
							gotNFTerr = true;
							break;
						}

						// verify the memo line, if present, has not changed
						String memo = orgOutput.m_Memo;
						if (!memo.isEmpty()) {
							if (!eNFTdecMetadata.getMemo().equals(memo)) {
								m_Log.error(lbl + "memo on eNFT, ID "
											+ eNFTdecMetadata.getID()
											+ " doesn't match org client req");
								m_FailureMsg = "eNFT data consistency error";
								gotNFTerr = true;
								break;
							}
						}

						// TBD: if required, deal with expiration/growth/cost
					}
					if (gotNFTerr) {
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// this completes this step; record success and move on
					recordSuccess(M_OB_Generated);
					ret = advanceState(m_CurrStep, M_OB_Signed, M_NoFailure,
										"");
				}
				break;

			case M_OB_Signed:
				/* Here we have generated or verified the {OperationsBlock}.
				 * If we're lead, we must now pick a MVO committee, and set up
				 * to forward it to each member.  We'll complete this step only
				 * once all signed replies are seen.  We also send a CC to
				 * all known and connected Auditors as a MVOAuditorBlock.
				 *
				 * If we're not lead, we sign the original OB with our key and
				 * send it back, which completes this step for us.  We will
				 * likewise broadcast our signed OB to all Auditors, included in
				 * a MVOAuditorBlock message.
				 *
				 * As a pre-step, because the entire OB cannot be passed to the
				 * SC for verification (too large to parse), we must generate
				 * the OperationsBlock.ArgumentsHash for the argument data
				 * which will actually get passed to the SC by the dApp.  The
				 * lead MVO will set this in the OB; committee members merely
				 * verify it.  All MVOs will sign this hash data in addition to
				 * the complete OB (which itself includes the args hash value).
				 */
				clientReq = (ClientMVOBlock) orgReq;
				scc = mvo.getSCConfig(clientReq.getChainId());

				// first get our signing key for this chain
				BlockchainConfig bConfig
					= mvoConf.getChainConfig(scc.getChainId());
				if (bConfig == null) {
					// shouldn't occur!
					m_Log.error(lbl + "no MVO config for chain "
								+ scc.getChainName() + ", cannot "
								+ "sign OBs for it");
					m_FailureMsg = "missing MVO signature key";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				BigInteger bcSigKey = bConfig.getSigningKey();

				// sign the ArgumentsBlock passed by the dApp to the SC's API
				OperationsBlock.ArgumentsHash argsHash = opBlock.getArgsBlock();
				// NB: this arguments data is opcode-dependent
				String argData = argsHash.buildArgsData();
				if (argData == null) {
					m_Log.error(lbl + "could not build OB args hash data");
					m_FailureMsg = "could not build operations arguments hash";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				argsHash.m_ArgumentsData = argData;
				if (m_IsLead) {
					String hashValue = argsHash.argsHash();
					argsHash.m_ArgsHash = hashValue;
				}
				else {
					// double-check this is the data lead MVO actually signed
					if (!argsHash.m_ArgsHash.equals(argsHash.argsHash())) {
						m_Log.error(lbl + "could not verify OB args hash");
						m_FailureMsg
							= "could not verify operations arguments hash";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// validate the hash sig of lead MVO
					String leadArgSig = "";
					leadMVOId = m_Parent.getTagField(mvoRep, 1);
					MVOSignature leadSig = MVOSignature.findMVOSig(leadMVOId,
													opBlock.getSignatures());
					if (leadSig != null) {
						leadArgSig = leadSig.m_ArgsSig;
					}
					if (leadArgSig.isEmpty()) {
						m_FailureMsg = "no OB args signature for lead MVO";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// find the signingAddress for the lead MVO on this chain
					Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
					MVOConfig leadConf = (MVOConfig) mvoMap.get(leadMVOId);
					String signingAddress = "";
					BlockchainConfig bcConf
						= leadConf.getChainConfig(clientReq.getChainId());
					if (bcConf != null) {	
						signingAddress = bcConf.getSigningAddress();
					}

					// verify this was the address which signed
					String sigAddress = EncodingUtils.getHashSignerAddress(
															argsHash.m_ArgsHash,
															leadArgSig);
					if (sigAddress.isEmpty()
						|| !sigAddress.equalsIgnoreCase(signingAddress))
					{
						m_FailureMsg = "lead MVO OB args signature failure";
						m_Log.error(lbl + m_FailureMsg);
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// produce our signature on the arguments hash
				MVOSignature obSig = new MVOSignature();
				obSig.m_Signer = mvo.getMVOId();
				String digestSig = EncodingUtils.signHash(argsHash.m_ArgsHash,
														  bcSigKey);
				if (digestSig == null || digestSig.length() != 130) {
					m_FailureMsg = "error signing OB args hash";
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				obSig.m_ArgsSig = digestSig;

				// everything was okay, so sign the whole OB (lead or not)
				String hashableData = opBlock.buildSignedData();
				if (hashableData == null || hashableData.isEmpty()) {
					m_Log.error(lbl + "could not build OB signed data");
					m_FailureMsg = "could not sign operations block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				String ourSig = EncodingUtils.signData(hashableData, bcSigKey);
				if (ourSig == null) {
					m_Log.error(lbl + "could not sign OB data");
					m_FailureMsg = "could not sign operations block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				obSig.m_Signature = ourSig;
				// NB: if we're lead, this will be the first sig
				opBlock.addSignature(obSig);

				// build the MVOAuditorBlock message corresponding to our OB
				AuditorBlock audBlk = new AuditorBlock(m_Log);
				audBlk.setOpcode(clientReq.M_OpDeposit);
				audBlk.setSender(clientReq.getSender());
				audBlk.setAsset(opBlock.getAsset());
				audBlk.setAmount(opBlock.getAmount().toString());
				ArrayList<OperationsBlock.ClientPayee> obOutputs
						= opBlock.getPayees();

				// add all the outputs
				boolean gotNFTerr = false;
				for (OperationsBlock.ClientPayee outp : obOutputs) {
					AuditorBlock.ClientPayee audPayee
						= audBlk.new ClientPayee();
					audPayee.m_Payee = outp.m_Payee;
					audPayee.m_Address = outp.m_Address;
					// asset always matches deposit asset type
					audPayee.m_OutputAsset = audBlk.getAsset();
					audPayee.m_ID = outp.m_ID;
					audPayee.m_DetailsHash = outp.m_DetailsHash;
					audPayee.m_EncMetadata = outp.m_EncMetadata;

					// the eNFT in an AuditorBlock must be constructed in clear
					audPayee.m_Metadata = new NFTmetadata(m_Log);
					// parse outer NFT with {enshrouded} data encrypted 
					if (!audPayee.m_Metadata.buildFromJSON(
														audPayee.m_EncMetadata))
					{
						m_Log.error(lbl + "could not parse eNFT in OB for "
									+ "payee " + audPayee.m_Payee
									+ ", bad metadata");
						gotNFTerr = true;
						break;
					}

					// to get amount, decrypt the {enshrouded} data
					eNFTmetadata realNFT = audPayee.m_Metadata.getProperties();
					if (realNFT == null || !realNFT.isEncrypted()) {
						m_Log.error(lbl + "eNFT metadata not found or "
									+ "not encrypted, payee " + outp.m_Payee);
						gotNFTerr = true;
						break;
					}
					eNFTmetadata eNft = new eNFTmetadata(m_Log);

					/* decrypt the entire thing including the signature,
					 * using the AES key corresponding to the hash:
					 * chainId+ID+address
					 */
					ArrayList<String> hashComps = new ArrayList<String>(3);
					hashComps.add(Long.toString(scc.getChainId()));
					hashComps.add(outp.m_ID);
					hashComps.add(outp.m_Address.toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashComps);
					String keyHash = EncodingUtils.sha3(keyIdx);
					String useKey = null;
					AuditorKeyBlock.KeySpec aesSpec
						= m_KeyResponse.getKeyForHash(keyHash);
					if (aesSpec != null) {
						useKey = aesSpec.m_KeyData;
					}
					if (useKey == null) {
						m_Log.error(lbl + "no AES key found for hash "
									+ keyHash + ", cannot "
									+ "decrypt eNFT with it");
						m_FailureMsg = "missing output eNFT decryption key";
						gotNFTerr = true;
						break;
					}

					// make key from this key data
					Base64.Decoder b64d = Base64.getUrlDecoder();
					byte[] keyData = b64d.decode(useKey);
					SecretKey secretKey = new SecretKeySpec(keyData, "AES");
					String encMeta = realNFT.getEnshrouded();
					String decMeta = EncodingUtils.decWithAES(secretKey,
															  encMeta);
					if (decMeta == null) {
						m_Log.error(lbl + "error decrypting eNFT "
									+ "metadata for " + outp.m_ID);
						m_FailureMsg = "eNFT metdata decryption error";
						gotNFTerr = true;
						break;
					}
					String parseTxt = "{" + decMeta + "}";
					Object parseObj = null;
					try {
						parseObj = JSON.parse(parseTxt);
					}
					catch (IllegalStateException ise) { /* log below */ }
					if (!(parseObj instanceof Map)) {
						m_Log.error(lbl + "error parsing JSON: \""
									+ parseTxt + "\"");
						m_FailureMsg = "eNFT metdata parse error";
						gotNFTerr = true;
						break;
					}
					Map parseMap = (Map) parseObj;
					if (!eNft.buildFromMap(parseMap)) {
						m_Log.error(lbl + "error parsing eNFT map");
						m_FailureMsg = "eNFT metdata parse error";
						gotNFTerr = true;
						break;
					}
					// NB: all fields of eNft should now be set properly
					audPayee.m_OutputAmount = eNft.getAmount();
					audPayee.m_Rand = eNft.getRand();

					// recompute the details hash and compare as a check
					String hashStr = EncodingUtils.buildDetailsHash(
														audPayee.m_Address,
														audPayee.m_ID,
														audPayee.m_OutputAsset,
										audPayee.m_OutputAmount.toString(16),
														audPayee.m_Rand);
					if (!hashStr.equals(audPayee.m_DetailsHash)) {
						m_Log.error(lbl + "Auditor block payee "
									+ audPayee.m_Payee + " hash did not match");
						m_FailureMsg = "hash error building AuditorBlock";
						gotNFTerr = true;
						break;
					}
					audBlk.addPayee(audPayee);
				}
				if (gotNFTerr) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// construct the AB's ArgumentsHash and sign it
				AuditorBlock.ArgumentsHash abArgsHash = audBlk.getArgsBlock();
				// NB: this arguments data is opcode-dependent
				String abArgData = abArgsHash.buildArgsData();
				if (argData == null) {
					m_Log.error(lbl + "could not build AB args hash data");
					m_FailureMsg = "could not build Auditor arguments hash";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				abArgsHash.m_ArgumentsData = abArgData;
				abArgsHash.m_ArgsHash = abArgsHash.argsHash();

				// produce our signature on the arguments hash
				MVOSignature audBlSig = new MVOSignature();
				audBlSig.m_Signer = mvo.getMVOId();
				String abDigestSig
					= EncodingUtils.signHash(abArgsHash.m_ArgsHash, bcSigKey);
				if (digestSig == null || digestSig.length() != 130) {
					m_FailureMsg = "error signing AB args hash";
					m_Log.error(lbl + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				audBlSig.m_ArgsSig = abDigestSig;

				// sign the entire AuditorBlock with bcSigKey
				String audHashData = audBlk.buildSignedData();
				if (audHashData == null || audHashData.isEmpty()) {
					m_Log.error(lbl
								+ "could not build AuditorBlock signed data");
					m_FailureMsg = "could not sign Auditor block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				String aBlkSig = EncodingUtils.signData(audHashData, bcSigKey);
				if (aBlkSig == null) {
					m_Log.error(lbl + "could not sign AB data");
					m_FailureMsg = "could not sign Auditor block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				audBlSig.m_Signature = aBlkSig;
				audBlk.addSignature(audBlSig);

				// build the MVOAuditorBlock wrapper
				MVOAuditorBlock mvoAud = new MVOAuditorBlock(m_Log);
				mvoAud.setMVOBlock(audBlk);
				mvoAud.setClientBlock(clientReq);
				StringBuilder audJson = new StringBuilder(2048);
				audBlk.addJSON(audJson);
				mvoAud.setMVOBlockJson(audJson.toString());
				// append original client deposit request text
				String clientJSON = clientReq.getDecryptedPayload();
				mvoAud.setClientBlockJson(clientJSON);

				ArrayList<MVOConfig> mvoCommittee = null;
				if (m_IsLead) {
					// select the MVO committee based on stakings etc.
					mvoCommittee
						= m_Parent.selectMVOCommittee(clientReq.getChainId());
					if (mvoCommittee == null
						|| mvoCommittee.size() < scc.getNumSigs()-1)
					{
						m_Log.error(lbl + "could not form committee of "
									+ (scc.getNumSigs()-1) + " MVOs for chain "
									+ scc.getChainName());
						m_FailureMsg = "could not form MVO committee";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}

					/* Blast the auditor report to all known auditors.  At least
					 * one must succeed, or we cannot proceed with the request.
					 */
					if (!m_Parent.broadcastToAuditors(mvoAud)) {
						m_Log.error(lbl + "CRIT - could not broadcast to any "
									+ "Auditors");
						m_FailureMsg = "no Auditors available, cannot proceed";
						ret = false;
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						break;
					}

					// send our data to the committee (asynchronously)
					if (!m_Parent.sendToMVOCommittee(mvoCommittee, opBlock)) {
						m_Log.error(lbl + "could not send to MVO "
									+ "committee, abort");
						m_FailureMsg = "could not send to MVO committee";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailMVO,
									 m_FailureMsg);
						ret = false;
						break;
					}
					/* NB:	the callback will advance state to M_ReceiptGen
					 * 		when the last committee response has been received
					 */
				}
				else {
					/* Blast the auditor report to all known auditors.  At least
					 * one Auditor must succeed, or we must nack the lead MVO.
					 */
					if (!m_Parent.broadcastToAuditors(mvoAud)) {
						m_Log.error(lbl + "CRIT - could not broadcast to any "
									+ "Auditors");
						m_FailureMsg = "no Auditors available, cannot proceed";
						ret = false;
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						break;
					}

					// go on to return the reply to the lead MVO
					recordSuccess(M_OB_Signed);
					ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				}
				break;

			case M_ReceiptGen:
				/* Here we have finished all OB verification and signing tasks,
				 * whether or not we were the lead in the transaction.  If we
				 * were the lead, then it also falls to us to build and sign all
				 * of the receipts required.  In the case of a deposit, this
				 * means one receipt for each unique payee address in the list
				 * of outputs.  Always the depositor gets one payer receipt.
				 *
				 * If we're not the lead, it's a state error to come here.
				 * However, we do not fail with an error, instead we'll just
				 * go to the next step.
				 */
				if (!m_IsLead) {
					// this is a state error, but do not return failure for it
					m_Log.error(lbl + "hit receipt generation step on "
								+ "!lead MVO, skipping to reply step");
					ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
					break;
				}
				if (!m_ExpectingKeyServerReply) {
					/* Generate receipts corresponding to outputs.  Also
					 * make a key generation request to an Auditor, and
					 * record the response in m_KeyResponse.  We can
					 * advance to the next step (for real) only once the
					 * Auditor replies to us successfully.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					ArrayList<OperationsBlock.ClientPayee> obPayees
						= opBlock.getPayees();

					// init a receipt for the depositor (payer)
					ReceiptBlock depReceipt = new ReceiptBlock(m_Log);
					depReceipt.setReceiptId(stObj.getNextID());
					depReceipt.setReceiptType(depReceipt.M_Sender);
					depReceipt.setSource(clientReq.getSender());
					depReceipt.setChainId(clientReq.getChainId());
					m_Receipts.add(depReceipt);

					// build a payee receipt for each payee in the OB
					int payeeIdx = 1;
					NumberFormat nf = NumberFormat.getIntegerInstance();
					nf.setMinimumIntegerDigits(3);
					// map of payee addresses + asset combos we've seen:
					HashMap<String, ReceiptBlock.ReceiptPayee> seenPayees
						= new HashMap<String, ReceiptBlock.ReceiptPayee>();

					for (OperationsBlock.ClientPayee payee : obPayees) {
						// allocate a receipt for the recipient
						ReceiptBlock receipt = new ReceiptBlock(m_Log);
						receipt.setReceiptType(receipt.M_Recipient);
						// pick a random ID for the receipt
						receipt.setReceiptId(stObj.getNextID());
						receipt.setSource(clientReq.getSender());
						receipt.setChainId(clientReq.getChainId());

						// make a single ReceiptPayee for the receiver
						ReceiptBlock.ReceiptPayee rPayee
							= receipt.new ReceiptPayee();

						// show payee only one payee (themselves)
						rPayee.m_Payee = "payee001";
						// assets always the same for a deposit
						rPayee.m_Asset
							= Keys.toChecksumAddress(opBlock.getAsset());
						rPayee.m_Address
							= Keys.toChecksumAddress(payee.m_Address);
						rPayee.m_Amount = payee.m_OutputAmount;
						rPayee.m_ID = new String(payee.m_ID);
						receipt.setTagID(payee.m_ID);
						// set tag for Sender receipt to first output
						if (depReceipt.getTagID().isEmpty()) {
							depReceipt.setTagID(payee.m_ID);
						}

						// memo is found only in the original client request
						ArrayList<ClientMVOBlock.ClientPayee> orgPayees
							= clientReq.getPayees();
						String memo = "";
						for (ClientMVOBlock.ClientPayee orgPayee : orgPayees) {
							// find the matching payee output
							if (orgPayee.m_Payee.endsWith(payee.m_Payee)) {
								memo = new String(orgPayee.m_Memo);
								break;
							}
						}
						rPayee.m_Memo = memo;
						receipt.addPayee(rPayee);		// the only one
						m_Receipts.add(receipt);

						/* Now build a list element ReceiptPayee for the sender.
						 * We consolidate totals and memo lines into one, if
						 * there are multiple occurrences of a given address
						 * and asset type.
						 */
						ReceiptBlock.ReceiptPayee sPayee = null;
						String rctKey
							= payee.m_Address + ":" + opBlock.getAsset();
						// use existing one if found
						ReceiptBlock.ReceiptPayee existPayee
							= seenPayees.get(rctKey);
						if (existPayee == null) {
							// make new one and add to map
							sPayee = depReceipt.new ReceiptPayee();
							sPayee.m_Address
								= Keys.toChecksumAddress(payee.m_Address);
							sPayee.m_Asset
								= Keys.toChecksumAddress(opBlock.getAsset());
							seenPayees.put(rctKey, sPayee);
						}
						else {
							sPayee = existPayee;
						}

						// re-use payee sequence number if required
						String payeeSeq = "payee" + nf.format(payeeIdx);
						if (sPayee.m_Payee.isEmpty()) {
							sPayee.m_Payee = new String(payeeSeq);
							payeeIdx++;
						}
						// else: leave it alone (i.e. use the existing value)

						// propagate memo
						if (sPayee.m_Memo.isEmpty()) {
							sPayee.m_Memo = memo;
						}
						else {
							// append
							sPayee.m_Memo += ("; " + memo);
						}
						// (NB: no ID is set for a sender receipt)

						// aggregate amount if required
						if (existPayee != null) {
							BigInteger newTot
								= sPayee.m_Amount.add(payee.m_OutputAmount);
							sPayee.m_Amount = newTot;
						}
						else {
							sPayee.m_Amount = payee.m_OutputAmount;
							depReceipt.addPayee(sPayee);	// add to list
						}
					}

					/* Now, for every unique receipt ID found in m_Receipts,
					 * request an AES key for it from the Auditor,
					 * using chain+ID+address as the hash key.  Send it
					 * asynchronously and advance when the callback sees
					 * the response, recorded in m_KeyResponse.
					 */
					// start by picking a random auditor
					String randAud2 = mvo.getRandomAuditor(null);
					if (randAud2.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get receipt keys failure, "
									+ "no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// now build the key block to send to the Auditor
					MVOKeyBlock rctKeyBlock = new MVOKeyBlock(m_Log);
					rctKeyBlock.setMapping(rctKeyBlock.M_MapReceipt);
					rctKeyBlock.setOpcode(rctKeyBlock.M_OpNew);

					// we must supply the user's orig req
					String cliJSON = clientReq.getDecryptedPayload();
					rctKeyBlock.setClientBlockJson(cliJSON);

					// for each receipt, compute chainId+ID+address and get key
					for (ReceiptBlock rct : m_Receipts) {
						String rId = rct.getReceiptId();
						String addy = "";
						if (rct.getReceiptType().equals(rct.M_Sender)) {
							addy = rct.getSource();
						}
						else {
							ReceiptBlock.ReceiptPayee rPay
								= rct.getPayees().get(0);
							if (rPay == null) {
								m_Log.error(lbl + "receipt Id " + rId
										+ " has no ReceiptPayee record, skip");
								continue;
							}
							addy = rPay.m_Address;
						}
						ArrayList<String> hashComps = new ArrayList<String>(3);
						hashComps.add(Long.toString(clientReq.getChainId()));
						hashComps.add(rId);
						hashComps.add(addy.toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						rctKeyBlock.addHashToList(keyHash);
					}

					// sign the MVOKeyBlock
					MVOSignature rctSig = new MVOSignature();
					rctSig.m_Signer = mvo.getMVOId();
					PrivateKey sigKy = mvoConf.getCommPrivKey();
					String sData = rctKeyBlock.buildSignedData();
					boolean sOk = true;
					if (rctKeyBlock.getErrCode() != 0) {
						sOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKy, sData);
						if (sig == null) {
							sOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							rctSig.m_Signature = sig;
						}
					}
					if (!sOk) {
						m_FailureMsg = "could not build key block for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					rctKeyBlock.setSignature(rctSig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud2, rctKeyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud2 + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud2);
							randAud2 = mvo.getRandomAuditor(excludeAIds);
							if (randAud2.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg
							= "error looking up receipt eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// processing key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String sigAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature adSig
						= m_KeyResponse.getSignature();
					boolean sigVer = true;
					PublicKey adKey = mvoConf.getPeerPubkey(adSig.m_Signer);
					if (adKey == null) {
						sigVer = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ adSig.m_Signer + ", cannot verify sig "
									+ "on returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(adKey,
														   sigAudData,
														   adSig.m_Signature))
						{
							sigVer = false;
							m_Log.error(lbl + "sig verify failed for "
										+ "Aud " + adSig.m_Signer
										+ ", cannot verify sig on returned "
										+ "AuditorKeyBlock");
						}
					}
					if (!sigVer) {
						m_FailureMsg = "error looking up receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + adSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up receipt AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all keys, move to next state
				/* (NB: we do not sign or encrypt receipts until we hit the
				 * M_Deposited state)
				 */
				recordSuccess(M_ReceiptGen);
				ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				break;

			case M_Replied:
				/* Here we're ready to reply to the party who sent us this
				 * whole request.  As a lead, that's a dApp client.  Otherwise
				 * it's the lead MVO who picked us for their committee.
				 */
				if (!m_IsLead) {
					// affix our signature to the passed OB block JSON text
					PrivateKey ourKey = mvoConf.getCommPrivKey();
					StringBuilder opbJSON = new StringBuilder(2048);
					opBlock.addJSON(opbJSON);
					String fullTxt = opbJSON.toString();
					String ourL2Sig = EncodingUtils.signStr(ourKey, fullTxt);

					// the reply ID in the broker entry specifies lead MVO
					if (mvoRep == null || mvoRep.isEmpty()) {
						m_Log.error(lbl + "could not send reply to "
									+ "lead MVO (no tag), abort");
						m_FailureMsg = "no reply tag to send committee reply "
										+ "to lead";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					leadMVOId = m_Parent.getTagField(mvoRep, 1);

					// encrypt to this pubkey
					PublicKey leadKey = mvoConf.getPeerPubkey(leadMVOId);
					if (leadKey == null) {
						m_Log.error(lbl + "could not encrypt OB to "
									+ "lead MVO, no pubkey");
						m_FailureMsg = "no pubkey found for lead MVO "
										+ leadMVOId;
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					String encTxt
						= EncodingUtils.base64PubkeyEncStr(fullTxt, leadKey);

					// send it back to the lead MVO as one message
					String leadMsg = mvoRep + "::" + encTxt + "::" + ourL2Sig;
					PeerConnector peerConn = stObj.getPeerConnection(leadMVOId);
					if (peerConn == null) {
						// this is an error, because we're doing a reply
						m_Log.error(lbl + "no connection to lead MVO Id "
									+ leadMVOId);
						m_FailureMsg = "no lead MVO connection";
						ret = false;
						// cannot nack w/out connection, so just clean up
						advanceState(m_PrevStep, m_FinalStep,
									 M_UnavailMVO, m_FailureMsg);
						break;
					}

					// send as asynchronous write
					if (!peerConn.sendString(leadMsg, null)) {
						m_Log.error(lbl + "could not write reply to "
									+ "lead MVO Id " + leadMVOId);
						m_FailureMsg = "lead MVO reply failure";
						ret = false;
						// cannot nack w/out connection, so just clean up
						advanceState(m_PrevStep, m_FinalStep,
									 M_UnavailMVO, m_FailureMsg);
						break;
					}

					// as a non-lead MVO, we are now done, go to cleanup
					recordSuccess(M_Replied);
					ret = advanceState(m_CurrStep, m_FinalStep,
									   M_NoFailure, "");
				}
				else {
					/* Send our reply to the original client request.  This
					 * includes the OB we created, with all signatures of MVO
					 * committee members attached to it.
					 */
					clientReq = (ClientMVOBlock) orgReq;
					HttpServletResponse httpRep
						= (HttpServletResponse) rep.getResponse();
					if (!cHandler.sendNormalRep(httpRep, opBlock,
												clientReq.getReplyKey()))
					{
						m_Log.error(lbl + "unable to send client "
									+ "reply after successful completion, "
									+ "transId = " + m_Parent.getTransId());
						rep.complete();
						ret = false;
						// skip over final 2 states to cleanup
						advanceState(m_CurrStep, m_FinalStep, M_ProcError,
									"client reply failure");
					}
					else {
						// normal progression to next step
						rep.complete();
						recordSuccess(M_Replied);
						ret = advanceState(m_CurrStep, M_Deposited,
										   M_NoFailure, "");
					}
				}
				break;

			case M_Deposited:
				/* Here we've sent our reply already.  In the lead MVO case
				 * (only), we need to know when the transaction has been mined
				 * on-chain, so that we can upload the transaction receipts we
				 * built in a previous step. To do this, we do have a list of
				 * eNFT IDs which the SC will only mint once the deposit trans
				 * has been performed and mined.  In particular, our m_Receipts
				 * list features tags which will prove that the transaction
				 * was finalized and minted.  We'll use the Web3j interface to
				 * watch for the mint event(s), based on the list of tags found
				 * of tags found in ReceiptBlock.m_TagID for our receipts.
				 *
				 * To avoid stopping our thread, we'll make an async ABI request
				 * and set a timer which will come back here to register an
				 * error in case of timeout (i.e. transaction never mined).
				 *
				 * We also note the block number of the minting, and record it
				 * in the ReceiptBlock.m_BlockNumber field for each receipt.
				 * We then sign and encrypt each receipt, and pass to the queue.
				 *
				 * It's an error to arrive here if not the lead MVO.
				 */
				if (!m_IsLead) {
					m_Log.error(lbl + "non-lead MVO hit deposit check");
					ret = inconsistentState(rep, mvoRep);
					break;
				}

				// must have gotten key response too
				if (m_KeyResponse == null || m_Achieved < M_ReceiptGen) {
					m_Log.error(lbl + "no Auditor key response but hit "
								+ "deposit check state");
					ret = false;
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 "receipts not generated before check for "
								 + "transaction mined");
					break;
				}

				/* Save all receipts with the queueing manager, marking each
				 * as not yet being ready to upload.  Save the keys also.
				 * This is done so that the manager can recover and upload the
				 * receipts for us in the event that we exit in the interim.
				 */
				clientReq = (ClientMVOBlock) orgReq;
				scc = mvo.getSCConfig(clientReq.getChainId());
				EnftCache eCache = scc.getCache();
				BigInteger currBlock = eCache.getLatestBlock();
				ReceiptQueue rctUploader = m_Parent.getMVO().getReceiptQueue();
				boolean queueOk = true;
				for (ReceiptBlock rct : m_Receipts) {
					ReceiptQueue.ReceiptEntry qEntry
						= rctUploader.new ReceiptEntry();
					qEntry.m_ChainId = rct.getChainId();
					qEntry.m_Receipt = rct;
					qEntry.m_BlockWhenQueued = currBlock;

					// find the corresponding key
					String rId = rct.getReceiptId();
					String addy = "";
					if (rct.getReceiptType().equals(rct.M_Sender)) {
						addy = rct.getSource();
					}
					else {
						ReceiptBlock.ReceiptPayee rPay
							= rct.getPayees().get(0);
						if (rPay == null) {
							m_Log.error(lbl + "receipt Id " + rId
										+ " has no ReceiptPayee record, skip");
							queueOk = false;
							continue;
						}
						addy = rPay.m_Address;
					}
					ArrayList<String> hashComps = new ArrayList<String>(3);
					hashComps.add(Long.toString(rct.getChainId()));
					hashComps.add(rId);
					hashComps.add(addy.toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashComps);
					String keyHash = EncodingUtils.sha3(keyIdx);
					AuditorKeyBlock.KeySpec aesKey
						= m_KeyResponse.getKeyForHash(keyHash);
					if (aesKey == null) {
						m_Log.error(lbl + "no AES key found for rId "
									+ rId + ", cannot queue");
						queueOk = false;
						continue;
					}
					qEntry.m_AESkey = aesKey.m_KeyData;

					/* What we actually store in the queue is the raw receipt,
					 * including any tagID values.  It is not yet signed or
					 * encrypted using the indicated key.
					 */
					qEntry.updateJson();
					if (!rctUploader.queueReceipt(qEntry)) {
						m_Log.error(lbl + "cannot queue receipt, "
									+ qEntry.asText());
						queueOk = false;
						continue;
					}
				}
				if (!queueOk) {
					m_Log.error(lbl + "error queueing prepared but "
								+ "unsigned receipts");
					ret = false;
					advanceState(m_CurrStep, m_FinalStep, M_ProcError,
								"receipt upload failure");
					break;
				}

				recordSuccess(M_Deposited);
				// go to cleanup
				ret = advanceState(m_CurrStep, M_Completed, M_NoFailure, "");
				break;

			case M_Completed:
				/* Here all of our tasks have been completed, and the last step
				 * is to clean up data structures, removing temporary data.
				 * Garbage collection and finalize() calls will auto-trigger
				 * the next time System.gc() runs.
				 */
				// if we're lead, remove ourselves from the broker map
				if (m_IsLead) {
					Map<String, MVOBrokerEntry> brokerMap
						= stObj.getBrokerMap();
					brokerMap.remove(m_Parent.getTransId(), m_Parent);
					m_Parent = null;
				}
				m_Receipts.clear();
				m_KeyResponse = null;
				// NB: we do not need to close any connections or listeners
				break;

			default:
				// all other states don't make sense
				m_Log.error(lbl + "illegal state transition to " + m_CurrStep);
				ret = inconsistentState(rep, mvoRep);
				break;
		}
		return ret;
	}

	/**
	 * obtain whether we're the lead MVO on this request
	 * @return true if lead, false if committee member
	 */
	public boolean isLead() { return m_IsLead; }

	/**
	 * set whether we're the lead MVO on this request
	 * @param lead true if lead
	 */
	public void setLead(boolean lead) { m_IsLead = lead; }

	/**
	 * method to send pending error reply to dApp via ClientHandler
	 * @param resp the dApp response object
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackDapp(AsyncContext resp, int sc, String errTxt) {
		if (resp != null) {
			HttpServletResponse httpRep
				= (HttpServletResponse) resp.getResponse();
			m_Parent.getMVO().getClientHandler().sendErrRep(httpRep,
															sc, errTxt);
			resp.complete();
			recordSuccess(M_Replied);
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * method to send pending error reply to lead MVO via MVOHandler
	 * @param mvoRep the MVOId:transId:msgId needed to send a response to the
	 * lead MVO
	 * @param the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackMVO(String mvoRep, int sc, String errTxt) {
		if (mvoRep != null && !mvoRep.isEmpty()) {
			// do actual response write on WebSocket
			String leadMVOId = m_Parent.getTagField(mvoRep, 1);
			MVOState mvoState = m_Parent.getMVO().getStateObj();
			PeerConnector peerConn = mvoState.getPeerConnection(leadMVOId);
			if (peerConn == null) {
				// very odd since we got the committee request from the lead
				m_Log.error("nackMVO(): no return connection to lead MVO to "
							+ "send nack: " + errTxt);
				return false;
			}

			// send with asynchronous write, opening connection if necessary
			String errRep = mvoRep + "::ERR" + sc + "::" + errTxt;
			if (!peerConn.sendString(errRep, null)) {
				m_Log.error("nackMVO(): error writing nack to " + leadMVOId);
			}
			else {
				m_Log.debug("nackMVO(): sent err reply to " + leadMVOId + ": "
							+ errRep);
				recordSuccess(M_Replied);
			}
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * method to send a generic reply to the originator
	 * @param resp the dApp response object
	 * @param mvoRep the MVO reply Id
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackReq(AsyncContext resp, String mvoRep,
							int sc, String errTxt)
	{
		if (m_IsLead && resp != null) {
			// don't try to send a reply if we're past that
			if (m_Achieved >= M_Replied) {
				return false;
			}
			return nackDapp(resp, sc, errTxt);
		}
		else if (mvoRep != null) {
			return nackMVO(mvoRep, sc, errTxt);
		}
		m_Log.error("Unable to send nack!");
		return false;
	}

	/**
	 * Method to log when reason versus state is inconsistent.  Note this
	 * implies an internal programming error. The dApp client is nack'd.
	 * @param rep the dApp response object
	 * @param mvoRep the MVO reply Id
	 * @return true on success
	 */
	private boolean inconsistentState(AsyncContext rep, String mvoRep) {
		m_Log.error("DepositState internal inconsistency, step " + m_CurrStep
					+ ", reason " + m_FailureMode + ", emsg \"" + m_FailureMsg
					+ "\"");
		return nackReq(rep, mvoRep,
					   HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
					   "Internal state error, abort");
	}

	// END methods
}
