/*
 * last modified---
 * 	01-26-25 purge PeerConnectionS with null client sockets before retries
 * 	09-15-22 keep list of sent entries for error processing
 * 	08-05-22 new
 *
 * purpose---
 * 	provide an object which can buffer up Auditor broadcasts to be uploaded,
 * 	and guarantee delivery of them in order when a connection becomes available
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;

import java.util.Iterator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.LinkedBlockingQueue;
import java.io.FileReader;
import java.io.BufferedReader;
import java.io.PrintWriter;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.io.File;
import java.io.FileWriter;
import java.net.URI;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.math.BigInteger;


/**
 * This class implements a mechanism for delayed (but guaranteed) delivery of
 * MVOAuditorBlockS which must be sent to Auditors.  Its behavior is such that
 * a separate work thread is provided which does nothing until an entry appears
 * in the queue, at which point staggered delivery attempts commence.  It is
 * assumed that the original request which caused the entry to be placed in
 * the queue has already been processed to completion.  Each Auditor node known
 * to the MVO gets its own dedicated PerAuditorQueue object.
 */
public final class AuditorQueue {
	// BEGIN data members
	/**
	 * constant defining length of time to wait between queue and connection
	 * checks (msec)
	 */
	public final long		M_CheckInterval = 1000L;

	/**
	 * constant defining spacing between messages sent when draining the queue
	 * (to prevent message flooding, msec)
	 */
	private final long		M_MsgSpacing = 3000L;

	/**
	 * constant defining how long to wait for the upload before timing out
	 * (msec)
	 */
	private final long		M_Timeout = 30000L;

	/**
	 * the MVO object which owns us
	 */
	private MVO				m_MVO;

	/**
	 * inner class to describe a queued entry (also represented as a single line
	 * in the backing file)
	 */
	public final class BroadcastEntry {
		/**
		 * ID of blockchain for which this entry was generated
		 */
		public long				m_ChainId;

		/**
		 * JSON text of MVOAuditorBlock to be broadcast
		 */
		public String			m_BlockJson;

		/**
		 * the MVOAuditorBlock itself
		 */
		public MVOAuditorBlock	m_Block;

		/**
		 * nullary constructor
		 */
		public BroadcastEntry() {
			m_BlockJson = "";
		}

		/**
		 * emit entry as text suitable for inclusion in the queue
		 */
		public String asText() {
			String output = m_ChainId + "::" + m_BlockJson;
			return output;
		}

		/**
		 * build Auditor block from JSON text
		 * @return true on success
		 */
		public boolean buildBlock() {
			if (m_Block == null) {
				// allocate one
				m_Block = new MVOAuditorBlock(m_Log);
			}
			if (!m_Block.buildFromString(m_BlockJson)) {
				m_Log.error("BroadcastEntry.buildBlock: parse error");
				return false;
			}
			return true;
		}

		/**
		 * update JSON text based on Auditor block
		 */
		public void updateJson() {
			StringBuilder json = new StringBuilder(2048);
			m_Block.addJSON(json);
			m_BlockJson = json.toString();
		}
	}

	/**
	 * local copy of MVO's logging object
	 */
	private Log				m_Log;

	/**
	 * inner class to represent the queue for a particular Auditor
	 */
	public final class PerAuditorQueue implements Runnable {
		// data members
		/**
		 * the Auditor Id
		 */
		private String			m_AudId;

		/**
		 * the message queue
		 */
		private LinkedBlockingQueue<BroadcastEntry>	m_Queue;

		/**
		 * the thread that does all the work
		 */
		private Thread			m_WorkThread;

		/**
		 * the stop trigger, to cause the work thread to exit when null
		 */
		private volatile Thread	m_StopTrigger;

		/**
		 * the path to the disc file backing our queue
		 */
		private String			m_QueuePath;

		/**
		 * list of transIds mapped to messages sent to the Auditor, used to
		 * match against any error replies returned
		 */
		private HashMap<String, MVOAuditorBlock>	m_TransIdHistory;

		// methods
		/**
		 * constructor
		 * @param audId the ID of the Auditor this queue serves
	 	 * @param qPath the pathname to the disc file used to back the queue
		 */
		public PerAuditorQueue(String audId, String qPath) {
			m_AudId = audId;
			m_Queue = new LinkedBlockingQueue<BroadcastEntry>();
			m_QueuePath = new String(qPath + "AuditorQueue-" + m_MVO.getMVOId()
									+ ".json");
			m_TransIdHistory = new HashMap<String, MVOAuditorBlock>();
		}

		/**
		 * start method
		 * @return true if all went okay, false if queue could not be started
		 */
		public boolean start() {
			final String lbl = "PerAuditorQueue.start(" + m_AudId + "): ";
			// pre-load the queue with any records found saved in the disc file
			if (m_QueuePath == null || m_QueuePath.isEmpty()) {
				m_Log.error(lbl + "queue path file not set");
				return false;
			}

			// test that we have a path to this Auditor (no connect yet)
			URI audURI = m_MVO.getURIforMVO(m_AudId);
			if (audURI == null) {
				m_Log.error(lbl + "no URI support for Auditor, cannot start");
				return false;
			}

			// make sure directory exists
			File queueFile = new File(m_QueuePath);
			File parentDir = queueFile.getParentFile();
			if (!parentDir.isDirectory()) {
				// try to create it
				if (!parentDir.mkdirs()) {
					m_Log.error(lbl + "could not create queue path directory");
					return false;
				}
			}

			// create file if not found
			if (!queueFile.isFile()) {
				try {
					if (!queueFile.createNewFile()) {
						m_Log.error(lbl + "could not create queue file");
						return false;
					}
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "could not create queue file", ioe);
					return false;
				}
			}

			// open reader on file
			FileReader fr = null;
			try {
				fr = new FileReader(m_QueuePath);
			}
			catch (FileNotFoundException fnfe) {
				m_Log.error(lbl + "could not open queue backing file", fnfe);
				return false;
			}
			BufferedReader br = new BufferedReader(fr);

			String line = null;
			boolean ok = true;
			int cnt = 0;
			do {
				// each line should be one saved request message
				try {
					line = br.readLine();
					// allow for empty line
					if (line == null || line.isEmpty()) {
						continue;
					}

					// try to interpret this as a queued BroadcastEntry
					String[] lineParts = line.split("::");
					if (lineParts.length != 2) {
						m_Log.error(lbl + "illegal format, "
									+ "broadcast queue entry: " + line);
						continue;
					}
					cnt++;

					String chain = lineParts[0];
					String broadcast = lineParts[1];
					// if there is no broadcast text, throw it out
					if (broadcast.isEmpty()) {
						m_Log.warning(lbl + "empty broadcast in queue, line "
									+ cnt);
						continue;
					}
					BroadcastEntry entry = new BroadcastEntry();
					boolean lineOk = true;

					// parse both of the parts for correctness
					try {
						Long chId = Long.parseLong(chain.trim());
						entry.m_ChainId = chId;
					}
					catch (NumberFormatException nfe) {
						m_Log.error(lbl + "illegal chainId, " + chain
									+ " in " + m_AudId + " queue, line " + cnt);
						lineOk = false;
					}
					entry.m_BlockJson = broadcast;

					// allocate a block entry (because constructor doesn't)
					entry.m_Block = new MVOAuditorBlock(m_Log);
					if (!entry.buildBlock()) {
						m_Log.error(lbl + "broadcast data in " + m_AudId
									+ " Auditor queue line " + cnt
									+ " does not parse, \"" + broadcast + "\"");
						lineOk = false;
					}
					else {
						/* build AuditorBlock JSON string and record
						 * (NB: this is done in buildBlock() for ClientMVOBlock)
						 */
						StringBuilder mvoBlock = new StringBuilder(10240);
						entry.m_Block.getMVOBlock().addJSON(mvoBlock);
						entry.m_Block.setMVOBlockJson(mvoBlock.toString());
					}

					if (lineOk) {
						// it's okay, add it to the queue
						try {
							m_Queue.put(entry);
						}
						catch (InterruptedException ie) {
							m_Log.error(lbl + "interrupted adding to "
										+ "broadcast queue from disc file", ie);
							ok = false;
						}
					}
					else {
						m_Log.error(lbl + "non-broadcast entry found in "
									+ m_AudId + " queue file, " + line);
						ok = false;
					}
				}
				catch (IOException ioe) {
					m_Log.error(lbl + m_AudId + " backing queue file "
								+ "read error", ioe);
					ok = false;
				}
			} while (line != null);
			try {
				fr.close();
			}
			catch (IOException ioe) { /* ignore */ }

			// now deal with thread startup
			m_WorkThread = new Thread(this);
			m_WorkThread.setDaemon(true);
			m_StopTrigger = m_WorkThread;
			m_WorkThread.start();
			m_Log.debug(lbl + "broadcast queue initialized with "
						+ m_Queue.size() + " elements from file");
			return ok;
		}

		/**
		 * stop method
		 * @return true if all queue contents were written to backing file,
		 * false otherwise
		 */
		public boolean stop() {
			// set the trigger that will make the run loop exit
			m_StopTrigger = null;

			// kill the history
			m_TransIdHistory.clear();

			// NB: this will overwrite with a blank file if queue is now empty
			return syncQueueFile();
		}

		/**
		 * helper method to write all queue elements to the backing file
		 * @return true if all queue contents were written to backing file,
		 * false otherwise
		 */
		private synchronized boolean syncQueueFile() {
			// take any remaining queue entries and save them to backing file
			PrintWriter pw = null;
			try {
				pw = new PrintWriter(m_QueuePath);
			}
			catch (FileNotFoundException fnfe) {
				m_Log.error("AuditorQueue.syncQueueFile: could not open "
							+ m_AudId
							+ " broadcast queue backing file for write");
				return false;
			}

			/* output the text for each entry in our list
			 * NB: if there are no elements, we will write an empty file.  This
			 * may be the intended effect, if we just removed the only element
			 * present.
			 */
			for (BroadcastEntry qEntry : m_Queue) {
				pw.println(qEntry.asText());
			}
			pw.flush();
			pw.close();
			return true;
		}

		// method to implement Runnable
		/**
		 * main method to poll queue and perform uploads
		 */
		public void run() {
			final String lbl = "PerAuditorQueue.run (" + m_AudId + "): ";
			for (;;) {
				// break out of loop if we've been stopped
				Thread thisThread = Thread.currentThread();
				if (thisThread != m_StopTrigger) {
					break;
				}

				/* See if anything is in the queue.  We use peek() and not
				 * take() because we cannot be guaranteed that this thread will
				 * never be terminated ungracefully, and we don't want to
				 * remove a message that wasn't actually processed.
				 */
				BroadcastEntry qEnt = m_Queue.peek();
				if (qEnt == null) {
					// wait the recheck interval (1 sec)
					try {
						Thread.sleep(M_CheckInterval);
					}
					catch (InterruptedException ie) { /* ignore */ }
					continue;
				}
				BroadcastEntry uploadBlk = qEnt;

				// see if we currently have a valid connection to that Auditor
				MVOState stObj = m_MVO.getStateObj();
				MVOClient mvoClient = m_MVO.getMVOClient();
				PeerConnector.MVOClientWebSocket webConn = null;
				PeerConnector peerConn = stObj.getPeerConnection(m_AudId);
				if (peerConn == null) {
					// open one, but don't wait for it to connect
					URI audURI = m_MVO.getURIforMVO(m_AudId);
					webConn = mvoClient.openPeerSocket(m_AudId, audURI, true);
					if (webConn == null) {
						m_Log.error(lbl + "CRIT - error opening websocket to "
									+ m_AudId);
					}
					else {
						// success; wait interval before retry
						try {
							Thread.sleep(M_CheckInterval);
						}
						catch (InterruptedException ie) { /* ignore */ }
						continue;
					}
				}
				else {
					// check whether connection is authenticated
					webConn = peerConn.getClientSocket();
					if (webConn != null && webConn.isAuthenticated()) {
						// upload the item to the auditor
						if (!uploadBlock(uploadBlk, peerConn)) {
							// failed; wait interval before retry
							try {
								Thread.sleep(M_CheckInterval);
							}
							catch (InterruptedException ie) { /* ignore */ }
							continue;
						}
						else {
							// done; remove from list
							if (!dequeueBlock(uploadBlk)) {
								m_Log.error(lbl + "error dequeuing uploaded "
											+ "broadcast: "
											+ uploadBlk.asText());
							}
						}
					}
					else if (webConn != null) {
						// wait half interval (0.5 sec) for auth before retry
						try {
							Thread.sleep(M_CheckInterval / 2);
						}
						catch (InterruptedException ie) { /* ignore */ }
						continue;
					}
					else {
						// purge old broken connection with null client socket
						peerConn.shutdown();
						stObj.purgePeerConnection(m_AudId);
						// now retry with no wait
					}
				}

				// wait drain interval and continue with next queue item
				try {
					Thread.sleep(M_MsgSpacing);
				}
				catch (InterruptedException ie) { /* ignore */ }
			}

			m_Log.debug(lbl + "work thread terminated");
		}

		/**
		 * method to queue a broadcast (which must be correct and complete)
		 * in both the memory queue and the backing flat file
		 * @param broadcast the wrapped broadcast to be sent
		 * @return true on success, false on failure
		 */
		public boolean queueBlock(BroadcastEntry broadcast) {
			if (broadcast == null) {
				m_Log.error("AuditorQueue.queueBlock: attempt to queue missing "
							+ "broadcast");
				return false;
			}

			// add to in-memory queue
			try {
				m_Queue.put(broadcast);
			}
			catch (InterruptedException ie) {
				m_Log.error("AuditorQueue.queueBlock: interrupted queuing "
							+ "broadcast", ie);
				return false;
			}

			// now add to backing flat file by re-writing queue into it
			return syncQueueFile();
		}

		/**
		 * method to remove a broadcast from the queue (both in-memory and
		 * flat file)
		 * @param broadcast the broadcast to be removed
		 * @return true on success, false on failure
		 */
		private boolean dequeueBlock(BroadcastEntry broadcast) {
			if (broadcast == null) {
				m_Log.error("AuditorQueue.dequeueBlock: attempt to dequeue "
							+ "improper broadcast");
				return false;
			}

			// take out of in-memory queue
			if (!m_Queue.remove(broadcast)) {
				return false;
			}

			/* now remove from backing flat file by re-writing queue into it
			 * (may result in overwriting our backing file with one of zero
			 * length)
			 */
			return syncQueueFile();
		}

		/**
		 * method to upload a broadcast to the indicated Auditor
		 * @param broadcast the broadcast to be uploaded
		 * @param conn the live connection to the desired Auditor
		 * @return true on successful send
		 */
		private boolean uploadBlock(BroadcastEntry broadcast,
									PeerConnector conn)
		{
			final String lbl = "PerAuditorQueue.uploadBlock(" + m_AudId + "): ";
			if (broadcast == null || conn == null) {
				m_Log.error(lbl + "missing inputs");
				return false;
			}

			/* We should already have a good MVOAuditorBlock object, with both
			 * of its constituent parts duly signed (one by the original client,
			 * the other by us).  We now need to generate a message that looks
			 * like this: tag::encrypted text::signature (on clear text).
			 * The tag is also in 3 parts: destAUDId:transId:sequence number.
			 * Since no reply is expected (this is a uni-directional broadcast),
			 * the values chosen for transId and seqNum are unimportant.
			 */
			MVOConfig ourConf = m_MVO.getConfig();
			PrivateKey ourKey = ourConf.getCommPrivKey();
			StringBuilder audJSON = new StringBuilder(20480);
			broadcast.m_Block.addJSON(audJSON);
			String msgText = audJSON.toString();
			String ourSig = EncodingUtils.signStr(ourKey, msgText);
			// encrypt to pubkey of Auditor
			PublicKey audKey = ourConf.getPeerPubkey(m_AudId);
			if (audKey == null) {
				m_Log.error(lbl + "could not encrypt to Auditor, no pubkey");
				return false;
			}
			String encText = EncodingUtils.base64PubkeyEncStr(msgText, audKey);

			// pick a transId
			MVOState stObj = m_MVO.getStateObj();
			BigInteger tId = new BigInteger(256, stObj.getRNG());
			String transId = tId.toString();
			long seqNum = stObj.getNextAUDid();
			String tag = m_AudId + ":" + transId + ":" + seqNum;
			String audMsg = tag + "::" + encText + "::" + ourSig;

			// send using synchronous write
			if (!conn.sendString(audMsg, null)) {
				m_Log.error(lbl + "could not write broadcast of queue entry");
				return false;
			}
			else {
				// record tId used, along with message sent (for error handling)
				m_TransIdHistory.put(transId, broadcast.m_Block);
			}

			return true;
		}

		/**
		 * fetch the broadcast block that corresponded to a given transId
		 * @param tId the transId
		 * @return the associated Auditor broadcast, or null if not found
		 */
		public MVOAuditorBlock getBlockForTid(String tId) {
			return m_TransIdHistory.get(tId);
		}
		// end methods
	}

	/**
	 * the list of queues maintained, one per initialized Auditor, as a mapping
	 * of audId to queue object
	 */
	private HashMap<String, PerAuditorQueue>	m_AuditorQueues;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the plugin object of which we are a part
	 */
	public AuditorQueue(MVO mvo) {
		m_MVO = mvo;
		m_Log = m_MVO.log();
		m_AuditorQueues = new HashMap<String, PerAuditorQueue>();
	}

	/**
	 * configure and start a queue for an Auditor
	 * @param audId the ID of the Auditor
	 * @param qPath the path to the backing store file we should use for it
	 * (ends with /, does not include the filename)
	 * @return true on successful setup
	 */
	public boolean configAuditorQueue(String audId, String qPath) {
		if (audId == null || !audId.startsWith("AUD") || qPath == null
			|| qPath.isEmpty())
		{
			m_Log.error("AuditorQueue.configAuditorQueue: missing inputs");
			return false;
		}
		PerAuditorQueue audQueue = new PerAuditorQueue(audId, qPath);
		if (!audQueue.start()) {
			m_Log.error("AuditorQueue.configAuditorQueue: error starting queue "
						+ "for " + audId);
			return false;
		}
		m_AuditorQueues.put(audId, audQueue);
		return true;
	}

	/**
	 * take down all running Auditor queues
	 * @return true on success
	 */
	public boolean stopQueues() {
		boolean gotErr = false;
		for (Map.Entry<String, PerAuditorQueue> entry
			: m_AuditorQueues.entrySet())
		{
			PerAuditorQueue queue = entry.getValue();
			if (!queue.stop()) {
				gotErr = true;
			}
		}
		m_AuditorQueues.clear();
		return !gotErr;
	}

	/**
	 * obtain the queue for a given Auditor
	 * @param audId the ID of the Auditor
	 * @return the queue object, or null if none found
	 */
	public PerAuditorQueue getAuditorQueue(String audId) {
		if (audId == null || !audId.startsWith("AUD")) {
			return null;
		}
		return m_AuditorQueues.get(audId);
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_AuditorQueues != null) {
				m_AuditorQueues.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
