/*
 * last modified---
 * 	08-07-25 switch to MariaDb Driver; do "set session wsrep_sync_wait=1" in
 * 			 sessions on all new connections; clear =0 on close
 * 	03-14-23 relocate from aud.db
 * 	09-22-22 new
 *
 * purpose---
 * 	provide an interface to pooled DB connections obtained via Proxool
 */

package cc.enshroud.jetty.db;

import cc.enshroud.jetty.log.Log;

import org.logicalcobwebs.proxool.ConnectionPoolDefinitionIF;
import org.logicalcobwebs.proxool.ProxoolException;
import org.logicalcobwebs.proxool.ProxoolFacade;
import org.logicalcobwebs.proxool.admin.SnapshotIF;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.Properties;


/**
 * Central manager of database connections.  MySQL is assumed, without use of
 * transactions.  This class is loosely based on two classes from Openfire:
 * org.jivesoftware.database.DbConnectionManager and
 * org.jivesoftware.database.DefaultConnectionProvider.
 */
public final class DbConnectionManager {
	// BEGIN data members
	/**
	 * minimum default number of connections to support
	 */
	private final int					M_MinConnections = 5;

	/**
	 * maximum default number of connections to support
	 */
	private final int					M_MaxConnections = 50;

	/**
	 * activity timeout, 15 min in msec
	 */
	private final int					M_ActiveTimeout = 900000;

	/**
	 * maximum connection lifetime, 8 hours in msec
	 */
	private final int					M_Lifetime = 21600000;

	/**
	 * max number of times to retry getting a connection
	 */
	private final int					M_RetryMax = 10;

	/**
	 * wait interval between retries, in msec
	 */
	private final int					M_RetryWait = 250;

	/**
	 * MySQL/MariaDB driver
	 */
	//private final String			M_SQLDriver = "com.mysql.cj.jdbc.Driver";
	private final String				M_SQLDriver = "org.mariadb.jdbc.Driver";

	/**
	 * pool name
	 */
	private final String				M_Pool = "enshroud";

	/**
	 * properties passed to Proxool
	 */
	private Properties					m_Settings;

	/**
	 * the DB URL
	 */
	private String						m_DbURL;

	/**
	 * the URL to pass to Proxool
	 */
	private String						m_ProxoolURL;

	/**
	 * the username to use to log into the database
	 */
	private String						m_DBuser;

	/**
	 * the password to use to connect to the database
	 */
	private String						m_DBpass;

	/**
	 * error logger
	 */
	private Log							m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param dbURL the URL describing where to find database, null for default
	 * @param dbUser the database connection user
	 * @param dbPass the database connection password
	 * @param logger the error logger to use
	 */
	public DbConnectionManager(String dbURL,
							   String dbUser,
							   String dbPass,
							   Log logger)
	{
		if (dbURL != null && !dbURL.isEmpty()) {
			m_DbURL = new String(dbURL);
		}
		else {
			// NB: actual database name will be enshroud_receipts (for MVOs) or
			// 	   enshroud_keys (for AUDs)
			m_DbURL = "jdbc:mysql://localhost:3306/enshroud?serverTimezone=UTC";
		}
		m_DBuser = new String(dbUser);
		m_DBpass = new String(dbPass);
		m_Log = logger;

		m_Settings = new Properties();
		m_ProxoolURL = "proxool." + M_Pool + ":" + M_SQLDriver + ":" + m_DbURL;
		m_Settings.setProperty("proxool.maximum-activetime",
								Integer.toString(M_ActiveTimeout));
		m_Settings.setProperty("proxool.maximum-connection-count", 
								Integer.toString(M_MaxConnections));
		m_Settings.setProperty("proxool.minimum-connection-count",
								Integer.toString(M_MinConnections));
		m_Settings.setProperty("proxool.maximum-connection-lifetime",
								Integer.toString(M_Lifetime));
		m_Settings.setProperty("proxool.house-keeping-test-sql",
								"select CURRENT_DATE");
		m_Settings.setProperty("user", m_DBuser);
		m_Settings.setProperty("password", m_DBpass);
	}

	/**
	 * utility method to obtain a low-level connection via Proxool
	 * @return the connection
	 * @throws SQLException
	 */
	private Connection getDbConnection() throws SQLException {
		try {
			Class.forName("org.logicalcobwebs.proxool.ProxoolDriver");
			return DriverManager.getConnection(m_ProxoolURL, m_Settings);
		}
		catch (ClassNotFoundException cnfe) {
			throw new SQLException("DbConnectionManager: could not find "
									+ "driver: " + cnfe);
		}
		catch (SQLException se) {
			throw new SQLException("DbConnectionManager: SQLException "
									+ "connecting to DB: " + se);
		}
	}

	/**
	 * obtain a connection from the pool and return it
	 * @return the connection
	 * @throws SQLException
	 */
	public Connection getConnection() throws SQLException {
		int retryCnt = 0;
		Connection con = null;
		SQLException lastException = null;
		do {
			try {
				con = getDbConnection();
				if (con != null) {
					/* Ensure that "critical reads" wait for Galera cluster sync
					 * to occur.  This (value = 1) will only affect SELECT or
					 * START_TRANSACTION statements, but not INSERT or DELETE.
					 * However, we sometimes perform reads in the context of
					 * write transactions, e.g. to check whether a change is
					 * needed before doing the write.
					 */
					final String setSql = "SET SESSION wsrep_sync_wait = 1;";
					Statement setSt = con.createStatement();
					setSt.execute(setSql);
					setSt.close();
					// got one, return it
					return con;
				}
			}
			catch (SQLException se) {
				lastException = se;
			}
			finally {
				try {
					Thread.sleep(M_RetryWait);
				}
				catch (InterruptedException ie) { /* ignored */ }
				++retryCnt;
			}
		} while (retryCnt <= M_RetryMax);
		throw new SQLException("DbConnectionManager.getConnection() "
								+ "failed to obtain a connection after "
								+ retryCnt + " retries.  Last exception was: "
								+ lastException);
	}

	/**
	 * close a statement.  For use within finally { }.
	 * @param stmt the statement to be closed
	 */
	public void closeStatement(Statement stmt) {
		if (stmt != null) {
			try {
				stmt.close();
			}
			catch (SQLException se) {
				m_Log.error("Error closing statement", se);
			}
		}
	}

	/**
	 * close a result set.  For use within finally { }.
	 * @param rs the result set to be closed
	 */
	public void closeResultSet(ResultSet rs) {
		if (rs != null) {
			try {
				rs.close();
			}
			catch (SQLException se) {
				m_Log.error("Error closing result set", se);
			}
		}
	}

	/**
	 * close a connection, returning it to the pool.  For use within finally
	 * { }.
	 * @param con the connection to be closed
	 */
	public void closeConnection(Connection con) {
		if (con != null) {
			try {
				if (!con.isClosed()) {
					// clear wsrep_sync_wait flag (will reset it at next get)
					final String setSql = "SET SESSION wsrep_sync_wait = 0;";
					Statement setSt = con.createStatement();
					setSt.execute(setSql);
					setSt.close();
				}
				con.close();
			}
			catch (SQLException se) {
				m_Log.error("Error closing connection", se);
			}
		}
	}

	/**
	 * close a statement and a connection.  For use within finally { }.
	 * @param stmt the statement to close
	 * @param con the connection to close
	 */
	public void closeConnection(Statement stmt, Connection con) {
		closeStatement(stmt);
		closeConnection(con);
	}

	/**
	 * close a result set, statement, and connection.  For use within finally
	 * { }.
	 * @param rs the result set to close
	 * @param stmt the statement to close
	 * @param con the connection to close
	 */
	public void closeConnection(ResultSet rs, Statement stmt, Connection con) {
		closeResultSet(rs);
		closeStatement(stmt);
		closeConnection(con);
	}

	/**
	 * provide an implementation of toString() which will show pool stats
	 * @return a string representation of this object
	 */
	@Override
	public String toString() {
		String ret = null;
		try {
			ConnectionPoolDefinitionIF poolDef
				= ProxoolFacade.getConnectionPoolDefinition(M_Pool);
			SnapshotIF poolStats = ProxoolFacade.getSnapshot(M_Pool, true);
			ret = poolDef.getMinimumConnectionCount() + ", "
					+ poolDef.getMaximumConnectionCount() + ", "
					+ poolStats.getAvailableConnectionCount() + ", "
					+ poolStats.getActiveConnectionCount();
		}
		catch (ProxoolException pe) {
			ret = "Proxool DB Connection Provider";
		}
		finally {
			return ret;
		}
	}

	// END methods
}
