/*
 * last modified---
 * 	01-06-26 remove debug about keys not found
 * 	03-14-23 generic classes now in .db
 * 	09-30-22 use same connection / transaction for multi-insert/delete
 * 	09-27-22 new
 *
 * purpose---
 * 	provide an API to manipulate records in the ENSH_ENFT_KEYS table
 */

package cc.enshroud.jetty.aud.db;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.GenericDb;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.db.DbConnectionManager;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Date;


/**
 * This class supplies atomic operations which act on the ENSH_ENFT_KEYS table
 * in the Auditor's database.  These methods correspond to logical operations,
 * e.g. create, get, delete.  They closely interoperate with the class that
 * instantiates a row in the table, {@link AESKey AESKey}.
 */
public final class EnftKeysDb extends GenericDb {
	// BEGIN data members

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param manager the DB connection manager
	 * @param logger the error/debug logger
	 */
	public EnftKeysDb(DbConnectionManager manager, Log logger) {
		super(manager, logger);
		m_Table = "ENSH_ENFT_KEYS";
	}

	/**
	 * obtain the AES key for a given hash
	 * @param hash the hash of the key sought
	 * @param conn DB connection to use (create one if passed as null)
	 * @return the key record for this ID, null on error or if not found
	 */
	public AESKey getKey(String hash, Connection conn) {
		final String lbl = this.getClass().getSimpleName() + ".getKey: ";
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing key hash");
			return null;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				return null;
			}
		}

		// build and execute query
		String getSql = "select * from " + m_Table + " where HASH=?;";
		PreparedStatement query = null;
		ResultSet keyRS = null;
		try {
			query = dbConn.prepareStatement(getSql);
			query.setString(1, hash);
			keyRS = query.executeQuery();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running eNFT key query for "
						+ m_Table, sqle);
		}

		// examine results and load a return object (index is unique)
		AESKey keyRec = null;
		try {
			while (keyRS != null && keyRS.next()) {
				String aesKey = keyRS.getString("AES_KEY");
				keyRec = new AESKey(hash, aesKey);
			}
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error parsing result set for " + m_Table, sqle);
			keyRec = null;
		}
		finally {
			// don't close a passed connection
			if (conn != null) {
				m_DbManager.closeResultSet(keyRS);
				m_DbManager.closeStatement(query);
			}
			else {
				m_DbManager.closeConnection(keyRS, query, dbConn);
			}
		}
		return keyRec;
	}

	/**
	 * obtain a list of keys from a list of hashes (note that caller should
	 * check to ensure that all input hashes were indeed returned in output)
	 * @param hashes the list of hashes for which keys are wanted
	 * @return list of keys found, or null on any errors (all or nothing)
	 */
	public ArrayList<AESKey> getKeys(ArrayList<String> hashes) {
		final String lbl = this.getClass().getSimpleName() + ".getKeys: ";
		if (hashes == null || hashes.isEmpty()) {
			m_Log.error(lbl + "missing key hashes");
			return null;
		}

		// get a DB connection
		Connection dbConn = getDbConnection();
		if (dbConn == null) {
			return null;
		}

		// build and execute query
		StringBuilder getSql = new StringBuilder(1024);
		getSql.append("select * from " + m_Table + " where HASH in (");
		int hashCnt = 1;
		for (String hash : hashes) {
			getSql.append("?");
			if (hashCnt++ < hashes.size()) {
				getSql.append(",");
			}
		}
		getSql.append(");");

		PreparedStatement query = null;
		ResultSet keyRS = null;
		try {
			query = dbConn.prepareStatement(getSql.toString());
			hashCnt = 1;
			for (String hash : hashes) {
				query.setString(hashCnt++, hash);
			}
			keyRS = query.executeQuery();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running eNFT keys query for "
						+ m_Table, sqle);
			m_DbManager.closeConnection(keyRS, query, dbConn);
		}

		// examine results and load a return object (each hash is unique)
		ArrayList<AESKey> keyRecs = new ArrayList<AESKey>(hashes.size());
		try {
			while (keyRS != null && keyRS.next()) {
				String hash = keyRS.getString("HASH");
				String aesKey = keyRS.getString("AES_KEY");
				keyRecs.add(new AESKey(hash, aesKey));
			}
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error parsing result set for " + m_Table, sqle);
			keyRecs = null;
		}
		finally {
			m_DbManager.closeConnection(keyRS, query, dbConn);
		}
		return keyRecs;
	}

	/**
	 * create a key record for an eNFT
	 * @param hash the index hash
	 * @param aesKey the key value
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on success or duplicate, false on bad input
	 * @throws EnshDbException on all database errors, dup record, etc.
	 */
	public boolean insertKey(String hash, String aesKey, Connection conn)
		throws EnshDbException
	{
		final String lbl = this.getClass().getSimpleName() + ".insertKey: ";
		// check inputs
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing eNFT hash");
			return false;
		}
		if (aesKey == null || aesKey.isEmpty()) {
			m_Log.error(lbl + "missing AES key value");
			return false;
		}

		// another Auditor may have already done this
		AESKey existKey = getKey(hash, conn);
		if (existKey != null) {
			// make sure value is the same
			if (aesKey.equals(existKey.getKey())) {
				// nothing to do here
				return true;
			}
			m_Log.error(lbl + "hash " + hash + " exists with different key "
						+ "value");
			return false;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				throw new EnshDbException("could not get database connection");
			}
		}

		// build and execute SQL insertion string
		String insSql = "insert into " + m_Table + " values(?, ?);";
		insSql += ";";
		PreparedStatement keyInsert = null;
		int insCnt = 0;
		try {
			keyInsert = dbConn.prepareStatement(insSql);
			keyInsert.setString(1, hash);
			keyInsert.setString(2, aesKey);
			insCnt = keyInsert.executeUpdate();
		}
		catch (SQLException sqle) { 
			m_Log.error(lbl + "exception running insert for " + m_Table, sqle);
			if (conn != null) {
				m_DbManager.closeStatement(keyInsert);
			}
			else {
				m_DbManager.closeConnection(keyInsert, dbConn);
			}
			throw new EnshDbException(sqle.getMessage(), insSql,
									  sqle.getErrorCode());
		}
		finally {
			// do not close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(keyInsert);
			}
			else {
				m_DbManager.closeConnection(keyInsert, dbConn);
			}
		}
		if (insCnt != 1) {
			m_Log.error(lbl + "unexpected insert count on " + m_Table
						+ ": " + insCnt);
			throw new EnshDbException("Bad insert count, " + insCnt, insSql);
		}
		return true;
	}

	/**
	 * delete a key entry (can be done in a transaction context by passing conn)
	 * @param hash the hash of the entry to delete (unique index)
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on deletion, false on errors or if not found
	 */
	public boolean purgeKey(String hash, Connection conn) {
		final String lbl = this.getClass().getSimpleName() + ".purgeKey: ";
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing key hash");
			return false;
		}

		// get a DB connection
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				return false;
			}
		}

		// build and execute update string
		String updSql = "delete from " + m_Table + " where HASH=?;";
		PreparedStatement delKey = null;
		int updCnt = 0;
		try {
			delKey = dbConn.prepareStatement(updSql);
			delKey.setString(1, hash);
			updCnt = delKey.executeUpdate();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "exception running " + updSql, sqle);
		}
		finally {
			// do not close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(delKey);
			}
			else {
				m_DbManager.closeConnection(delKey, dbConn);
			}
		}
		// unique index
		if (updCnt != 1) {
			m_Log.error(lbl + "bad update count deleting key hash " + hash);
			return false;
		}
		return true;
	}

	// END methods
}
