/*
 * last modified---
 * 	07-09-24 mute ping attempts if !m_PeerAlive
 * 	07-05-24 fixes for dup connect attempts by websocket layer
 * 	10-04-22 pass MVOAuditorBlock broadcasts to actual handler
 * 	09-08-22 new, based on mvo.PeerConnector
 *
 * purpose---
 * 	provide a connection object between an MVO and a peer MVO or AUD node
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.ClientReceiptBlock;

import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.CloseStatus;
import org.eclipse.jetty.websocket.api.StatusCode;
import org.eclipse.jetty.websocket.api.UpgradeRequest;
import org.eclipse.jetty.websocket.api.RemoteEndpoint;
import org.eclipse.jetty.websocket.api.ProtocolException;
import org.eclipse.jetty.websocket.common.WebSocketSession;
import org.eclipse.jetty.websocket.common.io.FutureWriteCallback;
import org.eclipse.jetty.websocket.api.annotations.WebSocket;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketConnect;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketClose;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketError;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketMessage;

import java.util.Base64;
import java.util.Timer;
import java.util.TimerTask;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.net.URI;


/**
 * This class represents the connections to a peer node (MVO or Auditor).  Due
 * to the way WebSocket connections work, we need a pair of connections, one
 * where we act as the server and one where we act as the client.  This class
 * manages the two types of WebSocket classes (AUDClientSocket and
 * AUDServerSocket), along with the logic required to create, authenticate,
 * and maintain active peer connections.  The WebSocket callbacks also handle
 * processing for MVO peer requests.  All of these are sent as TEXT messages.
 * It interoperates with the connection factory classes, {@link MVOHandler}
 * (for server-type connections), and {@link MVOClient}.
 */
public final class PeerConnector {
	// BEGIN data members
	/**
	 * the owning Auditor
	 */
	private AUD							m_AUDServer;

	/**
	 * the logging object (shared with owning AUD)
	 */
	private Log							m_Log;

	/**
	 * the ID of the peer we're talking to (MVO-NNN or AUD-NNN)
	 */
	private String						m_PartnerMVO;

	/**
	 * challenge message sent to authenticate m_PartnerMVO value, format:
	 * our AUTHREQ::MVO ID::base64(random 64 bytes)::our RSA sig on second
	 * and third fields only
	 */
	private String						m_ChallengeData;

	/**
	 * helper class to provide keep-alives on peer connections via Ping/Pong
	 */
	protected class PeerKeepAlive extends TimerTask {
		// data member
		/**
		 * random data used in the current outgoing ping
		 */
		public String					m_PingData;

		/**
		 * flag indicating whether peer connection is alive
		 */
		public boolean					m_PeerAlive;

		/**
		 * flag indicating whether we've seen a reply pong since last ping
		 */
		public boolean					m_PeerPonged;

		// methods
		/**
		 * nullary constructor
		 */
		public PeerKeepAlive() {
			super();
			m_PingData = "";
			m_PeerAlive = true;
			m_PeerPonged = true;
		}

		/**
		 * perform keep-alive tasks
		 */
		public void run() {
			if (m_PeerAlive && !m_PeerPonged) {
				m_Log.warning("PeerKeepAlive: peer " + m_PartnerMVO
							+ " did not PONG our last PING");
			}
			// if no reply from last send, mark dead
			m_PeerAlive = m_PeerPonged;

			// generate unique challenge data for this run
			Long runData = m_AUDServer.getStateObj().getRNG().nextLong();
			m_PingData = runData.toString();
			String pingTxt = m_PartnerMVO + "::PING::" + m_PingData;

			// send to client side (only)
			m_PeerPonged = false;
			if (m_PeerAlive) {
				if (!sendString(pingTxt, null)) {
					m_Log.error("PeerKeepAlive: could not ping "
								+ m_PartnerMVO);
					m_PeerAlive = false;
				}
			/*
				else {
					m_Log.debug("Pinged " + m_PartnerMVO);
				}
			 */
			}
		}
	}

	/**
	 * ping-pong keep-alive task
	 */
	protected PeerKeepAlive				m_PingPonger;

	/**
	 * timer that runs keep-alive task
	 */
	private Timer						m_PingTimer;

	/**
	 * helper class to act as the WebSocket for sessions with server behavior
	 */
	@WebSocket
	public class AUDServerWebSocket {
		// data members
		/**
		 * the actual web socket session
		 */
		private WebSocketSession		m_Session;

		/**
		 * whether our connection has passed encryption challenge yet
		 */
		private boolean					m_Authenticated;

		/**
		 * Whether we initiated this connection (via MVOClient).  This is
		 * always false, but is here simply so the session can tell itself
		 * whether it's a client or server websocket.
		 */
		private boolean					m_Outgoing;

		// methods
		/**
		 * nullary constructor
		 */
		public AUDServerWebSocket() {
		}

		// methods to implement WebSocket
		/**
		 * register new sessions with other MVOs (and Auditors)
		 * @param session the new Session (really a WebSocketSession)
		 */
		@OnWebSocketConnect
		public void onWebSocketConnect(Session session) {
			final String lbl = "ServerWebSocket.onWebSocketConnect: ";
			/* This will automatically be recorded with the SessionTracker.
			 * We also need to look at the attributes provided in the upgrade
			 * request to determine the MVO-ID, and register it with the
			 * list in the m_WebSocketHandlers hash table.
			 * NB: all session.close() calls cannot send status codes or reasons
			 * 	   because this will trigger the "Client MUST mask all frames"
			 * 	   error, which will be handed to the
			 * 	   ServerWebSocket.onWebSocketClose() method with status 1002.
			 */
			if (!(session instanceof WebSocketSession)) {
				m_Log.error(lbl + "non web socket session received, ignoring");
				if (session != null) {
					// not WSS; go ahead and risk close with status
					session.close(StatusCode.PROTOCOL,
								"Not a websocket session");
				}
				return;
			}
			setSession((WebSocketSession) session);
			m_PingPonger.m_PeerAlive = true;

			// make sure mode is correct
			if (m_Outgoing) {
				m_Log.error(lbl + "for outgoing");
			/*
				m_Session.close(StatusCode.POLICY_VIOLATION,
								"Outgoing server socket!");
			 */
				m_Session.close();
				return;
			}

			// obtain the peer Id passed by the other side
			UpgradeRequest upgReq = m_Session.getUpgradeRequest();
			String mvoId = upgReq.getHeader("PeerId");
			String mvoID = "";
			if (mvoId != null && !mvoId.isEmpty()) {
				mvoID = mvoId;
			}
			if (mvoID.isEmpty()) {
				m_Log.error(lbl + "missing Peer Id");
			/*
				m_Session.close(StatusCode.POLICY_VIOLATION,
								"Missing Peer Id");
			 */
				m_Session.close();
				return;
			}
			m_PartnerMVO = new String(mvoID);

			// if we already have an authenticated connection, close dup
			String ourID = m_AUDServer.getAUDId();
			AUDServerWebSocket existingSess = getServerSocket();
			//if (existingSess != null && existingSess.isAuthenticated()) {
			if (existingSess != null) {
				m_Log.debug(lbl + "closing dup connection to Peer Id "
							+ m_PartnerMVO);
				m_Session.close();
				return;
			}

			// record new socket
			registerServerSocket(this);
			m_Log.debug(lbl + "connected incoming WS from "
						+ m_PartnerMVO + ", at: "
						+ m_Session.getRemoteAddress());

			// trigger encrypted challenge / response to prove MVO ID
			AUDState stObj = m_AUDServer.getStateObj();
			byte[] challenge = new byte[64];
			stObj.getRNG().nextBytes(challenge);
			Base64.Encoder b64e = Base64.getEncoder();
			String challengeStr = b64e.encodeToString(challenge);
			setChallenge(challengeStr);
			String sigData = ourID + "::" + challengeStr;
			PrivateKey privKey = m_AUDServer.getConfig().getCommPrivKey();
			String ourSig = EncodingUtils.signStr(privKey, sigData);
			if (!sendString("AUTHREQ::" + sigData + "::" + ourSig, null)) {
				m_Log.error(lbl + "unable to send AUTHREQ to MVO Id "
							+ m_PartnerMVO);
			}
		}

		/**
		 * register closure of existing session
		 * @param session the departing Session (actually a WebSocketSession)
		 * @param status the code with which the Session closed
		 * @param reason textual reason to go with code
		 */
		@OnWebSocketClose
		public void onWebSocketClose(Session session, int status, String reason)
		{
			/* This will automatically be removed from the SessionTracker.
			 * We need to delete this session from the m_WebSocketHandlers.
			 */
			m_Log.debug("ServerWebSocket.onWebSocketClose: close of session to "
						+ m_PartnerMVO + ", status = " + status
						+ ", reason = " + reason);
			if (!m_PartnerMVO.isEmpty()) {
				// stop pings
				if (m_PingTimer != null) {
					m_PingTimer.cancel();
				}
				registerServerSocket(null);
			}
			else {
				m_Log.error("ServerWebSocket.onWebSocketClose: session was not "
							+ "for a known partner node");
			}
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle an exception on a WebSocket session
		 * @param session the session which got the exception
		 * @param cause the exception
		 */
		@OnWebSocketError
		public void onWebSocketError(Session session, Throwable cause) {
			m_Log.error("ServerWebSocket.onWebSocketError: WebSocket "
						+ "exception: " + cause.toString());
			// remove session from object
			if (!m_PartnerMVO.isEmpty()) {
				// stop pings
				if (m_PingTimer != null) {
					m_PingTimer.cancel();
				}
				registerServerSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ServerWebSocket.onWebSocketError: session was not "
							+ "for a known partner node");
			}
			// NB: no need to close, Jetty will do it automatically
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle a text message
		 * @param session the session sending the message
		 * @param text the entire message
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session, String text) {
			final String lbl = "ServerWebSocket.onWebSocketMessage: ";
			/* If this connection is the result of an outbound connection that
			 * was opened by the MVOClient, there's a chance of a race between
			 * the return of the connect() call (Future) and the arrival of the
			 * AUTHREQ message from the other side.  Therefore, if we don't yet
			 * have a m_Session value, we need to set it to the passed param.
			 */
			if (m_Session == null && session instanceof WebSocketSession) {
				m_Log.debug(lbl + "no Session "
							+ "yet, but received our first text message");
				setSession((WebSocketSession) session);
			}
			if (!(session instanceof WebSocketSession)
				|| session != m_Session)
			{
				m_Log.error(lbl + "bad session");
				session.close(StatusCode.PROTOCOL, "Not a websocket session");
				return;
			}

			/* NB: all session.close() calls cannot send status codes or reasons
			 * 	   because this will trigger the "Client MUST mask all frames"
			 * 	   error, which will be handed to the
			 * 	   ServerWebSocket.onWebSocketClose() method with status 1002.
			 */

			// check for authentication messages if not yet authenticated
			String ourID = m_AUDServer.getAUDId();
			if (!isAuthenticated()) {
				// check for an AUTH reply
				if (text.startsWith("AUTHREP::")) {
					// this is expected if we received the connection
					if (m_Outgoing) {
						m_Log.warning(lbl + "got AUTHREP on session we opened");
					/*
						session.close(StatusCode.POLICY_VIOLATION,
									"AUTHREP on client socket!");
					 */
						session.close();
						return;
					}

					// parse message, check challenge data and signature
					// format is: AUTHREP::ID::challenge::signature
					String[] authMsg = text.split("::");
					if (authMsg.length != 4) {
						m_Log.error(lbl + "improper AUTHREP, " + text);
					/*
						session.close(StatusCode.PROTOCOL,
									  "Bad AUTHREP format");
					 */
						session.close();
						return;
					}
					String peerId = authMsg[1];
					if (!peerId.equals(m_PartnerMVO)) {
						// this should not have changed
						m_Log.error(lbl + "peer Id "
									+ peerId + " in AUTHREP does not "
									+ "match established Id " + m_PartnerMVO);
					/*
						session.close(StatusCode.PROTOCOL,
									"Expected AUTHREP from " + m_PartnerMVO);
					 */
						session.close();
						return;
					}
					String b64Challenge = authMsg[2];
					String recChallenge = getChallenge();
					if (recChallenge.isEmpty()) {
						m_Log.error(lbl
									+ "no recorded challenge data on AUTHREP");
					}
					else {
						// make sure same challenge data was echoed back
						if (!b64Challenge.equals(recChallenge)) {
							// this should not have changed
							m_Log.error(lbl + "peer Id " + m_PartnerMVO
										+ " sent back changed challenge data "
										+ "with sig");
						/*
							session.close(StatusCode.PROTOCOL,
									"Challenge data was returned incorrectly");
						 */
							session.close();
							return;
						}
					}

					// fetch pubkey of m_PartnerMVO and check sig on reply
					String peerSig = authMsg[3];
					AUDConfig config = m_AUDServer.getConfig();
					PublicKey peerKey = config.getPeerPubkey(peerId);
					if (peerKey == null) {
						m_Log.error(lbl + "no pubkey for peerId "
									+ m_PartnerMVO);
					/*
						session.close(StatusCode.SERVER_ERROR,
								"No pubkey available for ID " + m_PartnerMVO);
					 */
					 	session.close();
						return;
					}
					String sigData = m_PartnerMVO + "::" + b64Challenge;

					// verify signature
					if (!EncodingUtils.verifySignedStr(peerKey,
													   sigData,
													   peerSig))
					{
						m_Log.error(lbl + "sig "
									+ "verification error on AUTHREP from "
									+ peerId);
					/*
						session.close(StatusCode.PROTOCOL, "Failure verifying "
									+ "signature on AUTHREP");
					 */
						session.close();
						return;
					}

					// this proves the other side is who they claim to be
					m_Authenticated = true;
				}
				else if (text.startsWith("AUTHREQ::")) {
					// this is expected if we initiated the connection
					if (m_Outgoing) {
						m_Log.warning(lbl + "got AUTHREQ on session we opened");
					/*
						session.close(StatusCode.POLICY_VIOLATION,
									"AUTHREP on client socket!");
					 */
						session.close();
						return;
					}

					// parse message, check signature and give reply
					// format is: AUTHREQ::ID::challenge::signature
					String[] authMsg = text.split("::");
					if (authMsg.length != 4) {
						m_Log.error(lbl + "improper AUTHREQ, " + text);
					/*
						session.close(StatusCode.PROTOCOL,
									"Bad AUTHREQ format");
					 */
						session.close();
						return;
					}
					String peerId = authMsg[1];
					// get pubkey for this peer
					AUDConfig config = m_AUDServer.getConfig();
					PublicKey peerKey = config.getPeerPubkey(peerId);
					if (peerKey == null) {
						m_Log.error(lbl + "no pubkey for peerId " + peerId);
					/*
						session.close(StatusCode.SERVER_ERROR,
									"No pubkey available for ID " + peerId);
					 */
						session.close();
						return;
					}
					String b64Challenge = authMsg[2];
					String peerSig = authMsg[3];
					String sigData = peerId + "::" + b64Challenge;

					// verify signature
					if (!EncodingUtils.verifySignedStr(peerKey,
													   sigData,
													   peerSig))
					{
						m_Log.error(lbl + "sig "
									+ "verification error on AUTHREQ from "
									+ peerId);
					/*
						session.close(StatusCode.PROTOCOL, "Failure verifying "
									+ "signature on AUTHREQ");
					 */
						session.close();
						return;
					}

					// this proves the other side is who they claim to be
					m_PartnerMVO = new String(peerId);
					if (m_ClientWS != null) {
						m_ClientWS.setAuthenticated(true);
					}

					/* send a AUTHREP reply, allowing the other side to
					 * authenticate us via the same process
					 */
					sigData = ourID + "::" + b64Challenge;
					PrivateKey privKey = config.getCommPrivKey();
					String ourSig = EncodingUtils.signStr(privKey, sigData);
					String repData = "AUTHREP::" + sigData + "::" + ourSig;
					if (!sendString(repData, null)) {
						m_Log.error(lbl + "unable to send AUTHREP to MVO Id "
									+ m_PartnerMVO);
						m_Session.close();
						return;
					}
				}
				else {
					// regular message in non-authenticated context (error)
					m_Log.error(lbl
								+ "got regular message in non-authenticated "
								+ "context; partner MVO Id = "
								+ m_PartnerMVO + ", msg = " + text);
				/*
					m_Session.close(StatusCode.BAD_PAYLOAD,
									"Unexpected message while unauthenticated");
				 */
					m_Session.close();
				}
				return;
			}

			// process normal session message, by examining tag and body
			int parseIdx = text.indexOf("::");
			if (parseIdx == -1) {
				// unrecognized message format
				m_Log.error(lbl + "got unknown msg format from "
							+ m_PartnerMVO + ", msg = \"" + text + "\"");
				return;
			}

			/* message should be in 3 parts:
			 * 	tag::encrypted text::signature (for a normal message); or:
			 *	tag::ERRcode::error text (if it's an error reply)
			 *	MVOId::PING::random text (if it's a ping message)
			 *	MVOId::PONG::random text (if it's a pong response)
			 */
			String[] msgParts = text.split("::");
			if (msgParts.length != 3) {
				m_Log.error(lbl + "got weird msg "
							+ "format from " + m_PartnerMVO + ", msg = \""
							+ text + "\"");
				return;
			}

			// check for a PING/PONG
			final String msg = msgParts[1];
			if (msg.equals("PING")) {
				// first element should be our ID
				if (!msgParts[0].equals(ourID)) {
					m_Log.error(lbl + "PING from "
								+ m_PartnerMVO + " not for us");
				}
				String pingTxt = msgParts[2];
				// send same text back as reply
				String pongMsg = m_PartnerMVO + "::PONG::" + pingTxt;
				if (!sendString(pongMsg, null)) {
					m_Log.error(lbl + "could not "
								+ "send PONG reply to " + m_PartnerMVO);
				}
				return;
			}
			else if (msg.equals("PONG")) {
				// check whether text is same as what we sent in last ping
				String pongTxt = msgParts[2];
				if (pongTxt.equals(m_PingPonger.m_PingData)) {
					m_PingPonger.m_PeerPonged = true;
				}
				return;
			}

			// the tag should also be in 3 parts: AUDId (us), transId, seqNum
			String tag = msgParts[0];
			parseIdx = tag.indexOf(":");
			if (parseIdx == -1) {
				m_Log.error(lbl + "bad message "
							+ "tag from " + m_PartnerMVO + ", tag = \""
							+ tag + "\"");
				return;
			}
			String[] tagParts = tag.split(":");
			if (tagParts.length != 3) {
				m_Log.error(lbl + "weird message "
							+ "tag from " + m_PartnerMVO + ", tag = \""
							+ tag + "\"");
				return;
			}
			String destAUD = tagParts[0];
			String transId = tagParts[1];
			String seqNum = tagParts[2];

			// double-check the message is for us
			if (!destAUD.equals(ourID)) {
				m_Log.error(lbl + "message tagged for " + destAUD
							+ ", not for us, ignoring");
				// return a generic error
				sendGenericErr(transId, seqNum,
								"Message sent for " + destAUD + " not us");
				return;
			}
			AUDState stObj = m_AUDServer.getStateObj();

			/* Check for an error response from the partner node.  These are
			 * in the format: tag::ERR{ec}::message text, where {ec} is an
			 * integer (a HTTP error status value).
			 */
			if (msg.startsWith("ERR")) {
				// this may be an error message
				String errStat = msg;
				String errCode = errStat.substring(3).trim();
				// if a real error message, this errCode will be an integer
				boolean realErr = errCode.length() <= 8;
				Integer code = null;
				if (realErr) {
					try {
						code = Integer.parseInt(errCode);
					}
					catch (NumberFormatException nfe) {
						m_Log.error(lbl + "non-integer error code from "
									+ m_PartnerMVO + "; value passed = "
									+ errCode);
						realErr = false;
					}
				}
				String errText = msgParts[2];

				/* Since errors are sent in response to outgoing requests, we
				 * do not expect to see any here (we make no requests to MVOs).
				 * Just log it.
				 */
				if (realErr) {
					m_Log.error(lbl + "partner " + m_PartnerMVO
								+ " sent us apparent error on tId "
								+ transId + ": code = " + code
								+ ", err = \"" + errText + "\"");

					return;
				}
				/* else: attempt interpretation as a regular message, just in
				 * case we had some B64 text that actually started with "ERR"
				 */
			}

			// this is a normal message, either a new request or a reply
			String encTxt = msg;
			String partnerSig = msgParts[2];

			// decrypt the message using our pubkey
			AUDConfig config = m_AUDServer.getConfig();
			PrivateKey ourKey = config.getCommPrivKey();
			String decMsg = EncodingUtils.getStrFromBase64PubkeyEnc(encTxt,
																	ourKey);
			if (decMsg == null) {
				m_Log.error(lbl + "message from "
							+ m_PartnerMVO + " with tag " + tag
							+ " did not decrypt properly, must ignore");
				// return a generic processing error (could be attacker)
				sendGenericErr(transId, seqNum,
								"Your message did not decrypt properly");
				return;
			}

			// verify partner signature
			PublicKey peerKey = config.getPeerPubkey(m_PartnerMVO);
			if (peerKey == null) {
				m_Log.error(lbl + "missing peer pubkey for " + m_PartnerMVO);
			}
			if (!EncodingUtils.verifySignedStr(peerKey, decMsg, partnerSig)) {
				m_Log.error(lbl + "message from "
							+ m_PartnerMVO + " with tag " + tag
							+ " did not sig validate, must ignore");
				// return a generic error (could be attacker)
				sendGenericErr(transId, seqNum,
								"Your message did not pass sig check");
				return;
			}

			/* This message must be from a peer MVO.  We need to create the
			 * appropriate infrastructure to handle and send a reply.
			 * We start by determining which case we're in.  If the text
			 * starts with '{"ClientMVOBlock":' then the message is a
			 * MVOAuditorBlock being broadcast.  We process this but do not
			 * send any reply (unless there is a parse error).
			 *
			 * If the text starts with '{"mapping":"eNFT|receipt"' then
			 * the message is a MVOKeyBlock, and we need to process it and
			 * return a reply (which we can do synchronously).
			 */
			MVOKeyServer keyServer = m_AUDServer.getKeyServer();
			if (decMsg.startsWith("{\"mapping\":")) {
				MVOKeyBlock keyReq = new MVOKeyBlock(m_Log);
				if (!keyReq.buildFromString(decMsg)) {
					m_Log.error(lbl + "apparent MVOKeyBlock from "
								+ m_PartnerMVO + " did not parse");
					sendGenericErr(transId, seqNum,
									"parse error on your MVOKeyBlock");
					return;
				}

				// pass to key server
				AuditorKeyBlock keyBlock = keyServer.handleKeyBlock(keyReq);
				if (keyBlock == null) {
					m_Log.error(lbl + "error processing MVOKeyBlock from "
								+ m_PartnerMVO);
					sendGenericErr(transId, seqNum,
									"error returned from key server");
					return;
				}

				// sign and encrypt reply
				String repTag = m_PartnerMVO + ":" + transId + ":" + seqNum;
				PrivateKey privKey = m_AUDServer.getConfig().getCommPrivKey();
				StringBuilder kbJSON = new StringBuilder(2048);
				keyBlock.addJSON(kbJSON);
				String fullTxt = kbJSON.toString();
				String ourL2Sig = EncodingUtils.signStr(ourKey, fullTxt);
				PublicKey mvoKey = config.getPeerPubkey(m_PartnerMVO);
				if (mvoKey == null) {
					m_Log.error(lbl + "could not encrypt AuditorKeyBlock to "
								+ "MVO, no pubkey");
					sendGenericErr(transId, seqNum,
									"do not have your pubkey");
					return;
				}
				String replyTxt
					= EncodingUtils.base64PubkeyEncStr(fullTxt, mvoKey);

				// send synchronously as one message
				String mvoReply = repTag + "::" + replyTxt + "::" + ourL2Sig;
				if (!sendString(mvoReply, null)) {
					m_Log.error(lbl + "unable to send AuditorKeyBlock back to "
								+ m_PartnerMVO);
					// NB: sending an error reply won't work, either
					return;
				}
			/*
				else {
					m_Log.debug("Sent key server reply to " + m_PartnerMVO
								+ " for tId " + transId);
				}
			 */
			}
			else if (decMsg.startsWith("{\"ClientMVOBlock\":")) {
				MVOAuditorBlock audBlock = new MVOAuditorBlock(m_Log);
				if (!audBlock.buildFromString(decMsg)) {
					m_Log.error(lbl + "apparent MVOAuditorBlock from "
								+ m_PartnerMVO + " did not parse");
					sendGenericErr(transId, seqNum,
									"parse error on your MVOAuditorBlock");
					return;
				}

				// pass to broadcast handler
				AudBlockHandler audProcessor = m_AUDServer.getAudBlockHandler();
				if (audProcessor == null
					|| !audProcessor.handleAuditorBlock(audBlock, m_PartnerMVO))
				{
					m_Log.error(lbl + "error processing MVOAuditorBlock from "
								+ m_PartnerMVO);
					// we send the MVO an error reply when we want it to requeue
					sendGenericErr(transId, seqNum,
									"error processing your MVOAuditorBlock, "
									+ "RETRY");
				}
			}
			else {
				m_Log.error(lbl + "unknown message type from "
							+ m_PartnerMVO + "; could not parse");
				sendGenericErr(transId, seqNum,
								"your message did not make sense");
			}
		}

		/**
		 * handle a binary message (not used)
		 * @param session the session sending the message
		 * @param buf the binary buffer
		 * @param offset where the message starts
		 * @param len message length in bytes
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session,
									   byte[] buf,
									   int offset,
									   int len)
		{
			m_Log.error("ServerWebSocket.onWebSocketMessage: unsupported "
						+ "binary message received, closing " + m_PartnerMVO);
			session.close();
		}

		/**
		 * obtain the session endpoint
		 * @return the actual connected web socket session
		 */
		public WebSocketSession getSession() { return m_Session; }

		/**
		 * obtain the associated MVO ID
		 * @return the ID of the MVO on the other end of the web socket
		 */
		public String getMVOId() { return m_PartnerMVO; }

		/**
		 * obtain whether connection has been authenticated by challenge
		 * @return true if authentication was successful
		 */
		public boolean isAuthenticated() { return m_Authenticated; }

		/**
		 * obtain whether connection was initiated by us (outgoing)
		 * @return true if the MVOClient set up this connection
		 */
		public boolean isOutgoing() { return m_Outgoing; }

		/**
		 * set the session (used for outbound connections)
		 * @param session the session to set
		 */
		public void setSession(WebSocketSession session) {
			if (m_Session != null) {
				m_Log.error("MVOWebSocket.setSession: session already set!");
			}
			m_Session = session;
		}

		/**
		 * configure the session as an outgoing connection
		 * @param outgoing true if outgoing (called by MVOClient)
		 */
		public void setOutgoing(boolean outgoing) { m_Outgoing = outgoing; }
	}

	/**
	 * WebSocket object for incoming messages (server side)
	 */
	private AUDServerWebSocket			m_ServerWS;

	/**
	 * helper class to act as the WebSocket for sessions with client behavior
	 */
	@WebSocket
	public class AUDClientWebSocket {
		// data members
		/**
		 * the actual web socket session
		 */
		private WebSocketSession		m_Session;

		/**
		 * whether our connection has passed encryption challenge yet
		 */
		private boolean					m_Authenticated;

		/**
		 * Whether we initiated this connection (via MVOClient).  This is
		 * always true, but is here simply so the session can tell itself
		 * whether it's a client or server websocket.
		 */
		private boolean					m_Outgoing;

		// methods
		/**
		 * nullary constructor
		 */
		public AUDClientWebSocket() {
			m_Outgoing = true;
		}

		// methods to implement WebSocket
		/**
		 * register new sessions with other MVOs
		 * @param session the new Session (really a WebSocketSession)
		 */
		@OnWebSocketConnect
		public void onWebSocketConnect(Session session) {
			m_Log.debug("ClientWebSocket.onWebSocketConnect: got connect to "
						+ m_PartnerMVO + " at " + session.getRemoteAddress());
			if (!(session instanceof WebSocketSession)) {
				m_Log.error("ClientWebSocket.onWebSocketConnect: non web "
							+ "socket session received, ignoring");
				if (session != null) {
					session.close(StatusCode.PROTOCOL,
								"Not a websocket session");
				}
				return;
			}
			setSession((WebSocketSession) session);
			m_PingPonger.m_PeerAlive = true;

			// no message context is relevant here; we already know Peer Id
		}

		/**
		 * register closure of existing session
		 * @param session the departing Session (actually a WebSocketSession)
		 * @param status the code with which the Session closed
		 * @param reason textual reason to go with code
		 */
		@OnWebSocketClose
		public void onWebSocketClose(Session session, int status, String reason)
		{
			/* This will automatically be removed from the SessionTracker.
			 * We need to delete this session from the m_WebSocketHandlers.
			 * It's normal to see a message about frame error when the server
			 * side closes the connection.
			 */
			if (!reason.startsWith("Client MUST mask all frames")) {
				m_Log.debug("ClientWebSocket.onWebSocketClose: close of "
							+ "session to " + m_PartnerMVO + ", status = "
							+ status + ", reason = " + reason);
			}
			if (!m_PartnerMVO.isEmpty()) {
				registerClientSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ClientWebSocket.onWebSocketClose: session was not "
							+ "for a known partner node");
			}
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle an exception on a WebSocket session
		 * @param session the session which got the exception
		 * @param cause the exception
		 */
		@OnWebSocketError
		public void onWebSocketError(Session session, Throwable cause) {
			/* A ProtocolException re non-masking of frames is normal when the
			 * server side closes, because it can't write the close message to
			 * us.  So we log no error for that case.
			 * Also, a failure opening due to connection refused or a timeout
			 * needs no log.
			 */
			if (!(cause instanceof ProtocolException)
				&& !(cause instanceof java.net.ConnectException)
				&& !(cause instanceof
					 org.eclipse.jetty.websocket.api.UpgradeException)
				&& !(cause instanceof java.util.concurrent.TimeoutException))
			{
				m_Log.error("ClientWebSocket.onWebSocketError: "
							+ "WebSocket exception: " + cause.toString());
			}

			// remove session from list
			if (!m_PartnerMVO.isEmpty()) {
				registerClientSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ClientWebSocket.onWebSocketError: session was not "
							+ "for a known partner node");
			}
			// NB: no need to close, Jetty will do it automatically
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle a text message
		 * @param session the session sending the message
		 * @param text the entire message
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session, String text) {
			/* If this connection is the result of an outbound connection that
			 * was opened by the MVOClient, there's a chance of a race between
			 * the return of the connect() call (Future) and the arrival of the
			 * AUTHREQ message from the other side.  Therefore, if we don't yet
			 * have a m_Session value, we need to set it to the passed param.
			 */
			if (m_Session == null && session instanceof WebSocketSession) {
				m_Log.debug("ClientWebSocket.onWebSocketMessage: no Session "
							+ "yet but received our first text message");
				setSession((WebSocketSession) session);
			}
			if (!(session instanceof WebSocketSession)
				|| session != m_Session)
			{
				m_Log.error("ClientWebSocket.onWebSocketMessage: bad session");
				session.close(StatusCode.PROTOCOL, "Not a websocket session");
				return;
			}

			// no messages should ever be recorded here
			m_Log.error("ClientWebSocket.onWebSocketMessage: unexpected msg "
						+ "received: \"" + text + "\"");
		}

		/**
		 * handle a binary message (not used)
		 * @param session the session sending the message
		 * @param buf the binary buffer
		 * @param offset where the message starts
		 * @param len message length in bytes
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session,
									   byte[] buf,
									   int offset,
									   int len)
		{
			m_Log.error("ClientWebSocket.onWebSocketMessage: unsupported "
						+ "binary message received, closing " + m_PartnerMVO);
			session.close(StatusCode.BAD_DATA, "Unexpected binary data");
		}

		/**
		 * obtain the session endpoint
		 * @return the actual connected web socket session
		 */
		public WebSocketSession getSession() { return m_Session; }

		/**
		 * obtain the associated MVO ID
		 * @return the ID of the MVO on the other end of the web socket
		 */
		public String getMVOId() { return m_PartnerMVO; }

		/**
		 * obtain whether connection has been authenticated by challenge
		 * @return true if authentication was successful
		 */
		public boolean isAuthenticated() { return m_Authenticated; }

		/**
		 * obtain whether connection was initiated by us (outgoing)
		 * @return true if the MVOClient set up this connection
		 */
		public boolean isOutgoing() { return m_Outgoing; }

		/**
		 * set the session (used for outbound connections)
		 * @param session the session to set
		 */
		public void setSession(WebSocketSession session) {
			if (m_Session != null) {
				m_Log.error("MVOWebSocket.setSession: session already set!");
			}
			m_Session = session;
		}

		/**
		 * configure the session as an outgoing connection
		 * @param outgoing true if outgoing (called by MVOClient)
		 */
		public void setOutgoing(boolean outgoing) { m_Outgoing = outgoing; }

		/**
		 * record that the session is now authenticated
		 * @param auth the authentication status
		 */
		public void setAuthenticated(boolean auth) {
			m_Authenticated = auth;
		}
	}

	/**
	 * WebSocket object for outgoing messages (client side)
	 */
	private AUDClientWebSocket			m_ClientWS;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param aud the Auditor which owns us
	 * @param peer the ID of the peer whose connections we manage
	 */
	public PeerConnector(AUD aud, String peer) {
		m_AUDServer = aud;
		m_Log = m_AUDServer.log();
		m_PartnerMVO = peer;
		m_ChallengeData = "";
		m_PingPonger = new PeerKeepAlive();
		m_PingTimer = new Timer("PingPong:" + m_PartnerMVO, true);
		// ping peer once every minute
		m_PingTimer.schedule(m_PingPonger, 60000L, 60000L);
	}

	/**
	 * obtain the socket for the peer which has a server mode behavior
	 * (that is, we listen for incoming messages and send replies)
	 * @return the WebSocket object, if one exists
	 */
	public AUDServerWebSocket getServerSocket() { return m_ServerWS; }

	/**
	 * obtain the socket for the peer which has a client mode behavior
	 * (that is, we send outgoing messages and await replies)
	 * @return the WebSocket object, if one exists
	 */
	public AUDClientWebSocket getClientSocket() { return m_ClientWS; }

	/**
	 * write a string to the outgoing (client) socket, creating the outbound
	 * connection if one does not exist
	 * @param message the message to send
	 * @param callback the method to invoke on write completion, if any
	 * @return true on success
	 */
	public boolean sendString(String message, FutureWriteCallback callback) {
		if (message == null || message.isEmpty()) {
			m_Log.error("PeerConnector.sendString: no message");
			return false;
		}

		// make sure we have a connection to write to
		if (m_ClientWS == null) {
			URI uri = m_AUDServer.getURIforMVO(m_PartnerMVO);
			if (uri == null) {
				m_Log.error("PeerConnector.sendString: no URI found for MVO Id "
							+ m_PartnerMVO);
				return false;
			}
			m_ClientWS = m_AUDServer.getMVOClient().openPeerSocket(m_PartnerMVO,
																   uri,
																   true);
			if (m_ClientWS == null) {
				m_Log.error("PeerConnector.sendString: unable to send msg to "
							+ m_PartnerMVO + " due to failure to open socket");
				return false;
			}
		}

		// send the message
		WebSocketSession wss = m_ClientWS.getSession();
		RemoteEndpoint remote = wss.getRemote();

		/* If a callback is given, remote.sendString() uses a NOOP_CALLBACK
		 * and still sends the message asynchronously.  This is what we
		 * want, because we're allowed up to 65535 (0xFFFF) overlapping
		 * async sends on the endpoint, as opposed to 1 for a sync send.
		 */
		remote.sendString(message, callback);	// asynchronous
		/*
			m_Log.debug("sendString(): sent async message \""
						+ message + "\" to " + m_PartnerMVO);
		 */
		return true;
	}

	/**
	 * record a new server mode socket (any old one will be replaced)
	 * @param socket the websocket to record
	 * @return true on success
	 */
	public boolean registerServerSocket(AUDServerWebSocket socket) {
		if (socket == m_ServerWS) {
			// nothing to do
			return true;
		}

		// socket must have correct mode
		if (socket != null && socket.isOutgoing()) {
			m_Log.error("PeerConnector.registerServerSocket: client mode "
						+ "socket passed");
			return false;
		}

		// purge any existing, unless we're resetting to null
		if (m_ServerWS != null && socket != null) {
			WebSocketSession wss = m_ServerWS.getSession();
			if (wss != null) {
				wss.close(StatusCode.NORMAL, "Connection replaced");
				if (m_ServerWS != null && m_ServerWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.registerServerSocket: auth "
								+ "server connection replaced by another");
				}
			}
		}
		m_ServerWS = socket;
		return true;
	}

	/**
	 * record a new client mode socket (any old one will be replaced)
	 * @param socket the websocket to record
	 * @return true on success
	 */
	public boolean registerClientSocket(AUDClientWebSocket socket) {
		if (socket == m_ClientWS) {
			// nothing to do
			return true;
		}

		// socket must have correct mode
		if (socket != null && !socket.isOutgoing()) {
			m_Log.error("PeerConnector.registerClientSocket: server mode "
						+ "socket passed");
			return false;
		}

		// purge any existing
		if (m_ClientWS != null && socket != null) {
			WebSocketSession wss = m_ClientWS.getSession();
			if (wss != null) {
				wss.close(StatusCode.NORMAL, "Connection replaced");
				if (m_ClientWS != null && m_ClientWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.registerClientSocket: auth "
								+ "client connection replaced by another");
				}
			}
		}
		m_ClientWS = socket;
		return true;
	}

	/**
	 * shut down connection, doing normal termination on both sockets
	 */
	public void shutdown() {
		// stop pings
		if (m_PingTimer != null) {
			m_PingTimer.cancel();
		}
		m_PingPonger.m_PeerAlive = false;

		// shut down server side connection
		if (m_ServerWS != null) {
			WebSocketSession wss = m_ServerWS.getSession();
			if (wss != null && wss.isOpen()) {
				/* Closing with a message causes a frame mask error, because
				 * this connection is not allowed to do writes.
				 * wss.close(StatusCode.SHUTDOWN, "Connection terminated");
				 */
				if (m_ServerWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.shutdown: auth server "
								+ "connection ending for " + m_PartnerMVO);
				}
			}
		}
		m_ServerWS = null;

		// shut down client side connection
		if (m_ClientWS != null) {
			WebSocketSession wss = m_ClientWS.getSession();
			if (wss != null && wss.isOpen()) {
				// this one is okay to close
				wss.close(StatusCode.SHUTDOWN,
						"Connection terminated on shutdown");
				if (m_ClientWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.shutdown: auth client "
								+ "connection ended for " + m_PartnerMVO);
				}
			}
		}
		m_ClientWS = null;
	}

	/**
	 * set the challenge data
	 * @param challenge the data to set
	 */
	protected void setChallenge(String challenge) {
		if (challenge == null) {
			m_ChallengeData = "";
		}
		else {
			m_ChallengeData = challenge;
		}
	}

	/**
	 * obtain the challenge data
	 * @return the last challenge recorded
	 */
	protected String getChallenge() { return m_ChallengeData; }

	/**
	 * return a generic error response, in cases where we don't have a clearly
	 * defined state object to utilize
	 * @param transId the transaction Id context
	 * @param seq the sequence number of the incoming request
	 * @param errText the error message to send
	 */
	protected void sendGenericErr(String transId, String seq, String errText) {
		if (transId == null || seq == null || errText == null) {
			m_Log.error("PeerConnector.sendGenericErr: missing inputs");
			return;
		}
		String errTag = m_PartnerMVO + ":" + transId + ":" + seq;
		String errMsg = errTag + "::ERR" + StatusCode.SERVER_ERROR + "::"
						+ errText;
		sendString(errMsg, null);
	}

	// END methods
}
