/*
 * last modified---
 * 	08-05-24 reduce policy default idle timeout from 1 hr to 5 pings (300 secs)
 * 	02-12-24 suppress SslContextFactory.Server setup, as we won't be using it
 * 	09-08-22 new, based on mvo.MVOHandler
 *
 * purpose---
 * 	manage connections with peers where we are the server side
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.servlet.ServletHolder;
import org.eclipse.jetty.servlet.ServletHandler;
import org.eclipse.jetty.server.Server;
import org.eclipse.jetty.server.NetworkTrafficServerConnector;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.eclipse.jetty.util.thread.QueuedThreadPool;
import org.eclipse.jetty.websocket.servlet.WebSocketServlet;
import org.eclipse.jetty.websocket.servlet.WebSocketServletFactory;
import org.eclipse.jetty.websocket.servlet.WebSocketCreator;
import org.eclipse.jetty.websocket.servlet.ServletUpgradeRequest;
import org.eclipse.jetty.websocket.servlet.ServletUpgradeResponse;
import org.eclipse.jetty.websocket.server.WebSocketServerFactory;
import org.eclipse.jetty.websocket.api.WebSocketPolicy;
import org.eclipse.jetty.websocket.api.CloseStatus;
import org.eclipse.jetty.websocket.api.StatusCode;
import org.eclipse.jetty.websocket.common.SessionTracker;
import org.eclipse.jetty.websocket.common.WebSocketSession;

import java.util.Properties;
import java.io.IOException;
import javax.management.timer.Timer;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;


/**
 * This class is instantiated once per AUD and provides a listener and request
 * handler for MVO peer requests to make key request and broadcast actions.
 * This class manages the connections factory for server-behavior WebSockets.
 * For client side, see {@link MVOClient}.  Both kinds of sockets are grouped
 * in {@link PeerConnector}.
 */
public final class MVOHandler extends WebSocketServlet
	implements WebSocketCreator
{
	// BEGIN data members
	/**
	 * the owning AUD
	 */
	private AUD							m_AUDServer;

	/**
	 * the logging object (shared with owning AUD)
	 */
	private Log							m_Log;

	/**
	 * the port number at which we are to listen for incoming requests
	 */
	private final int					m_ListenPort;

	/**
	 * the secure connection factory used by connectors, built using our
	 * configured keystore
	 */
	private SslContextFactory.Server	m_SslFactory;

	/**
	 * the Jetty server used to listen for incoming messages
	 */
	private Server						m_JettyServer;

	/**
	 * the policy object we configure (for server side)
	 */
	private WebSocketPolicy				m_Policy;

	/**
	 * the web socket factory we specify
	 */
	private WebSocketServerFactory		m_Factory;

	/**
	 * for aggregate functions, useful to have a session tracker
	 */
	private SessionTracker				m_SessionTracker;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param aud the AUD which owns us
	 * @param log the error/debug logger (normally mvo.log())
	 * @param listenPort the port at which we're to listen for requests
	 */
	public MVOHandler(AUD aud, Log log, int listenPort) {
		super();
		m_AUDServer = aud;
		m_Log = log;
		m_ListenPort = listenPort;
		m_SslFactory = new SslContextFactory.Server();
		// customize thread pool
		QueuedThreadPool qtp = new QueuedThreadPool(64, 4, 600000);
		m_JettyServer = new Server(qtp);
	}

	/**
	 * initialize the Jetty instance
	 * @return true on success
	 */
	public boolean initialize() {
		boolean secureOk = true;
	/* NB: wss TLS negotiation is broken in Jetty!  Also, we don't want to
	 * 	   require site certificates for MVOs, since they have no domains.
	 * 	   Therefore, we comment all of this out.
		// set up the SSL connector factory based on our keystore and properties
		Properties props = m_AUDServer.getProperties();
		m_SslFactory = new SslContextFactory.Server();
		String keystorePath = props.getProperty("PeerKeyStore");
		m_SslFactory.setKeyStorePath(keystorePath);
		String keystorePass = props.getProperty("KeyStorePass");
		m_SslFactory.setKeyStorePassword(keystorePass);
		String keystoreType = props.getProperty("KeyStoreType");
		m_SslFactory.setKeyStoreType(keystoreType);
		String truststorePath = props.getProperty("TrustStore");
		m_SslFactory.setTrustStorePath(truststorePath);
		String truststoreType = props.getProperty("TrustStoreType");
		m_SslFactory.setTrustStoreType(truststoreType);
		String trustStorePass = props.getProperty("TrustStorePass");
		if (trustStorePass != null) {
			m_SslFactory.setTrustStorePassword(trustStorePass);
		}
		String keyAlias = props.getProperty("MVOKeyAlias", "mvopeers");
		m_SslFactory.setCertAlias(keyAlias);
		m_SslFactory.setNeedClientAuth(false);
		m_SslFactory.setWantClientAuth(false);
		m_SslFactory.addExcludeProtocols("SSLv2Hello", "SSLv3");
		try {
			m_SslFactory.start();
		}
		catch (Exception eee) {
			m_Log.error("Unable to start SSL factory", eee);
			secureOk = false;
		}
	 */
		secureOk = false;

		if (secureOk) {
			final NetworkTrafficServerConnector httpsConnector
				= new NetworkTrafficServerConnector(m_JettyServer,
													m_SslFactory);
			httpsConnector.setHost(null);
			httpsConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpsConnector);
		}
		else {
			NetworkTrafficServerConnector httpConnector
				= new NetworkTrafficServerConnector(m_JettyServer);
			httpConnector.setHost(null);
			httpConnector.setPort(m_ListenPort);
			m_JettyServer.addConnector(httpConnector);
		}

		// define servlet handler
		ServletHandler handler = new ServletHandler();
		m_JettyServer.setHandler(handler);
		handler.addServletWithMapping(new ServletHolder(this), "/keyserver");

		// attempt to start the server and handler
		try {
			// this call supplies a default factory
			m_JettyServer.start();
			// this call initializes the ServletContext:
			handler.initialize();
			m_Log.debug("MVOHandler.initialize: started MVOHandler WebSocket "
						+ "listener (secure=" + secureOk + ") on port "
						+ m_ListenPort);
		}
		catch (Exception e) {
			m_Log.error("MVOHandler.initialize: could not start Jetty listener "
						+ "for MVOHandler", e);
			return false;
		}
		return true;
	}

	/**
	 * shutdown Jetty listener and everything associated
	 */
	public void shutdown() {
		// shut down the main listener, to stop new connections
		if (m_JettyServer != null && m_JettyServer.isRunning()) {
			try {
				m_JettyServer.stop();
				m_Log.debug("MVOHandler.shutdown: Jetty listener stopped");
			}
			catch (Exception e) {
				m_Log.error("MVOHandler.shutdown: error stopping MVOHandler "
							+ "Jetty listener", e);
			}
		}

		// shut down the factory
		if (m_Factory != null && m_Factory.isRunning()) {
			try {
				m_Factory.stop();
				m_Log.debug("MVOHandler.shutdown: factory stopped");
			}
			catch (Exception e) {
				m_Log.error("MVOHandler.shutdown: error stopping MVOHandler "
							+ "factory", e);
			}
		}

		// NB: session closures should be reflected in the tracker total
		if (m_SessionTracker != null && m_SessionTracker.isRunning()) {
			try {
				m_Log.debug("MVOHandler.shutdown: shutting down with "
							+ m_SessionTracker.getNumSessions() + " sessions");
				m_SessionTracker.stop();
			}
			catch (Exception e) {
				m_Log.error("MVOHandler.shutdown: error stopping MVOHandler "
							+ "session tracker", e);
			}
		}

		// shut down the SSL factory
		if (m_SslFactory != null && m_SslFactory.isRunning()) {
			try {
				m_SslFactory.stop();
			}
			catch (Exception ee) {
				m_Log.error("MVOHandler.shutdown: exception stopping SSL "
							+ "factory", ee);
			}
		}
	}

	// methods to extend the WebSocketServlet class
	/**
	 * configuration method to record the factory and policy items
	 */
	@Override
	public void configure(WebSocketServletFactory factory) {
		/* Due to the fact that we're a WebSocketServlet, a default
		 * WebSocketServletFactory was created internally by super().
		 * Since we were added in a ServletHolder, when Server.start() is
		 * called (on m_JettyServer), this method will be invoked.
		 * The default factory is passed to us here, giving us the chance to
		 * record and suitably configure it.
		 */
		m_Factory = (WebSocketServerFactory) factory;
		m_Policy = m_Factory.getPolicy();

		// do policy configuration adjustments
		m_Policy.setAsyncWriteTimeout(Timer.ONE_SECOND * 30L);
		// max idle timeout = 5 ping intervals (close after 5 pings missed)
		m_Policy.setIdleTimeout(Timer.ONE_SECOND * 300L);
		m_Policy.setInputBufferSize(1024 * 1000);
		m_Policy.setMaxTextMessageBufferSize(1024 * 1000);
		m_Policy.setMaxTextMessageSize(1024 * 1000);

		// register our WebSocket class
		m_Factory.register(PeerConnector.AUDServerWebSocket.class);
		/* another way to do it:
			factory.register(
				Class.forName(
					"cc.enshroud.jetty.aud.PeerConnector$AUDServerWebSocket"));
		 */

		/* because our WebSocket class is internal, in order for the constructor
		 * to work we need to define ourselves as WebSocketCreator so we can
		 * use our own operator new
		 */
		m_Factory.setCreator(this);

		// add our own session listener
		m_SessionTracker = new SessionTracker();
		m_Factory.addSessionListener(m_SessionTracker);

		// start these objects, per their lifecycle
		try {
			m_SessionTracker.start();
			m_Factory.start();
		}
		catch (Exception e) {
			m_Log.error("MVOHandler.configure: could not start session tracker "
						+ "and/or factory", e);
		}
	}

	/**
	 * handler method for POSTs from the dApp clients
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doPost(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("MVOHandler.doPost: null AUD reqeust");
		}
		if (response == null) {
			throw new IOException("MVOHandler.doPost: null AUD response "
								+ "object");
		}
		response.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED,
							"POST not supported for this function");
	}

	/**
	 * handler method for GETs from the dApp clients (not used)
	 * @param request the immutable request object, which may have been wrapped
	 * @param response the handle for the reply we will send
	 * @throws IOException on bad input
	 * @throws ServletException per definition
	 */
	@Override
	public void doGet(HttpServletRequest request, HttpServletResponse response)
		throws IOException, ServletException
	{
		if (request == null) {
			throw new IOException("MVOHandler.doGet: null AUD request");
		}
		if (response == null) {
			throw new IOException("MVOHandler.doGet: null AUD response object");
		}

		// GET is not used
		response.sendError(HttpServletResponse.SC_METHOD_NOT_ALLOWED,
							"GET not supported for this function");
	}

	/**
	 * method to supply servlet information
	 * @return the name of the servlet (arbitrary)
	 */
	@Override
	public String getServletInfo() {
		return "AUD-Jetty:MVOHandler servlet";
	}

	// method to implement the WebSocketCreator class
	/**
	 * handler method for creating server-mode WebSocket objects
	 * @param req the request details
	 * @param resp the response details
	 * @return the object created (will be a AUDWebSocket)
	 */
	@Override
	public Object createWebSocket(ServletUpgradeRequest req,
								  ServletUpgradeResponse resp)
	{
		String mvoId = req.getHeader("PeerId");
		if (mvoId == null || mvoId.isEmpty()) {
			m_Log.error("MVOHandler.createWebSocket: no PeerId in upgrade "
						+ "request");
			return null;
		}

		// obtain any existing server websocket
		AUDState mSt = m_AUDServer.getStateObj();
		PeerConnector peerConn = mSt.getPeerConnection(mvoId);
		if (peerConn == null) {
			peerConn = new PeerConnector(m_AUDServer, mvoId);
			mSt.addPeerConnection(mvoId, peerConn);
		}

		// obtain any existing server socket
		PeerConnector.AUDServerWebSocket existSocket
			= peerConn.getServerSocket();
		if (existSocket != null && existSocket.isAuthenticated()) {
			// return this one
			return existSocket;
		}
		PeerConnector.AUDServerWebSocket socket
			= peerConn.new AUDServerWebSocket();
		// accept text but not binary
		for (String subprotocol : req.getSubProtocols()) {
			if (subprotocol.equals("text")) {
				resp.setAcceptedSubProtocol(subprotocol);
				return socket;
			}
		}
		/* NB: caller will now create a Session for this object and pass the
		 * req and resp in the context of a @OnWebSocketConnect event
		 */
		// no valid subprotocol in request, ignore
		return null;
	}

	// utility methods
	/**
	 * obtain the Jetty server
	 * @return the Jetty server object we use to handle requests
	 */
	public Server getJettyServer() { return m_JettyServer; }

	/**
	 * obtain the SSL factory (shared with ReceiptHandler)
	 * @return the factory
	 */
	public SslContextFactory.Server getSslFactory() { return m_SslFactory; }

	/**
	 * obtain the web socket factory
	 * @return the factory which constructs AUD web socket sessions
	 */
	public WebSocketServerFactory getFactory() { return m_Factory; }

	/**
	 * obtain the session tracker
	 * @return the tracker
	 */
	public SessionTracker getSessionTracker() { return m_SessionTracker; }

	/**
	 * obtain the web socket policy
	 * @return the current policy configuration (modified locally from defaults)
	 */
	public WebSocketPolicy getSocketPolicy() { return m_Policy; }


	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			m_SslFactory = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
