/*
 * last modified---
 * 	03-28-25 allow fallback to dApp-supplied AES keys on spend/withdraw inputs
 * 	02-24-25 warn about memo lines that are too long (dApp/MVO should prevent)
 * 	01-27-25 in handleWithdrawBlock(), allow for case with zero eNFT outputs
 * 			 when the inputs have been burned
 * 	01-19-24 build new output detailsHash with passed m_Rand in withdraw case
 * 	01-10-24 do not include Code 406: etc. in greylisted reasons
 * 	12-06-23 add actual on-chain greylisting of failed eNFT IDs
 * 	07-04-23 after getNFTsById() calls, check for greylisting and dwell time
 * 	06-20-23 add acct to getNFTsById() calls
 * 	06-14-23 remove AssetConfig usage; use Web3j
 * 	06-13-23 use actual verifyClientSig(), copied from MVO
 * 	04-11-23 destroy ephemeral AES keys after use
 * 	03-14-23 generic DB classes now in .db
 * 	03-06-23 use validateENFTsig() to validate signatures
 * 	03-01-23 verify sigs on passed AB blocks
 * 	02-28-23 add verification of AB sigs; validate addresses
 * 	12-06-22 handle rand term in client and MVO payees
 * 	10-20-22 complete draft
 * 	10-04-22 new (stub)
 *
 * purpose---
 * 	provide processor for all MVOAuditorBlock messages broadcast from MVOs
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.MVOAuditorBlock;
import cc.enshroud.jetty.AuditorBlock;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.eNFTmetadata;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.DbConnectionManager;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;
import cc.enshroud.jetty.aud.db.EnftStatus;
import cc.enshroud.jetty.aud.db.EnftStatusDb;
import cc.enshroud.jetty.aud.db.EnftKeysDb;
import cc.enshroud.jetty.aud.db.AESKey;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.WalletUtils;
import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Savepoint;
import java.util.Date;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.Map;
import java.util.Base64;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.math.BigInteger;
import java.net.HttpURLConnection;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import javax.security.auth.DestroyFailedException;


/**
 * This class provides the functionality of the Auditor as a validator for
 * AuditorBlockS broadcast by MVOs.  Processing resembles that performed by a
 * non-lead committee member MVO.  There is no reply to the MVO; instead
 * entries are created in the ENSH_ENFT_STATUS table indicating the valid or
 * suspect status of any output eNFTs.  In certain cases, the Auditor might
 * mark the IDs of output eNFTs as "greylisted" in the smart contract, in
 * the event that "suspect" IDs are actually minted.
 */
public final class AudBlockHandler {
	// BEGIN data members
	/**
	 * owning Auditor
	 */
	private AUD				m_AUD;

	/**
	 * error logging object (inhereted from parent)
	 */
	private Log				m_Log;

	/**
	 * database connection manager (copied from owning AUD)
	 */
	private DbConnectionManager	m_DbMgr;

	/**
	 * helper class to describe an error report generated during processing
	 */
	public final class AudErrorReport {
		/**
		 * a textual error message which describes the nature of the failure
		 */
		public String			m_FailureMsg;

		/**
		 * a numeric error code, describing how a validation failed:
		 * 406 (HTTP_NOT_ACCEPTABLE) - some aspect of request was inconsistent
		 * 502 (HTTP_BAD_GATEWAY)	 - database could not be accessed, which
		 * 							   limited the processing that could be done
		 * 500 (HTTP_INTERNAL_ERROR) - an error occurred attempting to process
		 * Only 406 results in greylisting output IDs.
		 */
		public int				m_FailureCode;

		/**
		 * nullary constructor
		 */
		public AudErrorReport() {
			m_FailureMsg = "";
		}

		/**
		 * constructor
		 * @param errMsg the textual description of the error found
		 * @param code the error code indicating severity (per above)
		 */
		public AudErrorReport(String errMsg, int code) {
			m_FailureMsg = errMsg;
			m_FailureCode = code;
		}
	}

	/**
	 * the Id of the MVO whose broadcast we're currently processing
	 */
	private String			m_MVOId;

	/**
	 * the message tag from the MVO's broadcast (AUDId:transId:seqnum format),
	 * used for sending error replies as required
	 */
	private String			m_Tag;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param aud the Auditor which owns us
	 */
	public AudBlockHandler(AUD aud) {
		m_AUD = aud;
		m_Log = m_AUD.log();
		m_DbMgr = m_AUD.getDbManager();
	}

	/**
	 * actual method to handle MVOAuditorBlock sent by an MVO
	 * @param audBlock the block from the MVO, broadcast to all Auditors
	 * @param mvoId the MVO who sent us this broadcast
	 * @return true on success, false iff we want the MVO to resend the block
	 */
	public synchronized boolean handleAuditorBlock(MVOAuditorBlock audBlock,
												   String mvoId)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".handleAuditorBlock: ";
		if (audBlock == null || mvoId == null || !mvoId.startsWith("MVO-")) {
			m_Log.error(lbl + "missing input");
			return false;
		}
		m_MVOId = mvoId;
		boolean ret = true;

		// set MVOBlock JSON string too (buildFromString() doesn't do this)
		StringBuilder mvoBlk = new StringBuilder(10240);
		AuditorBlock mvoBlock = audBlock.getMVOBlock();
		mvoBlock.addJSON(mvoBlk);
		audBlock.setMVOBlockJson(mvoBlk.toString());

		// verify that it parses back into the same object
		ClientMVOBlock cliBlock = audBlock.getClientBlock();
		StringBuilder audBlk = new StringBuilder(20480);
		audBlock.addJSON(audBlk);
		MVOAuditorBlock rebuilt = new MVOAuditorBlock(m_Log);
		if (!rebuilt.buildFromString(audBlk.toString())) {
			m_Log.error(lbl + "could not re-parse MVOAuditorBlock");
			// no point in resending, it still won't parse
			return ret;
		}

		// initialize the error report list
		ArrayList<AudErrorReport> errReports = new ArrayList<AudErrorReport>();
		String errMsg = "";
		int errCode = 0;

		// get config for chain
		long chainId = cliBlock.getChainId();
		SmartContractConfig scc = m_AUD.getSCConfig(chainId);
		if (scc == null) {
			// this should be impossible given it would be unsupported for MVOs
			errMsg = "chain Id " + chainId + " is not supported";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
			errReports.add(new AudErrorReport(errMsg, errCode));
			// however, must abandon processing here
			return ret;
		}

		// make sure SCC is enabled (i.e. RPC-JSON client is working)
		if (!scc.isEnabled()) {
			errMsg = "chain Id " + chainId + " support not currently available";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			// however, must abandon processing here
			return ret;
		}

		// verify signature on client block
		if (!verifyClientSig(audBlock.getClientBlockJson(), cliBlock)) {
			errMsg = "client signature did not verify";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}

		StringBuilder AB = new StringBuilder(2048);
		audBlock.addJSON(AB);
	/*
		m_Log.debug(lbl + "processing request from " + m_MVOId
					+ ", AuditorBlock data =\n" + AB);
	 */

		// verify signature on MVO's block
		ArrayList<MVOSignature> mvoSigs = mvoBlock.getSignatures();
		MVOSignature mvoSig = MVOSignature.findMVOSig(m_MVOId, mvoSigs);
		if (mvoSig == null) {
			errMsg = "no sig of MVO " + m_MVOId + " on AuditorBlock";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		else {
			// get their signing key for this blockchain
			MVOConfig mvoConf
				= (MVOConfig) scc.getMVOMap().get(mvoSig.m_Signer);
			if (mvoConf == null) {
				errMsg = "no config for MVO " + m_MVOId + ", cannot check sig";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
			else {
				String signingAddress = mvoConf.getSigningAddress();

				// verify this address signed the passed AB
				String signedData = mvoBlock.buildSignedData();
				String sigAddress
					= EncodingUtils.getDataSignerAddress(signedData,
														 mvoSig.m_Signature);
				if (sigAddress.isEmpty()
					|| !sigAddress.equalsIgnoreCase(signingAddress))
				{
					errMsg = "signature does not match for AuditorBlock "
							+ "from MVOId " + m_MVOId;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// also verify their signature on the internal ArgumentsBlock
				AuditorBlock.ArgumentsHash argsHash = mvoBlock.getArgsBlock();
				sigAddress
					= EncodingUtils.getHashSignerAddress(argsHash.m_ArgsHash,
														 mvoSig.m_ArgsSig);
				if (sigAddress.isEmpty()
					|| !sigAddress.equalsIgnoreCase(signingAddress))
				{
					errMsg = "argsHash signature does not match for "
							+ "AuditorBlock from MVOId " + m_MVOId;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
			}
		}

		// branch to sub-handler based on opcode
		String opcode = mvoBlock.getOpcode();
		if (!opcode.equals(cliBlock.getOpcode())) {
			errMsg = "client and MVO opcodes do not match";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		// proceed to opcode-specific validation
		else {
			ArrayList<AudErrorReport> subErrs = null;
			if (opcode.equals(cliBlock.M_OpDeposit)) {
				subErrs = handleDepositBlock(cliBlock, mvoBlock);
			}
			else if (opcode.equals(cliBlock.M_OpSpend)) {
				subErrs = handleSpendBlock(cliBlock, mvoBlock);
			}
			else if (opcode.equals(cliBlock.M_OpWithdraw)) {
				subErrs = handleWithdrawBlock(cliBlock, mvoBlock);
			}
			else {
				errMsg = "illegal opcode, " + opcode;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
			if (subErrs != null) {
				// add to top-level list
				errReports.addAll(subErrs);
			}
		}

	/*
		//TEMPCODE
		// gin up a temporary problem for testing greylisting
		if (errReports.isEmpty()) {
			AudErrorReport aer
				= new AudErrorReport("FALSE error for greylist test",
									HttpURLConnection.HTTP_NOT_ACCEPTABLE);
			errReports.add(aer);
		}
		//TEMPCODE
	 */
		m_Log.debug(lbl + "got error list with " + errReports.size()
					+ " entries in it from " + mvoId);

		/* based on whether a problem was found or not, set all the output
		 * eNFTs to the ENSH_ENFT_STATUS table with appropriate status
		 */
		EnftStatusDb statDb = new EnftStatusDb(m_DbMgr, m_Log);
		ArrayList<AuditorBlock.ClientPayee> abPayees = mvoBlock.getPayees();
		// bail if no outputs
		if (abPayees.isEmpty()) {
			// a withdrawal can validly have zero output payees
			if (opcode.equals(cliBlock.M_OpWithdraw)) {
				if (!errReports.isEmpty()) {
					m_Log.error(lbl + "errors present on Withdraw, but no "
								+ "output IDs exist to apply status to");
				}
			}
			else {
				errMsg = "missing output IDs, " + opcode + " block from "
						+ m_MVOId;
				m_Log.error(lbl + errMsg);
			}
			// with or without errors, continuing below will achieve nothing
			return ret;
		}

		// build list of output IDs to which our error list applies
		ArrayList<String> outputIDs = new ArrayList<String>(abPayees.size());
		for (AuditorBlock.ClientPayee payee : abPayees) {
			outputIDs.add(payee.m_ID);
		}

		// start a transaction with a savepoint
		Connection dbConn = null;
		Savepoint savePt = null;
		boolean needRollback = false;
		// get a DB connection on which we can unset auto-commit
		try {
			dbConn = m_DbMgr.getConnection();
			dbConn.setAutoCommit(false);
			long savId = m_AUD.getStateObj().getNextAUDid();
			savePt = dbConn.setSavepoint("audStatus" + savId);
		}
		catch (SQLException se) {
			m_Log.error(lbl + "can't turn off commit for STATUS trans", se);
			return false;
		}

		// find out if these already exist in the database
		ArrayList<EnftStatus> existRecs
			= statDb.getStatus(outputIDs, chainId, dbConn);
		String existStat = "";
		if (existRecs == null) {
			// got a DB connect or other DB error
			errMsg = "error looking for existing STATUS records for output IDs";
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
		}
		if (existRecs.size() == outputIDs.size()) {
			// all records should have the same status, but check to confirm
			for (EnftStatus statRec : existRecs) {
				if (existStat.isEmpty()) {
					existStat = statRec.getStatus();
				}
				else if (!existStat.equals(statRec.getStatus())) {
					errMsg = "inconsistent existing status found on ID "
							+ statRec.getID();
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
			}
		}
		final String us = m_AUD.getAUDId();

		/* If all we have are HTTP_BAD_GATEWAY errors, we can ask for retry.
		 * Otherwise if we have only HTTP_INTERNAL_ERROR errors, do nada.
		 * Else add all outputs to 'suspect' list.
		 */
		boolean gotGtwy = false;
		boolean gotInternal = false;
		boolean gotUnacceptable = false;
		// build narrative (same for all IDs)
		StringBuilder narrative = new StringBuilder(10240);
		narrative.append("Auditor " + us + " report, errors found:\n");
		narrative.append("***BEGIN***\n");
		boolean gotDbErr = false;
		for (AudErrorReport report : errReports) {
			switch (report.m_FailureCode) {
				case HttpURLConnection.HTTP_NOT_ACCEPTABLE:
					gotUnacceptable = true;
					break;
				case HttpURLConnection.HTTP_INTERNAL_ERROR:
					gotInternal = true;
					break;
				case HttpURLConnection.HTTP_BAD_GATEWAY:
					gotGtwy = true;
					break;
				default:
					m_Log.error(lbl + "weird error status ("
								+ report.m_FailureCode + ") on AB block");
					break;
			}
			narrative.append("Code " + report.m_FailureCode + ": "
							+ report.m_FailureMsg + "\n");
		}
		narrative.append("***END***");
		if (!errReports.isEmpty()) {
			m_Log.debug(lbl + "errors: gtwy=" + gotGtwy + ", int="
						+ gotInternal + ", bad=" + gotUnacceptable);
		}

		// if we got no errors except gtwy errors, try to set outputs to valid
		if (errReports.isEmpty()
			|| (gotGtwy && !gotInternal && !gotUnacceptable))
		{
			// add all outputs to 'valid' list
			boolean statsOk = true;
			for (String outId : outputIDs) {
				// if existing records have status, no need to do loop
				if (!existStat.isEmpty()) {
					// no need to log complaint if status valid or burned
					if (!EnftStatus.M_Valid.equals(existStat)
						&& !EnftStatus.M_Deleted.equals(existStat))
					{
						// some other AUD marked an error, we cannot override
						m_Log.error(lbl + "cannot mark an ID list valid which "
									+ "is already marked invalid");
					}
					// NB: break, not continue, because all have same status
					break;
				}
				try {
					// get the corresponding output and find details hash
					String detHash = "";
					for (AuditorBlock.ClientPayee payee : abPayees) {
						if (outId.equals(payee.m_ID)) {
							if (opcode.equals(cliBlock.M_OpWithdraw)) {
								// NB: m_DetailsHash not set on a withdrawal
								detHash = EncodingUtils.buildDetailsHash(
															payee.m_Address,
															payee.m_ID,
															payee.m_OutputAsset,
											payee.m_OutputAmount.toString(16),
															payee.m_Rand);
							}
							else {
								detHash = payee.m_DetailsHash;
							}
							break;
						}
					}
					if (!statDb.setStatus(outId, chainId, us, detHash, null,
										  "", dbConn))
					{
						m_Log.error(lbl
									+ "error inserting STATUS record for ID "
									+ outId);
						statsOk = false;
						break;
					}
				}
				catch (EnshDbException edbe) {
					statsOk = false;
					m_Log.error(lbl
								+ "exception inserting STATUS record for ID "
								+ outId + ": " + edbe.toString());
					break;
				}
			}
			if (!statsOk) {
				// roll back DB changes
				needRollback = true;
				// counts as HTTP_BAD_GATEWAY; ask MVO to send to us again
				gotGtwy = true;
			}
			if (gotGtwy) {
				m_Log.debug(lbl + "asking MVO " + m_MVOId + " for AB retry");
				ret = false;
			}
		}
		else {
			// one or more errors present
			if (gotUnacceptable) {
				String reason = narrative.toString();
				ArrayList<BigInteger> greylistIds
					= new ArrayList<BigInteger>(outputIDs.size());
				// greylist all IDs with same report as reason
				for (String outId : outputIDs) {
					// record BigInteger value of outId for greylisting
					BigInteger outEid = Numeric.parsePaddedNumberHex(outId);
					greylistIds.add(outEid);

					// get the corresponding output and build details hash
					String detHash = "";
					for (AuditorBlock.ClientPayee payee : abPayees) {
						if (outId.equals(payee.m_ID)) {
							if (opcode.equals(cliBlock.M_OpWithdraw)) {
								// NB: m_DetailsHash not set on a withdrawal
								detHash = EncodingUtils.buildDetailsHash(
															payee.m_Address,
															payee.m_ID,
															payee.m_OutputAsset,
											payee.m_OutputAmount.toString(16),
															payee.m_Rand);
							}
							else {
								detHash = payee.m_DetailsHash;
							}
							break;
						}
					}
					try {
						// if we have existing records, use update
						if (!existStat.isEmpty()) {
							// illegal to override blocked/deleted with suspect
							if (EnftStatus.M_Blocked.equals(existStat)
								|| EnftStatus.M_Deleted.equals(existStat))
							{
								m_Log.error(lbl + "ignore attempt to update ID "
											+ outId + " to "
											+ EnftStatus.M_Suspect);
								continue;
							}
							if (!statDb.updStatus(outId,
												  chainId,
												  us,
												  EnftStatus.M_Suspect,
												  reason,
												  dbConn))
							{
								gotDbErr = true;
								m_Log.error(lbl + "error updating suspect "
											+ "status for ID " + outId);
								break;
							}
							else {
								m_Log.debug(lbl + "updated to suspect stat ID "
											+ outId);
							}
						}
						else {
							// NB: setStatus() does nothing if status is same
							if (!statDb.setStatus(outId,
												  chainId,
												  us,
												  detHash,
												  EnftStatus.M_Suspect,
												  reason,
												  dbConn))
							{
								gotDbErr = true;
								m_Log.error(lbl + "error setting suspect "
											+ "status for ID " + outId);
								break;
							}
							else {
								m_Log.debug(lbl + "set suspect stat ID "
											+ outId);
							}
						}
					}
					catch (EnshDbException edbe) {
						m_Log.error(lbl + "exception changing STATUS record "
									+ "for ID " + outId);
						gotDbErr = true;
						break;
					}
				}
				if (gotDbErr) {
					needRollback = true;
				}

				/* Regardless of any DB errors, mark all outputIDs as
				 * greyListed on-chain.  SCC is enabled, thus EnftCache and
				 * GreyLister both must exist.
				 */
				GreyLister greyLister
					= m_AUD.getConfig().getGreyLister(chainId);
				if (greyLister == null) {
					m_Log.error(lbl + "CRITICAL: no GreyLister defined for "
								+ "chainId " + chainId);
					ret = false;
				}
				else {
					// to save gas on chain, strip boilerplate out of narrative
					int startTagIdx = reason.indexOf("***BEGIN***\n");
					int endTagIdx = reason.indexOf("\n***END***");
					String chainReason
						= reason.substring(startTagIdx+12, endTagIdx);
					// also strip "Code NNN:" out of reason
					String codelessReason
						= chainReason.replaceAll("Code [0-9][0-9][0-9]:", "");
					
					// define bucket to be greylisted if any ID is ever minted
					greyLister.addBucket(greylistIds, codelessReason);
				}
			}
			// NB: do nothing if error was INTERNAL (nothing we can do about it)
		}
		if (needRollback) {
			try {
				dbConn.rollback(savePt);
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error rolling back to save point!", se);
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
				// counts as HTTP_BAD_GATEWAY
				ret = false;
			}
		}
		else {
			try {
				dbConn.commit();
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error committing STATUS transaction: "
							+ se.toString());
				// counts as HTTP_BAD_GATEWAY
				ret = false;
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
			}
		}
		return ret;
	}

	/**
	 * sub-processor for deposit operations
	 * @param clientBlock the original signed request sent by dApp client
	 * @param audBlock the detailed OB prepared and signed by the MVO
	 * @return the list of errors we generated (empty if none)
	 */
	private ArrayList<AudErrorReport> handleDepositBlock(
													ClientMVOBlock clientReq,
									   				AuditorBlock audBlock)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".handleDepositBlock: ";
		ArrayList<AudErrorReport> errReports = new ArrayList<AudErrorReport>();
		String errMsg = "";
		int errCode = 0;

		// begin by validating the original user request
		long chainId = clientReq.getChainId();
		SmartContractConfig scc = m_AUD.getSCConfig(chainId);
		// NB: scc == null was checked and eliminated in caller

		// confirm depositor address is valid
		if (!WalletUtils.isValidAddress(clientReq.getSender())) {
			errMsg = "sender address (" + clientReq.getSender()
					+ ") appears invalid";
			m_Log.error(lbl + "req verify failure: " + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}

		// check asset type is supported
		String asset = clientReq.getAsset();
		BigInteger amt = clientReq.getAmount();
		boolean valid = true;

		// check overall asset type matches
		if (!asset.equalsIgnoreCase(audBlock.getAsset())) {
			errMsg = "deposit asset (" + asset + ") conflicts with AB, "
							+ "which has " + audBlock.getAsset();
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			valid = false;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		// these issues are serious enough to bail right here
		if (!valid) return errReports;

		// ensure that all the client payee addresses/amounts appear valid
		ArrayList<ClientMVOBlock.ClientPayee> payees = clientReq.getPayees();
		boolean badAddress = false;
		BigInteger outputSum = BigInteger.ZERO;
		final BigInteger one100 = new BigInteger("100");
		for (ClientMVOBlock.ClientPayee payee : payees) {
			if (!payee.m_Address.isEmpty()
				&& !WalletUtils.isValidAddress(payee.m_Address))
			{
				badAddress = true;
				errMsg = "payee address " + payee.m_Address
						+ " appears invalid";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* Tally output amounts.  Note that the value of the m_Units field
			 * dictates whether amount is absolute or a percentage of the total.
			 */
			if (payee.m_Units.isEmpty()) {
				outputSum = outputSum.add(payee.m_OutputAmount);
			}
			else if (payee.m_Units.equals("%")) {
				// compute the percentage of the total
				BigInteger incr = amt.multiply(payee.m_OutputAmount);
				incr = incr.divide(one100);
				outputSum = outputSum.add(incr);
			}
			else {
				m_Log.error(lbl + "bad units on output " + payee.m_Payee
							+ ", ignoring amount");
			}
		}
		if (badAddress) {
			errMsg = "one or more payee addresses are invalid";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			valid = false;
		}

		// ensure that deposit amount offered matches outputs
		if (outputSum.compareTo(amt) != 0) {
			errMsg = "output amounts do not match total ("
					+ outputSum + " versus " + amt + ")";
			m_Log.error(lbl + errMsg);
			valid = false;
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		// these errors are serious enough to bail already
		if (!valid) return errReports;

		// obtain keys for all output eNFTs
		ArrayList<AuditorBlock.ClientPayee> abPayees = audBlock.getPayees();
		boolean noId = false;
		if (abPayees.isEmpty()) {
			errMsg = "output payees not found";
			m_Log.error(lbl + errMsg);
			noId = true;
		}
		ArrayList<String> keyHashes = new ArrayList<String>(abPayees.size());
		for (AuditorBlock.ClientPayee payee : abPayees) {
			if (payee.m_ID.isEmpty()) {
				noId = true;
				continue;
			}

			// compute chainId+ID+address hash and add to list
			ArrayList<String> hashComps = new ArrayList<String>(3);
			hashComps.add(Long.toString(chainId));
			hashComps.add(payee.m_ID);
			hashComps.add(payee.m_Address.toLowerCase());
			String keyIdx = String.join("+", hashComps);
			String keyHash = EncodingUtils.sha3(keyIdx);
			keyHashes.add(keyHash);
		}
		if (noId) {
			errMsg = "missing passed payee ID list in AuditorBlock";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			return errReports;
		}

		// fetch these keys from the database
		EnftKeysDb ekDb = new EnftKeysDb(m_DbMgr, m_Log);
		ArrayList<AESKey> aesKeys = ekDb.getKeys(keyHashes);
		if (aesKeys == null || aesKeys.size() != abPayees.size()) {
			errMsg = "error fetching output eNFT keys; ABORT!";
			m_Log.error(lbl + errMsg);
			// this is an internal error, doesn't mean anything wrong with input
			if (aesKeys == null) {
				// DB could not be accessed
				errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			}
			else {
				// not all keys were retrieved
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
			}
			errReports.add(new AudErrorReport(errMsg, errCode));
			return errReports;
		}

		// compare original client request payee outputs with AuditorBlock's
		boolean gotOutputErr = false;
		ArrayList<ClientMVOBlock.ClientPayee> orgOutputs
			= clientReq.getPayees();
		errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
		// make certain every original output is found in the AB
		for (ClientMVOBlock.ClientPayee orgOut : orgOutputs) {
			boolean present = false;
			AuditorBlock.ClientPayee abPayee = null;
			for (AuditorBlock.ClientPayee output : abPayees) {
				// strip off "payee"
				String ptag = orgOut.m_Payee.substring(5);
				if (ptag.equals(output.m_Payee)) {
					present = true;
					abPayee = output;
					break;
				}
			}
			if (!present) {
				errMsg = "client payee " + abPayee.m_Payee + " not found in AB";
				gotOutputErr = true;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// compare address
			if (!abPayee.m_Address.equalsIgnoreCase(orgOut.m_Address)) {
				errMsg = "AB payee " + abPayee.m_Payee
								+ " involves incorrect address, "
								+ abPayee.m_Address + " vs. "
								+ orgOut.m_Address;
				gotOutputErr = true;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// compare amount and asset (AB has these where OB does not)
			if (!abPayee.m_OutputAsset.equalsIgnoreCase(asset)) {
				errMsg = "AB payee " + abPayee.m_Payee
							+ " involves incorrect asset, "
							+ abPayee.m_OutputAsset;
				gotOutputErr = true;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
			BigInteger orgAmt = orgOut.m_OutputAmount;
			if (orgOut.m_Units.equals("%")) {
				// scale as percentage of total
				BigInteger pAmt = orgAmt.multiply(clientReq.getAmount());
				orgAmt = pAmt.divide(one100);
			}
			if (abPayee.m_OutputAmount.compareTo(orgAmt) != 0) {
				errMsg = "AB payee " + abPayee.m_Payee
								+ " is for an incorrect amount, "
								+ abPayee.m_OutputAmount
								+ " vs. client " + orgAmt;
				gotOutputErr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				m_Log.error(lbl + errMsg);
			}

			// compare randomizer value
			if (!orgOut.m_Rand.equals(abPayee.m_Rand)) {
				errMsg = "AB payee " + abPayee.m_Payee + " has salt = "
						+ abPayee.m_Rand + " vs. client " + orgOut.m_Rand;
				gotOutputErr = true;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		if (gotOutputErr) {
			return errReports;
		}

		/* Now we must validate every AB output payee versus the eNFT attached
		 * to it.  To do this, we must decrypt the encrypted text using the
		 * corresponding key we looked up above.  These must also match any
		 * existing ENSH_ENFT_STATUS records that might exist (these will be
		 * located if this request comes from a committee (!lead) MVO).
		 */
		// first, build list of IDs and get any STATUS records that exist
		ArrayList<String> outputIDs = new ArrayList<String>(abPayees.size());
		for (AuditorBlock.ClientPayee payee : abPayees) {
			outputIDs.add(payee.m_ID);
		}
		EnftStatusDb statDb = new EnftStatusDb(m_DbMgr, m_Log);
		ArrayList<EnftStatus> existRecs
			= statDb.getStatus(outputIDs, chainId, null);
		if (existRecs == null) {
			// this error means no DB access or DB error on fetch
			errMsg = "error looking for STATUS records for deposit IDs";
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			return errReports;
		}
		if (existRecs.size() < outputIDs.size() && !existRecs.isEmpty()) {
			m_Log.warning(lbl + "fewer STATUS records found than output IDs, "
						+ existRecs.size() + " of " + outputIDs.size());
		}

		// loop through all AB payee outputs
		boolean gotNFTerr = false;
		for (AuditorBlock.ClientPayee outp : abPayees) {
			// get the original client request payee that matches
			ClientMVOBlock.ClientPayee orgOutput = null;
			for (ClientMVOBlock.ClientPayee orgPayee : orgOutputs) {
				if (orgPayee.m_Payee.equals("payee" + outp.m_Payee)) {
					orgOutput = orgPayee;
					break;
				}
			}
			if (orgOutput == null) {
				// we don't have an original payee like this (error for deposit)
				errMsg = "AB payee " + outp.m_Payee
						+ " not found in original client request";
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// parse outer NFT with {enshrouded} data encrypted 
			NFTmetadata nft = new NFTmetadata(m_Log);
			if (!nft.buildFromJSON(outp.m_EncMetadata)) {
				errMsg = "could not parse eNFT in AB, bad metadata";
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// obtain and decrypt the {enshrouded} data
			eNFTmetadata metadata = nft.getProperties();
			if (metadata == null || !metadata.isEncrypted()) {
				errMsg = "eNFT metadata not found or not encrypted";
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			/* decrypt the entire thing including the signature,
			 * using the AES key corresponding to the hash:
			 * chainId+ID+address
			 */
			ArrayList<String> hashComps = new ArrayList<String>(3);
			hashComps.add(Long.toString(chainId));
			hashComps.add(outp.m_ID);
			hashComps.add(outp.m_Address.toLowerCase());
			String keyIdx = String.join("+", hashComps);
			String keyHash = EncodingUtils.sha3(keyIdx);
			// get key from retrieved list
			AESKey aesSpec = null;
			for (AESKey aKey : aesKeys) {
				if (aKey.getHash().equals(keyHash)) {
					aesSpec = aKey;
					break;
				}
			}
			if (aesSpec == null) {
				errMsg = "no AES key found for hash index " + keyIdx
								+ ", cannot decrypt eNFT with it";
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// make key from this key data
			Base64.Decoder b64d = Base64.getUrlDecoder();
			byte[] keyData = b64d.decode(aesSpec.getKey());
			SecretKey secretKey = new SecretKeySpec(keyData, "AES");
			String encMeta = metadata.getEnshrouded();
			String decMeta = EncodingUtils.decWithAES(secretKey,
													  encMeta);
			if (decMeta == null) {
				errMsg = "error decrypting eNFT metadata for ID "
								+ outp.m_ID;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			try {
				secretKey.destroy();
			}
			catch (DestroyFailedException dfe) { /* ignore */ }
			eNFTmetadata eNFTdecMetadata = new eNFTmetadata(m_Log);
			String parseTxt = "{" + decMeta + "}";
			Object parseObj = null;
			try {
				parseObj = JSON.parse(parseTxt);
			}
			catch (IllegalStateException ise) { /* log below */ }
			if (!(parseObj instanceof Map)) {
				errMsg = "eNFT metdata parse error";
				m_Log.error(lbl + "error parsing JSON: \""
							+ parseTxt + "\"");
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			Map parseMap = (Map) parseObj;
			if (!eNFTdecMetadata.buildFromMap(parseMap)) {
				errMsg = "eNFT metdata parse error";
				m_Log.error(lbl + "error parsing eNFT map");
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// get the signer (lead MVO) and check their signature on the eNFT
			String signer = eNFTdecMetadata.getSigner();
			// get their signing address for this blockchain
			MVOConfig leadConf = (MVOConfig) scc.getMVOMap().get(signer);
			if (leadConf == null) {
				errMsg = "cannot find config for lead MVO signer, "
								+ signer;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			String leadSigAddr = leadConf.getSigningAddress();

			// verify sig of leadSigAddr on decrypted eNFT
			if (!validateENFTsig(eNFTdecMetadata, leadSigAddr)) {
				errMsg = "eNFT signature validation failure on output ID "
						+ eNFTdecMetadata.getID();
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* We now have a decrypted version of the eNFT data.
			 * Compare each relevant bit against the output data,
			 * to ensure the lead MVO pulled no shenaningans.
			 */
			// check ID value
			if (!eNFTdecMetadata.getID().equals(outp.m_ID)) {
				// falsifying the ID of the eNFT is very serious...
				errMsg = "CRIT - output eNFT with "
								+ "ID = " + eNFTdecMetadata.getID()
								+ " while AB has " + outp.m_ID;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check owner
			if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(outp.m_Address)) {
				// falsifying the payee is also very serious...
				errMsg = "CRIT - output eNFT with payee "
								+ eNFTdecMetadata.getOwner()
								+ " while AB has " + outp.m_Address;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check asset
			if (!eNFTdecMetadata.getAsset().equalsIgnoreCase(asset)) {
				// changing asset is also very serious...
				errMsg =  "CRIT - output eNFT with asset "
								+ eNFTdecMetadata.getAsset()
								+ " while AB has " + asset;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check amount
			if (!outp.m_OutputAmount.equals(eNFTdecMetadata.getAmount())) {
				errMsg = "CRIT - output eNFT with amount "
								+ eNFTdecMetadata.getAmount().toString()
								+ " while AB output has " + outp.m_OutputAmount;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check randomizer value
			if (!outp.m_Rand.equals(eNFTdecMetadata.getRand())) {
				errMsg = "CRIT - output eNFT with salt "
								+ eNFTdecMetadata.getRand()
								+ " while AB output has " + outp.m_Rand;
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// make sure the details hash computes correctly
			String hashStr
				= EncodingUtils.buildDetailsHash(outp.m_Address,
												 outp.m_ID,
												 asset,
											outp.m_OutputAmount.toString(16),
												 outp.m_Rand);
			// check it matches
			if (!outp.m_DetailsHash.equals(hashStr)) {
				errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID()
							+ ", details hash doesn't match OB";
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// if we have a STATUS record for this ID, must also match
			EnftStatus existStat = null;
			for (EnftStatus stat : existRecs) {
				if (stat.getID().equals(outp.m_ID)) {
					existStat = stat;
					break;
				}
			}
			if (existStat != null) {
				// hash must match this record also
				if (!outp.m_DetailsHash.equals(existStat.getDetailsHash())) {
					errMsg = "CRIT - output eNFT with ID " + outp.m_ID
							+ ", details hash doesn't match STATUS record";
					m_Log.error(lbl + errMsg);
					gotNFTerr = true;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
			}

			// on a deposit, generation should be 1
			if (eNFTdecMetadata.getGeneration() != 1) {
				errMsg = "illegal generation, ID "
								+ eNFTdecMetadata.getID() + ": "
								+ eNFTdecMetadata.getGeneration();
				m_Log.error(lbl + errMsg);
				gotNFTerr = true;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// verify the memo line, if present, has not changed
			String memo = orgOutput.m_Memo;
			if (!memo.isEmpty()) {
				String eMemo = eNFTdecMetadata.getMemo();
				if (!eMemo.equals(memo)) {
					errMsg = "memo on eNFT, ID "
									+ eNFTdecMetadata.getID()
									+ " doesn't match org client req";
					m_Log.error(lbl + errMsg);
					gotNFTerr = true;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
				// warn about excessive length (shouldn't happen)
				if (eMemo.length() > eNFTmetadata.M_MEMO_MAX) {
					m_Log.warning(lbl + "memo on eNFT, ID "
								+ eNFTdecMetadata.getID() + " too long, "
								+ eMemo.length() + " chars");
				}
			}

			if (!gotNFTerr) {
				// TBD: if required, deal with expiration/growth/cost, etc.
			}
			else {
				errMsg = "one or more eNFT outputs inconsistent";
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		return errReports;
	}

	/**
	 * sub-processor for spend operations
	 * @param clientBlock the original signed request sent by dApp client
	 * @param audBlock the detailed OB prepared and signed by the MVO
	 * @return list of error reports we generated, empty if none
	 */
	private ArrayList<AudErrorReport> handleSpendBlock(
													ClientMVOBlock clientReq,
									 				AuditorBlock audBlock)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".handleSpendBlock: ";
		ArrayList<AudErrorReport> errReports = new ArrayList<AudErrorReport>();
		String errMsg = "";
		int errCode = 0;

		// begin by validating the original user request
		long chainId = clientReq.getChainId();
		SmartContractConfig scc = m_AUD.getSCConfig(chainId);
		// NB: scc == null was checked and eliminated in caller

		// confirm payer address is valid
		if (!WalletUtils.isValidAddress(clientReq.getSender())) {
			errMsg = "sender address (" + clientReq.getSender()
					+ ") appears invalid";
			m_Log.error(lbl + "req verify failure: " + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}

		final BigInteger one100 = new BigInteger("100");
		final String chgMemo = "auto-generated change amount";
		// original client inputs and outputs
		ArrayList<ClientMVOBlock.ClientInput> inputs = clientReq.getInputs();
		ArrayList<ClientMVOBlock.ClientPayee> payees = clientReq.getPayees();
		// NB: empty inputs or payees is disallowed by ClientMVOBlock parser

		// AB versions of inputs and outputs (should match client + change back)
		ArrayList<AuditorBlock.ClientInput> abInputs = audBlock.getInputs();
		// NB: empty abInputs is disallowed by AuditorBlock parser
		ArrayList<AuditorBlock.ClientPayee> abPayees = audBlock.getPayees();

		// list of assets and totals found in the input eNFTs (asset to amount)
		Hashtable<String, BigInteger> inputAssets
			= new Hashtable<String, BigInteger>();
		// list of assets and totals to be emitted as output eNFTs (asset/amt)
		Hashtable<String, BigInteger> outputAssets
			= new Hashtable<String, BigInteger>();
		// list of change outputs to catch any overage (assets mapped to amts)
		Hashtable<String, BigInteger> changeOutputs
			= new Hashtable<String, BigInteger>();
		// list of input eNFTs retrieved from on-chain
		ArrayList<NFTmetadata> fetchedInputs
			= new ArrayList<NFTmetadata>(inputs.size());
		// list of IDs of all input eNFTs
		ArrayList<String> idList = new ArrayList<String>(inputs.size());
		// list of IDs of all output eNFTs
		ArrayList<String> outList = new ArrayList<String>(abPayees.size());
		// lists of key hashes we'll need to fetch from database
		ArrayList<String> inpKeyHashes = new ArrayList<String>(inputs.size());
		ArrayList<String> outKeyHashes = new ArrayList<String>(abPayees.size());

		// AB and clientReq should have same number of inputs (outputs can vary)
		if (abInputs.size() != inputs.size()) {
			errMsg = "AB has " + abInputs.size() + " inputs but client block "
					+ "has " + inputs.size();
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			return errReports;
		}

		/* Check asset types is/are supported.  Build a tally of the amounts of
		 * each asset in the inputs.  Make a list of the IDs of all input eNFTs.
		 */
		boolean assetsOk = true;
		for (ClientMVOBlock.ClientInput reqInput : inputs) {
			String id = reqInput.m_eNFT.getID();
			// check for dup inputs
			if (idList.contains(id)) {
				errMsg = "input ID " + id + " included twice";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			idList.add(id);
			String asst = reqInput.m_eNFT.getAsset();
			BigInteger inpAmt = reqInput.m_eNFT.getAmount();
			if (!inputAssets.containsKey(asst)) {
				// initialize with amount
				inputAssets.put(asst, inpAmt);
			}
			else {
				// add amount
				BigInteger amt = inputAssets.get(asst);
				BigInteger sum = amt.add(inpAmt);
				inputAssets.replace(asst, sum);
			}

			// compute key hash and add to list
			ArrayList<String> hashComps = new ArrayList<String>(3);
			hashComps.add(Long.toString(chainId));
			hashComps.add(id);
			hashComps.add(clientReq.getSender().toLowerCase());
			String keyIdx = String.join("+", hashComps);
			String keyHash = EncodingUtils.sha3(keyIdx);
			inpKeyHashes.add(keyHash);
		}

		/* Before proceeding any further, validate that all inputs are
		 * still circulating.  We could simply trust the signature of the
		 * issuing MVO on the passed eNFT input, but we cannot trust
		 * that it is indeed still circulating and hasn't been burned.
		 * (Particularly since that status could change during the system's
		 * processing of a transaction.)  This check must also take into
		 * account possible greylisting and insufficient dwell time.
		 *
		 * Note however that because an MVO could have sent us this AB after
		 * the transaction was mined, it might be normal and expected that the
		 * input eNFTs have been burnt.  Therefore we cannot greylist outputs
		 * solely on the basis that an input was burnt.  To distinguish
		 * between these cases, we need to do a further check: see if it's the
		 * case that *all* inputs are burned, while outputs are now minted.  In
		 * this case the results are to be expected.
		 */
		BlockchainAPI web3j = m_AUD.getWeb3(chainId);
		Future<ArrayList<NFTmetadata>> nftFut
			= web3j.getNFTsById(chainId, scc.getABI(),
								clientReq.getSender(), idList, false);
		try {
			fetchedInputs = nftFut.get();
			if (fetchedInputs != null && fetchedInputs.isEmpty()) {
				// determine whether *all* inputs are found if burned allowed
				nftFut = web3j.getNFTsById(chainId, scc.getABI(),
										   clientReq.getSender(), idList, true);
				fetchedInputs = nftFut.get();
				if (fetchedInputs.size() == inputs.size()) {
					// look for outputs (burned or not)
					ArrayList<NFTmetadata> fetchedOutputs = null;
					Future<ArrayList<NFTmetadata>> outFut
						= web3j.getNFTsById(chainId, scc.getABI(),
											clientReq.getSender(), outList,
											true);
					fetchedOutputs = outFut.get();
					if (fetchedOutputs.size() < outList.size()) {
						// okay, this is an error
						errMsg = "one or more eNFTs not found on chain";
						m_Log.error(lbl + errMsg);
						errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
						assetsOk = false;
						errReports.add(new AudErrorReport(errMsg, errCode));
					}
					else {
						m_Log.debug(lbl + "hit corner case where inputs are "
									+ "burned but outputs are found onchain");
					}
				}
			}
		}
		catch (InterruptedException | ExecutionException
			   | CancellationException e)
		{
			errMsg = "could not confirm input eNFTs on chain";
			m_Log.error(lbl + errMsg, e);
			assetsOk = false;
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			//scc.setEnabled(false);
		}

		// check for possible greylisting and insufficient dwell time
		EnftCache eCache = scc.getCache();
		if (eCache != null) {
			boolean greyListedInput = false;
			boolean unconfInput = false;
			BigInteger curBlock = eCache.getLatestBlock();
			EnshroudProtocol protoWrapper = eCache.getProtocolWrapper();
			for (String Id : idList) {
				// see if ID is in the greylisted list in cache
				BigInteger eId = Numeric.parsePaddedNumberHex(Id);
				if (eCache.isIdGreyListed(eId)) {
					/* This means it's greylisted on-chain in a past block (not
					 * just in our local AudDb).  In this case either a client
					 * is trying shenanigans, or another AUD beat us to it a
					 * while back.  Either way, no point in further processing.
					 */
					errMsg = "greylisted input eNFT, ID " + Id;
					m_Log.error(lbl + errMsg);
					greyListedInput = true;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					break;
				}

				// see if ID is vested/confirmed at this time
				BigInteger unlockBlock = null;
				try {
					unlockBlock = protoWrapper.enftUnlockTime(eId).send();
				}
				catch (Exception eee) {
					m_Log.error(lbl + "unable to obtain enftUnlockTime "
								+ "for ID " + Id, eee);
				}
				if (unlockBlock == null || unlockBlock.compareTo(curBlock) > 0)
				{
					errMsg = "unconfirmed input eNFT, ID " + Id;
					m_Log.error(lbl + errMsg);
					unconfInput = true;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					break;
				}
			}
			if (greyListedInput || unconfInput) {
				assetsOk = false;
			}
		}
		// else: we have no cache to check against for this chain

		if (!assetsOk) {
			// no point in further processing
			return errReports;
		}

		// build list of output eNFT Ids
		for (AuditorBlock.ClientPayee outp : abPayees) {
			if (outList.contains(outp.m_ID)) {
				errMsg = "output ID " + outp.m_ID + " included twice";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
				assetsOk = false;
				continue;
			}
			outList.add(outp.m_ID);

			// compute key hash and add to list for fetch
			ArrayList<String> hashComps = new ArrayList<String>(3);
			hashComps.add(Long.toString(chainId));
			hashComps.add(outp.m_ID);
			hashComps.add(outp.m_Address.toLowerCase());
			String keyIdx = String.join("+", hashComps);
			String keyHash = EncodingUtils.sha3(keyIdx);
			outKeyHashes.add(keyHash);
		}

		// build a tally of the amounts of each asset in the outputs
		for (ClientMVOBlock.ClientPayee reqOutput : payees) {
			String asst = Keys.toChecksumAddress(reqOutput.m_OutputAsset);
			BigInteger inpTotal = inputAssets.get(asst);
			if (inpTotal == null) {
				// asset is not found in inputs
				assetsOk = false;
				errMsg = "output asset " + asst + " not found in inputs";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// confirm that payee address is valid
		 	if (!reqOutput.m_Address.isEmpty()
				&& !WalletUtils.isValidAddress(reqOutput.m_Address))
			{
				assetsOk = false;
				errMsg = "payee address " + reqOutput.m_Address
						+ " appears invalid";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			BigInteger outAmt = reqOutput.m_OutputAmount;
			// this could be an absolute or percentage amount
			if (reqOutput.m_Units.equals("%")) {
				// calculate percentage of input total
				BigInteger incr = inpTotal.multiply(outAmt);
				outAmt = incr.divide(one100);
			}
			// else: absolute amount
			if (!outputAssets.containsKey(asst)) {
				// initialize with amount
				outputAssets.put(asst, outAmt);
			}
			else {
				// add amount to running total
				BigInteger amt = outputAssets.get(asst);
				BigInteger sum = amt.add(outAmt);
				outputAssets.replace(asst, sum);
			}
		}

		// now check that all outputs are sufficiently funded by inputs
		for (String outAsset : outputAssets.keySet()) {
			BigInteger outAmt = outputAssets.get(outAsset);
			BigInteger inpAmt = inputAssets.get(outAsset);
			int amtComp = inpAmt.compareTo(outAmt);
			if (amtComp > 0) {
				// supply a catch-all eNFT back to sender for overage
				BigInteger diff = inpAmt.subtract(outAmt);
				changeOutputs.put(outAsset, diff);
			}
			else if (amtComp < 0) {
				// inputs are short
				assetsOk = false;
				errMsg = "output asset " + outAsset
						+ ": insufficient inputs to pay " + outAmt;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		if (!assetsOk) {
			errMsg = "one or more output assets misconfigured";
			m_Log.error(lbl + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			return errReports;
		}

		// fetch all eNFT keys from the database (inputs + outputs)
		EnftKeysDb ekDb = new EnftKeysDb(m_DbMgr, m_Log);
		ArrayList<String> keyHashes = new ArrayList<String>(inpKeyHashes);
		keyHashes.addAll(outKeyHashes);
		ArrayList<AESKey> aesKeys = ekDb.getKeys(keyHashes);
		if (aesKeys == null) {
			// DB could not be accessed
			errMsg = "error fetching eNFT keys; ABORT!";
			m_Log.error(lbl + errMsg);
			// this is an internal error, doesn't mean anything wrong with input
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			// cannot proceed without decryption keys
			return errReports;
		}

		// make a hash table out of the returned keys for easy lookups
		Hashtable<String, String> eNFTkeys
			= new Hashtable<String, String>(keyHashes.size());
		for (AESKey aesKey : aesKeys) {
			eNFTkeys.put(aesKey.getHash(), aesKey.getKey());
		}

		/* It is permissible for keys for inputs to be missing, *iff* we have
		 * an AES key supplied in the request by the dApp.  (These must be the
		 * correct keys; otherwise the decrypts will fail.)  However *all* of
		 * the output key hashes must be found.
		 */
		boolean keyMissing = false;
		if (aesKeys.size() < inputs.size() + abPayees.size()) {
			// loop through output keys
			for (String outHash : outKeyHashes) {
				if (eNFTkeys.get(outHash) == null) {
					keyMissing = true;
					m_Log.error(lbl + "key for output hash " + outHash
								+ " not found");
				}
			}

			if (!keyMissing) {
				// loop through input keys, using dApp replacements for missing
				for (String inHash : inpKeyHashes) {
					if (eNFTkeys.get(inHash) == null) {
						// find the input responsible for this missing hash
						ClientMVOBlock.ClientInput hashMatch = null;
						for (ClientMVOBlock.ClientInput reqInput : inputs) {
							String id = reqInput.m_eNFT.getID();
							ArrayList<String> hashComp
								= new ArrayList<String>(3);
							hashComp.add(Long.toString(chainId));
							hashComp.add(id);
							hashComp.add(clientReq.getSender().toLowerCase());
							String keyIdx = String.join("+", hashComp);
							String keyHash = EncodingUtils.sha3(keyIdx);
							if (keyHash.equals(inHash)) {
								hashMatch = reqInput;
								break;
							}
						}
						if (hashMatch != null) {
							// use the ClientInput's key
							if (!hashMatch.m_AESkey.isEmpty()) {
								AESKey dAppKey
									= new AESKey(inHash, hashMatch.m_AESkey);
								aesKeys.add(dAppKey);
								eNFTkeys.put(inHash, dAppKey.getKey());
								m_Log.debug(lbl + "found dApp key for Id "
											+ hashMatch.m_eNFT.getID());
							}
							else {
								keyMissing = true;
								m_Log.error(lbl + "no dApp passed key found "
											+ "for keyless hash " + inHash
											+ " re eNFT Id "
											+ hashMatch.m_eNFT.getID());
							}
						}
						else {
							keyMissing = true;
							m_Log.error(lbl + "no key found for hash " + inHash
										+ " re eNFT Id "
										+ hashMatch.m_eNFT.getID());
						}
					}
					// else: was found in DB
					else m_Log.debug(lbl + "found input hash " + inHash);
				}
			}
		}
		if (keyMissing) {
			// not all required keys were retrieved; no replacements available
			errMsg = "error fetching one or more eNFT keys; ABORT!";
			m_Log.error(lbl + errMsg);
			// this is an internal error, doesn't mean anything wrong with input
			errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
			errReports.add(new AudErrorReport(errMsg, errCode));
			// cannot proceed without decryption keys
			return errReports;
		}

		/* in addition, all the inputs must be found in the STATUS table
		 * with valid status
		 */
		EnftStatusDb statDb = new EnftStatusDb(m_DbMgr, m_Log);
		ArrayList<EnftStatus> inputStats
			= statDb.getStatus(idList, chainId, null);
		if (inputStats == null) {
			// this error means we couldn't connect to DB or got DB error
			errMsg = "error fetching STATUS records for input ID list";
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			return errReports;
		}
		boolean inputsOk = true;
		if (inputStats.size() < idList.size()) {
			// this means one or more were not found
			errMsg = "fewer STATUS records for input IDs than inputs, "
					+ inputStats.size() + " of " + idList.size();
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			inputsOk = false;
		}

		// loop through all input IDs
		for (String inpId : idList) {
			// get the corresponding AB input
			AuditorBlock.ClientInput cliInput = null;
			for (AuditorBlock.ClientInput cliIn : abInputs) {
				if (inpId.equals(cliIn.m_ID)) {
					cliInput = cliIn;
					break;
				}
			}
			if (cliInput == null) {
				errMsg = "client input ID " + inpId + " not found in AB inputs";
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				inputsOk = false;
				m_Log.error(lbl + errMsg);
				continue;
			}

			/* NB: cannot verify that the AB's details hash is constructed
			 * correctly, because cliInput.m_InputAsset, cliInput.m_InputRand,
			 * and cliInput.m_InputAmount will not be set for a spend.
			 */

			// get the corresponding original client input
			ClientMVOBlock.ClientInput clientInput = null;
			for (ClientMVOBlock.ClientInput clInp : inputs) {
				if (inpId.equals(clInp.m_eNFT.getID())) {
					clientInput = clInp;
					break;
				}
			}
			// NB: due to the way idList was constructed, clientInput non-null

			// get the corresponding fetched status record
			EnftStatus statRec = null;
			for (EnftStatus stt : inputStats) {
				if (inpId.equals(stt.getID())) {
					statRec = stt;
					break;
				}
			}
			if (statRec == null) {
				// wasn't returned
				errMsg = "STATUS record not found for input ID " + inpId;
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
				inputsOk = false;
				m_Log.error(lbl + errMsg);
			}
			else {
				// check fields by checking hashes 
				if (!statRec.getStatus().equals(EnftStatus.M_Valid)) {
					errMsg = "STATUS record for input ID " + inpId
							+ " has illegal status = " + statRec.getStatus();
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// compare details hash against input in AB
				String statDet = statRec.getDetailsHash();
				if (!statDet.equals(cliInput.m_DetailsHash)) {
					errMsg = "STATUS record for input ID " + inpId
							+ " details hash doesn't match AB input";
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// compare details hash against original client input eNFT
				String eNFThash = EncodingUtils.buildDetailsHash(
												clientInput.m_eNFT.getOwner(),
												clientInput.m_eNFT.getID(),
												clientInput.m_eNFT.getAsset(),
									clientInput.m_eNFT.getAmount().toString(16),
												clientInput.m_eNFT.getRand());
				if (!statDet.equals(eNFThash)) {
					errMsg = "STATUS record for input ID " + inpId
							+ " details hash doesn't match client input eNFT";
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// for completeness, check details hash vs. computed from eNFT
				if (!eNFThash.equals(cliInput.m_DetailsHash)) {
					// shouldn't occur given we've checked the constituent parts
					errMsg = "CRIT - input eNFT with ID "
							+ clientInput.m_eNFT.getID() + " has mismatched "
							+ "details hash vs AB input";
					inputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// NB: now eNFT, client input, STATUS rec, AB hashes all same
			}
		}
		if (!inputsOk) {
			return errReports;
		}

		// decrypt fetched input eNFTs using fetched keys
		boolean inputNFTsOk = true;
		int nftIdx = 0;
		ArrayList<String> hashParts = new ArrayList<String>(3);
		for (ClientMVOBlock.ClientInput reqInput : inputs) {
			// build the hash anew and find the key in the key list
			String id = reqInput.m_eNFT.getID();
			hashParts.add(Long.toString(chainId));
			hashParts.add(id);
			hashParts.add(clientReq.getSender().toLowerCase());
			String keyIdx = String.join("+", hashParts);
			String keyHash = EncodingUtils.sha3(keyIdx);
			hashParts.clear();

			// find the key in the table by looking up the hash
			String aesKey = eNFTkeys.get(keyHash);
			if (aesKey == null || aesKey.isEmpty()) {
				inputNFTsOk = false;
				errMsg = "no key avail for input eNFT " + id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				nftIdx++;
				continue;
			}

			// make key from this key data
			Base64.Decoder b64d = Base64.getUrlDecoder();
			byte[] keyData = b64d.decode(aesKey);
			SecretKey secretKey = null;
			// NB: this key could be user-supplied, cannot assume it's okay
			try {
				secretKey = new SecretKeySpec(keyData, "AES");
			}
			catch (IllegalArgumentException iae) {
				inputNFTsOk = false;
				errMsg = "bad AES key for input eNFT " + id;
				m_Log.error(lbl + errMsg, iae);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				nftIdx++;
				continue;
			}

			// get the corresponding NFT using the index
			NFTmetadata nft = null;
			try {
				nft = fetchedInputs.get(nftIdx);
			}
			catch (IndexOutOfBoundsException ioobe) {
				// should be impossible given how the fetchedInputs[] was built
				m_Log.error(lbl + "no NFT at index " + nftIdx, ioobe);
				inputNFTsOk = false;
				nftIdx++;
				continue;
			}
			nftIdx++;
			eNFTmetadata eNft = nft.getProperties();
			String enshrouded = eNft.getEnshrouded();

			// decrypt the eNFT using this key
			if (!eNft.isEncrypted() || enshrouded.isEmpty()) {
				m_Log.error(lbl + "weird, eNFT Id " + id
							+ " does not appear encrypted");
			}
			else {
				String decNFT = EncodingUtils.decWithAES(secretKey, enshrouded);
				if (decNFT == null || decNFT.isEmpty()) {
					m_Log.error(lbl + errMsg);
					errMsg = "error decrypting input eNFT Id " + id;
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputNFTsOk = false;
					continue;
				}
				try {
					secretKey.destroy();
				}
				catch (DestroyFailedException dfe) { /* ignore */ }
				String decMap = "{" + decNFT + "}";
				// parse this as JSON
				Object parsed = null;
				try {
					parsed = JSON.parse(decMap);
				}
				catch (IllegalStateException ise) {
					errMsg = "parse exception on decrypted input eNFT ID " + id;
					m_Log.error(lbl + errMsg, ise);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputNFTsOk = false;
					continue;
				}
				if (parsed instanceof Map) {
					Map eNFTMap = (Map) parsed;
					if (!eNft.buildFromMap(eNFTMap)) {
						errMsg = "could not parse decrypted eNFT ID " + id;
						errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
						inputNFTsOk = false;
						continue;
					}
				}
				else {
					errMsg = "could not parse decrypted eNFT ID " + id;
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					inputNFTsOk = false;
					continue;
				}
			}

			// get the signer and check their signature on the eNFT
			String signer = eNft.getSigner();
			// get their signing key for this blockchain
			MVOConfig leadConf = (MVOConfig) scc.getMVOMap().get(signer);
			if (leadConf == null) {
				errMsg = "cannot find config for lead MVO signer, " + signer;
				m_Log.error(lbl + errMsg);
				inputNFTsOk = false;
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			String leadSigAddr = leadConf.getSigningAddress();

			// verify sig of leadSigAddr on eNft
			if (!validateENFTsig(eNft, leadSigAddr)) {
				errMsg = "signature validation failure on input eNFT ID " + id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// we now check the signature also on the passed eNFT
			// NB: implicitly, the signer must be identical
			if (!validateENFTsig(reqInput.m_eNFT, leadSigAddr)) {
				errMsg = "signature validation failure on passed input eNFT ID "						+ id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* Now that we have the decrypted input eNFT, we need to compare
			 * two fields to establish equality with the one in client inputs:
			 * the ID and the signature value.  If the signature is the same,
			 * then the hash is the same, hence no need to compare all parts.
			 */
			if (!id.equals(eNft.getID())
				|| !reqInput.m_eNFT.getSigner().equals(eNft.getSigner())
				|| !reqInput.m_eNFT.getSignature().equals(eNft.getSignature()))
			{
				// this input does not match!
				inputNFTsOk = false;
				errMsg = "input eNFT ID " + id
						+ " does not match blockchain copy!";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		if (!inputNFTsOk) {
			// add summary conclusion
			errMsg = "one or more eNFT inputs not found or inconsistent";
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
			// cannot proceed further without decrypted eNFTs
			return errReports;
		}
		/* NB:	at the end of this loop, all NFTs in fetchedInputs are now in
		 * 		decrypted form.
		 */

		// loop through all AB outputs, decrypt them and compare fields
		boolean outputsOk = true;
		for (AuditorBlock.ClientPayee abOut : abPayees) {
			// get the original client request payee that matches
			ClientMVOBlock.ClientPayee orgOutput = null;
			for (ClientMVOBlock.ClientPayee orgPayee : payees) {
				if (orgPayee.m_Payee.equals("payee" + abOut.m_Payee)) {
					orgOutput = orgPayee;
					break;
				}
			}
			BigInteger changeOutput = changeOutputs.get(abOut.m_OutputAsset);
			if (orgOutput == null) {
				/* This could be a change payee.  In that case, the address
				 * MUST be the sender, and we must have a change amount for
				 * this asset type.
				 */
				if (!abOut.m_Address.equalsIgnoreCase(audBlock.getSender())
					|| changeOutput == null)
				{
					errMsg = "illegal payee address on output " + abOut.m_Payee;
					m_Log.error(lbl + errMsg);
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}
			}

			// parse outer NFT with {enshrouded} data encrypted
			NFTmetadata nft = new NFTmetadata(m_Log);
			if (!nft.buildFromJSON(abOut.m_EncMetadata)) {
				errMsg = "could not parse eNFT in AB, bad metadata";
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				outputsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
				m_Log.error(lbl + errMsg);
				continue;
			}

			// obtain the {enshrouded} data
			eNFTmetadata metadata = nft.getProperties();
			if (metadata == null || !metadata.isEncrypted()) {
				errMsg = "output eNFT metadata not found or not encrypted";
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				outputsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
				m_Log.error(lbl + errMsg);
				continue;
			}

			// make sure we have a AES key for it
			hashParts.add(Long.toString(chainId));
			hashParts.add(abOut.m_ID);
			hashParts.add(abOut.m_Address.toLowerCase());
			String keyIdx = String.join("+", hashParts);
			String keyHash = EncodingUtils.sha3(keyIdx);
			String decKey = eNFTkeys.get(keyHash);
			hashParts.clear();
			if (decKey == null) {
				errMsg = "no AES key found for AB output " + abOut.m_Payee
						+ " ID " + abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// decrypt the whole thing including the signature
			Base64.Decoder b64d = Base64.getUrlDecoder();
			byte[] keyData = b64d.decode(decKey);
			SecretKey secretKey = new SecretKeySpec(keyData, "AES");
			String encMeta = metadata.getEnshrouded();
			String decMeta = EncodingUtils.decWithAES(secretKey, encMeta);
			if (decMeta == null) {
				errMsg = "error decrypting eNFT metadata for output ID "
						+ abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			try {
				secretKey.destroy();
			}
			catch (DestroyFailedException dfe) { /* ignore */ }
			eNFTmetadata eNFTdecMetadata = new eNFTmetadata(m_Log);
			String parseTxt = "{" + decMeta + "}";
			Object parseObj = null;
			try {
				parseObj = JSON.parse(parseTxt);
			}
			catch (IllegalStateException ise) { /* log below */ }
			if (!(parseObj instanceof Map)) {
				errMsg = "parse error on eNFT metadata for output ID "
						+ abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			Map parseMap = (Map) parseObj;
			if (!eNFTdecMetadata.buildFromMap(parseMap)) {
				errMsg = "error parsing eNFT map for output ID "
						+ abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// get the signer and check their signature on the eNFT
			String signer = eNFTdecMetadata.getSigner();
			// get their signing key for this blockchain
			MVOConfig leadConf
				= (MVOConfig) scc.getMVOMap().get(signer);
			if (leadConf == null) {
				errMsg = "cannot find config for MVO signer " + signer
						+ " on output eNFT ID " + abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			String leadSigAddr = leadConf.getSigningAddress();

			// verify sig of leadSigAddr on decrypted eNFT
			if (!validateENFTsig(eNFTdecMetadata, leadSigAddr)) {
				errMsg = "signature does not match on output ID " + abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* We now have a decrypted version of the eNFT data.  Compare
			 * each relevant bit against the output's data, to ensure the
			 * lead MVO pulled no shenaningans.  We need to check against BOTH
			 * the values in the AB, and the original outputs specified in the
			 * signed client request which was forwarded.  If no client payee
			 * is available, we'll check for a change output.
			 */
			String asset = eNFTdecMetadata.getAsset();

			// check ID
			if (!abOut.m_ID.equals(eNFTdecMetadata.getID())) {
				errMsg = "CRIT - output eNFT with ID = " 
						+ eNFTdecMetadata.getID() + " while AB has "
						+ abOut.m_ID;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check payee address
			if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(abOut.m_Address)) {
				errMsg = "CRIT - output eNFT with ID "
						+ eNFTdecMetadata.getID() + " has payee "
						+ eNFTdecMetadata.getOwner() + " while AB has "
						+ abOut.m_Address;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check asset type
			if (!abOut.m_OutputAsset.equalsIgnoreCase(asset)) {
				errMsg = "CRIT - output with ID = "
						+ abOut.m_ID + " is not for eNFT asset " + asset;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check randomizer value
			if (!abOut.m_Rand.equals(eNFTdecMetadata.getRand())) {
				errMsg = "CRIT - output eNFT with ID = "
						+ eNFTdecMetadata.getID() + " has salt "
						+ eNFTdecMetadata.getRand() + " while AB has "
						+ abOut.m_Rand;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// check fields against client payee if there is one
			BigInteger orgAmt = null;
			if (orgOutput != null) {	// eNFT from client output
				// check address against the client's payee output
				if (!abOut.m_Address.equalsIgnoreCase(orgOutput.m_Address)) {
					errMsg = "CRIT - output eNFT with payee "
							+ eNFTdecMetadata.getOwner()
							+ " while org client request has "
							+ orgOutput.m_Address;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// check asset
				if (!asset.equalsIgnoreCase(orgOutput.m_OutputAsset)) {
					errMsg = "CRIT - output eNFT with asset " + asset
							+ ", while org client request has "
							+ orgOutput.m_OutputAsset;
					m_Log.error(lbl + errMsg);
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// determine amount from org request based on units
				orgAmt = orgOutput.m_OutputAmount;
				BigInteger orgInputTotal = inputAssets.get(asset);
				if (orgOutput.m_Units.equals("%")) {
					// interpret as a percentage of total
					BigInteger incr = orgInputTotal.multiply(orgAmt);
					orgAmt = incr.divide(one100);
				}

				// check original amount against output eNFT
				if (!orgAmt.equals(eNFTdecMetadata.getAmount())) {
					errMsg = "CRIT - output eNFT with amount "
							+ eNFTdecMetadata.getAmount().toString()
							+ " while org client request has "
							+ orgAmt.toString();
					m_Log.error(lbl + errMsg);
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// also check amount against AB
				if (!abOut.m_OutputAmount.equals(eNFTdecMetadata.getAmount())) {
					errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID()
							+ " is for amount " + eNFTdecMetadata.getAmount()
							+ ", but should be " + abOut.m_OutputAmount;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// check randomizer value
				if (!orgOutput.m_Rand.equals(eNFTdecMetadata.getRand())) {
					errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID() + " has salt "
							+ eNFTdecMetadata.getRand() + " while org client "
							+ "request has " + orgOutput.m_Rand;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// verify the memo line, if present, has not changed
				String memo = orgOutput.m_Memo;
				if (!memo.isEmpty()) {
					String eMemo = eNFTdecMetadata.getMemo();
					if (!memo.equals(eMemo)) {
						errMsg = "memo on eNFT, ID " + eNFTdecMetadata.getID()
								+ " doesn't match org client req";
						m_Log.error(lbl + errMsg);
						outputsOk = false;
						errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
						errReports.add(new AudErrorReport(errMsg, errCode));
					}
					// warn about excessive length (shouldn't happen)
					if (eMemo.length() > eNFTmetadata.M_MEMO_MAX) {
						m_Log.warning(lbl + "memo on eNFT, ID "
									+ eNFTdecMetadata.getID() + " too long, "
									+ eMemo.length() + " chars");
					}
				}
			}
			else {	// eNFT for change / overage
				// validate that this eNFT is to be issued to sender
				if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(
														clientReq.getSender()))
				{
					errMsg = "CRIT - output change eNFT with payee "
							+ eNFTdecMetadata.getOwner()
							+ " does not match sender";
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					m_Log.error(lbl + errMsg);
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				/* If this is a change eNFT (for the overage), it must correlate
				 * to the amount found in the list of change outputs.
				 */
				if (!eNFTdecMetadata.getAmount().equals(changeOutput)) {
					errMsg = "CRIT - output change eNFT seen for asset " + asset
							+ ", but no change value for that amount exists";
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					m_Log.error(lbl + errMsg);
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// NB: we checked abOut.m_Rand vs. eNFT above

				// memo line should say change amount
				if (!chgMemo.equals(eNFTdecMetadata.getMemo())) {
					errMsg = "memo on eNFT, ID " + eNFTdecMetadata.getID()
							+ " not as expected for change";
					m_Log.error(lbl + errMsg);
					// don't fail because of this, though
				}
			}

			// make sure both details hashes agree (AB payee and eNFT)
			String hashStr = EncodingUtils.buildDetailsHash(abOut.m_Address,
															abOut.m_ID,
															abOut.m_OutputAsset,
											abOut.m_OutputAmount.toString(16),
															abOut.m_Rand);
			if (!hashStr.equals(abOut.m_DetailsHash)) {
				errMsg = "CRIT - output " + abOut.m_Payee + " with ID "
				 		+ abOut.m_ID + ", details hash doesn't match AB value";
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
				outputsOk = false;
			}
			String eNFTHash = EncodingUtils.buildDetailsHash(
													eNFTdecMetadata.getOwner(),
													eNFTdecMetadata.getID(),
													eNFTdecMetadata.getAsset(),
									eNFTdecMetadata.getAmount().toString(16),
													eNFTdecMetadata.getRand());
			if (!eNFTHash.equals(hashStr)) {
				errMsg = "CRIT - output eNFT with ID " + eNFTdecMetadata.getID()
						+ ", details hash doesn't match AB";
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* cycle through inputs of this asset type to determine the
			 * generation, which is the minimum plus 1
			 */
			int minGen = 0;
			for (NFTmetadata iNft : fetchedInputs) {
				eNFTmetadata eNFT = iNft.getProperties();
				if (eNFT.getAsset().equalsIgnoreCase(asset)) {
					if (minGen == 0) {
						// first match
						minGen = eNFT.getGeneration();
					}
					else if (eNFT.getGeneration() < minGen) {
						minGen = eNFT.getGeneration();
					}
				}
			}
			minGen += 1;
			
			// check generation
			if (eNFTdecMetadata.getGeneration() != minGen) {
				errMsg = "output eNFT with ID " +  eNFTdecMetadata.getID()
						+ " has illegal generation, "
						+ eNFTdecMetadata.getGeneration();
				outputsOk = false;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		if (!outputsOk) {
			// add summary item
			errMsg = "one or more eNFT outputs inconsistent";
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		else {
			// TBD: if required, deal with expiration/growth/cost
		}
		return errReports;
	}

	/**
	 * sub-processor for withdraw operations
	 * @param clientBlock the original signed request sent by dApp client
	 * @param audBlock the detailed OB prepared and signed by the MVO
	 * @return true if everything checks out
	 */
	private ArrayList<AudErrorReport> handleWithdrawBlock(
													ClientMVOBlock clientReq,
													AuditorBlock audBlock)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".handleWithdrawBlock: ";
		ArrayList<AudErrorReport> errReports = new ArrayList<AudErrorReport>();
		String errMsg = "";
		int errCode = 0;

		// begin by validating the original user request
		long chainId = clientReq.getChainId();
		SmartContractConfig scc = m_AUD.getSCConfig(chainId);
		// NB: scc == null was checked and eliminated in caller

		// confirm withdrawer address is valid
		if (!WalletUtils.isValidAddress(clientReq.getSender())) {
			errMsg = "sender address (" + clientReq.getSender()
					+ ") appears invalid";
			m_Log.error(lbl + "req verify failure: " + errMsg);
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}

		String redAsset = clientReq.getAsset();
		BigInteger amt = clientReq.getAmount();
		errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
		boolean valid = true;

		// check overall asset type matches
		if (!redAsset.equalsIgnoreCase(audBlock.getAsset())) {
			errMsg = "deposit asset (" + redAsset + ") conflicts with AB, "
							+ "which has " + audBlock.getAsset();
			m_Log.error(lbl + errMsg);
			valid = false;
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		// these issues are serious enough to bail right here
		if (!valid) return errReports;

		// check input asset types all match given redemption asset
		boolean inputsOk = true;
		ArrayList<ClientMVOBlock.ClientInput> inputs = clientReq.getInputs();
		// NB: empty inputs is disallowed by ClientMVOBlock parser
		ArrayList<AuditorBlock.ClientInput> abInputs = audBlock.getInputs();
		// NB: empty abInputs is disallowed by AuditorBlock parser
		if (abInputs.size() != inputs.size()) {
			errMsg = "AB has " + abInputs.size() + " inputs but client block "
					+ "has " + inputs.size();
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			inputsOk = false;
			return errReports;
		}
		// list of IDs of all output eNFTs
		ArrayList<AuditorBlock.ClientPayee> abPayees = audBlock.getPayees();
		ArrayList<String> outList = new ArrayList<String>(abPayees.size());

		// build list of output eNFT Ids (should be exactly 1 or 0)
		for (AuditorBlock.ClientPayee outp : abPayees) {
			if (outList.contains(outp.m_ID)) {
				errMsg = "input ID " + outp.m_ID + " included twice";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			outList.add(outp.m_ID);
		}

		/* As we loop through inputs, build a tally of the amount of the asset
		 * in the inputs.  Also make a list of the IDs of all input eNFTs.
		 * Finally, build a list of the AES key hashes we'll need to look up.
		 */
		BigInteger inputTotal = new BigInteger("0");
		ArrayList<String> idList = new ArrayList<String>(inputs.size());
		if (abPayees.size() > 1) {
			// this is technically illegal
			errMsg = "AB has " + abPayees.size() + " outputs, only 1 allowed";
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		ArrayList<String> keyHashes
			= new ArrayList<String>(inputs.size() + abPayees.size());
		for (ClientMVOBlock.ClientInput reqInput : inputs) {
			String id = reqInput.m_eNFT.getID();
			// check for dup inputs
			if (idList.contains(id)) {
				errMsg = "input ID " + id + " included twice";
				m_Log.error(lbl + errMsg);
				inputsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			idList.add(id);
			String asst = reqInput.m_eNFT.getAsset();
			if (!asst.equalsIgnoreCase(redAsset)) {
				errMsg = "input ID " + id + " is for asset " + asst
						+ "; does not match " + redAsset;
				m_Log.error(lbl + errMsg);
				inputsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}

			// add amount to running total of inputs
			inputTotal = inputTotal.add(reqInput.m_eNFT.getAmount());

			// compute key hash and add to list
			ArrayList<String> hashComps = new ArrayList<String>(3);
			hashComps.add(Long.toString(chainId));
			hashComps.add(id);
			hashComps.add(clientReq.getSender().toLowerCase());
			String keyIdx = String.join("+", hashComps);
			String keyHash = EncodingUtils.sha3(keyIdx);
			keyHashes.add(keyHash);
		}
		if (!inputsOk) {
			errMsg = "withdrawal inputs are bad, aborting processing";
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
			return errReports;
		}

		/* Before proceeding any further, validate that all inputs are
		 * still circulating.  We could simply trust the signature of the
		 * issuing MVO on the passed eNFT input, but we cannot trust
		 * that it is indeed still circulating and hasn't been burned.
		 * (Particularly since that status could change during our
		 * processing of a transaction.)  This check must also take into
		 * account possible greylisting and insufficient dwell time.
		 *
		 * Note however that because an MVO could have sent us this AB after
		 * the transaction was mined, it might be normal and expected that the
		 * input eNFTs have been burnt.  Therefore we cannot greylist outputs
		 * solely on the basis that an input was burnt.  To distinguish
		 * between these cases, we need to do a further check: see if it's the
		 * case that *all* inputs are burned, and outputs are now minted.  In
		 * this case the results are to be expected.
		 */
		BlockchainAPI web3j = m_AUD.getWeb3(chainId);
		ArrayList<NFTmetadata> fetchedInputs = null;
		Future<ArrayList<NFTmetadata>> nftFut
			= web3j.getNFTsById(chainId, scc.getABI(), clientReq.getSender(),
								idList, false);
		try {
			fetchedInputs = nftFut.get();
			if (fetchedInputs != null && fetchedInputs.isEmpty()) {
				// determine whether *all* inputs are found when burned allowed
				nftFut = web3j.getNFTsById(chainId, scc.getABI(),
										   clientReq.getSender(), idList, true);
				fetchedInputs = nftFut.get();
				// allow for case where there are no outputs
				if (fetchedInputs.size() == inputs.size() && !outList.isEmpty())
				{
					// look for outputs (burned or not)
					ArrayList<NFTmetadata> fetchedOutputs = null;
					Future<ArrayList<NFTmetadata>> outFut
						= web3j.getNFTsById(chainId, scc.getABI(),
											clientReq.getSender(),
											outList, true);
					fetchedOutputs = outFut.get();
					if (fetchedOutputs != null
						&& fetchedOutputs.size() < outList.size())
					{
						// okay, this is an error
						errMsg
							= "one or more input eNFTs is not found on chain";
						m_Log.error(lbl + errMsg);
						errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
						inputsOk = false;
						errReports.add(new AudErrorReport(errMsg, errCode));
					}
					else {
						m_Log.debug(lbl + "hit corner case where inputs are "
									+ "deminted but outputs are found onchain");
					}
				}
			}
		}
		catch (InterruptedException | ExecutionException
			   | CancellationException e)
		{
			errMsg = "could not confirm input eNFTs on chain";
			m_Log.error(lbl + errMsg, e);
			inputsOk = false;
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			//scc.setEnabled(false);
		}

		// check for possible greylisting and insufficient dwell time
		EnftCache eCache = scc.getCache();
		if (eCache != null) {
			boolean greyListedInput = false;
			boolean unconfInput = false;
			BigInteger curBlock = eCache.getLatestBlock();
			EnshroudProtocol protoWrapper = eCache.getProtocolWrapper();
			for (String Id : idList) {
				// see if ID is in the greylisted list in cache
				BigInteger eId = Numeric.parsePaddedNumberHex(Id);
				if (eCache.isIdGreyListed(eId)) {
					/* This means it's greylisted on-chain in a past block (not
					 * just in our local AudDb).  In this case either a client
					 * is trying shenanigans, or another AUD beat us to it a
					 * while back.  Either way, no point in further processing.
					 */
					errMsg = "greylisted input eNFT, ID " + Id;
					m_Log.error(lbl + errMsg);
					greyListedInput = true;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					break;
				}

				// see if ID is vested/confirmed at this time
				BigInteger unlockBlock = null;
				try {
					unlockBlock = protoWrapper.enftUnlockTime(eId).send();
				}
				catch (Exception eee) {
					m_Log.error(lbl + "unable to obtain enftUnlockTime "
								+ "for ID " + Id, eee);
				}
				if (unlockBlock == null || unlockBlock.compareTo(curBlock) > 0)
				{
					errMsg = "unconfirmed input eNFT, ID " + Id;
					m_Log.error(lbl + errMsg);
					unconfInput = true;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					break;
				}
			}
			if (greyListedInput || unconfInput) {
				inputsOk = false;
			}
		}
		// else: we have no cache to check against for this chain

		if (!inputsOk) {
			// no point in examining invalid inputs
			return errReports;
		}

		// confirm the withdrawal is adequately funded by inputs
		BigInteger changeOutput = new BigInteger("0");
		int inpDiff = inputTotal.compareTo(amt);
		if (inpDiff < 0) {
			// error
			errMsg = "input eNFTs do not support a redemption of "
					+ amt.toString() + " of " + redAsset;
			m_Log.error(lbl + errMsg);
			inputsOk = false;
		}
		// see whether we need an implicit change output
		else if (inpDiff > 0) {
			changeOutput = inputTotal.subtract(amt);
			// add hashes for change output(s)
			for (AuditorBlock.ClientPayee payee : abPayees) {
				ArrayList<String> hashComps = new ArrayList<String>(3);
				hashComps.add(Long.toString(chainId));
				hashComps.add(payee.m_ID);
				hashComps.add(clientReq.getSender().toLowerCase());
				String keyIdx = String.join("+", hashComps);
				String keyHash = EncodingUtils.sha3(keyIdx);
				keyHashes.add(keyHash);
			}
		}

		// fetch eNFT keys from the database (inputs + any outputs)
		EnftKeysDb ekDb = new EnftKeysDb(m_DbMgr, m_Log);
		ArrayList<AESKey> aesKeys = ekDb.getKeys(keyHashes);
		if (aesKeys == null
			|| aesKeys.size() < (inputs.size() + abPayees.size()))
		{
			errMsg = "error fetching eNFT keys; ABORT!";
			m_Log.error(lbl + errMsg);
			// this is an internal error, doesn't mean anything wrong with input
			if (aesKeys == null) {
				// DB could not be accessed
				errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			}
			else {
				// not all keys were retrieved
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
			}
			errReports.add(new AudErrorReport(errMsg, errCode));
			// cannot proceed without decryption keys
			return errReports;
		}

		// make a hash table out of this for easy lookups
		Hashtable<String, String> eNFTkeys
			= new Hashtable<String, String>(keyHashes.size());
		for (AESKey aesKey : aesKeys) {
			eNFTkeys.put(aesKey.getHash(), aesKey.getKey());
		}

		/* in addition, all the inputs must be found in the STATUS table
		 * with valid status
		 */
		EnftStatusDb statDb = new EnftStatusDb(m_DbMgr, m_Log);
		ArrayList<EnftStatus> inputStats
			= statDb.getStatus(idList, chainId, null);
		if (inputStats == null) {
			// this means we couldn't connect to DB or got DB error
			errMsg = "error fetching STATUS records for input ID list";
			errCode = HttpURLConnection.HTTP_BAD_GATEWAY;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			return errReports;
		}
		if (inputStats.size() < idList.size()) {
			// this means one or more were not found
			errMsg = "fewer STATUS records for input IDs than inputs, "
					+ inputStats.size() + " of " + idList.size();
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			errReports.add(new AudErrorReport(errMsg, errCode));
			m_Log.error(lbl + errMsg);
			inputsOk = false;
		}
		for (String inpId : idList) {
			// get the corresponding AB input
			AuditorBlock.ClientInput cliInput = null;
			for (AuditorBlock.ClientInput cliIn : abInputs) {
				if (inpId.equals(cliIn.m_ID)) {
					cliInput = cliIn;
					break;
				}
			}
			if (cliInput == null) {
				errMsg = "client input ID " + inpId + " not found in AB inputs";
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				inputsOk = false;
				m_Log.error(lbl + errMsg);
			}

			// get the corresponding original client input
			ClientMVOBlock.ClientInput clientInput = null;
			for (ClientMVOBlock.ClientInput clInp : inputs) {
				if (inpId.equals(clInp.m_eNFT.getID())) {
					clientInput = clInp;
					break;
				}
			}
			// NB: due to the way idList was constructed, clientInput non-null

			/* The AB's input doesn't set the m_DetailsHash except for spends,
			 * so we need to build it separately from the other values.  If
			 * this matches, we don't need to check the constituent values.
			 */
			String ABinputHash = EncodingUtils.buildDetailsHash(
														audBlock.getSender(),
														cliInput.m_ID,
														cliInput.m_InputAsset,
											cliInput.m_InputAmount.toString(16),
														cliInput.m_InputRand);
			if (cliInput.m_DetailsHash.isEmpty()) {
				// set it to the calculated value
				cliInput.m_DetailsHash = ABinputHash;
			}

			// get the corresponding fetched status record (must exist)
			EnftStatus statRec = null;
			for (EnftStatus stt : inputStats) {
				if (inpId.equals(stt.getID())) {
					statRec = stt;
					break;
				}
			}
			if (statRec == null) {
				// wasn't returned
				errMsg = "STATUS record not found for input ID " + inpId;
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				inputsOk = false;
				m_Log.error(lbl + errMsg);
			}
			else {
				if (!statRec.getStatus().equals(EnftStatus.M_Valid)) {
					errMsg = "STATUS record for input ID " + inpId
							+ " has illegal status = " + statRec.getStatus();
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// compare details hash against input in AB (built above)
				String statDet = statRec.getDetailsHash();
				if (!statDet.equals(cliInput.m_DetailsHash)) {
					errMsg = "STATUS record for input ID " + inpId
							+ " details hash doesn't match AB input";
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// compare details hash against original client input eNFT
				String eNFThash = EncodingUtils.buildDetailsHash(
												clientInput.m_eNFT.getOwner(),
												clientInput.m_eNFT.getID(),
												clientInput.m_eNFT.getAsset(),
									clientInput.m_eNFT.getAmount().toString(16),
												clientInput.m_eNFT.getRand());
				if (!statDet.equals(eNFThash)) {
					errMsg = "STATUS record for input ID " + inpId
						+ " details hash doesn't match client input eNFT";
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputsOk = false;
					m_Log.error(lbl + errMsg);
				}

				// for completeness, check details hash vs. computed from eNFT
				if (!eNFThash.equals(cliInput.m_DetailsHash)) {
					// shouldn't occur given we've checked the constituent parts
					errMsg = "CRIT - input eNFT with ID "
							+ clientInput.m_eNFT.getID() + " has mismatched "
							+ "details hash vs AB input";
					inputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
			}
		}
		if (!inputsOk) {
			return errReports;
		}

		// decrypt fetched input eNFTs using fetched keys
		boolean inputNFTsOk = true;
		int nftIdx = 0;
		ArrayList<String> hashParts = new ArrayList<String>(3);
		for (ClientMVOBlock.ClientInput reqInput : inputs) {
			// build the hash anew and find the key in the key list
			String id = reqInput.m_eNFT.getID();
			hashParts.add(Long.toString(chainId));
			hashParts.add(id);
			hashParts.add(clientReq.getSender().toLowerCase());
			String keyIdx = String.join("+", hashParts);
			String keyHash = EncodingUtils.sha3(keyIdx);
			hashParts.clear();

			// find the key in the table by looking up the hash
			String aesKey = eNFTkeys.get(keyHash);
			if (aesKey == null || aesKey.isEmpty()) {
				inputNFTsOk = false;
				errMsg = "no key avail for input eNFT " + id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				nftIdx++;
				continue;
			}

			// make key from this key data
			Base64.Decoder b64d = Base64.getUrlDecoder();
			byte[] keyData = b64d.decode(aesKey);
			SecretKey secretKey = null;
			// NB: this key could be user-supplied, cannot assume it's okay
			try {
				secretKey = new SecretKeySpec(keyData, "AES");
			}
			catch (IllegalArgumentException iae) {
				inputNFTsOk = false;
				errMsg = "bad AES key for input eNFT " + id;
				m_Log.error(lbl + errMsg, iae);
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				nftIdx++;
				continue;
			}

			// get the corresponding NFT using the index
			NFTmetadata nft = null;
			try {
				nft = fetchedInputs.get(nftIdx);
			}
			catch (IndexOutOfBoundsException ioobe) {
				// should be impossible given how the fetchedInputs[] was built
				m_Log.error(lbl + "no NFT at index " + nftIdx, ioobe);
				inputNFTsOk = false;
				nftIdx++;
				continue;
			}
			nftIdx++;
			eNFTmetadata eNft = nft.getProperties();
			String enshrouded = eNft.getEnshrouded();

			// decrypt the eNFT using this key
			if (!eNft.isEncrypted() || enshrouded.isEmpty()) {
				m_Log.error(lbl + "weird, eNFT Id " + id
							+ " does not appear encrypted");
			}
			else {
				String decNFT = EncodingUtils.decWithAES(secretKey, enshrouded);
				if (decNFT == null || decNFT.isEmpty()) {
					m_Log.error(lbl + errMsg);
					errMsg = "error decrypting input eNFT Id " + id;
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputNFTsOk = false;
					continue;
				}
				try {
					secretKey.destroy();
				}
				catch (DestroyFailedException dfe) { /* ignore */ }
				String decMap = "{" + decNFT + "}";
				// parse this as JSON
				Object parsed = null;
				try {
					parsed = JSON.parse(decMap);
				}
				catch (IllegalStateException ise) {
					errMsg = "parse exception on decrypted input eNFT ID " + id;
					m_Log.error(lbl + errMsg, ise);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					inputNFTsOk = false;
					continue;
				}
				if (parsed instanceof Map) {
					Map eNFTMap = (Map) parsed;
					if (!eNft.buildFromMap(eNFTMap)) {
						errMsg = "could not parse decrypted eNFT ID " + id;
						errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
						inputNFTsOk = false;
						continue;
					}
				}
				else {
					errMsg = "could not parse decrypted eNFT ID " + id;
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					inputNFTsOk = false;
					continue;
				}
			}

			// get the signer and check their signature on the eNFT
			String signer = eNft.getSigner();
			// get their signing key for this blockchain
			MVOConfig leadConf = (MVOConfig) scc.getMVOMap().get(signer);
			if (leadConf == null) {
				errMsg = "cannot find config for eNFT signer, " + signer;
				m_Log.error(lbl + errMsg);
				inputNFTsOk = false;
				errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
				errReports.add(new AudErrorReport(errMsg, errCode));
				continue;
			}
			String leadSigAddr = leadConf.getSigningAddress();

			// verify sig of leadSigAddr on eNft
			if (!validateENFTsig(eNft, leadSigAddr)) {
				errMsg = "signature validation failure on input eNFT ID " + id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			// verify sig of leadSigAddr on reqInput.m_eNFT
			// NB: implicitly, the signer must be identical
			if (!validateENFTsig(reqInput.m_eNFT, leadSigAddr)) {
				errMsg = "signature validation failure on passed input "
						+ "eNFT ID " + id;
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}

			/* Now that we have the decrypted input eNFT, we need to compare
			 * two fields to establish equality with the one in client inputs:
			 * the ID and the signature value.  If the signature is the same,
			 * then the hash is the same, hence no need to compare all parts.
			 */
			if (!id.equals(eNft.getID())
				|| !reqInput.m_eNFT.getSigner().equals(eNft.getSigner())
				|| !reqInput.m_eNFT.getSignature().equals(eNft.getSignature()))
			{
				// this input does not match!
				inputNFTsOk = false;
				errMsg = "input eNFT ID " + id
						+ " does not match blockchain copy!";
				m_Log.error(lbl + errMsg);
				errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
				inputNFTsOk = false;
				errReports.add(new AudErrorReport(errMsg, errCode));
			}
		}
		if (!inputNFTsOk) {
			errMsg = "one or more eNFT inputs not found or inconsistent";
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
			// cannot proceed further without decrypted eNFTs
			return errReports;
		}
		/* NB:	at the end of this loop, all NFTs in fetchedInputs are now in
		 * 		decrypted form.
		 */

		// if outputs exist, confirm they're correct (== changeOutput)
		boolean outputsOk = true;
		if (!abPayees.isEmpty()) {
			final String chgMemo = "auto-generated change amount";
			BigInteger outTot = BigInteger.ZERO;

			/* cycle through inputs to determine the generation, which is
			 * the minimum plus 1
			 */
			int minGen = 0;
			for (NFTmetadata iNft : fetchedInputs) {
				eNFTmetadata eNFT = iNft.getProperties();
				if (minGen == 0) {
					// first match
					minGen = eNFT.getGeneration();
				}
				else if (eNFT.getGeneration() < minGen) {
					minGen = eNFT.getGeneration();
				}
			}
			minGen += 1;

			// loop should execute exactly once
			for (AuditorBlock.ClientPayee abOut : abPayees) {
				if (!abOut.m_Address.equalsIgnoreCase(audBlock.getSender())) {
					errMsg = "illegal payee address on output " + abOut.m_Payee;
					m_Log.error(lbl + errMsg);
					outputsOk = false;
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
				if (!abOut.m_OutputAsset.equalsIgnoreCase(redAsset)) {
					errMsg = "illegal asset " + abOut.m_OutputAsset
							+ " on output " + abOut.m_Payee;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
				outTot = abOut.m_OutputAmount.add(outTot);
				if (outTot.compareTo(changeOutput) > 0) {
					errMsg = "AB outputs sum to more than change output of "
							+ changeOutput;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// make sure we have a AES key for it
				hashParts.add(Long.toString(chainId));
				hashParts.add(abOut.m_ID);
				hashParts.add(abOut.m_Address.toLowerCase());
				String keyIdx = String.join("+", hashParts);
				String keyHash = EncodingUtils.sha3(keyIdx);
				String decKey = eNFTkeys.get(keyHash);
				hashParts.clear();
				if (decKey == null) {
					errMsg = "no AES key found for AB output " + abOut.m_Payee
							+ " ID " + abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}

				// parse outer NFT with {enshrouded} data encrypted
				NFTmetadata nft = new NFTmetadata(m_Log);
				if (!nft.buildFromJSON(abOut.m_EncMetadata)) {
					errMsg = "output eNFT did not parse";
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}

				// obtain the {enshrouded} data
				eNFTmetadata metadata = nft.getProperties();
				if (metadata == null || !metadata.isEncrypted()) {
					errMsg = "output eNFT metadata not found";
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}

				// decrypt the whole thing including the signature
				Base64.Decoder b64d = Base64.getUrlDecoder();
				byte[] keyData = b64d.decode(decKey);
				SecretKey secretKey = new SecretKeySpec(keyData, "AES");
				String encMeta = metadata.getEnshrouded();
				String decMeta = EncodingUtils.decWithAES(secretKey, encMeta);
				if (decMeta == null) {
					errMsg = "error decrypting eNFT metadata for output ID "
							+ abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}
				try {
					secretKey.destroy();
				}
				catch (DestroyFailedException dfe) { /* ignore */ }
				eNFTmetadata eNFTdecMetadata = new eNFTmetadata(m_Log);
				String parseTxt = "{" + decMeta + "}";
				Object parseObj = null;
				try {
					parseObj = JSON.parse(parseTxt);
				}
				catch (IllegalStateException ise) { /* log below */ }
				if (!(parseObj instanceof Map)) {
					errMsg = "parse error on eNFT metadata for output ID "
							+ abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}
				Map parseMap = (Map) parseObj;
				if (!eNFTdecMetadata.buildFromMap(parseMap)) {
					errMsg = "error parsing eNFT map for output ID "
							+ abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}

				// get the signer and check their signature on the eNFT
				String signer = eNFTdecMetadata.getSigner();
				// get their signing key for this blockchain
				MVOConfig leadConf
					= (MVOConfig) scc.getMVOMap().get(signer);
				if (leadConf == null) {
					errMsg = "cannot find config for MVO signer " + signer
							+ " on output eNFT ID " + abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_INTERNAL_ERROR;
					errReports.add(new AudErrorReport(errMsg, errCode));
					continue;
				}
				String leadSigAddr = leadConf.getSigningAddress();

				// verify sig of leadSigAddr on decrypted eNFT
				if (!validateENFTsig(eNFTdecMetadata, leadSigAddr)) {
					errMsg = "signature does not match on output ID "
							+ abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				/* We now have a decrypted version of the eNFT data.  Compare
				 * each relevant bit against the output's data, to ensure the
				 * lead MVO pulled no shenaningans.  We have no client outputs
				 * to check against, merely the single change output amount.
				 */
				// check ID
				if (!abOut.m_ID.equals(eNFTdecMetadata.getID())) {
					errMsg = "CRIT - output eNFT with ID = " 
							+ eNFTdecMetadata.getID() + " while AB has "
							+ abOut.m_ID;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// check asset type
				if (!abOut.m_OutputAsset.equalsIgnoreCase(
													eNFTdecMetadata.getAsset()))
				{
					errMsg = "CRIT - output eNFT with ID = "
							+ eNFTdecMetadata.getID() + " is not for asset "
							+ abOut.m_OutputAsset;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// check amount
				if (!abOut.m_OutputAmount.equals(eNFTdecMetadata.getAmount())) {
					errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID() + " is not for amount "
							+ abOut.m_OutputAmount;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// check payee address
				if (!eNFTdecMetadata.getOwner().equalsIgnoreCase(
															abOut.m_Address))
				{
					errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID() + " has payee "
							+ eNFTdecMetadata.getOwner() + " while AB has "
							+ abOut.m_Address;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				/* NB: Because the AB's payee (abOut) doesn't include a details
				 * hash (m_DetailsHash) on a withdrawal, we cannot double-check
				 * it.  We did however check all the constituent datums above.
				 */
				
				// check randomizer value
				if (!eNFTdecMetadata.getRand().equals(abOut.m_Rand)) {
					errMsg = "CRIT - output eNFT with ID "
							+ eNFTdecMetadata.getID() + " has randomizer value "
							+ eNFTdecMetadata.getRand() + " while AB has "
							+ abOut.m_Rand;
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}

				// memo line should say change amount
				if (!chgMemo.equals(eNFTdecMetadata.getMemo())) {
					m_Log.error(lbl + "memo on output eNFT, ID " + abOut.m_ID
								+ " not as expected");
					// don't fail because of this though
				}

				// check generation
				if (eNFTdecMetadata.getGeneration() != minGen) {
					errMsg = "output eNFT with ID " + abOut.m_ID
							+ " has illegal generation, "
							+ eNFTdecMetadata.getGeneration();
					outputsOk = false;
					m_Log.error(lbl + errMsg);
					errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
					errReports.add(new AudErrorReport(errMsg, errCode));
				}
			}
		}
		if (!outputsOk) {
			// add summary item
			errMsg = "one or more eNFT outputs inconsistent";
			errCode = HttpURLConnection.HTTP_NOT_ACCEPTABLE;
			m_Log.error(lbl + errMsg);
			errReports.add(new AudErrorReport(errMsg, errCode));
		}
		else {
			// TBD: if required, deal with expiration/growth/cost, etc.
		}

		return errReports;
	}

	/**
	 * helper method to verify client block signatures
	 * @param requestJson the entire JSON string received from the user
	 * @param cliRequest the parsed ClientMVOBlock
	 * @return true if client's signature was valid
	 */
	private boolean verifyClientSig(String requestJson,
								    ClientMVOBlock cliRequest)
	{
		if (cliRequest == null || requestJson == null || requestJson.isEmpty())
		{
			m_Log.error("AudBlockHandler.verifyClientSig: missing input");
			return false;
		}

		/* to extract the signed portion, take everything up to:
		 * ',\"signature\":'
		 */
		int sigIdx = requestJson.lastIndexOf(",\"signature\":");
		if (sigIdx == -1) {
			// json was not signed
			m_Log.error("AudBlockHandler.verifyClientSig: missing signature");
			return false;
		}
		String eip712Data = requestJson.substring(0, sigIdx) + "}";

		// obtain the signature from full input data
		String rawSig = requestJson.substring(sigIdx);
		int sigStart = rawSig.lastIndexOf(":\"");
		// strip off leading :" and trailing "}
		String signature = rawSig.substring(sigStart+2, rawSig.length()-2);
		String signingAddress = cliRequest.getSender();

		// determine which address actually signed the data
		String actualSigAddr
			= EncodingUtils.getEIP712SignerAddress(eip712Data, signature);
		return actualSigAddr.equalsIgnoreCase(signingAddress);
	}

	/**
	 * validate the signature on an eNFT
	 * @param eNFT the eNFT object
	 * @param sigAddr the address which is supposed to have signed it
	 * @return true if validate, false otherwise
	 */
	private boolean validateENFTsig(eNFTmetadata eNft, String sigAddr) {
		if (eNft == null || sigAddr == null || sigAddr.isEmpty()) {
			return false;
		}
		String signingAddress = eNft.getSigAddr();
		if (signingAddress.equalsIgnoreCase(sigAddr)) {
			return true;
		}
		return false;
	}

	// END methods
}
