/*
 * last modified---
 * 	12-19-25 add getFormerOwners()
 * 	12-04-25 add reinitWhileRunning(), based on initialize()
 * 	09-16-25 attempt fix to rare multiple parallel block subscription condition
 * 	04-25-25 query chainId in loadBlocks(), to catch value change on reconnect
 * 	04-21-25 in BlockSubscription.onError(), check for block decrements
 * 	04-09-25 catch WebsocketNotConnectedExceptionS
 * 	04-08-25 log any negative block increments
 * 	03-12-25 set BlockHandler.m_Init only in accept() method, to prevent
 * 			 ever reaching init state while connected to a dead block source
 * 	02-11-25 do not preserve old startBlock values unless startBlock < stopBlock
 * 	01-18-25 obviate bug with ongoing subscriptions not returning events for
 * 			 the startBlock if startBlock is also the LATEST block
 * 	11-26-24 fix issue with nesting initialize(true) calls skipping blocks
 * 	11-05-24 rework to use ScheduledThreadPoolExecutor rather than TimerS
 * 	10-30-24 invoke EnftCacheResetter instead of exiting on catastrophic fails
 * 	09-18-24 rework initialization procedures to avoid the need to exit on fails
 * 	09-10-24 add logging of cases where block numbers increment by != 1
 * 	08-05-24 prune error output dumping exception data
 * 	07-30-24 add reinit counter to CacheReinitTask
 * 	07-15-24 catch ClientConnectionExceptionS during ethGetLogs() calls
 * 	07-10-24 remove all System.exit() calls during init, except in Block sub
 * 	07-09-24 move reinit TimerTask into CacheReinitTask; destroy all old
 * 			 subscriptions on successful reinit so we don't end up with dups
 * 	07-03-24 trigger a reconnect if we're seen to be lagging behind blocks
 * 	06-19-24 record block progress again in a limited way
 * 	04-14-24 implement BlockTimer and setEventTimer() inside BlockHandler
 * 	04-08-24 rework to do all initial event downloads using Web3j.ethGetLogs()
 * 	03-27-24 additional tweaks to reconnect exception handling
 * 	03-12-24 improve exception error handling; add getLastProcessedBlock()
 * 	03-08-24 implement onError(Throwable) methods to restart subscriptions
 * 	01-05-24 GreyListAddEventResponse.audId is now a string
 * 	12-20-23 add getGreylistedIDs() and getUngreylistedIDs(); make
 * 			 isGreyListed() smarter
 * 	12-18-23 pass EnftListener.recordBlock() from BlockHandler
 * 	11-28-23 call an EnftListener if one is registered
 * 	10-16-23 debug event listeners
 * 	07-06-23 debug reconnect logic (initialize(true))
 * 	06-30-23 add caching of GreyListAdd and GreyListDeletion events; also
 * 			 subscribe to blocks without transactions
 * 	06-27-23 implement timer mechanism to get around ReactiveX bug where end
 * 			 events are never sent to the callback
 * 	06-21-23 new
 *
 * purpose---
 * 	provide a data cache for downloaded event data related to eNFTs
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;

import org.web3j.protocol.Web3j;
import org.web3j.protocol.core.methods.request.EthFilter;
import org.web3j.protocol.core.methods.response.EthLog;
import org.web3j.protocol.core.methods.response.EthBlockNumber;
import org.web3j.protocol.core.methods.response.EthBlock;
import org.web3j.protocol.core.methods.response.EthChainId;
import org.web3j.protocol.core.DefaultBlockParameterNumber;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.protocol.exceptions.ClientConnectionException;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.Keys;
import org.web3j.tx.Contract;
import org.web3j.tx.gas.DefaultGasProvider;
import org.web3j.utils.Numeric;
import org.web3j.abi.EventEncoder;
import org.web3j.abi.EventValues;
import org.web3j.abi.datatypes.Array;
import io.reactivex.Flowable;
import io.reactivex.disposables.Disposable;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Action;
import org.java_websocket.exceptions.WebsocketNotConnectedException;

import java.util.Collections;
import java.util.ArrayList;
import java.util.List;
import java.util.NavigableSet;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.RejectedExecutionHandler;
import java.util.concurrent.TimeUnit;
import java.math.BigInteger;
import java.io.IOException;


/**
 * Provide a data cache through which methods in {@link BlockchainAPI} can
 * access pre-downloaded data emitted into the blockchain's event log through
 * EnshroudProtocol events TransferSingle and URI.  This cache must be loaded
 * successfully before this node can accept client requests related to this
 * chain.
 */
public final class EnftCache implements RejectedExecutionHandler {
	// BEGIN data members
	/**
	 * number of seconds pause between data retrieval attempts
	 */
	private final long				M_PauseSecs = 5L;

	/**
	 * max value allowed for CacheReinitTask.m_ReinitCounter
	 */
	private final int				M_ReinitMax = 2;

	/**
	 * the smart contract config of which we're a part
	 */
	private SmartContractConfig		m_SCC;
	
	/**
	 * reference to the Web3j ABI access session for this chain
	 */
	private Web3j					m_Web3;

	/**
	 * the initialized (loaded) EnshroudProtocol wrapper
	 */
	private EnshroudProtocol		m_Wrapper;

	/**
	 * the latest deployed address for the EnshroudProtocol on this chain
	 */
	private String					m_ProtocolAddr;

	/**
	 * the latest block number we've seen
	 */
	private volatile BigInteger		m_LatestBlock;

	/**
	 * The last block we processed (a previous m_LatestBlock value).  Does not
	 * begin to increment until m_SCC.isEnabled().
	 */
	private volatile BigInteger		m_LastProcessedBlock;

	/**
	 * the lower limit block number from which we begin downloading events
	 */
	private BigInteger				m_StartInitBlock;

	/**
	 * the higher limit block number up to which we download events
	 */
	private BigInteger				m_StopInitBlock;

	/**
	 * flag indicating that we're doing startup initialization (or reinit)
	 */
	private boolean					m_DoingInit;

	/**
	 * the current block as of the last time the progress check timer went off
	 */
	private BigInteger				m_LastCheckedBlock;

	/**
	 * thread pool executor used to schedule all delayed retry tasks
	 */
	private ScheduledThreadPoolExecutor	m_RetryExecutor;

	/**
	 * Mapping of all minted eNFTs, indexed by owner to list of IDs.
	 * NB: we use BigInteger for IDs (uint256 in the contract) because of
	 * efficient sorting of twos-complement values.  However IDs as actually
	 * stored in eNFTs and utilized in MVOs and the dApp are zero-padded hex
	 * strings of length=64.  Therefore all access methods must take this
	 * into account by converting inputs/outputs appropriately.  Address keys
	 * are EIP-55 hex strings without 0x prefixed to make sorting easier.
	 */
	private ConcurrentSkipListMap<String, List<BigInteger>>	m_EnftsMinted;

	/**
	 * Mapping of all burned eNFTs, indexed by owner to list of IDs.  See
	 * notation re BigInteger usage for m_EnftsMinted.  Address keys are EIP-55
	 * hex strings without 0x prefixed.
	 */
	private ConcurrentSkipListMap<String, List<BigInteger>>	m_EnftsBurned;

	/**
	 * Mapping of all eNFT metadata (both minted and burned), indexed by ID.
	 * See notation re BigInteger usage for m_EnftsMinted.  The stored metadata
	 * values are JSON with the encoded strings in encrypted Base64 format,
	 * with the smart contract's baseURI prefix value stripped.
	 */
	private ConcurrentSkipListMap<BigInteger, String>	m_EnftMetadata;

	/**
	 * list of all the greylisted eNFT IDs
	 */
	private ConcurrentSkipListSet<BigInteger>	m_EnftsGreyListed;

	/**
	 * listener for mint and burn events, used to record block, mint, burn,
	 * and greylist events for use in other contexts
	 */
	private EnftListener			m_MintBurnListener;

	/**
	 * rebuilder for starting up a replacement EnftCache on a fresh ABI session
	 * for cases when we experience irrecoverable error conditions
	 */
	private EnftCacheResetter		m_CacheResetter;

	/**
	 * inner class to supply callbacks for events, errors, and completions
	 * on TransferSingle event downloads
	 */
	private final class TransferSingleHandler implements
		Consumer<EnshroudProtocol.TransferSingleEventResponse>, Action
	{
		/**
		 * inner class to provide a task which we can invoke for retries
		 */
		private final class TransferSingleRetry implements Runnable {
			/**
			 * nullary constructor
			 */
			public TransferSingleRetry() { }

			/**
			 * task run on expiration of wait interval
			 */
			public void run() {
				/* Having implemented a wait after a failure (either of the
				 * initial download or of the ongoing subscription), we need
				 * to do a re-init of all TransferSingle events in order to
				 * recover and ensure that we aren't missing any valid events.
				 * NB: we assume that the start and stop block values are good.
				 */
				loadENFTids();
			}
		}

        /**
         * retry task
         */
        private TransferSingleRetry m_RetryTask;

		/**
		 * init status (appears to be synchronized with blockchain)
		 */
		public volatile boolean		m_Init;

		/**
		 * current block to start loading events
		 */
		public BigInteger			m_StartBlock;

		/**
		 * current block to stop loading events
		 */
		public BigInteger			m_StopBlock;


		/**
		 * initiate retry
		 */
		public void attemptRetry() {
			if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.schedule(m_RetryTask, M_PauseSecs,
										 TimeUnit.SECONDS);
			}
		}

		// methods to implement the Consumer and Action interfaces
		/**
		 * method to consume events downloaded by loadENFTids() or retrieved
		 * from m_TransferSingleSub
		 * @param event the data extracted from the event log
		 * @throws Exception theoretically
		 */
		public void accept(EnshroudProtocol.TransferSingleEventResponse event)
			throws Exception
		{
			final String lbl = this.getClass().getSimpleName() + ".accept: ";

			// during init, reject anything after our stop block setting
			BigInteger evBlock = event.log.getBlockNumber();
			if (m_Init || evBlock.compareTo(m_StopBlock) < 1) {
				// we don't need operator (msg.sender) address for anything atm
				String opAddr = Numeric.cleanHexPrefix(
									Keys.toChecksumAddress(event.operator));
				final String
					zeroAddr = "0000000000000000000000000000000000000000";
				// strip 0x from these to facilitate sort by natural ordering
				String toAddr
					= Numeric.cleanHexPrefix(Keys.toChecksumAddress(event.to));
				String fromAddr = Numeric.cleanHexPrefix(
											Keys.toChecksumAddress(event.from));
				BigInteger eId = event.id;
				String idZP = Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
				// NB: the amount will be 1 for both a mint and a burn (ignored)
				m_Log.debug(lbl + "TransferSingleEvent on " +
							m_SCC.getChainName() + ", with msg.sender = "
							+ opAddr + ", to = " + toAddr + ", from = "
							+ fromAddr + ", Id = " + idZP + ", block = "
							+ evBlock);

				// if from == 0 and to != 0, it's a mint
				if (!toAddr.equals(zeroAddr) && fromAddr.equals(zeroAddr)) {
					// see if we have an entry in map for toAddr already
					List<BigInteger> bucket = m_EnftsMinted.get(toAddr);
					if (bucket == null) {
						// initialize bucket
						bucket = Collections.synchronizedList(
												new ArrayList<BigInteger>(100));
						m_EnftsMinted.put(toAddr, bucket);
					}
					if (!bucket.contains(eId)) {
						bucket.add(eId);
						// pass event to listener if we have one set
						if (m_MintBurnListener != null) {
							m_MintBurnListener.recordEnftMint(
															m_SCC.getChainId(),
															eId,
															toAddr,
															evBlock);
						}
					}
				}
				else if (toAddr.equals(zeroAddr) && !fromAddr.equals(zeroAddr))
				{
					// see if we have an entry in map for fromAddr already
					List<BigInteger> bucket = m_EnftsBurned.get(fromAddr);
					if (bucket == null) {
						// initialize bucket
						bucket = Collections.synchronizedList(
												new ArrayList<BigInteger>(100));
						m_EnftsBurned.put(fromAddr, bucket);
					}
					if (!bucket.contains(eId)) {
						bucket.add(eId);
						// pass event to listener if we have one set
						if (m_MintBurnListener != null) {
							m_MintBurnListener.recordEnftBurn(
															m_SCC.getChainId(),
															eId,
															fromAddr,
															evBlock);
						}
					}
				}
				else {
					m_Log.error(lbl + "impossible TransferSingle event, "
								+ "neither a mint nor a burn: operator = "
								+ opAddr + ", from = " + fromAddr + ", to = "
								+ toAddr + " (" + m_SCC.getChainName() + ")");
					throw new Exception("Non-mint, non-burn "
										+ "TransferSingle event");
				}
			}
		}

		/**
		 * method to consume errors occuring on m_TransferSingleSub
		 * -- reconnects to publisher to pick up from the block where the error
		 * occurred
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on TransferSingle subscription, init "
						+ "status = " + m_Init
						+ " (" + m_SCC.getChainName() + ")");

			// record that we're not enabled anymore until we recover
			m_SCC.setEnabled(false);
			m_Init = false;

			if (m_DoingInit) {
				// NB: we leave the start and stop blocks the same
			}
			else {
				// record new start and stop points (begin at -conf blocks)
				BigInteger confBlocks
					= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
				m_StartBlock = m_LatestBlock.subtract(confBlocks);
				m_StopBlock = m_LastProcessedBlock;
				if (m_StartBlock.compareTo(m_StopBlock) > 0) {
					m_StartBlock = m_StopBlock.subtract(BigInteger.ONE);
				}
			}

			// cancel any previous subscription now voided by exception
			if (m_TransferSingleSub != null
				&& !m_TransferSingleSub.isDisposed())
			{
				try {
					m_TransferSingleSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}

			m_Log.debug(lbl + "retrying TransferSingle event download from "
						+ m_StartBlock + " after interval ("
						+ m_SCC.getChainName() + ")");
			// try again after interval
			attemptRetry();
		}

		/**
		 * method to handle the completion action after last event retrieved.
		 * NB: this method is never called by Flowable publishers, apparently
		 * due to a bug in the io.ReactiveX libraries.  However we do invoke
		 * it manually to establish the ongoing subscription once the main
		 * event download has succeeded.
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			final String lbl = this.getClass().getSimpleName() + ".run: ";

			/* Restart subscription from m_StopBlock+1 to LATEST,
			 * using 2-arg subscribe().  This will allow us to catch up with
			 * any events we missed in the latest block, or during downloading.
			 * It will continue running until we call shutdown().
			 *
			 * NB: there appears to be a bug in the io.reactivex.Flowable code
			 * which causes the most recent block's events to not get returned
			 * when the stop block is set to LATEST.  To mitigate this situation
			 * we'll select as our *starting* block m_StopBlock+1 or
			 * m_LatestBlock-1, whichever is less.  In this way we'll be
			 * guaranteed to fetch the last block's events, as soon as the next
			 * block is generated.  Note that on non-advancing blockchains such
			 * as Ganache (where we also didn't subtract 1 from latest to get
			 * the stopBlock), this implies we'll download the stopBlock's
			 * events 2x.  (This is not a problem, only a minor inefficiency.)
			 */
			BigInteger stopPlusOne = m_StopBlock.add(BigInteger.ONE);
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(
					stopPlusOne.min(m_LatestBlock.subtract(BigInteger.ONE)));
			DefaultBlockParameterName endBlock
				= DefaultBlockParameterName.LATEST;

			// create a publisher to fetch all TransferSingleS between these pts
			Flowable<EnshroudProtocol.TransferSingleEventResponse> xferSinglePub
				= m_Wrapper.transferSingleEventFlowable(startBlock, endBlock);

			// use form with callback & error handler
			m_TransferSingleSub = xferSinglePub.subscribe(m_TransferHandler,
							throwable -> m_TransferHandler.onError(throwable));
			m_StartBlock = m_StopBlock;
			m_Init = true;
		}

		/**
		 * nullary constructor
		 */
		public TransferSingleHandler() {
			// begin at system start/stop blocks
			m_StartBlock = m_StartInitBlock;
			m_StopBlock = m_StopInitBlock;
			m_RetryTask = new TransferSingleRetry();
		}
	}

	/**
	 * handler for TransferSingleEvent subscriptions
	 */
	private TransferSingleHandler	m_TransferHandler;

	/**
	 * our subscription to TransferSingle events (one open at a time)
	 */
	private Disposable				m_TransferSingleSub;

	/**
	 * inner class to supply callbacks for events, errors, and completions
	 * on URI event downloads
	 */
	private final class URIHandler implements
		Consumer<EnshroudProtocol.URIEventResponse>, Action
	{
		/**
		 * inner class to provide a task which we can invoke
		 */
		private final class URIRetry implements Runnable {
			/**
			 * nullary constructor
			 */
			public URIRetry() { }

			/**
			 * task run on expiration of retry interval
			 */
			public void run() {
				/* Having implemented a wait after a failure (either of the
				 * initial download or of the ongoing subscription), we need
				 * to do a re-init of all URI events in order to recover and
				 * ensure that we aren't missing any valid events.
				 * NB: we assume that the start and stop block values are good.
				 */
				loadENFTs();
			}
		}

		/**
		 * retry task
		 */
		private URIRetry			m_RetryTask;

		/**
		 * init status (appears to be synchronized with blockchain)
		 */
		public volatile boolean		m_Init;

		/**
		 * current block to start loading events
		 */
		public BigInteger			m_StartBlock;

		/**
		 * current block to stop loading events
		 */
		public BigInteger			m_StopBlock;


		/**
		 * initiate retry
		 */
		public void attemptRetry() {
			if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.schedule(m_RetryTask, M_PauseSecs,
										 TimeUnit.SECONDS);
			}
		}

		/**
		 * method to consume events downloaded by loadENFTs() or retrieved
		 * from m_URISub
		 * @param event the data from the event log
		 * @throws Exception theoretically
		 */
		public void accept(EnshroudProtocol.URIEventResponse event)
			throws Exception
		{
			final String lbl = this.getClass().getSimpleName() + ".accept: ";

			// during init, reject anything after our stop block setting
			BigInteger evBlock = event.log.getBlockNumber();
			if (m_Init || evBlock.compareTo(m_StopBlock) < 1) {
				BigInteger eId = event.id;
				String idZP = Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
				String rawMetadata = event.value;
				// strip off prefixed baseURI value from this metadata
				final String prefix = m_SCC.getBaseURI();
				if (!rawMetadata.startsWith(prefix)) {
					m_Log.error(lbl + "invalid " + m_SCC.getChainName()
								+ " URI metadata, doesn't start with "
								+ prefix + ": " + rawMetadata);
					throw new Exception("invalid URI metadata");
				}
				String metadata = rawMetadata.substring(prefix.length());
				m_Log.debug("URI metadata seen for eId " + idZP + ", block = "
							+ evBlock + " (" + m_SCC.getChainName() + ")");

				// add to mapping
				if (m_EnftMetadata.putIfAbsent(eId, metadata) != null) {
				/*
					m_Log.warning(lbl + "metadata for ID " + idZP
								+ " already present in map");
				 */
				}
			}
		}

		/**
		 * method to consume errors occuring on m_URISub -- reconnects to
		 * publisher to pick up from the block where the error occurred
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on URI subscription, init status = "
						+ m_Init + " (" + m_SCC.getChainName() + ")");

			// record that we're not enabled anymore until we recover
			m_SCC.setEnabled(false);
			m_Init = false;

			if (m_DoingInit) {
				// NB: leave the start/stop blocks the same
			}
			else {
				// record new start and stop points (begin at -conf blocks)
				BigInteger confBlocks
					= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
				m_StartBlock = m_LatestBlock.subtract(confBlocks);
				m_StopBlock = m_LastProcessedBlock;
				if (m_StartBlock.compareTo(m_StopBlock) > 0) {
					m_StartBlock = m_StopBlock.subtract(BigInteger.ONE);
				}
			}

			// cancel any previous subscription now voided by exception
			if (m_URISub != null && !m_URISub.isDisposed()) {
				try {
					m_URISub.dispose();
				} 
				catch (Exception e) { /* ignore */ }
			}

			m_Log.debug(lbl + "retrying URI event download from "
						+ m_StartBlock + " after interval ("
						+ m_SCC.getChainName() + ")");
			// try again after interval
			attemptRetry();
		}

		/**
		 * method to handle the completion action after last event retrieved.
		 * NB: this method is never called by Flowable publishers, apparently
		 * due to a bug in the io.ReactiveX libraries.  However we do invoke
		 * it manually to establish the ongoing subscription once the main
		 * event download has succeeded.
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			//final String lbl = this.getClass().getSimpleName() + ".run: ";

			/* Restart subscription from m_StopBlock+1 to LATEST,
			 * using 2-arg subscribe().  This will allow us to catch up with
			 * any events we missed in the latest block, or during downloading.
			 * It will continue running until we call shutdown().
			 *
			 * NB: there appears to be a bug in the io.reactivex.Flowable code
			 * which causes the most recent block's events to not get returned
			 * when the stop block is set to LATEST.  To mitigate this situation
			 * we'll select as our *starting* block m_StopBlock+1 or
			 * m_LatestBlock-1, whichever is less.  In this way we'll be
			 * guaranteed to fetch the last block's events, as soon as the next
			 * block is generated.  Note that on non-advancing blockchains such
			 * as Ganache (where we also didn't subtract 1 from latest to get
			 * the stopBlock), this implies we'll download the stopBlock's
			 * events 2x.  (This is not a problem, only a minor inefficiency.)
			 */
			BigInteger stopPlusOne = m_StopBlock.add(BigInteger.ONE);
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(
					stopPlusOne.min(m_LatestBlock.subtract(BigInteger.ONE)));
			DefaultBlockParameterName endBlock
				= DefaultBlockParameterName.LATEST;

			// create a publisher to fetch all URI events between these points
			Flowable<EnshroudProtocol.URIEventResponse> uriPub
				= m_Wrapper.uRIEventFlowable(startBlock, endBlock);

			// use form with callback & error handler
			m_URISub = uriPub.subscribe(m_URIHandler,
								throwable -> m_URIHandler.onError(throwable));
			m_StartBlock = m_StopBlock;
			m_Init = true;
		}

		/**
		 * nullary constructor
		 */
		public URIHandler() {
			// begin at system start/stop blocks
			m_StartBlock = m_StartInitBlock;
			m_StopBlock = m_StopInitBlock;
			m_RetryTask = new URIRetry();
		}
	}

	/**
	 * handler for URI event subscriptions
	 */
	private URIHandler				m_URIHandler;

	/**
	 * our subscription to URI events (one open at a time)
	 */
	private Disposable				m_URISub;

	/**
	 * inner class to supply callbacks for events and errors on block downloads
	 */
	private final class BlockHandler implements Consumer<EthBlock> {
		/**
		 * inner class to provide a retry task which we can invoke
		 */
		private final class BlockRetry implements Runnable {
			/**
			 * nullary constructor
			 */
			public BlockRetry() { }

			/**
			 * task run on expiration of interval
			 */
			public void run() {
				/* After a wait following a failure on our subscription,
				 * we need to re-init the Block subscription.
				 */
				if (!m_Init) {
					// clear any existing subscription to prevent memory leak
					if (m_BlockSub != null && !m_BlockSub.isDisposed()) {
						try {
							m_BlockSub.dispose();
						}
						catch (Exception e) { /* ignore */ }
					}
					loadBlocks();
				}
			}
		}

		/**
		 * retry task
		 */
		private BlockRetry			m_RetryTask;

		/**
		 * init status (appears to be synchronized with blockchain)
		 */
		public volatile boolean		m_Init;

		/**
		 * initiate retry
		 */
		public void attemptRetry() {
			if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.schedule(m_RetryTask, M_PauseSecs,
										 TimeUnit.SECONDS);
			}
		}

		/**
		 * method to consume events retrieved from m_BlockSub
		 * @param event the data from the blockchain
		 * @throws Exception theoretically
		 */
		public void accept(EthBlock block) throws Exception {
			final String lbl = this.getClass().getSimpleName() + ".accept: ";
			if (block == null || block.getBlock() == null) {
				throw new RuntimeException("null EthBlock from subscription");
			}

			// check that block numbers are monotonically increasing
			BigInteger blockNumber = block.getBlock().getNumber();
			BigInteger blockDiff = blockNumber.subtract(m_LatestBlock);
			if (blockDiff.compareTo(BigInteger.ONE) > 0) {
				m_Log.error(lbl + "next block (" + blockNumber + ") on "
							+ m_SCC.getChainName() + " incremented by "
							+ blockDiff + "!");

				// reconnect to this chain if this happens outside init
				if (m_Init && m_SCC.isEnabled() && !m_DoingInit) {
					// cancel old timer task so we can replace it
					if (m_CacheReinitializer != null) {
						m_RetryExecutor.remove(m_CacheReinitializer);
					}

					// create a fresh task for initProgressTimer()
					createReinitializerTask(0);
					if (!initialize(true)) {
						// NB: this can only happen due to a configuration error
						m_Log.error(lbl + "CRITICAL: unable to reinit on chain "
									+ m_SCC.getChainName()
									+ " after blocks incremented by >1");
						System.exit(-7);
					}
					else {
						// do not process this block, as it may be invalid
						return;
					}
				}
			}
			// check for negative block increment (shouldn't occur)
			else if (m_LatestBlock.compareTo(blockNumber) > 0) {
				blockDiff = blockNumber.subtract(m_LatestBlock);
				m_Log.error(lbl + "next block (" + blockNumber + ") on "
							+ m_SCC.getChainName() + " decremented by "
							+ blockDiff + "!");
				// this block is almost certainly invalid
				return;
			}

			// all we really need is the block number
			m_LatestBlock = blockNumber;

			// this is block progress, so assume this inits us
			if (!m_Init) {
				m_Init = true;
				// account for case where we're the last component to init
				checkEnabled();
			}

			// log every 10x dwell time for this SCC
			BigInteger dwell
				= new BigInteger(Integer.toString(m_SCC.getDwellTime()))
								.multiply(BigInteger.TEN);
			if (dwell.compareTo(BigInteger.ONE) > 0
				&& m_LatestBlock.mod(dwell).equals(BigInteger.ZERO))
			{
				String currThread = Thread.currentThread().toString();
				m_Log.debug(lbl + "latest block on " + m_SCC.getChainName() 
							+ " now " + m_LatestBlock
							+ " (" + currThread + ")");
			}
			if (m_SCC.isEnabled()) {
				/* Record that we have processed subscribed events up to this
				 * block.  There is no guarantee that we've processed all
				 * subscribed events within the current block just received.
				 * But our event subscriptions should sync up between blocks.
				 */
				m_LastProcessedBlock = m_LatestBlock.subtract(BigInteger.ONE);
			}

			// pass event to listener if we have one set
			if (m_MintBurnListener != null) {
				m_MintBurnListener.recordBlock(m_SCC.getChainId(),
											   m_LatestBlock);
			}
		}

		/**
		 * method to consume errors occuring on m_BlockSub (reconnects to pub)
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			if (err instanceof NullPointerException) {
				// log so we can see traceback
				m_Log.error(lbl + "NPE on Block subscription for "
							+ m_SCC.getChainName(), err);
			}
			else {
				m_Log.error(lbl + "exception on Block subscription: "
						+ err.toString() + " (" + m_SCC.getChainName() + ")");
			}

			// record that we're not enabled anymore until we recover
			m_SCC.setEnabled(false);
			m_Init = false;

			/* get the latest block number emitted on the chain at this time
			 * (i.e. try for an immediate recovery before implementing a wait)
			 */
			EthBlockNumber ebn = null;
			try {
				ebn = m_Web3.ethBlockNumber().send();
			}
			catch (Exception ioe) { /* ignore */ }
			if (ebn != null) {
				// record block number seen
				BigInteger recovBlock = ebn.getBlockNumber();
				// check for a decrement
				if (m_LatestBlock.compareTo(recovBlock) > 0) {
					BigInteger blockDiff = recovBlock.subtract(m_LatestBlock);
					m_Log.error(lbl + "next block (" + recovBlock + ") on "
								+ m_SCC.getChainName() + " decremented by "
								+ blockDiff + "!");
					// this block is almost certainly invalid, do not accept
				}
				else {
					m_LatestBlock = recovBlock;
					// assume all blocks previous were processed
					m_LastProcessedBlock
						= m_LatestBlock.subtract(BigInteger.ONE);
					m_Log.debug(lbl + "recovering Block sub from block # "
							+ m_LatestBlock + " on " + m_SCC.getChainName());
				}
			}
			// recreate publisher to fetch all new blocks, after delay
			attemptRetry();
		}

		/**
		 * nullary constructor
		 */
		public BlockHandler() { m_RetryTask = new BlockRetry(); }
	}

	/**
	 * handler for new block subscriptions
	 */
	private BlockHandler			m_BlockHandler;

	/**
	 * our subscription to block events (one open at a time)
	 */
	private Disposable				m_BlockSub;

	/**
	 * inner class to supply callbacks for events, errors, and completions
	 * on GreyListAdd event downloads
	 */
	private final class GreyListAddHandler implements
		Consumer<EnshroudProtocol.GreyListAddEventResponse>, Action
	{
		/**
		 * inner class to provide a task which we can invoke
		 */
		private final class GreyListAddRetry implements Runnable {
			/**
			 * nullary constructor
			 */
			public GreyListAddRetry() { }

			/**
			 * task run on retries
			 */
			public void run() {
				/* Having implemented a wait after a failure (either of the
				 * initial download or of the ongoing subscription), we need
				 * to do a re-init of all GreyListAdd events in order to
				 * recover and ensure that we aren't missing any valid events.
				 * NB: we assume that the start and stop block values are good.
				 */
				loadGreyListAdds();
			}
		}

		/**
		 * retry task
		 */
		private GreyListAddRetry		m_RetryTask;

		/**
		 * init status (appears to be synchronized with blockchain)
		 */
		public volatile boolean			m_Init;

		/**
		 * current block to start loading events
		 */
		public BigInteger				m_StartBlock;

		/**
		 * current block to stop loading events
		 */
		public BigInteger				m_StopBlock;

		/**
		 * list of eNFT IDs which were greylisted more than once
		 */
		public ArrayList<BigInteger>	m_MultipleEntries;


		/**
		 * initiate retry
		 */
		public void attemptRetry() {
			if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.schedule(m_RetryTask, M_PauseSecs,
										 TimeUnit.SECONDS);
			}
		}

		/**
		 * method to consume events retrieved from m_GreyListAddSub
		 * @param event the data from the event log
		 * @throws Exception theoretically
		 */
		public void accept(EnshroudProtocol.GreyListAddEventResponse event)
			throws Exception
		{
			final String lbl = this.getClass().getSimpleName() + ".accept: ";

			// during init, reject anything after our stop block setting
			BigInteger evBlock = event.log.getBlockNumber();
			if (m_Init || evBlock.compareTo(m_StopBlock) < 1) {
				String audId = event.audId;
				// NB: we can ignore the matching reasons, only need eNFT IDs
				for (BigInteger eId : event.ids) {
					String numId
						= Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
					if (m_EnftsGreyListed.add(eId)) {
						m_Log.debug(lbl + "AUDId " + audId + " greylisted Id: "
									+ numId + " in block " + evBlock
									+ " on " + m_SCC.getChainName());
					}
					else {
						m_Log.debug(lbl + "AUDId " + audId + " greylisted Id: "
									+ numId + " AGAIN in block " + evBlock
									+ " on " + m_SCC.getChainName());
						// add to list of multiple entries for mergeGreyList()
						m_MultipleEntries.add(eId);
					}
				}
			}
		}

		/**
		 * method to consume errors occuring on m_GreyListAddSub -- reconnects
		 * to the publisher to pick up from the block where the error occurred
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on GreyListAdd subscription on "
						+ m_SCC.getChainName() + ", init status = " + m_Init);

			// record that we're not enabled anymore until we recover
			m_SCC.setEnabled(false);
			m_Init = false;

			if (m_DoingInit) {
				// leave block boundaries the same
			}
			else {
				// record new start and stop points (begin at -conf blocks)
				BigInteger confBlocks
					= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
				m_StartBlock = m_LatestBlock.subtract(confBlocks);
				m_StopBlock = m_LastProcessedBlock;
				if (m_StartBlock.compareTo(m_StopBlock) > 0) {
					m_StartBlock = m_StopBlock.subtract(BigInteger.ONE);
				}
			}

			// cancel any previous subscription now voided by exception
			if (m_GreyListAddSub != null && !m_GreyListAddSub.isDisposed()) {
				try {
					m_GreyListAddSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}

			m_Log.debug(lbl + "retrying GreyListAdd event download from "
						+ m_StartBlock + " after interval ("
						+ m_SCC.getChainName() + ")");
			// try again after interval
			attemptRetry();
		}

		/**
		 * method to handle the completion action after last event retrieved.
		 * NB: this method is never called by Flowable publishers, apparently
		 * due to a bug in the io.ReactiveX libraries.  However we do invoke
		 * it manually to establish the ongoing subscription once the main
		 * event download has succeeded.
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			//final String lbl = this.getClass().getSimpleName() + ".run: ";

			/* Restart subscription from m_StopBlock+1 to LATEST,
			 * using 2-arg subscribe().  This will allow us to catch up with
			 * any events we missed in the latest block, or during downloading.
			 * It will continue running until we call shutdown().
			 *
			 * NB: there appears to be a bug in the io.reactivex.Flowable code
			 * which causes the most recent block's events to not get returned
			 * when the stop block is set to LATEST.  To mitigate this situation
			 * we'll select as our *starting* block m_StopBlock+1 or
			 * m_LatestBlock-1, whichever is less.  In this way we'll be
			 * guaranteed to fetch the last block's events, as soon as the next
			 * block is generated.  Note that on non-advancing blockchains such
			 * as Ganache (where we also didn't subtract 1 from latest to get
			 * the stopBlock), this implies we'll download the stopBlock's
			 * events 2x.  (This is not a problem, only a minor inefficiency.)
			 */
			BigInteger stopPlusOne = m_StopBlock.add(BigInteger.ONE);
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(
					stopPlusOne.min(m_LatestBlock.subtract(BigInteger.ONE)));
			DefaultBlockParameterName endBlock
				= DefaultBlockParameterName.LATEST;

			// create a publisher to fetch all  between these
			Flowable<EnshroudProtocol.GreyListAddEventResponse> greyListAddPub
				= m_Wrapper.greyListAddEventFlowable(startBlock, endBlock);

			// use form with callback & error handler
			m_GreyListAddSub = greyListAddPub.subscribe(m_GreyListAddHandler,
						throwable -> m_GreyListAddHandler.onError(throwable));

			// if the GreyListDeletion download is also complete, merge
			if (m_GreyListDeletionHandler.m_Init) {
				m_GreyListDeletionHandler.mergeGreyList();
			}
			m_StartBlock = m_StopBlock;
			m_Init = true;
		}

		/**
		 * nullary constructor
		 */
		public GreyListAddHandler() {
			m_MultipleEntries = new ArrayList<BigInteger>();
			// begin at system start/stop blocks
			m_StartBlock = m_StartInitBlock;
			m_StopBlock = m_StopInitBlock;
			m_RetryTask = new GreyListAddRetry();
		}
	}

	/**
	 * handler for GreyListAdd subscriptions
	 */
	private GreyListAddHandler		m_GreyListAddHandler;

	/**
	 * our subscription to GreyListAdd events (one open at a time)
	 */
	private Disposable				m_GreyListAddSub;

	/**
	 * inner class to supply callbacks for events, errors, and completions
	 * on GreyListDeletion event downloads
	 */
	private final class GreyListDeletionHandler implements
		Consumer<EnshroudProtocol.GreyListDeletionEventResponse>, Action
	{
		/**
		 * inner class to provide a task which we can invoke
		 */
		private final class GreyListDeletionRetry implements Runnable {
			/**
			 * nullary constructor
			 */
			public GreyListDeletionRetry() { }

			/**
			 * timer task run on expiration of m_EventTimer
			 */
			public void run() {
				/* Having implemented a wait after a failure (either of the
				 * initial download or of the ongoing subscription), we need
				 * to do a re-init of all GreyListDeletion events in order to
				 * recover and ensure that we aren't missing any valid events.
				 * NB: we assume that the start and stop block values are good.
				 */
				loadGreyListDeletions();
			}
		}

		/**
		 * retry task
		 */
		private GreyListDeletionRetry	m_RetryTask;

		/**
		 * init status (appears to be synchronized with blockchain)
		 */
		public volatile boolean			m_Init;

		/**
		 * list of eNFT IDs which were de-greylisted
		 */
		public ArrayList<BigInteger>	m_UngreylistedEntries;

		/**
		 * current block to start loading events
		 */
		public BigInteger				m_StartBlock;

		/**
		 * current block to stop loading events
		 */
		public BigInteger				m_StopBlock;


		/**
		 * initiate retry
		 */
		public void attemptRetry() {
			if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.schedule(m_RetryTask, M_PauseSecs,
										 TimeUnit.SECONDS);
			}
		}

		/**
		 * method to consume events retrieved from m_GreyListDeletionSub
		 * @param event the data from the event log
		 * @throws Exception theoretically
		 */
		public void accept(EnshroudProtocol.GreyListDeletionEventResponse event)
			throws Exception
		{
			//final String lbl = this.getClass().getSimpleName() + ".accept: ";

			// during init, reject anything after our stop block setting
			BigInteger evBlock = event.log.getBlockNumber();
			if (m_Init || evBlock.compareTo(m_StopBlock) < 1) {
				/* Record ID in our list if not present.  We check for dups
				 * because our Disposable subscription sometimes hands the same
				 * event (with same txHash) back to us multiple times.
				 */
				if (!m_UngreylistedEntries.contains(event.id)) {
					m_UngreylistedEntries.add(event.id);
					String numId
						= Numeric.toHexStringNoPrefixZeroPadded(event.id, 64);
					if (m_MintBurnListener != null) {
						m_MintBurnListener.recordUnGreylist(m_SCC.getChainId(),
															event.id);
					}
				}
			}
		}

		/**
		 * method to consume errors occuring on m_GreyListDeletionSub
		 * -- reconnects to the subscription to pick up from where it left off
		 * @param err the exception
		 */
		public void onError(Throwable err) {
			final String lbl = this.getClass().getSimpleName() + ".onError: ";
			m_Log.error(lbl + "exception on GreyListDeletion subscription, "
						+ "init status = " + m_Init + " ("
						+ m_SCC.getChainName() + ")");

			// record that we're not enabled anymore until we recover
			m_SCC.setEnabled(false);
			m_Init = false;

			if (m_DoingInit) {
				// leave block boundaries the same
			}
			else {
				// record new start and stop points (begin at -conf blocks)
				BigInteger confBlocks
					= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
				m_StartBlock = m_LatestBlock.subtract(confBlocks);
				m_StopBlock = m_LastProcessedBlock;
				if (m_StartBlock.compareTo(m_StopBlock) > 0) {
					m_StartBlock = m_StopBlock.subtract(BigInteger.ONE);
				}
			}

			// cancel any previous subscription now voided by exception
			if (m_GreyListDeletionSub != null
				&& !m_GreyListDeletionSub.isDisposed())
			{
				try {
					m_GreyListDeletionSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}

			// reload GreyListDeletion events from where we left off & continue
			m_Log.debug(lbl + "retrying GreyListDeletion event download from "
						+ m_StartBlock + " after interval ("
						+ m_SCC.getChainName() + ")");
			// try again after interval
			attemptRetry();
		}

		/**
		 * method to handle the completion action after last event retrieved
		 * (called once per init cycle)
		 * @throws Exception theoretically
		 */
		public void run() throws Exception {
			//final String lbl = this.getClass().getSimpleName() + ".run: ";

			/* Restart subscription from m_StopBlock+1 to LATEST,
			 * using 2-arg subscribe().  This will allow us to catch up with
			 * any events we missed in the latest block, or during downloading.
			 * It will continue running until we call shutdown().
			 *
			 * NB: there appears to be a bug in the io.reactivex.Flowable code
			 * which causes the most recent block's events to not get returned
			 * when the stop block is set to LATEST.  To mitigate this situation
			 * we'll select as our *starting* block m_StopBlock+1 or
			 * m_LatestBlock-1, whichever is less.  In this way we'll be
			 * guaranteed to fetch the last block's events, as soon as the next
			 * block is generated.  Note that on non-advancing blockchains such
			 * as Ganache (where we also didn't subtract 1 from latest to get
			 * the stopBlock), this implies we'll download the stopBlock's
			 * events 2x.  (This is not a problem, only a minor inefficiency.)
			 */
			BigInteger stopPlusOne = m_StopBlock.add(BigInteger.ONE);
			DefaultBlockParameterNumber startBlock
				= new DefaultBlockParameterNumber(
					stopPlusOne.min(m_LatestBlock.subtract(BigInteger.ONE)));
			DefaultBlockParameterName endBlock
				= DefaultBlockParameterName.LATEST;

			// create a publisher to fetch all GreyListDeletionS between these
			Flowable<EnshroudProtocol.GreyListDeletionEventResponse>
				greyListDeletionPub
				= m_Wrapper.greyListDeletionEventFlowable(startBlock, endBlock);

			// use form with callback & error handler
			m_GreyListDeletionSub
				= greyListDeletionPub.subscribe(m_GreyListDeletionHandler,
					throwable -> m_GreyListDeletionHandler.onError(throwable));

			// if the GreyListAdd download is also complete, merge
			if (m_GreyListAddHandler.m_Init) {
				mergeGreyList();
			}
			m_StartBlock = m_StopBlock;
			m_Init = true;
		}

		/**
		 * delete removed items from list, then check for verification of
		 * status of IDs greylisted multiple times via the idToAuditorGreylist
		 */
		public void mergeGreyList() {
			final String lbl
				= this.getClass().getSimpleName() + ".mergeGreyList: ";

			// delete all recorded removed items from the greylist
			for (BigInteger Id : m_UngreylistedEntries) {
				String numId = Numeric.toHexStringNoPrefixZeroPadded(Id, 64);
				if (m_EnftsGreyListed.remove(Id)) {
					m_Log.debug(lbl + "removed deleted greylist Id " + numId
								+ " on " + m_SCC.getChainName());
				}
				// else: removed it previously (normal for previous block items)
			}

			/* Now, if the m_GreyListAddHandler discovered any
			 * entries which were added multiple times, it could be that we've
			 * erroneously removed an entry which really should still be listed.
			 * To catch this case, we'll audit any multiply greylisted Ids
			 * against the smart contract's idToAuditorGreylist mapping value.
			 */
			for (BigInteger multiId : m_GreyListAddHandler.m_MultipleEntries) {
				String numId
					= Numeric.toHexStringNoPrefixZeroPadded(multiId, 64);
				String greyReason = "";
				try {
					greyReason = m_Wrapper.idToAuditorGreylist(multiId).send();
				}
				catch (Exception eee) {
					m_Log.error(lbl + "error checking multiply greylisted Id "
								+ numId + " on " + m_SCC.getChainName(), eee);
				}
				if (greyReason != null && !greyReason.isEmpty()) {
					// add this ID back into the greylist
					if (m_EnftsGreyListed.add(multiId)) {
						m_Log.warning(lbl + "multiply-greylisted eId " + numId
									+ " still greylisted, added back in "
									+ "(" + m_SCC.getChainName() + ")");

					}
					else {
						// this is odd but acceptable, nevertheless log it
						m_Log.error(lbl + "multiply-greylisted eId " + numId
									+ " was still in the greylist, not added "
									 + "(" + m_SCC.getChainName() + ")");
					}
				}
			}

			// possibly enable processing on this chain
			checkEnabled();
		}

		/**
		 * nullary constructor
		 */
		public GreyListDeletionHandler() {
			m_UngreylistedEntries = new ArrayList<BigInteger>();
			// begin at system start/stop blocks
			m_StartBlock = m_StartInitBlock;
			m_StopBlock = m_StopInitBlock;
			m_RetryTask = new GreyListDeletionRetry();
		}
	}

	/**
	 * handler for GreyListDeletion subscriptions
	 */
	private GreyListDeletionHandler	m_GreyListDeletionHandler;

	/**
	 * our subscription to GreyListDeletion events (one open at a time)
	 */
	private Disposable				m_GreyListDeletionSub;

	/**
	 * inner class to provide a task which we can invoke for reconnects
	 */
	private final class CacheReinitTask implements Runnable {
		/**
		 * local count of which iteration this is in successive re-inits
		 */
		private int					m_ReinitCounter;

		/**
		 * nullary constructor
		 * @param counter number of times initialize(true) (a reinit) was
		 * called when we were created
		 */
		public CacheReinitTask(int counter) { m_ReinitCounter = counter; }

		/**
		 * task run on expiration of recheck interval (2 minutes)
		 */
		public void run() {
			final String lbl = this.getClass().getSimpleName() + ".run: ";

			// calculate how many blocks have elapsed since the last invocation
			BigInteger elapsedBlocks
				= m_LatestBlock.subtract(m_LastCheckedBlock);
			if (elapsedBlocks.compareTo(BigInteger.ZERO) <= 0) {
				/* We have experienced a lack of progress on our Block
				 * subscription.  This certainly means that our Block
				 * subscription is hosed, and may mean that event listeners
				 * (such as TransferSingleHandler) are also non-functional.
				 * The most correct action to take here is to reinitialize
				 * our entire JSON-RPC connection to the blockchain.
				 *
				 * Due to a probable bug / race condition in either the
				 * okHttp3 or io.reactiveX libraries, we can get into a state
				 * where a filter doesn't get cancelled, and thereafter no
				 * new replacement filter can download any data.  The effect of
				 * this is that we get into a loop where we keep doing a
				 * reinit (initialize(true)) which appears to succeed, but we
				 * never see any block progress after that.  An undeliverable
				 * exception may be triggered, or not, but either way there is
				 * no way to fix this except by starting over.
				 */
				if (++m_ReinitCounter > M_ReinitMax) {
					// invoke EnftCacheResetter (rather than ShutdownHandler)
					if (m_CacheResetter != null) {
						m_CacheResetter.requestEnftCacheRestart(
								m_SCC.getChainId(),
								"reinit loop on chain " + m_SCC.getChainName()
								+ " more than " + M_ReinitMax + " times");
					}
					else {
						m_Log.error(lbl + "CRITICAL: detected reinit loop on "
									+ "chain " + m_SCC.getChainName()
									+ ", no EnftCacheResetter, exiting app!");
						System.exit(-3);
					}
					return;
				}

				// log situation
				m_Log.error(lbl + "no block progress seen on chain "
							+ m_SCC.getChainName() + "; reinitializing cache, "
							+ "try # " + m_ReinitCounter);
				try {
					// cancel old timer task so we can replace it
					m_RetryExecutor.remove(this);
					createReinitializerTask(m_ReinitCounter);

					// reinitialize the cache for this chain
					if (!initialize(true)) {
						// NB: this can only happen due to a configuration error
						m_Log.error(lbl + "CRITICAL: unable to reinit on chain "
									+ m_SCC.getChainName()
									+ " after block stoppage");
						System.exit(-8);
					}
				}
				catch (Exception eee) {
					// invoke EnftCacheResetter (rather than ShutdownHandler)
					m_Log.error(lbl + "exception thrown during progress timer "
								+ "execution for " + m_SCC.getChainName()
								+ ": " +  eee.toString());
					if (m_CacheResetter != null) {
						m_CacheResetter.requestEnftCacheRestart(
									m_SCC.getChainId(),
									"exception thrown during progress timer "
									+ "execution for " + m_SCC.getChainName());
					}
					else {
						m_Log.error(lbl + "no resetter, exiting app!", eee);
						System.exit(-2);
					}
				}
			}
			else {
				m_LastCheckedBlock = m_LatestBlock;
				m_ReinitCounter = 0;
			}
		}
	}

	/**
	 * the task which checks need for and performs cache reinitializations
	 */
	private CacheReinitTask			m_CacheReinitializer;

	/**
	 * logging object (local copy)
	 */
	private Log						m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param scc the smart contract config we service
	 * @param web3 the ABI access object
	 * @param lastBlock initial stop block for initialize() event downloads 
	 * (will be BigInteger.ZERO if we have no previous cache to inherit from)
	 */
	public EnftCache(SmartContractConfig scc, Web3j web3, BigInteger lastBlock)
	{
		m_SCC = scc;
		m_Web3 = web3;
		m_Log = m_SCC.log();
		m_CacheResetter = m_SCC.getCacheRebuilder();
		// NB: natural ordering is sufficient for sorting all maps
		m_EnftsMinted = new ConcurrentSkipListMap<String, List<BigInteger>>();
		m_EnftsBurned = new ConcurrentSkipListMap<String, List<BigInteger>>();
		m_EnftMetadata = new ConcurrentSkipListMap<BigInteger, String>();
		m_EnftsGreyListed = new ConcurrentSkipListSet<BigInteger>();
		m_TransferHandler = new TransferSingleHandler();
		m_URIHandler = new URIHandler();
		m_BlockHandler = new BlockHandler();
		m_GreyListAddHandler = new GreyListAddHandler();
		m_GreyListDeletionHandler = new GreyListDeletionHandler();
		createReinitializerTask(0);
		m_RetryExecutor = new ScheduledThreadPoolExecutor(32, this);
		m_RetryExecutor.setRemoveOnCancelPolicy(true);
		m_RetryExecutor.setExecuteExistingDelayedTasksAfterShutdownPolicy(
																		false);
		m_LatestBlock = lastBlock;
	}

	// get methods
	/**
	 * obtain the smart contract config
	 * @param the config that owns us
	 */
	public SmartContractConfig getConfig() { return m_SCC; }

	/**
	 * determine whether greylist data has been downloaded (used by Auditor)
	 * @return true if greylist event data is completely downloaded
	 */
	public boolean isGreyListingComplete() {
		/* if both GreyListAddHandler.m_Init and GreyListDeletionHandler.m_Init
		 * are set, then both GreyListAdd and GreyListDeletion events have been
		 * downloaded, and mergeGreyList() has been called
		 */
		return m_GreyListAddHandler.m_Init && m_GreyListDeletionHandler.m_Init;
	}

	/**
	 * obtain the list of minted eNFT Ids for a given account address
	 * @param owner the account address, as EIP-55 without 0x prefix
	 * @return list of eNFT IDs which have been minted to this address, or null
	 * if none exist
	 */
	public ArrayList<BigInteger> getMintedIdsForAccount(String owner) {
		List<BigInteger> mints = m_EnftsMinted.get(owner);
		ArrayList<BigInteger> retList = null;
		if (mints != null) {
			retList = new ArrayList<BigInteger>(mints);
		}
		return retList;
	}

	/**
	 * obtain the list of burned eNFT Ids for a given account address
	 * @param owner the account address, as EIP-55 without 0x prefix
	 * @return list of eNFT IDs which have been burned for this address, or null
	 * if none exist
	 */
	public ArrayList<BigInteger> getBurnedIdsForAccount(String owner) {
		List<BigInteger> burns = m_EnftsBurned.get(owner);
		ArrayList<BigInteger> retList = null;
		if (burns != null) {
			retList = new ArrayList<BigInteger>(burns);
		}
		return retList;
	}

	/**
	 * obtain the list of accounts which have owned previously burned eNFTs
	 * @return list of account addresses, in EIP-55 format without 0x prefixes
	 */
	public NavigableSet<String> getFormerOwners() {
		return m_EnftsBurned.keySet();
	}

	/**
	 * obtain an eNFT's metadata based on its ID (minted or burned)
	 * @param eId the unique ID of the eNFT
	 * @return the metadata in JSON with secret data encrypted Base64 format
	 */
	public String getMetadataForId(BigInteger eId) {
		return m_EnftMetadata.get(eId);
	}

	/**
	 * obtain whether an eNFT is currently greylisted by an Auditor
	 * @param eId the unique ID of the eNFT
	 * @return true if greylisted, else false
	 */
	public boolean isIdGreyListed(BigInteger eId) {
		// first, check to see whether Id has ever been greylisted
		if (!m_EnftsGreyListed.contains(eId)) {
			// never greylisted, return false
			return false;
		}

		// next, check if it's also been ungreylisted
		if (!m_GreyListDeletionHandler.m_UngreylistedEntries.contains(eId)) {
			// never un-greylisted, return true
			return true;
		}

		// next, see whether it's been greylisted multiple times
		if (!m_GreyListAddHandler.m_MultipleEntries.contains(eId)) {
			/* Only greylisted once, but also un-greylisted.  Because only a
			 * greylisted ID can be un-greylisted, the ungreylist op must have
			 * occurred last.  Therefore we can return false here.
			 */
			return false;
		}

		/* Here we have an ID which has been re-greylisted after being
		 * un-greylisted (at least) once.  In this case the only proper thing
		 * to do is consult the status on-chain.
		 */
		String greyReason = "";
		String numId = Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
		try {
			greyReason = m_Wrapper.idToAuditorGreylist(eId).send();
			m_Log.debug("isIdGreyListed: forced to check " + numId
						+ " status on-chain on " + m_SCC.getChainName());
		}
		catch (Exception eee) {
			m_Log.error("isIdGreyListed: error checking multiply greylisted Id "
						+ numId + " (" + m_SCC.getChainName() + ")", eee);
		}
		if (greyReason != null && !greyReason.isEmpty()) {
			/* Still greylisted.  Remove from ungreylisted list to simplify
			 * processing if invoked again.  Return true.
			 */
			m_GreyListDeletionHandler.m_UngreylistedEntries.remove(eId);
			return true;
		}

		/* Not currently greylisted on-chain -- this is dispositive.  Note we
		 * can remove from m_EnftsGreyListed without affecting anything, since
		 * if eId ever gets greylisted again it will be re-added, and is already
		 * present in m_MultipleEntries.
		 */
		m_EnftsGreyListed.remove(eId);
		return false;
	}

	/**
	 * obtain the latest block number
	 * @return the most recently block number we've seen on this chain
	 */
	public BigInteger getLatestBlock() { return m_LatestBlock; }

	/**
	 * obtain the initialized wrapper for the EnshroudProtocol contract
	 * @return the wrapper
	 */
	public EnshroudProtocol getProtocolWrapper() { return m_Wrapper; }

	/**
	 * obtain the list of IDs that have been greylisted
	 * @return the list
	 */
	public ConcurrentSkipListSet<BigInteger> getGreylistedIDs() {
		return m_EnftsGreyListed;
	}

	/**
	 * obtain the list of IDs that have been un-greylisted
	 * @return the list
	 */
	public ArrayList<BigInteger> getUngreylistedIDs() {
		return m_GreyListDeletionHandler.m_UngreylistedEntries;
	}

	/**
	 * obtain the last completely processed block we're aware of
	 * @return the block
	 */
	public BigInteger getLastProcessedBlock() { return m_LastProcessedBlock; }

	/**
	 * create a reinitializer task
	 * @param counter retry counter (0-5)
	 */
	public void createReinitializerTask(int counter) {
		if (counter < 0 || counter > M_ReinitMax) {
			m_Log.error("createReinitializerTask: illegal counter, " + counter
						+ " (" + m_SCC.getChainName() + ")");
			counter = 0;
		}
		m_CacheReinitializer = new CacheReinitTask(counter);
	}

	/**
	 * register a mint/burn event listener (called by MVOs for receipt handling,
	 * and by AUDs for greylisting purposes)
	 * @param listener the method which receives eNFT mint/burn notifications
	 */
	public void registerListener(EnftListener listener) {
		m_MintBurnListener = listener;
	}

	/**
	 * method to trigger initialization of object by downloading chain events
	 * @param reconnect true if reinitializing after losing ABI connection
	 * @return true iff download is successfully started
	 */
	public boolean initialize(boolean reconnect) {
		final String lbl = this.getClass().getSimpleName() + ".initialize: ";
		if (m_SCC.isEnabled()) {
			if (!reconnect) {
				m_Log.warning(lbl + "called after SCC already enabled for "
							+ m_SCC.getChainName());
			}
		}
		m_DoingInit = !reconnect;
		m_Log.debug(lbl + "for chain " + m_SCC.getChainId() + " ("
					+ m_SCC.getChainName() + "), reconnect = " + reconnect);

		// handle case where we do a full reinit a second time while running
		if (m_DoingInit) {
			m_EnftsMinted.clear();
			m_EnftsBurned.clear();
			m_EnftMetadata.clear();
			m_EnftsGreyListed.clear();
			m_GreyListAddHandler.m_MultipleEntries.clear();
			m_GreyListDeletionHandler.m_UngreylistedEntries.clear();
		}
		else {
			// purge all existing tasks & subscriptions so we don't create dups
			shutdown();
			// NB: our caller was expected to create m_CacheReinitializer task

			// replace the executor object after shutdown() killed any tasks
			m_RetryExecutor = new ScheduledThreadPoolExecutor(32, this);
			m_RetryExecutor.setRemoveOnCancelPolicy(true);
			m_RetryExecutor.setExecuteExistingDelayedTasksAfterShutdownPolicy(
																		false);
		}

		// reset init states
		m_TransferHandler.m_Init = false;
		m_URIHandler.m_Init = false;
		m_GreyListAddHandler.m_Init = false;
		m_GreyListDeletionHandler.m_Init = false;
		m_BlockHandler.m_Init = false;

		// get the deployed EnshroudProtocol contract for this chain
		m_ProtocolAddr = EnshroudProtocol.getPreviouslyDeployedAddress(
											Long.toString(m_SCC.getChainId()));
		if (m_ProtocolAddr == null) {
			m_Log.error(lbl + "EnshroudProtocol contract not deployed on chain "
						+ "Id " + m_SCC.getChainId());
			return false;
		}

		// since we're strictly view-only, can dummy up Credentials (and no gas)
		Credentials credentials = null;
		try {
			credentials = Credentials.create(Keys.createEcKeyPair());
		}
		catch (Exception eee) {
			m_Log.error(lbl + "exception generating dummy Credentials", eee);
			return false;
		}
		m_Wrapper = EnshroudProtocol.load(m_ProtocolAddr, m_Web3, credentials,
										  new DefaultGasProvider());

		/* Confirm that the JSON-RPC node we're connected to serves
		 * the correct chain.  This is in case a DNS or routing misconfig
		 * causes us to (re)connect to a node serving the wrong blockchain.
		 * If this happens during a first pass initialization, or in the
		 * absence of a previously recorded m_LastestBlock, we must exit.
		 */
		EthChainId eci = null;
		try {
			eci = m_Web3.ethChainId().send();
		}
		catch (Exception ioe) { /* ignore */ }
		if (eci != null) {
			BigInteger servedChain = eci.getChainId();
			if (servedChain != null
				&& servedChain.longValue() != m_SCC.getChainId())
			{
				m_Log.error(lbl + "CRITICAL: RPC node is serving chainId "
							+ servedChain + ", while we expected "
							+ m_SCC.getChainId());
				// precipitate exit if doing init
				if (m_DoingInit) return false;

				/* Do not get a latest block from the wrong chain.  Unless we
				 * inherited a previous latest block, this must be fatal.
				 */
				if (m_LatestBlock == null
					|| m_LatestBlock.equals(BigInteger.ZERO))
				{
					m_Log.error(lbl + "CRITICAL: no latest block available "
								+ "for " + m_SCC.getChainName()
								+ ", exiting");
					System.exit(-1);
				}
			}
			else {
				// get the latest block number emitted on the chain at this time
				EthBlockNumber ebn = null;
				try {
					ebn = m_Web3.ethBlockNumber().send();
				}
				catch (Exception ioe) { /* ignore */ }
				if (ebn == null) {
					m_Log.error(lbl + "unable to obtain current block number "
								+ "on " + m_SCC.getChainName());
					// this must be fatal unless we inherited a previous latest
					if (m_LatestBlock == null
						|| m_LatestBlock.equals(BigInteger.ZERO))
					{
						m_Log.error(lbl + "CRITICAL: no latest block available "
									+ "for " + m_SCC.getChainName()
									+ ", exiting");
						System.exit(-1);
					}
				}
				else {
					m_LatestBlock = ebn.getBlockNumber();
				}
			}
		}

		/* To avoid duplicate event downloads, we'll preload from Logs all that
		 * occur from deployment to the current block less one.  The event
		 * subscriptions will then start beginning at the current block to
		 * LATEST, which may encompass more than one block depending upon how
		 * long the download process takes.
		 */
		if (m_SCC.getChainName().startsWith("Ganache")
			|| m_SCC.getChainId() == 1337L)
		{
			/* because this chain doesn't increment blocks absent new
			 * transactions occurring, we do not subtract 1 from latest block
			 * so that we'll download all the available events
			 */
			m_StopInitBlock = m_LatestBlock;
		}
		else {
			m_StopInitBlock = m_LatestBlock.subtract(BigInteger.ONE);
		}
		if (m_DoingInit) {
			// assume we'll process up to here due to subscription block range
			m_LastProcessedBlock = m_StopInitBlock;
			// start at "genesis" (block where smart contract was deployed)
			m_StartInitBlock = m_SCC.getDeploymentBlock();
		}
		else {
			/* NB: m_LastProcessedBlock value must be retained, but due to the
			 * possibility that we dropped some blocks before the exception
			 * that brought us here was raised, subtract the confirmation
			 * blocks count ("dwell time" for the chain).
			 */
			BigInteger confBlocks
				= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
			m_StartInitBlock = m_LastProcessedBlock.subtract(confBlocks);
			if (m_StartInitBlock.compareTo(m_StopInitBlock) > 0) {
				m_StartInitBlock = m_StopInitBlock;
			}
		}

		/* if we have a valid startBlock that's earlier and from a previous
		 * failed download attempt (i.e. that never hit run() or got onError()),
		 * then retain it so we don't risk a gap in our downloaded events
		 */
		if (m_TransferHandler.m_StartBlock != null
			&& m_TransferHandler.m_StartBlock.compareTo(
											m_TransferHandler.m_StopBlock) < 0)
		{
			m_TransferHandler.m_StartBlock
				= m_TransferHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_TransferHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_URIHandler.m_StartBlock != null
			&& m_URIHandler.m_StartBlock.compareTo(
												m_URIHandler.m_StopBlock) < 0)
		{
			m_URIHandler.m_StartBlock
				= m_URIHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_URIHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_GreyListAddHandler.m_StartBlock != null
			&& m_GreyListAddHandler.m_StartBlock.compareTo(
										m_GreyListAddHandler.m_StopBlock) < 0)
		{
			m_GreyListAddHandler.m_StartBlock
				= m_GreyListAddHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_GreyListAddHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_GreyListDeletionHandler.m_StartBlock != null
			&& m_GreyListDeletionHandler.m_StartBlock.compareTo(
									m_GreyListDeletionHandler.m_StopBlock) < 0)
		{
			m_GreyListDeletionHandler.m_StartBlock
				= m_GreyListDeletionHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_GreyListDeletionHandler.m_StartBlock = m_StartInitBlock;
		}
		m_TransferHandler.m_StopBlock = m_StopInitBlock;
		m_URIHandler.m_StopBlock = m_StopInitBlock;
		m_GreyListAddHandler.m_StopBlock = m_StopInitBlock;
		m_GreyListDeletionHandler.m_StopBlock = m_StopInitBlock;

		if (!m_DoingInit) {
			// clear any existing block sub to prevent duplicate subscriptions
			if (m_BlockSub != null && !m_BlockSub.isDisposed()) {
				try {
					m_BlockSub.dispose();
				}
				catch (Exception e) { /* ignore */ }
			}
		}

		// subscribe to all blocks from this point forward
		loadBlocks();

		// download all eNFT mint/burn events, and build maps of ownership
		loadENFTids();

		// download all eNFT encrypted metadata (URI events) and add to map
		loadENFTs();

		// download all GreyListAdd events, and build list of greylisted Ids
		loadGreyListAdds();

		// download all GreyListDeletion events, and build list of removed Ids
		loadGreyListDeletions();

		// we can now skip special handling in onError() handlers
		m_DoingInit = false;

		// start running the block progress check on timer (task exists)
		initProgressTimer();

		/* NB: this return is synchronous.  All init downloads should be done,
		 * unless there was a problem getting EthLogs for one or more events.
		 * In that case there will be a timer running to retry, and we won't
		 * actually enable this blockchain until it succeeds.
		 */
		return true;
	}

	/**
	 * method to initialize the progress timer for this blockchain
	 */
	private void initProgressTimer() {
		final String lbl
			= this.getClass().getSimpleName() + ".initProgressTimer: ";

		// on an EVM-compatible blockchain, 2 min without a block would be odd
		long progressInterval = 2L;
		if (m_SCC.getChainName().startsWith("Ganache")
			|| m_SCC.getChainId() == 1337L)
		{
			// skip progress timer on Ganache because blocks do not auto-advance
			m_Log.debug(lbl + "skipping progress timer init on non-advancing "
						+ "testnet Ganache");
			// force initialization of BlockHandler component
			m_BlockHandler.m_Init = true;
			checkEnabled();
		}
		else {
			// NB: m_CacheReinitializer (re)set by caller and removed from exec
			m_LastCheckedBlock = m_LatestBlock;
			if (m_RetryExecutor == null || m_CacheReinitializer == null) {
				m_Log.error(lbl + "missing thread pool or reinitializer task");
				System.exit(-5);
			}
			else if (!m_RetryExecutor.isShutdown()) {
				m_RetryExecutor.scheduleWithFixedDelay(m_CacheReinitializer,
													   progressInterval,
													   progressInterval,
													   TimeUnit.MINUTES);
			}
			// else: we (EnftCache) are shutting down and will get replaced
		}
	}

	/**
	 * method for reinitialization of object by downloading chain events.
	 * This is called from hourly HKP, in order to catch cases where one of our
	 * event subscriptions (e.g. for URI data) misses seeing a particular new
	 * datum.  Unfortunately this can happen due to JSON-RPC connections to
	 * block sources being fundamentally unreliable and non-deterministic,
	 * especially when subscribed over a https connection.  Due to drops of
	 * requests, replies, or various errors which can occur within the logic of
	 * the block data host (e.g. 502s or failures accessing the blockchain) we
	 * cannot guarantee that all subscribed data will indeed be received.  In
	 * order to maintain the consistency of our cache with the blockchain,
	 * we therefore re-download the subscribed data periodically.
	 *
	 * This method differs from initialize(false) in that it does not need to
	 * clear existing in-memory data maps.  It also differs from the effect of
	 * calling initialize(true) in that it doesn't start from the current block
	 * less the dwell time, but goes back to a time 1+ hours ago. This will 
	 * result in overlapping data being re-added to maps, but this does no harm
	 * and will result in any unexpected gaps getting filled.
	 *
	 * @return true iff re-download is successfully started
	 */
	public boolean reinitWhileRunning() {
		final String lbl
			= this.getClass().getSimpleName() + ".reinitWhileRunning: ";
		if (!m_SCC.isEnabled()) {
			// caller is supposed to prevent this
			m_Log.error(lbl + "called when SCC not enabled");
			return false;
		}
		m_Log.debug(lbl + "for chain " + m_SCC.getChainId() + " ("
					+ m_SCC.getChainName() + ")");

		/* Re-confirm that the JSON-RPC node we're connected to serves
		 * the correct chain.  This is in case a DNS or routing misconfig
		 * causes us to (re)connect to a node serving the wrong blockchain.
		 * Unlike in initialize(), we do not have to System.exit() here because
		 * a) we always have a m_LatestBlock if the SCC was enabled, and b) the
		 * worst thing that can happen is that we return false and abort the
		 * effect of the reinit, keeping our existing map data until next time.
		 */
		EthChainId eci = null;
		try {
			eci = m_Web3.ethChainId().send();
		}
		catch (Exception ioe) { /* ignore */ }
		if (eci != null) {
			BigInteger servedChain = eci.getChainId();
			if (servedChain != null
				&& servedChain.longValue() != m_SCC.getChainId())
			{
				m_Log.error(lbl + "CRITICAL: RPC node is serving chainId "
							+ servedChain + ", while we expected "
							+ m_SCC.getChainId());
				return false;
			}

			// get the latest block number emitted on the chain at this time
			EthBlockNumber ebn = null;
			try {
				ebn = m_Web3.ethBlockNumber().send();
			}
			catch (Exception ioe) { /* ignore */ }
			if (ebn == null) {
				m_Log.error(lbl + "unable to obtain current block number "
							+ "on " + m_SCC.getChainName());
				// this must be fatal unless we inherited a previous latest
				if (m_LatestBlock == null
					|| m_LatestBlock.equals(BigInteger.ZERO))
				{
					m_Log.error(lbl + "no latest block available "
								+ "for " + m_SCC.getChainName());
					return false;
				}
			}
			else {
				m_LatestBlock = ebn.getBlockNumber();
			}
		}

		// purge all existing tasks & subscriptions so we don't create dups
		m_SCC.setEnabled(false);
		shutdown();
		// cancel old timer task so we can replace it
		m_RetryExecutor.remove(m_CacheReinitializer);
		// create a fresh task for initProgressTimer()
		createReinitializerTask(0);

		// replace the executor object after shutdown() killed any tasks
		m_RetryExecutor = new ScheduledThreadPoolExecutor(32, this);
		m_RetryExecutor.setRemoveOnCancelPolicy(true);
		m_RetryExecutor.setExecuteExistingDelayedTasksAfterShutdownPolicy(
																		false);

		// reset component init states
		m_TransferHandler.m_Init = false;
		m_URIHandler.m_Init = false;
		m_GreyListAddHandler.m_Init = false;
		m_GreyListDeletionHandler.m_Init = false;
		m_BlockHandler.m_Init = false;

		// NB: we can reuse the same wrapper and Credentials set by initialize()

		/* To avoid duplicate event downloads, we'll preload from Logs all that
		 * occur from one hour ago to the current block less one.  The event
		 * subscriptions will then start beginning at the current block to
		 * LATEST, which may encompass more than one block depending upon how
		 * long the download process takes.
		 */
		if (m_SCC.getChainName().startsWith("Ganache")
			|| m_SCC.getChainId() == 1337L)
		{
			/* because this chain doesn't increment blocks absent new
			 * transactions occurring, we do not subtract 1 from latest block
			 * so that we'll download all the available events
			 */
			m_StopInitBlock = m_LatestBlock;
		}
		else {
			m_StopInitBlock = m_LatestBlock.subtract(BigInteger.ONE);
		}

		/* Compute the block that would have occurred approximately one hour
		 * ago (the duration of the housekeeping timer that calls us).  To do
		 * this, we take 4x the confirmations blocks dwell time for the chain.
		 * E.g. on Ethereum Mainnet this is 80 blocks, or about 16 minutes.
		 */
		BigInteger confBlocks
			= new BigInteger(Integer.toString(m_SCC.getDwellTime()));
		BigInteger distance = confBlocks.multiply(BigInteger.valueOf(4L));
		m_StartInitBlock = m_LastProcessedBlock.subtract(distance);

		/* if we have a valid startBlock that's earlier and from a previous
		 * failed download attempt (i.e. that never hit run() or got onError()),
		 * then retain it so we don't risk a gap in our downloaded events
		 */
		if (m_TransferHandler.m_StartBlock != null
			&& m_TransferHandler.m_StartBlock.compareTo(
											m_TransferHandler.m_StopBlock) < 0)
		{
			m_TransferHandler.m_StartBlock
				= m_TransferHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_TransferHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_URIHandler.m_StartBlock != null
			&& m_URIHandler.m_StartBlock.compareTo(
												m_URIHandler.m_StopBlock) < 0)
		{
			m_URIHandler.m_StartBlock
				= m_URIHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_URIHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_GreyListAddHandler.m_StartBlock != null
			&& m_GreyListAddHandler.m_StartBlock.compareTo(
										m_GreyListAddHandler.m_StopBlock) < 0)
		{
			m_GreyListAddHandler.m_StartBlock
				= m_GreyListAddHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_GreyListAddHandler.m_StartBlock = m_StartInitBlock;
		}
		if (m_GreyListDeletionHandler.m_StartBlock != null
			&& m_GreyListDeletionHandler.m_StartBlock.compareTo(
									m_GreyListDeletionHandler.m_StopBlock) < 0)
		{
			m_GreyListDeletionHandler.m_StartBlock
				= m_GreyListDeletionHandler.m_StartBlock.min(m_StartInitBlock);
		}
		else {
			m_GreyListDeletionHandler.m_StartBlock = m_StartInitBlock;
		}
		m_TransferHandler.m_StopBlock = m_StopInitBlock;
		m_URIHandler.m_StopBlock = m_StopInitBlock;
		m_GreyListAddHandler.m_StopBlock = m_StopInitBlock;
		m_GreyListDeletionHandler.m_StopBlock = m_StopInitBlock;

		// again clear any existing block sub to prevent duplicate subscriptions
		if (m_BlockSub != null && !m_BlockSub.isDisposed()) {
			try {
				m_BlockSub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}

		// subscribe to all blocks from this point forward
		loadBlocks();

		// download eNFT mint/burn events, and build maps of ownership
		loadENFTids();

		// download eNFT encrypted metadata (URI events) and add to map
		loadENFTs();

		// download GreyListAdd events, and build list of greylisted Ids
		loadGreyListAdds();

		// download GreyListDeletion events, and build list of removed Ids
		loadGreyListDeletions();

		// start running the block progress check on timer (task exists)
		initProgressTimer();

		/* NB: this return is synchronous.  All init downloads should be done,
		 * unless there was a problem getting EthLogs for one or more events.
		 * In that case there will be a timer running to retry, and we won't
		 * actually enable this blockchain until it succeeds.
		 */
		return true;
	}

	/**
	 * method to load the list of blocks (starts at current block when called)
	 */
	private void loadBlocks() {
		/* Confirm that the JSON-RPC node we're connected to serves
		 * the correct chain.  This is in case a DNS or routing misconfig
		 * causes us to (re)connect to a node serving the wrong blockchain.
		 */
		EthChainId eci = null;
		try {
			eci = m_Web3.ethChainId().send();
		}
		catch (Exception ioe) { /* ignore */ }
		if (eci != null) {
			BigInteger servedChain = eci.getChainId();
			if (servedChain != null
				&& servedChain.longValue() != m_SCC.getChainId())
			{
				m_Log.error(this.getClass().getSimpleName()
							+ ".loadBlocks: RPC node is serving chainId "
							+ servedChain + ", while we expected "
							+ m_SCC.getChainId());
				/* Go back and try again, hoping it gets fixed.  If it doesn't
				 * get corrected, we'll eventually replace the entire Web3
				 * due to no block progress.  In the meantime we'll be un-init.
				 */
				m_BlockHandler.attemptRetry();
				return;
			}
		}

		// create a publisher to fetch all new blocks, w/out transactions
		Flowable<EthBlock> blockPub = m_Web3.blockFlowable(false);
		// use form with callback & error handler (does not terminate)
		m_BlockSub = blockPub.subscribe(m_BlockHandler,
								throwable -> m_BlockHandler.onError(throwable));

		/* NB: in order for this component (the Block subscription) to become
		 * 	   initialized, it is necessary to see evidence of block progress.
		 * 	   This protects us from becoming initialized following the
		 * 	   successful download of all indicated events, in the event that
		 * 	   our RPC source allows this but isn't emitting new blocks.
		 * 	   Therefore, we set m_BlockSub.m_Init only in the
		 * 	   BlockHandler.accept() method, following demonstrated progress.
		 */
	}

	/**
	 * method to load the list of eNFT events (both mints and burns)
	 */
	private void loadENFTids() {
		final String lbl = this.getClass().getSimpleName() + ".loadENFTids: ";
		/* create a filter for all URI events between the starting block
		 * and the ending block
		 */
		DefaultBlockParameterNumber startBlock
			= new DefaultBlockParameterNumber(m_TransferHandler.m_StartBlock);
		DefaultBlockParameterNumber endBlock
			= new DefaultBlockParameterNumber(m_TransferHandler.m_StopBlock);
		EthFilter transSingFilter
			= new EthFilter(startBlock, endBlock, m_ProtocolAddr);
		transSingFilter.addSingleTopic(EventEncoder.encode(
										EnshroudProtocol.TRANSFERSINGLE_EVENT));
		// NB: no indexed or additional topics
		m_Log.debug(lbl + "downloading all TransferSingle events between "
					+ startBlock.getBlockNumber() + " and "
					+ endBlock.getBlockNumber() + " on "
					+ m_SCC.getChainName());
		boolean gotEvents = false;
		EthLog ethLog = null;
		try {
			// synchronous fetch
			ethLog = m_Web3.ethGetLogs(transSingFilter).send();
			if (ethLog == null) {
				m_Log.error(lbl + m_SCC.getChainName()
							+ " Web3 URI does not appear to support events");
			}
			else if (ethLog.getLogs() == null) {
				// not supposed to happen, but sometimes does on timeouts
				m_Log.error(lbl + "null log list returned ("
							+ m_SCC.getChainName() + ")");
				// if not doing cold start, adjust the stop block to current
				if (!m_DoingInit) {
					m_TransferHandler.m_StopBlock = m_LatestBlock;
				}
			}
			else {
				gotEvents = true;
			}
		}
		catch (IOException | ClientConnectionException
				| WebsocketNotConnectedException eee)
		{
			m_Log.error(lbl
						+ "unable to get EthLogs for TransferSingle events: "
						+ eee.toString());
			// if not doing full cold start, adjust the stop block to current
			if (!m_DoingInit) {
				m_TransferHandler.m_StopBlock = m_LatestBlock;
			}
		}
		if (gotEvents) {
			List<EthLog.LogResult> transEvents = ethLog.getLogs();
			for (EthLog.LogResult eLog : transEvents) {
				EthLog.LogObject logObj = (EthLog.LogObject) eLog;
				org.web3j.protocol.core.methods.response.Log log = logObj.get();
				EventValues eventValues = Contract.staticExtractEventParameters(
									EnshroudProtocol.TRANSFERSINGLE_EVENT, log);
				EnshroudProtocol.TransferSingleEventResponse tseResponse
					= new EnshroudProtocol.TransferSingleEventResponse();
				tseResponse.log = log;
				tseResponse.operator
					= (String) eventValues.getIndexedValues().get(0).getValue();
				tseResponse.from
					= (String) eventValues.getIndexedValues().get(1).getValue();
				tseResponse.to
					= (String) eventValues.getIndexedValues().get(2).getValue();
				tseResponse.id = (BigInteger) eventValues.getNonIndexedValues()
									.get(0).getValue();
				tseResponse.amount
					= (BigInteger) eventValues.getNonIndexedValues().get(1)
																	.getValue();
				
				// pass event to TransferSingleHandler object
				try {
					m_TransferHandler.accept(tseResponse);
				}
				catch (Exception e) { /* error logged by handler */ }
			}

			// move to subscription phase
			try {
				m_TransferHandler.run();

				// check whether m_SCC can now be enabled
				checkEnabled();
			}
			catch (Exception ee) { /* not actually thrown */ }
		}
		else {
			m_TransferHandler.m_Init = false;
			// set timer and try again after interval
			m_TransferHandler.attemptRetry();
		}
	}

	/**
	 * method to load the encrypted eNFT metadata for every non-burned eNFT
	 */
	private void loadENFTs() {
		final String lbl = this.getClass().getSimpleName() + ".loadENFTs: ";
		/* create a filter for all URI events between the starting block
		 * and the ending block
		 */
		DefaultBlockParameterNumber startBlock
			= new DefaultBlockParameterNumber(m_URIHandler.m_StartBlock);
		DefaultBlockParameterNumber endBlock
			= new DefaultBlockParameterNumber(m_URIHandler.m_StopBlock);
		EthFilter uriFilter
			= new EthFilter(startBlock, endBlock, m_ProtocolAddr);
		uriFilter.addSingleTopic(EventEncoder.encode(
												EnshroudProtocol.URI_EVENT));
		// NB: no indexed or additional topics
		m_Log.debug(lbl + "downloading all URI events between "
					+ startBlock.getBlockNumber() + " and "
					+ endBlock.getBlockNumber() + " on "
					+ m_SCC.getChainName());
		boolean gotEvents = false;
		EthLog ethLog = null;
		try {
			// synchronous fetch
			ethLog = m_Web3.ethGetLogs(uriFilter).send();
			if (ethLog == null) {
				m_Log.error(lbl + m_SCC.getChainName()
							+ " Web3 URI does not appear to support events");
			}
			else if (ethLog.getLogs() == null) {
				// not supposed to happen, but sometimes does on timeouts
				m_Log.error(lbl + "null log list returned ("
							+ m_SCC.getChainName() + ")");
				// if not doing cold start, adjust the stop block to current
				if (!m_DoingInit) {
					m_URIHandler.m_StopBlock = m_LatestBlock;
				}
			}
			else {
				gotEvents = true;
			}
		}
		catch (IOException | ClientConnectionException
				| WebsocketNotConnectedException eee)
		{
			m_Log.error(lbl + "unable to get EthLogs for URI events: "
						+ eee.toString());
			// if not doing full cold start, adjust the stop block to current
			if (!m_DoingInit) {
				m_URIHandler.m_StopBlock = m_LatestBlock;
			}
		}
		if (gotEvents) {
			List<EthLog.LogResult> uriEvents = ethLog.getLogs();
			for (EthLog.LogResult eLog : uriEvents) {
				EthLog.LogObject logObj = (EthLog.LogObject) eLog;
				org.web3j.protocol.core.methods.response.Log log = logObj.get();
				EventValues eventValues = Contract.staticExtractEventParameters(
											EnshroudProtocol.URI_EVENT, log);
				EnshroudProtocol.URIEventResponse uriResponse
					= new EnshroudProtocol.URIEventResponse();
				uriResponse.log = log;
				uriResponse.id = (BigInteger) eventValues.getIndexedValues()
									.get(0).getValue();
				uriResponse.value = (String) eventValues.getNonIndexedValues()
									.get(0).getValue();

				// pass event to URIHandler object
				try {
					m_URIHandler.accept(uriResponse);
				}
				catch (Exception e) { /* error logged by handler */ }
			}

			// move to subscription phase
			try {
				m_URIHandler.run();

				// check whether m_SCC can now be enabled
				checkEnabled();
			}
			catch (Exception ee) { /* not actually thrown */ }
		}
		else {
			m_URIHandler.m_Init = false;
			// set timer and try again after interval
			m_URIHandler.attemptRetry();
		}
	}

	/**
	 * method to load the greylisted Ids from the history of GreyListAdd events
	 */
	@SuppressWarnings("unchecked")
	private void loadGreyListAdds() {
		final String lbl
			= this.getClass().getSimpleName() + ".loadGreyListAdds: ";
		/* create a filter for all TransferSingle events between the block
		 * where the EnshroudProtocol contract was deployed, and our starting
		 * point penultimate block
		 */
		DefaultBlockParameterNumber startBlock
			= new DefaultBlockParameterNumber(
											m_GreyListAddHandler.m_StartBlock);
		DefaultBlockParameterNumber endBlock
			= new DefaultBlockParameterNumber(m_GreyListAddHandler.m_StopBlock);
		EthFilter glAddFilter
			= new EthFilter(startBlock, endBlock, m_ProtocolAddr);
		glAddFilter.addSingleTopic(EventEncoder.encode(
									EnshroudProtocol.GREYLISTADD_EVENT));
		// NB: no indexed or additional topics
		m_Log.debug(lbl + "downloading GreyListAdd events between "
					+ startBlock.getBlockNumber() + " and "
					+ endBlock.getBlockNumber() + " on "
					+ m_SCC.getChainName());
		boolean gotEvents = false;
		EthLog ethLog = null;
		try {
			// synchronous fetch
			ethLog = m_Web3.ethGetLogs(glAddFilter).send();
			if (ethLog == null) {
				m_Log.error(lbl + m_SCC.getChainName()
							+ " Web3 URI does not appear to support events");
			}
			else if (ethLog.getLogs() == null) {
				// not supposed to happen, but sometimes does on timeouts
				m_Log.error(lbl + "null log list returned ("
							+ m_SCC.getChainName() + ")");
				// if not doing cold start, adjust the stop block to current
				if (!m_DoingInit) {
					m_GreyListAddHandler.m_StopBlock = m_LatestBlock;
				}
			}
			else {
				gotEvents = true;
			}
		}
		catch (IOException | ClientConnectionException
				| WebsocketNotConnectedException eee)
		{
			m_Log.error(lbl + "unable to get EthLogs for GreyListAdd events: "
						+ eee.toString());
			// if not doing full cold start, adjust the stop block to current
			if (!m_DoingInit) {
				m_GreyListAddHandler.m_StopBlock = m_LatestBlock;
			}
		}
		if (gotEvents) {
			List<EthLog.LogResult> glEvents = ethLog.getLogs();
			for (EthLog.LogResult eLog : glEvents) {
				EthLog.LogObject logObj = (EthLog.LogObject) eLog;
				org.web3j.protocol.core.methods.response.Log log = logObj.get();
				EventValues eventValues = Contract.staticExtractEventParameters(
									EnshroudProtocol.GREYLISTADD_EVENT, log);
				EnshroudProtocol.GreyListAddEventResponse glResponse
					= new EnshroudProtocol.GreyListAddEventResponse();
				glResponse.log = log;
				glResponse.audId = (String) eventValues.getNonIndexedValues()
									.get(0).getValue();
				glResponse.ids = (List<BigInteger>) ((Array) eventValues.getNonIndexedValues().get(1)).getNativeValueCopy();

				// pass event to GreyListAddHandler object
				try {
					m_GreyListAddHandler.accept(glResponse);
				}
				catch (Exception e) { /* error logged by handler */ }
			}

			// move to subscription phase
			try {
				m_GreyListAddHandler.run();

				// check whether m_SCC can now be enabled
				checkEnabled();
			}
			catch (Exception ee) { /* not actually thrown */ }
		}
		else {
			m_GreyListAddHandler.m_Init = false;
			// set timer and try again after interval
			m_GreyListAddHandler.attemptRetry();
		}
	}

	/**
	 * method to winnow the greylisted Ids from the GreyListDeletion events
	 */
	private void loadGreyListDeletions() {
		final String lbl
			= this.getClass().getSimpleName() + ".loadGreyListDeletions: ";
		/* construct a filter for all GreyListDeletion events between the block
		 * where the EnshroudProtocol contract was deployed, and our starting
		 * point penultimate block
		 */
		DefaultBlockParameterNumber startBlock
			= new DefaultBlockParameterNumber(
										m_GreyListDeletionHandler.m_StartBlock);
		DefaultBlockParameterNumber endBlock
			= new DefaultBlockParameterNumber(
										m_GreyListDeletionHandler.m_StopBlock);
		EthFilter glDelFilter
			= new EthFilter(startBlock, endBlock, m_ProtocolAddr);
		glDelFilter.addSingleTopic(EventEncoder.encode(
									EnshroudProtocol.GREYLISTDELETION_EVENT));
		// NB: no indexed or additional topics
		m_Log.debug(lbl + "downloading GreyListDeletion events between "
					+ startBlock.getBlockNumber() + " and "
					+ endBlock.getBlockNumber() + " on "
					+ m_SCC.getChainName());
		boolean gotEvents = false;
		EthLog ethLog = null;
		try {
			// synchronous fetch
			ethLog = m_Web3.ethGetLogs(glDelFilter).send();
			if (ethLog == null) {
				m_Log.error(lbl + m_SCC.getChainName()
							+ " Web3 URI does not appear to support events");
			}
			else if (ethLog.getLogs() == null) {
				// not supposed to happen, but sometimes does on timeouts
				m_Log.error(lbl + "null log list returned ("
							+ m_SCC.getChainName() + ")");
				// if not doing cold start, adjust the stop block to current
				if (!m_DoingInit) {
					m_GreyListDeletionHandler.m_StopBlock = m_LatestBlock;
				}
			}
			else {
				gotEvents = true;
			}
		}
		catch (IOException | ClientConnectionException
				| WebsocketNotConnectedException eee)
		{
			m_Log.error(lbl + "unable to get EthLogs for GreyListDeletion "
						+ "events: " + eee.toString());
			// if not doing full cold start, adjust the stop block to current
			if (!m_DoingInit) {
				m_GreyListDeletionHandler.m_StopBlock = m_LatestBlock;
			}
		}
		if (gotEvents) {
			List<EthLog.LogResult> glDelEvents = ethLog.getLogs();
			for (EthLog.LogResult eLog : glDelEvents) {
				EthLog.LogObject logObj = (EthLog.LogObject) eLog;
				org.web3j.protocol.core.methods.response.Log log = logObj.get();
				EventValues eventValues = Contract.staticExtractEventParameters(
								EnshroudProtocol.GREYLISTDELETION_EVENT, log);
				EnshroudProtocol.GreyListDeletionEventResponse glDelResponse
					= new EnshroudProtocol.GreyListDeletionEventResponse();
				glDelResponse.log = log;
				glDelResponse.id = (BigInteger) eventValues.getNonIndexedValues().get(0).getValue();

				// pass event to m_GreyListDeletionHandler object
				try {
					m_GreyListDeletionHandler.accept(glDelResponse);
				}
				catch (Exception e) { /* error logged by handler */ }
			}

			// move to subscription phase
			try {
				m_GreyListDeletionHandler.run();

				// check whether m_SCC can now be enabled
				checkEnabled();
			}
			catch (Exception ee) { /* not actually thrown */ }
		}
		else {
			m_GreyListDeletionHandler.m_Init = false;
			// set timer and try again after interval
			m_GreyListDeletionHandler.attemptRetry();
		}
	}

	/**
	 * check if everything appears to be initialized now, and if so enable SCC
	 */
	private void checkEnabled() {
		if ((m_TransferHandler != null && m_TransferHandler.m_Init)
			&& (m_URIHandler != null && m_URIHandler.m_Init)
			&& (m_GreyListAddHandler != null && m_GreyListAddHandler.m_Init)
			&& (m_GreyListDeletionHandler != null
				&& m_GreyListDeletionHandler.m_Init)
			&& (m_BlockHandler != null && m_BlockHandler.m_Init))
		{
			if (!m_SCC.isEnabled()) {
				m_Log.debug("EnftCache.checkEnabled: all appears init, "
							+ "enabling " + m_SCC.getChainName());
				m_SCC.setEnabled(true);
			}
		}
	}

	/**
	 * reset the Web3j ABI connection (called during websocket reconnects)
	 * @param abiSession the connection to use to talk to this chain
	 */
	public void resetABI(Web3j abiSession) {
		if (abiSession != null) {
			m_Web3 = abiSession;
			// call to initialize(true) should now follow
		}
	}

	/**
	 * shut down the cache and clean up subscriptions
	 */
	public void shutdown() {
		final String lbl = this.getClass().getSimpleName() + ".shutdown: ";
		m_SCC.setEnabled(false);

		// clear all tasks in the thread pool
		if (m_RetryExecutor != null) {
			// terminate any executing or scheduled tasks
			m_RetryExecutor.shutdownNow();
		}

		/* clear any subscriptions that appear likely to exist
		 * NB: attempting to call .dispose() on subs may throw undeliverable
		 * exceptions, but it's better to do it anyway for request hygiene
		 */
		if (m_BlockSub != null && !m_BlockSub.isDisposed()) {
			try {
				m_BlockSub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}
		if (m_TransferSingleSub != null && !m_TransferSingleSub.isDisposed()) {
			try {
				m_TransferSingleSub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}
		if (m_URISub != null && !m_URISub.isDisposed()) {
			try {
				m_URISub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}
		if (m_GreyListAddSub != null && !m_GreyListAddSub.isDisposed()) {
			try {
				m_GreyListAddSub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}
		if (m_GreyListDeletionSub != null
			&& !m_GreyListDeletionSub.isDisposed())
		{
			try {
				m_GreyListDeletionSub.dispose();
			}
			catch (Exception e) { /* ignore */ }
		}

		m_BlockSub = null;
		m_TransferSingleSub = null;
		m_URISub = null;
		m_GreyListAddSub = null;
		m_GreyListDeletionSub = null;
	}

	// implement RejectedExecutionHandler interface
	/**
	 * method to handle issues with rejected executions from m_RetryExecutor
	 * @param task the Runnable that can't be executed
	 * @param executor the ThreadPoolExecutor (m_RetryExecutor)
	 */
	public void rejectedExecution(Runnable task, ThreadPoolExecutor executor) {
		final String lbl
			= this.getClass().getSimpleName() + ".rejectedExecution: ";

		/* This likely indicates that a cancelled thread attempted to schedule
		 * a new task after we called m_RetryExecutor.shutdownNow(), i.e.
		 * before it got killed but after the shutdown was initiated.
		 * If this does *not* appear to be the case, then something is hosed
		 * in our ScheduledThreadPoolExecutor, and we should reinit.
		 */
		m_Log.error(lbl + "abandoning cache instance and doing reinit "
					+ "due to RejectedExecutionException"); 
		// invoke EnftCacheResetter (rather than ShutdownHandler)
		if (m_CacheResetter != null) {
			// NB: this method guards against duplicate/re-entrant calls
			m_CacheResetter.requestEnftCacheRestart(m_SCC.getChainId(),
						"RejectedExecutionException on EnftCache handler");
		}
		else {
			m_Log.error(lbl + "RejectedExecutionException on Block "
						+ "sub, no resetter found, exiting");
			System.exit(-6);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		try {
			// clear the maps if they exist
			if (m_EnftsMinted != null) {
				m_EnftsMinted.clear();
			}
			if (m_EnftsBurned !=  null) {
				m_EnftsBurned.clear();
			}
			if (m_EnftMetadata != null) {
				m_EnftMetadata.clear();
			}
			if (m_EnftsGreyListed != null) {
				m_EnftsGreyListed.clear();
			}
			if (m_RetryExecutor != null) {
				m_RetryExecutor.purge();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
