/*
 * last modified---
 * 	06-07-23 add getEIP712SignerAddress()
 *	06-01-23 do digest sigs in R,S,V order; remove sha3String()
 * 	05-01-23 rewrite decWithECIES() with actual code; remove encWithECIES()
 * 	04-04-23 replace buildDetailsHash() with Solidity-compatible version
 * 	03-30-23 add buildSolDetailsHash()
 * 	03-21-23 switch buildDetailsHash() to use sha3String()
 * 	03-07-23 switch buildDetailsHash() to use ethPrefixHash() not sha3(); add
 * 			 0x prefix to amount in detailsHash for consistency
 * 	02-22-23 add ethPrefixHash(), signHash(), signData(), getDataSignerAddress()
 * 			 and getHashSignerAddress()
 * 	02-21-22 add sha3String()
 * 	12-01-22 add rand elt to buildDetailsHash(); convert to BC provider;
 *			 added encWithECIES(), decWithECIES(),
 * 			 getECPubkeyFromBase64Str(), getECPrivkeyFromBase64Str()
 * 	10-13-22 add buildDetailsHash()
 * 	06-15-22 in getStrFromBase64PubkeyEnc(), remove trailing padding chars
 * 	05-05-22 add temp methods toHexStringZeroPadded() and sha3()
 * 	04-14-22 add verifySignedStr() and signStr()
 * 	03-29-22 new
 *
 * purpose---
 * 	provide utility methods for dealing with encryption tasks
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.web3j.utils.Numeric;
import org.web3j.crypto.Hash;
import org.web3j.crypto.Sign;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.Keys;
import org.web3j.crypto.StructuredDataEncoder;
import org.web3j.abi.datatypes.Type;
import org.web3j.abi.datatypes.Address;
import org.web3j.abi.datatypes.generated.Uint256;
import org.web3j.abi.datatypes.generated.Bytes16;
import org.web3j.abi.DefaultFunctionEncoder;

import org.bouncycastle.jce.ECNamedCurveTable;
import org.bouncycastle.jce.interfaces.ECPrivateKey;
import org.bouncycastle.jce.interfaces.ECPublicKey;
import org.bouncycastle.jce.spec.ECParameterSpec;
import org.bouncycastle.jce.spec.ECPublicKeySpec;
import org.bouncycastle.jce.spec.ECNamedCurveParameterSpec;
import org.bouncycastle.jce.provider.BouncyCastleProvider;
import org.bouncycastle.jcajce.provider.asymmetric.ec.BCECPublicKey;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECMultiplier;

import java.util.Base64;
import java.util.Arrays;
import java.util.ArrayList;
import java.nio.charset.StandardCharsets;
import java.nio.ByteBuffer;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import javax.crypto.SecretKey;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.BadPaddingException;
import java.security.SecureRandom;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.InvalidKeyException;
import java.security.KeyFactory;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.security.Signature;
import java.security.SignatureException;
import java.security.MessageDigest;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.X509EncodedKeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.InvalidAlgorithmParameterException;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;


/**
 * Holds static methods to manipulate encryption objects.  This includes hashes
 * plus RSA-4096, AES-256, ECIES, and ECDSA sig/verify operations.
 */
public final class EncodingUtils {
	// constants
	/**
	 * block size for encrypting RSA blocks (must be smaller than key modulus,
	 * allowing for padding):
	 * mLen = kLenBits / 8 - 2 * hLenBits / 8 - 2
	 * For key len = 4096 bits, hashLen bits = 256, gives: 512 - 64 - 2 = 446
	 */
	private static final int	M_EncBlockSz = 446;

	/**
	 * block size for decrypting RSA blocks with 4096-bit keys (4096 / 8,
	 * includes padding of 66 bytes)
	 */
	private static final int	M_DecBlockSz = 512;

	/**
	 * tag length for GCM parameter (default)
	 */
	private static final int	M_GCMTagLen = 128;

	// BEGIN methods
	/**
	 * method to obtain a public RSA key from a base64-encoded string
	 * @param b64Str the encoded string
	 * @return the public key, or null if the string does not parse properly
	 */
	public static PublicKey getPubkeyFromBase64Str(String b64Str) {
		if (b64Str == null || b64Str.isEmpty()) {
			return null;
		}

		Base64.Decoder decoder = Base64.getDecoder();
		byte[] pubKeyBytes = decoder.decode(b64Str);
		X509EncodedKeySpec eks = new X509EncodedKeySpec(pubKeyBytes);
		PublicKey pubKey = null;
		KeyFactory keyFactory = null;
		try {
			keyFactory = KeyFactory.getInstance("RSA");
			pubKey = keyFactory.generatePublic(eks);
		}
		catch (NoSuchAlgorithmException nsae) {
			return null;
		}
		catch (InvalidKeySpecException ikse) {
			return null;
		}
		return pubKey;
	}

	/**
	 * method to obtain a public ECDSA key from a base64-encoded string
	 * @param b64Str the encoded string
	 * @return the public key, or null if the string does not parse properly
	 */
	public static PublicKey getECPubkeyFromBase64Str(String b64Str) {
		if (b64Str == null || b64Str.isEmpty()) {
			return null;
		}

		Base64.Decoder decoder = Base64.getDecoder();
		byte[] pubKeyBytes = decoder.decode(b64Str);
		X509EncodedKeySpec eks = new X509EncodedKeySpec(pubKeyBytes);
		PublicKey pubKey = null;
		KeyFactory keyFactory = null;
		try {
			keyFactory = KeyFactory.getInstance("EC", "BC");
			pubKey = keyFactory.generatePublic(eks);
		}
		catch (NoSuchAlgorithmException nsae) {
			return null;
		}
		catch (InvalidKeySpecException ikse) {
			return null;
		}
		catch (NoSuchProviderException nspe) {
			return null;
		}
		return pubKey;
	}

	/**
	 * method to obtain a private RSA key from a base64-encoded string
	 * @param b64Str the encoded string
	 * @return the private key, or null if the string does not parse properly
	 */
	public static PrivateKey getPrivkeyFromBase64Str(String b64Str) {
		if (b64Str == null || b64Str.isEmpty()) {
			return null;
		}

		Base64.Decoder decoder = Base64.getDecoder();
		byte[] privKeyBytes = decoder.decode(b64Str);
		PKCS8EncodedKeySpec eks = new PKCS8EncodedKeySpec(privKeyBytes);
		Arrays.fill(privKeyBytes, (byte)0);
		PrivateKey privKey = null;
		KeyFactory keyFactory = null;
		try {
			keyFactory = KeyFactory.getInstance("RSA");
			privKey = keyFactory.generatePrivate(eks);
		}
		catch (NoSuchAlgorithmException nsae) {
			return null;
		}
		catch (InvalidKeySpecException ikse) {
			return null;
		}
		return privKey;
	}

	/**
	 * method to obtain a private ECDSA key from a base64-encoded string
	 * @param b64Str the encoded string
	 * @return the private key, or null if the string does not parse properly
	 */
	public static PrivateKey getECPrivkeyFromBase64Str(String b64Str) {
		if (b64Str == null || b64Str.isEmpty()) {
			return null;
		}

		Base64.Decoder decoder = Base64.getDecoder();
		byte[] privKeyBytes = decoder.decode(b64Str);
		PKCS8EncodedKeySpec eks = new PKCS8EncodedKeySpec(privKeyBytes);
		Arrays.fill(privKeyBytes, (byte)0);
		PrivateKey privKey = null;
		KeyFactory keyFactory = null;
		try {
			keyFactory = KeyFactory.getInstance("EC", "BC");
			privKey = keyFactory.generatePrivate(eks);
		}
		catch (NoSuchAlgorithmException nsae) {
			return null;
		}
		catch (InvalidKeySpecException ikse) {
			return null;
		}
		catch (NoSuchProviderException nspe) {
			return null;
		}
		return privKey;
	}

	/**
	 * method to encrypt and base64-encode a string to a given public RSA key,
	 * used for encrypting payloads sent over WebSocket connections
	 * @param data the string to be encrypted
	 * @param key the public RSA key to use (assumed to be 4096 bits)
	 * @return the encrypted and encoded string, or null if encrypting failed
	 */
	public static String base64PubkeyEncStr(String data, PublicKey key) {
		if (data == null || data.isEmpty() || key == null) {
			return null;
		}

		Cipher pubkeyCipher = null;
		boolean gotErr = false;
		// init the cipher object with the public key
		KeyFactory keyFactory = null;
		try {
			keyFactory = KeyFactory.getInstance("RSA");
			pubkeyCipher
				= Cipher.getInstance("RSA/ECB/OAEPWithSHA-256AndMGF1Padding");
			pubkeyCipher.init(Cipher.ENCRYPT_MODE, key);
		}
		catch (NoSuchAlgorithmException nsae) {
			gotErr = true;
		}
		catch (NoSuchPaddingException nspe) {
			gotErr = true;
		}
		catch (InvalidKeyException ike) {
			gotErr = true;
		}
		if (gotErr) return null;

		/* Perform the encrypt.  To allow for the possibility that our data is
		 * longer than the max block size, we'll chain the blocks together as
		 * needed.
		 */
		ByteArrayInputStream encInput
			= new ByteArrayInputStream(data.getBytes(StandardCharsets.UTF_8));
		ByteArrayOutputStream encOutput = new ByteArrayOutputStream(2048);
		byte[] ibuff = new byte[M_EncBlockSz];
		int cnt = encInput.read(ibuff, 0, M_EncBlockSz);
		while (cnt != -1) {
			byte[] obuff = null;
			gotErr = false;

			// encrypt this block
			try {
				obuff = pubkeyCipher.doFinal(ibuff);
				Arrays.fill(ibuff, (byte)0);
			}
			catch (IllegalBlockSizeException ibse) {
				gotErr = true;
			}
			catch (BadPaddingException bpe) {
				gotErr = true;
			}
			if (gotErr) return null;

			encOutput.write(obuff, 0, obuff.length);
			ibuff = new byte[M_EncBlockSz];
			cnt = encInput.read(ibuff, 0, M_EncBlockSz);
		}

		byte[] encBlob = encOutput.toByteArray();
		Base64.Encoder encoder = Base64.getEncoder().withoutPadding();
		byte[] b64Blob = encoder.encode(encBlob);
		String encStr = new String(b64Blob, StandardCharsets.UTF_8);
		return encStr;
	}

	/**
	 * method to use a private key to decrypt a base64-encoded blob encrypted
	 * using the corresponding public key, used for decrypting messages received
	 * from other nodes
	 * @param encData the encoded and encrypted data
	 * @param privKey the private key to use for decrypting (assumed 4096 bits)
	 * @return the clear text, or null if decryption failed
	 */
	public static String getStrFromBase64PubkeyEnc(String encData,
												   PrivateKey privKey)
	{
		if (encData == null || encData.isEmpty() || privKey == null) {
			return null;
		}

		// init the cipher
		Cipher privkeyCipher = null;
		boolean gotErr = false;
		try {
			privkeyCipher
				= Cipher.getInstance("RSA/ECB/OAEPWithSHA-256AndMGF1Padding");
			privkeyCipher.init(Cipher.DECRYPT_MODE, privKey);
		}
		catch (NoSuchAlgorithmException nsae) {
			gotErr = true;
		}
		catch (NoSuchPaddingException nspe) {
			gotErr = true;
		}
		catch (InvalidKeyException ike) {
			gotErr = true;
		}
		if (gotErr) return null;

		// decrypt the input
		Base64.Decoder decoder = Base64.getDecoder();
		byte[] encDataBytes
			= decoder.decode(encData.getBytes(StandardCharsets.UTF_8));
		ByteArrayInputStream decInput = new ByteArrayInputStream(encDataBytes);
		ByteArrayOutputStream decOutput
			= new ByteArrayOutputStream(encDataBytes.length);
		byte[] ibuff = new byte[M_DecBlockSz];
		int cnt = decInput.read(ibuff, 0, M_DecBlockSz);
		while (cnt != -1) {
			gotErr = false;

			// decrypt this block
			byte[] obuff = null;
			try {
				obuff = privkeyCipher.doFinal(ibuff);
			}
			catch (IllegalBlockSizeException ibse) {
				gotErr = true;
			}
			catch (BadPaddingException bpe) {
				gotErr = true;
			}
			if (gotErr) return null;

			decOutput.write(obuff, 0, obuff.length);
			Arrays.fill(obuff, (byte)0);
			ibuff = new byte[M_DecBlockSz];
			cnt = decInput.read(ibuff, 0, M_DecBlockSz);
		}

		byte[] decBytes = decOutput.toByteArray();
		decOutput.reset();
		String rawDecStr = new String(decBytes, StandardCharsets.UTF_8);
		Arrays.fill(decBytes, (byte)0);

		// chop off any trailing padding characters
		int trailIdx = rawDecStr.indexOf(Character.MIN_VALUE);
		if (trailIdx != -1) {
			return rawDecStr.substring(0, trailIdx);
		}
		return rawDecStr;
	}

	/**
	 * method to generate a fresh AES-256 key
	 * @param rng the secure random number generator to use for key
	 * @return the key, or null on failure
	 */
	public static SecretKey genAES256Key(SecureRandom rng) {
		if (rng == null) {
			return null;
		}

		// generate a random key for AES-256 (bits determined by array size)
		byte[] key = new byte[32];
		rng.nextBytes(key);
		SecretKey secretKey = new SecretKeySpec(key, "AES");
		Arrays.fill(key, (byte)0);
		return secretKey;
	}

	/**
	 * method to encrypt and base64-encode a string with an AES-256 key
	 * @param aesKey the key to be used
	 * @param plainText the data to be encrypted
	 * @param rng the secure random number generator to use for the IV
	 * @return the cipher text in base64 (with the initialization vector as
	 * the first 12 bytes), or null on failures
	 */
	public static String encWithAES(SecretKey aesKey,
									String plainText,
									SecureRandom rng)
	{
		if (aesKey == null || plainText == null || plainText.isEmpty()
			|| rng == null)
		{
			return null;
		}

		byte[] iv = new byte[12];
		rng.nextBytes(iv);
		// use standard 128 bit tag length
		GCMParameterSpec parameterSpec = new GCMParameterSpec(M_GCMTagLen, iv);
		Cipher cipher = null;
		Base64.Encoder b64e = Base64.getEncoder();

		byte[] cipherText = null;
		String b64Cipher = "";
		boolean gotErr = false;
		try {
			cipher = Cipher.getInstance("AES/GCM/NoPadding");
			cipher.init(Cipher.ENCRYPT_MODE, aesKey, parameterSpec);
			byte[] plainBytes = plainText.getBytes(StandardCharsets.UTF_8);
			cipherText = cipher.doFinal(plainBytes);
			Arrays.fill(plainBytes, (byte)0);
			ByteBuffer byteBuffer
				= ByteBuffer.allocate(iv.length + cipherText.length);
			byteBuffer.put(iv);
			Arrays.fill(iv, (byte)0);
			// NB: GCM tag is appended as the final 16 bytes of the cipherText
			byteBuffer.put(cipherText);
			byte[] cipherMessage = byteBuffer.array();
			byte[] b64Msg = b64e.encode(cipherMessage);
			b64Cipher = new String(b64Msg, StandardCharsets.UTF_8);
		}
		catch (NoSuchAlgorithmException nsae) {
			gotErr = true;
		}
		catch (InvalidKeyException ike) {
			gotErr = true;
		}
		catch (IllegalBlockSizeException ibse) {
			gotErr = true;
		}
		catch (InvalidAlgorithmParameterException iape) {
			gotErr = true;
		}
		catch (NoSuchPaddingException nspe) {
			gotErr = true;
		}
		catch (BadPaddingException bpe) {
			gotErr = true;
		}
		if (gotErr) return null;

		return b64Cipher;
	}

	/**
	 * method to decrypt a string using an AES-256 key
	 * @param aesKey the key to use
	 * @param cipherText the encrypted text, base64-encoded
	 * @return the decrypted plain text, or null on errors
	 */
	public static String decWithAES(SecretKey aesKey, String cipherText) {
		if (aesKey == null || cipherText == null || cipherText.isEmpty()) {
			return null;
		}

		Cipher cipher = null;
		Base64.Decoder b64d = Base64.getDecoder();
		byte[] msgDecoded = b64d.decode(cipherText);
		String plainText = null;
		boolean gotErr = false;
		try {
			cipher = Cipher.getInstance("AES/GCM/NoPadding");
			// initialization vector is found in the first 12 cipherText bytes
			GCMParameterSpec gcmIv
				= new GCMParameterSpec(M_GCMTagLen, msgDecoded, 0, 12);
			cipher.init(Cipher.DECRYPT_MODE, aesKey, gcmIv);
			// everything from 12 bytes on is ciphertext, GCM tag at end
			byte[] decText
				= cipher.doFinal(msgDecoded, 12, msgDecoded.length - 12);
			plainText = new String(decText, StandardCharsets.UTF_8);
			Arrays.fill(decText, (byte)0);
		}
		catch (NoSuchAlgorithmException nsae) {
			gotErr = true;
		}
		catch (InvalidKeyException ike) {
			gotErr = true;
		}
		catch (IllegalBlockSizeException ibse) {
			gotErr = true;
		}
		catch (InvalidAlgorithmParameterException iape) {
			gotErr = true;
		}
		catch (NoSuchPaddingException nspe) {
			gotErr = true;
		}
		catch (BadPaddingException bpe) {
			gotErr = true;
		}
		if (gotErr) return null;

		return plainText;
	}

	/**
	 * method to create an RSA signature on a string
	 * @param privKey the private key to sign with
	 * @param sigData the data to be signed
	 * @return the signature (base64-encoded), or null on failures
	 */
	public static String signStr(PrivateKey privKey, String sigData) {
		if (privKey == null || sigData == null || sigData.isEmpty()) {
			return null;
		}

		boolean ok = true;
		Signature sig = null;
		byte[] signature = null;
		try {
			sig = Signature.getInstance("SHA256withRSA");
			sig.initSign(privKey);
			sig.update(sigData.getBytes(StandardCharsets.UTF_8));
			signature = sig.sign();
		}
		catch (NoSuchAlgorithmException nsae) {
			ok = false;
		}
		catch (InvalidKeyException ike) {
			ok = false;
		}
		catch (SignatureException se) {
			ok = false;
		}
		if (signature == null) {
			ok = false;
		}
		if (!ok) {
			return null;
		}
		Base64.Encoder b64e = Base64.getEncoder();
		return b64e.encodeToString(signature);
	}

	/**
	 * method to verify an RSA signature on a string 
	 * @param pubKey the public key used to verify the signature
	 * @param signed the data signed (cleartext)
	 * @param signature the signature to be verified (base64-encoded)
	 * @return true if signature is valid
	 */
	public static boolean verifySignedStr(PublicKey pubKey,
										  String signed,
										  String signature)
	{
		if (pubKey == null || signed == null || signature == null
			|| signature.isEmpty() || signed.isEmpty())
		{
			return false;
		}

		boolean ok = false;
		Base64.Decoder b64d = Base64.getDecoder();
		Signature sig = null;
		try {
			sig = Signature.getInstance("SHA256withRSA");
			sig.initVerify(pubKey);
			sig.update(signed.getBytes(StandardCharsets.UTF_8));
			ok = sig.verify(b64d.decode(signature));
		}
		catch (NoSuchAlgorithmException nsae) {
			ok = false;
		}
		catch (InvalidKeyException ike) {
			ok = false;
		}
		catch (SignatureException se) {
			ok = false;
		}
		return ok;
	}

	/**
	 * method to decrypt a string using ECIES and a private key
	 * @param privKey the private ECDSA key to be used
	 * @param cipherText the data to be decrypted, in Base64Url format
	 * @return the decrypted text in clear, or null on failures
	 */
	public static String decWithECIES(PrivateKey privKey, String cipherText) {
		if (cipherText == null || cipherText.isEmpty()
			|| !(privKey instanceof ECPrivateKey))
		{
			return null;
		}

		boolean gotErr = false;
		String plainText = null;
		try {
			Base64.Decoder b64d = Base64.getUrlDecoder();
			byte[] msgDecoded = b64d.decode(cipherText);
			/* The first 65 bytes of the cipherText is the uncompressed
			 * ephemeral EC public key.  The actual ciphertext follows.
			 */
			byte[] pubKey = new byte[65];
			System.arraycopy(msgDecoded, 0, pubKey, 0, pubKey.length);

			// build the ephemeral pubkey from the passed byte[]
			ECNamedCurveParameterSpec params
				= ECNamedCurveTable.getParameterSpec("secp256k1");
			ECPublicKeySpec passedPubkeySpec
				= new ECPublicKeySpec(params.getCurve().decodePoint(pubKey),
									  params);
			BCECPublicKey ephPub = new BCECPublicKey("ECDSA",
													 passedPubkeySpec,
									 		BouncyCastleProvider.CONFIGURATION);

			/* To build the AES-256 key that was used by the dApp to encrypt
			 * we must "decapsulate" the private key.  This involves taking
			 * the private key and multiplying it by the uncompressed public
			 * key, then taking sha256(pubKey + product).
			 *
			 * This works because for any two EC keypairs generated from the
			 * same curve, the product of one pubkey multiplied by another
			 * privkey will always equal the product of the opposite combo.
			 * That is, for any two keypairs on secp256k1:
			 * 	pub1 * priv2 == pub2 * priv1
			 * (NB: the operation '*' means using the multiplier function which
			 * was derived from the privkey, not literal multiplication.)
			 *
			 * Here the dApp has generated an ephemeral keypair and used our
			 * (the MVO's) pubkey as "pub1".  It multiplies this by its
			 * ephemeral privkey (priv2) and passes us pub2 ahead of the
			 * ciphertext.  The product is used to build the unique AES-256 key
			 * by taking the hash of the ephemeral pubkey concatenated with
			 * the unique product.
			 *
			 * NB: the dApp will also generate an ephemeral AES-256 key and
			 * pass it in the encrypted JSON map as "replyKey".  This will
			 * be used to encrypt the MVO's reply (unless it's a fatal error).
			 * Therefore we do not need to know the dApp's actual pubkey, which
			 * allows a throw-away ephemeral keypair to be used on that side.
			 */
			// obtain the multiplier that applies to our private key
			ECPrivateKey ecPrivKey = (ECPrivateKey) privKey;
			ECParameterSpec ecSpec = ecPrivKey.getParameters();
			ECCurve ecCurve = ecSpec.getCurve();
			ECMultiplier multiplier = ecCurve.getMultiplier();

			// compute the AES key: sha256(passedPub.uncompressed + product)
			ECPoint prod = multiplier.multiply(ephPub.getQ(), ecPrivKey.getD());
			ByteBuffer keyBuf = ByteBuffer.allocate(130);	// 65+65
			keyBuf.put(pubKey);
			keyBuf.put(prod.getEncoded(false));	// uncompressed format
			byte[] hashBuff = keyBuf.array();
			MessageDigest digest = MessageDigest.getInstance("SHA-256");
			byte[] aesKeyBytes = digest.digest(hashBuff);
			SecretKey aesKey = new SecretKeySpec(aesKeyBytes, "AES");
			
			// the first 16 bytes of the cipherText are the AES IV / nonce
			byte[] iv = new byte[16];
			System.arraycopy(msgDecoded, pubKey.length, iv, 0, iv.length);

			// set up to decrypt using this key and nonce with a 128-bit tag
			Cipher cipher = Cipher.getInstance("AES/GCM/NoPadding");
			GCMParameterSpec gcmIv = new GCMParameterSpec(M_GCMTagLen, iv);
			cipher.init(Cipher.DECRYPT_MODE, aesKey, gcmIv);

			// NB: JS (eciesjs) side prepends the tag rather than appending it
			ByteBuffer encText = ByteBuffer.allocate(
								msgDecoded.length - pubKey.length - iv.length);
			encText.put(msgDecoded,
						pubKey.length + iv.length + 16,
						msgDecoded.length - pubKey.length - iv.length - 16);
			encText.put(msgDecoded, pubKey.length + iv.length, 16);
			byte[] msgText = cipher.doFinal(encText.array());
			plainText = new String(msgText, StandardCharsets.UTF_8);
			Arrays.fill(msgText, (byte)0);
			Arrays.fill(iv, (byte)0);
		}
		// catch all exceptions which can be thrown by this code stanza
		catch (IllegalArgumentException iae) {
			gotErr = true;
		}
		catch (InvalidAlgorithmParameterException iape) {
			gotErr = true;
		}
		catch (NoSuchAlgorithmException nsae) {
			gotErr = true;
		}
		catch (InvalidKeyException ike) {
			gotErr = true;
		}
		catch (IllegalBlockSizeException ibse) {
			gotErr = true;
		}
		catch (NoSuchPaddingException nspe) {
			gotErr = true;
		}
		catch (BadPaddingException bpe) {
			gotErr = true;
		}
		catch (NullPointerException npe) {
			gotErr = true;
		}

		if (gotErr || plainText == null) {
			return null;
		}
		return plainText;
	}

	/**
	 * method to replicate the function of the method
	 * org.web3j.utils.Numeric.toHexStringZeroPadded()
	 * @param value the value to encode
	 * @param prefix if true, add "0x" prefix
	 * @return the hex value (with or without 0x prepended), zero-padded to
	 * length 64
	 */
	public static String toHexStringZeroPadded(BigInteger value, boolean prefix)	{
		if (value == null) {
			return "";
		}
		String result = value.toString(16);
		if (result.length() < 64) {
			int len = 64 - result.length();
			String pad
				= new String(new char[len]).replace("\0", String.valueOf('0'));
			result =  pad + result;
		}
		if (prefix) {
			return "0x" + result;
		}
		return result;
	}

	/**
	 * method to imitate the function of the method
	 * org.web3j.crypto.Hash.sha3String(), except with sha-256 not Keccak-256
	 * @param utf8 UTF-8 encoded string
	 * @return hash value as hex encoded string, without the leading 0x
	 */
	public static String sha3(String utf8) {
		String ret = "";
		if (utf8 == null || utf8.isEmpty()) {
			return ret;
		}
		byte[] hash = null;
		try {
			MessageDigest digest = MessageDigest.getInstance("SHA-256");
			hash = digest.digest(utf8.getBytes(StandardCharsets.UTF_8));
		}
		catch (NoSuchAlgorithmException nsae) {
			return ret;
		}
		StringBuilder hexBuilder = new StringBuilder(hash.length * 2);
		for (int iii = 0; iii < hash.length; iii++) {
			hexBuilder.append(String.format("%02x", hash[iii] & 0xFF));
		}
		ret = hexBuilder.toString();
		return ret;
	}

	/**
	 * method to produce a hex digest of an Ethereum-prefixed hash of data
	 * @param data the data to be hashed, a UTF-8 encoded string
	 * @return the keccak256 Ethereum-prefixed hash, as a hex value (no prefix)
	 */
	public static String ethPrefixHash(String data) {
		if (data == null || data.isEmpty()) {
			return null;
		}

		// generate the hash, prepending the proper Eth prefix + length
		byte[] dataBytes = data.getBytes(StandardCharsets.UTF_8);
		byte[] rawHash = Sign.getEthereumMessageHash(dataBytes);
		return Numeric.toHexStringNoPrefix(rawHash);
	}

	/**
	 * method to build a Solidity-style details hash of the form:
	 * keccak256(abi.encode(address,ID,asset,amount,rand))
	 * @param address the address field (0x prefix)
	 * @param id the ID field (zero-padded hex no prefix)
	 * @param asset the asset field (contract address, 0x prefix)
	 * @param amount the amount field, as a hex string
	 * @param rand the 16-bytes of random data (22 bytes Base64 no padding),
	 * can be empty
	 * @return the hash of the data encoded (no 0x prefix), empty on error
	 */
	public static String buildDetailsHash(String address,
										  String id,
										  String asset,
										  String amount,
										  String rand)
	{
		String hash = "";
		if (address == null || address.isEmpty() || id == null || id.isEmpty()
			|| asset == null || asset.isEmpty() || amount == null
			|| amount.isEmpty() || rand == null)
		{
			return hash;
		}

		String encodedVal = null;
		try {
			ArrayList<Type> params = new ArrayList<Type>(5);
			Address owner = new Address(address);
			params.add(owner);
			BigInteger idBig = new BigInteger(id, 16);
			Uint256 uId = new Uint256(idBig);
			params.add(uId);
			Address contract = new Address(asset);
			params.add(contract);
			BigInteger amtBig = new BigInteger(amount, 16);
			Uint256 amt = new Uint256(amtBig);
			params.add(amt);
			if (!rand.isEmpty()) {
				Base64.Decoder b64d = Base64.getDecoder();
				byte[] randBytes = b64d.decode(rand);
				if (randBytes.length != 16) {
					throw new NumberFormatException("illegal rand length, "
													+ randBytes.length);
				}
				Bytes16 randParam = new Bytes16(randBytes);
				params.add(randParam);
			}

			DefaultFunctionEncoder abiEncode = new DefaultFunctionEncoder();
			encodedVal = abiEncode.encodeParameters(params);
		}
		catch (NumberFormatException nfe) {
		}
		catch (UnsupportedOperationException uoe) {
		}
		catch (NullPointerException npe) {
		}
		if (encodedVal == null) {
			return hash;
		}

		// convert to byte array and take keccak256 hash
		hash = Hash.sha3(encodedVal);
		return Numeric.cleanHexPrefix(hash);
	}

	/**
	 * method to build a Solidity-style details hash of the form:
	 * keccak256(abi.encode(address,ID,asset,amount,rand)).  This method is the
	 * same as buildDetailsHash() except that it logs its passed arguments and
	 * any exceptions encountered.
	 * @param address the address field (0x prefix)
	 * @param id the ID field (zero-padded hex no prefix)
	 * @param asset the asset field (contract address, 0x prefix)
	 * @param amount the amount field, as a hex string
	 * @param rand the 16-bytes of random data (22 bytes Base64 no padding),
	 * @param log the logging object
	 * can be empty
	 * @return the hash of the data encoded (no 0x prefix), empty on error
	 */
	public static String buildDetailsHashDbg(String address,
										  String id,
										  String asset,
										  String amount,
										  String rand, Log log)
	{
		String hash = "";
		log.debug("buildDetailsHashDbg: address = " + address + ", id = " + id
				+ ", asset = " + asset + ", amount = " + amount + ", rand = "
				+ rand);
		if (address == null || address.isEmpty() || id == null || id.isEmpty()
			|| asset == null || asset.isEmpty() || amount == null
			|| amount.isEmpty() || rand == null)
		{
			return hash;
		}

		String encodedVal = null;
		try {
			ArrayList<Type> params = new ArrayList<Type>(5);
			Address owner = new Address(address);
			params.add(owner);
			BigInteger idBig = new BigInteger(id, 16);
			Uint256 uId = new Uint256(idBig);
			params.add(uId);
			Address contract = new Address(asset);
			params.add(contract);
			BigInteger amtBig = new BigInteger(amount, 16);
			Uint256 amt = new Uint256(amtBig);
			params.add(amt);
			if (!rand.isEmpty()) {
				Base64.Decoder b64d = Base64.getDecoder();
				byte[] randBytes = b64d.decode(rand);
				if (randBytes.length != 16) {
					throw new NumberFormatException("illegal rand length, "
													+ randBytes.length);
				}
				Bytes16 randParam = new Bytes16(randBytes);
				params.add(randParam);
			}

			DefaultFunctionEncoder abiEncode = new DefaultFunctionEncoder();
			encodedVal = abiEncode.encodeParameters(params);
			log.debug("buildDetailsHashDbg: hash encoding = \""
					+ encodedVal + "\"");
		}
		catch (NumberFormatException nfe) {
			log.error("NFE", nfe);
		}
		catch (UnsupportedOperationException uoe) {
			log.error("Unsupported Op", uoe);
		}
		catch (NullPointerException npe) {
			log.error("NPE", npe);
		}
		if (encodedVal == null) {
			log.error("Returning empty due to error");
			return hash;
		}

		// convert to byte array and take keccak256 hash
		hash = Hash.sha3(encodedVal);
		return Numeric.cleanHexPrefix(hash);
	}

	/**
	 * method to sign a hash with an ECDSA signature
	 * @param hash the hash of the data being signed (can be Ethereum prefixed)
	 * @param privKey the private key to use
	 * @return the signature as a hex encoded 130-char string, null on failure
	 */
	public static String signHash(String hash, BigInteger privKey) {
		if (privKey == null || hash == null || hash.isEmpty()) {
			return null;
		}

		// generate the signature
		ECKeyPair sigPair = ECKeyPair.create(privKey);
		byte[] hashBytes = Numeric.hexStringToByteArray(hash);
		Sign.SignatureData hashSig = null;
		try {
			hashSig = Sign.signMessage(hashBytes, sigPair, false);
		}
		catch (RuntimeException re) {
			return null;
		}

		// convert sig into hex format, no prefixes
		String digestSig = Numeric.toHexStringNoPrefix(hashSig.getR())
						+ Numeric.toHexStringNoPrefix(hashSig.getS())
						+ Numeric.toHexStringNoPrefix(hashSig.getV());
		return digestSig;
	}

	/**
	 * Method to sign an arbitrary string with a plain ECDSA signature.
	 * Data will not be hashed Ethereum-prefixed, so this method should not
	 * be used for any signature which will be validated on-chain.  If a sig
	 * for an on-chain message is desired, call ethPrefixHash() + signHash().
	 * @param message the data to be hashed and signed (UTF-8)
	 * @param privKey the private key to use
	 * @return the signature as a hex encoded string, null on failure
	 */
	public static String signData(String message, BigInteger privKey) {
		if (privKey == null || message == null || message.isEmpty()) {
			return null;
		}

		// generate the signature
		ECKeyPair sigPair = ECKeyPair.create(privKey);
		byte[] msgBytes = message.getBytes(StandardCharsets.UTF_8);
		Sign.SignatureData msgSig = null;
		try {
			msgSig = Sign.signMessage(msgBytes, sigPair, true);
		}
		catch (RuntimeException re) {
			return null;
		}

		// convert sig into hex format, no prefixes
		String digestSig = Numeric.toHexStringNoPrefix(msgSig.getR())
						+ Numeric.toHexStringNoPrefix(msgSig.getS())
						+ Numeric.toHexStringNoPrefix(msgSig.getV());
		return digestSig;
	}

	/**
	 * Method to validate a signature on generic data (not Ethereum prefixed).
	 * This method works with sigs generated by signData().
	 * @param signedData the data which was signed (UTF-8 encoded)
	 * @param signature the signature, in 130-char hex digest format (prefixed)
	 * @return the address which signed this data (EIP55 format), empty on error
	 */
	public static String getDataSignerAddress(String signedData,
											  String signature)
	{
		String retAddr = "";
		if (signedData == null || signedData.isEmpty()
			|| signature == null || signature.length() < 130) {
			return retAddr;
		}

		// transform the inputs into usable format
		String wholeSig = Numeric.cleanHexPrefix(signature);
		byte[] signedBytes = signedData.getBytes(StandardCharsets.UTF_8);
		String sigR = wholeSig.substring(0, 64);
		String sigS = wholeSig.substring(64, 128);
		String sigV = wholeSig.substring(128);
		byte[] R = Numeric.hexStringToByteArray(sigR);
		byte[] S = Numeric.hexStringToByteArray(sigS);
		byte[] V = Numeric.hexStringToByteArray(sigV);
		Sign.SignatureData sig = new Sign.SignatureData(V, R, S);
		BigInteger signerPubKey = null;
		try {
			signerPubKey = Sign.signedMessageToKey(signedBytes, sig);
		}
		catch (SignatureException se) {
			return retAddr;
		}
		retAddr = Keys.getAddress(signerPubKey);
		return Keys.toChecksumAddress(retAddr);
	}

	/**
	 * Method to validate a signature on a hash (may be Ethereum prefixed).
	 * This method works with sigs generated by signHash().
	 * @param signedHash the hash which was signed (UTF-8 encoded)
	 * @param signature the signature, in 130-char hex digest format
	 * @return the address which signed this data (EIP55 format), empty on error
	 */
	public static String getHashSignerAddress(String signedHash,
											  String signature)
	{
		String retAddr = "";
		if (signedHash == null || signedHash.isEmpty()
			|| signature == null || signature.length() < 130) {
			return retAddr;
		}

		// transform the inputs into usable format
		String wholeSig = Numeric.cleanHexPrefix(signature);
		byte[] signedBytes = Numeric.hexStringToByteArray(signedHash);
		String sigR = wholeSig.substring(0, 64);
		String sigS = wholeSig.substring(64, 128);
		String sigV = wholeSig.substring(128);
		byte[] R = Numeric.hexStringToByteArray(sigR);
		byte[] S = Numeric.hexStringToByteArray(sigS);
		byte[] V = Numeric.hexStringToByteArray(sigV);
		Sign.SignatureData sig = new Sign.SignatureData(V, R, S);
		BigInteger signerPubKey = null;
		try {
			signerPubKey = Sign.signedMessageHashToKey(signedBytes, sig);
		}
		catch (SignatureException se) {
			return retAddr;
		}
		retAddr = Keys.getAddress(signerPubKey);
		return Keys.toChecksumAddress(retAddr);
	}

	/**
	 * Method to validate a signature on EIP712 message data.
	 * @param eip712data the structured data that was signed
	 * @param signature the signature on the data
	 * @return the address which signed the data (EIP55 format), empty on error
	 */
	public static String getEIP712SignerAddress(String eip712data,
												String signature)
	{
		String retAddr = "";
		if (eip712data == null || eip712data.isEmpty()
			|| signature == null || signature.length() < 130) {
			return retAddr;
		}

		// parse the structured EIP712 message data and get its hash
		StructuredDataEncoder sde = null;
		byte[] dataHash = null;
		try {
			sde = new StructuredDataEncoder(eip712data);
			dataHash = sde.hashStructuredData();
		}
		catch (IOException | RuntimeException ire) {
			return retAddr;
		}

		// construct hash of the data
		String sigDataHash = Numeric.toHexStringNoPrefix(dataHash);

		// determine which address actually signed the data
		retAddr = getHashSignerAddress(sigDataHash, signature);
		return retAddr;
	}

	// END methods
}
