/*
 * last modified---
 * 	06-05-23 remove m_Signature, as this is now at a higher level
 * 	03-20-23 add m_ReplyKey on parse
 * 	07-12-22 improve error output labeling
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-30-22 extend ClientRequest
 * 	03-23-22 new
 *
 * purpose---
 * 	encapsulate wallet download request messages to MVOs from dApp clients
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.util.ArrayList;
import javax.servlet.http.HttpServletRequest;


/**
 * This class holds the data for dApp wallet download requests made to MVOs.
 * Such requests are received on the https endpoint, parsed, and stored in these
 * objects for processing by MVOs.  The result of MVO processing will be a
 * request/reply to the Auditor for all the necessary AES keys required to
 * decrypt all the eNFTs, followed by actual decryption of all extant eNFTs
 * validly owned by the signing address.  The MVOs reply will pass back both
 * the secret eNFT data and the keys necessary to decrypt them anew.
 * This class knows how to build itself from input JSON.
 */
public final class ClientWalletBlock extends ClientRequest {
	// BEGIN data members
	/**
	 * list of eNFT IDs being requested (if empty, indicates all)
	 */
	private ArrayList<String>	m_eNFTIds;

	/**
	 * logging object
	 */
	private Log					m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param orgReq the original HTTP level request
	 * @param logger the logging object
	 */
	public ClientWalletBlock(HttpServletRequest orgReq, Log logger) {
		super(orgReq);
		m_Log = logger;
		m_eNFTIds = new ArrayList<String>();
	}

	// GET methods
	/**
	 * obtain the list of file specifications
	 * @return the list, empty if none
	 */
	public ArrayList<String> getIDs() { return m_eNFTIds; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * config the chain Id
	 * @param id the ID of the blockchain this request is for, per chainlist.org
	 */
	public void setChainId(long id) {
		if (id > 0L) {
			m_ChainId = id;
		}
		else {
			m_Log.error("setChainId(): illegal chainId, " + id);
		}
	}

	/**
	 * set the sender of the request
	 * @param sender the wallet address which sent (and signed) the request
	 */
	public void setSender(String sender) {
		if (sender != null && !sender.isEmpty()) {
			m_Sender = new String(sender);
		}
		else {
			m_Log.error("ClientWalletBlock.setSender(): missing sender");
		}
	}

	/**
	 * add an ID to the list of eNFT IDs
	 * @param id the filespec to add
	 */
	public void addID(String id) {
		if (id != null) {
			m_eNFTIds.add(id);
		}
	}


	/**
	 * method to build object from a Map
	 * @param request the mapping, the result of a JSON parse
	 * @return true on success
	 */
	public boolean buildFromMap(Map request) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (request == null || request.isEmpty()) {
			m_Log.error(lbl + "missing top Map");
			return false;
		}
		boolean ret = true;

		// required: the chainId
		Object chain = request.get("chainId");
		if (chain instanceof String) {
			String chainId = (String) chain;
			try {
				Long cId = Long.parseLong(chainId.trim());
				setChainId(cId.longValue());
			}
			catch (NumberFormatException nfe) {
				ret = false;
				m_Log.error(lbl + "illegal chain Id, " + chainId, nfe);
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing chain Id");
		}

		// required: the sender address
		Object send = request.get("sender");
		if (send instanceof String) {
			String sender = (String) send;
			setSender(sender);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing sender");
		}

		// optional: the AES encryption reply key
		Object rKey = request.get("replyKey");
		if (rKey instanceof String) {
			String repKey = (String) rKey;
			setReplyKey(repKey);
		}

		// get ID inputs (none are required, empty list means "get all")
		Object ids = request.get("IDList");
		if (ids instanceof Object[]) {
			Object[] nftIds = (Object[]) ids;
			for (int iii = 0; iii < nftIds.length; iii++) {
				Object id = nftIds[iii];
				if (id instanceof String) {
					String idSpec = (String) id;
					addID(idSpec);
				}
				else {
					ret = false;
					m_Log.error(lbl + "non-String for ID input " + (iii+1));
				}
			}
		}

		return ret;
	}

	/**
	 * method to build entire request object from a JSON string
	 * @param reqData the request data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String reqData) {
		if (reqData == null | reqData.isEmpty()) {
			m_Log.error("ClientWalletBlock.buildFromString: missing client "
						+ "request data");
			return false;
		}
		boolean ret = false;
		Object req = null;
		try {
			req = JSON.parse(reqData);
		}
		catch (IllegalStateException ise) {
			m_Log.error("ClientWalletBlock.buildFromString: client JSON did "
						+ "not parse: \"" + reqData + "\"");
			return ret;
		}

		// parsed object should consist of a Map
		if (!(req instanceof Map)) {
			m_Log.error("ClientWalletBlock.buildFromString: client JSON was "
						+ "not a Map");
			return ret;
		}
		else {
			Map map = (Map) req;
			ret = buildFromMap(map);
		}
		return ret;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_eNFTIds != null) {
				m_eNFTIds.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
