/*
 * last modified---
 * 	04-05-23 make buildSignedData() emit EIP-55 for addresses and assets
 * 	04-04-23 replace buildArgsData() with buildSolArgsData() version
 * 	03-28-23 add buildSolArgsData()
 * 	03-22-23 switch ArgumentsHash.argsHash() to use sha3String()
 * 	02-28-23 add ArgumentsHash inner class
 * 	12-07-22 add optional ClientInput.m_InputRand for withdrawals only
 * 	12-02-22 add ClientPayee.m_Rand to store randomizer value
 * 	08-30-22 allow output eNFT amounts of zero
 * 	08-18-22 debug emission and parsing
 * 	08-12-22 allow outputs with zero amounts
 * 	08-03-22 corrections to array emissions
 * 	07-22-22 improve error output labels
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-04-22 new
 *
 * purpose---
 * 	encapsulate messages from MVOs to Auditors, replicating {OperationsBlock}s
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.Hash;
import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;
import org.web3j.abi.datatypes.Type;
import org.web3j.abi.datatypes.generated.Uint256;
import org.web3j.abi.datatypes.DynamicArray;
import org.web3j.abi.DefaultFunctionEncoder;

import java.util.ArrayList;
import java.util.TreeMap;
import java.util.Map;
import java.math.BigInteger;
import java.text.NumberFormat;
import java.io.IOException;


/**
 * This class holds the data necessary to build {AuditorBlock} JSON objects
 * which are sent to Auditors as broadcasts. These messages replicate the
 * {OperationsBlock} objects returned by MVOs to dApps for submission to smart
 * contracts, except with additional data (amounts, assets) shown in clear.
 * This class knows how to emit itself as JSON, and can also parse a JSON
 * string to populate its fields.
 */
public final class AuditorBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * opcode for operations block, one of deposit|spend|withdrawal
	 */
	private String			m_Opcode;

	/**
	 * single token asset represented, for deposits and withdrawals
	 * (as the address of the relevant token contract)
	 */
	private String			m_Asset;

	/**
	 * quantity of asset represented (a uint256 represented as a BigInteger),
	 * for deposits and withdrawals (spends are calculated from inputs/outputs)
	 */
	private BigInteger		m_Amount;

	/**
	 * originating user address (msg.sender), EIP-55
	 */
	private String			m_Sender;

	/**
	 * helper class describing a payee specifications (for all opcodes)
	 */
	public class ClientPayee {
		/**
		 * payee ID (sequence number, such as 001, 002 etc.)
		 */
		public String		m_Payee;

		/**
		 * address of the payee, to which an eNFT will be minted (EIP-55 hex)
		 */
		public String		m_Address;

		/**
		 * unique ID of the eNFT to be minted for this payee (a 64-digit hex
		 * number represented as a String)
		 */
		public String		m_ID;

		/**
		 * asset represented by eNFT (will equal m_Asset for deposit/withdraw)
		 */
		public String		m_OutputAsset;

		/**
		 * amount of the eNFT to be issued (uint256 in {decimals} precision)
		 */
		public BigInteger	m_OutputAmount;

		/**
		 * randomizer value to assure details hash uniqueness (optional)
		 */
		public String		m_Rand;

		/**
		 * details hash: (not used on a withdrawal)
		 * keccak256(abi.encode(m_Address,m_ID,asset,m_OutputAmount,m_Rand))
		 */
		public String		m_DetailsHash;

		/**
		 * eNFT metadata for this ID, verbatim from MVO as JSON string
		 */
		public String		m_EncMetadata;

		/**
		 * full eNFT metadata for this ID, with {enshrouded} encrypted to
		 * the pubkey of {m_Address} in base64 format (m_EncMetadata parsed)
		 */
		public NFTmetadata	m_Metadata;

		/**
		 * nullary constructor
		 */
		public ClientPayee() {
			m_Payee = m_Address = m_ID = m_DetailsHash = m_EncMetadata = "";
			m_OutputAsset = m_Rand = "";
			m_OutputAmount = new BigInteger("0");
		}
	}

	/**
	 * list of payees defined in reply block
	 */
	private ArrayList<ClientPayee>	m_Payees;

	/**
	 * helper class describing input specifications (for spends and withdrawals)
	 */
	public class ClientInput {
		/**
		 * input ID (sequence number, such as 001, 002, etc.)
		 */
		public String		m_Input;

		/**
		 * ID of the eNFT offered as input
		 */
		public String		m_ID;

		/**
		 * the details hash, which must match the value in the smart contract's
		 * mapping (used only for spends)
		 */
		public String		m_DetailsHash;

		/**
		 * the token asset the eNFT represents (used only for withdrawals)
		 */
		public String		m_InputAsset;

		/**
		 * the amount of the asset (used only for withdrawals)
		 */
		public BigInteger	m_InputAmount;

		/**
		 * the randomizer value (optional, used only for withdrawals)
		 */
		public String		m_InputRand;

		/**
		 * nullary constructor
		 */
		public ClientInput() {
			m_Input = m_ID = m_DetailsHash = m_InputRand = "";
			m_InputAmount = new BigInteger("0");
		}
	}

	/**
	 * list of input eNFTs provided by client for operation
	 */
	private ArrayList<ClientInput>	m_Inputs;

	/**
	 * helper class describing the hash of the arguments the dApp will need to
	 * pass to the smart contract API, which each MVO must sign separately
	 */
	public final class ArgumentsHash {
		/**
		 * Plain text for the arguments which are hashed.
		 * Format depends on the opcode context, as follows.
		 * 1. If the opcode=deposit, the hash is built thus:
		 * 		amount,ids[],detailsHashes[]
		 * 	where:	amount is the total output value
		 *
		 * 2. For opcode=spend, the hash is built thus:
		 * 		inputIds[],outputIds[],detailsHashes[]
		 * 	where:	inputIds[] are IDs of eNFTs to be burned
		 * 			detailsHashN is the details hash of outputId eNFT N
		 *
		 * 3. For opcode=withdraw, the hash is built thus:
		 * 		inputIds[],amount
		 * 		Iff there is a (solo) change output eNFT, append:
		 * 			,id[],detailsHash[]
		 * 	where:	IDs in inputIds[] are those of eNFTs to be burned
		 * 			amount is the total being withdrawn
		 * 			the id[] and detailsHash[] arrays are length 1
		 */
		public String		m_ArgumentsData;

		/**
		 * keccack256() of the m_ArgumentsData (emitted in output JSON)
		 */
		public String		m_ArgsHash;

		/**
		 * nullary constructor
		 */
		public ArgumentsHash() {
			m_ArgumentsData = m_ArgsHash = "";
		}

		/**
		 * alternate way to build arguments data to be hashed, the Solidity way
		 * @return the data on success, null if some required datum was missing
		 */
		public String buildArgsData() {
			final String errLbl = "ArgumentsHash.buildArgsData: ";
			StringBuilder argsData = new StringBuilder(2048);
			Uint256 amt = null;
			ArrayList<Type> params = new ArrayList<Type>();
			int outSize = 0;
			int inSize = 0;
			try {
				if (m_Opcode.equals(ClientMVOBlock.M_OpDeposit)) {
					if (m_Payees.isEmpty()) {
						m_Log.error(errLbl + "no deposit payees");
						return null;
					}
					outSize = m_Payees.size();

					// add amount
					amt = new Uint256(m_Amount);
					params.add(amt);

					// add IDs and detailsHashes for all output payees
					ArrayList<Uint256> ids = new ArrayList<Uint256>(outSize);
					ArrayList<Uint256> dets = new ArrayList<Uint256>(outSize);
					for (ClientPayee payee : m_Payees) {
						BigInteger id = new BigInteger(payee.m_ID, 16);
						ids.add(new Uint256(id));
						BigInteger hash
							= new BigInteger(payee.m_DetailsHash, 16);
						dets.add(new Uint256(hash));
					}
					DynamicArray<Uint256> idParms
						= new DynamicArray<Uint256>(Uint256.class, ids);
					params.add(idParms);
					DynamicArray<Uint256> detParms
						= new DynamicArray<Uint256>(Uint256.class, dets);
					params.add(detParms);
				}
				else if (m_Opcode.equals(ClientMVOBlock.M_OpSpend)) {
					if (m_Inputs.isEmpty()) {
						m_Log.error(errLbl + "no spend inputs");
						return null;
					}
					inSize = m_Inputs.size();
					if (m_Payees.isEmpty()) {
						m_Log.error(errLbl + "no spend payees");
						return null;
					}
					outSize = m_Payees.size();

					// add IDs for all inputs
					ArrayList<Uint256> inIds = new ArrayList<Uint256>(inSize);
					for (ClientInput input : m_Inputs) {
						BigInteger id = new BigInteger(input.m_ID, 16);
						inIds.add(new Uint256(id));
					}
					DynamicArray<Uint256> inIdParms
						= new DynamicArray<Uint256>(Uint256.class, inIds);
					params.add(inIdParms);

					// add IDs and detailsHashes for all output payees
					ArrayList<Uint256> outIds = new ArrayList<Uint256>(outSize);
					ArrayList<Uint256> dets = new ArrayList<Uint256>(outSize);
					for (ClientPayee payee : m_Payees) {
						BigInteger id = new BigInteger(payee.m_ID, 16);
						outIds.add(new Uint256(id));
						BigInteger hash
							= new BigInteger(payee.m_DetailsHash, 16);
						dets.add(new Uint256(hash));
					}
					DynamicArray<Uint256> outIdParms
						= new DynamicArray<Uint256>(Uint256.class, outIds);
					params.add(outIdParms);
					DynamicArray<Uint256> detParms
						= new DynamicArray<Uint256>(Uint256.class, dets);
					params.add(detParms);
				}
				else if (m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
					if (m_Inputs.isEmpty()) {
						m_Log.error(errLbl + "no withdraw inputs");
						return null;
					}
					inSize = m_Inputs.size();

					// add IDs for all inputs
					ArrayList<Uint256> inIds = new ArrayList<Uint256>(inSize);
					for (ClientInput input : m_Inputs) {
						BigInteger id = new BigInteger(input.m_ID, 16);
						inIds.add(new Uint256(id));
					}
					DynamicArray<Uint256> inIdParms
						= new DynamicArray<Uint256>(Uint256.class, inIds);
					params.add(inIdParms);

					// add amount
					amt = new Uint256(m_Amount);
					params.add(amt);

					// add change output ID + detailsHash if there is one
					if (!m_Payees.isEmpty()) {
						// there can only be a single change output
						ClientPayee payee = m_Payees.get(0);
						BigInteger id = new BigInteger(payee.m_ID, 16);
						ArrayList<Uint256> outIds = new ArrayList<Uint256>(1);
						outIds.add(new Uint256(id));
						DynamicArray<Uint256> idParm
							= new DynamicArray<Uint256>(Uint256.class, outIds);
						params.add(idParm);

						// NB: the detailsHash is not emitted on a withdraw
						ArrayList<Uint256> dets = new ArrayList<Uint256>(1);
						BigInteger hash = null;
						if (payee.m_DetailsHash.isEmpty()) {
							String detHash
								= EncodingUtils.buildDetailsHash(m_Sender,
																 payee.m_ID,
																 m_Asset,
											payee.m_OutputAmount.toString(16),
																 payee.m_Rand);
							hash = new BigInteger(detHash, 16);
						}
						else {
							hash = new BigInteger(payee.m_DetailsHash, 16);
						}
						dets.add(new Uint256(hash));
						DynamicArray<Uint256> detParms
							= new DynamicArray<Uint256>(Uint256.class, dets);
						params.add(detParms);
					}
				}
				else {
					m_Log.error(errLbl + "illegal opcode, " + m_Opcode);
					return null;
				}
			}
			catch (NumberFormatException nfe) {
				m_Log.error(errLbl
						+ "error parsing what should be a hex uint256", nfe);
				return null;
			}
			catch (UnsupportedOperationException uoe) {
				m_Log.error(errLbl + "error doing encoding", uoe);
				return null;
			}

			// encode the parameters (unpacked) to build obHash for SC
			DefaultFunctionEncoder abiEncode = new DefaultFunctionEncoder();
			try {
				argsData.append(abiEncode.encodeParameters(params));
			}
			catch (NullPointerException npe) {
				m_Log.error(errLbl + "NPE doing encoding", npe);
				return null;
			}
			return argsData.toString();
		}

		/**
		 * calculate the hash of the args data
		 * @return the Keccak-256 hash (null if arguments data not set first)
		 */
		public String argsHash() {
			if (m_ArgumentsData.isEmpty()) {
				m_Log.error("ArgumentsHash.argsHash: args data not built");
				return null;
			}

			// convert to byte array and take keccak256 hash
			String kec = Hash.sha3(m_ArgumentsData);
			return Numeric.cleanHexPrefix(kec);
		}
	}

	/**
	 * the arguments hash block
	 */
	private ArgumentsHash	m_ArgsBlock;

	/**
	 * list of MVO signatures appended to this output block
	 */
	private ArrayList<MVOSignature>	m_Signatures;

	/**
	 * logging object
	 */
	private Log				m_Log;

	/**
	 * error code status (must be 0 to indicate success)
	 */
	private int				m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger the logging object
	 */
	public AuditorBlock(Log logger) {
		m_Log = logger;
		m_Opcode = m_Asset = m_Sender = "";
		m_Amount = new BigInteger("0");
		m_Payees = new ArrayList<ClientPayee>(ClientMVOBlock.M_ARRAY_MAX);
		m_Inputs = new ArrayList<ClientInput>(ClientMVOBlock.M_ARRAY_MAX);
		m_Signatures = new ArrayList<MVOSignature>();
		m_ArgsBlock = new ArgumentsHash();
	}

	// GET methods
	/**
	 * obtain the opcode of this reply block
	 * @return the code, one of deposit|spend|withdrawal
	 */
	public String getOpcode() { return m_Opcode; }

	/**
	 * obtain the asset name as EIP-55 address
	 * @return the asset (token address) involved in a deposit or withdrawal
	 */
	public String getAsset() { return m_Asset; }

	/**
	 * obtain the quantity of the asset, uint256 but represented as a BigInteger
	 * @return the amount of this asset involved in the deposit or withdrawal
	 */
	public BigInteger getAmount() { return m_Amount; }

	/**
	 * obtain the sender address
	 * @return the address involved in the transaction
	 */
	public String getSender() { return m_Sender; }

	/**
	 * obtain the list of payee outputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<ClientPayee> getPayees() { return m_Payees; }

	/**
	 * obtain the list of eNFT inputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<ClientInput> getInputs() { return m_Inputs; }

	/**
	 * obtain the arguments hash block
	 * @return the block
	 */
	public ArgumentsHash getArgsBlock() { return m_ArgsBlock; }

	/**
	 * obtain the list of MVO signatures for this
	 */
	public ArrayList<MVOSignature> getSignatures() { return m_Signatures; }

	/**
	 * obtain the error code from the last addJSON() call
	 * @return the result code from building output JSON
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * set the opcode for the reply block
	 * @param op the opcode, one of deposit|spend|withdrawal
	 */
	public void setOpcode(String op) {
		if (op != null && !op.isEmpty()) {
			m_Opcode = new String(op);
		}
		else {
			m_Log.error("AuditorBlock.setOpcode: missing opcode");
		}
	}

	/**
	 * config the asset name (used for deposits and withdrawals)
	 * @param asset the token address referenced in the reply block
	 */
	public void setAsset(String asset) {
		if (asset != null && !asset.isEmpty()) {
			m_Asset = new String(asset);
		}
		else {
			m_Log.error("AuditorBlock.setAsset: missing asset");
		}
	}

	/**
	 * config the quantity of the asset, uint256 but represented as a string
	 * @param amt amount of this asset represented by the eNFT
	 */
	public void setAmount(String amt) {
		if (amt != null && !amt.isEmpty()) {
			try {
				m_Amount = new BigInteger(amt);
			}
			catch (NumberFormatException nfe) {
				m_Log.error("AuditorBlock.setAmount: illegal amount, " + amt);
			}
		}
		else {
			m_Log.error("AuditorBlock.setAmount: missing amount");
		}
	}

	/**
	 * config the sender address
	 * @param addr the address of the user doing the transaction
	 */
	public void setSender(String addr) {
		if (addr != null && !addr.isEmpty()) {
			m_Sender = new String(addr);
		}
		else {
			m_Log.error("AuditorBlock.setSender: missing address");
		}
	}

	/**
	 * add a client payee output to list
	 * @param payee the payee to add
	 */
	public void addPayee(ClientPayee payee) {
		if (payee != null) {
			m_Payees.add(payee);
		}
	}

	/**
	 * add an eNFT input to list
	 * @param eNFT an input
	 */
	public void addInput(ClientInput eNFT) {
		if (eNFT != null) {
			m_Inputs.add(eNFT);
		}
	}

	/**
	 * add a signature to the list
	 * @param sig the signature of an MVO
	 */
	public void addSignature(MVOSignature sig) {
		if (sig != null) {
			m_Signatures.add(sig);
		}
	}

	/**
	 * Build the {MVOSigned} element JSON that's actually signed.  This is done
	 * separately so that each MVO can replicate the exact same signed data.
	 * All address types are emitted in EIP-55 format.
	 * @return the JSON string for the {MVOSigned} element including delimiters,
	 * or null on error
	 */
	public String buildSignedData() {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{");
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);

		// use a TreeMap so the output will be lexically sorted
		TreeMap<String, String> jsonElts = new TreeMap<String, String>();
		final String lbl
			= this.getClass().getSimpleName() + ".buildSignedData: ";

		// add opcode
		if (m_Opcode.isEmpty()) {
			m_Log.error(lbl + "missing opcode");
			m_ErrCode = 1;
			return null;
		}
		jsonElts.put("opcode", m_Opcode);

		// add the sender (user performing transaction)
		if (m_Sender.isEmpty()) {
			m_Log.error(lbl + "missing withdrawal address");
			m_ErrCode = 13;
			return null;
		}
		jsonElts.put("sender", Keys.toChecksumAddress(m_Sender));

		// add arguments hash
		if (m_ArgsBlock.m_ArgsHash.isEmpty()) {
			m_Log.error(lbl + "argsHash is not set");
			m_ErrCode = 18;
			return null;
		}
		jsonElts.put("argsHash", m_ArgsBlock.m_ArgsHash);

		// other elements will depend upon the opcode
		if (m_Opcode.equals(ClientMVOBlock.M_OpDeposit)) {
			// add the amount being deposited
			if (m_Amount.compareTo(BigInteger.ZERO) <= 0) {
				m_Log.error(lbl + "missing or illegal deposit amount");
				m_ErrCode = 3;
				return null;
			}
			jsonElts.put("amount", m_Amount.toString());

			// specify the asset
			if (m_Asset.isEmpty()) {
				m_Log.error(lbl + "missing deposit asset");
				m_ErrCode = 4;
				return null;
			}
			jsonElts.put("asset", Keys.toChecksumAddress(m_Asset));

			// add the outputs array
			if (m_Payees.isEmpty()) {
				m_Log.error(lbl + "missing deposit outputs[]");
				m_ErrCode = 5;
				return null;
			}
			StringBuilder outputs = new StringBuilder(4096);
			int outSeq = 1;
			outputs.append("[");
			for (ClientPayee cp : m_Payees) {
				cp.m_Payee = nf.format(outSeq++);
				outputs.append("{\"" + cp.m_Payee + "\":{");
				// these fields too must be sorted
				TreeMap<String, String> outElts = new TreeMap<String, String>();

				// address to which eNFT will be minted
				if (cp.m_Address.isEmpty()) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has no address");
					m_ErrCode = 6;
					return null;
				}
				outElts.put("address", Keys.toChecksumAddress(cp.m_Address));

				// ID assigned to new eNFT
				if (cp.m_ID.isEmpty()) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has no ID");
					m_ErrCode = 7;
					return null;
				}
				outElts.put("id", cp.m_ID);

				// output asset for new eNFT
				if (cp.m_OutputAsset.isEmpty()) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has no output asset");
					m_ErrCode = 17;
					return null;
				}
				outElts.put("asset", Keys.toChecksumAddress(cp.m_OutputAsset));

				// output amount for new eNFT
				if (cp.m_OutputAmount.compareTo(BigInteger.ZERO) < 0) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has illegal output amount, "
								+ cp.m_OutputAmount);
					m_ErrCode = 16;
					return null;
				}
				outElts.put("amount", cp.m_OutputAmount.toString());

				// optional field for randomizer value
				if (!cp.m_Rand.isEmpty()) {
					outElts.put("rand", cp.m_Rand);
				}

				// details hash for new eNFT
				if (cp.m_DetailsHash.isEmpty()) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has no details hash");
					m_ErrCode = 8;
					return null;
				}
				outElts.put("hash", cp.m_DetailsHash);

				// encoded metadata for new eNFT
				if (cp.m_EncMetadata.isEmpty()) {
					m_Log.error(lbl + "deposit output "
								+ cp.m_Payee + " has no metadata");
					m_ErrCode = 9;
					return null;
				}
				outElts.put("metadata", cp.m_EncMetadata);

				// now emit the output elements for this array element
				int cnt = 0;
				for (String key : outElts.keySet()) {
					if (key.equals("metadata")) {
						// no quotes surrounding this one
						outputs.append("\"" + key + "\":" + outElts.get(key));
					}
					else {
						outputs.append("\"" + key + "\":\""
										+ outElts.get(key) + "\"");
					}
					// add a comma except for the last one
					if (++cnt < outElts.size()) {
						outputs.append(",");
					}
				}
				// end the m_Payee object, with comma unless last
				if (outSeq <= m_Payees.size()) {
					outputs.append("}},");
				}
				else {
					outputs.append("}}]");	// terminate array object
				}
			}
			// add entire outputs [] to output elements
			jsonElts.put("outputs", outputs.toString());

			// emit all JSON elements in order
			int idx = 0;
			for (String jKey : jsonElts.keySet()) {
				out.append("\"" + jKey + "\":");
				String jVal = jsonElts.get(jKey);
				if (jKey.equals("outputs")) {
					out.append(jVal);
				}
				else {
					out.append("\"" + jVal + "\"");
				}
				if (++idx < jsonElts.size()) {
					out.append(",");
				}
			}
		}
		else if (m_Opcode.equals(ClientMVOBlock.M_OpSpend)) {
			// add the inputs array
			if (m_Inputs.isEmpty()) {
				m_Log.error(lbl + "missing spend inputs[]");
				m_ErrCode = 10;
				return null;
			}
			StringBuilder inputs = new StringBuilder(4096);
			int inSeq = 1;
			inputs.append("[");
			for (ClientInput ci : m_Inputs) {
				ci.m_Input = nf.format(inSeq++);
				inputs.append("{\"" + ci.m_Input + "\":{");
				// these fields too must be sorted
				TreeMap<String, String> inElts = new TreeMap<String, String>();

				// ID assigned to old eNFT
				if (ci.m_ID.isEmpty()) {
					m_Log.error(lbl + "spend input " + ci.m_Input
								+ " has no ID");
					m_ErrCode = 11;
					return null;
				}
				inElts.put("id", ci.m_ID);

				// details hash for old eNFT
				if (ci.m_DetailsHash.isEmpty()) {
					m_Log.error(lbl + "spend output " + ci.m_Input
								+ " has no details hash");
					m_ErrCode = 12;
					return null;
				}
				inElts.put("hash", ci.m_DetailsHash);

				// now emit the output elements for this array element
				int cnt = 0;
				for (String key : inElts.keySet()) {
					inputs.append("\"" + key + "\":\"" + inElts.get(key));
					inputs.append("\"");
					// add a comma except for the last one
					if (++cnt < inElts.size()) {
						inputs.append(",");
					}
				}
				// end the m_Input object, with comma unless last
				if (inSeq <= m_Inputs.size()) {
					inputs.append("}},");
				}
				else {
					inputs.append("}}]");	// terminate array object
				}
			}
			// add entire inputs [] to input elements
			jsonElts.put("inputs", inputs.toString());

			// add the outputs array
			if (m_Payees.isEmpty()) {
				m_Log.error(lbl + "missing spend outputs[]");
				m_ErrCode = 5;
				return null;
			}
			StringBuilder outputs = new StringBuilder(4096);
			int outSeq = 1;
			outputs.append("[");
			for (ClientPayee cp : m_Payees) {
				cp.m_Payee = nf.format(outSeq++);
				outputs.append("{\"" + cp.m_Payee + "\":{");
				// these fields too must be sorted
				TreeMap<String, String> outElts = new TreeMap<String, String>();

				// address to which eNFT will be minted
				if (cp.m_Address.isEmpty()) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has no address");
					m_ErrCode = 6;
					return null;
				}
				outElts.put("address", Keys.toChecksumAddress(cp.m_Address));

				// ID assigned to new eNFT
				if (cp.m_ID.isEmpty()) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has no ID");
					m_ErrCode = 7;
					return null;
				}
				outElts.put("id", cp.m_ID);

				// output asset for new eNFT
				if (cp.m_OutputAsset.isEmpty()) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has no asset");
					m_ErrCode = 17;
					return null;
				}
				outElts.put("asset", Keys.toChecksumAddress(cp.m_OutputAsset));

				// output amount for new eNFT
				if (cp.m_OutputAmount.compareTo(BigInteger.ZERO) < 0) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has illegal amount, " + cp.m_OutputAmount);
					m_ErrCode = 16;
					return null;
				}
				outElts.put("amount", cp.m_OutputAmount.toString());

				// optional field for randomizer value
				if (!cp.m_Rand.isEmpty()) {
					outElts.put("rand", cp.m_Rand);
				}

				// details hash for new eNFT
				if (cp.m_DetailsHash.isEmpty()) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has no details hash");
					m_ErrCode = 8;
					return null;
				}
				outElts.put("hash", cp.m_DetailsHash);

				// metadata for new eNFT
				if (cp.m_EncMetadata.isEmpty()) {
					m_Log.error(lbl + "spend output " + cp.m_Payee
								+ " has no metadata");
					m_ErrCode = 9;
					return null;
				}
				outElts.put("metadata", cp.m_EncMetadata);

				// now emit the output elements for this array element
				int cnt = 0;
				for (String key : outElts.keySet()) {
					if (key.equals("metadata")) {
						// no quotes surrounding this one
						outputs.append("\"" + key + "\":" + outElts.get(key));
					}
					else {
						outputs.append("\"" + key + "\":\"" + outElts.get(key));
						outputs.append("\"");
					}
					// add a comma except for the last one
					if (++cnt < outElts.size()) {
						outputs.append(",");
					}
				}
				// end the m_Payee object, with comma unless last
				if (outSeq <= m_Payees.size()) {
					outputs.append("}},");
				}
				else {
					outputs.append("}}]");	// terminate array object
				}
			}
			// add entire outputs [] to output elements
			jsonElts.put("outputs", outputs.toString());

			// emit all JSON elements in order
			int idx = 0;
			for (String jKey : jsonElts.keySet()) {
				out.append("\"" + jKey + "\":");
				String jVal = jsonElts.get(jKey);
				if (jKey.equals("outputs") || jKey.equals("inputs")) {
					out.append(jVal);
				}
				else {
					out.append("\"" + jVal + "\"");
				}
				if (++idx < jsonElts.size()) {
					out.append(",");
				}
			}
		}
		else if (m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
			// specify the asset
			if (m_Asset.isEmpty()) {
				m_Log.error(lbl + "missing withdrawal asset");
				m_ErrCode = 4;
				return null;
			}
			jsonElts.put("asset", Keys.toChecksumAddress(m_Asset));

			// add the amount being withdrawn
			if (m_Amount.compareTo(BigInteger.ZERO) <= 0) {
				m_Log.error(lbl + "missing or illegal withdrawal amount");
				m_ErrCode = 3;
				return null;
			}
			jsonElts.put("amount", m_Amount.toString());

			// add the inputs array
			if (m_Inputs.isEmpty()) {
				m_Log.error(lbl + "missing withdraw inputs[]");
				m_ErrCode = 10;
				return null;
			}
			StringBuilder inputs = new StringBuilder(4096);
			int inSeq = 1;
			inputs.append("[");
			for (ClientInput ci : m_Inputs) {
				ci.m_Input = nf.format(inSeq++);
				inputs.append("{\"" + ci.m_Input + "\":{");
				// these fields too must be sorted
				TreeMap<String, String> inElts = new TreeMap<String, String>();

				// ID assigned to old eNFT
				if (ci.m_ID.isEmpty()) {
					m_Log.error(lbl + "withdraw input "
								+ ci.m_Input + " has no ID");
					m_ErrCode = 11;
					return null;
				}
				inElts.put("id", ci.m_ID);

				// asset of old eNFT
				if (ci.m_InputAsset.isEmpty()) {
					m_Log.error(lbl + "withdraw input "
								+ ci.m_Input + " has no asset");
					m_ErrCode = 14;
					return null;
				}
				inElts.put("asset", Keys.toChecksumAddress(ci.m_InputAsset));

				// amount of old eNFT
				if (ci.m_InputAmount.compareTo(BigInteger.ZERO) < 0) {
					m_Log.error(lbl + "withdraw input " + ci.m_Input
								+ " has illegal amount, " + ci.m_InputAmount);
					m_ErrCode = 15;
					return null;
				}
				inElts.put("amount", ci.m_InputAmount.toString());

				// include randomizer value if present
				if (!ci.m_InputRand.isEmpty()) {
					inElts.put("rand", ci.m_InputRand);
				}

				// now emit the output elements for this array element
				int cnt = 0;
				for (String key : inElts.keySet()) {
					inputs.append("\"" + key + "\":\"" + inElts.get(key));
					inputs.append("\"");
					// add a comma except for the last one
					if (++cnt < inElts.size()) {
						inputs.append(",");
					}
				}
				// end the m_Input object, with comma unless last
				if (inSeq <= m_Inputs.size()) {
					inputs.append("}},");
				}
				else {
					inputs.append("}}]");	// terminate array object
				}
			}
			// add entire inputs [] to input elements
			jsonElts.put("inputs", inputs.toString());

			// add the outputs array (can be empty, just means no change eNFT)
			if (!m_Payees.isEmpty()) {
				StringBuilder outputs = new StringBuilder(4096);
				int outSeq = 1;
				outputs.append("[");
				for (ClientPayee cp : m_Payees) {
					cp.m_Payee = nf.format(outSeq++);
					outputs.append("{\"" + cp.m_Payee + "\":{");
					// these fields too must be sorted
					TreeMap<String, String> outElts
						= new TreeMap<String, String>();

					// address to which eNFT will be minted
					if (cp.m_Address.isEmpty()) {
						m_Log.error(lbl + "withdraw output "
									+ cp.m_Payee + " has no address");
						m_ErrCode = 6;
						return null;
					}
					outElts.put("address",
								Keys.toChecksumAddress(cp.m_Address));

					// ID assigned to new eNFT
					if (cp.m_ID.isEmpty()) {
						m_Log.error(lbl + "withdraw output "
									+ cp.m_Payee + " has no ID");
						m_ErrCode = 7;
						return null;
					}
					outElts.put("id", cp.m_ID);

					// output asset for new eNFT
					if (cp.m_OutputAsset.isEmpty()) {
						m_Log.error(lbl + "withdraw output "
									+ cp.m_Payee + " has no asset");
						m_ErrCode = 17;
						return null;
					}
					outElts.put("asset",
								Keys.toChecksumAddress(cp.m_OutputAsset));

					// amount for this new eNFT
					if (cp.m_OutputAmount.compareTo(BigInteger.ZERO) < 0) {
						m_Log.error(lbl + "withdraw output " + cp.m_Payee
									+ " has illegal amount, "
									+ cp.m_OutputAmount);
						m_ErrCode = 16;
						return null;
					}
					outElts.put("amount", cp.m_OutputAmount.toString());

					// optional field for randomizer value
					if (!cp.m_Rand.isEmpty()) {
						outElts.put("rand", cp.m_Rand);
					}

					// metadata for new eNFT
					if (cp.m_EncMetadata.isEmpty()) {
						m_Log.error(lbl + "withdraw output "
									+ cp.m_Payee + " has no metadata");
						m_ErrCode = 9;
						return null;
					}
					outElts.put("metadata", cp.m_EncMetadata);

					// now emit the output elements for this array element
					int cnt = 0;
					for (String key : outElts.keySet()) {
						if (key.equals("metadata")) {
							// no quotes surrounding this one
							outputs.append("\"" + key + "\":"
											+ outElts.get(key));
						}
						else {
							outputs.append("\"" + key + "\":\""
											+ outElts.get(key) + "\"");
						}
						// add a comma except for the last one
						if (++cnt < outElts.size()) {
							outputs.append(",");
						}
					}
					// end the m_Payee object, with comma unless last
					if (outSeq <= m_Payees.size()) {
						outputs.append("}},");
					}
					else {
						outputs.append("}}]");	// terminate array object
					}
				}
				// add entire outputs [] to output elements
				jsonElts.put("outputs", outputs.toString());
			}

			// emit all JSON elements in order
			int idx = 0;
			for (String jKey : jsonElts.keySet()) {
				out.append("\"" + jKey + "\":");
				String jVal = jsonElts.get(jKey);
				if (jKey.equals("outputs") || jKey.equals("inputs")) {
					out.append(jVal);
				}
				else {
					out.append("\"" + jVal + "\"");
				}
				if (++idx < jsonElts.size()) {
					out.append(",");
				}
			}
		}
		else {
			m_Log.error(lbl + "illegal opcode, " + m_Opcode);
			m_ErrCode = 2;
			return null;
		}
		// terminate MVOSigned object
		out.append("}");
		return out.toString();
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in (sorted) JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{\"MVOSigned\":");
		// first, put the {MVOSigned} data in
		String mvoSigned = buildSignedData();
		if (mvoSigned == null || m_ErrCode != 0) {
			m_Log.error("AuditorBlock.addJSON: signed data could not be built");
			return;
		}
		out.append(mvoSigned);

		// add sigs []
		out.append(",\"signatures\":[");
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		int sigCnt = 1;
		for (MVOSignature sig : m_Signatures) {
			if (sig.m_Sequence.isEmpty()) {
				sig.m_Sequence = nf.format(sigCnt);
			}
			out.append("{\"" + sig.m_Sequence + "\":{");
			out.append("\"signer\":\"" + sig.m_Signer + "\",");
			out.append("\"argsSig\":\"" + sig.m_ArgsSig + "\",");
			out.append("\"sig\":\"" + sig.m_Signature + "\"}}");
			// add comma except on last
			if (sigCnt++ < m_Signatures.size()) {
				out.append(",");
			}
		}

		out.append("]}");
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("AuditorBlock.addJSON: exception appending", ioe);
		}
	}

	/**
	 * method to build object from a Map, the result of a JSON parse
	 * @param block the mapping
	 * @return true on success
	 */
	public boolean buildFromMap(Map block) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (block == null || block.isEmpty()) {
			m_Log.error(lbl + "missing top Map");
			return false;
		}
		boolean ret = true;

		// required: inner map {MVOSigned}
		Object mvoObj = block.get("MVOSigned");
		if (!(mvoObj instanceof Map)) {
			ret = false;
			m_Log.error(lbl + "missing {MVOSigned} block");
			return ret;
		}
		Map mvoSigned = (Map) mvoObj;

		// required: the opcode
		Object opc = mvoSigned.get("opcode");
		if (opc instanceof String) {
			String opcode = (String) opc;
			setOpcode(opcode);
			if (!(m_Opcode.equals(ClientMVOBlock.M_OpDeposit)
				  || m_Opcode.equals(ClientMVOBlock.M_OpSpend)
				  || m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)))
			{
				ret = false;
				m_Log.error(lbl + "illegal AB opcode," + m_Opcode);
				return ret;
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing AB opcode");
			return ret;
		}

		// required: the arguments hash
		Object argHash = mvoSigned.get("argsHash");
		if (argHash instanceof String) {
			String argsHash = (String) argHash;
			m_ArgsBlock.m_ArgsHash = argsHash;
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing AB argsHash");
			return ret;
		}

		// the amount (required for deposit and withdraw)
		Object amt = mvoSigned.get("amount");
		if (amt instanceof String) {
			String amount = (String) amt;
			setAmount(amount);
		}
		else if (!m_Opcode.equals(ClientMVOBlock.M_OpSpend)) {
			ret = false;
			m_Log.error(lbl + "missing AB amount");
		}

		// the asset (required for deposit and withdraw)
		Object ass = mvoSigned.get("asset");
		if (ass instanceof String) {
			String asset = (String) ass;
			setAsset(asset);
		}
		else if (!m_Opcode.equals(ClientMVOBlock.M_OpSpend)) {
			ret = false;
			m_Log.error(lbl + "missing AB asset");
		}

		// the sender
		Object user = mvoSigned.get("sender");
		if (user instanceof String) {
			String sender = (String) user;
			setSender(sender);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing AB sender");
		}

		// inputs[] (required for spend and withdraw)
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		Object inps = mvoSigned.get("inputs");
		if (inps instanceof Object[]) {
			Object[] inputs = (Object[]) inps;
			if (inputs.length < 1
				&& !m_Opcode.equals(ClientMVOBlock.M_OpDeposit))
			{
				ret = false;
				m_Log.error(lbl + "missing AB eNFT inputs");
			}
			for (int iii = 0; iii < inputs.length; iii++) {
				Object inp = inputs[iii];
				if (inp instanceof Map) {
					Map input = (Map) inp;
					// get input index
					long idx = iii + 1;
					String key = nf.format(idx);
					Object inputSpec = input.get(key);
					ClientInput ci = new ClientInput();
					if (inputSpec instanceof Map) {
						Map iSpec = (Map) inputSpec;
						ci.m_Input = new String(key);

						// get the ID
						Object ido = iSpec.get("id");
						if (ido instanceof String) {
							String id = (String) ido;
							ci.m_ID = new String(id);
						}
						else {
							ret = false;
							m_Log.error(lbl + "missing AB input ID");
						}

						// get the details hash (required for spends)
						Object det = iSpec.get("hash");
						if (det instanceof String) {
							String hash = (String) det;
							ci.m_DetailsHash = new String(hash);
						}
						else if (m_Opcode.equals(ClientMVOBlock.M_OpSpend)) {
							ret = false;
							m_Log.error(lbl + "missing AB input details hash");
						}

						// get the asset (required for withdraws)
						Object ast = iSpec.get("asset");
						if (ast instanceof String) {
							String asset = (String) ast;
							ci.m_InputAsset = new String(asset);
						}
						else if (m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
							ret = false;
							m_Log.error(lbl + "missing AB input asset");
						}

						// get the amount (required for withdraws)
						Object quan = iSpec.get("amount");
						if (quan instanceof String) {
							String quantity = (String) quan;
							try {
								ci.m_InputAmount = new BigInteger(quantity);
							}
							catch (NumberFormatException nfe) {
								m_Log.error(lbl + "illegal AB "
											+ "input amount, " + quantity, nfe);
								ret = false;
							}
						}
						else if (m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
							ret = false;
							m_Log.error(lbl + "missing AB input amount");
						}

						// on a withdraw, optional field is randomizer value
						if (m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
							Object rnd = iSpec.get("rand");
							if (rnd instanceof String) {
								String rand = (String) rnd;
								ci.m_InputRand = new String(rand);
							}
						}
					}
					else {
						ret = false;
						m_Log.error(lbl + "AB input " + key + " was not a Map");
					}
					if (ret) addInput(ci);
				}
				else {
					ret = false;
					m_Log.error(lbl + "AB input " + (iii+1)
								+ " was not a Map");
				}
				if (!ret) break;
			}
		}
		else if (!m_Opcode.equals(ClientMVOBlock.M_OpDeposit)) {
			ret = false;
			m_Log.error(lbl + "missing AB inputs[]");
		}

		// get outputs[] (required except for withdraw where optional)
		Object outs = mvoSigned.get("outputs");
		if (outs instanceof Object[]) {
			Object[] outputs = (Object[]) outs;
			if (outputs.length < 1
				&& !m_Opcode.equals(ClientMVOBlock.M_OpWithdraw))
			{
				ret = false;
				m_Log.error(lbl + "missing AB outputs[]");
			}
			for (int jjj = 0; jjj < outputs.length; jjj++) {
				Object out = outputs[jjj];
				if (out instanceof Map) {
					Map output = (Map) out;
					// get output index
					long idx = jjj + 1;
					String key = nf.format(idx);
					Object outSpec = output.get(key);
					if (outSpec instanceof Map) {
						Map outputSpec = (Map) outSpec;
						ClientPayee cp = new ClientPayee();
						cp.m_Payee = new String(key);

						// get eNFT minting address
						Object adr = outputSpec.get("address");
						if (adr instanceof String) {
							String addr = (String) adr;
							cp.m_Address = new String(addr);
						}
						// implicit on a change eNFT for a withdrawal
						else if (!m_Opcode.equals(ClientMVOBlock.M_OpWithdraw))
						{
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing address");
						}

						// get eNFT ID
						Object ido = outputSpec.get("id");
						if (ido instanceof String) {
							String id = (String) ido;
							cp.m_ID = new String(id);
						}
						else {
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing ID");
						}

						// get randomizer value (optional)
						Object rnd = outputSpec.get("rand");
						if (rnd instanceof String) {
							String rand = (String) rnd;
							cp.m_Rand = new String(rand);
						}

						// get eNFT asset
						Object ast = outputSpec.get("asset");
						if (ast instanceof String) {
							String asset = (String) ast;
							cp.m_OutputAsset = new String(asset);
						}
						else {
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing asset");
						}

						// get eNFT amount
						Object eamt = outputSpec.get("amount");
						if (eamt instanceof String) {
							String amount = (String) eamt;
							try {
								cp.m_OutputAmount = new BigInteger(amount);
							}
							catch (NumberFormatException nfe) {
								ret = false;
								m_Log.error(lbl + "AB output "
											+ key + " illegal amount, "
											+ amount, nfe);
							}
						}
						else {
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing amount");
						}

						// get the details hash
						Object det = outputSpec.get("hash");
						if (det instanceof String) {
							String hash = (String) det;
							cp.m_DetailsHash = new String(hash);
						}
						// implicit on a withdraw (calculated)
						else if (!m_Opcode.equals(ClientMVOBlock.M_OpWithdraw))
						{
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing details hash");
						}

						// get the metadata for the eNFT
						Object meta = outputSpec.get("metadata");
						if (meta instanceof Map) {
							Map mData = (Map) meta;

							// parse the JSON for the eNFT so we can use it
							cp.m_Metadata = new NFTmetadata(m_Log);
							if (!cp.m_Metadata.buildFromMap(mData)) {
								ret = false;
								m_Log.error(lbl + "AB output " + key
											+ " has invalid metadata");
							}

							// also set the metadata text (encrypted)
							StringBuilder metaStr = new StringBuilder(1024);
							cp.m_Metadata.addJSON(metaStr);
							cp.m_EncMetadata = metaStr.toString();
						}
						else {
							ret = false;
							m_Log.error(lbl + "AB output " + key
										+ " missing metadata");
						}
						if (ret) addPayee(cp);
					}
					else {
						ret = false;
						m_Log.error(lbl + "AB output " + key
									+ " was not a Map");
					}
				}
				else {
					ret = false;
					m_Log.error(lbl + "AB output " + (jjj+1)
								+ " was not a Map");
				}
				if (!ret) break;
			}
		}
		else if (!m_Opcode.equals(ClientMVOBlock.M_OpWithdraw)) {
			ret = false;
			m_Log.error(lbl + "missing AB outputs[]");
		}

		// get signatures[] (should have at least one if we're parsing)
		Object sigs = block.get("signatures");
		if (sigs instanceof Object[]) {
			Object[] signatures = (Object[]) sigs;
			if (signatures.length < 1) {
				ret = false;
				m_Log.error(lbl + "missing AB signatures[]");
			}
			for (int kkk = 0; kkk < signatures.length; kkk++) {
				// get output index
				long idx = kkk + 1;
				String key = nf.format(idx);
				Object sigSpec = signatures[kkk];
				if (sigSpec instanceof Map) {
					Map signature = (Map) sigSpec;
					Object signatureSpec = signature.get(key);
					if (!(signatureSpec instanceof Map)) {
						ret = false;
						m_Log.error(lbl + "AB sig " + key + " was not a Map");
						break;
					}
					signature = (Map) signatureSpec;
					MVOSignature sig = new MVOSignature();
					sig.m_Sequence = new String(key);

					// get MVO signer ID
					Object signer = signature.get("signer");
					if (signer instanceof String) {
						String mvoID = (String) signer;
						sig.m_Signer = new String(mvoID);
					}
					else {
						ret = false;
						m_Log.error(lbl + "AB sig " + key
									+ " missing signer");
					}

					// get actual signature on entire block
					Object mSig = signature.get("sig");
					if (mSig instanceof String) {
						String mvoSig = (String) mSig;
						sig.m_Signature = new String(mvoSig);
					}
					else {
						ret = false;
						m_Log.error(lbl + "AB sig " + key
									+ " missing signature");
					}

					// get args hash signature
					Object aSig = signature.get("argsSig");
					if (aSig instanceof String) {
						String argSig = (String) aSig;
						sig.m_ArgsSig = new String(argSig);
					}
					else {
						ret = false;
						m_Log.error(lbl + "AB sig " + key
									+ " missing argsSig");
					}
					if (ret) addSignature(sig);
				}
				else {
					ret = false;
					m_Log.error(lbl + "AB sig " + idx + " was not a Map");
				}
				if (!ret) break;
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing AB signatures[]");
		}

		// if all is okay, set the value of the arguments to be hashed
		if (ret) {
			m_ArgsBlock.m_ArgumentsData = m_ArgsBlock.buildArgsData();
			// confirm that this hashes to the expected value
			String aHash = m_ArgsBlock.argsHash();
			if (!m_ArgsBlock.m_ArgsHash.equals(aHash)) {
				m_Log.error(lbl + "AB args hash does not match parse result");
				ret = false;
			}
		}

		return ret;
	}

	/**
	 * method to build entire block from a JSON string (used by committee MVOs)
	 * @param req the request data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String req) {
		final String lbl
			= this.getClass().getSimpleName() + ".buildFromString: ";
		if (req == null || req.isEmpty()) {
			m_Log.error(lbl + "missing AB request data");
			return false;
		}
		Object jReq = null;
		boolean ret = false;
		try {
			jReq = JSON.parse(req);
		}
		catch (IllegalStateException ise) {
			m_Log.error(lbl + "AB JSON did not parse: \"" + req + "\"");
			return ret;
		}

		// parsed object should consist of a Map
		if (!(jReq instanceof Map)) {
			m_Log.error(lbl + "AB JSON was not a Map");
			return ret;
		}
		Map blockMap = (Map) jReq;
		ret = buildFromMap(blockMap);
		return ret;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Payees != null) {
				m_Payees.clear();
			}
			if (m_Inputs != null) {
				m_Inputs.clear();
			}
			if (m_Signatures != null) {
				m_Signatures.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
