/*
 * last modified---
 * 	03-08-24 try additional AUDs as key servers if we get a failure
 * 	06-29-23 getABI() moved to SmartContractConfig
 * 	06-12-23 remove standalone mode requirement
 * 	03-08-23 break out and go to successful reply if no eNFTs are found; check
 * 			 sigs on eNFTs and return only those which are valid
 * 	03-02-23 validate addresses, check and use real signatures
 * 	08-05-22 code for async key server accesses
 * 	07-20-22 don't fail on keys not found due to eNFT from foreign chain
 * 	07-08-22 complete debugging, implement input ID restrictions
 * 	06-24-22 first implementation draft
 * 	04-07-22 new (placeholder)
 *
 * purpose---
 * 	state machine transition object for handling dApp wallet download requests
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.MVOWalletBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.ClientWalletBlock;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.eNFTmetadata;
import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.WalletUtils;

import java.util.HashMap;
import java.util.Map;
import java.util.ArrayList;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.Base64;
import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.text.NumberFormat;
import javax.servlet.AsyncContext;
import javax.servlet.http.HttpServletResponse;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;


/**
 * This class provides the implementation of control logic to handle the various
 * stages required to perform a lookup of eNFTs and their keys owned by a
 * wallet. It also does error handling and recovery for any problems
 * encountered at each stage.
 */
public final class WalletState extends ReqSvcState {
	// BEGIN data members
	// aliases for steps in the deposit process
	/**
	 * request received from dApp (parsed okay, sig verified)
	 */
	public static final int M_ReqReceived = 1;
	/**
	 * eNFTs retrieved via ABI Web3j node, confirmed to belong to requestor
	 */
	public static final int M_GoteNFTs = 2;
	/**
	 * keys obtained from an Auditor for eNFT decryption
	 */
	public static final int M_GotKeys = 3;
	/**
	 * decrypts complete and output constructed
	 */
	public static final int	M_WalletDone = 4;
	/**
	 * reply sent to dApp
	 */
	public static final int M_Replied = 5;
	/**
	 * processing is done, clean up
	 */
	public static final int M_Completed = 6;

	/**
	 * the reply object we'll build, containing all eNFTs and keys in wallet
	 */
	private MVOWalletBlock	m_WalletReply;

	/**
	 * the NFTs found, while still encrypted, indexed by their IDs
	 */
	private HashMap<String, NFTmetadata>	m_RetrievedNFTs;

	/**
	 * local shorthand copy of MVO's logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param brokerEnt the parent object to which we are attached
	 */
	public WalletState(MVOBrokerEntry brokerEnt) {
		super(brokerEnt);
		m_FinalStep = M_Completed;
		m_Log = m_Parent.getLog();
		m_WalletReply = new MVOWalletBlock(m_Log);
		m_RetrievedNFTs = new HashMap<String, NFTmetadata>();
	}

	/**
	 * method to advance the state (required to extend ReqSvcState)
	 * @param last the last step we attempted to perform (the from state)
	 * @param next the step number to go to
	 * @param reason the success/failure code causing us to go to this event
	 * (if an error code, the error will have occurred doing the previous step)
	 * @param errMsg a text error message describing the failure, or empty
	 * @return whether the state advance worked successfully (can be true even
	 * if handling an error reason code)
	 */
	public boolean advanceState(int last, int next, int reason, String errMsg) {
		final String lbl = this.getClass().getSimpleName() + ".advanceState: ";
	/*
		m_Log.debug(lbl + "transition from " + last + " to " + next
					+ ", reason " + reason + ", emsg \"" + errMsg + "\"");
	 */

		// examine inputs
		if (next > m_FinalStep) {
			m_Log.error(lbl + "invalid next step number: " + next);
			return false;
		}
		if (next < last) {
			m_Log.error(lbl + "reverse state transition from " + next
						+ " to " + last);
			return false;
		}
		if (reason < 0) {
			m_Log.error(lbl + "invalid step advance reason: " + reason);
			return false;
		}
		m_PrevStep = last;
		m_CurrStep = next;
		m_FailureMode = reason;
		if (errMsg != null) {
			m_FailureMsg = errMsg;
		}

		// local convenience definitions
		MVO mvo = m_Parent.getMVO();
		MVOState stObj = mvo.getStateObj();
		AsyncContext rep = m_Parent.getDappReply();
		if (rep == null) {
			m_Log.error(lbl + "lead without an AsyncContext, abort");
			return false;
		}
		ReceiptHandler rHandler = mvo.getReceiptHandler();

		// handle errors, based on the 'last' step completed
		boolean ret = true;
		switch (reason) {
			case M_InvalidReq:
				// context: M_ReqReceived, bad request seen
				if (m_PrevStep != M_ReqReceived) {
					// nack with state error
					ret = inconsistentState(rep);
				}
				else {
					// nack original request
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_REQUEST,
								   "parse error: " + m_FailureMsg);
				}
				break;

			// we could not access the appropriate blockchain
			case M_UnavailABI:
				// context: M_ReqReceived --> M_GoteNFTs
				if (m_PrevStep == M_ReqReceived) {
					// couldn't obtain eNFTs from blockchain's event log
					ret = nackDapp(rep, HttpServletResponse.SC_BAD_GATEWAY,
								   "blockchain access error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep);
				}
				break;

			// we could not reach an Auditor (for decryption keys)
			case M_UnavailAud:
				// context: M_GoteNFTs --> M_GotKeys
				if (m_PrevStep == M_GoteNFTs) {
					// failed to get eNFT keys; nack original request
					ret = nackDapp(rep, HttpServletResponse.SC_GATEWAY_TIMEOUT,
								   "key server error: " + m_FailureMsg);
				}
				else {
					// nack with state error
					ret = inconsistentState(rep);
				}
				break;

			case M_ProcError:
				/* Internal processing error, theoretically valid for any step.
				 * Unless we have some kind of coding error, these should not
				 * occur. To avoid the possibility of a recursive endless loop,
				 * we always bail with a nack when one of these crops up.
				 */
				String fail = "";
				int rCode = HttpServletResponse.SC_INTERNAL_SERVER_ERROR;
				switch (m_PrevStep) {
					case M_ReqReceived:
						fail = "request processing error: " + m_FailureMsg;
						rCode = HttpServletResponse.SC_BAD_REQUEST;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_GoteNFTs:
						fail = "problem retrieving eNFTs: " + m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_GotKeys:
						fail = "key server access problem: " + m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_WalletDone:
						fail = "wallet decryption or construction problem: "
								+ m_FailureMsg;
						ret = nackDapp(rep, rCode, fail);
						break;
					
					case M_Replied:
						fail = "reply send problem: " + m_FailureMsg;
						// this will probably fail too of course:
						ret = nackDapp(rep, rCode, fail);
						break;

					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep);
						break;
				}
				m_Log.error(lbl + "processing error: " + fail);
				break;

			case M_ParseError:
				/* Partners can send us bogosity that didn't parse.  If they do,
				 * we have to bail because resending will probably generate the
				 * same bogosity again.  (This condition probably indicates a
				 * coding error in a partner server.)
				 */
				switch (m_PrevStep) {
					case M_ReqReceived:
					case M_GoteNFTs:
					case M_GotKeys:
						m_Log.error(lbl + "parse error seen from "
									+ "partner, " + m_FailureMsg);
						ret = nackDapp(rep,
								HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
								"error processing partner reply, aborting: "
								+ m_FailureMsg);
						break;

					case M_WalletDone:
					case M_Replied:
					case M_Completed:
					default:
						// should be impossible
						ret = inconsistentState(rep);
						break;
				}
				break;

			// partners can of course also send us error replies
			case M_GotNack:
				String nack = "";
				int nCode = HttpServletResponse.SC_BAD_GATEWAY;
				switch (m_PrevStep) {
					case M_GoteNFTs:
						nack = "nack obtaining eNFTs, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackDapp(rep, nCode, nack);
						break;

					case M_GotKeys:
						nack = "nack obtaining eNFT keys, " + m_FailureMsg;
						m_Log.error(lbl + nack);
						ret = nackDapp(rep, nCode, nack);
						break;

					// no peer replies expected in these states
					case M_ReqReceived:
					case M_WalletDone:
					case M_Replied:
					case M_Completed:
					default:
						// should be impossible
						m_Log.error(lbl + "unexpected reply parse error in "
									+ "state " + m_CurrStep + ", "
									+ m_FailureMsg);
						ret = inconsistentState(rep);
						break;
				}
				break;

			case M_NoFailure:
				// fall through below
				break;

			default:
				// should be impossible
				break;
		}
		if (reason != M_NoFailure) return ret;

		// handle normal (non-error) cases
		ret = true;
		m_FailureMode = M_NoFailure;
		ClientRequest orgReq = m_Parent.getDappRequest();
		ClientWalletBlock clientReq = null;
		SmartContractConfig scc = null;
		MVOConfig mvoConf = mvo.getConfig();
		switch (m_CurrStep) {
			case M_GoteNFTs:
				/* Here we have successfully parsed and signature-verified the
				 * request from the client.  Our next step is to do basic
				 * sanity-checking.  If any problems are found, we nack request.
				 */
				if (m_Achieved < M_ReqReceived) {
					m_Log.error(lbl + "verify requested without valid "
								+ "request received");
					ret = inconsistentState(rep);
					break;
				}

				// get the request as passed by dApp client
				if (!(orgReq instanceof ClientWalletBlock)) {
					m_FailureMsg = "client request is not a ClientWalletBlock";
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}
				clientReq = (ClientWalletBlock) orgReq;
				// get owner wallet without leading 0x
				String walletAcct = clientReq.getSender();
				if (!WalletUtils.isValidAddress(walletAcct)) {
					m_FailureMsg
						= "invalid client wallet address, " + walletAcct;
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// obtain SC config
				scc = mvo.getSCConfig(clientReq.getChainId());
				if (scc == null) {
					// unknown chain
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " is not supported";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_InvalidReq,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// confirm availability of RPC-JSON client
				if (!scc.isEnabled()) {
					// Web3j URL must be offline or not reachable
					m_FailureMsg = "chain Id " + clientReq.getChainId()
									+ " support is not currently available";
					m_Log.error(lbl + "req verify failure: " + m_FailureMsg);
					advanceState(m_PrevStep, m_CurrStep, M_UnavailABI,
								 m_FailureMsg);
					ret = false;
					break;
				}

				boolean gotNFTerr = false;
				BlockchainAPI web3j = mvo.getWeb3(scc.getChainId());
				Future<HashMap<String, NFTmetadata>> nftFuts
					= web3j.getAccountNFTs(scc.getChainId(),
										   scc.getABI(),
										   walletAcct,
										   false);
				HashMap<String, NFTmetadata> nfts = null;
				try {
					nfts = nftFuts.get();
					if (!nfts.isEmpty()) {
						// record (we'll decrypt in next step after keys)
						ArrayList<String> reqIDs = clientReq.getIDs();
						if (reqIDs.isEmpty()) {
							// return all found
							m_RetrievedNFTs.putAll(nfts);
						}
						else {
							for (String Id : reqIDs) {
								m_RetrievedNFTs.put(Id, nfts.get(Id));
							}
						}
					}
				}
				catch (InterruptedException | ExecutionException
						 | CancellationException e)
				{
					m_FailureMsg = "exception listing eNFTs on chain";
					m_Log.error(lbl + "exception checking for eNFTs for "
								+ "acct " + walletAcct + ": "
								+ e.getMessage());
					gotNFTerr = true;
					//scc.setEnabled(false);
				}
				if (gotNFTerr) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// this completes this step; record success and move on
				recordSuccess(M_GoteNFTs);

				// if we have no eNFTs, we are done; go to success return
				if (m_RetrievedNFTs.isEmpty()) {
					// not an error (no eNFTs in wallet is legal)
					recordSuccess(M_GotKeys);
					ret = advanceState(m_CurrStep, M_WalletDone, M_NoFailure,
										"");
				}
				else {
					ret = advanceState(m_CurrStep, M_GotKeys, M_NoFailure, "");
				}
				break;

			case M_GotKeys:
				/* At this point we have all eNFTs included in the wallet.  We
				 * now need to obtain the AES-256 encryption key from the key
				 * server (any Auditor node) for each ID found in the list.
				 * To do this, we pick a random Auditor and compute:
				 * chainId+ID+owner_address and add it to the fetch request.
				 */
				if (!m_ExpectingKeyServerReply) {
					String randAud = mvo.getRandomAuditor(null);
					if (randAud.isEmpty()) {
						// no Auditors are available
						m_FailureMsg = "no key servers available";
						m_Log.error(lbl + "get eNFT keys failure, no Auditors");
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// now build the key block to send to the Auditor
					MVOKeyBlock nftKeyBlock = new MVOKeyBlock(m_Log);
					nftKeyBlock.setMapping(nftKeyBlock.M_MapENFT);
					nftKeyBlock.setOpcode(nftKeyBlock.M_OpGet);

					// get owner wallet as a BigInteger (without leading 0x)
					clientReq = (ClientWalletBlock) orgReq;
					// must supply the user's orig req
					nftKeyBlock.setClientBlockJson(
											clientReq.getDecryptedPayload());
					String ownerAddr = clientReq.getSender();

					// for each eNFT, compute chainId+ID+owner and request key
					ArrayList<String> hashComps = new ArrayList<String>(3);
					for (String nftID : m_RetrievedNFTs.keySet()) {
						hashComps.add(Long.toString(clientReq.getChainId()));
						hashComps.add(nftID);
						hashComps.add(ownerAddr.toLowerCase());
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						nftKeyBlock.addHashToList(keyHash);
						hashComps.clear();
					}

					// sign the MVOKeyBlock
					MVOSignature keySig = new MVOSignature();
					keySig.m_Signer = mvo.getMVOId();
					PrivateKey sigKy = mvoConf.getCommPrivKey();
					String sData = nftKeyBlock.buildSignedData();
					boolean sOk = true;
					if (nftKeyBlock.getErrCode() != 0) {
						sOk = false;
						m_Log.error(lbl + "could not build MVOKeyBlock "
									+ "signed data, no sig");
					}
					else {
						String sig = EncodingUtils.signStr(sigKy, sData);
						if (sig == null) {
							sOk = false;
							m_Log.error(lbl + "could not sign MVOKeyBlock");
						}
						else {
							keySig.m_Signature = sig;
						}
					}
					if (!sOk) {
						m_FailureMsg
							= "could not build MVOKeyBlock for Auditor";
						advanceState(m_PrevStep, m_CurrStep, M_ProcError,
									 m_FailureMsg);
						ret = false;
						break;
					}
					nftKeyBlock.setSignature(keySig);

					/* repeat until we either succeed sending key block or run
					 * out of available Auditors we haven't tried yet
					 */
					boolean sentKeyBlk = false;
					ArrayList<String> excludeAIds = new ArrayList<String>();
					while (!sentKeyBlk) {
						// send the key block to the selected Auditor
						if (!m_Parent.sendKeyServerReq(randAud, nftKeyBlock)) {
							m_Log.error(lbl + "key lookup failure, Auditor "
										+ randAud + " could not be reached");
							// get another different AUD Id
							excludeAIds.add(randAud);
							randAud = mvo.getRandomAuditor(excludeAIds);
							if (randAud.isEmpty()) {
								// there are no untried auditors left; fail
								break;	// while
							}
						}
						else {
							sentKeyBlk = true;
						}
					}

					// check results
					if (!sentKeyBlk) {
						m_FailureMsg = "error getting eNFT decryption keys";
						advanceState(m_PrevStep, m_CurrStep, M_UnavailAud,
									 m_FailureMsg);
						ret = false;
						break;
					}
					m_ExpectingKeyServerReply = true;
					// wait for async response (will hit else{} below)
					return ret;
				}
				else {	// processing key server response
					// verify Auditor's signature
					m_ExpectingKeyServerReply = false;
					String sigAudData = m_KeyResponse.buildSignedData();
					AuditorKeyBlock.AuditorSignature adSig
						= m_KeyResponse.getSignature();
					boolean sigVer = true;
					PublicKey adKey = mvoConf.getPeerPubkey(adSig.m_Signer);
					if (adKey == null) {
						sigVer = false;
						m_Log.error(lbl + "no pubkey for Auditor "
									+ adSig.m_Signer + ", cannot verify sig on "
									+ "returned AuditorKeyBlock");
					}
					else {
						if (!EncodingUtils.verifySignedStr(adKey,
														   sigAudData,
														   adSig.m_Signature))
						{
							sigVer = false;
							m_Log.error(lbl + "sig verify failed for Aud "
										+ adSig.m_Signer
										+ ", cannot verify sig on "
										+ "returned AuditorKeyBlock");
						}
					}
					if (!sigVer) {
						m_FailureMsg = "error looking up eNFT AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}

					// verify the key server didn't return an error
					String errStat = m_KeyResponse.getStatus();
					if (!m_KeyResponse.M_Success.equals(errStat)) {
						m_Log.error(lbl + "Aud " + adSig.m_Signer
									+ " returned error: " + errStat);
						m_FailureMsg = "error looking up eNFT AES keys";
						advanceState(m_PrevStep, m_CurrStep, M_GotNack,
									 m_FailureMsg);
						ret = false;
						break;
					}
				}

				// we should have all the keys, move on to next state
				recordSuccess(M_GotKeys);
				ret = advanceState(m_CurrStep, M_WalletDone, M_NoFailure, "");
				break;

			case M_WalletDone:
				// for each eNFT, decrypt it and add its POJO to the output
				NumberFormat nf = NumberFormat.getNumberInstance();
				nf.setMinimumIntegerDigits(3);
				int nftIdx = 1;
				clientReq = (ClientWalletBlock) orgReq;
				if (clientReq == null
					|| (!m_RetrievedNFTs.isEmpty() && m_KeyResponse == null))
				{
					m_FailureMsg = "processing error, no keys found";
					m_Log.error(lbl + "missing client request and/or "
								+ "eNFT AES keys, cannot build wallet reply");
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				scc = mvo.getSCConfig(clientReq.getChainId());
				boolean allOk = true;
				ArrayList<String> hashParts = new ArrayList<String>(3);
				for (String nftID : m_RetrievedNFTs.keySet()) {
					MVOWalletBlock.WalletSpec walletEntry
						= m_WalletReply.new WalletSpec();
					String sequence = "eNFT" + nf.format(nftIdx++);
					walletEntry.m_Sequence = new String(sequence);
					hashParts.add(Long.toString(clientReq.getChainId()));
					hashParts.add(nftID);
					hashParts.add(clientReq.getSender().toLowerCase());
					String keyIdx = String.join(M_JoinChar, hashParts);
					String keyHash = EncodingUtils.sha3(keyIdx);
					hashParts.clear();

					// find the key in the list by looking up the hash
					AuditorKeyBlock.KeySpec keySpec
						= m_KeyResponse.getKeyForHash(keyHash);
					if (keySpec != null) {
						walletEntry.m_Key = keySpec.m_KeyData;
					}
					if (walletEntry.m_Key.isEmpty()) {
						/* In actual usage, we pass getAccountNFTs() the ABI
						 * URL for the real blockchain, and so if we get back
						 * an eNFT for which we don't have a key, and we're
						 * not looking at deminted ones, this is an error.
						 * But we can't treat this as a fatal error; simply
						 * go on.  Otherwise a single missing key would make
						 * fetching any wallet eNFTs impossible.
						 */
						m_FailureMsg = "processing error, missing eNFT key(s)";
						m_Log.error(lbl + "missing eNFT AES key for "
									+ "eNFT hash index " + keyIdx
									+ ", cannot build complete wallet reply");
						allOk = false;
						continue;
					}

					// make key from this key data
					Base64.Decoder b64d = Base64.getUrlDecoder();
					byte[] keyData = b64d.decode(walletEntry.m_Key);
					SecretKey secretKey = new SecretKeySpec(keyData, "AES");

					// decrypt the eNFT using this key
					NFTmetadata nft = m_RetrievedNFTs.get(nftID);
					eNFTmetadata eNft = nft.getProperties();
					String enshrouded = eNft.getEnshrouded();
					if (!eNft.isEncrypted() || enshrouded.isEmpty()) {
						m_Log.error(lbl + "weird, eNFT Id " + nftID
									+ " does not appear encrypted");
					}
					else {
						String decNFT = EncodingUtils.decWithAES(secretKey,
																 enshrouded);
						if (decNFT == null || decNFT.isEmpty()) {
							m_Log.error(lbl + "could not decrypt eNFT ID "
										+ nftID);
							m_FailureMsg = "processing error decrypting eNFTs";
							allOk = false;
							break;
						}
						String decMap = "{" + decNFT + "}";
						// parse this as JSON
						Object parsed = null;
						try {
							parsed = JSON.parse(decMap);
						}
						catch (IllegalStateException ise) {
							m_FailureMsg = "processing error parsing eNFTs";
							m_Log.error(lbl + "could not parse decrypted "
										+ "eNFT ID " + nftID, ise);
							allOk = false;
							break;
						}
						if (parsed instanceof Map) {
							Map eNFTMap = (Map) parsed;
							if (!eNft.buildFromMap(eNFTMap)) {
								m_FailureMsg = "processing error parsing eNFTs";
								m_Log.error(lbl + "could not parse Map for "
											+ "decrypted eNFT ID " + nftID);
								allOk = false;
								break;
							}
						}
						else {
							m_FailureMsg = "processing error parsing eNFTs";
							m_Log.error(lbl + "could not parse decrypted "
										+ "eNFT ID " + nftID);
							allOk = false;
							break;
						}

						// get signing key for signing MVO
						String eNFTsigner = eNft.getSigner();
						MVOConfig signerConf
							= (MVOConfig) scc.getMVOMap().get(eNFTsigner);
						if (signerConf == null) {
							m_FailureMsg = "error validating eNFT sig";
							m_Log.error(lbl + "cannot find config for "
										+ "eNFT signer, " + eNFTsigner);
							allOk = false;
							break;
						}
						BlockchainConfig signerChConf
							= signerConf.getChainConfig(clientReq.getChainId());
						if (signerChConf == null) {
							m_FailureMsg = "error validating eNFT sig";
							m_Log.error(lbl + "cannot find chain config for "
										+ "eNFT signer " + eNFTsigner
										+ " on chainId "
										+ clientReq.getChainId());
							allOk = false;
							break;
						}
						String signerAddr
							= signerChConf.getSigningAddress();

						// verify the signature on the eNFT
						if (!validateENFTsig(eNft, signerAddr)) {
							m_FailureMsg = "signature did not validate on "
											+ "decrypted eNFT";
							m_Log.error(lbl + "CRITICAL - sig verification "
										+ "failed on decrypted eNFT ID "
										+ nftID);
						}
						else {
							// add only valid ones to output list
							walletEntry.m_eNFTmetadata = eNft;
							m_WalletReply.addToWallet(walletEntry);
						}
					}
				}
				if (!allOk) {
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}

				// affix our signature to the block
				String sigData = m_WalletReply.buildSignedData();
				// get our signing key for this chain
				scc = mvo.getSCConfig(clientReq.getChainId());
				BlockchainConfig bConfig
					= mvoConf.getChainConfig(scc.getChainId());
				if (bConfig == null) {
					// shouldn't occur!
					m_Log.error(lbl + "no MVO config for chain "
								+ scc.getChainName() + ", cannot "
								+ "sign MVOWalletBlock");
					m_FailureMsg = "could not sign wallet reply block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				BigInteger bcSigKey = bConfig.getSigningKey();
				MVOSignature walletSig = new MVOSignature();
				walletSig.m_Signer = mvo.getMVOId();
				String ourSig = EncodingUtils.signData(sigData, bcSigKey);
				if (ourSig == null) {
					m_Log.error(lbl + "error signing receipt reply block");
					m_FailureMsg = "could not sign wallet reply block";
					advanceState(m_PrevStep, m_CurrStep, M_ProcError,
								 m_FailureMsg);
					ret = false;
					break;
				}
				walletSig.m_Signature = ourSig;
				m_WalletReply.setSignature(walletSig);

				// send back the reply to the dApp client
				recordSuccess(M_WalletDone);
				ret = advanceState(m_CurrStep, M_Replied, M_NoFailure, "");
				break;

			case M_Replied:
				/* Send our reply to the original client request.  This
				 * includes the MVOWalletBlock we created, with our signature.
				 */
				clientReq = (ClientWalletBlock) orgReq;
				HttpServletResponse httpRep
					= (HttpServletResponse) rep.getResponse();
				if (!rHandler.sendNormalRep(httpRep, m_WalletReply,
											clientReq.getReplyKey()))
				{
					m_Log.error(lbl + "unable to send client reply after"
								+ " successful completion");
					rep.complete();
					ret = false;
					advanceState(m_CurrStep, m_FinalStep, M_ProcError,
								"client reply failure");
				}
				else {
					// normal progression to next step
					rep.complete();
					recordSuccess(M_Replied);
					ret = advanceState(m_CurrStep, m_FinalStep, M_NoFailure,
										"");
				}
				break;

			case M_Completed:
				/* Here all of our tasks have been completed, and the last step
				 * is to clean up data structures, removing temporary data.
				 * Garbage collection and finalize() calls will auto-trigger
				 * the next time System.gc() runs.
				 */
				// remove ourselves from the broker map if we are in it
				Map<String, MVOBrokerEntry> brokerMap = stObj.getBrokerMap();
				String tId = m_Parent.getTransId();
				if (tId != null && !tId.isEmpty()) {
					brokerMap.remove(tId, m_Parent);
					m_Parent = null;
				}

				// if we have a wallet reply built, forget the eNFTs
				m_RetrievedNFTs.clear();
				ArrayList<MVOWalletBlock.WalletSpec> walletEntries
					= m_WalletReply.getWallet();
				walletEntries.clear();
				m_KeyResponse = null;
				break;

			default:
				// all other states don't make sense
				m_Log.error(lbl + "illegal state transition to " + m_CurrStep);
				ret = inconsistentState(rep);
				break;
		}
		return ret;
	}

	/**
	 * obtain the wallet reply
	 * @return the list of wallet items, including keys
	 */
	public MVOWalletBlock getWalletReply() { return m_WalletReply; }

	/**
	 * method to send pending error reply to dApp via ReceiptHandler
	 * @param resp the dApp response object
	 * @param sc the SC_* error code
	 * @param errTxt the text to go with the code
	 * @return true on successful error reply send, false on failure
	 */
	private boolean nackDapp(AsyncContext resp, int sc, String errTxt) {
		if (resp != null) {
			HttpServletResponse httpRep
				= (HttpServletResponse) resp.getResponse();
			m_Parent.getMVO().getReceiptHandler().sendErrRep(httpRep,
															sc, errTxt);
			resp.complete();
			recordSuccess(M_Replied);
		}

		// proceed to cleanup
		return advanceState(M_Replied, m_FinalStep, M_NoFailure, "");
	}

	/**
	 * Method to log when reason versus state is inconsistent.  Note this
	 * implies an internal programming error. The dApp client is nack'd.
	 * @param rep the dApp response object
	 * @return true on success
	 */
	private boolean inconsistentState(AsyncContext rep) {
		m_Log.error("WalletState internal inconsistency, step " + m_CurrStep
					+ ", reason " + m_FailureMode + ", emsg \"" + m_FailureMsg
					+ "\"");
		return nackDapp(rep, HttpServletResponse.SC_INTERNAL_SERVER_ERROR,
					    "Internal state error, abort");
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_RetrievedNFTs != null) {
				m_RetrievedNFTs.clear();
			}
			m_WalletReply = null;
		} finally {
			super.finalize();
		}
	}

	// END methods
}
