/*
 * last modified---
 * 	07-16-25 in getConfig(), use VPNEVEN/VPNODD to compute, with .props override
 * 	03-18-24 in getMVOConfig(), save active value in own MVOConfig record
 * 	09-27-23 ensure we record EIP-55 format addresses
 * 	06-20-23 add checkIfIdUsed(), pass acct to getNFTsById()
 * 	06-12-23 new
 *
 * purpose---
 * 	provide an actual remote blockchain implementation of BlockchainAPI
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.NFTmetadata;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.BlockchainAPI;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;
import cc.enshroud.jetty.wrappers.MVOStaking;

import org.web3j.crypto.WalletUtils;
import org.web3j.crypto.Credentials;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.CipherException;
import org.web3j.crypto.Keys;
import org.web3j.protocol.Web3j;
import org.web3j.utils.Numeric;
import org.web3j.tx.gas.DefaultGasProvider;
import org.web3j.tuples.generated.Tuple8;

import java.math.BigInteger;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Properties;
import java.util.Hashtable;
import java.util.HashMap;
import java.util.concurrent.Future;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.FileFilter;


/**
 * Implementation of BlockchainAPI (for Web3j functions) using the blockchain.
 * Iff the Web ABI URL for a chain in the Properties file is not local file://,
 * the methods of this class should be used.  Otherwise use LocalBlockchainAPI.
 */
public final class RemoteBlockchainAPI implements BlockchainAPI, FileFilter {
	// BEGIN data members
	/**
	 * owning MVO which creates us
	 */
	private MVO			m_MVO;

	/**
	 * the chain we support
	 */
	private long		m_ChainId;

	/**
	 * logging object (from owning MVO)
	 */
	private Log			m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which creates us
	 * @param chId the chainId per chainlist.org
	 */
	public RemoteBlockchainAPI(MVO mvo, long chId) {
		m_MVO = mvo;
		m_Log = mvo.log();
		m_ChainId = chId;
	}

	/**
	 * init handler
	 * @return true on success
	 */
	public boolean init() {
		// nothing to do here
		return true;
	}

	/**
	 * shutdown handler
	 * @return true on success
	 */
	public boolean shutdown() {
		// nothing to do here
		return true;
	}

	/**
	 * obtain chain we support
	 * @return supported chain Id
	 */
	public long getChainId() { return m_ChainId ; }


	// methods required to implement BlockchainAPI
	/**
	 * obtain the smart contract configuration for a given blockchain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiSession the Web3j object at which we can access the chain
	 * @return the configuration, with data supplied appropriately
	 */
	public Future<SmartContractConfig> getConfig(long chainId,
												 String chain,
												 Web3j abiSession)
	{
		final String lbl = this.getClass().getSimpleName() + ".getConfig: ";
		if (chainId <= 0L || chain == null || chain.isEmpty()
			|| abiSession == null)
		{
			m_Log.error(lbl + "missing input");
			return null;
		}
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}

		// init a new record to return
		CompletableFuture<SmartContractConfig> retFut
			= new CompletableFuture<SmartContractConfig>();
		SmartContractConfig scc
			= new SmartContractConfig(chain, chainId, m_Log, m_MVO);
		Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();

		// whip up dummy creds (no gas required here)
		Credentials credentials = null;
		try {
			credentials = Credentials.create(Keys.createEcKeyPair());
		}
		catch (Exception ee) {
			m_Log.error(lbl + "exception generating dummy Credentials", ee);
			return null;
		}

		/* access the EnshroudProtocol contract to obtain these settings:
		 * 	scc.m_NumSigs 	(requiredSigs())
		 * 	scc.m_DwellTime	(confirmationBlocks())
		 * 	scc.m_BaseURI	(baseURI())
		 */
		String protocolAddr = EnshroudProtocol.getPreviouslyDeployedAddress(
														Long.toString(chainId));
		if (protocolAddr == null) {
			// contract has not been deployed here
			final String noDep = "EnshroudProtocol contract not deployed on "
								+ "chain Id " + chainId;
			Exception ee = new Exception(noDep);
			retFut.completeExceptionally(ee);
			return retFut;
		}
		EnshroudProtocol enshProtocol
			= EnshroudProtocol.load(protocolAddr, abiSession,
								  	credentials, new DefaultGasProvider());
		BigInteger numSigs = null;
		BigInteger delayBlocks = null;
		String baseURI = null;
		try {
			numSigs = enshProtocol.requiredSigs().send();
			delayBlocks = enshProtocol.confirmationBlocks().send();
			baseURI = enshProtocol.baseURI().send();
		}
		catch (Exception eee) {
			m_Log.error(lbl + "exception fetching EnshroudProtocol values");
			retFut.completeExceptionally(eee);
			return retFut;
		}
		scc.setNumSigs(numSigs.intValue());
		scc.setDwellTime(delayBlocks.intValue());
		scc.setBaseURI(baseURI);

		/* now access the MVOStaking contract to obtain this data:
		 * 1. Count of struct MVO{} records
		 * 2. List of MVOIds that support this contract
		 * 3. For each MVOId listed:
		 *  	stakingQuantum
         *  	active
		 *    We'll store this data in a MVOConfig inserted into scc.m_MVOs.
		 *    (Except for the MVOConfig for us, which is built separately.)
		 * 4. For each record where active=true, add its stakingQuantum to
		 * 	  running total to build scc.m_TotalStaked.  All of them go in
		 * 	  the list regardless of active, so we can check sigs on eNFTs.
		 */
		String mvoStakingAddr
			= MVOStaking.getPreviouslyDeployedAddress(Long.toString(chainId));
		if (mvoStakingAddr == null) {
			// contract has not been deployed here; check Eth mainnet
			mvoStakingAddr = MVOStaking.getPreviouslyDeployedAddress("1");
			if (mvoStakingAddr == null) {
				final String noDep = "MVOStaking contract not deployed on "
									+ "chain Id " + chainId + " or mainnet";
				Exception ee = new Exception(noDep);
				retFut.completeExceptionally(ee);
				return retFut;
			}
		}
		MVOStaking mvoStaking = MVOStaking.load(mvoStakingAddr,
												abiSession,
												credentials,
												new DefaultGasProvider());

		// get count of MVOs and list of MVOIds
		BigInteger numMVOs = null;
		int MVOcnt = 0;
		ArrayList<String> mvoIdList = null;
		try {
			numMVOs = mvoStaking.mvoIndex().send();
			if (numMVOs != null) {
				MVOcnt = numMVOs.intValue();
				mvoIdList = new ArrayList<String>(MVOcnt);
				for (int mmm = 0; mmm < MVOcnt; mmm++) {
					BigInteger idx = new BigInteger(Integer.toString(mmm));
					String mvoId = mvoStaking.mvoIds(idx).send();
					if (mvoId != null) {
						mvoIdList.add(mvoId);
					}
					else {
						m_Log.error(lbl + "no Id for MVO #" + mmm);
						throw new Exception("missing MVOId index " + mmm);
					}
				}
			}
			else {
				m_Log.error(lbl + "could not get MVO index");
				throw new Exception("could not get MVO count");
			}
		}
		catch (Exception eee) {
			retFut.completeExceptionally(eee);
			return retFut;
		}

		/* We utilize properties files to obtain the MVOURI values for MVOs.
		 * These act as overrides in cases where the computed URI is incorrect.
		 */
		String secDir = m_MVO.getProperties().getProperty("RunDir", "");
		StringBuilder keyPath = new StringBuilder(64);
		if (secDir.isEmpty()) {
			keyPath.append("security");
		}
		else {
			keyPath.append(secDir + File.separator + "security");
		}
		// find all files MVO-NNN.properties
		File securityDir = new File(keyPath.toString());
		File[] mvoConfigs = null;
		try {
			mvoConfigs = securityDir.listFiles((FileFilter)this);
		}
		catch (SecurityException se) {
			m_Log.error(lbl + "no permission to access peer properties");
			retFut.completeExceptionally(se);
			return retFut;
		}
		final int mvoConfCnt = (mvoConfigs == null ? 0 : mvoConfigs.length);
		if (mvoConfCnt == 0 || mvoConfCnt != MVOcnt) {
			m_Log.debug(lbl + "found " + MVOcnt + " MVOs on chain " + chain
						+ ", but " + mvoConfCnt + " local props files");
		}
		boolean propErr = false;

		// build running total of active stakings on this chain
		BigInteger totalStakings = BigInteger.ZERO;
		for (String mvoId : mvoIdList) {
			// return type corresponds to contract's struct MVO{}
			Tuple8<String,		// listenURI (not used by MVOs)
				   String,		// stakingAddress
				   String,		// signingAddress
				   String,		// encrPubkey (not used by MVOs)
				   BigInteger,	// stakingQuantum
				   BigInteger,	// memberPoolId (not used by MVOs)
				   BigInteger,	// rating (not yet used by MVOs)
				   Boolean>		// active
					mvoData = null;
			try {
				mvoData = mvoStaking.idToMVO(mvoId).send();
			}
			catch (Exception eee) {
				m_Log.error(lbl + "exception fetching MVOStaking record for "
							+ "MVOId " + mvoId);
			}
			finally {
				if (mvoData == null) {
					m_Log.error(lbl + "error fetching MVOStaking record for "
								+ "MVOId " + mvoId);
					continue;
				}
			}
			BigInteger stakingQuantum = mvoData.component5();
			Boolean stakingActive = mvoData.component8();
			if (stakingActive.booleanValue()) {
				// add into total
				totalStakings = totalStakings.add(stakingQuantum);
			}
			// else: we still need to record signingAddr for signature checks

			/* We must not build a MVOConfig if the MVOId is our own, because
			 * we use getMVOConfig() to do this for our own node.  All we
			 * needed to do was include our staking in the total computation.
			 */
			if (mvoId.equals(m_MVO.getMVOId())) {
				continue;
			}
			String stakingAddr = mvoData.component2();
			String signingAddr = mvoData.component3();
		/*
			m_Log.debug(lbl + "for mvoId " + mvoId + ", got stakingAddr = "
						+ stakingAddr + ", signingAddr = " + signingAddr
						+ ", stakingQuantum = " + stakingQuantum + ", active = "
						+ stakingActive);
		 */

			// create a record to store data about this MVO
			MVOConfig mConf = new MVOConfig(mvoId, m_Log);
			mConf.setStatus(stakingActive.booleanValue());
			mConf.addBlockchainConfig(chainId, chain);
			BlockchainConfig mbc = mConf.getChainConfig(chainId);
			mbc.configSigningAddress(Keys.toChecksumAddress(signingAddr));
			mbc.configStakingAddress(Keys.toChecksumAddress(stakingAddr));
			mbc.setStaking(stakingQuantum.toString());
			// insert this record into map for this chain
			mvoMap.put(mvoId, mConf);

			// compute the MVOURI value for this MVO
			URI calcURI = m_MVO.computePeerURI(mvoId, false);

			// also obtain the MVOURI value for this MVO from properties files
			File propFile = null;
			// find .properties for this MVO, if such a file exists
			for (int ppp = 0; ppp < mvoConfCnt; ppp++) {
				File pFile = mvoConfigs[ppp];
				String fName = pFile.getName();
				// get the MVOId from the filename
				int dot = fName.indexOf(".properties");
				if (fName.substring(0, dot).equals(mvoId)) {
					propFile = pFile;
					break;
				}
			}
			if (propFile != null && propFile.exists() && propFile.canRead()) {
				// read the file
				Properties mvoProps = new Properties();
				FileInputStream fis = null;
				try {
					fis = new FileInputStream(propFile);
					mvoProps.load(fis);
					fis.close();
				}
				catch (FileNotFoundException fnfe) {
					// (we already checked for this)
					propErr = true;
				}
				catch (IOException ioe) {
					m_Log.error(lbl + "error reading properties file, "
								+ propFile, ioe);
					propErr = true;
				}

				// record the data from the file in the config, and add to list
				String mvoURI = mvoProps.getProperty("MVOURI");
				URI uri = null;
				try {
					uri = new URI(mvoURI);
					m_Log.debug(lbl + "found file URI override for MVOId "
								+ mvoId + ": " + uri);
					// set override
					mConf.setMVOURI(uri);
				}
				catch (URISyntaxException ufe) {
					m_Log.error(lbl + "bad format, " + mvoURI, ufe);
					propErr = true;
				}
			}
			else {
				// use calculated value
				mConf.setMVOURI(calcURI);
			}
		}
		scc.setTotalStaking(totalStakings.toString());

		// bail if we got an error with properties
		if (propErr) {
			final String err = "error configuring MVOURI values for one or "
								+ "more MVOs from .properties files, for "
								+ "chain " + chain;
			Exception ee = new Exception(err);
			retFut.completeExceptionally(ee);
			return retFut;
		}

		// normal completion
		retFut.complete(scc);
		return retFut;
	}

	/**
	 * obtain the configuration particular to a given MVO on the chain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiSession the Web3j object at which we can access the chain
	 * @param mvoId the ID of the MVO
	 * @return the configuration (public parts only), exception or null on error
	 */
	public Future<BlockchainConfig> getMVOConfig(long chainId,
												 String chain,
												 Web3j abiSession,
												 String mvoId)
	{
		final String lbl = this.getClass().getSimpleName() + ".getMVOConfig: ";
		if (mvoId == null || mvoId.isEmpty() || abiSession == null) {
			m_Log.error(lbl + "missing input");
			return null;
		}
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}

		CompletableFuture<BlockchainConfig> retFut
			= new CompletableFuture<BlockchainConfig>();
		BlockchainConfig bConfig = new BlockchainConfig(chainId, chain, m_Log);
		Credentials credentials = null;

		// see if we're calling this on self
		if (mvoId.equals(m_MVO.getMVOId())) {
			// find our signing wallet credentials for this chain
			int numIdx = mvoId.indexOf("-") + 1;
			String mvoNum = mvoId.substring(numIdx);
			Properties props = m_MVO.getProperties();
			String credTag = "SigningWallet-" + chainId;
			String passTag = "SigningWalletPass-" + chainId;
			String credPass = props.getProperty(passTag);
			if (credPass == null || credPass.isEmpty()) {
				m_Log.error(lbl + "no " + passTag + " property defined");
				Exception err = new Exception("no " + passTag + " property");
				retFut.completeExceptionally(err);
				return retFut;
			}
			String credWallet = props.getProperty(credTag,
												  chainId + "-signwallet.json");
			if (credWallet == null || credWallet.isEmpty()) {
				m_Log.error(lbl + "no " + credWallet + " property defined");
				Exception err
					= new Exception("no " + credWallet + " property");
				retFut.completeExceptionally(err);
				return retFut;
			}
			String secDir = props.getProperty("RunDir", "");
			StringBuilder keyPath = new StringBuilder(128);
			if (secDir.isEmpty()) {
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + mvoId);
			String walletDir = keyPath.toString();
			String walletFile = walletDir + File.separator + credWallet;
			try {
				credentials = WalletUtils.loadCredentials(credPass, credWallet);
			}
			catch (CipherException ce) {
				m_Log.error(lbl + "exception loading signing Credentials", ce);
				retFut.completeExceptionally(ce);
				return retFut;
			}
			catch (IOException ioe) {
				m_Log.error(lbl + "exception loading signing Credentials", ioe);
				retFut.completeExceptionally(ioe);
				return retFut;
			}

			// pull data we need from the Credentials
			ECKeyPair ecKeys = credentials.getEcKeyPair();
			// (private key is of course not found on the blockchain)
			bConfig.configSigningKey(ecKeys.getPrivateKey());
			bConfig.configSigningAddress(Keys.toChecksumAddress(
													credentials.getAddress()));
		/*
			m_Log.debug(lbl + "signing private key = "
						+ Numeric.toHexStringNoPrefixZeroPadded(
												ecKeys.getPrivateKey(), 64));
		 */
		}
		else {
			// whip up dummy creds
			try {
				credentials = Credentials.create(Keys.createEcKeyPair());
			}
			catch (Exception ee) {
				m_Log.error("exception creating dummy Credentials", ee);
				return null;
			}
			m_Log.debug(lbl + "called on non-self: " + mvoId);
		}

		/* now access the MVOStaking contract to obtain this data:
		 *  bConfig.m_Staked
         *  bConfig.m_SigningAddress
         *  bConfig.m_StakingAddress
		 *  m_Active (stored in MVOConfig)
		 */
		String deployedAddr
			= MVOStaking.getPreviouslyDeployedAddress(Long.toString(chainId));
		if (deployedAddr == null) {
			// contract has not been deployed here; check Eth mainnet
			deployedAddr = MVOStaking.getPreviouslyDeployedAddress("1");
			if (deployedAddr == null) {
				final String noDep = "MVOStaking contract not deployed on "
									+ "chain Id " + chainId + " or mainnet";
				Exception ee = new Exception(noDep);
				retFut.completeExceptionally(ee);
				return retFut;
			}
		}
		MVOStaking mvoStaking = MVOStaking.load(deployedAddr,
												abiSession,
												credentials,
												// view only; no gas required
												new DefaultGasProvider());
		// return type corresponds to contract's struct MVO{}
		Tuple8<String,		// listenURI (not used by MVOs)
			   String,		// stakingAddress
			   String,		// signingAddress
			   String,		// encrPubkey (not used by MVOs)
			   BigInteger,	// stakingQuantum
			   BigInteger,	// memberPoolId (not used by MVOs)
			   BigInteger,	// rating
			   Boolean>		// active
			   	mvoData = null;
		try {
			mvoData = mvoStaking.idToMVO(mvoId).send();
		}
		catch (Exception eee) {
			m_Log.error(lbl + "exception fetching MVOStaking record");
			retFut.completeExceptionally(eee);
			return retFut;
		}
		finally {
			if (mvoData == null) {
				final String err404 = "MVOStaking record for " + "MVOId "
									+ mvoId + " not found";
				m_Log.error(lbl + err404);
				Exception err = new Exception(err404);
				retFut.completeExceptionally(err);
				return retFut;
			}
		}
		String stakingAddr = mvoData.component2();
		String signingAddr = mvoData.component3();
		String encrPubkey = mvoData.component4();
		BigInteger stakingQuantum = mvoData.component5();
		Boolean active = mvoData.component8();
	/*
		m_Log.debug(lbl + "for mvoId " + mvoId + ", got stakingAddr = "
					+ stakingAddr + ", signingAddr = " + signingAddr
					+ ", encrPubkey = " + encrPubkey + ", stakingQuantum = "
					+ stakingQuantum);
	 */
		if (mvoId.equals(m_MVO.getMVOId())) {
			// check signing address is the one we have Credentials for
			if (!credentials.getAddress().equals(signingAddr)) {
				m_Log.error(lbl + "signingAddress we have privkey for = "
							+ credentials.getAddress()
							+ ", but MVOStaking record has: " + signingAddr);
				Exception err = new Exception(
							"signingAddress does not match on-chain record");
				retFut.completeExceptionally(err);
				return retFut;
			}
			// set active value in our own config
			m_MVO.getConfig().setStatus(active.booleanValue());
		}
		bConfig.configSigningAddress(Keys.toChecksumAddress(signingAddr));
		bConfig.setStaking(stakingQuantum.toString());
		bConfig.configStakingAddress(Keys.toChecksumAddress(stakingAddr));

		// normal completion
		retFut.complete(bConfig);
		return retFut;
	}

	/**
	 * obtain all the eNFTs on-chain for a given account address
	 * @param chainId the chain Id per standard, used to select files
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @param acct the account address (as EIP-55)
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted), null or exception on errors
	 */
	public Future<HashMap<String, NFTmetadata>> getAccountNFTs(long chainId,
															   Web3j abiSession,
														 	   String acct,
														 	boolean demintedOk)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".getAccountNFTs: ";
		if (acct == null || acct.isEmpty()) {
			m_Log.error(lbl + "missing acct Id");
			return null;
		}
		if (abiSession == null) {
			m_Log.error(lbl + "missing ABI session");
			return null;
		}
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}
		String acctId = Numeric.cleanHexPrefix(Keys.toChecksumAddress(acct));

		CompletableFuture<HashMap<String, NFTmetadata>> retFut
			= new CompletableFuture<HashMap<String, NFTmetadata>>();

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null || !scc.isEnabled()) {
			// don't have this chain defined
			Exception err2 = new Exception("chainId " + chainId
											+ " data is not available");
			retFut.completeExceptionally(err2);
			return retFut;
		}

		HashMap<String, NFTmetadata> eNFTlist
			= new HashMap<String, NFTmetadata>();
		boolean gotErr = false;

		// get the cached data object for this chain
		EnftCache eCache = scc.getCache();

		// obtain all eNFT Ids ever minted to this account
		ArrayList<BigInteger> acctIds = eCache.getMintedIdsForAccount(acctId);
		if (acctIds == null || acctIds.isEmpty()) {
			// nothing to return
			retFut.complete(eNFTlist);
			return retFut;
		}

		// if deminted are *not* permitted, subtract list of burned Ids
		if (!demintedOk) {
			ArrayList<BigInteger> burntIds
				= eCache.getBurnedIdsForAccount(acctId);
			if (burntIds != null) {
				acctIds.removeAll(burntIds);
			}
		}

		// for each owned Id retrieve metadata and parse it
		for (BigInteger eId : acctIds) {
			String paddedId = Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
			String metadataJson = eCache.getMetadataForId(eId);
			if (metadataJson == null) {
				gotErr = true;
				m_Log.error(lbl + "metadata not found for minted Id "
							+ paddedId);
				continue;
			}

			// build eNFT (encrypted details) from metadata
			NFTmetadata eNFT = new NFTmetadata(m_Log);
			if (eNFT.buildFromJSON(metadataJson)) {
				eNFTlist.put(paddedId, eNFT);
				//m_Log.debug(lbl + "put eNFT on list, " + metadataJson);
			}
			else {
				m_Log.error(lbl + "error building NFT for Id " + paddedId);
				gotErr = true;
			}
		}

		if (gotErr) {
			Exception errRet = new Exception("got errors processing eNFTs");
			retFut.completeExceptionally(errRet);
		}
		else {
			retFut.complete(eNFTlist);
		}
		return retFut;
	}

	/**
	 * obtain a list of eNFTs, given their IDs
	 * @param chainId the chain Id per standard, used to select files
	 * @param abiSession the Web3j object at which we can access chain (ignored)
	 * @param acct the account address (as EIP-55)
	 * @param idList the list of eNFT IDs we want
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted), null or exception on errors
	 */
	public Future<ArrayList<NFTmetadata>> getNFTsById(long chainId,
													  Web3j abiSession,
													  String acct,
													  ArrayList<String> idList,
													  boolean demintedOk)
	{
		final String lbl = this.getClass().getSimpleName() + ".getNFTsById: ";
		if (acct == null || acct.isEmpty()) {
			m_Log.error(lbl + "missing acct Id");
			return null;
		}
		if (idList == null || idList.isEmpty()) {
			m_Log.error(lbl + "missing idList");
			return null;
		}
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}
		if (abiSession == null) {
			m_Log.error(lbl + "missing ABI session");
			return null;
		}
		String acctId = Numeric.cleanHexPrefix(Keys.toChecksumAddress(acct));

		CompletableFuture<ArrayList<NFTmetadata>> retFut
			= new CompletableFuture<ArrayList<NFTmetadata>>();

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null || !scc.isEnabled()) {
			// don't have this chain defined
			Exception err2 = new Exception("chainId " + chainId
											+ " data is not available");
			retFut.completeExceptionally(err2);
			return retFut;
		}

		// get the cached data object for this chain
		EnftCache eCache = scc.getCache();
		ArrayList<NFTmetadata> eNFTlist
			= new ArrayList<NFTmetadata>(idList.size());
		boolean gotErr = false;

		// obtain all eNFT Ids ever minted to this account
		ArrayList<BigInteger> acctIds = eCache.getMintedIdsForAccount(acctId);
		if (acctIds == null || acctIds.isEmpty()) {
			// nothing to return
			retFut.complete(eNFTlist);
			return retFut;
		}

		// if deminted are *not* permitted, subtract list of burned Ids
		ArrayList<BigInteger> burntIds = eCache.getBurnedIdsForAccount(acctId);
		if (!demintedOk) {
			if (burntIds != null) {
				acctIds.removeAll(burntIds);
			}
		}

		// look up each requested ID on the chain
		EnshroudProtocol wrapper = eCache.getProtocolWrapper();
		for (String id : idList) {
			BigInteger numericId = Numeric.parsePaddedNumberHex(id);
			if (acctIds.contains(numericId)) {
				String metadataJson = eCache.getMetadataForId(numericId);
				if (metadataJson == null) {
					gotErr = true;
					m_Log.error(lbl + "metadata not found for minted Id " + id);
					continue;
				}

				// build eNFT (encrypted details) from metadata
				NFTmetadata eNFT = new NFTmetadata(m_Log);
				if (eNFT.buildFromJSON(metadataJson)) {
					eNFTlist.add(eNFT);
				}
				else {
					m_Log.error(lbl + "error building NFT for Id " + id);
					gotErr = true;
				}
			}
			else {
				m_Log.error(lbl + "input Id " + id + " metadata not found");
			}
		}

		if (gotErr) {
			Exception errRet = new Exception("got errors processing eNFTs");
			retFut.completeExceptionally(errRet);
		}
		else {
			retFut.complete(eNFTlist);
		}
		return retFut;
	}

	/**
	 * method to determine whether a given ID is used (including deminted eNFTs)
	 * @param chainId the chain Id per chainlist.org
	 * @param abiService the object at which we can access Web3j service
	 * @param id the eNFT Id
	 * @return true if an eNFT exists with this Id
	 */
	public Future<Boolean> checkIfIdUsed(long chainId,
										 Web3j abiService,
										 String id)
	{
		final String lbl = this.getClass().getSimpleName() + ".checkIfIdUsed: ";
		if (chainId != m_ChainId) {
			m_Log.error(lbl + "invoked for invalid chain, " + chainId);
			return null;
		}
		if (id == null || id.isEmpty()) {
			m_Log.error(lbl + "missing eNFT Id");
			return null;
		}
		if (abiService == null) {
			m_Log.error(lbl + "missing ABI session");
			return null;
		}

		CompletableFuture<Boolean> retFut = new CompletableFuture<Boolean>();

		// get smart contract config for this chain
		SmartContractConfig scc = m_MVO.getSCConfig(chainId);
		if (scc == null || !scc.isEnabled()) {
			// don't have this chain defined
			Exception err2 = new Exception("chainId " + chainId
											+ " data is not available");
			retFut.completeExceptionally(err2);
			return retFut;
		}
		Boolean used = Boolean.valueOf(false);

		// access EnftCache URL event mapping
		EnftCache eCache = scc.getCache();
		BigInteger numericId = Numeric.parsePaddedNumberHex(id);
		String metadata = eCache.getMetadataForId(numericId);
		if (metadata != null && !metadata.isEmpty()) {
			used = Boolean.valueOf(true);
		}

		// alt: access EnshroudProtocol.nftIdUsed(id) and see if maps to true
		// (however this map is internal not public)

		retFut.complete(used);
		return retFut;
	}

	// implement FileFilter
	/**
	 * method to select files matching security/MVO-*.properties
	 * @param path the file to examine
	 * @return true if we should accept this file
	 */
	public boolean accept(File path) {
		if (path == null) {
			return false;
		}
		String name = path.getName();
		if (name.startsWith("MVO-") && name.endsWith(".properties")) {
			return true;
		}
		return false;
	}

	// END methods
}
