/*
 * last modified---
 * 	07-09-24 mute ping attempts if !m_PeerAlive
 * 	07-05-24 fixes for dup connect attempts by websocket layer
 * 	03-07-24 reset m_Sequence value on signatures before adding, to avoid
 * 			 multiple copies of "002" since every !lead MVO sends that value
 * 	03-01-23 verify main OB signature on replies
 * 	02-23-23 verify args hash signature on OB replies
 * 	09-15-22 handle errors returned by Auditors to broadcasts
 * 	09-13-22 debug Auditor async connect handling
 * 	08-17-22 add sendGenericErr() and its invocations
 * 	08-04-22 supply handling for Auditor messages
 * 	07-12-22 disable pings when peer appears not to be alive
 * 	06-23-22 add ping/pong frames as keepalives
 * 	06-17-22 implement handling of error messages returned from partners
 * 	06-10-22 new
 *
 * purpose---
 * 	provide a connection object between an MVO and a peer MVO or AUD node
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.MVOAuditorBlock;

import org.eclipse.jetty.websocket.api.Session;
import org.eclipse.jetty.websocket.api.CloseStatus;
import org.eclipse.jetty.websocket.api.StatusCode;
import org.eclipse.jetty.websocket.api.UpgradeRequest;
import org.eclipse.jetty.websocket.api.RemoteEndpoint;
import org.eclipse.jetty.websocket.api.ProtocolException;
import org.eclipse.jetty.websocket.common.WebSocketSession;
import org.eclipse.jetty.websocket.common.io.FutureWriteCallback;
import org.eclipse.jetty.websocket.api.annotations.WebSocket;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketConnect;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketClose;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketError;
import org.eclipse.jetty.websocket.api.annotations.OnWebSocketMessage;

import java.util.Base64;
import java.util.Properties;
import java.util.Hashtable;
import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.Timer;
import java.util.TimerTask;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.net.URI;


/**
 * This class represents the connections to a peer node (MVO or Auditor).  Due
 * to the way WebSocket connections work, we need a pair of connections, one
 * where we act as the server and one where we act as the client.  This class
 * manages the two types of WebSocket classes (MVOClientSocket and
 * MVOServerSocket), along with the logic required to create, authenticate,
 * and maintain active peer connections.  The WebSocket callbacks also handle
 * processing for MVO peer requests.  All of these are sent as TEXT messages.
 * It interoperates with the connection factory classes, {@link MVOHandler}
 * (for server-type connections), and {@link MVOClient}.
 */
public final class PeerConnector {
	// BEGIN data members
	/**
	 * the owning MVO
	 */
	private MVO							m_MVOServer;

	/**
	 * the logging object (shared with owning MVO)
	 */
	private Log							m_Log;

	/**
	 * the ID of the peer we're talking to (MVO-NNN or AUD-NNN)
	 */
	private String						m_PartnerMVO;

	/**
	 * challenge message sent to authenticate m_PartnerMVO value, format:
	 * our AUTHREQ::MVO ID::base64(random 64 bytes)::our RSA sig on second
	 * and third fields only
	 */
	private String						m_ChallengeData;

	/**
	 * helper class to provide keep-alives on peer connections via Ping/Pong
	 */
	protected class PeerKeepAlive extends TimerTask {
		// data member
		/**
		 * random data used in the current outgoing ping
		 */
		public String					m_PingData;

		/**
		 * flag indicating whether peer connection is alive
		 */
		public boolean					m_PeerAlive;

		/**
		 * flag indicating whether we've seen a reply pong since last ping
		 */
		public boolean					m_PeerPonged;

		// methods
		/**
		 * nullary constructor
		 */
		public PeerKeepAlive() {
			super();
			m_PingData = "";
			m_PeerAlive = true;
			m_PeerPonged = true;
		}

		/**
		 * perform keep-alive tasks
		 */
		public void run() {
			if (m_PeerAlive && !m_PeerPonged) {
				m_Log.warning("PeerKeepAlive: peer " + m_PartnerMVO
							+ " did not PONG our last PING");
			}
			// if no reply from last send, mark dead
			m_PeerAlive = m_PeerPonged;

			// generate unique challenge data for this run
			Long runData = m_MVOServer.getStateObj().getRNG().nextLong();
			m_PingData = runData.toString();
			String pingTxt = m_PartnerMVO + "::PING::" + m_PingData;

			// send to client side (only)
			m_PeerPonged = false;
			if (m_PeerAlive) {
				if (!sendString(pingTxt, null)) {
					m_Log.error("PeerKeepAlive: could not ping "
								+ m_PartnerMVO);
					m_PeerAlive = false;
				}
			/*
				else {
					m_Log.debug("Pinged " + m_PartnerMVO);
				}
			 */
			}
		}
	}

	/**
	 * ping-pong keep-alive task
	 */
	protected PeerKeepAlive				m_PingPonger;

	/**
	 * timer that runs keep-alive task
	 */
	private Timer						m_PingTimer;

	/**
	 * helper class to act as the WebSocket for sessions with server behavior
	 */
	@WebSocket
	public class MVOServerWebSocket {
		// data members
		/**
		 * the actual web socket session
		 */
		private WebSocketSession		m_Session;

		/**
		 * whether our connection has passed encryption challenge yet
		 */
		private boolean					m_Authenticated;

		/**
		 * Whether we initiated this connection (via MVOClient).  This is
		 * always false, but is here simply so the session can tell itself
		 * whether it's a client or server websocket.
		 */
		private boolean					m_Outgoing;

		// methods
		/**
		 * nullary constructor
		 */
		public MVOServerWebSocket() {
		}

		// methods to implement WebSocket
		/**
		 * register new sessions with other MVOs (and Auditors)
		 * @param session the new Session (really a WebSocketSession)
		 */
		@OnWebSocketConnect
		public void onWebSocketConnect(Session session) {
			final String lbl = "ServerWebSocket.onWebSocketConnect: ";
			/* This will automatically be recorded with the SessionTracker.
			 * We also need to look at the attributes provided in the upgrade
			 * request to determine the MVO-ID, and register it with the
			 * list in the m_WebSocketHandlers hash table.
			 * NB: all session.close() calls cannot send status codes or reasons
			 * 	   because this will trigger the "Client MUST mask all frames"
			 * 	   error, which will be handed to the
			 * 	   ServerWebSocket.onWebSocketClose() method with status 1002.
			 */
			if (!(session instanceof WebSocketSession)) {
				m_Log.error(lbl + "non web socket session received, ignoring");
				if (session != null) {
					// not WSS; go ahead and risk close with status
					session.close(StatusCode.PROTOCOL,
								"Not a websocket session");
				}
				return;
			}
			setSession((WebSocketSession) session);
			m_PingPonger.m_PeerAlive = true;
			
			// make sure mode is correct
			if (m_Outgoing) {
				m_Log.error(lbl + "for outgoing");
			/*
				m_Session.close(StatusCode.POLICY_VIOLATION,
								"Outgoing server socket!");
			 */
				m_Session.close();
				return;
			}

			// obtain the peer Id passed by the other side
			UpgradeRequest upgReq = m_Session.getUpgradeRequest();
			String mvoId = upgReq.getHeader("PeerId");
			String mvoID = "";
			if (mvoId != null && !mvoId.isEmpty()) {
				mvoID = mvoId;
			}
			if (mvoID.isEmpty()) {
				m_Log.error(lbl + "missing Peer Id");
			/*
				m_Session.close(StatusCode.POLICY_VIOLATION,
								"Missing Peer Id");
			 */
				m_Session.close();
				return;
			}
			m_PartnerMVO = new String(mvoID);

			// if we already have a connection, close dup
			String ourID = m_MVOServer.getMVOId();
			MVOServerWebSocket existingSess = getServerSocket();
			//if (existingSess != null && existingSess.isAuthenticated()) {
			if (existingSess != null) {
				m_Log.debug(lbl + "closing dup connection to Peer Id "
							+ m_PartnerMVO);
				m_Session.close();
				return;
			}

			// record new socket
			registerServerSocket(this);
			m_Log.debug(lbl + "connected incoming WS from "
						+ m_PartnerMVO + ", at: "
						+ m_Session.getRemoteAddress());

			// trigger encrypted challenge / response to prove MVO ID
			MVOState stObj = m_MVOServer.getStateObj();
			byte[] challenge = new byte[64];
			stObj.getRNG().nextBytes(challenge);
			Base64.Encoder b64e = Base64.getEncoder();
			String challengeStr = b64e.encodeToString(challenge);
			setChallenge(challengeStr);
			String sigData = ourID + "::" + challengeStr;
			PrivateKey privKey = m_MVOServer.getConfig().getCommPrivKey();
			String ourSig = EncodingUtils.signStr(privKey, sigData);
			if (!sendString("AUTHREQ::" + sigData + "::" + ourSig, null)) {
				m_Log.error(lbl + "unable to send AUTHREQ to MVO Id "
							+ m_PartnerMVO);
			}
		}

		/**
		 * register closure of existing session
		 * @param session the departing Session (actually a WebSocketSession)
		 * @param status the code with which the Session closed
		 * @param reason textual reason to go with code
		 */
		@OnWebSocketClose
		public void onWebSocketClose(Session session, int status, String reason)
		{
			/* This will automatically be removed from the SessionTracker.
			 * We need to delete this session from the m_WebSocketHandlers.
			 */
			m_Log.debug("ServerWebSocket.onWebSocketClose: close of session to "
						+ m_PartnerMVO + ", status = " + status
						+ ", reason = " + reason);
			if (!m_PartnerMVO.isEmpty()) {
				// stop pings
				if (m_PingTimer != null) {
					m_PingTimer.cancel();
				}
				registerServerSocket(null);
			}
			else {
				m_Log.error("ServerWebSocket.onWebSocketClose: session was not "
							+ "for a known partner node");
			}
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle an exception on a WebSocket session
		 * @param session the session which got the exception
		 * @param cause the exception
		 */
		@OnWebSocketError
		public void onWebSocketError(Session session, Throwable cause) {
			m_Log.error("ServerWebSocket.onWebSocketError: WebSocket "
						+ "exception: " + cause.toString());
			// remove session from object
			if (!m_PartnerMVO.isEmpty()) {
				// stop pings
				if (m_PingTimer != null) {
					m_PingTimer.cancel();
				}
				registerServerSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ServerWebSocket.onWebSocketError: session was not "
							+ "for a known partner node");
			}
			// NB: no need to close, Jetty will do it automatically
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle a text message
		 * @param session the session sending the message
		 * @param text the entire message
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session, String text) {
			final String lbl = "ServerWebSocket.onWebSocketMessage: ";
			/* If this connection is the result of an outbound connection that
			 * was opened by the MVOClient, there's a chance of a race between
			 * the return of the connect() call (Future) and the arrival of the
			 * AUTHREQ message from the other side.  Therefore, if we don't yet
			 * have a m_Session value, we need to set it to the passed param.
			 */
			if (m_Session == null && session instanceof WebSocketSession) {
				m_Log.debug(lbl + "no Session "
							+ "yet, but received our first text message");
				setSession((WebSocketSession) session);
			}
			if (!(session instanceof WebSocketSession)
				|| session != m_Session)
			{
				m_Log.error(lbl + "bad session");
				session.close(StatusCode.PROTOCOL, "Not a websocket session");
				return;
			}

			/* NB: all session.close() calls cannot send status codes or reasons
			 * 	   because this will trigger the "Client MUST mask all frames"
			 * 	   error, which will be handed to the
			 * 	   ServerWebSocket.onWebSocketClose() method with status 1002.
			 */

			// check for authentication messages if not yet authenticated
			String ourID = m_MVOServer.getMVOId();
			if (!isAuthenticated()) {
				// check for an AUTH reply
				if (text.startsWith("AUTHREP::")) {
					// this is expected if we received the connection
					if (m_Outgoing) {
						m_Log.warning(lbl + "got AUTHREP on session we opened");
					/*
						session.close(StatusCode.POLICY_VIOLATION,
									"AUTHREP on client socket!");
					 */
						session.close();
						return;
					}

					// parse message, check challenge data and signature
					// format is: AUTHREP::ID::challenge::signature
					String[] authMsg = text.split("::");
					if (authMsg.length != 4) {
						m_Log.error(lbl + "improper AUTHREP, " + text);
					/*
						session.close(StatusCode.PROTOCOL,
									  "Bad AUTHREP format");
					 */
						session.close();
						return;
					}
					String peerId = authMsg[1];
					if (!peerId.equals(m_PartnerMVO)) {
						// this should not have changed
						m_Log.error(lbl + "peer Id "
									+ peerId + " in AUTHREP does not "
									+ "match established Id " + m_PartnerMVO);
					/*
						session.close(StatusCode.PROTOCOL,
									"Expected AUTHREP from " + m_PartnerMVO);
					 */
						session.close();
						return;
					}
					String b64Challenge = authMsg[2];
					String recChallenge = getChallenge();
					if (recChallenge.isEmpty()) {
						m_Log.error(lbl
									+ "no recorded challenge data on AUTHREP");
					}
					else {
						// make sure same challenge data was echoed back
						if (!b64Challenge.equals(recChallenge)) {
							// this should not have changed
							m_Log.error(lbl + "peer Id " + m_PartnerMVO
										+ " sent back changed challenge data "
										+ "with sig");
						/*
							session.close(StatusCode.PROTOCOL,
									"Challenge data was returned incorrectly");
						 */
							session.close();
							return;
						}
					}

					// fetch pubkey of m_PartnerMVO and check sig on reply
					String peerSig = authMsg[3];
					MVOConfig config = m_MVOServer.getConfig();
					PublicKey peerKey = config.getPeerPubkey(peerId);
					if (peerKey == null) {
						m_Log.error(lbl + "no pubkey for peerId "
									+ m_PartnerMVO);
					/*
						session.close(StatusCode.SERVER_ERROR,
								"No pubkey available for ID " + m_PartnerMVO);
					 */
						session.close();
						return;
					}
					String sigData = m_PartnerMVO + "::" + b64Challenge;

					// verify signature
					if (!EncodingUtils.verifySignedStr(peerKey,
													   sigData,
													   peerSig))
					{
						m_Log.error(lbl + "sig "
									+ "verification error on AUTHREP from "
									+ peerId);
					/*
						session.close(StatusCode.PROTOCOL, "Failure verifying "
									+ "signature on AUTHREP");
					 */
						session.close();
						return;
					}

					// this proves the other side is who they claim to be
					m_Authenticated = true;
				}
				else if (text.startsWith("AUTHREQ::")) {
					// this is expected if we initiated the connection
					if (m_Outgoing) {
						m_Log.warning(lbl + "got AUTHREQ on session we opened");
					/*
						session.close(StatusCode.POLICY_VIOLATION,
									"AUTHREP on client socket!");
					 */
						session.close();
						return;
					}

					// parse message, check signature and give reply
					// format is: AUTHREQ::ID::challenge::signature
					String[] authMsg = text.split("::");
					if (authMsg.length != 4) {
						m_Log.error(lbl + "improper AUTHREQ, " + text);
					/*
						session.close(StatusCode.PROTOCOL,
									"Bad AUTHREQ format");
					 */
						session.close();
						return;
					}
					String peerId = authMsg[1];
					// get pubkey for this peer
					MVOConfig config = m_MVOServer.getConfig();
					PublicKey peerKey = config.getPeerPubkey(peerId);
					if (peerKey == null) {
						m_Log.error(lbl + "no pubkey for peerId " + peerId);
					/*
						session.close(StatusCode.SERVER_ERROR,
									"No pubkey available for ID " + peerId);
					 */
						session.close();
						return;
					}
					String b64Challenge = authMsg[2];
					String peerSig = authMsg[3];
					String sigData = peerId + "::" + b64Challenge;

					// verify signature
					if (!EncodingUtils.verifySignedStr(peerKey,
													   sigData,
													   peerSig))
					{
						m_Log.error(lbl + "sig "
									+ "verification error on AUTHREQ from "
									+ peerId);
					/*
						session.close(StatusCode.PROTOCOL, "Failure verifying "
									+ "signature on AUTHREQ");
					 */
						session.close();
						return;
					}

					// this proves the other side is who they claim to be
					m_PartnerMVO = new String(peerId);
					if (m_ClientWS != null) {
						m_ClientWS.setAuthenticated(true);
					}

					/* send a AUTHREP reply, allowing the other side to
					 * authenticate us via the same process
					 */
					sigData = ourID + "::" + b64Challenge;
					PrivateKey privKey = config.getCommPrivKey();
					String ourSig = EncodingUtils.signStr(privKey, sigData);
					String repData = "AUTHREP::" + sigData + "::" + ourSig;
					if (!sendString(repData, null)) {
						m_Log.error(lbl + "unable to send AUTHREP to MVO Id "
									+ m_PartnerMVO);
						m_Session.close();
						return;
					}

					MVOState stObj = m_MVOServer.getStateObj();
					Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();

					// see if this is an MVO or an Auditor
					if (m_PartnerMVO.startsWith("MVO-")) {
						/* We need to check whether our context here is that we
						 * opened this connection to the other MVO so that we
						 * can forward them a committee request.  If this is
						 * the case then we'll have the peerId recorded in a
						 * committee list in one of the broker map entries.  So
						 * we search for that and if found, immediately forward
						 * the pending request.
						 */
						boolean sent = false;
						for (String transId : brMap.keySet()) {
							// find all committees that include this MVO
							MVOBrokerEntry brEntry = brMap.get(transId);
							ArrayList<MVOBrokerEntry.MVOCommitteeMember>
								committee = brEntry.getCommittee();
							if (committee != null && !committee.isEmpty()) {
								for (MVOBrokerEntry.MVOCommitteeMember cMember
																	: committee)
								{
									if (cMember.m_Id.equals(m_PartnerMVO)
										&& cMember.m_ConnectWait)
									{
										// confirm we have data queued to send
										if (cMember.m_Context == null) {
											m_Log.error(lbl + "no queued "
													+ "committee data for tId "
														+ transId + ", MVO Id "
														+ m_PartnerMVO);
											return;
										}

										// build our signature on actual data
										StringBuilder completeJSON
											= new StringBuilder(4096);
										cMember.m_Context.addJSON(completeJSON);
										String fullTxt
											= completeJSON.toString();
										ourSig = EncodingUtils.signStr(privKey,
																	   fullTxt);

										// encrypt the message to their pubkey
										String encTxt
											= EncodingUtils.base64PubkeyEncStr(
															fullTxt, peerKey);

										// get a message tag
										String tag = brEntry.getMVOMsgTag(
																m_PartnerMVO);

										// build and send async message
										String mvoMsg = tag + "::" + encTxt
														+ "::" + ourSig;
										cMember.m_MsgTag = tag;
										cMember.m_ConnectWait = false;
										if (sendString(mvoMsg, cMember)) {
											brEntry.setBrokerMVOReq(tag,
																	mvoMsg);
											sent = true;
										/*
											m_Log.debug("Sent queued committee "
													+ "request tId " + transId
													+ " MVO Id " + m_PartnerMVO
														+ " with tag "
														+ cMember.m_MsgTag);
										 */
											/* NB: this MVO won't be in the
											 * committee more than once
											 */
											break;
										}
										else {
											m_Log.error(lbl
													+ "error writing the "
													+ "committee data for tId "
													+ transId + ", MVO Id "
													+ m_PartnerMVO);
											return;
										}
									}
								}
							}
						}
						if (!sent) {
							// perfectly normal
						/*
							m_Log.debug("No pending committee req found for "
										+ "newly connected MVO Id "
										+ m_PartnerMVO);
						 */
						}
					}
					else {	// Auditor
						/* We need to look for a queued up key server request,
						 * or a broadcast AuditorBlock.  Because we're looping
						 * through the whole map, we'll find any and all
						 * outstanding requests involving this Auditor, and
						 * send each one.
						 */
						boolean sent = false;
						for (String transId : brMap.keySet()) {
							MVOBrokerEntry brEntry = brMap.get(transId);
							MVOBrokerEntry.KeyServerTarget audTarget
								= brEntry.getKeyServer();
							if (audTarget.m_Id.equals(m_PartnerMVO)
								&& audTarget.m_ConnectWait)
							{
								// confirm we have data queued to send
								if (audTarget.m_Context == null) {
									m_Log.error(lbl + "no queued "
												+ "key request for tId "
												+ transId + ", AUD Id "
												+ m_PartnerMVO);
									return;
								}

								// build our signature on actual data
								StringBuilder completeJSON
									= new StringBuilder(2048);
								audTarget.m_Context.addJSON(completeJSON);
								String fullTxt = completeJSON.toString();
								ourSig = EncodingUtils.signStr(privKey,
															   fullTxt);

								// encrypt the message to their pubkey
								String encTxt
									= EncodingUtils.base64PubkeyEncStr(
														fullTxt, peerKey);

								// get a message tag
								String tag = brEntry.getAUDMsgTag(m_PartnerMVO);

								// build and send key server request (async)
								String audMsg
									= tag + "::" + encTxt + "::" + ourSig;
								audTarget.m_MsgTag = tag;
								audTarget.m_ConnectWait = false;
								if (sendString(audMsg, audTarget)) {
									brEntry.setBrokerAudReq(tag, audMsg);
									sent = true;
								/*
									m_Log.debug("Sent queued key server "
												+ "request tId " + transId
												+ " AUD Id " + m_PartnerMVO
												+ " with tag "
												+ audTarget.m_MsgTag);
								 */
								}
								else {
									m_Log.error(lbl + "error writing the "
											+ "key server req data for tId "
											+ transId + ", AUD Id "
											+ m_PartnerMVO);
									return;
								}
							}
						}
						if (!sent) {
							// this occurs when the AUD (re-)connects to us
							m_Log.debug("No pending requests found for newly "
									+ "(re-)connected AUD Id " + m_PartnerMVO);
						}
					}
				}
				else {
					// regular message in non-authenticated context (error)
					m_Log.error(lbl
								+ "got regular message in non-authenticated "
								+ "context; partner MVO Id = "
								+ m_PartnerMVO + ", msg = " + text);
					m_Session.close();
				}
				return;
			}

			// process normal session message, by examining tag and body
			int parseIdx = text.indexOf("::");
			if (parseIdx == -1) {
				// unrecognized message format
				m_Log.error(lbl + "got unknown msg format from "
							+ m_PartnerMVO + ", msg = \"" + text + "\"");
				return;
			}

			/* message should be in 3 parts:
			 * 	tag::encrypted text::signature (for a normal message); or:
			 *	tag::ERRcode::error text (if it's an error reply)
			 *	MVOId::PING::random text (if it's a ping message)
			 *	MVOId::PONG::random text (if it's a pong response)
			 */
			String[] msgParts = text.split("::");
			if (msgParts.length != 3) {
				m_Log.error(lbl + "got weird msg "
							+ "format from " + m_PartnerMVO + ", msg = \""
							+ text + "\"");
				return;
			}

			// check for a PING/PONG
			final String msg = msgParts[1];
			if (msg.equals("PING")) {
				// first element should be our ID
				if (!msgParts[0].equals(ourID)) {
					m_Log.error(lbl + "PING from "
								+ m_PartnerMVO + " not for us");
				}
				String pingTxt = msgParts[2];
				// send same text back as reply
				String pongMsg = m_PartnerMVO + "::PONG::" + pingTxt;
				if (!sendString(pongMsg, null)) {
					m_Log.error(lbl + "could not "
								+ "send PONG reply to " + m_PartnerMVO);
				}
				return;
			}
			else if (msg.equals("PONG")) {
				// check whether text is same as what we sent in last ping
				String pongTxt = msgParts[2];
				if (pongTxt.equals(m_PingPonger.m_PingData)) {
					m_PingPonger.m_PeerPonged = true;
				}
				return;
			}

			// the tag should also be in 3 parts: MVOId (us), transId, seqNum
			String tag = msgParts[0];
			parseIdx = tag.indexOf(":");
			if (parseIdx == -1) {
				m_Log.error(lbl + "bad message "
							+ "tag from " + m_PartnerMVO + ", tag = \""
							+ tag + "\"");
				return;
			}
			String[] tagParts = tag.split(":");
			if (tagParts.length != 3) {
				m_Log.error(lbl + "weird message "
							+ "tag from " + m_PartnerMVO + ", tag = \""
							+ tag + "\"");
				return;
			}
			String destMVO = tagParts[0];
			String transId = tagParts[1];
			String seqNum = tagParts[2];

			// double-check the message is for us
			if (!destMVO.equals(ourID)) {
				m_Log.error(lbl + "message tagged for " + destMVO
							+ ", not for us, ignoring");
				// return a generic error
				sendGenericErr(transId, seqNum,
								"Message sent for " + destMVO + " not us");
				return;
			}

			// obtain the req broker entry for this transaction Id (if any)
			MVOState stObj = m_MVOServer.getStateObj();
			Map<String, MVOBrokerEntry> brMap = stObj.getBrokerMap();
			MVOBrokerEntry brEntry = brMap.get(transId);

			/* Check for an error response from the partner node.  These are
			 * in the format: tag::ERR{ec}::message text, where {ec} is an
			 * integer (a HTTP error status value).
			 */
			String errStat = null;
			String errCode = null;
			String errText = null;
			boolean realErr = false;
			if (msg.startsWith("ERR")) {
				// this may be an error message -- first, we parse it
				errStat = msg;
				errCode = errStat.substring(3).trim();
				realErr = errCode.length() <= 8;
				// if a real error message, the errCode will be an integer
				Integer code = null;
				if (realErr) {
					try {
						code = Integer.parseInt(errCode);
					}
					catch (NumberFormatException nfe) {
						m_Log.error(lbl + "non-integer error code from "
									+ m_PartnerMVO + "; value passed = "
									+ errCode);
						realErr = false;
					}
				}
				errText = msgParts[2];
			}
			if (realErr) {
				/* Since errors are sent in response to outgoing requests, we
				 * expect to have a valid broker entry here.  If we do, we'll
				 * report an error appropriately to the state engine.  If we
				 * don't, there's nothing we can do but log the issue.
				 */
				if (brEntry == null) {
					if (m_PartnerMVO.startsWith("AUD-")) {
						/* There's a chance this is an error related to an
						 * MVOAuditorBlock previously broadcast by a queue.
						 * If so, we should log which broadcast failed.
						 */
						AuditorQueue aQ = m_MVOServer.getAuditorManager();
						AuditorQueue.PerAuditorQueue audQ
							= aQ.getAuditorQueue(m_PartnerMVO);
						if (audQ != null) {
							MVOAuditorBlock audBlock
								= audQ.getBlockForTid(transId);
							if (audBlock != null) {
								StringBuilder blOut = new StringBuilder(10240);
								audBlock.addJSON(blOut);
								// log failure
								m_Log.error(lbl + "Auditor " + m_PartnerMVO
											+ " sent us an error on tId "
											+ transId + ": code = " + errCode
											+ ", err = \"" + errText + "\"");

								// if errCode == 1011, re-queue message
								if (errCode.equals(
									Integer.toString(StatusCode.SERVER_ERROR)))
								{
									AuditorQueue.BroadcastEntry retryEnt
										= aQ.new BroadcastEntry();
									retryEnt.m_ChainId
									= audBlock.getClientBlock().getChainId();
									retryEnt.m_BlockJson = blOut.toString();
									retryEnt.m_Block = audBlock;
									if (!audQ.queueBlock(retryEnt)) {
										m_Log.error(lbl + "Failed requeueing "
													+ "AUD tId " + transId
													+ " to " + m_PartnerMVO);
									}
									else {
										m_Log.debug(lbl + "Success requeueing "
													+ "AUD tId " + transId
													+ " to " + m_PartnerMVO);
									}
								}
							}
							else {
								m_Log.error(lbl + "got ERR " + "reply from AUD "
											+ m_PartnerMVO + " without any "
											+ "transId context, tId was "
											+ transId);
							}
						}
						else {
							m_Log.error(lbl + "no PerAuditorQueue found for "
										+ m_PartnerMVO + ", which sent ERR");
						}
					}
					else {
						m_Log.error(lbl + "got ERR " + "reply from "
									+ m_PartnerMVO + " without broker request "
									+ "context, tId was " + transId);
					}
					return;
				}

				// otherwise do normal error processing
				m_Log.debug(lbl + "partner "
							+ m_PartnerMVO + " sent us an error on tId "
							+ transId + ": code = " + errCode
							+ ", err = \"" + errText + "\"");
				if (m_PartnerMVO.startsWith("MVO-")) {
					// this counts as a reply for any outstanding async reqs
					boolean foundMatch = false;
					for (MVOBrokerEntry.MVOCommitteeMember cMember
						: brEntry.getCommittee())
					{
						String matchTag
							= m_PartnerMVO + ":" + transId + ":" + seqNum;
						if (matchTag.equals(cMember.m_MsgTag)) {
							// look for a timer to cancel
							if (cMember.m_Timer != null) {
								cMember.m_Timer.cancel();
								foundMatch = true;
								break;
							}
						}
					}

					if (foundMatch) {
						// register a partner error reply on the current step
						ReqSvcState stateEngine = brEntry.getStateTracker();
						int failStep = stateEngine.getCurrentStep();
						int lastStep = stateEngine.getPreviousStep();
						stateEngine.advanceState(lastStep, failStep,
												 ReqSvcState.M_GotNack,
												 errText);
					}
					else {
						m_Log.error(lbl + "MVO error received without a "
									+ "committee context, from "
									+ m_PartnerMVO);
					}
				}
				else if (m_PartnerMVO.startsWith("AUD-")) {
					// this counts as a reply for any outstanding async reqs
					MVOBrokerEntry.KeyServerTarget keyServer
						= brEntry.getKeyServer();
					if (keyServer != null) {
						String matchTag
							= m_PartnerMVO + ":" + transId + ":" + seqNum;
						if (matchTag.equals(keyServer.m_MsgTag)) {
							keyServer.m_Error = true;
							keyServer.m_RepReceived = true;

							// cancel any timer
							keyServer.cancel();

							// register a partner error reply on current step
							ReqSvcState stateEngine = brEntry.getStateTracker();
							int failStep = stateEngine.getCurrentStep();
							int lastStep = stateEngine.getPreviousStep();
							stateEngine.advanceState(lastStep, failStep,
													 ReqSvcState.M_GotNack,
													 errText);
						}
						else {
							m_Log.error(lbl + "got ERR from " + m_PartnerMVO
										+ " without a key request tag match");
						}

					}
					else {
						m_Log.error(lbl + "key server error received without a "
									+ "request outstanding, from "
									+ m_PartnerMVO);
					}
				}
				else {
					m_Log.error(lbl + "hit "
								+ "unknown partner type error reply case");
				}
				return;
			}

			// this is a normal message, either a new request or a reply
			String encTxt = msg;
			String partnerSig = msgParts[2];

			// decrypt the message using our pubkey
			MVOConfig config = m_MVOServer.getConfig();
			PrivateKey ourKey = config.getCommPrivKey();
			String decMsg = EncodingUtils.getStrFromBase64PubkeyEnc(encTxt,
																	ourKey);
			if (decMsg == null) {
				m_Log.error(lbl + "message from "
							+ m_PartnerMVO + " with tag " + tag
							+ " did not decrypt properly, must ignore");
				// return a generic processing error (could be attacker)
				sendGenericErr(transId, seqNum,
								"Your message did not decrypt properly");
				return;
			}

			// verify signature
			PublicKey peerKey = config.getPeerPubkey(m_PartnerMVO);
			if (peerKey == null) {
				m_Log.error(lbl + "missing peer pubkey for " + m_PartnerMVO);
			}
			if (!EncodingUtils.verifySignedStr(peerKey, decMsg, partnerSig)) {
				m_Log.error(lbl + "message from "
							+ m_PartnerMVO + " with tag " + tag
							+ " did not sig validate, must ignore");
				// return a generic error (could be attacker)
				sendGenericErr(transId, seqNum,
								"Your message did not pass sig check");
				return;
			}

			// check for an Auditor's key server reply
			if (m_PartnerMVO.startsWith("AUD-")) {
				// this should be in the context of a brokered request
				if (brEntry == null) {
					m_Log.error("Got non-brokered message from " + m_PartnerMVO
								+ " with decrypted text: \"" + decMsg + "\", "
								+ "tId = " + transId);
					return;
				}

				/* The only (valid) case here is that we are receiving a reply
				 * to a key server request we previously submitted to this Aud.
				 * We need to locate the reply object and register it in the
				 * event handler.
				 */
				MVOBrokerEntry.KeyServerTarget keyServer
					= brEntry.getKeyServer();
				if (keyServer != null) {
					String matchTag
						= m_PartnerMVO + ":" + transId + ":" + seqNum;
					if (matchTag.equals(keyServer.m_MsgTag)) {
						// record reply with same tag
						brEntry.setBrokerAudRep(keyServer.m_MsgTag, text);
						keyServer.m_RepReceived = true;

						// cancel any timer
						keyServer.cancel();

						// this should be a AuditorKeyBlock
						ReqSvcState stateEngine = brEntry.getStateTracker();
						AuditorKeyBlock audKeyBlock
							= new AuditorKeyBlock(m_Log);
						if (!audKeyBlock.buildFromString(decMsg)) {
							m_Log.error(lbl + "AUD reply with tag "
										+ matchTag + " does not parse as "
										+ "AuditorKeyBlock");
							keyServer.m_Error = true;
							int failStep = stateEngine.getCurrentStep();
							int lastStep = stateEngine.getPreviousStep();
							stateEngine.advanceState(lastStep, failStep,
													 ReqSvcState.M_ParseError,
							"Auditor key server reply not an AuditorKeyBlock");
						}
						else {
							// record the reply
							// NB: this launches a new thread for processing
							stateEngine.recordKeyServerReply(audKeyBlock);
						}
					}
					else {
						m_Log.error(lbl + "got message from " + m_PartnerMVO
									+ " without a key request tag match");
					}

				}
				else {
					m_Log.error(lbl + "key server error received without a "
								+ "request outstanding, from " + m_PartnerMVO);
				}
				// done
				return;
			}
			// else: message is from a peer MVO
			/* There are two basic cases here:
			 * 1) We have no open broker entry for this transId.  This implies
			 *	  it's a new request coming from a lead MVO.  We need to create
			 *	  the appropriate infrastructure to handle and send a reply.
			 * 2) We do have a broker entry for this transId.  This implies we
			 * 	  are receiving a reply to a request we emitted as lead MVO.
			 */
			if (brEntry == null) {
				// non-lead; should be a MVOVerifyBlock, try to parse it
				MVOVerifyBlock leadMVOVerf = new MVOVerifyBlock(m_Log);
			/*
				m_Log.debug("Got non-brokered message from " + m_PartnerMVO
							+ " with decrypted text: \"" + decMsg + "\"");
			 */
				if (leadMVOVerf.buildFromString(decMsg)) {
					// build broker map entry and add to map
					String repTag = m_PartnerMVO + ":" + transId + ":" + seqNum;
					ClientMVOBlock cBlock = leadMVOVerf.getClientBlock();
					// record original signed client JSON
					cBlock.setDecryptedPayload(
						leadMVOVerf.getClientBlockJson());
					brEntry = new MVOBrokerEntry(cBlock,
											 	 leadMVOVerf.getMVOBlock(),
											 	 repTag,
											 	 m_MVOServer);
					brMap.put(transId, brEntry);
					// replicate transId so we can use for Auditor req/rep tags
					brEntry.setTransId(transId);

					// create proper state object
					DepositState depSt = null;
					SpendState spSt = null;
					WithdrawState withSt = null;
					String opcode = cBlock.getOpcode();
					boolean advOk = true;
					if (opcode.equals(cBlock.M_OpDeposit)) {
						// deposit
						depSt = new DepositState(brEntry);
						brEntry.setStateTracker(depSt);
						depSt.recordSuccess(DepositState.M_ReqReceived);
						// go on to verify request vs SC config
						advOk = depSt.advanceState(DepositState.M_ReqReceived,
												   DepositState.M_ReqVerified,
												   ReqSvcState.M_NoFailure, "");
					}
					else if (opcode.equals(cBlock.M_OpSpend)) {
						// spend
						spSt = new SpendState(brEntry);
						brEntry.setStateTracker(spSt);
						spSt.recordSuccess(SpendState.M_ReqReceived);
						// go on to verify request vs SC config
						advOk = spSt.advanceState(SpendState.M_ReqReceived,
												  SpendState.M_ReqVerified,
												  ReqSvcState.M_NoFailure, "");
					}
					else if (opcode.equals(cBlock.M_OpWithdraw)) {
						// withdraw
						withSt = new WithdrawState(brEntry);
						brEntry.setStateTracker(withSt);
						withSt.recordSuccess(WithdrawState.M_ReqReceived);
						// go on to verify request vs SC config
						advOk = withSt.advanceState(WithdrawState.M_ReqReceived,
												 	WithdrawState.M_ReqVerified,
												 	ReqSvcState.M_NoFailure,
													"");
					}
					else {
						// impossible opcode, shouldn't have parsed
						m_Log.error(lbl + "impossible opcode on MVO request, "
									+ opcode + ", json = \""
									+ leadMVOVerf.getClientBlockJson() + "\"");
						advOk = false;
						// return a generic error
						sendGenericErr(transId, seqNum, "Client opcode "
										+ opcode + " not supported");
					}
					if (!advOk) {
						// error reply already sent by state engine
						m_Log.error(lbl + "due "
									+ "to processing error, abandoning tId "
									+ transId + " opcode " + opcode
									+ " sent from lead " + m_PartnerMVO);
					}
				}
				else {
					m_Log.error(lbl + "unable to interpret request from "
								+ m_PartnerMVO);
					// send error reply
					sendGenericErr(transId, seqNum,
									"MVOVerifyBlock did not parse");
					return;
				}
			}
			else {
				/* This is a reply to a verify request we sent as lead MVO.
				 * Our algorithm here is as follows:
				 * 1) Look for the tag in the broker map entry's m_BrokerMVOReqs
				 * 	  and record this corresponding reply in m_BrokerMVOReps.
				 * 2) Find the MVOCommitteeMember with the tag in its m_MsgTag.
				 * 3) Evaluate the reply for consistency (OB matches, sig good)
				 * 	  and if all is okay, mark m_SigReceived = true.
				 * 4) Examine the rest of the committee and check whether this
				 * 	  is the last reply to be received.  If yes, advance to the
				 * 	  client reply step and cleanup.
				 */
				// recreate the tag we would have used to send request
				String sendTag = m_PartnerMVO + ":" + transId + ":" + seqNum;
				ConcurrentHashMap<String, String> reqs
					= brEntry.getBrokerMVOReqs();
				String ourReq = reqs.get(sendTag);
				if (ourReq == null) {
					m_Log.error(lbl + "no stored "
								+ "req found for tag " + sendTag
								+ " on apparent reply");
				}
				else {
					// record raw reply against same tag
					brEntry.setBrokerMVORep(sendTag, text);
				}

				// get the state object we need to send replies
				ClientMVOBlock clReq
					= (ClientMVOBlock) brEntry.getDappRequest();
				if (clReq == null) {
					m_Log.error(lbl + "CRIT - MVO"
								+ " reply with tag " + sendTag + " has no "
								+ "original client request; cannot process");
					return;
				}
				ReqSvcState stTrack = brEntry.getStateTracker();
				int fromSt = 0;
				int toSt = 0;
				OperationsBlock orgOB = brEntry.getOperationsBlock();
				String op = orgOB.getOpcode();
				if (op.equals(clReq.M_OpDeposit)) {
					fromSt = DepositState.M_OB_Signed;
					toSt = DepositState.M_ReceiptGen;
				}
				else if (op.equals(clReq.M_OpSpend)) {
					fromSt = SpendState.M_OB_Signed;
					toSt = SpendState.M_ReceiptGen;
				}
				else {
					fromSt = WithdrawState.M_OB_Signed;
					toSt = WithdrawState.M_ReceiptGen;
				}

				// search the committee list for the send tag
				ArrayList<MVOBrokerEntry.MVOCommitteeMember> committee
					= brEntry.getCommittee();
				if (committee == null || committee.isEmpty()) {
					// bizarre, and should not occur
					m_Log.error(lbl + "reply found for tag " + sendTag
								+ " but no MVO committee found; dropping");
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ProcError,
										"Missing MVO committee");
					return;
				}
				MVOBrokerEntry.MVOCommitteeMember replier = null;
				for (MVOBrokerEntry.MVOCommitteeMember member : committee) {
					if (member.m_MsgTag.equals(sendTag)) {
						replier = member;
						break;
					}
				}
				if (replier == null) {
					m_Log.error(lbl + "no MVO in committee has tag " + sendTag
								+ "; dropping");
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ProcError,
										"no MVO committee member with tag "
										+ sendTag);
					return;
				}

				// cancel any reply timer this committee member has going
				if (replier.m_Timer != null) {
					replier.m_Timer.cancel();
				}

				// examine the reply, which should be an {OperationsBlock}
				OperationsBlock replyOB = new OperationsBlock(m_Log);
				if (!replyOB.buildFromString(decMsg)) {
					m_Log.error(lbl + "MVO reply "
								+ "with tag " + sendTag + " does not parse as "
								+ "OperationsBlock");
					replier.m_Error = true;
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ParseError,
										"MVO committee reply not an OB");
					return;
				}

				/* This replyOB should contain a signature from the partner MVO.
				 * (It may also contain other signatures, such as ours.)
				 * In order to verify that nothing in the OB has changed from
				 * the original OB which we sent to this committee MVO, we'll
				 * check the supplied signature against both this returned copy
				 * and the original we have stored.
				 */
				// get signature
				ArrayList<MVOSignature> obSigs = replyOB.getSignatures();
				MVOSignature mvoSig
					= MVOSignature.findMVOSig(m_PartnerMVO, obSigs);
				if (mvoSig == null) {
					m_Log.error(lbl + "OB signature on tag " + sendTag
								+ " not available");
					replier.m_Error = true;
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ParseError,
										"MVO committee reply not signed");
					return;
				}

				// obtain the signing key of the partner MVO
				long chId = clReq.getChainId();
				SmartContractConfig scc = m_MVOServer.getSCConfig(chId);
				Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
				MVOConfig mvoConf = (MVOConfig) mvoMap.get(m_PartnerMVO);
				BlockchainConfig bcConf = mvoConf.getChainConfig(chId);
				String signingAddress = "";
				if (bcConf != null) {
					signingAddress = bcConf.getSigningAddress();
				}

				// verify the signature on the OB args hash
				OperationsBlock.ArgumentsHash argHash = replyOB.getArgsBlock();
				String sigAddress = EncodingUtils.getHashSignerAddress(
															argHash.m_ArgsHash,
															mvoSig.m_ArgsSig);
				if (sigAddress.isEmpty() || !sigAddress.equals(signingAddress))
				{
					m_Log.error(lbl + "OB args hash signature on tag " + sendTag
								+ " did not verify");
					replier.m_Error = true;
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ParseError,
								"MVO committee reply args hash sig failure");
					return;
				}

				// first, check that returned OB is the same as the original OB
				String replyOBdata = replyOB.buildSignedData();
				String orgOBdata = orgOB.buildSignedData();
				if (!replyOBdata.equals(orgOBdata)) {
					// committee MVO changed something before signing, invalid
					replier.m_Error = true;
					m_Log.error(lbl + "OB in "
								+ "reply with tag " + sendTag + " does not "
								+ "match original OB:\n" + replyOBdata
								+ "\nversus:\n" + orgOBdata);
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ProcError,
										"MVO committee reply OB match fail");
					return;
				}

				/* Now we'll take the signature (mvoSig.m_Signature) and
				 * validate that it signed the block, returning the MVO's
				 * public address signingAddress.
				 */
				sigAddress = EncodingUtils.getDataSignerAddress(replyOBdata,
															mvoSig.m_Signature);
				if (sigAddress.isEmpty() || !sigAddress.equals(signingAddress))
				{
					replier.m_Error = true;
					m_Log.error(lbl + "OB sig on "
								+ "reply with tag " + sendTag + " failed");
					// proceed to error state
					stTrack.advanceState(fromSt, toSt, ReqSvcState.M_ProcError,
										"MVO committee reply sig check fail");
				}
				else {
					replier.m_SigReceived = true;
					// attach this sig to our master copy, resetting sequence
					mvoSig.m_Sequence = "";
					orgOB.addSignature(mvoSig);

					/* now review whether sigs were received without errors
					 * for the rest of the committee (i.e. if this signature was
					 * the last one to report)
					 */
					boolean sigsDone = true;
					for (MVOBrokerEntry.MVOCommitteeMember member : committee) {
						if (!member.m_SigReceived || member.m_Error) {
							sigsDone = false;
							break;
						}
					}
					if (sigsDone) {
						stTrack.recordSuccess(fromSt);
						// advance to next state (reply, receipt generation)
						stTrack.advanceState(fromSt, toSt,
										     ReqSvcState.M_NoFailure, "");
					}
				}
			}
		}

		/**
		 * handle a binary message (not used)
		 * @param session the session sending the message
		 * @param buf the binary buffer
		 * @param offset where the message starts
		 * @param len message length in bytes
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session,
									   byte[] buf,
									   int offset,
									   int len)
		{
			m_Log.error("ServerWebSocket.onWebSocketMessage: unsupported "
						+ "binary message received, closing " + m_PartnerMVO);
			session.close();
		}

		/**
		 * obtain the session endpoint
		 * @return the actual connected web socket session
		 */
		public WebSocketSession getSession() { return m_Session; }

		/**
		 * obtain the associated MVO ID
		 * @return the ID of the MVO on the other end of the web socket
		 */
		public String getMVOId() { return m_PartnerMVO; }

		/**
		 * obtain whether connection has been authenticated by challenge
		 * @return true if authentication was successful
		 */
		public boolean isAuthenticated() { return m_Authenticated; }

		/**
		 * obtain whether connection was initiated by us (outgoing)
		 * @return true if the MVOClient set up this connection
		 */
		public boolean isOutgoing() { return m_Outgoing; }

		/**
		 * set the session (used for outbound connections)
		 * @param session the session to set
		 */
		public void setSession(WebSocketSession session) {
			if (m_Session != null) {
				m_Log.error("MVOWebSocket.setSession: session already set!");
			}
			m_Session = session;
		}

		/**
		 * configure the session as an outgoing connection
		 * @param outgoing true if outgoing (called by MVOClient)
		 */
		public void setOutgoing(boolean outgoing) { m_Outgoing = outgoing; }
	}

	/**
	 * WebSocket object for incoming messages (server side)
	 */
	private MVOServerWebSocket			m_ServerWS;

	/**
	 * helper class to act as the WebSocket for sessions with client behavior
	 */
	@WebSocket
	public class MVOClientWebSocket {
		// data members
		/**
		 * the actual web socket session
		 */
		private WebSocketSession		m_Session;

		/**
		 * whether our connection has passed encryption challenge yet
		 */
		private boolean					m_Authenticated;

		/**
		 * Whether we initiated this connection (via MVOClient).  This is
		 * always true, but is here simply so the session can tell itself
		 * whether it's a client or server websocket.
		 */
		private boolean					m_Outgoing;

		// methods
		/**
		 * nullary constructor
		 */
		public MVOClientWebSocket() {
			m_Outgoing = true;
		}

		// methods to implement WebSocket
		/**
		 * register new sessions with other MVOs (and Auditors)
		 * @param session the new Session (really a WebSocketSession)
		 */
		@OnWebSocketConnect
		public void onWebSocketConnect(Session session) {
			m_Log.debug("ClientWebSocket.onWebSocketConnect: got connect to "
						+ m_PartnerMVO + " at " + session.getRemoteAddress());
			if (!(session instanceof WebSocketSession)) {
				m_Log.error("ClientWebSocket.onWebSocketConnect: non web "
							+ "socket session received, ignoring");
				if (session != null) {
					session.close(StatusCode.PROTOCOL,
								"Not a websocket session");
				}
				return;
			}
			setSession((WebSocketSession) session);
			m_PingPonger.m_PeerAlive = true;

			// no message context is relevant here; we already know Peer Id
		}

		/**
		 * register closure of existing session
		 * @param session the departing Session (actually a WebSocketSession)
		 * @param status the code with which the Session closed
		 * @param reason textual reason to go with code
		 */
		@OnWebSocketClose
		public void onWebSocketClose(Session session, int status, String reason)
		{
			/* This will automatically be removed from the SessionTracker.
			 * We need to delete this session from the m_WebSocketHandlers.
			 * It's normal to see a message about frame error when the server
			 * side closes the connection.
			 */
			if (!reason.startsWith("Client MUST mask all frames")) {
				m_Log.debug("ClientWebSocket.onWebSocketClose: close of "
							+ "session to " + m_PartnerMVO + ", status = "
							+ status + ", reason = " + reason);
			}
			if (!m_PartnerMVO.isEmpty()) {
				registerClientSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ClientWebSocket.onWebSocketClose: session was not "
							+ "for a known partner node");
			}
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle an exception on a WebSocket session
		 * @param session the session which got the exception
		 * @param cause the exception
		 */
		@OnWebSocketError
		public void onWebSocketError(Session session, Throwable cause) {
			/* A ProtocolException re non-masking of frames is normal when the
			 * server side closes, because it can't write the close message to
			 * us.  So we log no error for that case.
			 * Also, a failure opening due to connection refused or a timeout
			 * needs no log.
			 */
			if (!(cause instanceof ProtocolException)
				&& !(cause instanceof java.net.ConnectException)
				&& !(cause instanceof
					 org.eclipse.jetty.websocket.api.UpgradeException)
				&& !(cause instanceof java.util.concurrent.TimeoutException))
			{
				m_Log.error("ClientWebSocket.onWebSocketError: "
							+ "WebSocket exception: " + cause.toString());
			}

			// remove session from list
			if (!m_PartnerMVO.isEmpty()) {
				registerClientSocket(null);
			}
			else {
				// should be impossible
				m_Log.error("ClientWebSocket.onWebSocketError: session was not "
							+ "for a known partner node");
			}
			// NB: no need to close, Jetty will do it automatically
			m_PingPonger.m_PeerAlive = false;
		}

		/**
		 * handle a text message
		 * @param session the session sending the message
		 * @param text the entire message
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session, String text) {
			/* If this connection is the result of an outbound connection that
			 * was opened by the MVOClient, there's a chance of a race between
			 * the return of the connect() call (Future) and the arrival of the
			 * AUTHREQ message from the other side.  Therefore, if we don't yet
			 * have a m_Session value, we need to set it to the passed param.
			 */
			if (m_Session == null && session instanceof WebSocketSession) {
				m_Log.debug("ClientWebSocket.onWebSocketMessage: no Session "
							+ "yet but received our first text message");
				setSession((WebSocketSession) session);
			}
			if (!(session instanceof WebSocketSession)
				|| session != m_Session)
			{
				m_Log.error("ClientWebSocket.onWebSocketMessage: bad session");
				session.close(StatusCode.PROTOCOL, "Not a websocket session");
				return;
			}

			// no messages should ever be recorded here
			m_Log.error("ClientWebSocket.onWebSocketMessage: unexpected msg "
						+ "received: \"" + text + "\"");
		}

		/**
		 * handle a binary message (not used)
		 * @param session the session sending the message
		 * @param buf the binary buffer
		 * @param offset where the message starts
		 * @param len message length in bytes
		 */
		@OnWebSocketMessage
		public void onWebSocketMessage(Session session,
									   byte[] buf,
									   int offset,
									   int len)
		{
			m_Log.error("ClientWebSocket.onWebSocketMessage: unsupported "
						+ "binary message received, closing " + m_PartnerMVO);
			session.close(StatusCode.BAD_DATA, "Unexpected binary data");
		}

		/**
		 * obtain the session endpoint
		 * @return the actual connected web socket session
		 */
		public WebSocketSession getSession() { return m_Session; }

		/**
		 * obtain the associated MVO ID
		 * @return the ID of the MVO on the other end of the web socket
		 */
		public String getMVOId() { return m_PartnerMVO; }

		/**
		 * obtain whether connection has been authenticated by challenge
		 * @return true if authentication was successful
		 */
		public boolean isAuthenticated() { return m_Authenticated; }

		/**
		 * obtain whether connection was initiated by us (outgoing)
		 * @return true if the MVOClient set up this connection
		 */
		public boolean isOutgoing() { return m_Outgoing; }

		/**
		 * set the session (used for outbound connections)
		 * @param session the session to set
		 */
		public void setSession(WebSocketSession session) {
			if (m_Session != null) {
				m_Log.error("MVOWebSocket.setSession: session already set!");
			}
			m_Session = session;
		}

		/**
		 * configure the session as an outgoing connection
		 * @param outgoing true if outgoing (called by MVOClient)
		 */
		public void setOutgoing(boolean outgoing) { m_Outgoing = outgoing; }

		/**
		 * record that the session is now authenticated
		 * @param auth the authentication status
		 */
		public void setAuthenticated(boolean auth) {
			m_Authenticated = auth;
		}
	}

	/**
	 * WebSocket object for outgoing messages (client side)
	 */
	private MVOClientWebSocket			m_ClientWS;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param mvo the MVO which owns us
	 * @param peer the ID of the peer whose connections we manage
	 */
	public PeerConnector(MVO mvo, String peer) {
		m_MVOServer = mvo;
		m_Log = m_MVOServer.log();
		m_PartnerMVO = peer;
		m_ChallengeData = "";
		m_PingPonger = new PeerKeepAlive();
		m_PingTimer = new Timer("PingPong:" + m_PartnerMVO, true);
		// ping peer once every minute
		m_PingTimer.schedule(m_PingPonger, 60000L, 60000L);
	}

	/**
	 * obtain the socket for the peer which has a server mode behavior
	 * (that is, we listen for incoming messages and send replies)
	 * @return the WebSocket object, if one exists
	 */
	public MVOServerWebSocket getServerSocket() { return m_ServerWS; }

	/**
	 * obtain the socket for the peer which has a client mode behavior
	 * (that is, we send outgoing messages and await replies)
	 * @return the WebSocket object, if one exists
	 */
	public MVOClientWebSocket getClientSocket() { return m_ClientWS; }

	/**
	 * write a string to the outgoing (client) socket, creating the outbound
	 * connection if one does not exist
	 * @param message the message to send
	 * @param callback the method to invoke on write completion, if any
	 * @return true on success
	 */
	public boolean sendString(String message, FutureWriteCallback callback) {
		if (message == null || message.isEmpty()) {
			m_Log.error("PeerConnector.sendString: no message");
			return false;
		}

		// make sure we have a connection to write to
		if (m_ClientWS == null) {
			URI uri = m_MVOServer.getURIforMVO(m_PartnerMVO);
			if (uri == null) {
				m_Log.error("PeerConnector.sendString: no URI found for MVO Id "
							+ m_PartnerMVO);
				return false;
			}
			m_ClientWS = m_MVOServer.getMVOClient().openPeerSocket(
																m_PartnerMVO,
																uri,
																true);
			if (m_ClientWS == null) {
				m_Log.error("PeerConnector.sendString: unable to send msg to "
							+ m_PartnerMVO + " due to failure to open socket");
				return false;
			}
		}

		// send the message
		WebSocketSession wss = m_ClientWS.getSession();
		RemoteEndpoint remote = wss.getRemote();
		/* If a callback is given, remote.sendString() uses a NOOP_CALLBACK
		 * and still sends the message asynchronously.  This is what we
		 * want, because we're allowed up to 65535 (0xFFFF) overlapping
		 * async sends on the endpoint, as opposed to 1 for a sync send.
		 */
		remote.sendString(message, callback);	// asynchronous
		/*
			m_Log.debug("sendString(): sent async message \""
						+ message + "\" to " + m_PartnerMVO);
		 */
		return true;
	}

	/**
	 * record a new server mode socket (any old one will be replaced)
	 * @param socket the websocket to record
	 * @return true on success
	 */
	public boolean registerServerSocket(MVOServerWebSocket socket) {
		if (socket == m_ServerWS) {
			// nothing to do
			return true;
		}

		// socket must have correct mode
		if (socket != null && socket.isOutgoing()) {
			m_Log.error("PeerConnector.registerServerSocket: client mode "
						+ "socket passed");
			return false;
		}

		// purge any existing, unless we're resetting to null
		if (m_ServerWS != null && socket != null) {
			WebSocketSession wss = m_ServerWS.getSession();
			if (wss != null) {
				wss.close(StatusCode.NORMAL, "Connection replaced");
				if (m_ServerWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.registerServerSocket: auth "
								+ "server connection replaced by another");
				}
			}
		}
		m_ServerWS = socket;
		return true;
	}

	/**
	 * record a new client mode socket (any old one will be replaced)
	 * @param socket the websocket to record
	 * @return true on success
	 */
	public boolean registerClientSocket(MVOClientWebSocket socket) {
		if (socket == m_ClientWS) {
			// nothing to do
			return true;
		}

		// socket must have correct mode
		if (socket != null && !socket.isOutgoing()) {
			m_Log.error("PeerConnector.registerClientSocket: server mode "
						+ "socket passed");
			return false;
		}

		// purge any existing
		if (m_ClientWS != null && socket != null) {
			WebSocketSession wss = m_ClientWS.getSession();
			if (wss != null) {
				wss.close(StatusCode.NORMAL, "Connection replaced");
				if (m_ClientWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.registerClientSocket: auth "
								+ "client connection replaced by another");
				}
			}
		}
		m_ClientWS = socket;
		return true;
	}

	/**
	 * shut down connection, doing normal termination on both sockets
	 */
	public void shutdown() {
		// stop pings
		if (m_PingTimer != null) {
			m_PingTimer.cancel();
		}
		m_PingPonger.m_PeerAlive = false;

		// shut down server side connection
		if (m_ServerWS != null) {
			WebSocketSession wss = m_ServerWS.getSession();
			if (wss != null && wss.isOpen()) {
				/* Closing with a message causes a frame mask error, because
				 * this connection is not allowed to do writes.
				 * wss.close(StatusCode.SHUTDOWN, "Connection terminated");
				 */
				if (m_ServerWS != null && m_ServerWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.shutdown: auth server "
								+ "connection ending for " + m_PartnerMVO);
				}
			}
		}
		m_ServerWS = null;

		// shut down client side connection
		if (m_ClientWS != null) {
			WebSocketSession wss = m_ClientWS.getSession();
			if (wss != null && wss.isOpen()) {
				// this one is okay to close
				wss.close(StatusCode.SHUTDOWN,
						"Connection terminated on shutdown");
				if (m_ClientWS != null && m_ClientWS.isAuthenticated()) {
					m_Log.debug("PeerConnector.shutdown: auth client "
								+ "connection ended for " + m_PartnerMVO);
				}
			}
		}
		m_ClientWS = null;
	}

	/**
	 * set the challenge data
	 * @param challenge the data to set
	 */
	protected void setChallenge(String challenge) {
		if (challenge == null) {
			m_ChallengeData = "";
		}
		else {
			m_ChallengeData = challenge;
		}
	}

	/**
	 * obtain the challenge data
	 * @return the last challenge recorded
	 */
	protected String getChallenge() { return m_ChallengeData; }

	/**
	 * return a generic error response, in cases where we don't have a clearly
	 * defined state object to utilize
	 * @param transId the transaction Id context
	 * @param seq the sequence number of the incoming request
	 * @param errText the error message to send
	 */
	protected void sendGenericErr(String transId, String seq, String errText) {
		if (transId == null || seq == null || errText == null) {
			m_Log.error("PeerConnector.sendGenericErr: missing inputs");
			return;
		}
		String errTag = m_PartnerMVO + ":" + transId + ":" + seq;
		String errMsg = errTag + "::ERR" + StatusCode.SERVER_ERROR + "::"
						+ errText;
		sendString(errMsg, null);
	}

	// END methods
}
