/*
 * last modified---
 * 	06-12-23 make m_ChainConfigs a HashMap for efficiency
 * 	09-07-22 derive from MVOGenConfig
 * 	08-04-22 make m_AuditorList a HashMap
 * 	07-12-22 improve error message labeling
 * 	04-22-22 move .BlockchainConfig to its own separate class
 * 	04-21-22 add m_AuditorList + methods
 * 	04-14-22 move to .mvo; add public key Hashtable for other MVOs
 * 	03-23-22 add m_UserDownloadURL, m_StakingAddress, BlockchainConfig
 * 				  class to support multiple chains, add private keys, finalize()
 * 	02-18-22 new
 *
 * purpose---
 * 	encapsulate certain configuration data for an MVO
 */

package cc.enshroud.jetty.mvo;

import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.MVOGenConfig;
import cc.enshroud.jetty.log.Log;

import java.net.URL;
import java.net.URI;
import java.security.PublicKey;
import java.security.PrivateKey;
import java.util.HashMap;
import java.util.Hashtable;


/**
 * This class holds the configuration parameters specific to a particular
 * MVO server.  The values should be loaded from the corresponding records in
 * the smart contract for the blockchain.  Configs will exist for the same
 * MVO in every relevant blockchain's configuration.  Certain fields are set
 * only for the current running MVO; others are set for both it and peers.
 */
public final class MVOConfig extends MVOGenConfig {
	// BEGIN data members
	/**
	 * endpoint where this MVO listens for user requests
	 * (not actually used for anything because we don't talk to ourselves)
	 */
	private URL								m_UserURL;

	/**
	 * endpoint where this MVO listens for user housekeeping requests regarding
	 * receipts and wallet downloads (not actually used for anything because we
	 * don't talk to ourselves)
	 */
	private URL								m_UserDownloadURL;

	/**
	 * configurations for every blockchain supported on this MVO, indexed by
	 * chainId as per chainlist.org
	 */
	private HashMap<Long, BlockchainConfig>	m_ChainConfigs;

	/**
	 * private key used to decrypt and sign Layer2 communications
	 * (only set for self)
	 */
	private PrivateKey						m_CommPrivKey;

	/**
	 * the configs of Auditors we know about, ID mapped to port number
	 * (only set for self)
	 */
	private HashMap<String, Integer>		m_AuditorList;

	/**
	 * the map of public keys for other peer MVOs and Auditors, indexed by ID
	 * (Auditor IDs from m_AuditorList must also be present in this map; only
	 * populated for self)
	 */
	private Hashtable<String, PublicKey>	m_PeerPubkeys;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param mvoId Id of server
	 * @param log logging object
	 */
	public MVOConfig(String mvoId, Log log) {
		super(mvoId, log);
		m_ChainConfigs = new HashMap<Long, BlockchainConfig>();
		m_AuditorList = new HashMap<String, Integer>();
		m_PeerPubkeys = new Hashtable<String, PublicKey>(100);
	}

	// GET methods
	/**
	 * obtain user request URL
	 * @return the endpoint where this MVO listens for proxied user requests
	 */
	public URL getUserURL() { return m_UserURL; }

	/**
	 * obtain user request URL
	 * @return the endpoint where this MVO listens for proxied user requests
	 * related to receipt and wallet downloads
	 */
	public URL getUserDownloadURL() { return m_UserDownloadURL; }

	/**
	 * obtain the configuration for a particular blockchain
	 * @param chain the ID of the chain, per the standard
	 * @return the chain-specific config, or null if not found
	 */
	public BlockchainConfig getChainConfig(long chain) {
		return m_ChainConfigs.get(chain);
	}

	/**
	 * obtain the privkey used to decrypt messages sent to this MVO, and sign
	 * messages originating from it
	 * @return the key used for decrypt/sign of peer L2 messages
	 */
	public PrivateKey getCommPrivKey() { return m_CommPrivKey; }

	/**
	 * look up a pubkey for a given peer
	 * @param peerId the ID of the other MVO or Auditor
	 * @return the public key for that ID, or null if not found
	 */
	public PublicKey getPeerPubkey(String peerId) {
		if (peerId == null || peerId.isEmpty()) {
			return null;
		}
		return m_PeerPubkeys.get(peerId);
	}

	/**
	 * obtain the list of mandatory Auditor IDs and listen port numbers
	 * @return the list, as configured in properties
	 */
	public HashMap<String, Integer> getAuditorList() { return m_AuditorList; }


	// SET methods
	/**
	 * configure the user request endpoint
	 * @param url the relevant network endpoint serving requests
	 */
	public void setUserURL(URL url) {
		if (url != null) {
			m_UserURL = url;
		}
		else {
			m_Log.error("MVOConfig.setUserURL: missing URL");
		}
	}

	/**
	 * configure the user request endpoint for receipts and wallet eNFTs
	 * @param url the relevant network endpoint serving requests
	 */
	public void setUserDownloadURL(URL url) {
		if (url != null) {
			m_UserDownloadURL = url;
		}
		else {
			m_Log.error("MVOConfig.setUserDownloadURL: missing URL");
		}
	}

	/**
	 * configure the private key used to secure messages sent to this MVO
	 * @param privkey the appropriate private key
	 */
	public void configCommPrivkey(PrivateKey privkey) {
		if (privkey != null) {
			m_CommPrivKey = privkey;
		}
		else {
			m_Log.error("MVOConfig.configCommPrivkey: missing key");
		}
	}

	/**
	 * add a blockchain configuration to the list
	 * @param chain the chain Id per the standard
	 * @param name the colloquial name of the blockchain
	 * @return the record added, or null if one already exists
	 */
	public BlockchainConfig addBlockchainConfig(long chain, String name) {
		BlockchainConfig bc = getChainConfig(chain);
		if (bc != null) {
			m_Log.error("MVOConfig.addBlockchainConfig: chain Id " + chain
						+ " already has record for MVOId " + getId());
			return null;
		}
		if (name == null || name.isEmpty()) {
			m_Log.error("MVOConfig.addBlockchainConfig: missing chain name");
			return null;
		}
		bc = new BlockchainConfig(chain, name, m_Log);
		m_ChainConfigs.put(chain, bc);
		return bc;
	}

	/**
	 * add an Auditor Id to the list
	 * @param id the ID to add (from properties)
	 * @param port the port number to construct VPN URI (from properties)
	 */
	public void addAuditor(String id, Integer port) {
		if (id != null && !id.isEmpty()
			&& port != null && port.intValue() > 0)
		{
			m_AuditorList.put(id, port);
		}
	}

	/**
	 * add a public key to the map of peer keys
	 * @param peerId the ID of the MVO or Auditor
	 * @param pubKey their public key
	 * @return previous value, or null if there wasn't one
	 */
	public PublicKey addPeerPubkey(String peerId, PublicKey pubKey) {
		if (peerId == null || peerId.isEmpty() || pubKey == null) {
			return null;
		}
		return m_PeerPubkeys.put(peerId, pubKey);
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			if (m_CommPrivKey != null) {
				m_CommPrivKey.destroy();
			}
			if (m_ChainConfigs != null) {
				// zero blockchain signing keys
				for (BlockchainConfig bc : m_ChainConfigs.values()) {
					bc.configSigningKey(null);
				}
				m_ChainConfigs.clear();
			}
			if (m_PeerPubkeys != null) {
				m_PeerPubkeys.clear();
			}
			if (m_AuditorList != null) {
				m_AuditorList.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
