/*
 * last modified---
 *	02-25-22 new
 *
 * purpose---
 *	class for writing messages to logfiles
 */

package cc.enshroud.jetty.log;

import java.util.Date;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;


/**
 * This class is used for general server logging, for MVOs, Auditors, etc.
 */
public final class Log {
	/**
	 * the file provided for writing logs
	 */
	private File				m_Logfile;

	/**
	 * the output stream opened on the logfile
	 */
	private FileOutputStream	m_LogStrm;

	/**
	 * the formatting stream which writes to the output stream
	 */
	private PrintStream			m_LogOut;

	/**
	 * whether the object has been initialized
	 */
	private boolean				m_Init;

	/**
	 * constructor
	 * @param lfilename the log file filename to write to
	 */
	public Log(String lfilename) {
		m_Logfile = new File(lfilename);
	}

	// get accessor methods
	/**
	 * obtain the log file written to
	 * @return the filesystem object used
	 */
	public File getFile() { return m_Logfile; }

	/**
	 * obtain the PrintStream output
	 * @return the output (tied to file) used for writing messages
	 */
	public PrintStream getLogStream() {
		return m_LogOut;
	}

	/**
	 * initializer -- this reassigns syserr and sysout for the caller
	 * @return true on success, false otherwise
	 */
	public boolean Init() {
		if (m_Logfile == null) {
			return false;
		}
		try {
			// create empty file if it doesn't exist
			if (!m_Logfile.exists()) {
				m_Logfile.createNewFile();
			}
			m_LogStrm = new FileOutputStream(m_Logfile.getPath(), true);
			m_LogOut = new PrintStream(m_LogStrm);
		}
		catch (FileNotFoundException fnfe) {
			System.err.println("Unable to create logfile for writing: "
								 + m_Logfile);
			return false;
		}
		catch (IOException ioe) {
			System.err.println("Unable to initialize logfile: " + m_Logfile);
			return false;
		}
		m_Init = true;
		return true;
	}

	/**
	 * de-initializer
	 */
	public void deInit() {
		m_LogOut.close();
		m_Init = false;
	}

	/**
	 * debug method
	 * @param message the debug message
	 */
	public void debug(String message) {
		if (message == null || message.isEmpty() || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer dOut = new StringBuffer(1024);
		dOut.append("DEBUG " + dnow.toString() + ":\n\t");
		dOut.append(message);
		m_LogOut.println(dOut.toString());
	}

	/**
	 * error method
	 * @param message the error message
	 */
	public void error(String message) {
		if (message == null || message.isEmpty() || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer eOut = new StringBuffer(1024);
		eOut.append("ERROR " + dnow.toString() + ":\n\t");
		eOut.append(message);
		m_LogOut.println(eOut.toString());
	}

	/**
	 * error method
	 * @param message the error message
	 * @param ex exception related to the message
	 */
	public void error(String message, Throwable ex) {
		if (message == null || message.isEmpty() || ex == null || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer eOut = new StringBuffer(1024);
		eOut.append("ERROR " + dnow.toString() + ":\n\t");
		eOut.append(message);
		eOut.append("\nCaused by: ");
		eOut.append(ex.getMessage() + "\n");
		m_LogOut.println(eOut.toString());
		ex.printStackTrace(m_LogOut);
	}

	/**
	 * error method
	 * @param ex exception describing the error
	 */
	public void error(Throwable ex) {
		if (ex == null || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer eOut = new StringBuffer(256);
		eOut.append("ERROR " + dnow.toString() + ":\n\t");
		eOut.append(ex.getMessage() + "\n");
		m_LogOut.println(eOut.toString());
		ex.printStackTrace(m_LogOut);
	}

	/**
	 * warning method
	 * @param message the warning message
	 */
	public void warning(String message) {
		if (message == null || message.isEmpty() || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer wOut = new StringBuffer(1024);
		wOut.append("WARNING " + dnow.toString() + ":\n\t");
		wOut.append(message);
		m_LogOut.println(wOut.toString());
	}

	/**
	 * warning method
	 * @param message the error message
	 * @param ex exception related to the message
	 */
	public void warning(String message, Throwable ex) {
		if (message == null || message.isEmpty() || !m_Init) {
			return;
		}

		Date dnow = new Date();
		StringBuffer wOut = new StringBuffer(1024);
		wOut.append("WARNING " + dnow.toString() + ":\n\t");
		wOut.append(message);
		wOut.append("\nCaused by: ");
		wOut.append(ex.getMessage() + "\n");
		m_LogOut.println(wOut.toString());
		ex.printStackTrace(m_LogOut);
	}
}
