/*
 * last modified---
 * 	02-24-25 add M_MEMO_MAX
 * 	11-16-23 do not complain about empty memo lines
 * 	04-05-23 make address types get emitted as EIP-55
 * 	03-02-23 add calcSignature() and getSigAddr()
 * 	12-01-22 add m_Rand
 * 	07-12-22 improve error message labeling
 * 	06-16-22 do not surround encrypted {enshrouded} with {}, just ""
 * 	05-13-22 add buildSignedData()
 * 	03-30-22 add m_Owner address, to prevent hijacking of valid eNFTs
 * 	03-23-22 emit the ID first, to reduce commonality of initial bytes
 * 	02-22-22 new
 *
 * purpose---
 * 	encapsulate encrypted NFT metadata
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.Keys;

import java.util.Map;
import java.math.BigInteger;
import java.io.IOException;


/**
 * This class holds the metadata necessary to represent Enshrouded NFT values.
 * These data fields are nested in the generic {@link NFTmetadata} object.
 * This class knows how to parse and emit itself as JSON, and can do so in one
 * of two modes: encrypted and unencrypted.
 */
public final class eNFTmetadata implements JSON.Generator {
	// BEGIN data members
	/**
	 * maximum size of memo line fields
	 */
	public final static int	M_MEMO_MAX = 1024;

	/**
	 * the enveloping field, which can be encrypted
	 */
	private String			m_Enshrouded;

	/**
	 * whether the object is currently set for encrypted or decrypted format
	 */
	private boolean			m_Encrypted;

	/**
	 * eNFT ID (a zero-padded 64-digit hex value, represented as a String)
	 */
	private String			m_ID;

	/**
	 * schema version
	 */
	private String			m_Schema;

	/**
	 * the owning address, to which the eNFT was minted (EIP-55)
	 */
	private String			m_Owner;

	/**
	 * token asset represented, given as the smart contract address (EIP-55)
	 */
	private String			m_Asset;

	/**
	 * quantity of asset represented (a uint256 represented as a BigInteger)
	 */
	private BigInteger		m_Amount;

	/**
	 * generation since original backing value deposit
	 */
	private int				m_Generation;

	/**
	 * randomizing data (base64-encoding of 16 random bytes) included to make it
	 * difficult to guess the details hash for this eNFT
	 */
	private String			m_Rand;

	/**
	 * the ID of the MVO who signed the data
	 */
	private String			m_MVOid;

	/**
	 * the actual MVO signature, 130-char hex digest format
	 */
	private String			m_Signature;

	/**
	 * expiration date, as Unix timestamp (OPTIONAL field)
	 */
	private long			m_Expiration;

	/**
	 * growth rate, including units (such as 1 bpd, OPTIONAL field)
	 */
	private String			m_Growth;

	/**
	 * cost rate, including units (such as 5% APR, OPTIONAL field)
	 */
	private String			m_Cost;

	/**
	 * memo line (encrypted to payee)
	 */
	private String			m_Memo;

	/**
	 * logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger logging object
	 */
	public eNFTmetadata(Log logger) {
		m_Log = logger;
		m_Schema = "v1.0";
		m_Enshrouded = m_ID = m_Asset = m_Growth = m_MVOid = m_Signature = "";
		m_Cost = m_Memo = m_Owner = m_Rand = "";
		m_Amount = new BigInteger("0");
		m_Generation = 1;
	}

	// GET methods
	/**
	 * obtain the text of the {enshrouded} field
	 * @return the text of the element, which could be encrypted or not
	 */
	public String getEnshrouded() { return m_Enshrouded; }

	/**
	 * obtain whether the {enshrouded} element is encrypted or not
	 * @return true if encrypted
	 */
	public boolean isEncrypted() { return m_Encrypted; }

	/**
	 * obtain the token ID, by which it's indexed in the blockchain's event log
	 * @return the ID of the eNFT, a zero-padded 64-digit hex number
	 */
	public String getID() { return m_ID; }

	/**
	 * obtain the schema version
	 * @return the revision of the schema used to create this eNFT
	 */
	public String getSchema() { return m_Schema; }

	/**
	 * obtain the token owner
	 * @return the address to which this eNFT was originally minted (EIP-55)
	 */
	public String getOwner() { return m_Owner; }

	/**
	 * obtain the asset
	 * @return the asset (token address) represented by the eNFT (EIP-55)
	 */
	public String getAsset() { return m_Asset; }

	/**
	 * obtain the quantity of the asset, uint256 but represented as a BigInteger
	 * @return the amount of this asset represented by the eNFT
	 */
	public BigInteger getAmount() { return m_Amount; }

	/**
	 * obtain the generation value
	 * @return number of eNFT generations since backing value was deposited
	 */
	public int getGeneration() { return m_Generation; }

	/**
	 * obtain the randomizing data
	 * @return the random data (22 bytes base64(random 16 bytes))
	 */
	public String getRand() { return m_Rand; }

	/**
	 * obtain the signing MVO
	 * @return the MVO's ID
	 */
	public String getSigner() { return m_MVOid; }

	/**
	 * obtain the MVO signature
	 * @return the MVO's signature
	 */
	public String getSignature() { return m_Signature; }

	/**
	 * obtain the expiration date
	 * @return expiration, Unix timestamp (0 if none)
	 */
	public long getExpiration() { return m_Expiration; }

	/**
	 * obtain the growth rate, if any
	 * @return the rate including units, empty if none
	 */
	public String getGrowth() { return m_Growth; }

	/**
	 * obtain the cost rate, if any
	 * @return the rate including units, empty if none
	 */
	public String getCost() { return m_Cost; }

	/**
	 * obtain the memo line
	 * @return the memo passed from depositor/payer to payee
	 */
	public String getMemo() { return m_Memo; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * set the contents of the {enshrouded} element, base64 when encrypted and
	 * JSON when not
	 * @param contents the text of the element
	 */
	public void setEnshrouded(String contents) {
		if (contents != null) {
			m_Enshrouded = new String(contents);
		}
		else {
			m_Log.error("eNFTmetadata.setEnshrouded: missing {enshrouded} "
						+ "contents");
		}
	}

	/**
	 * set whether the object is encrypted
	 * @param crypt true if encrypted
	 */
	public void setEncrypted(boolean crypt) { m_Encrypted = crypt; }

	/**
	 * config the token ID
	 * @param id the id, a 64-digit zero-padded hex number
	 */
	public void setID(String id) {
		if (id != null && !id.isEmpty()) {
			m_ID = new String(id);
		}
		else {
			m_Log.error("eNFTmetadata.setID: missing eNFT ID");
		}
	}

	/**
	 * set the schema revision
	 * @param schema revision string
	 */
	public void setSchema(String schema) {
		if (schema != null && !schema.isEmpty()) {
			m_Schema = new String(schema);
		}
		else {
			m_Log.error("eNFTmetadata.setSchema(): missing schema");
		}
	}

	/**
	 * config the owner address
	 * @param owner the address to which this eNFT was originally minted
	 */
	public void setOwner(String owner) {
		if (owner != null && !owner.isEmpty()) {
			m_Owner = new String(owner);
		}
		else {
			m_Log.error("eNFTmetadata.setOwner(): missing eNFT owning address");
		}
	}

	/**
	 * config the asset
	 * @param asset the asset (token address) represented by the eNFT
	 */
	public void setAsset(String asset) {
		if (asset != null && !asset.isEmpty()) {
			m_Asset = new String(asset);
		}
		else {
			m_Log.error("eNFTmetadata.setAsset(): missing eNFT asset");
		}
	}

	/**
	 * config the quantity of the asset, uint256 but represented as a string
	 * @param amt amount of this asset represented by the eNFT
	 */
	public void setAmount(String amt) {
		if (amt != null && !amt.isEmpty()) {
			try {
				m_Amount = new BigInteger(amt);
			}
			catch (NumberFormatException nfe) {
				m_Log.error("eNFTmetadata.setAmount(): illegal amount, "
							+ amt, nfe);
			}
		}
		else {
			m_Log.error("eNFTmetadata.setAmount(): missing eNFT amount");
		}
	}

	/**
	 * configure the generation
	 * @param gen the number of eNFT generations since deposit
	 */
	public void setGeneration(int gen) {
		if (gen > 0) {
			m_Generation = gen;
		}
		else {
			m_Log.error("eNFTmetadata.setGeneration(): illegal eNFT "
						+ "generation, " + gen);
		}
	}

	/**
	 * configure the randomizing data
	 * @param rand the random data (22 bytes base64(random 16 bytes))
	 */
	public void setRand(String rand) {
		if (rand != null) {
			m_Rand = new String(rand);
		}
	}

	/**
	 * set the expiration date
	 * @param exp the expiration date, as a unix timestamp
	 */
	public void setExpiration(long exp) { m_Expiration = exp; }

	/**
	 * set MVO signer
	 * @param mId ID of signing MVO
	 */
	public void setSigner(String mId) {
		if (mId != null && !mId.isEmpty()) {
			m_MVOid = new String(mId);
		}
		else {
			m_Log.error("eNFTmetadata.setSigner(): missing MVO Id");
		}
	}

	/**
	 * set MVO signature
	 * @param mSig signature of indicated MVO
	 */
	public void setSignature(String mSig) {
		if (mSig != null && !mSig.isEmpty()) {
			m_Signature = new String(mSig);
		}
		else {
			m_Log.error("eNFTmetadata.setSignature(): missing MVO signature");
		}
	}

	/**
	 * set growth rate
	 * @param growth rate plus units
	 */
	public void setGrowth(String growth) {
		if (growth != null) {
			m_Growth = new String(growth);
		}
		else {
			m_Log.error("eNFTmetadata.setGrowth(): missing growth");
		}
	}

	/**
	 * set cost rate
	 * @param cost rate plus units
	 */
	public void setCost(String cost) {
		if (cost != null) {
			m_Cost = new String(cost);
		}
		else {
			m_Log.error("eNFTmetadata.setCost(): missing cost");
		}
	}

	/**
	 * set memo line
	 * @param memo the memo line passed to payee (empty ones are ok)
	 */
	public void setMemo(String memo) {
		if (memo != null) {
			m_Memo = new String(memo);
		}
	}

	/**
	 * method to build object from a string (this implies object is encrypted)
	 * @param propData the properties data (this should be base64 format)
	 */
	public void buildFromString(String propData) {
		setEnshrouded(propData);
		setEncrypted(true);
	}

	/**
	 * method to build object from a JSON map
	 * @param mapData the properties data (unencrypted)
	 * @return true on success
	 */
	public boolean buildFromMap(Map mapData) {
		if (mapData == null || mapData.isEmpty()) {
			return false;
		}
		Object ensh = mapData.get("enshrouded");
		Map enshMap = null;
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if ((ensh instanceof Map)) {
			enshMap = (Map) ensh;
			setEncrypted(false);
		}
		else if (ensh instanceof String) {
			// simply record string and we're done
			String enshEncStr = (String) ensh;
			buildFromString(enshEncStr);
			return true;
		}
		else {
			m_Log.error(lbl + "missing {enshrouded}");
			return false;
		}

		boolean ret = true;
		// required field: id
		Object id = enshMap.get("id");
		if (id instanceof String) {
			setID((String)id);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT ID");
		}

		// required field: schema
		Object sch = enshMap.get("schema");
		if (sch instanceof String) {
			setSchema((String)sch);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT schema");
		}

		// required field: owner address
		Object own = enshMap.get("owner");
		if (own instanceof String) {
			setOwner((String)own);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT owner address");
		}

		// required field: asset
		Object asset = enshMap.get("asset");
		if (asset instanceof String) {
			m_Asset = new String((String)asset);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT asset");
		}

		// required field: amount
		Object amt = enshMap.get("amount");
		if (amt instanceof String) {
			setAmount((String)amt);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT amount");
		}

		// optional field (early test eNFTs did not have this): randomizing data
		Object rand = enshMap.get("rand");
		if (rand instanceof String) {
			setRand((String)rand);
		}

		// required field: generation
		Object gen = enshMap.get("generation");
		if (gen instanceof String) {
			String gention = (String) gen;
			try {
				setGeneration(new Integer(gention.trim()));
			}
			catch (NumberFormatException nfe) {
				ret = false;
				m_Log.error(lbl + "illegal eNFT generation, " + gen, nfe);
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT generation");
		}

		// required field: signer
		Object signer = enshMap.get("signer");
		if (signer instanceof String) {
			setSigner((String)signer);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT signer");
		}

		// required field: signature
		Object sig = enshMap.get("signature");
		if (sig instanceof String) {
			setSignature((String)sig);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing eNFT signature");
		}

		// optional field: expiration
		Object exp = enshMap.get("expiration");
		if (exp instanceof String) {
			String expire = (String) exp;
			try {
				setExpiration(Long.parseLong(expire.trim()));
			}
			catch (NumberFormatException nfe) {
				ret = false;
				m_Log.error(lbl + "illegal expiration timestamp, " + exp, nfe);
			}
		}

		// optional field: growth
		Object growth = enshMap.get("growth");
		if (growth instanceof String) {
			setGrowth((String)growth);
		}

		// optional field: cost
		Object cost = enshMap.get("cost");
		if (cost instanceof String) {
			setCost((String)cost);
		}

		// optional field: memo
		Object memo = enshMap.get("memo");
		if (memo instanceof String) {
			setMemo((String)memo);
		}
		return ret;
	}

	/**
	 * build the portion of the object which is actually signed
	 * @return the signed data, or null on error (object cannot be encrypted)
	 */
	public String buildSignedData() {
		StringBuilder signed = new StringBuilder(1024);
		if (m_Encrypted) {
			m_Log.error("eNFTmetadata.buildSignedData: cannot sign encrypted "
						+ "eNFTmetadata");
			return null;
		}
		signed.append("\"id\":\"" + m_ID + "\",");
		signed.append("\"schema\":\"" + m_Schema + "\",");
		signed.append("\"owner\":\"" + Keys.toChecksumAddress(m_Owner) + "\",");
		signed.append("\"asset\":\"" + Keys.toChecksumAddress(m_Asset) + "\",");
		signed.append("\"amount\":\"" + m_Amount + "\",");
		if (!m_Rand.isEmpty()) {
			signed.append("\"rand\":\"" + m_Rand + "\",");
		}
		signed.append("\"generation\":\"" + m_Generation + "\",");
		if (m_Expiration > 0L) {
			signed.append("\"expiration\":\"" + m_Expiration + "\",");
		}
		if (!m_Growth.isEmpty()) {
			signed.append("\"growth\":\"" + m_Growth + "\",");
		}
		if (!m_Cost.isEmpty()) {
			signed.append("\"cost\":\"" + m_Cost + "\",");
		}
		if (!m_Memo.isEmpty()) {
			signed.append("\"memo\":\"" + m_Memo + "\",");
		}
		return signed.toString();
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the Appendable data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		// emit without any whitespace; doesn't need to be sorted
		StringBuilder out = new StringBuilder(1024);
		if (!m_Encrypted) {
			out.append("\"enshrouded\":{");
			out.append(buildSignedData());
			// add signer and sig if these exist
			if (!m_MVOid.isEmpty()) {
				out.append("\"signer\":\"" + m_MVOid + "\",");
			}
			if (!m_Signature.isEmpty()) {
				out.append("\"signature\":\"" + m_Signature + "\"");
			}
			out.append("}");
		}
		else {
			out.append("\"enshrouded\":");
			out.append("\"" + m_Enshrouded + "\"");
		}
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("eNFTmetadata.addJSON(): exception appending", ioe);
		}
	}

	/**
	 * calculate a signature for this eNFT (must not be signed already)
	 * @param sigKey the private key for the signature
	 * @return the signature, in 130-char hex digest format (null on failure)
	 */
	public String calcSignature(BigInteger privKey) {
		final String lbl = this.getClass().getSimpleName() + ".calcSignature: ";
		if (privKey == null) {
			m_Log.error(lbl + "missing sig key");
			return null;
		}
		String sigData = buildSignedData();
		if (sigData == null) {
			// can't sign if still in encrypted format
			m_Log.error(lbl + "cannot build signed data");
			return null;
		}

		// build an Ethereum-prefixed hash for signature
		String ethHash = EncodingUtils.ethPrefixHash(sigData);

		// sign with key
		String sig = EncodingUtils.signHash(ethHash, privKey);
		if (sig == null) {
			m_Log.error(lbl + "error generating signature");
		}
		return sig;
	}

	/**
	 * obtain the address which signed this eNFT
	 * @return the address, in 0x format (empty on failure)
	 */
	public String getSigAddr() {
		final String lbl = this.getClass().getSimpleName() + ".getSigAddr: ";
		String sigAddr = "";

		// eNFT must be signed
		if (m_Signature.isEmpty()) {
			m_Log.error(lbl + "eNFT does not appear to be signed");
			return sigAddr;
		}

		String sigData = buildSignedData();
		if (sigData == null) {
			// can't sign if still in encrypted format
			m_Log.error(lbl + "cannot build signed data");
			return sigAddr;
		}

		// build Ethereum-prefixed hash for verification
		String ethHash = EncodingUtils.ethPrefixHash(sigData);

		// determine the address which produced the signature
		sigAddr = EncodingUtils.getHashSignerAddress(ethHash, m_Signature);
		if (sigAddr.isEmpty()) {
			m_Log.error(lbl + "unable to recover signing address");
		}
		return sigAddr;
	}

	// END methods
}
