/*
 * last modified---
 * 	04-27-24 handle SQLIntegrityConstraintViolationException to catch an
 * 			 insert by another AUD after we checked for existence
 * 	03-14-23 generic DB classes now in .db
 * 	10-25-22 differentiate between getStatus() returning empty vs. null
 * 	10-11-22 add DETAILS_HASH column
 * 	10-10-22 add NARRATIVE column
 * 	10-04-22 add support for multiple inserts/mods in a single transaction
 * 	09-27-22 new
 *
 * purpose---
 * 	provide an API to manipulate records in the ENSH_ENFT_STATUS table
 */

package cc.enshroud.jetty.aud.db;

import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.GenericDb;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.db.DbConnectionManager;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.util.Date;
import java.util.ArrayList;


/**
 * This class supplies atomic operations which act on the ENSH_ENFT_STATUS table
 * in the Auditor's database.  These methods correspond to logical operations,
 * e.g. create, get.  They closely interoperate with the class that
 * instantiates a row in the table, {@link EnftStatus EnftStatus}.  Note there
 * is no deletion method, as this is not required.  Read-write methods allow
 * for passed connections to permit use within a transaction.
 */
public final class EnftStatusDb extends GenericDb {
	// BEGIN data members

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param manager the DB connection manager
	 * @param logger the error/debug logger
	 */
	public EnftStatusDb(DbConnectionManager manager, Log logger) {
		super(manager, logger);
		m_Table = "ENSH_ENFT_STATUS";
	}

	/**
	 * obtain the status records for eNFTs
	 * @param Ids the IDs of the eNFTs whose status is sought
	 * @param chainId the ID of the relevant blockchain
	 * @param conn DB connection to use (create one if passed as null)
	 * @return the list of status records for these IDs, empty if not found, or
	 * null on DB errors
	 */
	public ArrayList<EnftStatus> getStatus(ArrayList<String> Ids,
										   long chainId,
										   Connection conn)
	{
		final String lbl = this.getClass().getSimpleName() + ".getStatus: ";
		if (Ids == null || Ids.isEmpty() || chainId <= 0L) {
			m_Log.error(lbl + "missing input");
			return null;
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				return null;
			}
		}
		ArrayList<EnftStatus> statRecs = new ArrayList<EnftStatus>(Ids.size());

		// build and execute query
		StringBuilder getSql = new StringBuilder(1024);
		getSql.append("select * from " + m_Table + " where ENFT_ID in (");
		int idCnt = 1;
		for (String id : Ids) {
			getSql.append("?");
			if (idCnt++ < Ids.size()) {
				getSql.append(",");
			}
		}
		getSql.append(") and CHAIN_ID=? order by INSERT_DATE;");
		PreparedStatement query = null;
		ResultSet stRS = null;
		try {
			query = dbConn.prepareStatement(getSql.toString());
			idCnt = 1;
			for (String id : Ids) {
				query.setString(idCnt++, id);
			}
			query.setLong(idCnt, chainId);
			stRS = query.executeQuery();
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error running eNFT status query for "
						+ m_Table, sqle);
			statRecs = null;
		}

		// examine results and load a return object
		try {
			while (statRecs != null && stRS != null && stRS.next()) {
				String eId = stRS.getString("ENFT_ID");
				EnftStatus statRec = new EnftStatus(eId, chainId);
				statRec.setDetailsHash(stRS.getString("DETAILS_HASH"));
				statRec.setStatus(stRS.getString("STATUS"));
				statRec.setSubmitter(stRS.getString("SUBMITTER"));
				statRec.setNarrative(stRS.getString("NARRATIVE"));
				statRec.setInsertDate(stRS.getTimestamp("INSERT_DATE"));
				statRec.setUpdater(stRS.getString("UPDATER"));
				statRec.setLmodDate(stRS.getTimestamp("LMOD_DATE"));
				statRecs.add(statRec);
			}
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "error parsing result set for " + m_Table, sqle);
			statRecs = null;
		}
		finally {
			// don't close a passed connection
			if (conn != null) {
				m_DbManager.closeResultSet(stRS);
				m_DbManager.closeStatement(query);
			}
			else {
				m_DbManager.closeConnection(stRS, query, dbConn);
			}
		}
		return statRecs;
	}

	/**
	 * create a status record for an eNFT
	 * @param eId the ID of the eNFT
	 * @param chainId the ID of the blockchain
	 * @param aud the Auditor making the entry
	 * @param hash details hash (keccak256("[{address},{id},{asset},{amount}]")
	 * @param stat the initial status to set (null for database default)
	 * @param reason an optional reason for setting this status
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on success, false on bad input
	 * @throws EnshDbException on all database errors, dup record, etc.
	 */
	public boolean setStatus(String eId,
							 long chainId,
							 String aud,
							 String hash,
							 String stat,
							 String reason,
							 Connection conn)
		throws EnshDbException
	{
		final String lbl = this.getClass().getSimpleName() + ".setStatus: ";
		// check inputs
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain ID");
			return false;
		}
		if (eId == null || eId.isEmpty()) {
			m_Log.error(lbl + "missing eNFT ID");
			return false;
		}
		if (aud == null || !aud.startsWith("AUD-")) {
			m_Log.error(lbl + "illegal Auditor ID");
			return false;
		}
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing details hash");
			return false;
		}
		if (reason == null) {
			reason = "";
		}
		String status = EnftStatus.M_Valid;
		if (stat != null && !stat.isEmpty()) {
			status = stat;
		}

		// another Auditor may have already done this
		ArrayList<String> ID = new ArrayList<String>(1);
		ID.add(eId);
		ArrayList<EnftStatus> existStats = getStatus(ID, chainId, conn);
		if (existStats != null && !existStats.isEmpty()) {
			EnftStatus existStat = existStats.get(0);
			// make sure status is the same
			if (status.equals(existStat.getStatus())) {
				// nothing to do here
				return true;
			}
		}

		// get a DB connection if we don't have one
		Connection dbConn = conn;
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				throw new EnshDbException("unable to get database connection");
			}
		}

		// build and execute SQL insertion string
		String insSql = "insert into " + m_Table + " set ENFT_ID=?, "
						+ "CHAIN_ID=?, DETAILS_HASH=?, SUBMITTER=?, "
						+ "INSERT_DATE=now()";
		if (stat != null) {
			insSql += ", STATUS=?";
		}
		// no reason is required when status = M_Valid
		if (!reason.isEmpty() && !EnftStatus.M_Valid.equals(status)) {
			insSql += ", NARRATIVE=?";
		}
		insSql += ";";
		PreparedStatement statInsert = null;
		int insCnt = 0;
		try {
			int nxtFld = 1;
			statInsert = dbConn.prepareStatement(insSql);
			statInsert.setString(nxtFld++, eId);
			statInsert.setLong(nxtFld++, chainId);
			statInsert.setString(nxtFld++, hash);
			statInsert.setString(nxtFld++, aud);
			if (stat != null) {
				statInsert.setString(nxtFld++, stat);
			}
			if (!reason.isEmpty() && !stat.equals(EnftStatus.M_Valid)) {
				statInsert.setString(nxtFld, reason);
			}
			insCnt = statInsert.executeUpdate();
		}
		catch (SQLException sqle) { 
			if (sqle instanceof SQLIntegrityConstraintViolationException
				&& sqle.getMessage().startsWith("Duplicate entry"))
			{
				/* This implies that another AUD performed the insertion in
				 * parallel with us, but _after_ our check for existence above
				 * using getStatus().  Treat this as success.
				 */
				m_Log.debug(lbl + "DUP entry for eId " + eId
							+ ", assuming another AUD did parallel insert");
			}
			else {
				m_Log.error(lbl + "error running insert for " + m_Table, sqle);
				if (conn != null) {
					m_DbManager.closeStatement(statInsert);
				}
				else {
					m_DbManager.closeConnection(statInsert, dbConn);
				}
				throw new EnshDbException(sqle.getMessage(), insSql,
										  sqle.getErrorCode());
			}
		}
		finally {
			// don't close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(statInsert);
			}
			else {
				m_DbManager.closeConnection(statInsert, dbConn);
			}
		}
		if (insCnt != 1) {
			m_Log.error(lbl + "unexpected insert count on " + m_Table
						+ ": " + insCnt);
			throw new EnshDbException("Bad insert count, " + insCnt, insSql);
		}
		return true;
	}

	/**
	 * update a status record for an eNFT
	 * @param eId the ID of the eNFT (zero-padded hex)
	 * @param chainId the ID of the blockchain
	 * @param aud the Auditor making the update
	 * @param stat the new status to set
	 * @param reason a reason for setting this status (required for M_Suspect)
	 * @param conn DB connection to use (create one if passed as null)
	 * @return true on success, false on bad input (no change is success)
	 * @throws EnshDbException on all database errors, no record, etc.
	 */
	public boolean updStatus(String eId,
							 long chainId,
							 String aud,
							 String stat,
							 String reason,
							 Connection conn)
		throws EnshDbException
	{
		final String lbl = this.getClass().getSimpleName() + ".updStatus: ";
		// check inputs
		if (chainId <= 0L) {
			m_Log.error(lbl + "missing chain ID");
			return false;
		}
		if (eId == null || eId.isEmpty()) {
			m_Log.error(lbl + "missing eNFT ID");
			return false;
		}
		if (aud == null || !aud.startsWith("AUD-")) {
			m_Log.error(lbl + "illegal Auditor ID");
			return false;
		}
		if (stat == null || (!stat.equals(EnftStatus.M_Valid)
							 && !stat.equals(EnftStatus.M_Suspect)
							 && !stat.equals(EnftStatus.M_Blocked)
							 && !stat.equals(EnftStatus.M_Deleted)))
		{
			m_Log.error(lbl + "illegal status value, " + stat);
			return false;
		}
		// reason required when updating to M_Suspect
		if (EnftStatus.M_Suspect.equals(stat)
			&& (reason == null || reason.isEmpty()))
		{
			m_Log.error(lbl + "missing reason for status " + stat);
			return false;
		}

		// another Auditor may have already done this
		ArrayList<String> ID = new ArrayList<String>(1);
		ID.add(eId);
		ArrayList<EnftStatus> existStats = getStatus(ID, chainId, conn);
		if (existStats != null && !existStats.isEmpty()) {
			EnftStatus existStat = existStats.get(0);
			// make sure status is the same
			if (stat.equals(existStat.getStatus())) {
				// nothing to do here
				return true;
			}
		}

		// get a DB connection if one was not passed
		Connection dbConn = getDbConnection();
		if (dbConn == null) {
			dbConn = getDbConnection();
			if (dbConn == null) {
				throw new EnshDbException("unable to get database connection");
			}
		}

		// build and execute SQL update string
		String updSql = "update " + m_Table + " set STATUS=?, "
						+ "UPDATER=?, LMOD_DATE=now()";
		if (reason != null && !reason.isEmpty()) {
			updSql += ", NARRATIVE=?";
		}
		updSql += " where ENFT_ID=? and CHAIN_ID=?;";
		PreparedStatement statUpd = null;
		int updCnt = 0;
		try {
			statUpd = dbConn.prepareStatement(updSql);
			statUpd.setString(1, stat);
			statUpd.setString(2, aud);
			if (reason != null && !reason.isEmpty()) {
				statUpd.setString(3, reason);
				statUpd.setString(4, eId);
				statUpd.setLong(5, chainId);
			}
			else {
				statUpd.setString(3, eId);
				statUpd.setLong(4, chainId);
			}
			updCnt = statUpd.executeUpdate();
		}
		catch (SQLException sqle) { 
			m_Log.error(lbl + "error running update for " + m_Table, sqle);
			if (conn != null) {
				m_DbManager.closeStatement(statUpd);
			}
			else {
				m_DbManager.closeConnection(statUpd, dbConn);
			}
			throw new EnshDbException(sqle.getMessage(), updSql,
									  sqle.getErrorCode());
		}
		finally {
			// don't close a passed persistent connection
			if (conn != null) {
				m_DbManager.closeStatement(statUpd);
			}
			else {
				m_DbManager.closeConnection(statUpd, dbConn);
			}
		}
		// we must have updated something (LMOD_DATE if nothing else)
		if (updCnt != 1) {
			m_Log.error(lbl + "unexpected update count on " + m_Table
						+ ": " + updCnt);
			throw new EnshDbException("Bad update count, " + updCnt, updSql);
		}
		return true;
	}

	// END methods
}
