/*
 * last modified---
 * 	01-06-26 remove some debug on eNFT key deletions
 * 	03-28-25 cosmetic
 * 	01-17-24 complete implementation of user authorization checking
 * 	09-27-23 check user sigs and hash authorizations more fully
 * 	06-13-23 use actual verifyClientSig(), copied from AudBlockHandler
 * 	03-20-23 implement checks against original client request on get & delete;
 * 			 add verifyClientSig()
 * 	03-14-23 generic DB classes now in .db
 * 	12-14-22 log every successful MVO request
 * 	10-04-22 remove temp version of handleAuditorBlock()
 * 	09-28-22 new, drawn from file-based MVOKeyServer
 *
 * purpose---
 * 	provide an Auditor's key server function and response
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.MVOKeyBlock;
import cc.enshroud.jetty.AuditorKeyBlock;
import cc.enshroud.jetty.ClientRequest;
import cc.enshroud.jetty.ClientReceiptBlock;
import cc.enshroud.jetty.ClientWalletBlock;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.MVOSignature;
import cc.enshroud.jetty.EncodingUtils;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.DbConnectionManager;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.aud.db.AESKey;
import cc.enshroud.jetty.aud.db.EnftKeysDb;
import cc.enshroud.jetty.aud.db.ReceiptKeysDb;

import java.util.ArrayList;
import java.util.Date;
import java.util.Base64;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.sql.Savepoint;
import java.sql.Connection;
import java.sql.SQLException;
import javax.crypto.SecretKey;


/**
 * This class provides the functionality of the Auditor as a key server.  It
 * uses a central database, maintained among Auditors.  Methods are provided
 * to create, look up, and delete keys, in both the eNFT mapping and the
 * receipt mapping.
 */
public final class MVOKeyServer {
	// BEGIN data members
	/**
	 * the character used to join components of key hashes
	 */
	private final String  		M_JoinChar = "+";

	/**
	 * owning Auditor
	 */
	private AUD					m_AUD;

	/**
	 * error logging object (inherited from parent)
	 */
	private Log					m_Log;

	/**
	 * database connection manager (copied from owning AUD)
	 */
	private DbConnectionManager	m_DbMgr;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param aud the Auditor which owns us
	 */
	public MVOKeyServer(AUD aud) {
		m_AUD = aud;
		m_Log = m_AUD.log();
		m_DbMgr = m_AUD.getDbManager();
	}

	/**
	 * initializer (a stub in this version)
	 * @return true on success
	 */
	public boolean init() { return true; }

	/**
	 * shutdown handler (a stub in this version)
	 * @return true on success
	 */
	public boolean stop() { return true; }

	/**
	 * backup key list to database (a stub in this version)
	 * @return true on success
	 */
	public boolean backup() { return true; }

	/**
	 * get a new key in String form
	 * @return the AES256 key, base64url encoded, or null on error
	 */
	private String getNewAESkey() {
		SecretKey sekrit
			= EncodingUtils.genAES256Key(m_AUD.getStateObj().getRNG());
		if (sekrit == null) {
			m_Log.error("MVOKeyServer.getNewAESkey: could not generate new "
						+ "AES key");
			return null;
		}
		byte[] keyEncoding = sekrit.getEncoded();
		if (keyEncoding == null) {
			m_Log.error("MVOKeyServer.getNewAESkey: no encoding bytes for "
						+ "new AES key");
			return null;
		}
		Base64.Encoder b64e = Base64.getUrlEncoder();
		return b64e.encodeToString(keyEncoding);
	}

	/**
	 * get an existing key from the database
	 * @param eNFT true if key is for an eNFT, false if for receipt
	 * @param hash the hash at which key is stored
	 * @return the String representation of the key, or null if not found
	 */
	public String getKey(boolean eNFT, String hash) {
		final String lbl = this.getClass().getSimpleName() + ".getKey: ";
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing hash");
		}
		String retKey = null;
		if (!eNFT) {
			ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);
			AESKey rctKey = rctDb.getKey(hash, null);
			if (rctKey == null) {
				m_Log.error(lbl + "receipt key not found for hash " + hash);
			}
			else {
				retKey = rctKey.getKey();
			}
		}
		else {
			EnftKeysDb enftDb = new EnftKeysDb(m_DbMgr, m_Log);
			AESKey nftKey = enftDb.getKey(hash, null);
			if (nftKey == null) {
				m_Log.error(lbl + "eNFT key not found for hash " + hash);
			}
			else {
				retKey = nftKey.getKey();
			}
		}
		return retKey;
	}

	/**
	 * generate a new key, first checking it doesn't already exist
	 * @param eNFT true if the key is for an eNFT, false if for a receipt
	 * @param hash the hash where the key will be stored
	 * @return the new key value
	 */
	public String getNewKey(boolean eNFT, String hash) {
		final String lbl = this.getClass().getSimpleName() + ".getNewKey: ";
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing new key hash");
			return null;
		}

		// generate the new key
		String newKey = getNewAESkey();
		if (newKey == null) {
			m_Log.error(lbl + "error generating AES key");
			return null;
		}

		// add it to the database for the appropriate table
		// (NB: if key already exists the insert will fail)
		try {
			if (eNFT) {
				EnftKeysDb enftDb = new EnftKeysDb(m_DbMgr, m_Log);
				if (!enftDb.insertKey(hash, newKey, null)) {
					m_Log.error(lbl + "error inserting eNFT key hash " + hash);
					return null;
				}
			}
			else {
				ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);
				if (!rctDb.insertKey(hash, newKey, null)) {
					m_Log.error(lbl + "error inserting receipt key hash "
								+ hash);
					return null;
				}
			}
		}
		catch (EnshDbException edbe) {
			m_Log.error(lbl + "exception inserting key hash " + hash, edbe);
			return null;
		}
		return newKey;
	}

	/**
	 * generate a set of new keys, first checking each doesn't already exist
	 * @param eNFT true if the key is for an eNFT, false if for a receipt
	 * @param hashes the hashes where the keys will be stored
	 * @return the new key values, null on errors (means no keys were inserted)
	 */
	public ArrayList<AESKey> getNewKeys(boolean eNFT,
										ArrayList<String> hashes)
	{
		final String lbl = this.getClass().getSimpleName() + ".getNewKey: ";
		if (hashes == null || hashes.isEmpty()) {
			m_Log.error(lbl + "missing new key hashes");
			return null;
		}

		// if we're doing more than one key, create a bracketing transaction
		Connection dbConn = null; 
		Savepoint savePt = null;
		boolean needRollback = false;
		if (hashes.size() > 1) {
			// get a DB connection on which we can unset auto-commit
			try {
				dbConn = m_DbMgr.getConnection();
				dbConn.setAutoCommit(false);
				savePt = dbConn.setSavepoint("newKeys");
			}
			catch (SQLException se) {
				m_Log.error(lbl + "can't turn off auto-commit or establish a "
							+ "savepoint for new key generation", se);
				return null;
			}
		}

		ArrayList<AESKey> newKeys = new ArrayList<AESKey>(hashes.size());
		EnftKeysDb enftDb = new EnftKeysDb(m_DbMgr, m_Log);
		ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);

		// process each hash
		for (String hash : hashes) {
			// generate the new key value
			String newKey = getNewAESkey();
			if (newKey == null) {
				m_Log.error(lbl + "error generating AES key");
				needRollback = true;
				break;
			}

			// add it to the database for the appropriate table, within trans
			// (NB: if key already exists the insert will fail)
			try {
				if (eNFT) {
					if (!enftDb.insertKey(hash, newKey, dbConn)) {
						m_Log.error(lbl
									+ "error inserting eNFT key hash " + hash);
						needRollback = true;
						break;
					}
				}
				else {
					if (!rctDb.insertKey(hash, newKey, dbConn)) {
						m_Log.error(lbl + "error inserting receipt key hash "
									+ hash);
						needRollback = true;
						break;
					}
				}
				AESKey key = new AESKey(hash, newKey);
				newKeys.add(key);
			}
			catch (EnshDbException edbe) {
				m_Log.error(lbl + "exception inserting key hash " + hash, edbe);
				needRollback = true;
				break;
			}
		}

		// if we never set a transaction or savepoint, we're done
		if (dbConn == null || savePt == null) {
			return newKeys;
		}

		// if an error occurred, roll back to savepoint
		if (needRollback) {
			try {
				dbConn.rollback(savePt);
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error rolling back key insert trans to save "
							+ "point!", se);
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
				newKeys = null;
			}
		}
		else {
			// commit all inserts and return list
			try {
				dbConn.commit();
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error committing key insert trans", se);
				newKeys = null;
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
			}
		}
		return newKeys;
	}

	/**
	 * remove a key from the database
	 * @param eNFT true if key is for an eNFT, false if for receipt
	 * @param hash the index to remove
	 * @return the key which was stored there, or null if none or on failures
	 */
	public String removeKey(boolean eNFT, String hash) {
		final String lbl = this.getClass().getSimpleName() + ".removeKey: ";
		if (hash == null || hash.isEmpty()) {
			m_Log.error(lbl + "missing hash");
			return null;
		}
		String oldKey = getKey(eNFT, hash);
		if (oldKey != null) {
			// delete from DB
			if (eNFT) {
				EnftKeysDb enftDb = new EnftKeysDb(m_DbMgr, m_Log);
				if (!enftDb.purgeKey(hash, null)) {
					m_Log.error(lbl + "error deleting eNFT key hash " + hash);
					return null;
				}
			}
			else {
				ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);
				if (!rctDb.purgeKey(hash, null)) {
					m_Log.error(lbl + "error deleting rct key hash " + hash);
					return null;
				}
			}
		}
		return oldKey;
	}

	/**
	 * remove a list of keys from the database
	 * @param eNFT true if keys are for eNFTs, false if for receipt
	 * @param hashes the indices to remove
	 * @return list of hash/key values removed, null on fatal errors
	 */
	public ArrayList<AESKey> removeKeys(boolean eNFT,
										ArrayList<String> hashes)
	{
		final String lbl = this.getClass().getSimpleName() + ".removeKey: ";
		if (hashes == null || hashes.isEmpty()) {
			m_Log.error(lbl + "missing hashes");
			return null;
		}

		// if we're doing more than one key, create a bracketing transaction
		Connection dbConn = null; 
		Savepoint savePt = null;
		boolean needRollback = false;
		if (hashes.size() > 1) {
			// get a DB connection on which we can unset auto-commit
			try {
				dbConn = m_DbMgr.getConnection();
				dbConn.setAutoCommit(false);
				savePt = dbConn.setSavepoint("delKeys");
			}
			catch (SQLException se) {
				m_Log.error(lbl + "can't turn off auto-commit or establish a "
							+ "savepoint for key deletion", se);
				return null;
			}
		}

		EnftKeysDb enftDb = new EnftKeysDb(m_DbMgr, m_Log);
		ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);
		ArrayList<AESKey> delKeys = new ArrayList<AESKey>(hashes.size());

		// process all hashes
		for (String hash : hashes) {
			AESKey oldKey = null;
			// delete from appropriate DB table
			if (eNFT) {
				oldKey = enftDb.getKey(hash, dbConn);
				if (oldKey != null) {
					if (!enftDb.purgeKey(hash, dbConn)) {
						m_Log.error(lbl + "error deleting eNFT key hash "
									+ hash);
						needRollback = true;
						break;
					}
					else {
						m_Log.debug(lbl + "deleted eNFT key hash " + hash
									+ "; prev value was " + oldKey.getKey());
						AESKey delKey = oldKey;
						delKeys.add(delKey);
					}
				}
				// else: key did not exist, no need to delete
			}
			else {
				oldKey = rctDb.getKey(hash, dbConn);
				if (oldKey != null) {
					if (!rctDb.purgeKey(hash, dbConn)) {
						m_Log.error(lbl + "error deleting rct key hash "
									+ hash);
						needRollback = true;
						break;
					}
					else {
						AESKey delKey = oldKey;
						delKeys.add(delKey);
					}
				}
				// else: key did not exist, no need to delete
			}
		}

		// if we never set a transaction or savepoint, we're done
		if (dbConn == null || savePt == null) {
			return delKeys;
		}

		// if an error occurred, roll back to savepoint
		if (needRollback) {
			try {
				dbConn.rollback(savePt);
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error rolling back key delete trans to save "
							+ "point!", se);
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
				delKeys = null;
			}
		}
		else {
			// commit all inserts and return list
			try {
				dbConn.commit();
				// because our connections are pooled, reset auto-commit
				dbConn.setAutoCommit(true);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "error committing key delete trans", se);
				delKeys = null;
			}
			finally {
				m_DbMgr.closeConnection(dbConn);
			}
		}
		return delKeys;
	}

	/**
	 * process a received {@link MVOKeyBlock MVOKeyBlock} sent by an MVO
	 * @param keyBlock the key block, describing a request
	 * @return the {@link AuditorKeyBlock} Auditor's response, which may be an
	 * error (has status=error text), or null (which prevents processing)
	 */
	public AuditorKeyBlock handleKeyBlock(MVOKeyBlock keyBlock) {
		final String lbl
			= this.getClass().getSimpleName() + ".handleKeyBlock: ";
		if (keyBlock == null) {
			m_Log.error(lbl + "missing MVOKeyBlock");
			return null;
		}
		StringBuilder keyBl = new StringBuilder(2048);
		keyBlock.addJSON(keyBl);
		AuditorKeyBlock retBlock = new AuditorKeyBlock(m_Log);

		// determine which mapping the MVO request involves
		boolean receipts
			= keyBlock.getMapping().equals(keyBlock.M_MapReceipt);

		// validate the MVO's signature on the input keyBlock
		MVOSignature mvoSig = keyBlock.getSignature();
		if (mvoSig == null) {
			m_Log.error(lbl + "no MVO signature found");
			return null;
		}
		String signedData = keyBlock.buildSignedData();
		PublicKey verfKey = m_AUD.getConfig().getPeerPubkey(mvoSig.m_Signer);
		if (verfKey == null) {
			m_Log.error(lbl + "no pubkey for signer " + mvoSig.m_Signer
						+ " to check signature");
			return null;
		}
		if (!EncodingUtils.verifySignedStr(verfKey, signedData,
										   mvoSig.m_Signature))
		{
			m_Log.error(lbl + "signature did not verify for signer "
						+ mvoSig.m_Signer + "; sig data was: " + signedData);
			return null;
		}

		// NB: from here we don't return null, but set an error return message
		String errTxt = "";

		// see what the opcode is
		String opcode = keyBlock.getOpcode();
		ArrayList<String> inputHashes = keyBlock.getHashList();
		boolean gotErr = false;
		final String badHash = "one or more key hashes not client-authorized";

		// verify sig on original client request (always done)
		ClientRequest cliReq = keyBlock.getClientBlock();
		if (!verifyClientSig(keyBlock.getClientBlockJson(), cliReq)) {
			errTxt = "clientAuth signature did not verify";
			m_Log.error(lbl + errTxt);
			gotErr = true;
		}

		/* In cases where we're creating new receipts or eNFTs, there
		 * is no need to do anything beyond validating the user's
		 * signed request.  (That is, there's no way to confirm that
		 * the user explicitly authorized access to the new objects.)
		 *
		 * In cases where existing records are being accessed (and then
		 * decrypted using the fetched keys), we (the Auditor node) must
		 * make sure that the client specifically authorized access to
		 * the relevant records.  This is done by computing the hashes
		 * again from data in the signed object and comparing them to
		 * the passed hash list.  These situations are:
		 * 	1. spendspec.inputs[] - lookup of keys for existing eNFTs
		 * 	2. withdrawspec.inputs[] - lookup of keys for existing eNFTs
		 * 	3. receiptRequest.filespecs[] - for Get or Del of receipts
		 * 	4. walletDownload.IDList[] - lookup of keys for existing eNFTs
		 * 	(note normal case in last situation is an empty list == all)
		 *
		 * The keyBlock.M_OpNew case is always considered okay.
		 */

		/* Handle receipts case.  On two opcodes we must verify compliance
		 * with client's request.  (List only returns receiptID.json filenames.)
		 */
		if (!gotErr && receipts && (keyBlock.M_OpGet.equals(opcode)
			 			 			|| keyBlock.M_OpDelete.equals(opcode)))
		{
			ClientReceiptBlock rBlock = (ClientReceiptBlock) cliReq;
			// build list of all authorized hashes
			String sender = rBlock.getSender().toLowerCase();
			String chain = Long.toString(rBlock.getChainId());
			ArrayList<String> authHashes
				= new ArrayList<String>(inputHashes.size());
			ArrayList<String> hashComps = new ArrayList<String>(3);

			// examine all passed filespecs
			for (String file : rBlock.getFileSpecs()) {
				int jsonIdx = file.indexOf(".json");
				if (jsonIdx == -1) {
					m_Log.error(lbl + "illegal fileSpec, " + file);
					gotErr = true;
					break;
				}
				// parse receipt Id from filename
				String rId = file.substring(0, jsonIdx);
				hashComps.add(chain);
				hashComps.add(rId);
				hashComps.add(sender);
				String keyIdx = String.join(M_JoinChar, hashComps);
				String keyHash = EncodingUtils.sha3(keyIdx);
				authHashes.add(keyHash);
				hashComps.clear();
			}

			// verify every hash requested is referenced in orig req
			for (String hash : inputHashes) {
				if (!authHashes.contains(hash)) {
					m_Log.error(lbl + "hash " + hash + " not authorized");
					gotErr = true;
				}
			}
			if (gotErr) {
				errTxt = badHash;
			}
		}

		// handle other cases verifying inputs[] (i.e. for eNFT key gets)
		else if (!gotErr && !receipts && keyBlock.M_OpGet.equals(opcode)) {
			// check for deposit/spend/withdraw operations
			if (cliReq instanceof ClientMVOBlock) {
				ClientMVOBlock mBlock = (ClientMVOBlock) cliReq;
				ArrayList<ClientMVOBlock.ClientInput> inputs
					= mBlock.getInputs();
				String mOpcode = mBlock.getOpcode();
				if (inputs.isEmpty()) {
					// must be a deposit operation
					if (!mOpcode.equals(ClientMVOBlock.M_OpDeposit)) {
						m_Log.error(lbl + "opcode " + mOpcode
									+ " without any hash inputs");
						gotErr = true;
						errTxt = mOpcode + " opcode without inputs";
					}
				}
				else {
					// build list of all authorized hashes
					String sender = mBlock.getSender().toLowerCase();
					String chain = Long.toString(mBlock.getChainId());
					ArrayList<String> authHashes
						= new ArrayList<String>(inputHashes.size());
					ArrayList<String> hashComps = new ArrayList<String>(3);

					// examine all passed input eNFTs
					for (ClientMVOBlock.ClientInput input : inputs) {
						hashComps.add(chain);
						hashComps.add(input.m_eNFT.getID());
						hashComps.add(sender);
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						authHashes.add(keyHash);
						hashComps.clear();
					}

					/* There are two cases we could be in here:
					 *
					 * 1) Any MVO (lead or not) doing a fetch on the keys for
					 * eNFTs passed in the dApp's client request.  These must
					 * *all* be stipulated as inputs in the user's signed req.
					 *
					 * 2) A non-lead MVO doing a fetch on the keys for *output*
					 * eNFTs in an operation.  (Could be any of the three, but
					 * we eliminated deposit above due to empty inputs.)  In
					 * this case, because the lead MVO previously sent a M_OpNew
					 * request to create these keys, *none* will be found in
					 * the user's signed request, since the eNFTs didn't even
					 * exist at the time the dApp signed it.
					 *
					 * Therefore, to differentiate between these cases, either
					 * all must be found, or none must be found.  A partial
					 * find is evidence of an error.
					 */
					boolean foundAHash = false;
					boolean foundEveryHash = true;
					for (String hash : inputHashes) {
						if (authHashes.contains(hash)) {
							foundAHash = true;
						}
						else {
							foundEveryHash = false;
						}
					}
					if (foundAHash && !foundEveryHash) {
						gotErr = true;
						m_Log.error(lbl + "only a subset of hashes found");
					}
					if (gotErr) {
						errTxt = badHash;
					}
				}
			}

			// check for a wallet operation
			else if (cliReq instanceof ClientWalletBlock) {
				ClientWalletBlock wBlock = (ClientWalletBlock) cliReq;
				ArrayList<String> eIds = wBlock.getIDs();
				/* nothing to do here unless a specific subset of eNFT IDs was
				 * supplied by the user (NB: currently the dApp doesn't do this)
				 */
				if (!eIds.isEmpty()) {
					// build list of all authorized hashes
					String sender = wBlock.getSender().toLowerCase();
					String chain = Long.toString(wBlock.getChainId());
					ArrayList<String> authHashes
						= new ArrayList<String>(inputHashes.size());
					ArrayList<String> hashComps = new ArrayList<String>(3);

					// examine all passed eNFT Ids
					for (String eId : eIds) {
						hashComps.add(chain);
						hashComps.add(eId);
						hashComps.add(sender);
						String keyIdx = String.join(M_JoinChar, hashComps);
						String keyHash = EncodingUtils.sha3(keyIdx);
						authHashes.add(keyHash);
						hashComps.clear();
					}

					// verify every hash requested is referenced in orig req
					for (String hash : inputHashes) {
						if (!authHashes.contains(hash)) {
							m_Log.error(lbl + "hash " + hash
										+ " not authorized");
							gotErr = true;
						}
					}
					if (gotErr) {
						errTxt = badHash;
					}
				}
				// else: normal case
			}
			else {
				gotErr = true;
				m_Log.error(lbl + "unsupported ClientRequest type, "
							+ cliReq.getClass().getSimpleName());
			}
		}

		// if no errors, process opcode
		if (!gotErr) {
			if (keyBlock.M_OpNew.equals(opcode)) {
				// ask for a new key for each hash
				ArrayList<AESKey> aesKeys = getNewKeys(!receipts, inputHashes);
				if (aesKeys == null) {
					errTxt = "error generating new keys";
					m_Log.error(lbl + errTxt);
					gotErr = true;
				}
				else {
					for (AESKey newKey : aesKeys) {
						AuditorKeyBlock.KeySpec newSpec
							= retBlock.new KeySpec();
						newSpec.m_Hash = newKey.getHash();
						newSpec.m_KeyData = newKey.getKey();
						// NB: buildSignedData() adds the index
						retBlock.addKey(newSpec);
					}
				}
			}
			else if (keyBlock.M_OpGet.equals(opcode)) {
				// get existing key for all hashes, as a block
				if (!receipts) {
					EnftKeysDb ekDb = new EnftKeysDb(m_DbMgr, m_Log);
					ArrayList<AESKey> aesKeys = ekDb.getKeys(inputHashes);
					if (aesKeys == null) {
						errTxt = "error fetching eNFT keys";
						m_Log.error(lbl + errTxt);
						gotErr = true;
					}
					else {
						for (AESKey newKey : aesKeys) {
							AuditorKeyBlock.KeySpec oldSpec
								= retBlock.new KeySpec();
							oldSpec.m_Hash = new String(newKey.getHash());
							oldSpec.m_KeyData = new String(newKey.getKey());
							// NB: buildSignedData() adds the index
							retBlock.addKey(oldSpec);
						}
					}
				}
				else {
					ReceiptKeysDb rctDb = new ReceiptKeysDb(m_DbMgr, m_Log);
					ArrayList<AESKey> aesKeys = rctDb.getKeys(inputHashes);
					if (aesKeys == null) {
						errTxt = "error fetching receipt keys";
						m_Log.error(lbl + errTxt);
						gotErr = true;
					}
					else {
						for (AESKey newKey : aesKeys) {
							AuditorKeyBlock.KeySpec oldSpec
								= retBlock.new KeySpec();
							oldSpec.m_Hash = new String(newKey.getHash());
							oldSpec.m_KeyData = new String(newKey.getKey());
							// NB: buildSignedData() adds the index
							retBlock.addKey(oldSpec);
						}
					}
				}
			}
			else if (keyBlock.M_OpDelete.equals(opcode)) {
				/* In practice, the MVO never generates a delete request for
				 * deletion of eNFT keys, only receipt keys.  But in theory
				 * the protocol supports it.  This could however be extremely
				 * dangerous (in the sense of a malicious dApp / MVO action).
				 * Therefore we catch it here.
				 *
				 * NB: The correct way to purge old eNFT keys is some amount of
				 * time after the corresponding eNFT was burned on-chain. (TBD)
				 */
				if (!receipts) {
					gotErr = true;
					m_Log.error(lbl + "CRITICAL: attempt to delete eNFT keys "
								+ "sent by MVO for user " + cliReq.getSender());
					errTxt = "illegal attempt to delete eNFT keys";
				}
				else {
					// remove each key hash
					ArrayList<AESKey> deletedKeys
						= removeKeys(!receipts, inputHashes);
					if (deletedKeys == null) {
						errTxt = "error deleting keys";
						m_Log.error(lbl + errTxt);
						gotErr = true;
					}
					else {
						for (AESKey delKey : deletedKeys) {
							// key was removed, echo it in response
							AuditorKeyBlock.KeySpec delSpec
								= retBlock.new KeySpec();
							delSpec.m_Hash = delKey.getHash();
							delSpec.m_KeyData = delKey.getKey();
							// NB: buildSignedData() adds the index
							retBlock.addKey(delSpec);
						}
					}
				}
			}
			else {
				errTxt = "illegal opcode, " + opcode;
				m_Log.error(lbl + errTxt);
				gotErr = true;
			}
		}

		// send back error if we got one (overriding "success" as status)
		if (gotErr) {
			retBlock.setStatus(errTxt);
		}
		else {
			// in accordance with the white paper, log the MVO's request
			StringBuilder hashList = new StringBuilder(inputHashes.size() * 65);
			int hashIdx = 0;
			for (String hash : inputHashes) {
				if (++hashIdx < inputHashes.size()) {
					hashList.append(hash + ",");
				}
				else {
					hashList.append(hash);
				}
			}
			m_Log.debug("KEYLOG - " + mvoSig.m_Signer + " "
						+ (receipts ? "receipt" : "eNFT")
						+ " key request: opcode = " + opcode
						+ "; hash list = " + hashList.toString());
		}

		// affix our signature
		AuditorKeyBlock.AuditorSignature aSig = retBlock.new AuditorSignature();
		aSig.m_Signer = m_AUD.getAUDId();
		signedData = retBlock.buildSignedData();
		PrivateKey privKey = m_AUD.getConfig().getCommPrivKey();
		String sig = EncodingUtils.signStr(privKey, signedData);
		if (sig != null) {
			aSig.m_Signature = sig;
			retBlock.setSignature(aSig);
		}
		else {
			m_Log.error(lbl + "error signing AuditorKeyBlock reply");
			errTxt = "error signing reply";
			retBlock.setStatus(errTxt);
		}
		return retBlock;
	}

	/**
	 * helper method to verify client block signatures
	 * @param requestJson the entire JSON string received from the user
	 * @param cliRequest the parsed ClientReceiptBlock
	 * @return true if client's signature was valid
	 */
	private boolean verifyClientSig(String requestJson,
								    ClientRequest cliRequest)
	{
		if (cliRequest == null || requestJson == null || requestJson.isEmpty())
		{
			m_Log.error("MVOKeyServer.verifyClientSig: missing input");
			return false;
		}

		/* to extract the signed portion, take everything up to:
		 * ',\"signature\":'
		 */
		int sigIdx = requestJson.lastIndexOf(",\"signature\":");
		if (sigIdx == -1) {
			// json was not signed
			m_Log.error("MVOKeyServer.verifyClientSig: missing signature");
			return false;
		}
		String eip712Data = requestJson.substring(0, sigIdx) + "}";

		// obtain the signature from full input data
		String rawSig = requestJson.substring(sigIdx);
		int sigStart = rawSig.lastIndexOf(":\"");
		// strip off leading :" and trailing "}
		String signature = rawSig.substring(sigStart+2, rawSig.length()-2);
		String signingAddress = cliRequest.getSender();

		// determine which address actually signed the data
		String actualSigAddr
			= EncodingUtils.getEIP712SignerAddress(eip712Data, signature);
		return actualSigAddr.equalsIgnoreCase(signingAddress);
	}

	// END methods
}
