/*
 * last modified---
 * 	09-15-25 use chain-specific logfile, since greylisting is chain-specific
 * 	04-04-25 fix DB errors if greylist or ungreylist lists are empty
 * 	02-12-25 if greyListAuditAtStartup() called to early, use StartupAuditRetry
 * 	09-20-24 in greyListAuditAtStartup(), require only greylist add/delete done
 * 	12-20-23 draft complete
 * 	12-14-23 new
 *
 * purpose---
 * 	provide a mechanism for greylisting eNFT IDs on-chain
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.SmartContractConfig;
import cc.enshroud.jetty.BlockchainConfig;
import cc.enshroud.jetty.EnftListener;
import cc.enshroud.jetty.EnftCache;
import cc.enshroud.jetty.ClientMVOBlock;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.DbConnectionManager;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.wrappers.EnshroudProtocol;
import cc.enshroud.jetty.aud.db.EnftStatus;
import cc.enshroud.jetty.aud.db.EnftStatusDb;

import org.web3j.crypto.Credentials;
import org.web3j.crypto.ECKeyPair;
import org.web3j.crypto.Keys;
import org.web3j.utils.Numeric;
import org.web3j.protocol.core.methods.response.TransactionReceipt;
import org.web3j.protocol.core.methods.response.EthGetBalance;
import org.web3j.protocol.core.DefaultBlockParameterName;
import org.web3j.tx.gas.StaticEIP1559GasProvider;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.Timer;
import java.util.TimerTask;
import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Savepoint;


/**
 * Class to handle sending greyList requests to the blockchain.  This class
 * bundles up sets of output IDs into "buckets," which are all greylisted at
 * once in order to save gas.  It implements EnftListener to receive events
 * regarding mints etc., and takes action accordingly.  It also manages
 * database and EnftCache queries, and DB status updates.  It supports a method
 * to audit greylisted status vs "suspect" status in DB upon startup, which
 * may have been changed on-chain during downtime.  Actual on-chain greylisting
 * requests utilize Credentials for the specific chain (built from the private
 * key), and manage gas limits and retries, using individual execution threads.
 * There will be one GreyLister object for each blockchain the AUD supports.
 * (Note each AUD is supposed to support every chain where Enshroud deploys.)
 * In the event of large numbers of bogus IDs being minted (implying an attack
 * coordinated by multiple MVOs), multiple AUD nodes will start at opposite ends
 * of the list and avoid stepping on each other or repeating each other's work.
 */
public final class GreyLister implements EnftListener {
	// BEGIN data members
	/**
	 * the AUD object which owns us
	 */
	private AUD						m_AUD;

	/**
	 * the cache for this chain (also provides SmartContractConfig)
	 */
	private EnftCache				m_Cache;

	/**
	 * our initialized EnshroudProtocol wrapper object
	 */
	private EnshroudProtocol		m_Wrapper;

	/**
	 * logging object
	 */
	private Log						m_Log;

	/**
	 * inner class to represent a "bucket" of IDs which are associated with the
	 * same transaction output (and thus should be greylisted all at once if
	 * any of them are ever minted)
	 */
	private final class GreylistBucket implements Runnable {
		/**
		 * Interval to wait between on-chain greylist attempts (msec).
		 * TBD: this should perhaps be configurable on a per-chain basis.
		 */
		private final long				M_RETRY_INTERVAL = 60000L;

		/**
		 * the list of eNFT IDs we're supposed to greyList in this batch
		 */
		public ArrayList<BigInteger>	m_GreyListIDs;

		/**
		 * the reason for greylisting this group (same for all IDs)
		 */
		public String					m_Reason;

		/**
		 * status of bucket:
		 * 1 = pending (waiting for an ID in the bucket to get minted)
		 * 2 = in progress (greylisting initiated)
		 * 3 = completed (bucket will be deleted after this stage reached)
		 */
		private final int				M_PENDING = 1;
		private final int				M_PROGRESS = 2;
		private final int				M_COMPLETE = 3;
		private int						m_Status;

		/**
		 * thread which is running to attempt on-chain updates, if any
		 */
		public Thread					m_Greylister;

		/**
		 * stop flag, set to make thread exit
		 */
		public boolean					m_StopFlag;

		/**
		 * constructor
	 	 * @param idList the IDs to greylist
		 * @param reason the reason to greylist these IDs
		 */
		public GreylistBucket(ArrayList<BigInteger> idList, String reason) {
			m_GreyListIDs = idList;
			m_Reason = reason;
			m_Status = M_PENDING;
		}

		/**
		 * trigger run attempt (can be called from recordEnftMint(),
		 * recordBlock(), or as a retry after failure within m_Greylister.run())
		 * @return false on failure
		 */
		public boolean doGreyListing() {
			if (m_StopFlag) {
				// we're supposed to exit, so do nothing
				return true;
			}
			if (m_Status != M_PENDING) {
				return false;
			}

			// record we are in progress
			m_Status = M_PROGRESS;

			// spawn thread and return to caller
			m_Greylister = new Thread(this);
			m_Greylister.start();
			return true;
		}

		// implement Runnable to do actual work on-chain
		public void run() {
			final String lbl = this.getClass().getSimpleName() + ".run: ";
			if (m_Cache == null) {
				m_Log.error(lbl + "no EnftCache passed to constructor, abort");
				return;
			}

			// check all IDs individually to see if they're already done
			ArrayList<BigInteger> delIds
				= new ArrayList<BigInteger>(m_GreyListIDs.size());
			for (BigInteger eId : m_GreyListIDs) {
				/* See whether ID is in the greylisted list in cache.  Note
				 * that this check reflects the net status, based on the final
				 * outcome of possible multiple greylist/ungreylist operations.
				 */
				if (m_Cache.isIdGreyListed(eId)) {
					/* This means it's greylisted on-chain in a past block (not
					 * just in our local AudDb).  In this case either a client
					 * is (uselessly) trying shenanigans, or another AUD beat
					 * us to it a while back.  Either way, no point in further
					 * processing.
					 */
					delIds.add(eId);
				}
			}
			if (!delIds.isEmpty()) {
				// remove already-greylisted items
				m_GreyListIDs.removeAll(delIds);
			}
			if (m_GreyListIDs.isEmpty()) {
				m_Log.debug(lbl + "all IDs to be greylisted already done");
				m_Buckets.remove(this);
				return;
			}
			// construct an array copying the (same) reason for each ID
			ArrayList<String> glReasons
				= new ArrayList<String>(m_GreyListIDs.size());
			for (int idx = 0; idx < m_GreyListIDs.size(); idx++) {
				glReasons.add(m_Reason);
			}

			// do actual greylisting on the blockchain
			boolean gotErr = false;
			TransactionReceipt greyListRes = null;
			try {
				greyListRes = m_Wrapper.auditorGreyList(m_AUD.getAUDId(),
														m_GreyListIDs,
														glReasons)
					.send();

				// record success
				if (greyListRes.isStatusOK()) {
					m_Status = M_COMPLETE;
					String gP = Numeric.cleanHexPrefix(
											greyListRes.getEffectiveGasPrice());
					BigInteger gasPrice = new BigInteger(gP, 16);
					m_Log.debug(lbl + "greyListed " + m_GreyListIDs.size()
								+ " bad output Ids, txHash = "
								+ greyListRes.getTransactionHash()
								+ ", block " + greyListRes.getBlockNumber()
								+ ", gas used = " + greyListRes.getGasUsed()
								+ ", @price " + gasPrice);
				}
				else {
					m_Log.error(lbl + "bad status returned from greyList tx, \""
								+ greyListRes.getRevertReason() + "\"");
					gotErr = true;
				}
			}
			catch (Exception eee) {
				gotErr = true;
				if (greyListRes != null) {
					m_Log.error(lbl + "reason for greyList failure: \""
								+ greyListRes.getRevertReason() + "\"", eee);
				}
				else {
					m_Log.error(lbl + "TransactionReceipt was null, exception "
								+ "follows", eee);
				}
			}

			if (gotErr) {
				// log error
				StringBuilder idList = new StringBuilder(1024);
				for (BigInteger idFail : m_GreyListIDs) {
					idList.append(idFail.toString(16) + ",");
				}
				m_Log.error(lbl + "CRITICAL: unable to greyList failed "
							+ "output IDs! List: " + idList.toString());

				// see if we've been shut down while submitting trans on-chain
				if (m_StopFlag) return;

				// wait interval and retry
				try {
					Thread.sleep(M_RETRY_INTERVAL);
				}
				catch (InterruptedException ie) { /* ignore */ }

				// see if we've been shut down while sleeping
				if (m_StopFlag) return;

				/* shouldn't happen, but cover the case where another event
				 * started a doGreyListing() attempt and succeeded while we
				 * were asleep
				 */
				if (m_Status != M_COMPLETE) {
					// reset to try again
					m_Status = M_PENDING;
					doGreyListing();
					return;
				}
			}

			// change DB status from 'suspect' to 'blocked' for all IDs
			DbConnectionManager dbMgr = m_AUD.getDbManager();
			EnftStatusDb statDb = new EnftStatusDb(dbMgr, m_Log);
			// start a transaction with a savepoint
			Connection dbConn = null;
			Savepoint savePt = null;
			boolean needRollback = false;
			// get a DB connection on which we can unset auto-commit
			try {
				dbConn = dbMgr.getConnection();
				dbConn.setAutoCommit(false);
				long savId = m_AUD.getStateObj().getNextAUDid();
				savePt = dbConn.setSavepoint("greyListStatus" + savId);
			}
			catch (SQLException se) {
				m_Log.error(lbl + "can't turn off commit for STATUS trans", se);
				return;
			}

			// update all records
			long chainId = m_Cache.getConfig().getChainId();
			String us = m_AUD.getAUDId();
			for (BigInteger eId : m_GreyListIDs) {
				boolean dbErr = false;
				String ID = Numeric.toHexStringNoPrefixZeroPadded(eId, 64);
				try {
					if (!statDb.updStatus(ID, chainId, us,
										  EnftStatus.M_Blocked, "", dbConn))
					{
						dbErr = true;
						m_Log.error(lbl + "unable to update ID " + ID
									+ " status to " + EnftStatus.M_Blocked);
					}
				}
				catch (EnshDbException edbe) {
					m_Log.error(lbl + "exception updating ID " + ID
								+ " status to " + EnftStatus.M_Blocked, edbe);
					dbErr = true;
				}
				if (dbErr) {
					needRollback = true;
				}
			}

			// commit DB changes if any
			if (needRollback) {
				try {
					dbConn.rollback(savePt);
					// because our connections are pooled, reset auto-commit
					dbConn.setAutoCommit(true);
				}
				catch (SQLException se) {
					m_Log.error(lbl + "error rolling back to save point!", se);
				}
				finally {
					dbMgr.closeConnection(dbConn);
				}
			}
			else {
				try {
					dbConn.commit();
					// because our connections are pooled, reset auto-commit
					dbConn.setAutoCommit(true);
				}
				catch (SQLException se) {
					m_Log.error(lbl + "error committing status blocked trans",
								se);
				}
				finally {
					dbMgr.closeConnection(dbConn);
				}
			}

			// remove ourselves from the list of buckets to be processed
			m_Buckets.remove(this);
		}
	}

	/**
	 * inner class to retry calls to greyListAuditAtStartup() as required
	 */
	private final class StartupAuditRetry extends TimerTask {
		/**
		 * delay between retry attempts
		 */
		public final long		M_GREYLIST_RETRY = 30000L;

		/**
		 * nullary constructor
		 */
		public StartupAuditRetry() {
			super();
		}

		/**
		 * task run on expiration of wait interval
		 */
		@Override
		public void run() {
			final String lbl = this.getClass().getSimpleName() + ".run: ";
			// see if we're GTG yet
			if (m_Cache.isGreyListingComplete()) {
				if (greyListAuditAtStartup()) {
					m_Log.debug(lbl + "startup greylist audit succeeded");
				}
				else {
					m_Log.debug(lbl + "startup greylist audit failed");
				}
				// either way, we don't invoke again
			}
			else {
				m_Log.debug(lbl + m_Cache.getConfig().getChainName()
							+ " EnftCache still not ready; rescheduling "
							+ "startup greylist audit");
				// replace cancelled task
				m_RetryGreylistAudit = new StartupAuditRetry();
				// schedule retry after interval
				Timer retrier = new Timer("StartupAuditRetry", true);
				retrier.schedule(m_RetryGreylistAudit, M_GREYLIST_RETRY);
			}
		}
	}

	/**
	 * timer task to do startup greylist audit retries
	 */
	private StartupAuditRetry						m_RetryGreylistAudit;

	/**
	 * list of buckets related to this chain
	 */
	private ConcurrentLinkedQueue<GreylistBucket>	m_Buckets;

	/**
	 * map of block numbers to buckets where we saw any unknown eNFTs get minted
	 */
	private Hashtable<BigInteger, GreylistBucket>	m_SuspectBlocks;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param aud the owning AUD object
	 * @param cache the cache object for this chain
	 */
	public GreyLister(AUD aud, EnftCache cache) {
		m_AUD = aud;
		m_Cache = cache;
		m_Log = m_Cache.getConfig().log();
		m_RetryGreylistAudit = new StartupAuditRetry();
		m_Buckets = new ConcurrentLinkedQueue<GreylistBucket>();
		m_SuspectBlocks = new Hashtable<BigInteger, GreylistBucket>();
	}

	/**
	 * initialization method
	 * @return true on success
	 */
	public boolean initialize() {
		/* initialize EnshroudProtocol contract wrapper using our
		 * actual credentials and a real EIP-1559 gas provider
		 */
		// get the deployed EnshroudProtocol contract for this chain
		if (m_Cache == null) {
			m_Log.error("GreyLister.initialize: no EnftCache set");
			return false;
		}
		SmartContractConfig scc = m_Cache.getConfig();
		Long chainId = scc.getChainId();
		String protocolAddr = EnshroudProtocol.getPreviouslyDeployedAddress(
															chainId.toString());
		HashMap<Long, BlockchainConfig> chainConfigs
			= m_AUD.getConfig().getChainConfigs();
		BlockchainConfig bcConf = chainConfigs.get(chainId);
		/* Configure gas limits, used for sending greylisting requests.
		 * We set both a gwei limit and a general gas limit.
		 * TBD: we should be able to vary these settings on a per-chain basis.
		 */
		BigInteger maxGwei = new BigInteger("200");
		BigInteger maxPriFee = new BigInteger("2");
		BigInteger gasLimit = new BigInteger("2000000");
		StaticEIP1559GasProvider gasProvider
			= new StaticEIP1559GasProvider(chainId, maxGwei, maxPriFee,
										   gasLimit);
		Credentials creds = null;
		try {
			ECKeyPair ecKP = ECKeyPair.create(bcConf.getSigningKey());
			creds = Credentials.create(ecKP);
			m_Wrapper = EnshroudProtocol.load(protocolAddr,
											  scc.getABI(),
											  creds,
											  gasProvider);
			// obtain ETH (or native token) balance
			String addr = creds.getAddress();
			EthGetBalance balanceWei = scc.getABI().ethGetBalance(addr,
											DefaultBlockParameterName.LATEST)
				.send();
			// TBD: possibly send an alert somewhere if balance is too low
			m_Log.debug("GreyLister.initialize: created creds for addr "
						+ Keys.toChecksumAddress(addr) + " on chain " + chainId
						+ ", native token bal = " + balanceWei.getBalance());
		}
		catch (Exception eee) {
			m_Log.error("GreyLister.initialize: exception loading greyList "
						+ "Credentials for chain " + chainId, eee);
			return false;
		}
		return true;
	}

	/**
	 * shutdown method
	 * @return true on success
	 */
	public boolean shutdown() {
		if (!m_SuspectBlocks.isEmpty()) {
			m_Log.warning("GreyLister.shutdown: suspect blocks existed");
			m_SuspectBlocks.clear();
		}
		if (m_Buckets.isEmpty()) {
			return true;
		}
		m_Log.warning("GreyLister.shutdown: buckets existed");

		// for all buckets, stop any threads which appear to be running
		Iterator<GreylistBucket> bucketIt = m_Buckets.iterator();
		boolean stopErr = false;
		while (bucketIt.hasNext()) {
			GreylistBucket bucket = bucketIt.next();
			if (bucket.m_Greylister != null && bucket.m_Greylister.isAlive()) {
				// tell the thread that it's supposed to stop
				bucket.m_StopFlag = true;
				try {
					// interrupt in case it's sleeping awaiting retry
					bucket.m_Greylister.interrupt();
				}
				catch (SecurityException se) {
					m_Log.error("GreyLister.shutdown: exception interrupting "
								+ "bucket thread", se);
					stopErr = true;
				}
			}
		}
		m_Buckets.clear();

		// cancel any greylist startup retry timer
		if (m_RetryGreylistAudit != null) {
			m_RetryGreylistAudit.cancel();
		}
		return !stopErr;
	}

	/**
	 * add a bucket of IDs to the greylist
	 * @param bucket the list of IDs to handle at once
	 * @param reason the reason the IDs were greylisted (same for all IDs)
	 */
	public void addBucket(ArrayList<BigInteger> idList, String reason) {
		if (idList == null || idList.isEmpty()) {
			m_Log.error("GreyLister.addBucket: missing or empty ID list");
			return;
		}
		if (reason == null || reason.isEmpty()) {
			m_Log.error("GreyLister.addBucket: missing greylist reason");
			return;
		}
		GreylistBucket bucket = new GreylistBucket(idList, reason);
		m_Buckets.add(bucket);
	}

	/**
	 * method to update a DB status record to the indicated status
	 * @param eId the eNFT Id (as zero-padded string of len 64)
	 * @param status the status value to be set
	 * @return true on success, false on any DB error
	 */
	private boolean setDbStatus(String eId, String status) {
		final String lbl = this.getClass().getSimpleName() + ".setDbStatus: ";

		// status cannot be 'suspect' because we'd need a reason
		if (status == null || status.isEmpty()
			&& EnftStatus.M_Suspect.equals(status))
		{
			m_Log.error(lbl + "illegal status value, " + status);
			return false;
		}
		DbConnectionManager dbMgr = m_AUD.getDbManager();
		EnftStatusDb statDb = new EnftStatusDb(dbMgr, m_Log);

		// update record
		long chainId = m_Cache.getConfig().getChainId();
		String us = m_AUD.getAUDId();
		boolean dbErr = false;
		try {
			// NB: this call is a no-op if specified status is already set
			if (!statDb.updStatus(eId, chainId, us, status, "", null)) {
					dbErr = true;
					m_Log.error(lbl + "unable to update ID " + eId
								+ " status to " + status);
			}
		}
		catch (EnshDbException edbe) {
			m_Log.error(lbl + "exception updating ID " + eId
						+ " status to " + status, edbe);
			dbErr = true;
		}
		return !dbErr;
	}

	/**
	 * Method to figure out based on database status and on-chain status,
	 * which IDs need to be greylisted at startup.  Can only be called after
	 * EnftCache is fully initialized, at least in regard to greylist events.
	 * @return true if audit passed (including any corrections implemented)
	 */
	public boolean greyListAuditAtStartup() {
		final String lbl
			= this.getClass().getSimpleName() + ".greyListAtStartup: ";

		// we can only be called after our GL cache is fully initialized
		SmartContractConfig scc = m_Cache.getConfig();
		long chainId = scc.getChainId();
		/* make sure GreyListAdd and GreyListDeletion events have been
		 * processed, and mergeGreyList() run
		 */
		if (!m_Cache.isGreyListingComplete()) {
			m_Log.error(lbl + "called prior to EnftCache initialzation for "
						+ "chainId " + chainId);
			// schedule retry after interval
			Timer retrier = new Timer("StartupAuditRetry", true);
			retrier.schedule(m_RetryGreylistAudit,
							 m_RetryGreylistAudit.M_GREYLIST_RETRY);
			return false;
		}

		/* Conduct an audit of greylist/ungreylist status and compare with DB.
		 * Note that EnftCache.getGreylistedIDs() returns all Ids that have
		 * ever been greylisted, regardless of whether that Id still is.
		 */
		ConcurrentSkipListSet<BigInteger> allGreylisted
			= m_Cache.getGreylistedIDs();
		if (allGreylisted.isEmpty()) {
			// nothing to do
			m_Log.debug(lbl + "no greylisted items on chainId " + chainId);
			return true;
		}

		// filter list of greylisted IDs through ungreylisted list
		ArrayList<String> mustBeGreyed
			= new ArrayList<String>(allGreylisted.size());
		ArrayList<String> mustNotBeGreyed
			= new ArrayList<String>(allGreylisted.size());
		Iterator<BigInteger> greylistedIt = allGreylisted.iterator();
		while (greylistedIt.hasNext()) {
			BigInteger glId = greylistedIt.next();
			String ID = Numeric.toHexStringNoPrefixZeroPadded(glId, 64);
			/* Confirm whether still greylisted in light of ungreylisted list.
			 * Note that EnftCache.isIdGreyListed() returns a *net* status,
			 * meaning that if an Id has been greylisted more times than it's
			 * been un-greylisted, it will be returned.
			 */
			if (m_Cache.isIdGreyListed(glId)) {
				mustBeGreyed.add(ID);
			}
			else {
				// this one should be changed back to valid status
				mustNotBeGreyed.add(ID);
			}
		}
		if (mustBeGreyed.isEmpty() && mustNotBeGreyed.isEmpty()) {
			// nothing to do
			m_Log.debug(lbl + "no greylist item changes on chainId " + chainId);
			return true;
		}

		// confirm that every greylisted Id is marked as such in the DB
		DbConnectionManager dbMgr = m_AUD.getDbManager();
		EnftStatusDb statDb = new EnftStatusDb(dbMgr, m_Log);
		ArrayList<EnftStatus> glStatusRecs
			= new ArrayList<EnftStatus>(allGreylisted.size());
		if (!mustBeGreyed.isEmpty()) {
			glStatusRecs = statDb.getStatus(mustBeGreyed, chainId, null);
			if (glStatusRecs == null) {
				// got DB error
				m_Log.error(lbl + "error checking status records of "
							+ mustBeGreyed.size() + " greylisted Ids");
				return false;
			}
		}
		if (glStatusRecs.size() != mustBeGreyed.size()) {
			m_Log.error(lbl + mustBeGreyed.size() + " GL records on-chain, but "
						+ glStatusRecs.size() + " records in DB");
			// don't fail due to this
		}

		ArrayList<EnftStatus> uglStatusRecs
			= new ArrayList<EnftStatus>(allGreylisted.size());
		if (!mustNotBeGreyed.isEmpty()) {
			uglStatusRecs = statDb.getStatus(mustNotBeGreyed, chainId, null);
			if (uglStatusRecs == null) {
				// got DB error
				m_Log.error(lbl + "error checking status records of "
							+ mustNotBeGreyed.size() + " non-greylisted Ids");
				return false;
			}
		}
		if (uglStatusRecs.size() != mustNotBeGreyed.size()) {
			m_Log.error(lbl + mustNotBeGreyed.size()
						+ " !GL records on-chain, but "
						+ uglStatusRecs.size() + " records in DB");
			// don't fail due to this
		}

		// see which records we need to update to blocked, if any
		boolean dbErr = false;
		ArrayList<String> idsToBlock
			= new ArrayList<String>(glStatusRecs.size());
		for (String eId : mustBeGreyed) {
			EnftStatus dbRec = null;
			for (EnftStatus statRec : glStatusRecs) {
				if (eId.equals(statRec.getID())) {
					dbRec = statRec;
					break;
				}
			}
			if (dbRec != null) {
				String stat = dbRec.getStatus();
				// compare value, should be 'blocked'
				if (EnftStatus.M_Suspect.equals(stat)) {
					// some AUD greylisted on-chain without updating DB
					m_Log.error(lbl + "greylisted Id " + eId
							+ " marked suspect rather than blocked; fixing");
					idsToBlock.add(dbRec.getID());
				}
				else if (EnftStatus.M_Valid.equals(stat)) {
					m_Log.error(lbl + "greylisted Id " + eId
							+ " marked valid rather than blocked; fixing");
					idsToBlock.add(dbRec.getID());
				}
				else if (EnftStatus.M_Deleted.equals(stat)) {
					// we don't fix this one because it's burned
					m_Log.error(lbl + "WEIRD: greylisted Id " + eId
						+ " marked deleted rather than blocked; not fixing");
				}
				// else: blocked as expected, nothing to do
			}
			else {
				m_Log.error(lbl + "WEIRD: greylisted Id " + eId + " on chain "
							+ chainId + " not found in status DB");
			}
		}
		if (idsToBlock.isEmpty()) {
			if (!mustBeGreyed.isEmpty()) {
				m_Log.debug(lbl + "all " + mustBeGreyed.size()
							+ " greylisted items on chainId " + chainId
							+ " duly marked blocked in DB");
			}
		}
		else {
			// mark all as 'blocked' since they're greyed on-chain
			for (String bId : idsToBlock) {
				if (!setDbStatus(bId, EnftStatus.M_Blocked)) {
					m_Log.error(lbl + "unable to mark eId " + bId
								+ " as blocked in DB");
					dbErr = true;
				}
			}
			if (!dbErr) {
				m_Log.debug(lbl + "Marked " + idsToBlock.size() + " greylisted "
							+ "items on chainId " + chainId + " blocked in DB");
			}
		}

		// see which records we need to update to valid, if any
		ArrayList<String> idsToUnBlock
			= new ArrayList<String>(uglStatusRecs.size());
		for (String eId : mustNotBeGreyed) {
			EnftStatus dbRec = null;
			for (EnftStatus statRec : uglStatusRecs) {
				if (eId.equals(statRec.getID())) {
					dbRec = statRec;
					break;
				}
			}
			if (dbRec != null) {
				String stat = dbRec.getStatus();
				// compare value, should be 'valid'
				if (EnftStatus.M_Suspect.equals(stat)) {
					// some AUD greylisted on-chain without updating DB
					m_Log.error(lbl + "ungreylisted Id " + eId
							+ " marked suspect rather than valid; fixing");
					idsToUnBlock.add(dbRec.getID());
				}
				else if (EnftStatus.M_Blocked.equals(stat)) {
					m_Log.debug(lbl + "ungreylisted Id " + eId
							+ " marked blocked rather than valid; fixing");
					idsToUnBlock.add(dbRec.getID());
				}
				else if (EnftStatus.M_Deleted.equals(stat)) {
					// we don't want to alter this because it's burned
					m_Log.error(lbl + "WEIRD: ungreylisted Id " + eId
							+ " marked deleted rather than valid; not fixing");
				}
				// else: valid as expected, nothing to do
			}
			else {
				m_Log.error(lbl + "WEIRD: ungreylisted Id " + eId + " on chain "
							+ chainId + " not found in status DB");
			}
		}
		if (idsToUnBlock.isEmpty()) {
			if (!mustNotBeGreyed.isEmpty()) {
				m_Log.debug(lbl + "all " + mustNotBeGreyed.size()
							+ " un-greylisted items on chainId " + chainId
							+ " duly marked valid in DB");
			}
		}
		else {
			// mark all as 'valid' since they're now ungreyed on-chain
			for (String bId : idsToUnBlock) {
				if (!setDbStatus(bId, EnftStatus.M_Valid)) {
					m_Log.error(lbl + "unable to mark eId " + bId
								+ " as again valid in DB");
					dbErr = true;
				}
			}
			if (!dbErr) {
				m_Log.debug(lbl + "Marked " + idsToUnBlock.size()
							+ " ungreylisted items on chainId " + chainId
							+ " valid in DB");
			}
		}

		/* NB: We do not now examine all status=suspect records in the DB,
		 * 	   and try to greylist them if minted but not burned or ungrelisted.
		 * 	   Arguably we could do this, but given that at least one AUD is
		 * 	   supposed to be operating at all times, the only case we would be
		 * 	   covering by doing so would be the case where:
		 * 	   	1) A bad output eNFT was generated, and an AUD marks it suspect.
		 * 	   	2) All AUD nodes become unavailable.
		 * 	   	3) The bad output eNFT(s) get minted, and no AUD sees it happen.
		 * 	   	4) At least one AUD comes back up and calls this method prior
		 * 	   	   to the dwell time in elapsed blocks, and greylists the ID(s).
		 * 	   	   (Alternatively, that the bad IDs are not spent right away.)
		 * 	   This is a very unlikely set of circumstances, and if all AUDs
		 * 	   indeed went offline, there'd be no guarantee that at least one
		 * 	   would come back online in time for the greylisting to matter.
		 * 	   The possible negative consequence would be opening the door to
		 * 	   malicious greylisting via DB manipulation, if a 'suspect' status
		 * 	   automatically and inevitably leads to on-chain greylisting.
		 * 	   If this sort of circumstance ever arises, we can revisit this.
		 */
		return !dbErr;
	}

	// methods required to implement EnftListener (called from m_Cache)
	/**
	 * register a eNFT mint event
	 * @param chainId the chain on which the mint occurred
	 * @param enftId the ID of the eNFT minted
	 * @param addr the wallet address to which the eNFT was minted
	 * @param block the block number in which the mint occurred
	 */
	public void recordEnftMint(long chainId,
							   BigInteger enftId,
							   String addr,
							   BigInteger block)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".recordEnftMint: ";

		// ignore if not for our chain (shouldn't happen anyway)
		SmartContractConfig scc = m_Cache.getConfig();
		if (chainId != scc.getChainId()) {
			return;
		}

		// ignore calls prior to SCC enablement (i.e. during init event dnld)
		if (!scc.isEnabled()) {
			return;
		}
		String ID = Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
	/*
		m_Log.debug(lbl + "cId " + chainId + ", eId " + ID
					+ ", to 0x" + addr + ", block " + block);
	 */

		/* step 1: search existing buckets to see if this eNFT ID is included.
		 * NB: a) Finding a bucket implies that we previously processed an
		 * 	   AuditorBlock which caused us to create a bucket for a set of
		 * 	   outputs including the given eNFT Id.
		 * 	   b) This means we expect a EnftStatus record marked 'suspect'
		 * 	   (or 'blocked' if we or another AUD already got to this one)
		 * 	   must already exist.
		 * 	   c) If the bucket is already in progress, repeating the call to
		 * 	   its doGreyListing() method will exit without spawning a thread.
		 * 	   (This can occur if we're processing a subsequent mint event for
		 * 	   another Id that was also in the bucket.)
		 */
		Iterator<GreylistBucket> bucketIt = m_Buckets.iterator();
		GreylistBucket match = null;
		while (bucketIt.hasNext()) {
			GreylistBucket bucket = bucketIt.next();
			if (bucket.m_GreyListIDs.contains(enftId)) {
				match = bucket;
				break;
			}
		}
		if (match != null) {
			// invoke greylisting for this bucket (states handled per NB above)
			match.doGreyListing();
			// we need to nothing further here
			return;
		}

		/* step 2: check whether an EnftStatus record exists for this eNFT Id.
		 * a) If it doesn't, implies neither we nor any other AUD ever saw an
		 * 	  AuditorBlock involving this Id.  This could be indicative of an
		 * 	  attempt by a group of colluding MVOs to mint eNFTs without the
		 * 	  required AUD broadcasts.  Accordingly, any such Id must be flagged
		 * 	  for greylisting.
		 * b) If an EnftStatus record exists, examine status:
		 * 		o If status = valid, no action.
		 * 		o If status = suspect, flag Id.
		 * 		o If status = blocked, confirm on-chain status.  If greylisted,
		 * 		  nothing to do.  If not, check for a GreyListDeletion item.
		 * 		  If found, set status back to valid.
		 * 		o If status = deleted, no action / ignore.
		 */
		DbConnectionManager dbMgr = m_AUD.getDbManager();
		Connection dbConn = null;
		// get a DB connection for multiple uses here
		try {
			dbConn = dbMgr.getConnection();
		}
		catch (SQLException se) {
			m_Log.error(lbl + "CRITICAL: can't get DB connection to check "
						+ "status of newly minted eNFT Id " + ID, se);
			return;
		}
		EnftStatusDb statDb = new EnftStatusDb(dbMgr, m_Log);
		String us = m_AUD.getAUDId();
		ArrayList<String> checkId = new ArrayList<String>(1);
		checkId.add(ID);
		boolean flagId = false;
		EnftStatus eStatus = null;
		String reason = "";
		ArrayList<EnftStatus> existStat
			= statDb.getStatus(checkId, chainId, dbConn);
		if (existStat == null) {
			// This indicates a DB error.  Log but don't greylist.
			m_Log.error(lbl + "CRITICAL: DB error checking for EnftStatus of "
						+ "eId " + ID);
			dbMgr.closeConnection(dbConn);
			return;
		}
		if (existStat.isEmpty()) {
			// No record found.  Flag for greylisting!
			m_Log.error(lbl + "ALERT: no status record found for eId " + ID
						+ "; greyListing");
			flagId = true;
			reason = "Unexpected mintage of unknown ID";

			// also add a record now indicating suspect status (hash is bogus)
			try {
				if (statDb.setStatus(ID, chainId, us, "UNKNOWN",
									 EnftStatus.M_Suspect, reason, dbConn))
				{
					m_Log.debug(lbl + "added suspect status record for eNFT Id "
								+ ID);
				}
				else {
					m_Log.error(lbl + "CRITICAL: error adding suspect status "
								+ "record for eNFT Id " + ID);
				}
			}
			catch (EnshDbException edbe) {
				m_Log.error(lbl + "CRITICAL: exception adding suspect status "
							+ "record for eNFT Id " + ID, edbe);
			}
			finally {
				dbMgr.closeConnection(dbConn);
			}
		}
		else {
			eStatus = existStat.get(0);
			dbMgr.closeConnection(dbConn);
		}

		// if we fetched a status record, check status field
		if (eStatus != null) {
			reason = eStatus.getNarrative();
			String stat = eStatus.getStatus();
			if (EnftStatus.M_Suspect.equals(stat)) {
				/* Note this case would normally be caught during bucket check
				 * above.  The exception is if another AUD node is ahead of us
				 * in processing mint events, and has seen this one and added
				 * a EnftStatus record pending actual on-chain greylisting
				 * (which could still be waiting for mining).  We can eliminate
				 * this case by checking for another AUD Id and the "UNKNOWN"
				 * value of the detailsHash field.
				 */
				if (!eStatus.getSubmitter().equals(us)
					&& eStatus.getDetailsHash().equals("UNKNOWN"))
				{
					m_Log.debug(lbl + "skipping GL of eId " + ID
								+ " because " + eStatus.getSubmitter()
								+ " is handling it");
				}
				else {
					flagId = true;
				}
			}
			else if (EnftStatus.M_Blocked.equals(stat)) {
				/* Flag only if not already greylisted on-chain.  Note this
				 * shouldn't occur, because it implies that an AUD node
				 * previously marked the DB 'blocked' without doing the work
				 * on-chain first, which GreylistBucket.run() doesn't do.
				 */
				if (!m_Cache.isIdGreyListed(enftId)) {
					// determine whether this is because ID was un-greylisted
					ArrayList<BigInteger> ungreylisted
						= m_Cache.getUngreylistedIDs();
					if (!ungreylisted.contains(enftId)) {
						// this Id still needs to be greylisted by us
						flagId = true;
					}
					else {
						/* This Id is net non-greylisted because it was later
						 * ungreylisted by 3 admins.  Restore its valid status
						 * so that it can be spent by its owner.  Note that this
						 * is a very unlikely case to hit, because it implies
						 * that the eNFT was both greylisted before this mint
						 * attempt, *and* ungreylisted by admins before getting
						 * minted.  The latter being a manual process, the more
						 * likely scenario is that this output was minted and
						 * then greylisted, and will only get back to valid
						 * status when recordUnGreylist() gets called later.
						 */
						if (setDbStatus(ID, EnftStatus.M_Valid)) {
							m_Log.debug(lbl + "set eId " + ID + " back to "
										+ EnftStatus.M_Valid + " status on "
										+ "ungreylist detected at minting");
						}
					}
				}
				// else: blocked Id already greylisted, nothing to do
			}
			// else: valid or deleted status means no flag required
		}

		/* step 3: if this Id must be flagged, see whether we have an entry in
		 * 		   the m_SuspectBlocks hashtable for this block.
		 * 		   a) If not, create one first and add to hashtable.
		 * 		   b) Append this ID to the bucket.
		 * NB: this bucket will collect such IDs until the end of the block,
		 * 	   and the greylisting of the bucket's contents is triggered in
		 * 	   recordBlock() below.
		 */
		if (flagId) {
			GreylistBucket blockBucket = m_SuspectBlocks.get(block);
			if (blockBucket == null) {
				// add a new suspect bucket for this block
				ArrayList<BigInteger> idList
					= new ArrayList<BigInteger>(ClientMVOBlock.M_ARRAY_MAX);
				idList.add(enftId);
				blockBucket = new GreylistBucket(idList, reason);
				m_SuspectBlocks.put(block, blockBucket);
				// NB: we do not add the bucket to the m_Buckets queue yet
				m_Log.debug(lbl + "added eId " + ID + " to bucket for block "
							+ block);
			}
			else {
				// append Id to the existing bucket's Id list
				if (!blockBucket.m_GreyListIDs.contains(enftId)) {
					blockBucket.m_GreyListIDs.add(enftId);
					m_Log.debug(lbl + "appended eId " + ID
								+ " to bucket for block " + block);
				}
			}
		}
	}

	/**
	 * register a eNFT burn event
	 * @param chainId the chain on which the burn occurred
	 * @param enftId the ID of the eNFT burned
	 * @param addr the wallet address by which the eNFT was burned
	 * @param block the block number in which the burn occurred
	 */
	public void recordEnftBurn(long chainId,
							   BigInteger enftId,
							   String addr,
							   BigInteger block)
	{
		final String lbl
			= this.getClass().getSimpleName() + ".recordEnftBurn: ";

		// ignore if not for our chain (shouldn't occur)
		SmartContractConfig scc = m_Cache.getConfig();
		if (chainId != scc.getChainId()) {
			return;
		}

		// ignore calls prior to SCC enablement (i.e. during init event dnld)
		if (!scc.isEnabled()) {
			return;
		}
		String ID = Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);
	/*
		m_Log.debug(lbl + "cId " + chainId + ", eId " + ID
					+ ", by 0x" + addr + ", block " + block);
	 */

		/* Walk through buckets and see if enftId occurs in any of them.  If it
		 * does, remove from the list as a burn cannot be undone and it means
		 * nothing to greylist it.  Log though, because this may be evidence
		 * that someone "got away with one."
		 */
		Iterator<GreylistBucket> bucketIt = m_Buckets.iterator();
		GreylistBucket match = null;
		while (bucketIt.hasNext()) {
			GreylistBucket bucket = bucketIt.next();
			if (bucket.m_GreyListIDs.contains(enftId)) {
				m_Log.error(lbl + "CRITICAL: found burned eId " + ID
							+ " in GreylistBucket; removing");
				match = bucket;
				break;
			}
		}
		if (match != null) {
			// remove from the bucket
			match.m_GreyListIDs.remove(enftId);
		}

		// mark as deleted in DB, since a burn cannot be undone
		if (!setDbStatus(ID, EnftStatus.M_Deleted)) {
			m_Log.error(lbl + "could not set burned eId " + ID
						+ " to deleted in DB");
		}
	}

	/**
	 * Register a new block received, and greylist contents of any suspect
	 * block that exists.  Note this method greylists one block behind, so that
	 * we can be certain we have the complete list of illegal/unknown eNFT IDs
	 * minted in the block.
	 * @param chainId the chain on which the block was mined
	 * @param block the new latest block number
	 */
	public void recordBlock(long chainId, BigInteger block) {
		final String lbl = this.getClass().getSimpleName() + ".recordBlock: ";

		// compute previous block
		BigInteger prevBlock = block.subtract(BigInteger.ONE);
		// see if there were any accumulated suspect IDs for this block
		GreylistBucket blockBucket = m_SuspectBlocks.get(prevBlock);
		if (blockBucket == null) {
			// nothing to do
			return;
		}

		// sort the list according to natural ordering of Comparable<BigInteger>
		blockBucket.m_GreyListIDs.sort(null);

		// if required, break them up into separate buckets max of 20 Ids apiece
		final int chunkSz = ClientMVOBlock.M_ARRAY_MAX;
		int cnt = blockBucket.m_GreyListIDs.size();
		ArrayList<GreylistBucket> blockBuckets
			= new ArrayList<GreylistBucket>();
		for (int blkLim = 0; blkLim < cnt; blkLim += chunkSz) {
			int endIdx = Math.min(blkLim + chunkSz, cnt);
			ArrayList<BigInteger> bucket
				= new ArrayList<BigInteger>(ClientMVOBlock.M_ARRAY_MAX);
			List<BigInteger> chunk
				= blockBucket.m_GreyListIDs.subList(blkLim, endIdx);
			bucket.addAll(chunk);
			GreylistBucket chunkBucket
				= new GreylistBucket(bucket, blockBucket.m_Reason);
			blockBuckets.add(chunkBucket);
		}

		// we should only get called once per block; nevertheless nix from table
		m_SuspectBlocks.remove(prevBlock);

		/* If we're an odd-numbered AUD, go through these buckets from 0-n.
		 * If we're even-numbered, go through them in reverse order from n-0.
		 * This is done so that duplicate efforts among AUD nodes can be
		 * avoided to some extent.
		 * NB: GreylistBucket.run() itself checks for evidence of greylisting
		 * 	   on-chain (by querying EnftCache) before attempting to greylist.
		 */
		if (m_AUD.m_AUDNumber % 2 == 0) {
			// reverse order
			for (int iii = blockBuckets.size(); iii > 0; iii--) {
				GreylistBucket buck = blockBuckets.get(iii);
				buck.doGreyListing();
			}
		}
		else {
			// forward order
			for (int jjj = 0; jjj < blockBuckets.size(); jjj++) {
				GreylistBucket buck = blockBuckets.get(jjj);
				buck.doGreyListing();
			}
		}
	}

	/**
	 * record that an un-greylist event has occurred (post initialization)
	 * @param chainId the chain on which the greylist deletion occurred
	 * @param enftId the Id of the eNFT removed from greylist by admins
	 */
	public void recordUnGreylist(long chainId, BigInteger enftId) {
		final String lbl
			= this.getClass().getSimpleName() + ".recordUnGreylist: ";

		// ignore if not for our chain (shouldn't occur)
		SmartContractConfig scc = m_Cache.getConfig();
		if (chainId != scc.getChainId()) {
			return;
		}

		// ignore calls prior to SCC enablement (i.e. during init event dnld)
		if (!m_Cache.isGreyListingComplete()) {
			return;
		}
		String ID = Numeric.toHexStringNoPrefixZeroPadded(enftId, 64);

		// set back to valid status (does nothing if already valid)
		if (setDbStatus(ID, EnftStatus.M_Valid)) {
			// NB: the narrative in DB still retains original status report
			m_Log.debug(lbl + "cId " + chainId + ", eId " + ID + " made valid");
		}
		else {
			m_Log.error(lbl + "cId " + chainId + ", eId " + ID
						+ " could not make valid");
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Buckets != null) {
				m_Buckets.clear();
			}
			if (m_SuspectBlocks != null) {
				m_SuspectBlocks.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
