/*
 * last modified---
 * 	09-09-22 new, based on MVOState
 *
 * purpose---
 * 	class to hold Auditor state-tracking information
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.EncodingUtils;

import java.util.ArrayList;
import java.util.concurrent.ConcurrentHashMap;
import java.security.SecureRandom;
import java.security.NoSuchAlgorithmException;
import java.math.BigInteger;


/**
 * State-keeping engine for an Auditor.
 */
public final class AUDState {
	// BEGIN data members
	/**
	 * the current value for constructing IDs of outbound WebSocket requests to
	 * a MVO
	 */
	private long						m_NextMVOid;

	/**
	 * the current value for constructing IDs of outbound WebSocket requests to
	 * Auditors
	 */
	private long						m_NextAUDid;

	/**
	 * secure random number generator
	 */
	private SecureRandom				m_RNG;

	/**
	 * table of connectors to peers, indexed by their MVO/AUD ID
	 */
	private ConcurrentHashMap<String, PeerConnector>	m_PeerConnections;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 */
	public AUDState() {
		try {
			/* Use a NativePRNGBlocking RNG to get a random seed.  We cannot
			 * use this RNG in general because it quickly runs out of entropy
			 * and blocks for long periods.  See:
			 * https://docs.oracle.com/javase/8/docs/technotes/guides/security/StandardNames.html#SecureRandom
			 */
			SecureRandom seedRNG = SecureRandom.getInstanceStrong();
			byte[] seed = seedRNG.generateSeed(32);
			m_RNG = new SecureRandom(seed);
		}
		catch (NoSuchAlgorithmException nsae) {
			// fallback
			System.out.println("AUDState: warning - init default RNG");
		}
		m_PeerConnections = new ConcurrentHashMap<String, PeerConnector>();
	}

	/**
	 * obtain the next available Id for a WebSocket message outgoing to an MVO
	 * @return the unique sequence number
	 */
	public synchronized long getNextMVOid() { return ++m_NextMVOid; }

	/**
	 * obtain the next available Id for a WebSocket message outgoing to an Aud
	 * @return the unique sequence number
	 */
	public synchronized long getNextAUDid() { return ++m_NextAUDid; }
	
	/**
	 * obtain the secure random number generator
	 * @return the RNG used by this MVO
	 */
	public SecureRandom getRNG() { return m_RNG; }

	/**
	 * utility method to obtain a random ID
	 * @return the 256-bit value generated, as a 64-digit zero-padded hex number
	 */
	public String getNextID() {
		BigInteger val = new BigInteger(256, m_RNG);
		return EncodingUtils.toHexStringZeroPadded(val, false);
	}

	/**
	 * obtain the connection object for a given peer
	 * @param peerId the ID of the peer to look up
	 * @return the connector, or null if none exists
	 */
	public PeerConnector getPeerConnection(String peerId) {
		if (peerId == null || peerId.isEmpty()) {
			return null;
		}
		return m_PeerConnections.get(peerId);
	}

	/**
	 * purge a connection for a given peer
	 * @param peerId the ID of the peer to remove
	 * @return the value which was removed, or null if not found
	 */
	public PeerConnector purgePeerConnection(String peerId) {
		if (peerId == null || peerId.isEmpty()) {
			return null;
		}
		return m_PeerConnections.remove(peerId);
	}

	/**
	 * add a connection for a given peer
	 * @param peerId the ID of the peer to add
	 * @param conn the connector to add
	 * @return the previous value, if any, or null
	 */
	public PeerConnector addPeerConnection(String peerId, PeerConnector conn) {
		if (peerId == null || peerId.isEmpty() || conn == null) {
			return null;
		}
		return m_PeerConnections.put(peerId, conn);
	}

	/**
	 * obtain the list of all connected peers
	 * @return the list
	 */
	public ArrayList<String> getConnectedPeers() {
		ArrayList<String> allPeers
			= new ArrayList<String>(m_PeerConnections.keySet());
		return allPeers;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data
		try {
			if (m_PeerConnections != null) {
				m_PeerConnections.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
