/*
 * last modified---
 * 	01-06-26 version 0.2.7
 * 	12-19-25 add purging of burned eNFT keys to HKP
 * 	12-05-25 add EnftCache.reinitWhileRunning() call to run() HKP
 * 	10-06-25 do logging for requestEnftCacheRestart() in SCC's logfile
 * 	08-07-25 version 0.2.6; switch to MariaDB driver with wsrep_sync_wait set
 * 	07-16-25 version 0.2.5; implement 2nd VPN failover, live release candidate
 * 	05-06-25 version 0.2.4 (final test versions of MVOStaking/EnshroudProtocol)
 * 	04-01-25 prevent duplicate WSS connections created by
 * 			 requestEnftCacheRestart() invocations
 * 	03-28-25 version 0.2.3; support fallback to AES keys supplied by dApp
 * 	11-15-24 bump timeouts to 60 secs; prevent dup EnftCacheResetter calls
 * 	10-30-24 implement EnftCacheResetter
 * 	09-24-24 specify 30 sec timeouts for http connect/write, 60 secs for read;
 * 			 ensure ShutdownHook is created prior to initSmartContracts()
 * 	05-21-24 add override to RxJavaPlugins.setErrorHandler() to debug
 * 			 UndeliverableExceptionS that occur during .dispose() calls
 * 	04-15-23 add m_HKPTimer
 * 	03-12-24 provide dummy BlockchainConnectListener.setupMVOStakingListener()
 * 	01-22-24 use SmartContractConfigS to find MVO URIs in getURIforMVO()
 * 	12-21-23 run GreyLister.greyListAuditAtStartup() after successful cache init
 * 	12-15-23 add GreyLister initialization
 * 	09-25-23 changes to initialization order of objects
 * 	07-07-23 implement BlockchainConnectListener
 * 	06-20-23 set genesis deployment block in initSmartContractConfigs()
 * 	06-14-23 use new BlockchainAPI mechanism; remove AssetConfig logic; remove
 * 			 standalone mode
 * 	03-14-23 generic DB classes now in .db
 * 	12-06-22 read Credentials and set AUDConfig values
 * 	10-04-22 add m_AudBlockHandler
 * 	09-28-22 add DB initialization, fork Jetty logging into JettyLog
 * 	09-06-22 new, based on MVO
 *
 * purpose---
 * 	main class for Auditor servers
 */

package cc.enshroud.jetty.aud;

import cc.enshroud.jetty.*;
import cc.enshroud.jetty.log.Log;
import cc.enshroud.jetty.db.DbConnectionManager;
import cc.enshroud.jetty.db.EnshDbException;
import cc.enshroud.jetty.aud.db.AESKey;
import cc.enshroud.jetty.aud.db.EnftKeysDb;
import cc.enshroud.jetty.aud.db.EnftStatus;
import cc.enshroud.jetty.aud.db.EnftStatusDb;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.protocol.Web3j;
import org.web3j.protocol.http.HttpService;
import org.web3j.protocol.websocket.WebSocketService;
import org.web3j.protocol.exceptions.ClientConnectionException;
import org.web3j.protocol.core.filters.FilterException;
import org.web3j.utils.Numeric;
import okhttp3.OkHttpClient;
import io.reactivex.plugins.RxJavaPlugins;
import io.reactivex.exceptions.UndeliverableException;
import org.java_websocket.exceptions.WebsocketNotConnectedException;

import java.util.TimerTask;
import java.util.Timer;
import java.util.Date;
import java.util.Properties;
import java.util.Hashtable;
import java.util.Map;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.NavigableSet;
import java.util.concurrent.Future;
import java.util.concurrent.CancellationException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.IOException;
import java.io.PrintStream;
import java.io.FilenameFilter;
import java.text.NumberFormat;
import java.security.PrivateKey;
import java.security.PublicKey;
import java.security.SecureRandom;
import java.net.URI;
import java.net.URL;
import java.net.URISyntaxException;
import java.net.MalformedURLException;
import java.net.ConnectException;
import java.net.SocketException;
import java.sql.Connection;
import java.sql.SQLException;
import java.math.BigInteger;


/**
 * This class serves as the main for the Auditor Jetty server.  It manages all
 * the handlers, endpoints, listeners, and contexts for the Auditor, both as a
 * server and as a client of MVOs.
 */
public final class AUD extends TimerTask
	implements FilenameFilter, BlockchainConnectListener, EnftCacheResetter
{
	// BEGIN data members
	/**
	 * constant for version/date string
	 */
	private final static String	M_Version = "0.2.7, 06 January, 2026";

	/**
	 * our local configuration properties, as read from the file
	 * $jetty_home/AUD-N.properties (where N is the AUD number of the instance)
	 */
	private Properties			m_PropConfig;

	/**
	 * our instance number (which AUD we are)
	 */
	public static int			m_AUDNumber = 1;

	/**
	 * our Auditor ID
	 */
	private String				m_AUDId;

	/**
	 * our AUD configuration, as read from the smart contract
	 */
	private AUDConfig			m_Config;

	/**
	 * processor for handling peer MVO requests for user transactions
	 */
	private MVOHandler			m_MVOHandler;

	/**
	 * processor for outgoing connections to peer MVOs
	 */
	private MVOClient			m_MVOClient;

	/**
	 * actual blockchain API objects which talk to real JSON-RPC nodes,
	 * indexed by chainId values
	 */
	private Hashtable<Long, BlockchainAPI>	m_BlockchainAPIs;

	/**
	 * state-tracking object
	 */
	private AUDState			m_State;

	/**
	 * list of blockchain smart contract configurations this AUD supports,
	 * indexed by chain Id
	 */
	private Hashtable<Long, SmartContractConfig>	m_SmartContracts;

	/**
	 * callback handler for key server requests from MVOs
	 */
	private MVOKeyServer		m_KeyServer;

	/**
	 * callback handler for AuditorBlock requests broadcast from MVOs
	 */
	private AudBlockHandler		m_AudBlockHandler;

	/**
	 * database manager for accessing key server store
	 */
	private DbConnectionManager	m_DbManager;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * inner class to provide a shutdown hook for the server
	 */
	private class ShutdownHook extends Thread {
		/**
		 * the server we're to shut down
		 */
		private AUD			m_Server;

		/**
		 * constructor
		 * @param server the server we're a hook for
		 */
		public ShutdownHook(AUD server) {
			m_Server = server;
		}

		/**
		 * invokes shutdown methods and logs event
		 */
		@Override
		public void run() {
			if (m_HKPTimer != null) {
				m_HKPTimer.cancel();
			}

			// shut down the handlers
			if (m_Server.m_MVOClient != null) {
				m_Server.m_MVOClient.shutdown();
			}
			if (m_Server.m_MVOHandler != null) {
				m_Server.m_MVOHandler.shutdown();
			}
			if (m_Server.m_KeyServer != null) {
				m_Server.m_KeyServer.stop();
			}
			if (m_Server.m_Config != null) {
				m_Server.m_Config.setStatus(false);
			}

			// shut down all BlockchainAPI interfaces
			for (Long chain : m_BlockchainAPIs.keySet()) {
				// flushes the properties file to disk iff LocalBlockchainAPI
				// (does nothing if a true RemoteBlockchainAPI)
				BlockchainAPI chainAPI = m_BlockchainAPIs.get(chain);
				if (!chainAPI.shutdown()) {
					m_Server.log().error("AUD " + m_Server.m_AUDNumber
										+ " got error shutting down chain API "
										+ "for chain " + chain);
				}
			}

			// stop all the Web3j objects
			for (SmartContractConfig scc : m_SmartContracts.values()) {
				// shut down the GreyLister
				GreyLister chGL = m_Config.getGreyLister(scc.getChainId());
				if (chGL != null) {
					if (!chGL.shutdown()) {
						m_Server.log().error("AUD " + m_Server.m_AUDNumber
											+ " got error shutting down the "
											+ "GreyLister for chain "
											+ scc.getChainName());
					}
				}
				
				// shut down the cache
				EnftCache eCache = scc.getCache();
				if (eCache != null) {
					eCache.shutdown();
				}
				Web3j abiInterface = scc.getABI();
				if (abiInterface != null) {
					abiInterface.shutdown();
				}
				scc.log().deInit();
			}

			m_Server.log().debug("AUD " + m_Server.m_AUDNumber
								+ " server shutdown");
			m_Server.log().deInit();
		}
	}

	/**
	 * timer to trigger running hourly housekeeping
	 */
	private Timer				m_HKPTimer;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param props the properties we read from our config file
	 */
	public AUD(Properties props) {
		super();
		m_PropConfig = props;

		// init logging
		String logDir = m_PropConfig.getProperty("LogDir");
		if (logDir == null) {
			logDir = "/var/log/enshroud";
		}
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		// NB: assumption is that static m_AUDNumber was set by main()
		String logFile = m_PropConfig.getProperty("LogFile");
		if (logFile != null && !logFile.isEmpty()) {
			m_Log = new Log(logDir + File.separator + logFile);
		}
		else {
			m_Log = new Log(logDir + File.separator + "AUD-"
							+ nf.format(m_AUDNumber) + ".log");
		}
		m_Log.Init();

		// assign System.out and System.err to our logging object
		PrintStream logOut = m_Log.getLogStream();
		System.setOut(logOut);
		System.setErr(logOut);

		// set AUD Id, with number in minimum 3 digits
		m_AUDId = m_PropConfig.getProperty("AUDId",
											"AUD-" + nf.format(m_AUDNumber));
		// also set in system properties
		System.setProperty("PEER_ID", m_AUDId);

		// configure Jetty logging to use our Logger class
		System.setProperty("org.eclipse.jetty.util.log.class",
							"cc.enshroud.jetty.log.JettyLog");

		// create config
		m_Config = new AUDConfig(m_AUDId, m_Log);

		// create state tracker
		m_State = new AUDState();

		// create smart contract config map
		m_SmartContracts = new Hashtable<Long, SmartContractConfig>();

		// create blockchain API object map
		m_BlockchainAPIs = new Hashtable<Long, BlockchainAPI>();

		// create the MVO request handler
		String mvoPort = m_PropConfig.getProperty("MVOPort");
		if (mvoPort == null || mvoPort.isEmpty()) {
			mvoPort = "12441";
			m_PropConfig.setProperty("MVOPort", mvoPort);
		}
		int mPort = 12441;
		try {
			mPort = Integer.parseInt(mvoPort.trim());
		}
		catch (NumberFormatException nfe) {
			m_Log.error("Bad MVOPort properties value, " + mvoPort);
		}
		m_MVOHandler = new MVOHandler(this, m_Log, mPort);
		m_MVOClient = new MVOClient(this);

		// configure Apache logging to use our LogFactory class (for Proxool)
		System.setProperty("org.apache.commons.logging.LogFactory",
							"cc.enshroud.jetty.log.CommonsLogFactory");

		// init key server DB interface
		String dbUrl = m_PropConfig.getProperty("DBUrl");
		if (dbUrl == null) {
			dbUrl = "jdbc:mysql://localhost:3306/enshroud";
		}
		String dbUser = m_PropConfig.getProperty("DBUser");
		if (dbUser == null) {
			m_Log.error("Missing DB user");
		}
		String dbPw = m_PropConfig.getProperty("DBPassword");
		if (dbPw == null) {
			m_Log.error("Missing DB pass");
		}
		m_DbManager = new DbConnectionManager(dbUrl, dbUser, dbPw, m_Log);

		// init key server (which uses m_DbManager)
		m_KeyServer = new MVOKeyServer(this);

		// create the AuditorBlock processor
		m_AudBlockHandler = new AudBlockHandler(this);
	}

	/**
	 * main run method
	 * @param args array of arguments passed at invocation
	 */
	public static void main(String[] args) {
		// look for a -n config option, which tells us our instance number
		int instanceNum = 1;
		for (int iii = 0; iii < args.length; iii++) {
			String arg = args[iii];
			if (arg.equals("-n")) {
				// next arg will be the instance number
				String num = args[iii+1];
				try {
					int instance = Integer.parseInt(num.trim());
					instanceNum = instance;
				}
				catch (NumberFormatException nfe) {
					System.err.println("Illegal instance number, " + num);
					System.exit(2);
				}
			}
			else if (arg.startsWith("-")) {
				// something unsupported
				System.err.println("Unrecognized flag, " + arg);
				System.exit(1);
			}
		}

		// read the properties file for our instance
		m_AUDNumber = instanceNum;
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		String confFile = "AUD-" + nf.format(m_AUDNumber) + ".properties";
		Properties props = new Properties();
		FileInputStream fis = null;
		try {
			fis = new FileInputStream(confFile);
			props.load(fis);
			fis.close();
		}
		catch (FileNotFoundException fnfe) {
			System.err.println("Could not find properties file, " + confFile);
			System.exit(3);
		}
		catch (IOException ioe) {
			System.err.println("Error reading properties file, " + confFile);
			System.exit(4);
		}

		// establish the server object
		final AUD audServer = new AUD(props);
		if (!audServer.initialize()) {
			audServer.log().error("Unable to initialize AUD server, exiting");
			System.exit(5);
		}
		audServer.log().debug("Enshroud AUD instance " + nf.format(m_AUDNumber)
								+ " initialized, version " + M_Version);
	}

	/**
	 * initialize AUD server
	 * @return true on success
	 */
	public boolean initialize() { 
		final String lbl = this.getClass().getSimpleName() + ".initialize: ";
		// provide additional properties defaults
		if (m_PropConfig.getProperty("TrustStore") == null) {
			m_PropConfig.setProperty("TrustStore", "security/truststore");
		}
		if (m_PropConfig.getProperty("TrustStoreType") == null) {
			m_PropConfig.setProperty("TrustStoreType", "JKS");
		}
		if (m_PropConfig.getProperty("ClientKeyStore") == null) {
			m_PropConfig.setProperty("ClientKeyStore",
									 "security/keystore_client");
		}
		if (m_PropConfig.getProperty("PeerKeyStore") == null) {
			m_PropConfig.setProperty("PeerKeyStore",
									 "security/keystore_peer");
		}
		if (m_PropConfig.getProperty("KeyStoreType") == null) {
			m_PropConfig.setProperty("KeyStoreType", "PKCS12");
		}
		if (m_PropConfig.getProperty("KeyStorePass") == null) {
			m_PropConfig.setProperty("KeyStorePass", "changeit");
		}

		// populate m_Config with keypair used on L2
		StringBuilder keyPath = new StringBuilder(128);
		String secDir = m_PropConfig.getProperty("RunDir", "");

		// private key first
		String privKeyPath = m_PropConfig.getProperty("PrivKeyFile");
		if (privKeyPath == null || privKeyPath.isEmpty()) {
			// build a default
			if (secDir.isEmpty()) {
				// rely on relative path
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + m_AUDId + File.separator);
			keyPath.append("privkey.asc");
		}
		else {
			if (!secDir.isEmpty()) {
				keyPath.append(secDir + File.separator + privKeyPath);
			}
			else {
				keyPath.append(privKeyPath);
			}
		}
		File privKeyFile = new File(keyPath.toString());
		if (!privKeyFile.exists() || !privKeyFile.canRead()) {
			m_Log.error(lbl + "cannot open private key file " + privKeyFile
						+ " for read");
			return false;
		}
		// skip newline at end
		int keyLen = (int) privKeyFile.length() - 1;
		char[] keyBuff = new char[keyLen];
		try {
			FileReader kfr = new FileReader(privKeyFile);
			kfr.read(keyBuff, 0, keyLen);
			kfr.close();
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "unable to read private key from file", ioe);
			return false;
		}
		String keyData = new String(keyBuff);
		PrivateKey privKey = EncodingUtils.getPrivkeyFromBase64Str(keyData);
		Arrays.fill(keyBuff, ' ');
		if (privKey == null) {
			m_Log.error(lbl + "bad AUD private key data, " + keyPath);
			return false;
		}
		m_Config.configCommPrivkey(privKey);

		// now the public key
		keyPath = new StringBuilder(128);
		String pubKeyPath = m_PropConfig.getProperty("PubKeyFile");
		if (pubKeyPath == null || pubKeyPath.isEmpty()) {
			// build a default
			if (secDir.isEmpty()) {
				// rely on relative path
				keyPath.append("security");
			}
			else {
				keyPath.append(secDir + File.separator + "security");
			}
			keyPath.append(File.separator + m_AUDId + File.separator);
			keyPath.append("pubkey.asc");
		}
		else {
			if (!secDir.isEmpty()) {
				keyPath.append(secDir + File.separator + pubKeyPath);
			}
			else {
				keyPath.append(pubKeyPath);
			}
		}
		File pubKeyFile = new File(keyPath.toString());
		if (!pubKeyFile.exists() || !pubKeyFile.canRead()) {
			m_Log.error(lbl + "cannot open public key file " + pubKeyFile
						+ " for read");
			return false;
		}
		// skip newline at end
		keyLen = (int) pubKeyFile.length() - 1;
		keyBuff = new char[keyLen];
		try {
			FileReader kfr = new FileReader(pubKeyFile);
			kfr.read(keyBuff, 0, keyLen);
			kfr.close();
		}
		catch (IOException ioe) {
			m_Log.error(lbl + "unable to read public key from file", ioe);
			return false;
		}
		keyData = new String(keyBuff);
		PublicKey pubKey = EncodingUtils.getPubkeyFromBase64Str(keyData);
		if (pubKey == null) {
			m_Log.error(lbl + "bad AUD public key data, " + keyPath);
			return false;
		}
		m_Config.configCommPubkey(pubKey);

		// init the default URI at which we listen for peer MVO connections
		int idIdx = m_AUDId.indexOf("-") + 1;
		String aId = m_AUDId.substring(idIdx);
		String audPort = m_PropConfig.getProperty("MVOPort", "12441");
		String vpnOdd = m_PropConfig.getProperty("VPNODD", "10.1.0.");
		String vpnEven = m_PropConfig.getProperty("VPNEVEN", "10.2.0.");
		int audId = Integer.parseUnsignedInt(aId);
		Integer aIP = audId + MVOGenConfig.AUD_VPN_OFFSET;
		String uriStr = "ws://";
		if (audId % 2 == 0) {
			uriStr += vpnEven;
		}
		else {
			uriStr += vpnOdd;
		}
		uriStr += (aIP + ":" + audPort + "/keyserver");
		URI mvoURI = null;
		try {
			mvoURI = new URI(uriStr);
			/* NB: this is overridden in MVOHandler by listening on $MVOPort on
			 * 	   all supported interfaces, using ws://
			 */
			m_Config.setMVOURI(mvoURI);
		}
		catch (URISyntaxException use) {
			m_Log.error(lbl + "error building AUD URI from " + uriStr, use);
		}

		// init list of configured MVOs
		String mvoIdList = m_PropConfig.getProperty("MVOList");
		if (mvoIdList == null || mvoIdList.isEmpty()) {
			m_Log.error(lbl + "missing MVOList in properties");
			return false;
		}
		Object mListObj = null;
		try {
			mListObj = JSON.parse(mvoIdList);
		}
		catch (IllegalStateException ise) { /* log below */ }
		if (!(mListObj instanceof Object[])) {
			m_Log.error(lbl + "improper MVOList in properties, " + mvoIdList);
			return false;
		}
		Object[] mvoListObj = (Object[]) mListObj;
		for (int mmm = 0; mmm < mvoListObj.length; mmm++) {
			Object mvoIdObj = mvoListObj[mmm];
			if (!(mvoIdObj instanceof String)) {
				m_Log.error(lbl + "illegal MVO spec in MVOList, " + mvoIdList);
				return false;
			}
			// this must be a string MVO-NNN:port
			String mvoTxt = (String) mvoIdObj;
			String[] mvoSpec = mvoTxt.split(":");
			if (mvoSpec.length != 2) {
				m_Log.error(lbl + "illegal MVO spec in MVOList, " + mvoTxt);
				return false;
			}
			String mvoId = mvoSpec[0];
			String portId = mvoSpec[1];
			Integer mPort = null;
			try {
				mPort = Integer.parseInt(portId.trim());
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "illegal MVO port in MVOList, " + portId);
				return false;
			}
			m_Config.addMVO(mvoId, mPort);
		}
		// check we have at least one
		HashMap<String, Integer> allMVOs = m_Config.getMVOList();
		if (allMVOs.isEmpty()) {
			m_Log.error(lbl + "no valid MVO configurations in properties");
			return false;
		}

		// init peer pubkeys
		if (!initPeerKeys(secDir)) {
			m_Log.error(lbl + "error initializing peer public keys");
			return false;
		}

		// now confirm that we have a key for every configured MVO
		boolean mvoKeyErr = false;
		for (String mId : allMVOs.keySet()) {
			if (m_Config.getPeerPubkey(mId) == null) {
				m_Log.error(lbl + "no peer pubkey found for MVO Id " + mId);
				mvoKeyErr = true;
			}
		}
		if (mvoKeyErr) return false;

		/* set the override handler for RxJavaPlugins.onError() to help debug
		 * UndeliverableException handling when doing Subscription.dispose()
		 */
		RxJavaPlugins.setErrorHandler(e -> {
			if (e instanceof UndeliverableException) {
				e = e.getCause();
			}
			if ((e instanceof IOException)
				|| (e instanceof SocketException))
			{
				// fine, network problem or API that throws on cancelation
				m_Log.warning(lbl + "network error disposing of subscription: "
							  + e.toString());
				return;
			}
			if (e instanceof InterruptedException) {
				// fine, some blocking code was interrupted by a dispose()
				m_Log.warning(lbl + "blocking code interrupted by dispose", e);
				return;
			}
			if (e instanceof ClientConnectionException) {
				// fine, a dispose() call could not talk to the other side
				m_Log.warning(lbl + "connection failure during a dispose: "
							 + e.getMessage());
				return;
			}
			if ((e instanceof NullPointerException)
				|| (e instanceof IllegalArgumentException))
			{
				// that's likely a bug in the application
				m_Log.error(lbl + "possible bug due to NPE/IAE", e);
				Thread.currentThread().getUncaughtExceptionHandler()
					.uncaughtException(Thread.currentThread(), e);
				return;
			}
			if (e instanceof IllegalStateException) {
				// that's a bug in RxJava or in a custom operator
				m_Log.error(lbl + "possible bug in RxJava", e);
				Thread.currentThread().getUncaughtExceptionHandler()
					.uncaughtException(Thread.currentThread(), e);
				return;
			}
			if (e instanceof FilterException) {
				/* This occurs when the calling thread in the Executor has
				 * exited.  Because we call Executor.shutdownNow() first before
				 * calling .dispose(), this is expected.
				 */
				return;
			}
			if (e instanceof WebsocketNotConnectedException) {
				/* This occurs when a wss-based JSON-RPC connection is lost
				 * and reconnects, and EnftCache.shutdown() is called, which
				 * has no place to deliver the exception.  Expected.
				 */
				return;
			}
			m_Log.warning(lbl + "Undeliverable exception received, cause: "
						  + e.toString());
		});

		// establish a shutdown hook
		ShutdownHook sdh = new ShutdownHook(this);
		Runtime.getRuntime().addShutdownHook(sdh);

		// init blockchain configs from properties and from SCs
		if (!initSmartContractConfigs()) {
			m_Log.error(lbl
						+ "unable to initialize smart contract configurations");
			return false;
		}

		// test the DB connection (do before key server)
		Connection conn = null;
		try {
			conn = m_DbManager.getConnection();
			m_DbManager.closeConnection(conn);
		}
		catch (SQLException sqle) {
			m_Log.error(lbl + "could not obtain test DB connection", sqle);
			return false;
		}

		// start up the MVO key server
		if (!m_KeyServer.init()) {
			m_Log.error(lbl + "error initializing MVOKeyServer");
			return false;
		}

		// trigger every GreyLister to perform its startup audit tasks
		for (Long chaId : m_SmartContracts.keySet()) {
			GreyLister greyLister = m_Config.getGreyLister(chaId);
			if (greyLister != null) {
				if (!greyLister.greyListAuditAtStartup()) {
					m_Log.error(lbl + "errors running GL startup audit for "
								+ "chain Id " + chaId);
				}
			}
			// else: file:// test chains will not have a GreyLister defined
		}

		/* now that we have pubkeys and chains set up, establish the server
		 * and client objects for handling WebSocket connections to peers
		 */
		// start up the websocket incoming connection handler
		if (!m_MVOHandler.initialize()) {
			m_Log.error(lbl + "unable to start MVOHandler, exit");
			return false;
		}

		// start up the websocket outgoing handler (depends on m_MVOHandler)
		if (!m_MVOClient.initialize()) {
			m_Log.error(lbl + "unable to start MVOClient, exit");
			return false;
		}

		// schedule hourly housekeeping
		m_HKPTimer = new Timer("housekeeping", true);
		m_HKPTimer.scheduleAtFixedRate(this,
									   javax.management.timer.Timer.ONE_HOUR,
									   javax.management.timer.Timer.ONE_HOUR);

		// make an attempt to connect to every MVO we have configured
		for (String mvoID : allMVOs.keySet()) {
			mvoURI = getURIforMVO(mvoID);
			if (mvoURI != null) {
				if (m_MVOClient.openPeerSocket(mvoID, mvoURI, true) == null) {
					m_Log.error(lbl + "unable to make initial connection to "
								+ "MVO " + mvoID);
					/* take the connection back out so we don't confuse state
					 * whenever the MVO connects to us
					 */
					PeerConnector peer = m_State.getPeerConnection(mvoID);
					if (peer != null) {
						// this also kills pings
						peer.shutdown();
					}
					m_State.purgePeerConnection(mvoID);
				}
				else {
					m_Log.debug(lbl + "made initial connection to MVO "
								+ mvoID);
				}
			}
			else {
				m_Log.error(lbl + "no connect URI for configured MVO " + mvoID);
			}
		}
		return true;
	}


	// GET methods
	/**
	 * obtain the AUD Id
	 * @return the ID of our signing key
	 */
	public String getAUDId() { return m_AUDId; }

	/**
	 * obtain the AUD config (also loaded from supported smart contracts)
	 * @return the configuration, which spans all the blockchains we support
	 */
	public AUDConfig getConfig() { return m_Config; }

	/**
	 * obtain the properties config
	 * @return the local config object
	 */
	public Properties getProperties() { return m_PropConfig; }

	/**
	 * obtain the state-tracker
	 * @return the object used to track state
	 */
	public AUDState getStateObj() { return m_State; }

	/**
	 * obtain the logging object
	 * @return the logger used for errors and debug messages
	 */
	public Log log() { return m_Log; }

	/**
	 * obtain the MVO WebSocket processor
	 * @return the object which handles peer MVO WebSocket requests
	 */
	public MVOHandler getMVOHandler() { return m_MVOHandler; }

	/**
	 * obtain the MVO WebSocket client
	 * @return the object which handles creating MVO WebSocket connections
	 */
	public MVOClient getMVOClient() { return m_MVOClient; }

	/**
	 * obtain the key server interface handler
	 * @return the object which handles key server requests from MVOs
	 */
	public MVOKeyServer getKeyServer() { return m_KeyServer; }

	/**
	 * obtain the AuditorBlock processor
	 * @return the object which handles broadcast requests from MVOs
	 */
	public AudBlockHandler getAudBlockHandler() { return m_AudBlockHandler; }

	/**
	 * obtain the DB connection manager
	 * @return the manager object
	 */
	public DbConnectionManager  getDbManager() { return m_DbManager; }

	/**
	 * obtain a random MVO
	 * @return the ID of the MVO selected
	 */
	public String getRandomMVO() {
		HashMap<String, Integer> allMVOs = m_Config.getMVOList();
		SecureRandom rng = m_State.getRNG();
		ArrayList<String> mvoIDs = new ArrayList<String>(allMVOs.keySet());
		int idx = rng.nextInt(mvoIDs.size());
		String mId = "";
		try {
			mId = mvoIDs.get(idx);
		}
		catch (IndexOutOfBoundsException ioobe) {
			// ArrayList somehow changed during this call! (impossible)
			m_Log.error("No such MVO index, " + idx, ioobe);
			mId = mvoIDs.get(0);
		}
		return mId;
	}

	/**
	 * obtain the smart contract config object for a given blockchain Id
	 * @param chainId the Id of the chain we want the config for
	 * @return the config, or null if not found
	 */
	public SmartContractConfig getSCConfig(long chainId) {
		return m_SmartContracts.get(chainId);
	}

	/**
	 * calculate the URI for an MVO or AUD node, on the red/black VPNs, based
	 * on its ID
	 * @param nodeId the MVOId or AUDId
	 * @param reverse true if we want to use the IP from the opposite VPN
	 * @return the websocket URI to communicate with the node
	 */
	public URI computePeerURI(String nodeId, boolean reverse) {
		final String lbl
			= this.getClass().getSimpleName() + ".computePeerURI: ";
		URI nodeURI = null;
		String uriStr = "ws://";
		String vpnEven = m_PropConfig.getProperty("VPNEVEN", "10.2.0.");
		String vpnOdd = m_PropConfig.getProperty("VPNODD", "10.1.0.");

		// determine whether MVO or AUD
		if (nodeId.startsWith("MVO-")) {
			/* Build the URI based on properties values.  If the MVO's Id is
			 * even, use the formula VPNODD with IP = Id+20.  If it's even, use
			 * the formula VPNEVEN with IP = Id+20.  Examples: MVO-001 would be
			 * calculated as 10.1.0.21, MVO-002 as 10.2.0.22.  Specifying
			 * reverse=true results in the opposite.
			 */
			int idIdx = nodeId.indexOf("-") + 1;
			String mId = nodeId.substring(idIdx);
			try {
				int mvoId = Integer.parseUnsignedInt(mId);
				Integer mIP = mvoId + MVOGenConfig.MVO_VPN_OFFSET;
				if (mvoId % 2 == 0) {
					uriStr += (reverse ? vpnOdd : vpnEven);
				}
				else {
					uriStr += (reverse ? vpnEven : vpnOdd);
				}
				/* NB: a MVO with :port != 10444 must have its URI specified in
				 * a security/MVOId.properties file containing MVOURI=
				 */
				uriStr += (mIP + ":10444/mvo");
				nodeURI = new URI(uriStr);
			}
			catch (NumberFormatException nfe) {
				m_Log.error(lbl + "illegal MVOId, " + nodeId, nfe);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "error building MVO URI from " + uriStr, use);
			}
		}
		else {
			// AUD nodes don't contact other AUDs
			m_Log.error(lbl + "illegal nodeId, " + nodeId);
		}
		return nodeURI;
	}

	/**
	 * find a URI for a given MVO or Auditor
	 * @param mvoId
	 * @return the URI at which this MVO can be contacted, or null if not found
	 */
	public URI getURIforMVO(String mvoId) {
		if (mvoId == null || mvoId.isEmpty()) {
			return null;
		}

		URI mvoURI = null;
		if (mvoId.startsWith("MVO-")) {
			/* We make the assumption that the URI will be the same in the
			 * config for any blockchain in which the MVO appears.  Therefore,
			 * we simply take the first occurrence found.  These URIs are
			 * built by RemoteBlockchainAPI.getConfig(), but sourced via the
			 * algo used in computePeerURI(), with the MVOURI property
			 * found in any security/mvoId.properties file as an override.
			 */
			for (SmartContractConfig scc : m_SmartContracts.values()) {
				Hashtable<String, MVOGenConfig> availMVOs = scc.getMVOMap();
				MVOGenConfig mvoConf = availMVOs.get(mvoId);
				if (mvoConf != null) {
					mvoURI = mvoConf.getMVOURI();
					break;
				}
			}
		}
		// NB: AUD nodes don't connect to other AUDs
		else {
			m_Log.error("getURIforMVO: illegal MVO/AUD Id passed, " + mvoId);
		}
		return mvoURI;
	}

	/**
	 * obtain the blockchain interface handler
	 * @param chainId the chain for which we want the interface
	 * @return the object we use to talk to blockchains
	 */
	public BlockchainAPI getWeb3(long chainId) {
		return m_BlockchainAPIs.get(chainId);
	}


	// SET methods
	/**
	 * initialize peer pubkeys for MVOs and Auditors from static files
	 * @param secDir security directory where these files are stored
	 * @return true on success
	 */
	public boolean initPeerKeys(String secDir) {
		if (secDir == null) {
			return false;
		}

		StringBuilder keyPath = new StringBuilder(64);
		if (secDir.isEmpty()) {
			// rely on relative path
			keyPath.append("security");
		}
		else {
			keyPath.append(secDir + File.separator + "security");
		}

		/* list all files in this directory which are of the form MVO-NNN.pubkey
		 * or AUD-NNN.pubkey, for peer MVOs and Auditors
		 */
		File securityDir = new File(keyPath.toString());
		File[] peerPubkeys = null;
		try {
			peerPubkeys = securityDir.listFiles(this);
		}
		catch (SecurityException se) {
			m_Log.error("No permission to access peer pubkeys", se);
			return false;
		}
		boolean keyErr = false;
		if (peerPubkeys == null || peerPubkeys.length == 0) {
			m_Log.error("No peer pubkeys found in config");
			keyErr = true;
		}
		else {
			for (int ppp = 0; ppp < peerPubkeys.length; ppp++) {
				File pubKeyFile = peerPubkeys[ppp];
				if (!pubKeyFile.exists() || !pubKeyFile.canRead()) {
					m_Log.error("Cannot open public key file " + pubKeyFile
								+ " for read");
					keyErr = true;
					continue;
				}
				// skip newline at end
				int keyLen = (int) pubKeyFile.length() - 1;
				char[] keyBuff = new char[keyLen];
				try {
					FileReader kfr = new FileReader(pubKeyFile);
					kfr.read(keyBuff, 0, keyLen);
					kfr.close();
				}
				catch (IOException ioe) {
					m_Log.error("Unable to read public key from file", ioe);
					keyErr = true;
					continue;
				}
				PublicKey pubKey
					= EncodingUtils.getPubkeyFromBase64Str(new String(keyBuff));
				if (pubKey == null) {
					m_Log.error("Bad peer public key data, "
								+ pubKeyFile.getPath());
					keyErr = true;
				}
				else {
					int dotPos = pubKeyFile.getName().indexOf(".");
					String peerId = pubKeyFile.getName().substring(0, dotPos);
					m_Config.addPeerPubkey(peerId, pubKey);
				}
			}
		}
		return !keyErr;
	}

	/**
	 * Initialize the smart contract configurations, from the blockchains
	 * themselves.  In our properties we have only a list
	 * of chains to deal with.  Our ABI partner(s) must provide access
	 * via a URL.  This method is broken out from inside initialize() so
	 * that we can invoke it separately whenever an update event is received
	 * on a supported blockchain.
	 * @return true on success
	 */
	public boolean initSmartContractConfigs() {
		final String lbl = this.getClass().getSimpleName()
							+ ".initSmartContractConfigs: ";
		String scArray = m_PropConfig.getProperty("SmartContractList");
		if (scArray == null || scArray.isEmpty()) {
			m_Log.error("Missing smart contract config");
			return false;
		}
		Object scObj = null;
		try {
			scObj = JSON.parse(scArray);
		}
		catch (IllegalStateException ise) { /* log below */ }
		if (!(scObj instanceof Object[])) {
			m_Log.error(lbl + "parse error on SmartContractList value, "
						+ scArray);
			return false;
		}
		Object[] scAry = (Object[]) scObj;
		boolean scConfErr = false;
		for (int sss = 0; sss < scAry.length; sss++) {
			Object scItem = scAry[sss];
			// should have a list of longs
			if (!(scItem instanceof Long)) {
				m_Log.error(lbl + "item " + (sss+1)
							+ " in SmartContractList is not a long");
				return false;
			}

			// look for detail entry in properties
			Long chainId = (Long) scItem;
			String scDetails
				= m_PropConfig.getProperty("SmartContract-" + chainId);
			if (scDetails == null || scDetails.isEmpty()) {
				m_Log.error(lbl + "missing SmartContract-" + chainId
							+ " properties");
				scConfErr = true;
				continue;
			}

			/**
			 * should yield a Map with these entries -
			 * name = common name for blockchain
			 * ABI = local URI at which we can contact the full blockchain node
			 * Genesis = block number in which EnshroudProtocol was deployed
			 * (NB: AUD nodes are not concerned with ReceiptStore)
			 */
			Object scMap = null;
			try {
				scMap = JSON.parse(scDetails);
			}
			catch (IllegalStateException ise) { /* log below */ }
			if (!(scMap instanceof Map)) {
				m_Log.error(lbl + "SmartContract-" + chainId
							+ " details didn't parse");
				scConfErr = true;
				continue;
			}
			Map scDetMap = (Map) scMap;
			Object nmObj = scDetMap.get("name");
			String chainName = "";
			if (nmObj instanceof String) {
				chainName = (String) nmObj;
			}
			if (chainName == null || chainName.isEmpty()) {
				m_Log.error(lbl + "missing name= for " + chainId + " config");
				scConfErr = true;
			}
			Object uriObj = scDetMap.get("ABI");
			String abiUri = "";
			if (uriObj instanceof String) {
				abiUri = (String) uriObj;
			}
			if (abiUri == null || abiUri.isEmpty()) {
				m_Log.error(lbl + "missing ABI= for " + chainId + " config");
				scConfErr = true;
			}
			URI web3jURI = null;
			try {
				web3jURI = new URI(abiUri);
			}
			catch (URISyntaxException use) {
				m_Log.error(lbl + "Web3j URL for chainId " + chainId
							+ ", bad format", use);
				scConfErr = true;
			}
			Object genesisObj = scDetMap.get("Genesis");
			BigInteger deployBlock = null;
			if (genesisObj instanceof String) {
				String depBlock = (String) genesisObj;
				try {
					deployBlock = new BigInteger(depBlock);
				}
				catch (NumberFormatException nfe) {
					m_Log.error(lbl + "Genesis block for chainId " + chainId
								+ ", invalid block = " + depBlock);
				}
			}
			if (deployBlock == null) {
				m_Log.error(lbl + "missing Genesis= for " + chainId
							+ " config");
				// default to 1
				deployBlock = BigInteger.ONE;
			}
			if (scConfErr) {
				m_Log.error(lbl + "fatal errors in SC properties config for "
							+ "chainId " + chainId + " or a predecessor");
				continue;
			}

			// create a smart contract config object and init it
			SmartContractConfig scConfig
				= new SmartContractConfig(chainName, chainId, m_Log, this);
			scConfig.setWeb3jURI(web3jURI);
			scConfig.setDeploymentBlock(deployBlock);
			scConfig.configCacheRebuilder(this);

			// create the BlockchainConfig entry in our AUDConfig
			BlockchainConfig bConfig
				= new BlockchainConfig(chainId, chainName, m_Log);
			m_Config.addChainConfig(chainId, bConfig);

			// form a connection to the Web3j URL
			Web3j abiSession = null;
			BlockchainAPI web3Api = null;
			String proto = web3jURI.getScheme();
			boolean deferChainInit = false;
			Log scLog = scConfig.log();
			scLog.debug(lbl + chainName
						+ ": connecting to JSON-RPC @" + web3jURI.toString());
			if (proto.equals("file")) {
				// use test flat-file implementation
				web3Api = new LocalBlockchainAPI(this, chainId);
				if (!web3Api.init()) {
					m_Log.error(lbl + "error initializing LocalBlockchainAPI");
					scConfErr = true;
					continue;
				}
			}
			else if (proto.startsWith("http")) {
				// NB: regular HttpClient will be slow and shouldn't be used
				OkHttpClient okClient = new OkHttpClient();
				/* configure 60 sec timeout for fetching events, to match value
				 * org.web3j.protocol.websocket.WebSocketService.REQUEST_TIMEOUT
				 */
				OkHttpClient configClient = okClient.newBuilder()
										.readTimeout(60L, TimeUnit.SECONDS)
										.writeTimeout(60L, TimeUnit.SECONDS)
										.connectTimeout(60L, TimeUnit.SECONDS)
					.build();
				abiSession = Web3j.build(new HttpService(web3jURI.toString(),
														 configClient));
			}
			else if (proto.startsWith("ws")) {
				// websockets are persistent, therefore we must connect first
				WebSocketService wss
					= new WebSocketService(web3jURI.toString(), false);
				try {
					/* We use connect(Consumer<String> onMessage,
					 * 				Consumer<Throwable> onError,
					 * 				Runnable onClose())
					 * with the methods located in scConfig object,
					 * so we can tell when it drops and restore it.
					 */
					wss.connect(scConfig,
								throwable -> scConfig.accept(throwable),
								scConfig);
					abiSession = Web3j.build(wss);
				}
				catch (ConnectException ce) {
					m_Log.error(lbl + "error connecting to websocket for "
								+ web3jURI.toString());
					// defer to reconnect logic by invoking disconnect handler
					scConfig.run();		// (forks thread)
					// pre-config a BlockchainAPI so we'll have one
					web3Api = new RemoteBlockchainAPI(this, chainId);
					web3Api.init();		// (doesn't currently do anything)
					// certain inits below which need JSON-RPC we can't do yet
					deferChainInit = true;
				}
			}
			if (abiSession != null) {
				web3Api = new RemoteBlockchainAPI(this, chainId);
				scConfig.configABISession(abiSession);
				web3Api.init();		// (doesn't currently do anything)

				// create and init the eNFT data cache
				EnftCache eCache
					= new EnftCache(scConfig, abiSession, BigInteger.ZERO);
				scConfig.configCache(eCache);
				if (!eCache.initialize(false)) {
					// implies configuration error; we should exit app
					m_Log.error(lbl + "error initializing EnftCache for "
								+ chainName);
					scConfErr = true;
					continue;
				}
			}
			if (web3Api == null) {
				m_Log.error(lbl + "unsupported Web3j ABI protocol, " + proto);
				scConfErr = true;
				continue;
			}
			m_BlockchainAPIs.put(chainId, web3Api);

			// record the config
			m_SmartContracts.put(chainId, scConfig);

			if (!deferChainInit) {
				// get our own config for this chain (only inits our Creds)
				if (!getOwnNodeConfig(scConfig)) {
					scLog.error(lbl + "error fetching self BlockchainConfig "
								+ "for chainId " + chainId);
					scConfErr = true;
					continue;
				}

				// get MVO configs also and check they conform to our properties
				if (!getOtherNodeConfigs(scConfig)) {
					scLog.error(lbl + "errors building MVOConfig for MVOs on "
								+ "chain " + chainName);
					scConfErr = true;
					continue;
				}

				// setup the listener for events which handle MVOStaking updates
				if (!setupMVOStakingListener(scConfig)) {
					scLog.error(lbl + "error establishing listener for changes "
								+ "to MVOStakings for chainId " + chainId);
					scConfErr = true;
				}
			}
		}
		if (scConfErr) {
			m_Log.error(lbl + "fatal errors in SC properties configs");
			return false;
		}
		return true;
	}

	// override run() to implement TimerTask
	@Override
	public void run() {
		final String lbl = this.getClass().getSimpleName() + ".run: ";
		for (SmartContractConfig scc : m_SmartContracts.values()) {
			// reinitialize blockchain subscriptions since last HKP run
			EnftCache eCache = scc.getCache();
			if (eCache != null) {
				// reinit the cache
				if (!eCache.reinitWhileRunning()) {
					scc.log().error(lbl + "HKP unable to reinit for chain "
									+ scc.getChainName());
				}
			}

			// if property set, purge all keys for eNFTs burned > 2 weeks ago
			Log scLog = scc.log();
			if (m_PropConfig.getProperty("PURGE_ENFT_KEYS", "false")
				.equals("false"))
			{
				continue;
			}

			// 1. get all accounts which have ever owned burned eNFTs
			NavigableSet<String> pastOwners = eCache.getFormerOwners();

			// 2. get list of burned eNFTs each owner used to have; compute
			// 	  hash value for each, and merge into maps of burned hashes
			HashMap<String, String> burnedHashes
				= new HashMap<String, String>(pastOwners.size());
			HashMap<String, String> burnedIdMap
				= new HashMap<String, String>(pastOwners.size());
			for (String owner : pastOwners) {
				ArrayList<BigInteger> burntIDs
					= eCache.getBurnedIdsForAccount(owner);
				for (BigInteger bId : burntIDs) {
					ArrayList<String> hashComp = new ArrayList<String>(3);
					hashComp.add(Long.toString(scc.getChainId()));
					// convert cache's BigInteger to 64-char zero-padded ID
					String eId = Numeric.toHexStringNoPrefixZeroPadded(bId, 64);
					hashComp.add(eId);
					// convert from EIP-55 and prepend 0x
					hashComp.add(Numeric.prependHexPrefix(owner.toLowerCase()));
					String keyIdx = String.join("+", hashComp);
					String keyHash = EncodingUtils.sha3(keyIdx);
					burnedHashes.put(keyHash, eId);
					burnedIdMap.put(eId, keyHash);
				}
			}

			// 3. check which of these hashes still correspond to keys in DB
			EnftKeysDb eKeyDb = new EnftKeysDb(m_DbManager, m_Log);
			boolean gotKeys = true;
			ArrayList<String> hashesToBurn
				= new ArrayList<String>(burnedHashes.keySet());
			ArrayList<AESKey> extantKeys = eKeyDb.getKeys(hashesToBurn);
			if (extantKeys == null) {
				m_Log.error(lbl
							+ "unable to fetch extant burned eNFT keys");
				gotKeys = false;
			}
			else {
				if (extantKeys.isEmpty()) {
					gotKeys = false;
				}
			}

			// 4. for each key still in DB, fetch the ENSH_ENFT_STATUS record
			if (gotKeys) {
				boolean gotStatus = true;
				ArrayList<String> extantIDs
					= new ArrayList<String>(extantKeys.size());
				for (AESKey bKey : extantKeys) {
					String kId = burnedHashes.get(bKey.getHash());
					if (kId != null) {
						extantIDs.add(kId);
					}
				}
				EnftStatusDb statDb = new EnftStatusDb(m_DbManager, m_Log);
				ArrayList<EnftStatus> statRecs
					= statDb.getStatus(extantIDs, scc.getChainId(), null);
				if (statRecs == null) {
					m_Log.error(lbl + "unable to fetch status records");
					gotStatus = false;
				}
				else {
					if (statRecs.isEmpty()) {
						gotStatus = false;
					}
				}

				// 5. loop through all found status records and see if they were
				// 	  deleted more than 2 weeks ago
				if (gotStatus) {
					ArrayList<String> purgeHashes
						= new ArrayList<String>(statRecs.size());
					for (EnftStatus statRec : statRecs) {
						String id = statRec.getID();
						if (statRec.getStatus().equals(EnftStatus.M_Deleted)) {
							Date lMod = statRec.getLmodDate();
							if (lMod == null) {
								m_Log.error(lbl + "burned Id " + id
											+ " doesn't have lmod timestamp");
								continue;
							}
							Date now = new Date();
							Date twoWeeksAgo = new Date(now.getTime()
								- (2L * javax.management.timer.Timer.ONE_WEEK));
							if (lMod.before(twoWeeksAgo)) {
								// add to list of hashes to be purged
								String delHash = burnedIdMap.get(id);
								if (delHash != null) {
									purgeHashes.add(delHash);
								}
								else {
									m_Log.error(lbl + "burned Id " + id
												+ " doesn't have a known hash");
								}
							}
						}
						else {
							scLog.warning(lbl + "burned Id " + statRec.getID()
											+ " doesn't have deleted status");
						}
					}

					if (!purgeHashes.isEmpty()) {
						// 6. tell the key server to purge these hashes
						ArrayList<AESKey> purgedHashes
							= m_KeyServer.removeKeys(true, purgeHashes);
						if (purgedHashes == null) {
							m_Log.error(lbl + "error purging deleted hashes");
						}
						else {
							scLog.debug(lbl + "purged " + purgedHashes.size()
										+ " burned eNFT keys");
						}
					}
				}
				else {
					scLog.error(lbl + "no STATUS records found for keys");
				}
			}
		}

		// now reclaim any freed heap memory
		System.gc();
	}

	// implement FilenameFilter
	/**
	 * method to select files in RunDir/security that are peer MVO pubkeys
	 * @param dir the directory the file was found in
	 * @param name the name of the file
	 * @return true if we should accept this file
	 */
	public boolean accept(File dir, String name) {
		if (dir == null || name == null) {
			return false;
		}
		if (dir.getPath().contains("security")) {
			if (name.endsWith(".pubkey") && (name.startsWith("MVO-"))) {
				return true;
			}
		}
		return false;
	}

	// methods to implement BlockchainConnectListener
	/**
	 * obtain node Id
	 * @return our Id
	 */
	public String getSelfId() { return m_AUDId; }

	/**
	 * perform download tasks related to obtaining our own on-chain config
	 * @param scc the smart contract we just achieved a connection to
	 * @return true on success
	 */
	public boolean getOwnNodeConfig(SmartContractConfig scc) {
		final String lbl
			= this.getClass().getSimpleName() + ".getOwnNodeConfig: ";
		if (scc == null) {
			m_Log.error(lbl + "no SmartContractConfig input");
			return false;
		}
		long chainId = scc.getChainId();

		// obtain our on-chain config (doesn't actually utilize abiSession)
		BlockchainAPI web3Api = m_BlockchainAPIs.get(chainId);
		Future<BlockchainConfig> audConfFuture
			= web3Api.getMVOConfig(chainId, scc.getChainName(),
								   scc.getABI(), getSelfId());
		BlockchainConfig audBCconf = null;
		try {
			audBCconf = audConfFuture.get();
		}
		catch (Exception ee) {
			m_Log.error(lbl + "exception getting BlockchainConfig for self: "
						+ ee.toString());
		}
		if (audBCconf == null) {
			m_Log.error(lbl + "could not fetch BlockchainConfig for self");
			return false;
		}

		// record data in allocated blockchain config record
		HashMap<Long, BlockchainConfig> bcMap = m_Config.getChainConfigs();
		BlockchainConfig bConfig = bcMap.get(chainId);
		if (bConfig != null) {
			bConfig.configSigningKey(audBCconf.getSigningKey());
			bConfig.configSigningAddress(audBCconf.getSigningAddress());
		}
		else {
			m_Log.error(lbl + "no BlockchainConfig to store self record");
			return false;
		}

		// if we have an EnftCache already, configure GreyLister as listener
		EnftCache eCache = scc.getCache();
		if (eCache != null) {
			// if we do not already have one, create a GreyLister for this chain
			GreyLister chGL = m_Config.getGreyLister(chainId);
			if (chGL == null) {
				// first time we've connected to ABI for this chain
				chGL = new GreyLister(this, eCache);
				m_Config.addGreyLister(chainId, chGL);

				// configure the GreyLister as the EnftListener in the SCC
				scc.configEnftListener(chGL);
			}

			/* (re-)init GreyLister credentials and ABI/wrapper.  Implicit
			 * assumtion here: reconnect case should not involve a change of the
			 * signing key recorded on-chain, but if it does will be picked up
			 * here.  (This is reasonable.)
			 */
			if (!chGL.initialize()) {
				m_Log.error(lbl + "unable to initialize GreyLister for "
							+ scc.getChainName());
				return false;
			}
			eCache.registerListener(chGL);
		}
		// else: there's no need for a GreyLister because we're a file:// chain
		return true;
	}

	/**
	 * perform download tasks related to obtaining other nodes' configs
	 * @param scc the smart contract we just achieved a connection to
	 * @return true on success
	 */
	public boolean getOtherNodeConfigs(SmartContractConfig scc) {
		final String lbl
			= this.getClass().getSimpleName() + ".getOtherNodeConfigs: ";
		if (scc == null) {
			m_Log.error(lbl + "no SmartContractConfig input");
			return false;
		}
		long chainId = scc.getChainId();
		String chainName = scc.getChainName();

		/* use the Web3j connection to obtain and record this data:
		 *	scc.m_MVOs (i.e. populate rest of mvoMap)
		 *	scc.m_ReqSigs
		 *	scc.m_DwellTime
		 * 	scc.m_TotalStaked
		 * 	scc.m_BaseURI
		 */
		BlockchainAPI web3Api = m_BlockchainAPIs.get(chainId);
		Future<SmartContractConfig> scFuture
			= web3Api.getConfig(chainId, chainName, scc.getABI());
		SmartContractConfig scConf = null;
		try {
			scConf = scFuture.get();
		}
		catch (InterruptedException | ExecutionException
				| CancellationException ee)
		{
			m_Log.error(lbl + "Exception getting SmartContractConfig "
						+ "from Web3j for chain " + chainName, ee);

		}
		if (scConf == null) {
			m_Log.error(lbl + "could not fetch SmartContractConfig "
						+ "from Web3j for chain " + chainName);
			return false;
		}
		Hashtable<String, MVOGenConfig> mvoMap = scc.getMVOMap();
		mvoMap.putAll(scConf.getMVOMap());
		scc.setNumSigs(scConf.getNumSigs());
		scc.setDwellTime(scConf.getDwellTime());
		scc.setTotalStaking(scConf.getTotalStaking().toString());
		scc.setBaseURI(scConf.getBaseURI());

		/* For each MVO in the map for this blockchain, we need to check
		 * that we have a peer pubkey for it, and obtain a valid MVOURI.
		 */
		boolean mvoConfsOk = true;
		for (String chainMVO : mvoMap.keySet()) {
			// propConf == the one fetched from .properties or chain
			MVOConfig propConf = (MVOConfig) mvoMap.get(chainMVO);

			// double-check we have a pubkey for this MVO
			PublicKey pKey = m_Config.getPeerPubkey(chainMVO);
			if (pKey == null) {
				m_Log.error(lbl + "MVO " + chainMVO
							+ " defined for chain " + chainName
							+ ", but we have no pubkey for it");
				mvoConfsOk = false;
				continue;
			}
			propConf.configCommPubkey(pKey);

			// the signing address is required, unique per chain
			String signAddr = propConf.getSigningAddress();
			if (signAddr == null || signAddr.isEmpty()) {
				m_Log.error(lbl + "MVO " + chainMVO
							+ " defined for chain " + chainName
							+ ", but no signing address found");
				mvoConfsOk = false;
				continue;
			}

			// record URI so it's correct when we reference it
			URI propURI = propConf.getMVOURI();
			// fallback for URI calculated based on VPNDOMAIN default
			URI mvoURI = getURIforMVO(chainMVO);
			if (propURI == null) {
				m_Log.error(lbl + "MVO " + chainMVO
							+ " defined for chain " + chainName
							+ ", but no connect URI is set");
				mvoConfsOk = false;
				propConf.setMVOURI(mvoURI);
			}

		/* this code assumes that all MVOs are on the same domain or IP subnet
			// verify that what's in the properties matches calculated
			if (!mvoURI.equals(propURI)) {
				m_Log.warning(lbl + "MVO " + chainMVO
							+ " defined for chain " + chainName
							+ ", URI inconsistent, " + propURI
							+ " vs " + mvoURI + " calculated");
			}
		 */
		}
		return mvoConfsOk;
	}

	/**
	 * begin watching for MVOStaking record change events, after our calls to
	 * getOwnNodeConfig() and getOtherNodeConfigs()
	 * @param scc the smart contract we just achieved a connection t
	 * @return true on success
	 */
	public boolean setupMVOStakingListener(SmartContractConfig scc) {
		/* As an AUD, we don't care about updates to MVOStaking records,
		 * because we never select MVOs.  Adding completely new MVO requires
		 * properties config changes and therefore a restart.
		 */
		return true;
	}

	// method to implement EnftCacheResetter
	/**
	 * rebuild the ABI session and EnftCache for a specified blockchain
	 * @param chainId the chain ID (used to look up SmartContractConfig)
	 * @param reason the reason we were invoked (provided by old EnftCache)
	 * @return false if cache init is being deferred on a WS connection, true
	 * if cache init is proceeding normally
	 */
	public boolean requestEnftCacheRestart(long chainId, String reason) {
		final String lbl
			= this.getClass().getSimpleName() + ".requestEnftCacheRestart: ";

		// find the SCC object
		SmartContractConfig scConfig = m_SmartContracts.get(chainId);
		if (scConfig == null) {
			m_Log.error(lbl + "CRITICAL: no SCC found for existing chain Id "
						+ chainId);
			return false;
		}
		Log scLog = scConfig.log();
		scLog.error(lbl + "invoked due to: \"" + reason + "\"");

		// to prevent overlapping calls, check flag and set if required
		if (scConfig.isCacheReinitInProgress()) {
			scLog.error(lbl + "overlapping call for " + scConfig.getChainName()
						+ ", ignoring");
			return false;
		}
		scConfig.setCacheReinitInProgress(true);

		// shut down the existing cache
		scConfig.setEnabled(false);
		EnftCache oldCache = scConfig.getCache();
		if (oldCache != null) {
			oldCache.shutdown();
		}

		// pull latest block from old cache
		BigInteger previousLatestBlock = oldCache.getLatestBlock();

		// form a new connection to the Web3j URL
		Web3j abiSession = null;
		BlockchainAPI web3Api = null;
		URI web3jURI = scConfig.getWeb3jURI();
		String proto = web3jURI.getScheme();
		boolean deferCacheInit = false;
		if (proto.startsWith("http")) {
			// stop existing ABI interface
			Web3j abiInterface = scConfig.getABI();
			if (abiInterface != null) {
				abiInterface.shutdown();
			}

			scLog.debug(lbl + "reconnecting to " + scConfig.getChainName()
						+ " JSON-RPC @" + web3jURI.toString());
			// NB: regular HttpClient will be slow and shouldn't be used
			OkHttpClient okClient = new OkHttpClient();
			/* configure 60 sec timeout for fetching events, to match value
			 * org.web3j.protocol.websocket.WebSocketService.REQUEST_TIMEOUT
			 */
			OkHttpClient configClient = okClient.newBuilder()
									.readTimeout(60L, TimeUnit.SECONDS)
									.writeTimeout(60L, TimeUnit.SECONDS)
									.connectTimeout(60L, TimeUnit.SECONDS)
				.build();
			abiSession = Web3j.build(new HttpService(web3jURI.toString(),
									 configClient));
		}
		else if (proto.startsWith("ws")) {
			/* Defer to reconnect logic by invoking disconnect handler.
			 * We do this in preference to abiInterface.shutdown() because the
			 * latter will call scConfig.run() a second time on the close event.
			 */
			scConfig.run();		// (forks thread via WSS_Connector)

			/* Certain inits which need JSON-RPC we can't do yet.  Do not
			 * init EnftCache yet because it needs ABI session to operate.
			 */
			deferCacheInit = true;
		}

		// iff we created new ABI session (http case), init cache now
		if (abiSession != null && !deferCacheInit) {
			// record API object
			scConfig.configABISession(abiSession);

			// create the eNFT data cache, forwarding old previous last block
			EnftCache eCache
				= new EnftCache(scConfig, abiSession, previousLatestBlock);
			scConfig.configCache(eCache);
			if (!eCache.initialize(false)) {
				// NB: this can only happen due to a configuration error
				m_Log.error(lbl + "CRITICAL: error initializing EnftCache for "
							+ scConfig.getChainName() + ", exiting");
				System.exit(6);
			}
		}

		// reset nested call protection flag unless we deferred
		if (!deferCacheInit) {
			scConfig.setCacheReinitInProgress(false);
		}

		return !deferCacheInit;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			if (m_PropConfig != null) {
				m_PropConfig.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
