/*
 * last modified---
 * 	09-15-25 utilize unique logfile to isolate chain-specific outputs
 * 	04-08-25 prevent creation of duplicate WSS connections on EnftCache reinits
 * 	04-01-25 do full init on EnftCache if called by EnftCacheResetter
 * 	11-11-24 add m_DoingCacheReinit and get/set methods
 * 	11-07-24 pass BigInteger.ZERO as third arg to EnftCache constructor
 * 	10-30-24 add m_EnftCacheRebuilder
 * 	09-20-24 invoke EnftCache.createReinitializerTask() on reconnection cases
 * 	03-12-24 invoke setupMVOStakingListener() on reconnects
 * 	11-28-23 add m_EnftListener
 * 	07-07-23 add m_ConnectListener and its use
 * 	07-06-23 add m_WSS and configWSS(), WSS_Connector
 * 	06-29-23 add methods Consumer<String>.accept(), Runnable.onClose(), and
 * 			 Consumer<Throwable>.accept();
 * 			 move m_ABISession from BlockchainConfig
 * 	06-21-23 add m_DeploymentBlock, m_EnftCache, log(), m_BaseURI
 * 	06-14-23 remove AssetConfig usage
 * 	10-20-22 add m_Enabled field and methods
 * 	09-07-22 use abstract class MVOGenConfig
 * 	04-22-22 move m_TotalStaked here from BlockchainConfig
 * 	04-20-22 add m_ABIUrl
 * 	03-30-22 add m_DwellTime
 * 	02-17-22 new
 *
 * purpose---
 * 	encapsulate configuration data for a supported blockchain
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.web3j.protocol.websocket.WebSocketService;
import org.web3j.protocol.Web3j;

import java.util.Hashtable;
import java.util.function.Consumer;
import java.net.URI;
import java.net.ConnectException;
import java.math.BigInteger;
import java.security.SecureRandom;


/**
 * This class holds the configuration parameters specific to a particular
 * blockchain which Enshroud supports.  An ArrayList of these classes will
 * exist in the main classes for system objects.  The values will be loaded
 * from the corresponding records in the smart contract for the blockchain.
 * Certain values such as MVO IDs will be replicated in more than one instance.
 */
public final class SmartContractConfig implements Runnable, Consumer<String> {
	// BEGIN data members
	/**
	 * colloquial name of the blockchain, such as "mainnet"
	 */
	private String							m_Blockchain;

	/**
	 * chain ID for this chain, according to the standard (see chainlist.org)
	 */
	private long							m_ChainId;

	/**
	 * URI where we can connect to our ABI partner node to access this chain
	 * using Web3j (RPC-JSON endpoint, one of file://, http://, or ws://)
	 */
	private URI								m_ABIUri;

	/**
	 * the current connection to the ABI partner server (at m_ABIUri)
	 */
	private Web3j							m_ABI_Session;

	/**
	 * the active websocket service (if our JSON-RPC connection is ws://)
	 */
	private WebSocketService				m_WSS;

	/**
	 * Flag whether our RPC-JSON endpoint appears accessible (without this,
	 * we can't load our data for the other fields, or access event log, etc.).
	 * Volatile because numerous threads can access and manipulate this value.
	 */
	private volatile boolean				m_Enabled;

	/**
	 * base URI where user receipts are stored (not a URL in case the scheme is
	 * not http)
	 */
	private URI								m_ReceiptStore;

	/**
	 * MVOs which support this blockchain, IDs mapped to configs
	 */
	private Hashtable<String, MVOGenConfig>	m_MVOs;

	/**
	 * number of MVO signatures required
	 */
	private int								m_ReqSigs;

	/**
	 * dwell time in blocks on this chain (number of blocks that must elapse
	 * before an eNFT can be used as an input to another transaction)
	 */
	private int								m_DwellTime;

	/**
	 * total of $ENSHROUD staking on this blockchain by all MVOs
	 */
	private BigInteger						m_TotalStaked;

	/**
	 * genesis block at which EnshroudProtocol contract was deployed here
	 */
	private BigInteger						m_DeploymentBlock;

	/**
	 * EnshroudProtocol.baseURI value, prefaced on all eNFT metadata
	 */
	private String							m_BaseURI;

	/**
	 * cache object which downloads and holds eNFT data
	 */
	private EnftCache						m_EnftCache;

	/**
	 * flag used to prevent overlapping attempts to reinitialize m_EnftCache,
	 * which may be triggered by exception handling in multiple threads
	 */
	private volatile boolean				m_DoingCacheReinit;

	/**
	 * listener for blockchain ABI connection events
	 */
	private BlockchainConnectListener		m_ConnectListener;

	/**
	 * inner class to provide WSS connection drop listener, to handle reconnects
	 */
	private class WSS_Connector implements Runnable {
		/**
		 * flag indicating we're currently reconnecting
		 */
		public boolean		m_DoingReconnect;

		/**
		 * nullary constructor
		 */
		public WSS_Connector() { }

		/**
		 * handle connection closure
		 * NB: in cases where the connection is closed purposefully, this
		 * object should be removed first so that reconnection is not attempted.
		 * @param rng pseudo-random number generator
		 * @return true if the connection succeeded
		 */
		public boolean doReconnect(SecureRandom rng) {
			final String lbl
				= this.getClass().getSimpleName() + ".doReconnect: ";
			if (rng == null) {
				// this will loop if we return false; allow NPE below
				m_Log.error(lbl + "missing input RNG");
			}

			// wait random 5-10 seconds, so all L2 nodes don't reconnect at once
			int wait = rng.nextInt(5001) + 5000;
			try {
				Thread.sleep((long)wait);
			}
			catch (InterruptedException ie) {
				m_Log.error(lbl + "wait was interrupted", ie);
			}

			// perform reconnection
			m_WSS = new WebSocketService(m_ABIUri.toString(), false);

			/* Try to reconnect.  This will block (okay since we're in our
			 * own thread here), and use WebSocketClient.connectBlocking()
			 * logic rather than WebsocketClient.connect().
			 */
			try {
				m_WSS.connect(SmartContractConfig.this,
						throwable -> SmartContractConfig.this.accept(throwable),
							  SmartContractConfig.this);
				m_Log.debug(lbl + m_Blockchain + ": success reconnecting to "
							+ m_ABIUri);
			}
			catch (ConnectException ce) {
				m_Log.error(lbl + m_Blockchain + ": FAILED to reconnect to "
							+ m_ABIUri);
				return false;
			}
			return true;
		}

		/**
		 * implement Runnable (invoked from SmartContractConfig.run())
		 */
		public void run() {
			final String lbl = this.getClass().getSimpleName() + ".run: ";
			m_Log.warning(lbl + m_Blockchain
						+ ": WSS connection needs hard reconnect");
			// ensure that we do not trigger botchy library reconnect logic
			m_WSS = null;

			/* close old Web3j session if we had one (should have shutdown
			 * already on close unless we were called by EnftCacheResetter)
			 */
			if (m_ABI_Session != null) {
				m_DoingReconnect = true;
				m_ABI_Session.shutdown();
				m_ABI_Session = null;
				m_DoingReconnect = false;
			}

			// make repeated attempts to re-establish connection w/ random delay
			SecureRandom rng = new SecureRandom();
			boolean reconn = false;
			do {
				reconn = doReconnect(rng);
			} while (!reconn);
			// NB: m_WSS now set to open connection

			// restore ABI session in Web3j library
			m_ABI_Session = Web3j.build(m_WSS);

			// if we're called by EnftCacheResetter, do full init
			boolean reinit = !m_DoingCacheReinit;
			// we should have a cache already
			if (m_EnftCache == null) {
				/* this can happen if our "reconnect" is due to having started
				 * with the JSON-RPC node down or inaccessible
				 */
				m_Log.warning(lbl
							+ "should have EnftCache, but had to create new");
				m_EnftCache = new EnftCache(SmartContractConfig.this,
											m_ABI_Session,
											BigInteger.ZERO);
				reinit = false;
			}
			else {
				m_EnftCache.resetABI(m_ABI_Session);
				m_EnftCache.createReinitializerTask(0);
			}

			// init or reinit the cache
			if (m_EnftCache.initialize(reinit)) {
				m_Log.debug(lbl + "successfully re-ran init for " + m_Blockchain
							+ " following websocket reconnection");
				// NB: setEnabled(true) will be invoked by the EnftCache itself.
				// set EnftListener in the EnftCache to what we have recorded
				m_EnftCache.registerListener(m_EnftListener);

				// clear any indication we were reinitializing the cache
				setCacheReinitInProgress(false);
			}
			else {
				// NB: this can only happen due to a configuration error
				m_Log.error(lbl + "CRITICAL: error re-running cache init for "
							+ m_Blockchain
							+ " following websocket reconnection");
				/* TBD: MVO will not be able to serve this chain until
				 * fixed. This probably warrants a reconnect(chain)
				 * operator function on the future web admin console.
				 */
				System.exit(-4);
			}

			// do (or re-do) on-chain config download tasks
			if (!m_ConnectListener.getOwnNodeConfig(SmartContractConfig.this)) {
				m_Log.error(lbl + "init of self on-chain config failed");
				// NB: we do not abort here because we may have valid data avail
			}
			if (!m_ConnectListener.getOtherNodeConfigs(
													SmartContractConfig.this))
			{
				m_Log.error(lbl + "init of peer on-chain configs failed");
				// NB: we do not abort here because we may have valid data avail
			}
			if (!m_ConnectListener.setupMVOStakingListener(
													SmartContractConfig.this))
			{
				m_Log.error(lbl + "setup of MVOStaking listener failed");
				// NB: we do not abort here because we may have valid data avail
			}
		}
	}

	/**
	 * reconnection handler (used only when an existing connection drops)
	 */
	private WSS_Connector					m_WssReconnector;

	/**
	 * object capable of listening for eNFT mint/burn events
	 */
	private EnftListener					m_EnftListener;

	/**
	 * object capable of rebuilding the EnftCache after ABI session failure
	 */
	private EnftCacheResetter				m_EnftCacheRebuilder;

	/**
	 * logging object
	 */
	private Log								m_Log;

	// END data members
	
	// BEGIN methods
	/**
	 * constructor
	 * @param name blockchain name
	 * @param id chain Id
	 * @param log logging object
	 * @param listener ABI connect task handler
	 */
	public SmartContractConfig(String name,
							   long id,
							   Log log,
							   BlockchainConnectListener listener)
	{
		m_Blockchain = new String(name);
		m_ChainId = id;
		m_MVOs = new Hashtable<String, MVOGenConfig>();
		m_ReqSigs = 2;
		m_DwellTime = 80;
		m_TotalStaked = m_DeploymentBlock = BigInteger.ZERO;
		// add chainId to create a unique log file
		String logfile = log.getFile().getPath();
		int logIdx = logfile.indexOf(".log");
		String chainLog
			= logfile.substring(0, logIdx) + "-" + m_ChainId + ".log";
		m_Log = new Log(chainLog);
		m_Log.Init();
		m_BaseURI = Long.toString(m_ChainId) + "-URI:";
		m_ConnectListener = listener;
	}

	/**
	 * Receive a message from the Web3j session connected to our m_ABIUri.
	 * This method is added as a listener onto the underlying websocket
	 * connection.  Therefore it will see copies of replies to blockchain
	 * queries, subscribed events, etc.  We don't need to do anything with
	 * these messages, since they're also forwarded to the "real" callbacks
	 * established by Web3j.build() in those various contexts.  This method
	 * could however be used to tap into ABI communications for debugging.
	 * @param message the message sent to us as a listener on the ABI websocket
	 */
	public void accept(String message) {
	/*
		m_Log.debug("SCC.accept for " + m_Blockchain + ": got message: \""
					+ message + "\"");
	 */
	}

	/**
	 * Receive an exception thrown on the Web3j websocket connected to m_ABIURI.
	 * This method is required by the 3-argument WebSocketService.connect().
	 * In practical fact, this should never be called absent exceptions at the
	 * websocket transport layer.  So we simply log anything received.
	 * @param exception the exception reported
	 */
	public void accept(Throwable exception) {
		m_Log.error("SCC.accept for " + m_Blockchain + ": got exception: "
					+ exception.toString());
	}

	/**
	 * Implements a Runnable which is called when the websocket connection to
	 * m_ABIURI is closed (for any reason).  When this occurs, there are
	 * certain actions we need to take to reconnect and restore the completion
	 * of the eNFT data in the m_EnftCache before becoming "ready" to continue
	 * processing requests for this chain.
	 */
	public void run() {
		final String lbl = this.getClass().getSimpleName() + ".run: ";
		m_Log.error(lbl + m_Blockchain + ": got websocket close event");

		// we can no longer handle client events on this chain
		m_Enabled = false;

		/* We can get called during a reconnect by the WebSocket layer if we
		 * do a shutdown() on an existing connection not already terminated.
		 * In this case we'll already have a WSS_Connector with the flag set.
		 */
		if (m_WssReconnector != null && m_WssReconnector.m_DoingReconnect) {
			m_Log.debug(lbl + "skipped nested reconnection attempt");
			return;
		}

		// reconnect with a delay
		m_WssReconnector = new WSS_Connector();
		new Thread(m_WssReconnector).start();
	}

	// GET methods
	/**
	 * obtain chain name
	 * @return the colloquial name of the blockchain
	 */
	public String getChainName() { return m_Blockchain; }

	/**
	 * obtain chain Id
	 * @return the Id, set according to the industry standards
	 */
	public long getChainId() { return m_ChainId; }

	/**
	 * obtain the ABI node URI
	 * @return the URI where we can connect to our local partner node for Web3j
	 */
	public URI getWeb3jURI() { return m_ABIUri; }

	/**
	 * obtain the ABI interface object
	 * @return the object used to talk to the blockchain
	 */
	public Web3j getABI() { return m_ABI_Session; }

	/**
	 * obtain whether this MVO/AUD process is able to access the RPC-JSON URI
	 * @return true if ABI connection established and initial events downloaded
	 */
	public boolean isEnabled() { return m_Enabled; }

	/**
	 * obtain the receipt storage URI
	 * @return the resource identifier for accessing stored receipts
	 */
	public URI getReceiptURI() { return m_ReceiptStore; }

	/**
	 * obtain the complete map of all configured MVOs, indexed by MVO Id
	 * @return the mapping
	 */
	public Hashtable<String, MVOGenConfig> getMVOMap() { return m_MVOs; }

	/**
	 * obtain the number of MVO sigs required
	 * @return the number of signatures (N, in "N of M")
	 */
	public int getNumSigs() { return m_ReqSigs; }

	/**
	 * obtain the number of blocks an eNFT must "dwell" before being an input
	 * @return the number of blocks to season an eNFT
	 */
	public int getDwellTime() { return m_DwellTime; }

	/**
	 * obtain the number of MVOs. (Does not account for disabled MVOs; it is a
	 * problem if this value is less than number of required sigs.)
	 * @return the number of MVOs (M, in "N of M")
	 */
	public int getNumMVOs() { return m_MVOs.size(); }

	/**
	 * obtain the total quantity of $ENSHROUD staked by all MVOs
	 * @return the amount, in 18 digit precision
	 */
	public BigInteger getTotalStaking() { return m_TotalStaked; }

	/**
	 * obtain the genesis block for the EnshroudProtocol contract
	 * @return the block number in which the contract was deployed on this chain
	 */
	public BigInteger getDeploymentBlock() { return m_DeploymentBlock; }

	/**
	 * obtain the baseURI prefix for this chain
	 * @return the string prepended onto all eNFT encrypted metadata
	 */
	public String getBaseURI() { return m_BaseURI; }

	/**
	 * obtain the cache object
	 * @return the eNFT data cache for this chain
	 */
	public EnftCache getCache() { return m_EnftCache; }

	/**
	 * determine whether the cache object is currently being reinitialized
	 * @return true if cache reinit is in progress
	 */
	public boolean isCacheReinitInProgress() { return m_DoingCacheReinit; }

	/**
	 * obtain the eNFT event listener
	 * @return the listener
	 */
	public EnftListener getEnftListener() { return m_EnftListener; }

	/**
	 * obtain the cache rebuildier
	 * @return the object which can rebuild the m_ABI_Session and m_EnftCache
	 */
	public EnftCacheResetter getCacheRebuilder() {
		return m_EnftCacheRebuilder;
	}

	/**
	 * obtain the logging record
	 * @return the logger object
	 */
	public Log log() { return m_Log; }


	// SET methods
	/**
	 * configure the ABI URI
	 * @param url the local URI where we have Web3j access to this chain
	 */
	public void setWeb3jURI(URI uri) {
		if (uri != null) {
			m_ABIUri = uri;
		}
		else {
			m_Log.error("setWeb3jURI(): missing ABI URI");
		}
	}

	/**
	 * configure the ABI object
	 * @param web3j the object used to communicate with the blockchain
	 */
	public void configABISession(Web3j web3j) {
		if (web3j != null) {
			m_ABI_Session = web3j;
		}
	}

	/**
	 * configure the WSS object
	 * @param wss the websocket service
	 */
	public void configWSS(WebSocketService wss) {
		if (wss != null) {
			m_WSS = wss;
		}
	}

	/**
	 * set whether we have access to the RPC-JSON URI and can do stuff on chain
	 * @param enabled true if access appears available (set to false on failure)
	 */
	public void setEnabled(boolean enabled) { m_Enabled = enabled; }

	/**
	 * configure the receipt storage URI
	 * @param storage where receipts can be obtained or stored
	 */
	public void setReceiptStore(URI storage) {
		if (storage != null) {
			m_ReceiptStore = storage;
		}
		else {
			m_Log.error("setReceiptStore(): missing storage URI");
		}
	}

	/**
	 * configure the number of required signatures
	 * @param sigs number of MVO signatures required
	 */
	public void setNumSigs(int sigs) {
		if (sigs > 0) {
			m_ReqSigs = sigs;
		}
		else {
			m_Log.error("setNumSigs(): illegal req sigs, " + sigs);
		}
	}

	/**
	 * configure the number of required blocks to season a new eNFT
	 * @param blocks number of blocks an eNFT must dwell
	 */
	public void setDwellTime(int blocks) {
		if (blocks > 0) {
			m_DwellTime = blocks;
		}
		else {
			m_Log.error("setDwellTime(): illegal dwell time, " + blocks);
		}
	}

	/**
	 * configure the staking total
	 * @param stake the amount of $ENSHROUD all MVOs have staked, 18 digit
	 *	precision, represented as ascii digits
	 */
	public void setTotalStaking(String stake) {
		if (stake != null && !stake.isEmpty()) {
			try {
				m_TotalStaked = new BigInteger(stake);
			}
			catch (NumberFormatException nfe) {
				m_Log.error("setTotalStaking(): illegal total stake, " + stake,
							nfe);
			}
		}
		else {
			m_Log.error("setTotalStaking(): missing total stake");
		}
	}

	/**
	 * configure the genesis block
	 * @param block the block number in which EnshroudProtocol was deployed here
	 */
	public void setDeploymentBlock(BigInteger block) {
		if (block != null) {
			m_DeploymentBlock = block;
		}
		else {
			m_Log.error("setDeploymentBlock(): missing block number");
		}
	}

	/**
	 * configure the baseURI value
	 * @param prefix the chars prefixed to eNFT encrypted metadata on URI events
	 */
	public void setBaseURI(String prefix) {
		if (prefix != null) {
			m_BaseURI = new String(prefix);
		}
	}

	/**
	 * configure the eNFT data cache (points back to us)
	 * @param cache the caching object
	 */
	public void configCache(EnftCache cache) {
		if (cache == null || cache.getConfig() != this) {
			m_Log.error("SmartContractConfig.configCache: improper cache obj");
		}
		else {
			m_EnftCache = cache;
		}
	}

	/**
	 * set or clear cache reinit flag
	 * @param reinit whether reinit is in progress
	 */
	public void setCacheReinitInProgress(boolean reinit) {
		m_DoingCacheReinit = reinit;
	}

	/**
	 * configure the eNFT mint/burn listener (points at ReceiptQueue for MVOs,
	 * at a GreyLister for AUDs)
	 * @param listener the class which implements EnftListener
	 */
	public void configEnftListener(EnftListener listener) {
		m_EnftListener = listener;
	}

	/**
	 * configure the EnftCache rebuilder (points at MVO for MVOs, AUD for AUDs)
	 * @param rebuilder the class which implements EnftCacheResetter
	 */
	public void configCacheRebuilder(EnftCacheResetter rebuilder) {
		m_EnftCacheRebuilder = rebuilder;
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_MVOs != null) {
				m_MVOs.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
