/*
 * last modified---
 * 	04-05-23 make address types get emitted as EIP-55
 * 	03-02-23 add calcSignature() and getSigAddr()
 * 	07-20-22 fix extra comma before a blank memo
 * 	07-11-22 move {signature} inside {receipt}, add buildFromDecrypted()
 * 	07-06-22 allow block number to be empty in buildSignedData()
 * 	06-29-22 add buildFromMap(), buildFromString() and setSignature()
 * 	05-27-22 correct array JSON emission
 * 	03-23-22 emit {source} first, to reduce commonality of early bytes;
 * 			 add m_ReceiptId and methods
 * 	03-04-22 new
 *
 * purpose---
 * 	encapsulate receipts created for payers and payees of spend transactions
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import org.web3j.crypto.Keys;

import java.util.ArrayList;
import java.util.Map;
import java.math.BigInteger;
import java.text.NumberFormat;
import java.io.IOException;


/**
 * This class holds the data necessary to build {ReceiptBlock} JSON objects
 * which are stored for participants in eNFT spend transactions.  These receipts
 * are encrypted using the AES key for each one, and then stored on external
 * storage at the hash of the user's address.  This class knows how to emit
 * itself as JSON, and parse itself from JSON.
 */
public final class ReceiptBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * constant for recipient receipts
	 */
	public final String		M_Recipient = "recipient";

	/**
	 * constant for sender receipts
	 */
	public final String		M_Sender = "sender";

	/**
	 * the ID of the receipt record (set by MVO, identifies AES key used for
	 * encrypting the record)
	 */
	private String			m_ReceiptId;

	/**
	 * type of receipt, sender or recipient
	 */
	private String			m_ReceiptType;

	/**
	 * chain Id of the blockchain on which the transaction took place, according
	 * to chainlist.org
	 */
	private long			m_ChainId;

	/**
	 * block number on the chain in which the transaction took place
	 * (identifies approximate time of transaction).  This is almost certainly
	 * an integer quantity, but is stored as a String just in case some chain
	 * uses alphanumeric blocks.
	 */
	private String			m_BlockNumber;

	/**
	 * tag ID of the eNFT whose appearance in a mint or burn operation within
	 * a block triggers the finalization and storage of this receipt.  (This
	 * field is not actually emitted in the signed JSON output.)
	 */
	private String			m_TagID;

	/**
	 * payment source address (for a sender receipt, this is the user's own
	 * address; for a recipient receipt this identifies the payer; EIP-55)
	 */
	private String			m_Source;

	/**
	 * helper class describing a payee in a receipt context
	 */
	public class ReceiptPayee {
		/**
		 * payee sequence number, such as payee001, payee002, etc.
		 */
		public String		m_Payee;

		/**
		 * payee address (EIP-55 format)
		 */
		public String		m_Address;

		/**
		 * asset paid to payee (token address, EIP-55)
		 */
		public String		m_Asset;

		/**
		 * amount of asset paid to payee (quantity, in {decimals} precision)
		 */
		public BigInteger	m_Amount;

		/**
		 * ID of eNFT minted to payee (not included for sender type receipts)
		 */
		public String		m_ID;

		/**
		 * optional memo field from payer to payee
		 */
		public String		m_Memo;

		/**
		 * nullary constructor
		 */
		public ReceiptPayee() {
			m_Payee = m_Address = m_Asset = m_ID = m_Memo = "";
			m_Amount = new BigInteger("0");
		}
	}

	/**
	 * list of payees defined in receipt
	 */
	private ArrayList<ReceiptPayee>	m_Payees;

	/**
	 * the MVO signature on the receipt
	 */
	private MVOSignature	m_MVOSignature;

	/**
	 * the complete JSON (signed), encrypted to the appropriate AES key
	 */
	private String			m_EncReceipt;

	/**
	 * error logging object
	 */
	private Log				m_Log;

	/**
	 * error code status (0 indicates success)
	 */
	private int				m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param log the logging object
	 */
	public ReceiptBlock(Log log) {
		m_Log = log;
		m_ReceiptType = m_TagID = m_Source = m_BlockNumber = m_EncReceipt = "";
		m_ReceiptId = "";
		m_Payees = new ArrayList<ReceiptPayee>();
		m_MVOSignature = new MVOSignature();
	}

	// GET methods
	/**
	 * obtain the receipt ID
	 * @return the ID, used to identify the AES key needed for decryption
	 */
	public String getReceiptId() { return m_ReceiptId; }

	/**
	 * obtain the receipt type
	 * @return the type, one of sender or recipient
	 */
	public String getReceiptType() { return m_ReceiptType; }

	/**
	 * obtain the chain Id on which the transaction occurred
	 * @return the Id of the blockchain, per the standard
	 */
	public long getChainId() { return m_ChainId; }

	/**
	 * obtain the number of the block in which the transaction was a part
	 * @return the block ID (probably numeric, but string just in case)
	 */
	public String getBlockId() { return m_BlockNumber; }

	/**
	 * obtain the block number if it's a number, or 0 if it wasn't
	 * @return the block number as a numeric quantity, or 0
	 */
	public long getBlockNumber() {
		try {
			Long block = Long.valueOf(m_BlockNumber.trim());
			return block.longValue();
		}
		catch (NumberFormatException nfe) { /* ignore */ }
		return 0L;
	}

	/**
	 * obtain the tag ID of the trigger eNFT
	 * @return the ID of the eNFT whose presence in a block triggers the receipt
	 */
	public String getTagID() { return m_TagID; }

	/**
	 * obtain the source (payer) address
	 * @return the address on the chain which provided the source of funds
	 */
	public String getSource() { return m_Source; }

	/**
	 * obtain the list of receipt payees
	 * @return the list of payees (not a clone to permit modification)
	 */
	public ArrayList<ReceiptPayee> getPayees() { return m_Payees; }

	/**
	 * obtain the signature
	 * @return the MVO signature on the receipt
	 */
	public MVOSignature getSignature() { return m_MVOSignature; }

	/**
	 * obtain the encrypted data
	 * @return the encrypted version of the complete JSON, empty if not enc
	 */
	public String getEncData() { return m_EncReceipt; }

	/**
	 * obtain the last error code
	 * @return the last error encountered by buildSignedData(), 0 is success
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods
	/**
	 * set the receipt ID
	 * @param rId the ID of the receipt (set by MVO)
	 */
	public void setReceiptId(String rId) {
		if (rId != null && !rId.isEmpty()) {
			m_ReceiptId = new String(rId);
		}
	}

	/**
	 * set the receipt type
	 * @param type (must be one of "recipient" or "sender")
	 */
	public void setReceiptType(String type) {
		if (type == null
			|| !(type.equals(M_Sender) || type.equals(M_Recipient)))
		{
			return;
		}
		m_ReceiptType = new String(type);
	}

	/**
	 * set the chain Id
	 * @param cid the ID of the blockchain (according to the standard)
	 */
	public void setChainId(long cid) {
		if (cid > 0L) m_ChainId = cid;
	}

	/**
	 * set the block ID
	 * @param block the ID of the block in which the receipt trans occurred
	 */
	public void setBlockId(String block) {
		if (block != null && !block.isEmpty()) {
			m_BlockNumber = new String(block);
		}
	}

	/**
	 * set the block number
	 * @param bNum the ID of the block the transaction was mined in (as long)
	 */
	public void setBlockNumber(long bNum) {
		if (bNum > 0) {
			m_BlockNumber = Long.toString(bNum);
		}
	}

	/**
	 * set the tag ID
	 * @param tId the tag ID indicating the receipt transaction took place
	 */
	public void setTagID(String tId) {
		if (tId != null) {
			m_TagID = new String(tId);
		}
	}

	/**
	 * set the source address
	 * @param src the address which provided the input eNFTs for the transaction
	 */
	public void setSource(String src) {
		if (src != null && !src.isEmpty()) {
			m_Source = Keys.toChecksumAddress(src);
		}
	}

	/**
	 * add a destination payee to a receipt
	 * @param payee the payee specification
	 */
	public void addPayee(ReceiptPayee payee) {
		if (payee != null) {
			m_Payees.add(payee);
		}
	}

	/**
	 * set the encrypted data
	 * @param enc the full JSON data (from addJSON()) encrypted to AES key
	 */
	public void setEncData(String enc) {
		if (enc != null) {
			m_EncReceipt = new String(enc);
		}
	}

	/**
	 * configure the signature
	 * @param sig the MVO signature on the receipt
	 */
	public void setSignature(MVOSignature sig) {
		if (sig != null) {
			m_MVOSignature = sig;
		}
	}

	/**
	 * emit the signed part of the object as JSON ({receipt} element)
	 * @param tags true if tags should be included in the output (this is not
	 * the case when we're actually preparing data for signature)
	 * @param sig true if we should emit signature as well (used for building
	 * the data for encryption)
	 * @return the exact JSON string that the MVO will sign, or null on error
	 */
	public String buildSignedData(boolean tags, boolean sig) {
		StringBuilder out = new StringBuilder(4096);
		m_ErrCode = 0;
		out.append("{");
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		final String label
			= this.getClass().getSimpleName() + ".buildSignedData: ";

		// add source of the inputs for the transaction (identifies payer)
		if (m_Source.isEmpty()) {
			m_ErrCode = 4;
			m_Log.error(label + "missing source address");
			return null;
		}
		// NB: address is already in EIP-55 format due to setSource()
		out.append("\"source\":\"" + m_Source + "\",");

		// add receipt type
		if (m_ReceiptType.isEmpty()) {
			m_ErrCode = 1;
			m_Log.error(label + "missing receipt type");
			return null;
		}
		out.append("\"receiptType\":\"" + m_ReceiptType + "\",");

		// add chain Id
		if (m_ChainId <= 0L) {
			m_ErrCode = 2;
			m_Log.error(label + "missing chain Id");
			return null;
		}
		out.append("\"chainId\":\"" + m_ChainId + "\",");

		// add block number
		// NB: this can be empty when the receipt isn't finalized or signed yet
		out.append("\"block\":\"" + m_BlockNumber + "\",");

		// NB: tagID is not included in the signed JSON output
		if (tags && !m_TagID.isEmpty()) {
			out.append("\"tagID\":\"" + m_TagID + "\",");
		}

		// add all the payee destinations defined
		if (m_Payees.isEmpty()) {
			m_ErrCode = 5;
			m_Log.error(label + "missing payee spec");
			return null;
		}
		if (m_Payees.size() > 1 && m_ReceiptType.equals(M_Recipient)) {
			m_ErrCode = 6;
			m_Log.error(label + "more than one payee "
						+ "included for a recipient receipt");
			return null;
		}
		int outSeq = 1;
		out.append("\"destinations\":[");
		for (ReceiptPayee rp : m_Payees) {
			// auto-fill payee if not already set
			if (rp.m_Payee.isEmpty()) {
				rp.m_Payee = nf.format(outSeq);
			}
			out.append("{\"" + rp.m_Payee + "\":{");
			out.append("\"address\":\"" + Keys.toChecksumAddress(rp.m_Address)
						+ "\",");
			out.append("\"asset\":\"" + Keys.toChecksumAddress(rp.m_Asset)
						+ "\",");
			out.append("\"amount\":\"" + rp.m_Amount.toString() + "\"");
			if (m_ReceiptType.equals(M_Recipient)) {
				out.append(",\"id\":\"" + rp.m_ID + "\"");
			}
			if (!rp.m_Memo.isEmpty()) {
				out.append(",\"memo\":\"" + rp.m_Memo + "\"");
			}
			// end the m_Payee object, with comma unless last
			if (outSeq++ < m_Payees.size()) {
				out.append("}},");
			}
			else {
				out.append("}}]");
			}
		}

		// add signature if requested
		if (sig) {
			out.append(",\"signature\":{");
			out.append("\"signer\":\"" + m_MVOSignature.m_Signer + "\",");
			out.append("\"sig\":\"" + m_MVOSignature.m_Signature + "\"}");
		}
		out.append("}");
		return out.toString();
	}

	/**
	 * method to build receipt from a Map
	 * @param receipt the mapping, the result of a JSON parse
	 * @return true on success
	 */
	public boolean buildFromMap(Map receipt) {
		final String label
			= this.getClass().getSimpleName() + ".buildFromMap: ";
		if (receipt == null || receipt.isEmpty()) {
			m_Log.error("buildFromMap(): missing receipt Map");
			return false;
		}
		boolean ret = true;

		// first we must have a ReceiptBlock
		Object rBlock = receipt.get("ReceiptBlock");
		Map rctMap = null;
		if (rBlock instanceof Map) {
			rctMap = (Map) rBlock;

			// required: the receiptId
			Object rId = rctMap.get("receiptId");
			if (rId instanceof String) {
				String id = (String) rId;
				setReceiptId(id);
			}
			else {
				ret = false;
				m_ErrCode = 3;
				m_Log.error(label + "missing receiptId");
			}
		}
		else {
			m_Log.error(label + "{ReceiptBlock} JSON appears corrupt");
			m_ErrCode = 16;
			ret = false;
		}
		// no point in going further
		if (!ret) return ret;

		// required: the receipt section
		Object rctObj = rctMap.get("receipt");
		// if parsed object is a simple string, it's encrypted
		if (rctObj instanceof String) {
			String encData = (String) rctObj;
			setEncData(encData);
		}
		// if it's a Map, try to parse it in the regular manner
		else if (rctObj instanceof Map) {
			Map receiptMap = (Map) rctObj;

			// required: source
			Object src = receiptMap.get("source");
			if (src instanceof String) {
				String source = (String) src;
				setSource(source);
			}
			else {
				m_Log.error(label + "missing source");
				m_ErrCode = 4;
				ret = false;
			}

			// required: receiptType
			Object typ = receiptMap.get("receiptType");
			if (typ instanceof String) {
				String type = (String) typ;
				setReceiptType(type);
			}
			else {
				m_Log.error(label + "missing receiptType");
				m_ErrCode = 5;
				ret = false;
			}

			// required: chainId
			Object chId = receiptMap.get("chainId");
			if (chId instanceof String) {
				String chain = (String) chId;
				try {
					Long cId = Long.parseLong(chain.trim());
					setChainId(cId);
				}
				catch (NumberFormatException nfe) {
					m_Log.error(label + "illegal chainId", nfe);
					m_ErrCode = 6;
					ret = false;
				}
			}
			else {
				m_Log.error(label + "missing chainId");
				m_ErrCode = 7;
				ret = false;
			}

			// required: block number (can be empty)
			Object blk = receiptMap.get("block");
			if (blk instanceof String) {
				String block = (String) blk;
				setBlockId(block);
			}
			else {
				m_Log.error(label + "missing block num");
				m_ErrCode = 8;
				ret = false;
			}

			// optional: tagID
			Object tag = receiptMap.get("tagID");
			if (tag instanceof String) {
				String tagId = (String) tag;
				setTagID(tagId);
			}

			// required: destinations[]
			Object dest = receiptMap.get("destinations");
			if (dest instanceof Object[]) {
				NumberFormat nf = NumberFormat.getIntegerInstance();
				nf.setMinimumIntegerDigits(3);
				Object[] payees = (Object[]) dest;
				for (int pIdx = 0; pIdx < payees.length; pIdx++) {
					ReceiptPayee rPayee = new ReceiptPayee();

					// each array elt must be a map
					Object payObj = payees[pIdx];
					if (payObj instanceof Map) {
						Map payee = (Map) payObj;
						String idx = "payee" + nf.format(pIdx+1);
						rPayee.m_Payee = new String(idx);
						Object payeeObj = payee.get(idx);

						// this also must be a map
						if (payeeObj instanceof Map) {
							Map payeeMap = (Map) payeeObj;

							// required: address
							Object adr = payeeMap.get("address");
							if (adr instanceof String) {
								String addr = (String) adr;
								rPayee.m_Address = new String(addr);
							}
							else {
								m_Log.error(label + "missing address for "
											+ idx);
								m_ErrCode = 9;
								ret = false;
								break;
							}

							// required: asset
							Object ast = payeeMap.get("asset");
							if (ast instanceof String) {
								String asset = (String) ast;
								rPayee.m_Asset = new String(asset);
							}
							else {
								m_Log.error(label + "missing asset for "
											+ idx);
								m_ErrCode = 10;
								ret = false;
								break;
							}

							// required: amount
							Object amt = payeeMap.get("amount");
							if (amt instanceof String) {
								String amount = (String) amt;
								rPayee.m_Amount = new BigInteger(amount);
							}
							else {
								m_Log.error(label + "missing amount for "
											+ idx);
								m_ErrCode = 11;
								ret = false;
								break;
							}

							// recipient type receipts only: eNFT id
							if (m_ReceiptType.equals(M_Recipient)) {
								Object idObj = payeeMap.get("id");
								if (idObj instanceof String) {
									String eId = (String) idObj;
									rPayee.m_ID = new String(eId);
								}
								else {
									m_Log.error(label + "missing ID for "
												+ idx);
									m_ErrCode = 12;
									ret = false;
									break;
								}
							}

							// optional: memo
							Object memoObj = payeeMap.get("memo");
							if (memoObj instanceof String) {
								String memo = (String) memoObj;
								rPayee.m_Memo = new String(memo);
							}
						}
						else {
							m_Log.error(label + "no map for " + idx);
							m_ErrCode = 13;
							ret = false;
							break;
						}
					}
					else {
						m_Log.error(label + "no map for payee " + (pIdx+1));
						m_ErrCode = 14;
						ret = false;
						break;
					}
					addPayee(rPayee);
				}
			}
			else {
				m_Log.error(label + "missing destinations");
				m_ErrCode = 15;
				ret = false;
			}

			// {signature} section
			Object sigObj = receiptMap.get("signature");
			if (sigObj instanceof Map) {
				Map sigSection = (Map) sigObj;

				// required: signer
				Object signerObj = sigSection.get("signer");
				if (signerObj instanceof String) {
					String signer = (String) signerObj;
					m_MVOSignature.m_Signer = new String(signer);
				}
				else {
					m_Log.error(label + "missing signature.signer");
					m_ErrCode = 17;
					ret = false;
				}

				// required: sig
				Object sigObject = sigSection.get("sig");
				if (sigObject instanceof String) {
					String sig = (String) sigObject;
					m_MVOSignature.m_Signature = new String(sig);
				}
				else {
					m_Log.error(label + "missing signature.sig");
					m_ErrCode = 18;
					ret = false;
				}
			}
			else {
				// at this point the object should be signed
				m_Log.warning(label + "{receipt} JSON is not signed");
			}
		}
		else {
			m_ErrCode = 2;
			ret = false;
			m_Log.error(label + "{receipt} JSON appears corrupt");
		}
		return ret;
	}

	/**
	 * method to build entire receipt object from a JSON string
	 * @param rctData the receipt data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String rctData) {
		boolean ret = false;
		final String label
			= this.getClass().getSimpleName() + ".buildFromString: ";
		if (rctData == null || rctData.isEmpty()) {
			m_Log.error(label + "missing receipt data");
			return ret;
		}
		Object rct = null;
		m_ErrCode = 0;
		try {
			rct = JSON.parse(rctData);
			// must be a map
			if (rct instanceof Map) {
				Map rctMap = (Map) rct;
				ret = buildFromMap(rctMap);
			}
			else {
				m_ErrCode = 2;
				m_Log.error(label + "missing ReceiptBlock Map");
			}
		}
		catch (IllegalStateException ise) {
			m_ErrCode = 1;
			m_Log.error(label + "receipt JSON parse failed", ise);
			return ret;
		}
		return ret;
	}

	/**
	 * build the receipt and set its data fields based on the values found in a
	 * decrypted version of m_EncReceipt
	 * @param decReceipt the decrypted {receipt} section, including delimiters
	 * @return true on success
	 */
	public boolean buildFromDecrypt(String decReceipt) {
		boolean ret = false;
		final String label
			= this.getClass().getSimpleName() + ".buildFromDecrypt: ";
		if (decReceipt == null || !decReceipt.startsWith("{")) {
			m_Log.error("missing decrypted JSON");
			return ret;
		}

		Object rct = null;
		m_ErrCode = 0;
		try {
			rct = JSON.parse(decReceipt);
		}
		catch (IllegalStateException ise) {
			m_ErrCode = 1;
			m_Log.error(label + "decrypted {receipt} JSON parse failed",
						ise);
			return ret;
		}

		// must be a map
		if (!(rct instanceof Map)) {
			m_Log.error(label + "input not a Map");
			m_ErrCode = 2;
			return ret;
		}
		Map receiptMap = (Map) rct;
		ret = true;

		// required: source
		Object src = receiptMap.get("source");
		if (src instanceof String) {
			String source = (String) src;
			setSource(source);
		}
		else {
			m_Log.error(label + "missing source");
			m_ErrCode = 4;
			ret = false;
		}

		// required: receiptType
		Object typ = receiptMap.get("receiptType");
		if (typ instanceof String) {
			String type = (String) typ;
			setReceiptType(type);
		}
		else {
			m_Log.error(label + "missing receiptType");
			m_ErrCode = 5;
			ret = false;
		}

		// required: chainId
		Object chId = receiptMap.get("chainId");
		if (chId instanceof String) {
			String chain = (String) chId;
			try {
				Long cId = Long.parseLong(chain.trim());
				setChainId(cId);
			}
			catch (NumberFormatException nfe) {
				m_Log.error(label + "illegal chainId", nfe);
				m_ErrCode = 6;
				ret = false;
			}
		}
		else {
			m_Log.error(label + "missing chainId");
			m_ErrCode = 7;
			ret = false;
		}

		// required: block number
		Object blk = receiptMap.get("block");
		if (blk instanceof String) {
			String block = (String) blk;
			// at this point (decrypting a signed object) must have block #
			if (block.isEmpty()) {
				m_Log.error(label + "missing block num");
				m_ErrCode = 8;
				ret = false;
			}
			else {
				setBlockId(block);
			}
		}
		else {
			m_Log.error(label + "missing block num");
			m_ErrCode = 8;
			ret = false;
		}

		// optional: tagID
		Object tag = receiptMap.get("tagID");
		if (tag instanceof String) {
			String tagId = (String) tag;
			setTagID(tagId);
		}

		// required: destinations[]
		Object dest = receiptMap.get("destinations");
		if (dest instanceof Object[]) {
			NumberFormat nf = NumberFormat.getIntegerInstance();
			nf.setMinimumIntegerDigits(3);
			Object[] payees = (Object[]) dest;
			for (int pIdx = 0; pIdx < payees.length; pIdx++) {
				ReceiptPayee rPayee = new ReceiptPayee();

				// each array elt must be a map
				Object payObj = payees[pIdx];
				if (payObj instanceof Map) {
					Map payee = (Map) payObj;
					String idx = "payee" + nf.format(pIdx+1);
					rPayee.m_Payee = new String(idx);
					Object payeeObj = payee.get(idx);

					// this also must be a map
					if (payeeObj instanceof Map) {
						Map payeeMap = (Map) payeeObj;

						// required: address
						Object adr = payeeMap.get("address");
						if (adr instanceof String) {
							String addr = (String) adr;
							rPayee.m_Address = new String(addr);
						}
						else {
							m_Log.error(label + "missing address for "
										+ idx);
							m_ErrCode = 9;
							ret = false;
							break;
						}

						// required: asset
						Object ast = payeeMap.get("asset");
						if (ast instanceof String) {
							String asset = (String) ast;
							rPayee.m_Asset = new String(asset);
						}
						else {
							m_Log.error(label + "missing asset for "
										+ idx);
							m_ErrCode = 10;
							ret = false;
							break;
						}

						// required: amount
						Object amt = payeeMap.get("amount");
						if (amt instanceof String) {
							String amount = (String) amt;
							rPayee.m_Amount = new BigInteger(amount);
						}
						else {
							m_Log.error(label + "missing amount for "
										+ idx);
							m_ErrCode = 11;
							ret = false;
							break;
						}

						// recipient type receipts only: eNFT id
						if (m_ReceiptType.equals(M_Recipient)) {
							Object idObj = payeeMap.get("id");
							if (idObj instanceof String) {
								String eId = (String) idObj;
								rPayee.m_ID = new String(eId);
							}
							else {
								m_Log.error(label + "missing ID for "
											+ idx);
								m_ErrCode = 12;
								ret = false;
								break;
							}
						}

						// optional: memo
						Object memoObj = payeeMap.get("memo");
						if (memoObj instanceof String) {
							String memo = (String) memoObj;
							rPayee.m_Memo = new String(memo);
						}
					}
					else {
						m_Log.error(label + "no map for " + idx);
						m_ErrCode = 13;
						ret = false;
						break;
					}
				}
				else {
					m_Log.error(label + "no map for payee " + (pIdx+1));
					m_ErrCode = 14;
					ret = false;
					break;
				}
				addPayee(rPayee);
			}
		}
		else {
			m_Log.error(label + "missing destinations");
			m_ErrCode = 15;
			ret = false;
		}

		// {signature} section
		Object sigObj = receiptMap.get("signature");
		if (sigObj instanceof Map) {
			Map sigSection = (Map) sigObj;

			// required: signer
			Object signerObj = sigSection.get("signer");
			if (signerObj instanceof String) {
				String signer = (String) signerObj;
				m_MVOSignature.m_Signer = new String(signer);
			}
			else {
				m_Log.error(label + "missing signature.signer");
				m_ErrCode = 17;
				ret = false;
			}

			// required: sig
			Object sigObject = sigSection.get("sig");
			if (sigObject instanceof String) {
				String sig = (String) sigObject;
				m_MVOSignature.m_Signature = new String(sig);
			}
			else {
				m_Log.error(label + "missing signature.sig");
				m_ErrCode = 18;
				ret = false;
			}
		}
		else {
			// at this point the object should be signed
			m_Log.error(label + "{receipt} JSON is not signed");
			m_ErrCode = 19;
			ret = false;
		}
		return ret;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream on which to write
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(5120);
		m_ErrCode = 0;
		String rctText = "";
		if (m_EncReceipt.isEmpty()) {
			// emit object, including tags (set that field to "" to avoid)
			rctText = buildSignedData(true, true);
			if (m_ErrCode != 0) {
				m_Log.error("ReceiptBlock.addJSON: buildSignedData() failed");
				return;		// abort
			}
		}
		else {
			// emit encrypted version (as string not map)
			rctText = "\"" + m_EncReceipt + "\"";
		}
		out.append("{\"ReceiptBlock\":{\"receiptId\":\"" + m_ReceiptId + "\"");
		out.append(",\"receipt\":" + rctText + "}}");
		try {
			stream.append(out.toString());
		}
		catch (IOException ioe) {
			m_Log.error("ReceiptBlock.addJSON(): exception appending receipt",
						ioe);
		}
	}

	/**
	 * calculate a signature for this receipt (must not be signed already)
	 * @param sigKey the private key for the signature
	 * @return the signature, in 130-char hex digest format (null on failure)
	 */
	public String calcSignature(BigInteger privKey) {
		final String lbl = this.getClass().getSimpleName() + ".calcSignature: ";
		if (privKey == null) {
			m_Log.error(lbl + "missing sig key");
			return null;
		}
		String sigData = buildSignedData(false, false);
		if (sigData == null) {
			m_Log.error(lbl + "cannot build signed data");
			return null;
		}

		// build an Ethereum-prefixed hash for signature
		String ethHash = EncodingUtils.ethPrefixHash(sigData);

		// sign with key
		String sig = EncodingUtils.signHash(ethHash, privKey);
		if (sig == null) {
			m_Log.error(lbl + "error generating signature");
		}
		return sig;
	}

	/**
	 * obtain the address which signed this receipt
	 * @return the address, in 0x format (empty on failure)
	 */
	public String getSigAddr() {
		final String lbl = this.getClass().getSimpleName() + ".getSigAddr: ";
		String sigAddr = "";

		// receipt must be signed
		if (m_MVOSignature.m_Signature.isEmpty()) {
			m_Log.error(lbl + "receipt does not appear to be signed");
			return sigAddr;
		}

		String sigData = buildSignedData(false, false);
		if (sigData == null) {
			m_Log.error(lbl + "cannot build signed data");
			return sigAddr;
		}

		// build Ethereum-prefixed hash for verification
		String ethHash = EncodingUtils.ethPrefixHash(sigData);

		// determine the address which produced the signature
		sigAddr = EncodingUtils.getHashSignerAddress(ethHash,
													m_MVOSignature.m_Signature);
		if (sigAddr.isEmpty()) {
			m_Log.error(lbl + "unable to recover signing address");
		}
		return sigAddr;
	}

	
	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Payees != null) {
				m_Payees.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
