/*
 * last modified---
 * 	07-22-22 alphabetize JSON outputs for RLP
 * 	07-12-22 improve error message labeling
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	02-21-22 new
 *
 * purpose---
 * 	encapsulate NFT metadata required for ERC-1155
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.io.IOException;


/**
 * This class holds the metadata necessary to implement the ERC-1155 standard.
 * Most of the fields here will have constant, or not very interesting values,
 * since the important fields are in the nested generic {properties} object,
 * which is normally encrypted and represented by an {@link eNFTmetadata}.
 */
public final class NFTmetadata implements JSON.Generator {
	// BEGIN data members
	/**
	 * generic title field
	 */
	private String			m_Title;

	/**
	 * name of asset, also generic
	 */
	private String			m_Name;

	/**
	 * number of decimals of precision of balances in this asset (such as 18)
	 */
	private int				m_Decimals;

	/**
	 * generic description
	 */
	private String			m_Description;

	/**
	 * image (always empty for a eNFT)
	 */
	private String			m_Image;

	/**
	 * the actual nested properties object, where everything interesting goes
	 */
	private eNFTmetadata	m_Properties;

	/**
	 * logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger logging object
	 */
	public NFTmetadata(Log logger) {
		m_Log = logger;
		m_Title = "eNFT Metadata";
		m_Name = "Enshrouded";
		m_Decimals = 18;
		m_Description = m_Image = "";
	}

	// GET methods
	/**
	 * obtain the title
	 * @return the generic title
	 */
	public String getTitle() { return m_Title; }

	/**
	 * obtain the asset name
	 * @return the generic name (which does not reveal the real asset involved)
	 */
	public String getName() { return m_Name; }

	/**
	 * obtain the digits of precision of balances in this token
	 * @return number of digits in uint256 representation
	 */
	public int getDecimals() { return m_Decimals; }

	/**
	 * obtain the description
	 * @return the generic description
	 */
	public String getDescription() { return m_Description; }

	/**
	 * obtain the image URL
	 * @return path to image, if any (normally empty for this kind of NFT)
	 */
	public String getImage() { return m_Image; }

	/**
	 * obtain the nested (actual) properties metadata
	 * @return the object which contains the encrypted metadata
	 */
	public eNFTmetadata getProperties() { return m_Properties; }

	// SET methods
	/**
	 * set the title (included for bean completeness)
	 * @param title the title to set
	 */
	public void setTitle(String title) { m_Title = title; }

	/**
	 * set the name (included for bean completeness)
	 * @param name the name to set
	 */
	public void setName(String name) { m_Name = name; }

	/**
	 * configure the digits of precision
	 * @param prec the number of digits of precision for amounts of this asset
	 */
	public void setDecimals(int prec) {
		if (prec > 0) {
			m_Decimals = prec;
		}
		else {
			m_Log.error("NFTmetadata.setDecimals: illegal {decimals}, " + prec);
		}
	}

	/**
	 * set the description (included for bean completeness)
	 * @param desc the description to set
	 */
	public void setDescription(String desc) { m_Description = desc; }

	/**
	 * set the image (included for bean completeness)
	 * @param img the image URL to set
	 */
	public void setImage(String img) { m_Image = img; }

	/**
	 * set the inner properties object
	 * @param prop the nested (encrypted) properties object
	 */
	public void setProperties(eNFTmetadata prop) {
		if (prop != null) {
			m_Properties = prop;
		}
		else {
			m_Log.error("NFTmetadata.setProperties: missing eNFT properties");
		}
	}

	/**
	 * build from a Map object, the result of a JSON parse
	 * @param map the mapping
	 * @return true on success
	 */
	public boolean buildFromMap(Map meta) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (meta == null || meta.isEmpty()) {
			m_Log.error(lbl + "missing top Map");
			return false;
		}
		boolean ret = true;

		// required field: title
		Object title = meta.get("title");
		if (title instanceof String) {
			setTitle((String)title);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing title");
		}
		// required field: properties
		Object props = meta.get("properties");
		if (props instanceof Map) {
			Map properties = (Map) props;
			// required field: name
			Object name = properties.get("name");
			if (name instanceof String) {
				setName((String)name);
			}
			else {
				ret = false;
				m_Log.error(lbl + "missing name");
			}
			// required field: decimals
			Object decimals = properties.get("decimals");
			if (decimals instanceof String) {
				Long dec = Long.valueOf((String)decimals);
				setDecimals(dec.intValue());
			}
			else {
				ret = false;
				m_Log.error(lbl + "missing decimals");
			}
			// required field: description
			Object desc = properties.get("description");
			if (desc instanceof String) {
				setDescription((String)desc);
			}
			else {
				ret = false;
				m_Log.error(lbl + "missing description");
			}
			// optional meaningless field: image
			Object image = properties.get("image");
			if (image instanceof String) {
				setImage((String)image);
			}
			// deal with nested properties object (the important bits)
			Object eProps = properties.get("properties");
			if (eProps instanceof Map) {
				Map eProperties = (Map) eProps;
				m_Properties = new eNFTmetadata(m_Log);
				ret = m_Properties.buildFromMap(eProperties);
			}
			else {
				// missing; error
				ret = false;
				m_Log.error(lbl + "missing eNFT properties");
			}
		}
		else {
			// missing; error
			ret = false;
			m_Log.error(lbl + "missing NFT properties");
		}
		return ret;
	}

	/**
	 * method to build object from a JSON string representing one
	 * @param json the JSON string
	 * @return true on success
	 */
	public boolean buildFromJSON(String json) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromJSON: ";
		if (json == null) {
			m_Log.error(lbl + "missing JSON");
			return false;
		}
		if (json.isEmpty()) {
			m_Log.error(lbl + "missing JSON");
			return true;
		}
		boolean ret = true;
		Object meta = null;
		try {
			meta = JSON.parse(json);
		}
		catch (IllegalStateException ise) {
			m_Log.error(lbl + "NFT properties were not JSON: \"" + json + "\"");
			ret = false;
			return ret;
		}
		if (meta instanceof Map) {
			Map metadata = (Map) meta;
			// this will also parse any nested eNFT metadata
			ret = buildFromMap(metadata);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing NFT properties");
		}
		return ret;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the Appendable data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		// emit without any whitespace; needs to be sorted
		StringBuilder out = new StringBuilder(256);
		out.append("{\"properties\":{");
		out.append("\"decimals\":\"" + m_Decimals + "\",");
		out.append("\"description\":\"" + m_Description + "\",");
		out.append("\"image\":\"" + m_Image + "\",");
		out.append("\"name\":\"" + m_Name + "\",");
		out.append("\"properties\":{");
		try {
			stream.append(out);
			// add any nested object
			if (m_Properties != null) {
				m_Properties.addJSON(stream);
			}
			else {
				stream.append("\"enshrouded\":{}");
			}
			stream.append("}},\"title\":\"" + m_Title + "\"}");
		}
		catch (IOException ioe) {
			m_Log.error("NFTmetadata.addJSON(): exception appending", ioe);
		}
	}

	// END methods
}
