/*
 * last modified---
 * 	07-12-22 improve error output labeling
 * 	05-27-22 correct array JSON emission
 * 	03-23-22 new
 *
 * purpose---
 * 	encapsulate reply messages from MVOs to dApps related to wallet downloads
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.ArrayList;
import java.util.Map;
import java.text.NumberFormat;
import java.io.IOException;


/**
 * This class holds the data necessary to build JSON objects related to wallet
 * downloads which are sent to dApp clients as replies.  This class knows how
 * to emit itself as JSON.
 */
public final class MVOWalletBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * constant for success status return
	 */
	public final String			M_Success = "success";

	/**
	 * return status of the receipt request, either M_Success or an error text
	 */
	private String				m_Status;

	/**
	 * helper class describing a eNFT wallet object
	 */
	public class WalletSpec {
		/**
		 * the eNFT index (eNFT001, eNFT002, etc.)
		 */
		public String		m_Sequence;

		/**
		 * the eNFT's decrypted data, in POJO form
		 */
		public eNFTmetadata	m_eNFTmetadata;

		/**
		 * the AES key which decrypts the record stored on the event log
		 * (base64url-encoded)
		 */
		public String		m_Key;

		/**
		 * nullary constructor
		 */
		public WalletSpec() {
			m_Sequence = m_Key = "";
			m_eNFTmetadata = new eNFTmetadata(m_Log);
		}
	}

	/**
	 * list of eNFTs being returned
	 */
	private ArrayList<WalletSpec>	m_Wallet;

	/**
	 * MVO signature on the receipt operation reply
	 */
	private MVOSignature		m_Signature;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * error code status (must be 0 to indicate success)
	 */
	private int					m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger the logging object
	 */
	public MVOWalletBlock(Log logger) {
		m_Log = logger;
		m_Status = M_Success;
		m_Signature = new MVOSignature();
		m_Wallet = new ArrayList<WalletSpec>();
	}

	// GET methods
	/**
	 * obtain the status message
	 * @return the return status, either "success" or an error message
	 */
	public String getStatus() { return m_Status; }

	/**
	 * obtain the list of wallet outputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<WalletSpec> getWallet() { return m_Wallet; }

	/**
	 * obtain the MVO signatures for this reply block
	 */
	public MVOSignature getSignature() { return m_Signature; }

	/**
	 * obtain the error code from the last addJSON() call
	 * @return the result code from building output JSON
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * config the status return value (either M_Success or an error value)
	 * @param stat the status to set
	 */
	public void setStatus(String stat) {
		if (stat != null && !stat.isEmpty()) {
			m_Status = new String(stat);
		}
		else {
			m_Log.error("MVOWalletBlock.setStatus(): missing status return "
						+ "value");
		}
	}

	/**
	 * add an eNFT to the output list
	 * @param nft a eNFT descriptor
	 */
	public void addToWallet(WalletSpec nft) {
		if (nft != null) {
			m_Wallet.add(nft);
		}
	}

	/**
	 * configure the signature on the signed fields
	 * @param sig the signature of the MVO generating this reply
	 */
	public void setSignature(MVOSignature sig) {
		if (sig != null) {
			m_Signature = sig;
		}
		else {
			m_Log.error("MVOWalletBlock.setSignature(): missing MVO signature");
		}
	}

	/**
	 * Build the JSON text that's actually signed.
	 * @return the JSON string for the element without delimiters, or null on
	 * error
	 */
	public String buildSignedData() {
		StringBuilder out = new StringBuilder(5120);
		m_ErrCode = 0;
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);

		// add status
		if (m_Status.isEmpty()) {
			m_ErrCode = 1;
			m_Log.error("MVOWalletBlock.buildSignedData(): missing status "
						+ "return value");
			return null;
		}
		out.append("\"status\":\"" + m_Status + "\",");

		// add array of eNFT specifications
		out.append("\"eNFTs\":[");
		int eIdx = 1;
		for (WalletSpec wSpec : m_Wallet) {
			wSpec.m_Sequence = "eNFT" + nf.format(eIdx++);
			if (wSpec.m_eNFTmetadata.getID().isEmpty()) {
				m_ErrCode = 2;
				m_Log.error("MVOWalletBlock.buildSignedData(): wallet "
							+ wSpec.m_Sequence
							+ " has missing decrypted eNFT data");
				return null;
			}
			out.append("{\"" + wSpec.m_Sequence + "\":{");
			wSpec.m_eNFTmetadata.addJSON(out);
			out.append(",\"key\":\"" + wSpec.m_Key + "\"}}");
			// add comma except on last
			if (eIdx <= m_Wallet.size()) {
				out.append(",");
			}
		}
		out.append("]");

		// NB: signature should be added to this
		return out.toString();
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{");
		// first, put the signed data in
		String mvoSigned = buildSignedData();
		if (mvoSigned == null || m_ErrCode != 0) {
			return;
		}
		out.append(mvoSigned);

		// add sig on mvoSigned
		if (m_Signature.m_Signature.isEmpty()) {
			m_Log.error("MVOWalletBlock.addJSON(): missing signature");
			m_ErrCode = 7;
			return;
		}
		out.append(",\"signature\":{");
		out.append("\"signer\":\"" + m_Signature.m_Signer + "\",");
		out.append("\"sig\":\"" + m_Signature.m_Signature + "\"");
		out.append("}}");	// end Signature obj, outer obj
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("MVOWalletBlock.addJSON(): exception appending", ioe);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_Wallet != null) {
				m_Wallet.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
