/*
 * last modified---
 * 	07-12-22 improve error output labeling
 * 	05-27-22 correct array JSON emission
 * 	03-23-22 return decrypted data, not encrypted
 * 	03-08-22 new
 *
 * purpose---
 * 	encapsulate reply messages from MVOs to dApps related to receipt operations
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.ArrayList;
import java.util.Map;
import java.text.NumberFormat;
import java.io.IOException;


/**
 * This class holds the data necessary to build JSON objects related to receipts
 * which are sent to dApp clients as replies.  This class knows how to emit
 * itself as JSON.
 */
public final class MVOReceiptBlock implements JSON.Generator {
	// BEGIN data members
	/**
	 * constant for success status return
	 */
	public final String			M_Success = "success";

	/**
	 * opcode for reply block, one of list|get|delete (echo of request opcode)
	 */
	private String				m_Opcode;

	/**
	 * return status of the receipt request, either "success" or an error text
	 */
	private String				m_Status;

	/**
	 * list of file specifications returned, if any (list and delete opcodes)
	 */
	private ArrayList<String>	m_FileSpecs;

	/**
	 * helper class describing a receipt object (get opcode)
	 */
	public class ReceiptSpec {
		/**
		 * the receipt index (receipt001, receipt002, etc.)
		 */
		public String		m_Sequence;

		/**
		 * the filename of the receipt
		 */
		public String		m_Filename;

		/**
		 * the receipt data, decrypted
		 */
		public ReceiptBlock	m_ReceiptData;

		/**
		 * nullary constructor
		 */
		public ReceiptSpec() {
			m_Sequence = m_Filename = "";
			m_ReceiptData = new ReceiptBlock(m_Log);
		}
	}

	/**
	 * list of receipts being returned (get opcode only)
	 */
	public ArrayList<ReceiptSpec>	m_Receipts;

	/**
	 * MVO signature on the receipt operation reply
	 */
	private MVOSignature		m_Signature;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * error code status (must be 0 to indicate success)
	 */
	private int					m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param logger the logging object
	 */
	public MVOReceiptBlock(Log logger) {
		m_Log = logger;
		m_Opcode = "";
		m_Status = M_Success;
		m_FileSpecs = new ArrayList<String>();
		m_Signature = new MVOSignature();
		m_Receipts = new ArrayList<ReceiptSpec>();
	}

	// GET methods
	/**
	 * obtain the opcode of this reply block
	 * @return the code, one of list|get|delete
	 */
	public String getOpcode() { return m_Opcode; }

	/**
	 * obtain the status message
	 * @return the return status, either "success" or an error message
	 */
	public String getStatus() { return m_Status; }

	/**
	 * obtain the list of filespec outputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<String> getFileSpecs() { return m_FileSpecs; }

	/**
	 * obtain the list of receipt outputs for this reply block
	 * @return the list, empty if none
	 */
	public ArrayList<ReceiptSpec> getReceipts() { return m_Receipts; }

	/**
	 * obtain the MVO signatures for this reply block
	 */
	public MVOSignature getSignature() { return m_Signature; }

	/**
	 * obtain the error code from the last addJSON() call
	 * @return the result code from building output JSON
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * set the opcode for the reply block
	 * @param op the opcode, one of deposit|spend|withdrawal
	 */
	public void setOpcode(String op) {
		if (op != null && !op.isEmpty()) {
			m_Opcode = new String(op);
		}
		else {
			m_Log.error("MVOReceiptBlock.setOpcode(): missing opcode");
		}
	}

	/**
	 * config the status return value (either M_Success or an error value)
	 * @param stat the status to set
	 */
	public void setStatus(String stat) {
		if (stat != null && !stat.isEmpty()) {
			m_Status = new String(stat);
		}
		else {
			m_Log.error("MVOReceiptBlock.setStatus(): missing status return "
						+ "value");
		}
	}

	/**
	 * add a receipt filespec output to list
	 * @param spec the filespec to add
	 */
	public void addFileSpec(String spec) {
		if (spec != null) {
			m_FileSpecs.add(spec);
		}
	}

	/**
	 * add a receipt to the output list
	 * @param rct a receipt descriptor
	 */
	public void addReceipt(ReceiptSpec rct) {
		if (rct != null) {
			m_Receipts.add(rct);
		}
	}

	/**
	 * configure the signature on the signed fields
	 * @param sig the signature of the MVO generating this reply
	 */
	public void setSignature(MVOSignature sig) {
		if (sig != null) {
			m_Signature = sig;
		}
		else {
			m_Log.error("MVOReceiptBlock.setSignature(): missing MVO "
						+ "signature");
		}
	}

	/**
	 * Build the JSON text that's actually signed.
	 * @return the JSON string for the element without delimiters, or null on
	 * error (m_ErrCode set to some appropriate value)
	 */
	public String buildSignedData() {
		StringBuilder out = new StringBuilder(5120);
		m_ErrCode = 0;
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		final String lbl
			= this.getClass().getSimpleName() + ".buildSignedData: ";

		// add opcode
		if (m_Opcode.isEmpty()) {
			m_ErrCode = 1;
			m_Log.error(lbl + "missing opcode");
			return null;
		}
		out.append("\"opcode\":\"" + m_Opcode + "\",");

		// add status
		if (m_Status.isEmpty()) {
			m_ErrCode = 2;
			m_Log.error(lbl + "missing status return value");
			return null;
		}
		out.append("\"status\":\"" + m_Status + "\",");

		// other elements will depend upon the opcode
		if (m_Opcode.equals(ClientReceiptBlock.M_ReceiptList)
			|| m_Opcode.equals(ClientReceiptBlock.M_ReceiptDel))
		{
			// add array of file specifications
			out.append("\"filespecs\":[");
			int fIdx = 1;
			for (String filespec : m_FileSpecs) {
				out.append("\"" + filespec + "\"");
				// add comma except on last
				if (fIdx++ < m_FileSpecs.size()) {
					out.append(",");
				}
			}
			out.append("]");
		} else if (m_Opcode.equals(ClientReceiptBlock.M_ReceiptGet)) {
			int rIdx = 1;
			out.append("\"receipts\":[");
			for (ReceiptSpec rSpec : m_Receipts) {
				rSpec.m_Sequence = "receipt" + nf.format(rIdx);
				if (rSpec.m_Filename.isEmpty()) {
					m_ErrCode = 4;
					m_Log.error(lbl + "receipt " + rSpec.m_Sequence
								+ " has missing filename");
					return null;
				}
				if (rSpec.m_ReceiptData.getReceiptId().isEmpty()) {
					m_ErrCode = 5;
					m_Log.error(lbl + "receipt " + rSpec.m_Sequence
								+ " has missing receipt data");
					return null;
				}
				out.append("{\"" + rSpec.m_Sequence + "\":{");
				out.append("\"filename\":\"" + rSpec.m_Filename + "\",");
				out.append("\"receiptData\":");
				// add the actual JSON text of the receipt itself
				rSpec.m_ReceiptData.addJSON(out);
				out.append("}}");
				// add comma except on last
				if (rIdx++ < m_Receipts.size()) {
					out.append(",");
				}
			}
			out.append("]");
		}
		else {
			m_ErrCode = 6;
			m_Log.error(lbl + "illegal opcode, " + m_Opcode);
			return null;
		}
		// NB: signature should be added to this
		return out.toString();
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{");
		// first, put the signed data in
		String mvoSigned = buildSignedData();
		if (mvoSigned == null || m_ErrCode != 0) {
			return;
		}
		out.append(mvoSigned);

		// add sig on mvoSigned
		if (m_Signature.m_Signature.isEmpty()) {
			m_Log.error("MVOReceiptBlock.addJSON(): missing signature");
			m_ErrCode = 7;
			return;
		}
		out.append(",\"signature\":{");
		out.append("\"signer\":\"" + m_Signature.m_Signer + "\",");
		out.append("\"sig\":\"" + m_Signature.m_Signature + "\"");
		out.append("}}");	// end Signature obj, outer obj
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("MVOReceiptBlock.addJSON(): exception appending", ioe);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_FileSpecs != null) {
				m_FileSpecs.clear();
			}
			if (m_Receipts != null) {
				m_Receipts.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
