/*
 * last modified---
 * 	09-27-23 support user signature validation on Auditor side via clientAuth
 * 	09-30-22 allow receipt opcode of delete
 * 	09-14-22 parse signature too; allow receiptRequest element to be either
 * 				  a ClientReceiptBlock or a ClientMVOBlock
 * 	08-03-22 corrections to emission and parsing
 * 	07-12-22 improve error output labeling
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-24-22 new
 *
 * purpose---
 * 	encapsulate MVO requests to an Auditor to create or fetch AES keys
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.util.ArrayList;
import java.io.IOException;


/**
 * This class holds the data for a request (over a WebSocket connection)
 * from one MVO to an Auditor.  The first (lead) MVO creates AES-256 keys for
 * encrypting eNFTs, and all MVOs in the committee must access them.  Any MVO
 * can field a request from a client for their wallet eNFTs and keys.  Also,
 * receipts are generated by lead MVOs and likewise require keys.  Again, any
 * MVO can be asked by a client dApp to perform operations on receipts (fetch
 * or delete).  In such cases, the client's original {@link ClientRequest}
 * will be included by the MVO, to prove the client authorized the key access.
 * This class knows how to construct itself from input JSON, and emit itself
 * as JSON.
 *
 * The Auditor will return a signed {@link AuditorKeyBlock} reply.
 */
public final class MVOKeyBlock implements JSON.Generator {
	// BEGIN data members
	// the two types of AES key mappings: eNFT | receipt
	/**
	 * access the Hazelcast mapping related to eNFTs
	 */
	public final String			M_MapENFT = "eNFT";

	/**
	 * access the Hazelcast mapping related to receipts
	 */
	public final String			M_MapReceipt = "receipt";

	// the opcodes: one of new | get | delete
	/**
	 * new: create a key not in the map
	 */
	public final String			M_OpNew = "new";

	/**
	 * get: obtain a key already in the indicated map
	 */
	public final String			M_OpGet = "get";

	/**
	 * delete: remove a key from the receipt map (illegal for eNFT map)
	 */
	public final String			M_OpDelete = "delete";

	/**
	 * the mapping the request applies to
	 */
	public String				m_Mapping;

	/**
	 * the actual opcode (must be one of the three constants above)
	 */
	public String				m_Opcode;

	/**
	 * list of hashes for which keys are wanted (each is a keccak256 hash of
	 * either chainId+ID+address (for eNFTs) or chainId+receiptID+address
	 * (for receipts))
	 */
	public ArrayList<String>	m_HashList;

	/**
	 * the dApp client's original request (will be either a ClientMVOBlock,
	 * a ClientReceiptBlock, or a ClientWalletBlock)
	 */
	private ClientRequest		m_ClientBlock;

	/**
	 * original received JSON string representing the ClientRequest
	 */
	private String				m_ClientBlockJson;

	/**
	 * MVO signature on the key operation request
	 */
	private MVOSignature		m_Signature;

	/**
	 * logging object
	 */
	private Log					m_Log;

	/**
	 * error code status (must be 0 to indicate success)
	 */
	private int					m_ErrCode;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param log logging object
	 */
	public MVOKeyBlock(Log log) {
		m_Log = log;
		m_ClientBlockJson = m_Opcode = m_Mapping = "";
		m_HashList = new ArrayList<String>();
	}

	// GET methods
	/**
	 * obtain the mapping
	 * @return the applicable mapping, one of the two M_Map* constants
	 */
	public String getMapping() { return m_Mapping; }

	/**
	 * obtain the opcode
	 * @return the code, one of the three M_Op* constants
	 */
	public String getOpcode() { return m_Opcode; }

	/**
	 * obtain the hash list
	 * @return the list of requested key index hashes
	 */
	public ArrayList<String> getHashList() { return m_HashList; }

	/**
	 * obtain the client request block
	 * @return the client's original request, forwarded
	 */
	public ClientRequest getClientBlock() { return m_ClientBlock; }

	/**
	 * obtain the client request block JSON
	 * @return the original string received before parsing
	 */
	public String getClientBlockJson() { return m_ClientBlockJson; }

	/**
	 * obtain the client request signature
	 * @return the signature of the MVO client
	 */
	public MVOSignature getSignature() { return m_Signature; }

	/**
	 * obtain the error code
	 * @return the last error code value set
	 */
	public int getErrCode() { return m_ErrCode; }


	// SET methods
	/**
	 * configure the mapping
	 * @param map the mapping to set (must equal one of the M_Map* constants)
	 */
	public void setMapping(String map) {
		if (map != null) {
			m_Mapping = new String(map);
		}
	}

	/**
	 * configure the opcode
	 * @param op the opcode to set (must equal one of the M_Op* constants)
	 */
	public void setOpcode(String op) {
		if (op != null) {
			m_Opcode = new String(op);
		}
	}

	/**
	 * configure the client block
	 * @param the client's original forwarded block
	 */
	public void setClientBlock(ClientRequest cBlock) {
		if (cBlock != null) {
			m_ClientBlock = cBlock;
		}
	}

	/**
	 * configure the client block json
	 * @param json the original JSON string received
	 */
	public void setClientBlockJson(String json) {
		if (json != null && !json.isEmpty()) {
			m_ClientBlockJson = new String(json);
		}
	}

	/**
	 * add a hash index to the list
	 * @param hash the hash to add
	 * @return true if the list changed
	 */
	public boolean addHashToList(String hash) {
		if (hash == null || hash.isEmpty()) {
			return false;
		}
		return m_HashList.add(hash);
	}

	/**
	 * configure the signature on the signed fields
	 * @param sig the signature of the MVO generating this reply
	 */
	public void setSignature(MVOSignature sig) {
		if (sig != null) {
			m_Signature = sig;
		}
		else {
			m_Log.error("MVOKeyBlock.setSignature: missing MVO signature");
		}
	}


	/**
	 * parse a JSON string containing this kind of block (called by Auditors)
	 * @param msg the message from the MVO
	 * @return true on success
	 */
	public boolean buildFromString(String msg) {
		if (msg == null || msg.isEmpty()) {
			m_Log.error("MVOKeyBlock.buildFromString: missing data");
			return false;
		}
		boolean ret = true;
		Object data = null;
		try {
			data = JSON.parse(msg);
		}
		catch (IllegalStateException ise) {
			m_Log.error("MVOKeyBlock.buildFromString: forwarded data was not "
						+ "JSON: \"" + msg + "\"");
			ret = false;
			return ret;
		}
		if (data instanceof Map) {
			Map reqData = (Map) data;
			ret = buildFromMap(reqData, msg);
		}
		else {
			m_Log.error("MVOKeyBlock.buildFromString: forwarded data was not "
						+ "a Map");
			ret = false;
		}
		return ret;
	}

	/**
	 * build the objects from a passed Map, created from this JSON structure.
	 * @param map the mapping, created by parsing JSON
	 * @param orgJson the original JSON message that was parsed
	 * @return true on success
	 */
	public boolean buildFromMap(Map map, String orgJson) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (map == null || map.isEmpty()
			|| orgJson == null || orgJson.isEmpty())
		{
			m_Log.error(lbl + "missing or empty top Map");
			return false;
		}
		boolean ret = true;

		// get the mapping type
		Object mapping = map.get("mapping");
		if (mapping instanceof String) {
			String mapType = (String) mapping;
			if (mapType.equals(M_MapENFT) || mapType.equals(M_MapReceipt)) {
				setMapping(mapType);
			}
			else {
				m_Log.error(lbl + "illegal map type, " + mapping);
				ret = false;
			}
		}

		// get the opcode
		Object opCode = map.get("opcode");
		if (opCode instanceof String) {
			String op = (String) opCode;
			if (op.equals(M_OpNew) || op.equals(M_OpGet)
				|| op.equals(M_OpDelete))
			{
				setOpcode(op);
			}
			else {
				m_Log.error(lbl + "illegal opcode, " + op);
				ret = false;
			}
			// check for illegal combo
			if (m_Opcode.equals(M_OpDelete) && !m_Mapping.equals(M_MapReceipt))
			{
				m_Log.error(lbl + "cannot delete from eNFT key map");
				ret = false;
			}
		}

		if (ret) {
			// obtain the hash list
			Object hashes = map.get("hashes");
			if (hashes instanceof Object[]) {
				Object[] hashList = (Object[]) hashes;
				for (int iii = 0; iii < hashList.length; iii++) {
					Object hashItem = hashList[iii];
					if (hashItem instanceof String) {
						String hash = (String) hashItem;
						addHashToList(hash);
					}
				}
			}
			else {
				m_Log.error(lbl + "missing hash list");
				ret = false;
			}
		}

		// we should have a copy of the client's authorizing request
		if (ret) {
			Object clientAuth = map.get("clientAuth");
			if (clientAuth instanceof Map) {
				Map authMap = (Map) clientAuth;

				/* This Map was generated from an EIP-712 signed client
				 * message.  It will contain top-level elements including:
				 * domain, message, primaryType, and types.  To validate client
				 * signature, we need the entire message, which we'll store in
				 * m_ClientBlockJson.
				 *
				 * The actual client message will be found in
				 * message.requestJson, and will correspond to one of these
				 * cases:
				 * 	msg.requestJson.depositspec - a mint request
				 * 	msg.requestJson.spendspec - a spend request
				 * 	msg.requestJson.withdrawspec - a burn request
				 * 	msg.requestJson.receiptRequest - receipt manipulation
				 * 	msg.requestJson.walletDownload - eNFT listing request
				 */
				// enable signature validation by finding clientAuth elt text
				final String startTag = "\"clientAuth\":";
				int authReqStart = orgJson.indexOf(startTag);
				// NB: including the { in the end tag selects the MVO's sig @end
				int authReqEnd = orgJson.lastIndexOf(",\"signature\":{");
				String authJson = orgJson.substring(
								authReqStart + startTag.length(), authReqEnd);
				setClientBlockJson(authJson);

				// find message.payloadJson, which should be a Map
				Object messageObj = authMap.get("message");
				if (!(messageObj instanceof Map)) {
					m_Log.error(lbl + "could not find message in signed data");
					ret = false;
				}
				else {
					Map messageMap = (Map) messageObj;
					Object requestJson = messageMap.get("requestJson");
					if (!(requestJson instanceof Map)) {
						m_Log.error(lbl + "could not find message.requestJson");
						ret = false;
					}
					else {
						Map jsonMap = (Map) requestJson;

						// look for the various situations listed above
						ClientReceiptBlock rBlock = null;
						ClientWalletBlock wBlock = null;
						ClientMVOBlock cBlock = null;
						Map payloadMap = null;
						Object payloadReq = jsonMap.get("receiptRequest");
						if (payloadReq instanceof Map) {
							payloadMap = (Map) payloadReq;
							rBlock = new ClientReceiptBlock(null, m_Log);
							if (!rBlock.buildFromMap(payloadMap)) {
								m_Log.error(lbl + "error parsing forwarded "
											+ "ClientReceiptBlock");
								ret = false;
							}
							else {
								rBlock.setDecryptedPayload(authJson);
								m_ClientBlock = rBlock;
							}
						}
						else {
							payloadReq = jsonMap.get("walletDownload");
							if (payloadReq instanceof Map) {
								payloadMap = (Map) payloadReq;
								wBlock = new ClientWalletBlock(null, m_Log);
								if (!wBlock.buildFromMap(payloadMap)) {
									m_Log.error(lbl + "error parsing forwarded "
												+ "ClientWalletBlock");
									ret = false;
								}
								else {
									wBlock.setDecryptedPayload(authJson);
									m_ClientBlock = wBlock;
								}
							}
							else {
								// must be depositspec, spendspec, withdrawspec
								cBlock = new ClientMVOBlock(null, m_Log);
								cBlock.setDecryptedPayload(authJson);
								payloadReq = jsonMap.get("depositspec");
								if (payloadReq instanceof Map) {
									payloadMap = (Map) payloadReq;
									if (!cBlock.buildFromMap(payloadMap)) {
										ret = false;
									}
								}
								else {
									payloadReq = jsonMap.get("spendspec");
									if (payloadReq instanceof Map) {
										payloadMap = (Map) payloadReq;
										if (!cBlock.buildFromMap(payloadMap)) {
											ret = false;
										}
									}
									else {
										payloadReq
											= jsonMap.get("withdrawspec");
										if (payloadReq instanceof Map) {
											payloadMap = (Map) payloadReq;
											if (!cBlock.buildFromMap(
																	payloadMap))
											{
												ret = false;
											}
										}
										else {
											ret = false;
											m_Log.error(lbl
														+ "clientAuth element "
														+ "is not recognized");
										}
									}
								}
								if (!ret) {
									m_Log.error(lbl + "error parsing forwarded "
												+ "ClientMVOBlock");
								}
								else {
									m_ClientBlock = cBlock;
								}
							}
						}
					}
				}
			}
			else if (clientAuth instanceof String) {
				// this is okay; it's probably empty
				m_ClientBlockJson = (String) clientAuth;
			}
		}

		// get signature (should have one if we're parsing)
		Object sigSpec = map.get("signature");
		if (sigSpec instanceof Map) {
			Map signature = (Map) sigSpec;
			MVOSignature sig = new MVOSignature();

			// get MVO signer ID
			Object signer = signature.get("signer");
			if (signer instanceof String) {
				String mvoID = (String) signer;
				sig.m_Signer = new String(mvoID);
			}
			else {
				ret = false;
				m_Log.error(lbl + "sig missing signer");
			}

			// get actual signature
			Object mSig = signature.get("sig");
			if (mSig instanceof String) {
				String mvoSig = (String) mSig;
				sig.m_Signature = new String(mvoSig);
			}
			else {
				ret = false;
				m_Log.error(lbl + "sig missing signature");
			}
			if (ret) setSignature(sig);
		}
		else {
			ret = false;
			m_Log.error(lbl + "sig missing or was not a Map");
		}
		return ret;
	}

	/**
	 * build the JSON text that's actually signed by the MVO
	 * @return the JSON string for the element, without delimiters, or null on
	 * any error (m_ErrCode set to some appropriate value)
	 */
	public String buildSignedData() {
		StringBuilder out = new StringBuilder(5120);
		m_ErrCode = 0;
		final String lbl
			= this.getClass().getSimpleName() + ".buildSignedData: ";

		// add the mapping
		if (m_Mapping.isEmpty()) {
			m_ErrCode = 1;
			m_Log.error(lbl + "missing mapping type");
			return null;
		}
		out.append("\"mapping\":\"" + m_Mapping + "\",");

		// add opcode
		if (m_Opcode.isEmpty()) {
			m_ErrCode = 2;
			m_Log.error(lbl + "missing opcode");
			return null;
		}
		out.append("\"opcode\":\"" + m_Opcode + "\",");

		// add the list of hashes
		if (m_HashList.isEmpty()) {
			m_ErrCode = 3;
			m_Log.error(lbl + "missing hash list");
			return null;
		}
		out.append("\"hashes\":[");
		int iHash = 1;
		for (String hash : m_HashList) {
			out.append("\"" + hash + "\"");
			// add comma except for last
			if (iHash++ < m_HashList.size()) {
				out.append(",");
			}
		}
		out.append("]");

		// if available, add authorizing request from user
		if (m_ClientBlockJson.isEmpty()) {
			// this happens when we're asking for keys to generate receipts
			out.append(",\"clientAuth\":\"\"");
		}
		else {
			// recipient (Auditor) will need to verify client's signature
			out.append(",\"clientAuth\":");
			out.append(m_ClientBlockJson);
		}

		return out.toString();
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the object in JSON format
	 * @param stream the data stream to write on
	 */
	@Override
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);
		m_ErrCode = 0;
		out.append("{");
		// first, put the signed data in
		String mvoSigned = buildSignedData();
		if (mvoSigned == null || m_ErrCode != 0) {
			return;
		}
		out.append(mvoSigned);

		// NB: caller is responsible for adding signature previous to this call
		if (m_Signature.m_Signature.isEmpty()) {
			m_Log.error("MVOKeyBlock.addJSON: missing signature");
			m_ErrCode = 7;
			return;
		}
		out.append(",\"signature\":{");
		out.append("\"signer\":\"" + m_Signature.m_Signer + "\",");
		out.append("\"sig\":\"" + m_Signature.m_Signature + "\"");
		out.append("}}");   // end Signature obj, outer obj
		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("MVOKeyBlock.addJSON: exception appending", ioe);
		}
	}

	/**
	 * finalize the object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out sensitive data if present
		try {
			if (m_HashList != null) {
				m_HashList.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
