/*
 * last modified---
 * 	06-05-23 remove m_Signature (now at a higher level due to EIP-712)
 * 	03-20-23 add m_ReplyKey
 * 	03-17-23 move m_ChainId, m_Sender, m_Signature here; add m_DecryptedPayload
 * 	03-30-22 new
 *
 * purpose---
 * 	base class for every type of dApp client request
 */

package cc.enshroud.jetty;

import javax.servlet.http.HttpServletRequest;


/**
 * This class is implemented by every type of client request object.  A suitable
 * subclass instance will be created by ClientHandler or ReceiptHandler when a
 * validated request from a dApp user is received.
 */
public class ClientRequest {
	// BEGIN data members
	/**
	 * the original raw request received from a dApp. ClientHandler,
	 * WalletHandler, or ReceiptHandler will parse this into a subclass of
	 * this class.  If it was passed encrypted, the decrypted text will be
	 * stored in m_DecryptedPayload.
	 */
	private HttpServletRequest	m_OrigRequest;

	/**
	 * the chain ID, which identifies the smart contract and event log involved
	 * (common to all derived classes).  This is a key index hash constituent,
	 * and also identifies receipt storage and SC API URIs.
	 */
	protected long				m_ChainId;

	/**
	 * Sender address (requestor), which must match signature.
	 * This is also built into key index hashes.
	 */
	protected String			m_Sender;

	/**
	 * the decrypted payload value, if any
	 */
	private String				m_DecryptedPayload;

	/**
	 * client's indicated AES key for encrypting replies (base64 format)
	 */
	protected String			m_ReplyKey;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param orgReq the original request received from the dApp in the callback
	 */
	public ClientRequest(HttpServletRequest orgReq) {
		m_OrigRequest = orgReq;
		m_Sender = m_DecryptedPayload = m_ReplyKey = "";
	}

	// GET methods
	/**
	 * obtain the original request
	 * @return the HTTP level request
	 */
	public HttpServletRequest getOrigRequest() { return m_OrigRequest; }

	/**
	 * obtain the ID of the chain this request refers to
	 * @return the ID of the blockchain, according to standard (chainlist.org)
	 */
	public long getChainId() { return m_ChainId; }

	/**
	 * obtain the client sender address
	 * @return the dApp client wallet address
	 */
	public String getSender() { return m_Sender; }

	/**
	 * obtain the decrypted payload (this will include JSON for a subclass)
	 * @return the decrypted original client payload
	 */
	public String getDecryptedPayload() { return m_DecryptedPayload; }

	/**
	 * obtain the reply key
	 * @return the AES key the client wants to use to encrypt our reply to it
	 */
	public String getReplyKey() { return m_ReplyKey; }


	// SET methods
	/**
	 * set the decrypted payload, after successful decryption
	 * @param decPayload the JSON which will parse into a subclass
	 */
	public void setDecryptedPayload(String decPayload) {
		if (decPayload != null && !decPayload.isEmpty()) {
			m_DecryptedPayload = new String(decPayload);
		}
	}

	/**
	 * set the reply key
	 * @param repKey the AES key the client set for encrypting our reply to it
	 */
	public void setReplyKey(String repKey) {
		if (repKey != null && !repKey.isEmpty()) {
			m_ReplyKey = new String(repKey);
		}
	}

	// END methods
}
