/*
 * last modified---
 * 	03-27-25 add ClientInput.m_AESkey
 * 	09-27-23 mark addJSON() depcrecated
 * 	09-22-23 change inputs[] and payees[] to use *Label and *Spec fields
 * 	06-05-23 remove m_Signature, as this is now at a higher level
 * 	04-11-23 add M_ARRAY_MAX
 * 	04-05-23 make getAsset*Total() compare assets ignoring case
 * 	03-20-23 add handling of m_ReplyKey if present
 * 	02-20-23 make M_Op* static
 * 	12-05-22 add ClientPayee.m_Rand value
 * 	09-14-22 also emit asset on deposits and withdrawals; don't emit
 * 			 payees[] on withdraws
 * 	08-18-22 add getAssetInputTotal() and getAssetOutputTotal()
 * 	07-22-22 disallow negative payee output amounts
 * 	07-12-22 improve error output labeling
 * 	06-15-22 implement addJSON()
 * 	05-26-22 catch IllegalStateException from JSON.parse()
 * 	03-30-22 extend ClientRequest
 * 	02-23-22 new
 *
 * purpose---
 * 	encapsulate request messages to MVOs from dApp clients
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import org.eclipse.jetty.util.ajax.JSON;

import java.util.Map;
import java.util.ArrayList;
import java.math.BigInteger;
import java.text.NumberFormat;
import java.io.IOException;
import javax.servlet.http.HttpServletRequest;


/**
 * This class holds the data for dApp requests made to MVOs.  Such requests are
 * received on the https endpoint, parsed, and stored in these objects for
 * processing by MVOs.  The output of MVO processing will be a
 * {@link OperationsBlock}, which is passed back to the dApp in the reply.
 * This class knows how to build itself from input JSON.
 */
public final class ClientMVOBlock extends ClientRequest
	implements JSON.Generator
{
	// BEGIN data members
	/**
	 * opcode constant: deposit
	 */
	public final static String	M_OpDeposit = "deposit";
	/**
	 * opcode constant: spend
	 */
	public final static String	M_OpSpend = "spend";
	/**
	 * opcode constant: withdraw
	 */
	public final static String	M_OpWithdraw = "withdraw";

	/**
	 * maximum number of input or output eNFTs allowed by the smart contract
	 */
	public final static int	M_ARRAY_MAX = 20;

	/**
	 * opcode for request, one of deposit|spend|withdraw
	 */
	private String			m_Opcode;

	/**
	 * single token asset represented, for deposits and withdrawals
	 * (as the erc20 contract address)
	 */
	private String			m_Asset;

	/**
	 * quantity of asset represented (a uint256 represented as a BigInteger),
	 * for deposits and withdrawals
	 */
	private BigInteger		m_Amount;

	/**
	 * helper class describing a payee specifications (for all opcodes)
	 */
	public class ClientPayee {
		/**
		 * payee ID (sequence number, such as payee001, payee002 etc.)
		 */
		public String		m_Payee;

		/**
		 * address of the payee, to which an eNFT will be minted (hex number)
		 */
		public String		m_Address;

		/**
		 * asset type (used for spends only)
		 */
		public String		m_OutputAsset;

		/**
		 * amount of the eNFT to be issued (uint256 in {decimals} precision)
		 */
		public BigInteger	m_OutputAmount;

		/**
		 * units: empty if m_OutputAmount is an absolute amount, or "%" to
		 * indicate that m_OutputAmount is interpreted as a percentage of the
		 * m_Amount (deposit/withdraw only)
		 */
		public String		m_Units;

		/**
		 * random seed value; if set gets included by the MVO in the details
		 * hash for the generated eNFT.  The dApp must include it also to check
		 * the validity of the details hash before approving the operation.
		 */
		public String		m_Rand;

		/**
		 * memo from m_Sender to m_Address, if any
		 */
		public String		m_Memo;

		/**
		 * nullary constructor
		 */
		public ClientPayee() {
			m_Payee = m_Address = m_OutputAsset = m_Units = m_Memo = m_Rand
				= "";
			m_OutputAmount = new BigInteger("0");
		}
	}

	/**
	 * list of payees defined in request
	 */
	private ArrayList<ClientPayee>	m_Payees;

	/**
	 * helper class describing input specifications (for spends and withdrawals)
	 */
	public class ClientInput {
		/**
		 * input ID (sequence number, such as input001, input002, etc.)
		 */
		public String		m_Input;

		/**
		 * JSON representing the {enshrouded} element, containing all details
		 * and the MVO signature
		 */
		public String		m_Enshrouded;

		/**
		 * the rest is an eNFT object (decrypted)
		 */
		public eNFTmetadata	m_eNFT;

		/**
		 * base64-encoded AES key which decrypts this eNFT on-chain (optional)
		 */
		public String		m_AESkey;

		/**
		 * nullary constructor
		 */
		public ClientInput() {
			m_Input = m_Enshrouded = m_AESkey = "";
		}
	}

	/**
	 * list of input eNFTs provided by client for operation
	 */
	private ArrayList<ClientInput>	m_Inputs;

	/**
	 * original client data, including the signature
	 */
	private String			m_RequestData;

	/**
	 * logging object
	 */
	private Log				m_Log;

	// END data members

	// BEGIN methods
	/**
	 * constructor
	 * @param orgReq the original HTTP level request
	 * @param logger the logging object
	 */
	public ClientMVOBlock(HttpServletRequest orgReq, Log logger) {
		super(orgReq);
		m_Log = logger;
		m_Opcode = m_Asset = m_RequestData = "";
		m_Amount = new BigInteger("0");
		m_Payees = new ArrayList<ClientPayee>(M_ARRAY_MAX);
		m_Inputs = new ArrayList<ClientInput>(M_ARRAY_MAX);
	}

	// GET methods
	/**
	 * obtain the opcode of this request
	 * @return the code, one of deposit|spend|withdraw
	 */
	public String getOpcode() { return m_Opcode; }

	/**
	 * obtain the asset
	 * @return the asset (token address) involved in a deposit or withdraw
	 */
	public String getAsset() { return m_Asset; }

	/**
	 * obtain the quantity of the asset, uint256 but represented as a BigInteger
	 * @return the amount of this asset involved in the deposit or withdraw
	 */
	public BigInteger getAmount() { return m_Amount; }

	/**
	 * obtain the list of payee outputs for this request
	 * @return the list, empty if none
	 */
	public ArrayList<ClientPayee> getPayees() { return m_Payees; }

	/**
	 * obtain the list of eNFT inputs for this request
	 * @return the list, empty if none
	 */
	public ArrayList<ClientInput> getInputs() { return m_Inputs; }

	/**
	 * obtain the original request data
	 * @return the client wallet's original data before we parsed it
	 */
	public String getRequestData() { return m_RequestData; }

	/**
	 * obtain the total of all inputs in a given asset type
	 * @param asset the asset type of interest
	 * @return the total of all inputs in that asset
	 */
	public BigInteger getAssetInputTotal(String asset) {
		BigInteger total = new BigInteger("0");
		if (asset == null || asset.isEmpty()) {
			m_Log.error("ClientMVOBlock.getAssetInputTotal: missing asset");
			return total;
		}
		for (ClientInput inp : m_Inputs) {
			eNFTmetadata inpData = inp.m_eNFT;
			if (inpData != null && inpData.getAsset().equalsIgnoreCase(asset)) {
				// count this one
				total = total.add(inpData.getAmount());
			}
		}
		return total;
	}

	/**
	 * obtain the total of all outputs in a given asset type
	 * @param asset the asset type of interest
	 * @return the total of all outputs in that asset
	 */
	public BigInteger getAssetOutputTotal(String asset) {
		BigInteger total = new BigInteger("0");
		if (asset == null || asset.isEmpty()) {
			m_Log.error("ClientMVOBlock.getAssetOutputTotal: missing asset");
			return total;
		}
		// to handle percentages of inputs, get that total first
		BigInteger inpTotal = getAssetInputTotal(asset);
		final BigInteger one100 = new BigInteger("100");
		for (ClientPayee outp : m_Payees) {
			if (asset.equalsIgnoreCase(outp.m_OutputAsset)) {
				// include this one
				if (outp.m_Units.equals("%")) {
					// interpret as a percentage of total
					BigInteger amt = outp.m_OutputAmount.multiply(inpTotal);
					BigInteger incr = amt.divide(one100);
					total.add(incr);
				}
				else {
					total.add(outp.m_OutputAmount);
				}
			}
		}
		if (total.compareTo(inpTotal) != 0) {
			m_Log.warning("ClientMVOBlock.getAssetOutputTotal: total for asset "
						+ asset + " (" + total + ") doesn't match input tot of "
						+ inpTotal);
		}
		return total;
	}

	// SET methods (use operator new to convert from stack to heap variables)
	/**
	 * config the chain Id
	 * @param id the ID of the blockchain this request is for, per chainlist.org
	 */
	public void setChainId(long id) {
		if (id > 0L) {
			m_ChainId = id;
		}
		else {
			m_Log.error("ClientMVOBlock.setChainId(): illegal chainId, " + id);
		}
	}

	/**
	 * set the opcode for the request
	 * @param op the opcode, one of deposit|spend|withdraw
	 */
	public void setOpcode(String op) {
		if (op != null && !op.isEmpty()) {
			m_Opcode = new String(op);
		}
		else {
			m_Log.error("ClientMVOBlock.setOpcode(): missing opcode");
		}
	}

	/**
	 * set the sender of the request
	 * @param sender the wallet address which sent (and signed) the request
	 */
	public void setSender(String sender) {
		if (sender != null && !sender.isEmpty()) {
			m_Sender = new String(sender);
		}
		else {
			m_Log.error("ClientMVOBlock.setSender(): missing sender");
		}
	}

	/**
	 * config the asset (used for deposits and withdrawals)
	 * @param asset the asset (token address) referenced in the request
	 */
	public void setAsset(String asset) {
		if (asset != null && !asset.isEmpty()) {
			m_Asset = new String(asset);
		}
		else {
			m_Log.error("ClientMVOBlock.setAsset(): missing asset");
		}
	}

	/**
	 * config the quantity of the asset, uint256 but represented as a string
	 * @param amt amount of this asset represented by the eNFT
	 */
	public void setAmount(String amt) {
		if (amt != null && !amt.isEmpty()) {
			try {
				m_Amount = new BigInteger(amt);
			}
			catch (NumberFormatException nfe) {
				m_Log.error("ClientMVOBlock.setAmount(): illegal amount, "
							+ amt);
			}
		}
		else {
			m_Log.error("ClientMVOBlock.setAmount(): missing amount");
		}
	}

	/**
	 * record sender's original request data
	 * @param request the exact text received, before we parsed it
	 */
	public void setRequestData(String request) {
		if (request != null && !request.isEmpty()) {
			m_RequestData = new String(request);
		}
		else {
			m_Log.error("ClientMVOBlock.setRequestData(): missing request "
						+ "data");
		}
	}

	/**
	 * add a client payee output to list
	 * @param payee the payee to add
	 */
	public void addPayee(ClientPayee payee) {
		if (payee != null) {
			m_Payees.add(payee);
		}
	}

	/**
	 * add an eNFT input to list
	 * @param eNFT an input (may or may not be parsed yet)
	 */
	public void addInput(ClientInput eNFT) {
		if (eNFT != null) {
			m_Inputs.add(eNFT);
		}
	}

	/**
	 * method to build object from a Map
	 * @param request the mapping, the result of a JSON parse
	 * @return true on success
	 */
	public boolean buildFromMap(Map request) {
		final String lbl = this.getClass().getSimpleName() + ".buildFromMap: ";
		if (request == null || request.isEmpty()) {
			m_Log.error(lbl + "missing client request Map");
			return false;
		}
		boolean ret = true;

		// required: the chainId
		Object chain = request.get("chainId");
		if (chain instanceof String) {
			String chainId = (String) chain;
			try {
				Long cId = Long.parseLong(chainId.trim());
				setChainId(cId.longValue());
			}
			catch (NumberFormatException nfe) {
				ret = false;
				m_Log.error(lbl + "illegal chain Id, " + chainId, nfe);
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing chain Id");
		}

		// required: the opcode
		Object op = request.get("opcode");
		if (op instanceof String) {
			String opcode = (String) op;
			setOpcode(opcode);
			if (!(m_Opcode.equals(M_OpDeposit) || m_Opcode.equals(M_OpSpend)
				  || m_Opcode.equals(M_OpWithdraw)))
			{
				ret = false;
				m_Log.error(lbl + "illegal client opcode, " + m_Opcode);
				return ret;
			}
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing opcode");
			return ret;
		}

		// required: the sender address
		Object send = request.get("sender");
		if (send instanceof String) {
			String sender = (String) send;
			setSender(sender);
		}
		else {
			ret = false;
			m_Log.error(lbl + "missing sender");
		}

		// the asset (required for deposit and withdraw)
		Object ass = request.get("asset");
		if (ass instanceof String) {
			String asset = (String) ass;
			setAsset(asset);
		}
		else if (!m_Opcode.equals(M_OpSpend)) {
			ret = false;
			m_Log.error(lbl + "missing client asset type");
		}

		// the total amount (required for deposit and withdraw)
		Object tot = request.get("amount");
		if (tot instanceof String) {
			String total = (String) tot;
			setAmount(total);
		}
		else if (!m_Opcode.equals(M_OpSpend)) {
			ret = false;
			m_Log.error(lbl + "missing client amount");
		}

		// optional: the AES encryption reply key
		Object rKey = request.get("replyKey");
		if (rKey instanceof String) {
			String repKey = (String) rKey;
			setReplyKey(repKey);
		}

		// get inputs (at least one required except for deposits)
		NumberFormat nf = NumberFormat.getIntegerInstance();
		nf.setMinimumIntegerDigits(3);
		Object inps = request.get("inputs");
		if (inps instanceof Object[]) {
			Object[] inputs = (Object[]) inps;
			if (inputs.length < 1 && !m_Opcode.equals(M_OpDeposit)) {
				ret = false;
				m_Log.error(lbl + "missing eNFT inputs");
			}
			for (int iii = 0; iii < inputs.length; iii++) {
				Object inp = inputs[iii];
				if (inp instanceof Map) {
					Map input = (Map) inp;
					// get input index
					long idx = iii + 1;
					String key = "input" + nf.format(idx);
					Object inputLab = input.get("inputLabel");
					String inputLabel = key;
					if (inputLab instanceof String) {
						inputLabel = (String) inputLab;
						if (!inputLabel.equals(key)) {
							m_Log.warning(lbl + "inputLabel = " + inputLabel
										+ ", expected " + key);
						}
					}
					Object inputSpec = input.get("inputSpec");
					if (inputSpec instanceof Map) {
						Map eNFT = (Map) inputSpec;
						ClientInput ci = new ClientInput();
						ci.m_Input = new String(inputLabel);
						if (!buildInputFromMap(eNFT, ci)) {
							ret = false;
							m_Log.error(lbl + "eNFT parse error");
						}
						else {
							addInput(ci);

							// if present, add AES key to ClientInput
							Object aesKey = input.get("key");
							if (aesKey instanceof String) {
								ci.m_AESkey = (String) aesKey;
							}
						}
					}
					else {
						m_Log.error(lbl + "illegal input eNFT "
									+ "spec for key = " + key);
					}
				}
				else {
					ret = false;
					m_Log.error(lbl + "non-Map eNFT input");
				}
			}
		}
		else if (!m_Opcode.equals(M_OpDeposit)) {
			ret = false;
			m_Log.error(lbl + "missing eNFT inputs");
		}

		// get payees (at least one required except for withdrawals)
		Object pays = request.get("payees");
		if (pays instanceof Object[]) {
			Object[] payees = (Object[]) pays;
			if (payees.length < 1 && !m_Opcode.equals(M_OpWithdraw)) {
				ret = false;
				m_Log.error(lbl + "missing payee outputs");
			}
			for (int iii = 0; iii < payees.length; iii++) {
				Object pay = payees[iii];
				if (pay instanceof Map) {
					Map payee = (Map) pay;
					boolean goodPayee = true;
					// get payee index
					long idx = iii + 1;
					String key = "payee" + nf.format(idx);
					Object payeeLab = payee.get("payeeLabel");
					String payeeLabel = key;
					if (payeeLab instanceof String) {
						payeeLabel = (String) payeeLab;
						if (!payeeLabel.equals(key)) {
							m_Log.warning(lbl + "payeeLabel = " + payeeLabel
										+ ", expected " + key);
						}
					}
					Object paySpec = payee.get("payeeSpec");
					ClientPayee cp = new ClientPayee();
					if (paySpec instanceof Map) {
						cp.m_Payee = new String(payeeLabel);
						Map payeeSpec = (Map) paySpec;

						// get address
						Object addr = payeeSpec.get("address");
						if (addr instanceof String) {
							String address = (String) addr;
							cp.m_Address = new String(address);
						}
						else if (!m_Opcode.equals(M_OpSpend)) {
							// this is okay; use sender
							cp.m_Address = getSender();
						} else {
							goodPayee = false;
							m_Log.error(lbl + "missing payee address");
						}

						// get asset
						Object ast = payeeSpec.get("asset");
						if (ast instanceof String) {
							String asset = (String) ast;
							cp.m_OutputAsset = new String(asset);
						}
						else if (!m_Opcode.equals(M_OpSpend)) {
							// this is okay; use asset
							cp.m_OutputAsset = getAsset();
						} else {
							goodPayee = false;
							m_Log.error(lbl + "missing payee asset");
						}

						// get amount
						Object amt = payeeSpec.get("amount");
						if (amt instanceof String) {
							String amount = (String) amt;
							try {
								cp.m_OutputAmount = new BigInteger(amount);
							}
							catch (NumberFormatException nfe) {
								goodPayee = false;
								m_Log.error(lbl + "illegal payee "
											+ "amount, " + amt, nfe);
							}

							// disallow amounts of less than zero
							if (cp.m_OutputAmount.compareTo(BigInteger.ZERO)
								< 0)
							{
								goodPayee = false;
								m_Log.error(lbl + "negative payee amount, "
											+ amt);
							}
						}
						else {
							goodPayee = false;
							m_Log.error(lbl + "missing payee amount");
						}

						// get units (can be blank)
						Object unit = payeeSpec.get("units");
						if (unit instanceof String) {
							String units = (String) unit;
							cp.m_Units = new String(units);
						}
						else {
							goodPayee = false;
							m_Log.error(lbl + "illegal units");
						}

						// get random hash seed (optional)
						Object rnd = payeeSpec.get("rand");
						if (rnd instanceof String) {
							String rand = (String) rnd;
							cp.m_Rand = new String(rand);
						}

						// get memo (optional)
						Object mem = payeeSpec.get("memo");
						if (mem instanceof String) {
							String memo = (String) mem;
							// disallow quote characters in string
							if (memo.indexOf('"') != -1) {
								memo = memo.replaceAll("\"", "");
							}
							cp.m_Memo = new String(memo);
						}
					}
					else {
						m_Log.error(lbl + "payee for key " + key
									+ " was not a Map");
						goodPayee = false;
					}
					if (goodPayee) {
						addPayee(cp);
					}
					else {
						ret = false;
						break;
					}
				}
				else {
					ret = false;
					m_Log.error(lbl + "non-Map payee list");
					break;
				}
			}
		}
		else if (!m_Opcode.equals(M_OpWithdraw)) {
			ret = false;
			m_Log.error(lbl + "missing eNFT outputs");
		}

		return ret;
	}

	/**
	 * method to build entire request object from a JSON string
	 * @param reqData the request data (a JSON object)
	 * @return true on success
	 */
	public boolean buildFromString(String reqData) {
		boolean ret = false;
		if (reqData == null | reqData.isEmpty()) {
			m_Log.error("ClientMVOBlock.buildFromString(): missing client "
						+ "request data");
			return ret;
		}
		setRequestData(reqData);
		Object req = null;
		try {
			req = JSON.parse(reqData);
		}
		catch (IllegalStateException ise) {
			m_Log.error("ClientMVOBlock.buildFromString(): client JSON parse "
						+ "failed", ise);
			return ret;
		}

		// parsed object should consist of a Map
		if (!(req instanceof Map)) {
			m_Log.error("ClientMVOBlock.buildFromString(): client JSON was not "
						+ "a Map");
			return ret;
		}
		Map map = (Map) req;
		ret = buildFromMap(map);
		return ret;
	}

	/**
	 * method to build object from a JSON map
	 * @param mapData the properties data (unencrypted)
	 * @param clientInp the client input object we're building
	 * @return true on success
	 */
	public boolean buildInputFromMap(Map mapData, ClientInput clientInp) {
		if (mapData == null || mapData.isEmpty() || clientInp == null) {
			return false;
		}

		// use eNFTmetadata class to parse this (data should be one)
		eNFTmetadata eNFT = new eNFTmetadata(m_Log);
		if (eNFT.buildFromMap(mapData)) {
			clientInp.m_eNFT = eNFT;
			// also store original JSON text
			StringBuilder enftData = new StringBuilder(2048);
			eNFT.addJSON(enftData);
			clientInp.m_Enshrouded = enftData.toString();
		}
		else {
			m_Log.error("ClientMVOBlock.buildInputFromMap(): eNFT data parse "
						+ "failure");
			return false;
		}
		return true;
	}

	// method to implement interface JSON.Generator
	/**
	 * emit the elements of the object in JSON format.
	 * NB: this method should not be used to generate JSON for inclusion in a
	 * larger object such as an MVOAuditorBlock, MVOVerifyBlock, or MVOKeyBlock,
	 * because it does not include the EIP-712 elements (types, message, etc.)
	 * which are required to validate the user's wallet signature on the data.
	 * @param stream the data stream to write on
	 */
	@Override
	@Deprecated
	public void addJSON(Appendable stream) {
		StringBuilder out = new StringBuilder(10240);

		/* NB: because this object was parsed successfully from an incoming
		 * 	   message, we assume here that all values are present and correct.
		 */
		// add chain Id
		out.append("{\"chainId\":\"" + m_ChainId + "\",");

		// add opcode
		out.append("\"opcode\":\"" + m_Opcode + "\",");

		// add sender
		out.append("\"sender\":\"" + m_Sender + "\",");

		// if we have one, add the AES reply key
		if (!m_ReplyKey.isEmpty()) {
			out.append("\"replyKey\":\"" + m_ReplyKey + "\",");
		}

		// add asset and amount except for spends
		if (!m_Opcode.equals(M_OpSpend)) {
			out.append("\"asset\":\"" + m_Asset + "\",");
			out.append("\"amount\":\"" + m_Amount + "\",");
		}

		// add inputs, except for deposits
		if (!m_Opcode.equals(M_OpDeposit)) {
			out.append("\"inputs\":[");
			int iii = 1;
			for (ClientInput input : m_Inputs) {
				// input label flag
				out.append("{\"inputLabel\":\"" + input.m_Input + "\",");
				out.append("\"inputSpec\":{");
				// enshrouded eNFT data, as text
				out.append(input.m_Enshrouded + "}");
				// AES key if present
				if (!input.m_AESkey.isEmpty()) {
					out.append(",\"key\":\"" + input.m_AESkey + "\"}");
				}
				else {
					out.append("}");
				}
				// add comma except for last time
				if (iii++ < m_Inputs.size()) {
					out.append(",");
				}
			}
			out.append("],");
		}

		// add payees, except for withdraws
		if (!m_Opcode.equals(M_OpWithdraw)) {
			int ppp = 1;
			out.append("\"payees\":[");
			for (ClientPayee payee : m_Payees) {
				// sequence number
				out.append("{\"payeeLabel\":\"" + payee.m_Payee + "\",");
				out.append("\"payeeSpec\":{");
				// minting address
				out.append("\"address\":\"" + payee.m_Address + "\",");
				// asset type (spends only)
				if (m_Opcode.equals(M_OpSpend)) {
					out.append("\"asset\":\"" + payee.m_OutputAsset + "\",");
				}
				// amount
				out.append("\"amount\":\"" + payee.m_OutputAmount + "\",");
				// units
				out.append("\"units\":\"" + payee.m_Units + "\"");
				// optional rand hash seed
				if (!payee.m_Rand.isEmpty()) {
					out.append(",\"rand\":\"" + payee.m_Rand + "\"");
				}
				// optional memo
				if (!payee.m_Memo.isEmpty()) {
					out.append(",\"memo\":\"" + payee.m_Memo + "\"");
				}
				out.append("}}");
				// add comma except for last time
				if (ppp++ < m_Payees.size()) {
					out.append(",");
				}
			}
			out.append("]");
		}
		out.append("}");

		try {
			stream.append(out);
		}
		catch (IOException ioe) {
			m_Log.error("ClientMVOBlock.addJSON(): exception appending", ioe);
		}
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero sensitive data
		try {
			if (m_Inputs != null) {
				m_Inputs.clear();
			}
			if (m_Payees != null) {
				m_Payees.clear();
			}
		} finally {
			super.finalize();
		}
	}

	// END methods
}
