/*
 * last modified---
 * 	06-29-23 move m_ABISession to SmartContractConfig
 * 	06-09-23 activate m_ABISession
 * 	04-11-23 add finalize()
 * 	12-01-22 add m_ECPrivKey, m_ECPubKey, and methods
 * 	09-07-22 move up from .mvo for Auditor access
 * 	07-12-22 improve error message labeling
 * 	04-22-22 new, move here from subclass in MVOConfig
 *
 * purpose---
 * 	describe smart contract configuration items for a MVO on a given blockchain
 */

package cc.enshroud.jetty;

import cc.enshroud.jetty.log.Log;

import java.math.BigInteger;
import java.security.PrivateKey;
import java.security.PublicKey;


/**
 * class to describe MVO configuration items specific to particular blockchains
 * (used by both MVOs and Auditors)
 */
public final class BlockchainConfig {
	// data members
	/**
	 * chain ID for blockchain, according to the standard
	 */
	private long			m_ChainId;

	/**
	 * colloquial name for blockchain
	 */
	private String			m_Chain;

	/**
	 * quantity of $ENSHROUD staked by this MVO (normally on chainId = 1)
	 */
	private BigInteger		m_Staked;

	/**
	 * address associated with on-chain signing credentials (0x value)
	 */
	private String			m_SigningAddress;

	/**
	 * ECDSA private key used to sign as m_SigningAddress (only set for self)
	 */
	private BigInteger		m_SigningKey;

	/**
	 * ECDSA private key used to decrypt incoming client messages (the
	 * corresponding public key is on the blockchain in our config record;
	 * only set for self)
	 */
	private PrivateKey		m_ECPrivKey;

	/**
	 * ECDSA public key used by clients to encrypt incoming messages.
	 * (This public key is also on the blockchain in the staking record, but
	 * isn't used by MVOs since separate RSA keys are utilized instead.)
	 */
	private PublicKey		m_ECPubKey;

	/**
	 * address associated with on-chain staking of m_Staked $ENSHROUD
	 */
	private String			m_StakingAddress;

	/**
	 * local convenience copy of logging object
	 */
	private Log				m_Log;

	// methods
	/**
	 * constructor
	 * @param cId the chain Id per the standard
	 * @param chain the colloquial name of the blockchain
	 * @param logger the logging object
	 */
	public BlockchainConfig(long cId, String chain, Log logger) {
		m_ChainId = cId;
		m_Chain = chain;
		m_Log = logger;
		m_Staked = BigInteger.ZERO;
		m_SigningKey = BigInteger.ZERO;
		m_SigningAddress = m_StakingAddress = "";
	}

	/**
	 * obtain chain Id
	 * @return the Id, set according to the industry standards
	 */
	public long getChainId() { return m_ChainId; }

	/**
	 * obtain chain name
	 * @return the colloquial name of the chain
	 */
	public String getChain() { return m_Chain; }

	/**
	 * obtain the quantity of $ENSHROUD staked by this MVO
	 * @return the amount, in 18 digit precision
	 */
	public BigInteger getStaking() { return m_Staked; }

	/**
	 * obtain the address used to verify signatures of this MVO
	 * @return the address corresponding to the MVO's private signing key
	 */
	public String getSigningAddress() { return m_SigningAddress; }

	/**
	 * obtain the signing key that goes with m_SigningAddress
	 * @return the key, as a number
	 */
	public BigInteger getSigningKey() { return m_SigningKey; }

	/**
	 * obtain the decryption key for incoming client messages
	 * @return the key
	 */
	public PrivateKey getECPrivKey() { return m_ECPrivKey; }

	/**
	 * obtain the public key for encrypting incoming client messages
	 * @return the key
	 */
	public PublicKey getECPubKey() { return m_ECPubKey; }

	/**
	 * obtain the address used to hold staking for this MVO
	 * @return the address corresponding to the MVO's staking of m_Staked
	 */
	public String getStakingAddress() { return m_StakingAddress; }

	/**
	 * configure the staking quantity
	 * @param stake the amount of $ENSHROUD this MVO has staked, 18 digit
	 * prec, represented as ascii digits
	 */
	public void setStaking(String stake) {
		if (stake != null && !stake.isEmpty()) {
			try {
				m_Staked = new BigInteger(stake);
			}
			catch (NumberFormatException nfe) {
				m_Log.error("BlockchainConfig.setStaking: illegal stake, "
							+ stake, nfe);
			}
		}
		else {
			m_Log.error("BlockchainConfig.setStaking: missing stake");
		}
	}

	/**
	 * configure the address used to verify this MVO's signatures
	 * @param addr the appropriate blockchain address
	 */
	public void configSigningAddress(String addr) {
		if (addr != null) {
			m_SigningAddress = addr;
		}
		else {
			m_Log.error("BlockchainConfig.configSigningAddress: missing "
						+ "address");
		}
	}

	/**
	 * configure the address used to hold this MVO's $ENSHROUD staking
	 * @param addr the appropriate blockchain address
	 */
	public void configStakingAddress(String addr) {
		if (addr != null) {
			m_StakingAddress = addr;
		}
		else {
			m_Log.error("BlockchainConfig.configStakingAddress: missing "
						+ "address");
		}
	}

	/**
	 * configure the signing key for the blockchain (m_SigningAddress)
	 * @param key the key to assign
	 * (set by using org.web3j.crypto.Credentials to create keypair
	 * from wallet JSON + password taken from .properties)
	 */
	public void configSigningKey(BigInteger key) {
		m_SigningKey = key;
	}

	/**
	 * configure the private key used for decryption (separate from signing key)
	 * @param privKey the key
	 */
	public void setECPrivateKey(PrivateKey privKey) {
		m_ECPrivKey = privKey;
	}

	/**
	 * configure the public key used for encryption (separate from signing key)
	 * @param privKey the key
	 */
	public void setECPublicKey(PublicKey pubKey) {
		m_ECPubKey = pubKey;
	}

	/**
	 * finalize object when garbage-collected
	 * @throws Throwable on fatal error
	 */
	@Override
	protected void finalize() throws Throwable {
		// zero out any sensitive data
		try {
			if (m_ECPrivKey != null) {
				m_ECPrivKey.destroy();
			}
		} finally {
			super.finalize();
		}
	}
}
