/*
 * last modified---
 * 	06-20-23 add new method checkIfIdUsed(); add acct to getNFTsById()
 * 	06-12-23 pass org.web3j.protocol.Web3j instead of URLs
 * 	10-26-22 add chainId arg to getAccountNFTs() and getNFTsById()
 * 	09-06-22 move up from .mvo
 * 	07-29-22 add demintedOk arg to getNFTsById()
 * 	06-27-22 make getAccountNFTs() return a HashMap<ID, NFT> so we have IDs
 * 	04-22-22 new
 *
 * purpose---
 * 	define an interface for making requests through the Web3j ABI
 */

package cc.enshroud.jetty;

import org.web3j.protocol.Web3j;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.concurrent.Future;


/**
 * This interface abstracts what we need to be able to acquire from our full
 * blockchain partner nodes.  At runtime, these methods will all be provided by
 * the implementing class, which will utilize Web3j calls (synchronous or
 * asynchronous) to fetch the actual data from the indicated blockchain.  An
 * implementing class can also be provided for testing, which pulls the data
 * from a filesystem or database.
 */
public interface BlockchainAPI {
	/**
	 * initialize object
	 * @return true on success
	 */
	public boolean init();

	/**
	 * obtain the smart contract configuration for a given blockchain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiService the object at which we can access Web3j service
	 * @return the configuration, with data supplied appropriately
	 */
	public Future<SmartContractConfig> getConfig(long chainId,
												 String chain,
												 Web3j abiService);

	/**
	 * obtain the configuration particular to a given MVO on the chain
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param chain the blockchain's colloquial name
	 * @param abiService the object at which we can access Web3j service
	 * @param mvoId the ID of the MVO
	 * @return the configuration (public parts only)
	 */
	public Future<BlockchainConfig> getMVOConfig(long chainId,
												 String chain,
												 Web3j abiService,
												 String mvoId);

	/**
	 * obtain all the eNFTs on-chain for a given account address
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param abiService the object at which we can access Web3j service
	 * @param acct the account address
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted)
	 */
	public Future<HashMap<String, NFTmetadata>> getAccountNFTs(long chainId,
															   Web3j abiService,
														 	   String acct,
														 	boolean demintedOk);

	/**
	 * obtain a list of eNFTs, given their IDs
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param abiService the object at which we can access Web3j service
	 * @param acct the account address
	 * @param idList the list of eNFT IDs we want
	 * @param demintedOk if true, include those that have been deminted
	 * @return the list of eNFTs (encrypted)
	 */
	public Future<ArrayList<NFTmetadata>> getNFTsById(long chainId,
													  Web3j abiService,
													  String acct,
													  ArrayList<String> idList,
													  boolean demintedOk);

	/**
	 * determine whether a given ID has ever been used in an eNFT (including
	 * deminted ones)
	 * @param chainId the blockchain ID (as sanity check, per standard)
	 * @param abiService the object at which we can access Web3j service
	 * @param id the eNFT Id candidate
	 * @return true if an eNFT exists with that Id, else false (null on error)
	 */
	public Future<Boolean> checkIfIdUsed(long chainId,
										 Web3j abiService,
										 String id);

	/**
	 * shut down the object
	 * @return true on success
	 */
	public boolean shutdown();
}
